from __future__ import annotations
from abc import abstractmethod, ABC
from copy import copy
from itertools import chain
import re
from typing import Dict, List, Mapping, Optional, Tuple, Type, TYPE_CHECKING, final

from netlistdb import Database, Instance
from netlistdb.impl.sql.sql_model import Net

from api_service.exim.isf_command import ISFAbstractCommand, ISFAssignBlockResourceCommand, ISFCreateBlockCommand, \
                                        ISFCreateClkoutGPIOCommand, ISFCreateInoutGPIOCommand, ISFCreateInputGPIOCommand, \
                                        ISFCreateOutputGPIOCommand, ISFLockBlockPropertyCommand, \
                                        ISFSetBlockPropertyCommand
from api_service.unified_netlist.peri_library import get_peri_netlist
from util.excp import pt_assert

if TYPE_CHECKING:
    from netlistdb import Netlist, Instance, Connection, NetBus, Net, Database


def get_module_name(module: Netlist):
    name = module.module_name
    if "(" in name:
        name = name.split("(")[0]
    # Hack: ignore the random id appended to the module name
    supported_modules = {
        'EFX_GPIO_V1',
        'EFX_GPIO_V2',
        'EFX_GPIO_V3',
        'EFX_IBUF',
        'EFX_OBUF',
        'EFX_IO_BUF',
        'EFX_IREG',
        'EFX_OREG',
        'EFX_IOREG',
        'EFX_PLL_V1',
        'EFX_OSC_V1',
        'EFX_CLKOUT',
        'EFX_PLL_V2',
        'EFX_PLL_V3',
        'EFX_OSC_V2',
        'EFX_OSC_V3',
        'EFX_CLKMUX_V1',
        'EFX_ODDIO',
        'EFX_IDDIO',
        'EFX_FPLL_V1',
        'EFX_JTAG_V1',
        'EFX_JTAG_CTRL'
    }
    for m in supported_modules:
        if name.startswith(m):
            return m

    return name

def is_verilog_number(a: str) -> bool:
    match_obj = re.match(r"(\d*)\'(b|d|h|o)(\w+)", a)
    return match_obj is not None

def verilog_num_to_int(a: str) -> int:
    if not is_verilog_number(a):
        raise TypeError("Must be verilog number")
    match_obj = re.match(r"(\d*)\'(b|d|h|o)(\w+)", a)
    match match_obj.group(2):
        case 'b':
            value = int(match_obj.group(3), 2)
        case 'd':
            value = int(match_obj.group(3), 10)
        case 'h':
            value = int(match_obj.group(3), 16)
        case 'o':
            value = int(match_obj.group(3), 8)
        case _:
            raise TypeError()
    # TODO: range check
    return value

def is_constant_net(net: Net) -> bool:
    return net.name in {'GND', 'VCC'}

def get_connected_user_interface_name(ndb: Database, net: Net | None) -> str:
    """
    Get the user interface name that the net connected to
    """
    top = ndb.get_top_module()
    assert top is not None

    if net is None:
        # Missing Connection
        return ""

    user_inst = ndb.get_instance(top, inst_name=f'{top.cell_name}~core~inst')
    assert user_inst is not None
    if net.name == 'GND':
        return '__gnd__'
    elif net.name == 'VCC':
        return '__vcc__'
    else:
        ports = net.get_connected_inst_ports(user_inst)
        if len(ports) == 0:
            # Missing connection
            # FIXME: We may need to raise exception to tell user about the connectivity error in their design
            return ""
        elif len(ports) == 1:
            return ports[0].name
        else:
            raise Exception(f'Unhandled case where net connected to multiple user ports: {[str(p) for p in ports]}')

def get_connected_user_bus_interface_name(ndb: Database, nets: List[Net] | None ) -> str:
    """
    Get the user interface name that the nets connected to, expecting the interface is a portbus
    """
    top = ndb.get_top_module()
    assert top is not None

    if len(nets) == 0:
        # Missing Connection
        return ""

    user_inst = ndb.get_instance(top, inst_name=f'{top.cell_name}~core~inst')
    assert user_inst is not None

    # Check all nets in the list are having name 'GND"
    is_all_connected_to_gnd = all([n.name == 'GND' for n in nets])
    is_all_connected_to_vcd = all([n.name == 'VCC' for n in nets])
    if is_all_connected_to_gnd:
        return '__gnd__'
    elif is_all_connected_to_vcd:
        return '__vcc__'
    else:
        bus_name = None
        for n in nets:
            ports = n.get_connected_inst_ports(user_inst)
            if len(ports) == 0:
                if bus_name is None:
                    bus_name = ""
                elif bus_name != "":
                    raise Exception(f'Unexpected connection, should connect to the same port bus: ')
            elif len(ports) == 1:
                p = ports[0]
                if p.bus is None:
                    raise Exception(f'Unexpected connection, should connect to port bus')

                if bus_name is None:
                    bus_name = p.bus.name
                elif bus_name != p.bus.name:
                    raise Exception(f'Unexpected connection, should connect to the same port bus: {p.bus}')
            else:
                raise Exception(f'Unexpected connection, should connect to one port')

        assert bus_name is not None
        return bus_name


def get_connected_clkout_name(ndb: Database, net: Net) -> str:
    top = ndb.get_top_module()
    assert top is not None

    user_inst = ndb.get_instance(top, inst_name=f'{top.cell_name}~core~inst')
    src_inst, src_port = net.get_source()
    if src_inst == user_inst:
        return src_port.name
    else:
        if src_inst is None:
            raise Exception(f'Not Handled now')
        else:
            # Use the net name if it is driven by other periphery blocks
            return net.name


def remove_double_quote(value: str) -> str:
    assert value.startswith('"')
    assert value.endswith('"')
    return value[1:-1]


def find_command(commands: List[ISFAbstractCommand], cmd_type: Type, block_name: str, prop_name: Optional[str] = None) -> Optional[ISFAbstractCommand]:
    # Helper function to get the command for specific type / block_name / prop_name, raise exception if not found
    for cmd in commands:
        if isinstance(cmd, cmd_type):
            if cmd.block_name == block_name:
                if prop_name is not None and isinstance(cmd, ISFSetBlockPropertyCommand):
                    if cmd.prop_name == prop_name:
                        return cmd
                else:
                    return cmd
    return None


class AbstractISFConverter(ABC):
    """
    Use to convert a instance in netlistdb into list of ISF Commands
    """

    def __init__(self, ndb: Database, partial_design_name: str | None = None):
        self.ndb = ndb
        self.partial_design_name = partial_design_name

    @abstractmethod
    def process_block(self, inst: Instance) -> Tuple[List[ISFAbstractCommand], List[ISFAbstractCommand]] | None:
        raise NotImplementedError

    @property
    @final
    def design_api_handle(self) -> str:
        return "design"

    @property
    @abstractmethod
    def block_type(self) -> str:
        """
        Should return value for the block_type arg the ISF command
        """
        raise NotImplementedError

    def build_prop_cmds(self, block_name: str, prop_name: str, prop_value: str, lock: bool) -> List[ISFSetBlockPropertyCommand | ISFLockBlockPropertyCommand ]:
        cmd = [ISFSetBlockPropertyCommand(self.design_api_handle, block_name, self.block_type, prop_name, prop_value)]
        if lock:
            cmd.append(ISFLockBlockPropertyCommand(self.design_api_handle, block_name, self.block_type, prop_name, prop_value, self.partial_design_name))
        return cmd


class GPIOConnectionPropConverter():

    def __init__(self, ndb: Database, partial_design_name: str | None = None):
        self.ndb = ndb
        self.partial_design_name = partial_design_name

    def convert(self, inst: Instance, conn: Connection, name_overwrite: str = None) -> Optional[List[ISFAbstractCommand]]:
        if inst is None:
            return None
        if conn is None:
            return None

        isf_instance_id = inst.name
        if name_overwrite:
            isf_instance_id = name_overwrite


        design_api_handle = "design"
        def build_lock_cmd(cmd: ISFSetBlockPropertyCommand) -> ISFLockBlockPropertyCommand:
            return ISFLockBlockPropertyCommand(cmd.design_api_handle, cmd.block_name, cmd.block_type, cmd.prop_name, cmd.prop_value, self.partial_design_name)

        def build_prop_conn_cmds(pin_prop_name: str, conn: Connection, lock: bool) -> List[ISFAbstractCommand]:
            name = get_connected_user_interface_name(self.ndb, conn.net)
            cmd = ISFSetBlockPropertyCommand(design_api_handle, isf_instance_id, 'GPIO', pin_prop_name, name)
            if lock:
                lock_cmd = build_lock_cmd(cmd)
                return [cmd, lock_cmd]
            return [cmd]

        def build_simple_prop_cmds(prop_name: str, prop_value: str, lock: bool) -> List[ISFAbstractCommand]:
            cmd = ISFSetBlockPropertyCommand(design_api_handle, isf_instance_id, 'GPIO', prop_name, prop_value)
            if lock:
                lock_cmd = build_lock_cmd(cmd)
                return [cmd, lock_cmd]
            return [cmd]

        name = get_module_name(inst.view)
        port = conn.port

        if name == "EFX_IBUF":
            if port.name == 'O':
                cmds = build_prop_conn_cmds("IN_PIN", conn, lock=True)
                added_conn_type = False
                for conn in conn.net.connections:
                    conn_inst = conn.instance
                    if conn_inst == inst:
                        continue

                    if get_module_name(conn_inst.view) in ['EFX_PLL_V1', 'EFX_PLL_V2', 'EFX_PLL_V3']:
                        if conn.port.name.startswith('CLKIN'):
                            cmds += build_simple_prop_cmds("CONN_TYPE", "PLL_CLKIN", lock=True)
                            added_conn_type = True
                            break

                    # Support GCLK
                    gclk_attr = conn.port.get_attr('global_clock')
                    if gclk_attr and gclk_attr.value == 'TRUE':
                        cmds += build_simple_prop_cmds("CONN_TYPE", "GCLK", lock=True)
                        added_conn_type = True
                        break

                if not added_conn_type:
                    # Don't lock Conn Type if the connection is not a PLL or GCLK
                    cmds += build_simple_prop_cmds('CONN_TYPE', 'NONE', lock=False)

                return cmds

        if name == "EFX_OBUF":
            if port.name == 'I':
                return build_prop_conn_cmds("OUT_PIN", conn, lock=True)

        if name == "EFX_IO_BUF":
            if port.name == 'I':
                return build_prop_conn_cmds('OUT_PIN', conn, lock=True)
            elif port.name == 'O':
                return build_prop_conn_cmds('IN_PIN', conn, lock=True) + \
                        build_simple_prop_cmds('CONN_TYPE', 'NONE', lock=False)
            elif port.name == "OE":
                return build_prop_conn_cmds('OE_PIN', conn, lock=True)

        if name == "EFX_CLKOUT":
            if port.name == "CLK":
                return build_prop_conn_cmds('OUT_CLK_PIN', conn, lock=True)

        if name == "EFX_IREG":
            if port.name == 'O':
                return build_prop_conn_cmds('IN_PIN', conn, lock=True) + build_simple_prop_cmds('CONN_TYPE', 'NONE', lock=False)
            elif port.name == "CLK":
                return build_prop_conn_cmds('IN_CLK_PIN', conn, lock=True)

        if name == "EFX_OREG":
            if port.name == 'I':
                return build_prop_conn_cmds('OUT_PIN', conn, lock=True)
            elif port.name == "CLK":
                return build_prop_conn_cmds('OUT_CLK_PIN', conn, lock=True)

        if name == "EFX_IOREG":
            if port.name == 'I':
                return build_prop_conn_cmds('OUT_PIN', conn, lock=True)
            elif port.name == 'O':
                return build_prop_conn_cmds('IN_PIN', conn, lock=True) + build_simple_prop_cmds('CONN_TYPE', 'NONE', lock=False)
            elif port.name == "OE":
                return build_prop_conn_cmds('OE_PIN', conn, lock=True)
            elif port.name == "INCLK":
                return build_prop_conn_cmds('IN_CLK_PIN', conn, lock=True)
            elif port.name == "OUTCLK":
                return build_prop_conn_cmds('OUT_CLK_PIN', conn, lock=True)

        if name == "EFX_IDDIO":
            if port.name == "CLK":
                return build_prop_conn_cmds('IN_CLK_PIN', conn, lock=True)
            elif port.name == "O_HI":
                return build_prop_conn_cmds('IN_HI_PIN', conn, lock=True) + build_simple_prop_cmds('CONN_TYPE', 'NONE', lock=False)
            elif port.name == "O_LO":
                return build_prop_conn_cmds('IN_LO_PIN', conn, lock=True)

        if name == "EFX_ODDIO":
            if port.name == "CLK":
                return build_prop_conn_cmds('OUT_CLK_PIN', conn, lock=True)
            elif port.name == "I_HI":
                return build_prop_conn_cmds('OUT_HI_PIN', conn, lock=True)
            elif port.name == "I_LO":
                return build_prop_conn_cmds('OUT_LO_PIN', conn, lock=True)

        if name == "EFX_GPIO_V1":
            if port.name == "I":
                return build_prop_conn_cmds("IN_PIN", conn, lock=True) + build_simple_prop_cmds('CONN_TYPE', 'NONE', lock=False)
            elif port.name == "O":
                return build_prop_conn_cmds("OUT_PIN", conn, lock=True)
            elif port.name == 'OE':
                return build_prop_conn_cmds("OE_PIN", conn, lock=True)
            elif port.name == 'INCLK':
                return build_prop_conn_cmds("IN_CLK_PIN", conn, lock=True)
            elif port.name == 'OUTCLK':
                return build_prop_conn_cmds("OUT_CLK_PIN", conn, lock=True)

        def has_connection_on_port(inst, port_name) -> bool:
            result = list(filter(lambda c: c.port.name == port_name, inst.connections))
            return len(result) > 0

        if name == "EFX_GPIO_V2":
            portbus = port.bus
            if portbus:
                if port.name == "I[0]":
                    if has_connection_on_port(inst, "I[1]"):
                        return build_prop_conn_cmds("IN_HI_PIN", conn, lock=True)
                    else:
                        return build_prop_conn_cmds("IN_PIN", conn, lock=True)
                elif port.name == "I[1]":
                    if has_connection_on_port(inst, "I[0]"):
                        return build_prop_conn_cmds("IN_LO_PIN", conn, lock=True)
                    else:
                        raise Exception(f'Unhandled connection {conn}')

                if port.name == "O[0]":
                    if has_connection_on_port(inst, "O[1]"):
                        return build_prop_conn_cmds("OUT_HI_PIN", conn, lock=True)
                    else:
                        return build_prop_conn_cmds("OUT_PIN", conn, lock=True)
                elif port.name == "O[1]":
                    if has_connection_on_port(inst, "O[0]"):
                        return build_prop_conn_cmds("OUT_LO_PIN", conn, lock=True)
                    else:
                        raise Exception('Unhandled connection {conn}')
            else:
                if port.name == "OE":
                    return build_prop_conn_cmds("OE_PIN", conn, lock=True)
                elif port.name == "INCLK":
                    return build_prop_conn_cmds("IN_CLK_PIN", conn, lock=True)
                elif port.name == "OUTCLK":
                    return build_prop_conn_cmds("OUT_CLK_PIN", conn, lock=True)

        if name == "EFX_GPIO_V3":
            pass

        return None


class PLLISFConverter(AbstractISFConverter):

    @property
    def block_type(self) -> str:
        return "PLL"

    def process_simple_port(self, inst: Instance, port_name: str, isf_prop_name: str) -> List[ISFAbstractCommand]:
        net = inst.get_connected_net(inst.get_port(port_name))
        prop_value = ''
        if net is not None:
            prop_value = get_connected_user_interface_name(self.ndb, net)
        return self.build_prop_cmds(inst.name, isf_prop_name, prop_value, lock=True)

    def process_clkout_port(self, inst: Instance, clk_idx: int) -> List[ISFAbstractCommand]:
        name = get_module_name(inst.view)
        net = inst.get_connected_net(inst.get_port(f'CLKOUT{clk_idx}'))
        cmds = []
        if net is None:
            cmds += self.build_prop_cmds(inst.name, f'CLKOUT{clk_idx}_EN', '0', lock=True)
        else:
            cmds += self.build_prop_cmds(inst.name, f'CLKOUT{clk_idx}_EN', '1', lock=True)
            prop_value = get_connected_user_interface_name(self.ndb, net)
            cmds += self.build_prop_cmds(inst.name, f'CLKOUT{clk_idx}_PIN', prop_value, lock=True)
            cmds += self.build_prop_cmds(inst.name, f'CLKOUT{clk_idx}_DIV', str(inst.view.get_param(f'CLKOUT{clk_idx}_DIV').resolve_value()), lock=False)
            if name == 'EFX_PLL_V2':
                cmds += self.build_prop_cmds(inst.name, f'CLKOUT{clk_idx}_PHASE', str(inst.view.get_param(f'CLKOUT{clk_idx}_PHASE').resolve_value()), lock=False)
            elif name in {'EFX_PLL_V3', 'EFX_FPLL_V1'}:
                cmds += self.build_prop_cmds(inst.name, f'CLKOUT{clk_idx}_PHASE_STEP', str(inst.view.get_param(f'CLKOUT{clk_idx}_PHASE_STEP').resolve_value()), lock=False)

                # Inverted and connection type are connection based parameters so we should lock them
                clk_inverted = inst.view.get_param(f'IS_CLKOUT{clk_idx}_INVERTED').resolve_value()
                pt_assert(isinstance(clk_inverted, bool), 'IS_CLKOUT_INVERTED parameter must be a boolean', ValueError)
                cmds += self.build_prop_cmds(inst.name, f'IS_CLKOUT{clk_idx}_INVERTED', '1' if clk_inverted else '0', lock=True)

                if clk_idx == 3 or clk_idx == 4:
                    cmds += self.build_prop_cmds(inst.name, f'CLKOUT{clk_idx}_CONN_TYPE', str(inst.view.get_param(f'CLKOUT{clk_idx}_CONN_TYPE').resolve_value()), lock=True)

                if clk_idx == 1 and name == 'EFX_FPLL_V1':
                    cmds += self.build_prop_cmds(inst.name, 'CLKOUT1_PDIV', str(inst.view.get_param('CLKOUT1_DC_PDIV').resolve_value()), lock=False)
                    cmds += self.build_prop_cmds(inst.name, 'CLKOUT1_SDIV', str(inst.view.get_param('CLKOUT1_DC_SDIV').resolve_value()), lock=False)
                    cmds += self.process_prog_duty_cycle(inst)

        return cmds

    def process_reference_clk(self, inst: Instance) -> List[ISFAbstractCommand]:
        cmds = []

        ref_clk_src = None
        ref_clk_conns: List[Tuple[str, Connection]] = []
        CLK_SOURCE_PORTS = [f'CLKIN[{i}]' for i in range(4)]
        for port_name in CLK_SOURCE_PORTS:
            inst_port = inst.get_port(port_name)
            net = inst.get_connected_net(inst_port)
            if net is None:
                continue

            driver = net.get_source()
            if driver is None:
                continue

            driver_inst, driver_port = driver
            if driver_inst is None:
                continue

            if get_module_name(driver_inst.view) in ('EFX_IBUF', 'EFX_GPIO_V1', 'EFX_GPIO_V2', 'EFX_GPIO_V3'):
                ref_clk_src = 'EXTERNAL'
                ref_clk_conns.append((ref_clk_src, driver))
            elif is_user_instance(driver_inst) and not driver_port.name.startswith("gnd"):
                ref_clk_src = 'CORE'
                ref_clk_conns.append((ref_clk_src, driver))

        pt_assert(ref_clk_src is not None, f'Unable to determine reference clock source for {inst}, Most likely caused by unexpected connection', ValueError)
        if len(ref_clk_conns) > 1:
            cmds += self.build_prop_cmds(inst.name, 'REFCLK_SOURCE', 'DYNAMIC', lock=True)
        else:
            cmds += self.build_prop_cmds(inst.name, 'REFCLK_SOURCE', ref_clk_src, lock=True)

        core_clk_cnt = 0
        for tup in filter(lambda t: t[0] == 'CORE', ref_clk_conns):
            _, driver = tup
            if core_clk_cnt == 0:
                cmds += self.build_prop_cmds(inst.name, 'CORE_CLK_PIN', driver[1].name, lock=True)
            elif core_clk_cnt == 1:
                cmds += self.build_prop_cmds(inst.name, 'CORE_CLK1_PIN', driver[1].name, lock=True)
            else:
                raise ValueError(f'Unsupported more than two core ref clocks')
            core_clk_cnt += 1

        # Find REFCLK_SOURCE from connection
        CLK_SOURCE_CTRL_PORTS = {f'CLKSEL[{i}]' for i in range(2)}
        for port_name in CLK_SOURCE_CTRL_PORTS:
            inst_port = inst.get_port(port_name)
            net = inst.get_connected_net(inst_port)
            if net is None:
                continue

            driver = net.get_source()
            if driver is None:
                continue

            driver_inst, driver_port = driver
            if driver_inst is None:
                continue

            if is_user_instance(driver_inst) and not driver_port.name.startswith('gnd'):
                bus = driver_port.bus
                if bus is not None:
                    cmds += self.build_prop_cmds(inst.name, 'DYN_CLK_SEL_PIN', bus.name, lock=True)
        return cmds

    def process_feedback_clk(self, inst: Instance) -> List[ISFAbstractCommand]:
        net = inst.get_connected_net(inst.get_port('FBK'))

        view = inst.view
        name = get_module_name(view)
        fbk_clk = view.get_param("FEEDBACK_CLK")
        fbk_mode = view.get_param("FEEDBACK_MODE")

        pt_assert(fbk_clk is not None, "FEEDBACK_CLK parameter missing from RTL", ValueError)
        pt_assert(fbk_mode is not None, "FEEDBACK_MODE parameter missing from RTL", ValueError)

        fbk_clk = str(fbk_clk.resolve_value())
        fbk_mode = str(fbk_mode.resolve_value())

        cmds = []

        if fbk_mode == '"INTERNAL"':
            pt_assert(fbk_clk == '"INTERNAL"', 'FEEDBACK_CLK parameter should be "INTERNAL" for internal feedback mode', ValueError)

            cmds += self.build_prop_cmds(inst.name, "FEEDBACK_MODE", "INTERNAL", lock=True)
        elif fbk_mode == '"CORE"':
            is_valid_conn = False
            for i in range(3 if name == 'EFX_PLL_V2' else 5):
                clkout_port = inst.get_port(f'CLKOUT{i}')
                if clkout_port is None:
                    continue
                if net.is_connected_to((inst, clkout_port)):
                    is_valid_conn = True
                    break

            pt_assert(is_valid_conn, "Unexpected FBK port connection", ValueError)
            cmds += self.build_prop_cmds(inst.name, 'FEEDBACK_MODE', 'CORE', lock=True)
            cmds += self.build_prop_cmds(inst.name, 'FEEDBACK_CLK', remove_double_quote(fbk_clk), lock=True)
        elif fbk_mode == '"LOCAL"':
            is_valid_conn = False

            clkout_indexs = set()
            if name == 'EFX_PLL_V2' or name == 'EFX_PLL_V3':
                clkout_indexs.add(0)
            elif name == 'EFX_FPLL_V1':
                clkout_indexs.add(0)
                clkout_indexs.add(1)

            for i in clkout_indexs:
                clkout_port = inst.get_port(f'CLKOUT{i}')
                if clkout_port is None:
                    continue
                if net.is_connected_to((inst, clkout_port)):
                    is_valid_conn = True

            pt_assert(is_valid_conn, "Unexpected FBK port connection", ValueError)
            cmds += self.build_prop_cmds(inst.name, 'FEEDBACK_MODE', 'LOCAL', lock=True)
            cmds += self.build_prop_cmds(inst.name, 'FEEDBACK_CLK', remove_double_quote(fbk_clk), lock=True)

        return cmds

    def process_dynamic_phase_shift(self, inst: Instance) -> List[ISFAbstractCommand]:
        name = get_module_name(inst.view)
        pt_assert(name != "EFX_PLL_V2" and name != "EFX_PLL_V1", "Dynamic phase shift feature not supported in EFX_PLL_V2 and EFX_PLL_V1", ValueError)
        cmds = []

        shift_ena_net = inst.get_connected_net(inst.get_port('SHIFT_ENA'))
        shift_nets = [inst.get_connected_net(inst.get_port(f'SHIFT[{i}]')) for i in range(3)]
        shift_sel_nets = [inst.get_connected_net(inst.get_port(f'SHIFT_SEL[{i}]')) for i in range(5)]
        is_all_ports_connected = all([n is not None and not is_constant_net(n) for n in chain([shift_ena_net], shift_nets, shift_sel_nets)])
        is_incomplete_conn = any([n is not None and not is_constant_net(n) for n in chain([shift_ena_net], shift_nets, shift_sel_nets)])
        has_param_en = any([inst.view.get_param(f'CLKOUT{i}_DYNPHASE_EN').resolve_value() is True for i in range(5)])

        if has_param_en:
            if not is_all_ports_connected and is_incomplete_conn:
                raise ValueError(f'SHIFT / SHIFT_ENA / SHIFT_SEL ports need to be connected for dynamic phase shift feature')
        else:
            if is_all_ports_connected:
                raise ValueError('Dynamic Phase shift feature requires any of the CLKOUT<N>_DYNPHASE_EN parameters set to 1.')

        if has_param_en and is_all_ports_connected:
            shift_ena_name = get_connected_user_interface_name(self.ndb, shift_ena_net)
            shift_bus_name = get_connected_user_bus_interface_name(self.ndb, shift_nets)
            shift_sel_bus_name = get_connected_user_bus_interface_name(self.ndb, shift_sel_nets)
            cmds += self.build_prop_cmds(inst.name, 'PHASE_SHIFT_ENA_PIN', shift_ena_name, lock=True)
            cmds += self.build_prop_cmds(inst.name, 'PHASE_SHIFT_PIN', shift_bus_name, lock=True)
            cmds += self.build_prop_cmds(inst.name, 'PHASE_SHIFT_SEL_PIN', shift_sel_bus_name, lock=True)

            for i in range(5):
                clkout_net = inst.get_connected_net(inst.get_port(f'CLKOUT{i}'))
                if clkout_net is None:
                    continue
                dyn_phase_en = inst.view.get_param(f'CLKOUT{i}_DYNPHASE_EN').resolve_value()
                pt_assert(isinstance(dyn_phase_en, bool), f'CLKOUT{i}_DYNPHASE_EN must be a boolean', ValueError)
                cmds += self.build_prop_cmds(inst.name, f'CLKOUT{i}_DYNPHASE_EN', '1' if dyn_phase_en is True else '0', lock=True)

        return cmds

    def process_dynamic_cfg(self, inst: Instance) -> List[ISFAbstractCommand]:
        name = get_module_name(inst.view)
        pt_assert(name == "EFX_FPLL_V1", "Dynamic configuration feature only supported in EFX_FPLL_V1", ValueError)

        cmds = []
        dyn_cfg_en = inst.view.get_param('DYNAMIC_CFG_EN').resolve_value()
        pt_assert(isinstance(dyn_cfg_en, bool), f'DYNAMIC_CFG_EN parameter value must be boolean', ValueError)

        cmds += self.build_prop_cmds(inst.name, 'DYNAMIC_CFG_EN', '1' if dyn_cfg_en else '0', lock=True)

        for port_name in {'CFG_CLK', 'CFG_DATA_IN', 'CFG_DATA_OUT', 'CFG_SEL'}:
            net = inst.get_connected_net(inst.get_port(port_name))
            prop_name = f'{port_name}_PIN'
            prop_value = ''
            if net is not None:
                prop_value = get_connected_user_interface_name(self.ndb, net)

            cmds += self.build_prop_cmds(inst.name, prop_name, prop_value, lock=True)

        return cmds

    def process_fractional_mode(self, inst: Instance) -> List[ISFAbstractCommand]:
        name = get_module_name(inst.view)
        pt_assert(name == 'EFX_FPLL_V1', 'Fractional Mode feature only supported in EFX_FPLL_V1', ValueError)
        cmds = []

        frac_en = inst.view.get_param(f'CLKOUT1_FRAC_EN').resolve_value()
        pt_assert(isinstance(frac_en, bool), 'CLKOUT1_FRAC_EN must be a boolean', ValueError)

        cmds += self.build_prop_cmds(inst.name, 'FRACTIONAL_MODE_EN', '1' if frac_en else '0', lock=True)
        cmds += self.build_prop_cmds(inst.name, 'FRACTIONAL_COEFFICIENT', str(inst.view.get_param('CLKOUT1_FRAC_K').resolve_value()), lock=False)
        return cmds

    def process_ssc_mode(self, inst: Instance) -> List[ISFAbstractCommand]:
        name = get_module_name(inst.view)
        pt_assert(name == 'EFX_FPLL_V1', 'SSC mode only supported for EFX_FPLL_V1', ValueError)

        cmds = []
        ssc_mode = inst.view.get_param('SSC_MODE').resolve_value()
        pt_assert(isinstance(ssc_mode, str), 'SSC_MODE must be a string', ValueError)

        ssc_en_net = inst.get_connected_net(inst.get_port('USER_SSC_EN'))
        user_ssc_en_pin_value = get_connected_user_interface_name(self.ndb, ssc_en_net)
        cmds += self.build_prop_cmds(inst.name, 'USER_SSC_EN_PIN', user_ssc_en_pin_value, lock=True)
        cmds += self.build_prop_cmds(inst.name, 'SSC_MODE', remove_double_quote(ssc_mode), lock=True)
        cmds += self.build_prop_cmds(inst.name, 'SSC_FREQUENCY', str(inst.view.get_param('SSC_FREQUENCY').resolve_value()), lock=False)
        cmds += self.build_prop_cmds(inst.name, 'SSC_AMPLITUDE', str(inst.view.get_param('SSC_AMPLITUDE').resolve_value()), lock=False)
        cmds += self.build_prop_cmds(inst.name, 'SSC_MODULATION_TYPE', remove_double_quote(inst.view.get_param('SSC_MODULATION_TYPE').resolve_value()), lock=False)
        return cmds

    def process_prog_duty_cycle(self, inst: Instance) -> List[ISFAbstractCommand]:
        cmds = []
        cmds += self.build_prop_cmds(inst.name, 'CLKOUT1_DUTY_CYCLE', str(inst.view.get_param('CLKOUT1_DUTY_CYCLE').resolve_value()), lock=False)
        cmds += self.build_prop_cmds(inst.name, 'CLKOUT1_PROG_DUTY_CYCLE_EN', '1' if inst.view.get_param('CLKOUT1_PROG_DUTY_CYCLE_EN').resolve_value() is True else '0', lock=False)
        cmds += self.build_prop_cmds(inst.name, 'CLKOUT1_DC_ODD', '1' if inst.view.get_param('CLKOUT1_DC_ODD').resolve_value() is True else '0', lock=False)
        return cmds

    def process_block(self, inst: Instance):
        create_commands = []
        set_prop_commands = []

        view = inst.view
        name = get_module_name(view)
        peri_view = get_peri_netlist(name)
        if peri_view is not None and name in ("EFX_PLL_V1", "EFX_PLL_V2", "EFX_PLL_V3", "EFX_FPLL_V1"):
            peri_view: Netlist = peri_view()
        else:
            return

        design_api_handle = "design"
        create_commands += [
            ISFCreateBlockCommand(design_api_handle, inst.name, "PLL")
        ]

        def build_set_prop_cmd(prop_name, prop_value):
            return ISFSetBlockPropertyCommand(design_api_handle, inst.name, "PLL", str(prop_name), str(prop_value))

        set_prop_commands += self.process_simple_port(inst, 'RSTN', 'RSTN_PIN')
        set_prop_commands += self.process_simple_port(inst, 'LOCKED', 'LOCKED_PIN')

        for i in range(5 if name in {'EFX_PLL_V3', 'EFX_FPLL_V1'} else 3):
            set_prop_commands += self.process_clkout_port(inst, i)

        if name == "EFX_PLL_V3" or name == "EFX_FPLL_V1":
            set_prop_commands += self.process_dynamic_phase_shift(inst)

        if name == 'EFX_FPLL_V1':
            set_prop_commands += self.process_dynamic_cfg(inst)
            set_prop_commands += self.process_fractional_mode(inst)
            set_prop_commands += self.process_ssc_mode(inst)

        special_cmds = []
        if name != 'EFX_PLL_V1':
            special_cmds += self.process_feedback_clk(inst)
            special_cmds += self.process_reference_clk(inst)

        set_prop_commands += self.build_prop_cmds(inst.name, 'M', str(inst.view.get_param('M').resolve_value()), lock=False)
        set_prop_commands += self.build_prop_cmds(inst.name, 'N', str(inst.view.get_param('N').resolve_value()), lock=False)
        set_prop_commands += self.build_prop_cmds(inst.name, 'O', str(inst.view.get_param('O').resolve_value()), lock=False)
        set_prop_commands += self.build_prop_cmds(inst.name, 'REFCLK_FREQ', str(inst.view.get_param('REFCLK_FREQ').resolve_value()), lock=False)

        if special_cmds:
            set_prop_commands += special_cmds

        return create_commands, set_prop_commands


class GPIOISFConverter(AbstractISFConverter):

    @property
    def block_type(self) -> str:
        return "GPIO"

    def process_bus_blocks(self, bus_info: NetBus, instances: List[Instance]):
        # INSTANCES MUST HAVE SAME PARAMETERS VALUE
        create_commands = []
        set_prop_commands = []

        assert len(instances) > 0
        view = instances[0].view
        assert all(i.view == view for i in instances)
        name = get_module_name(view)

        design_api_handle = "design"
        if name in ("EFX_IBUF", "EFX_IDDIO", "EFX_IREG"):
            cmd = ISFCreateInputGPIOCommand(design_api_handle,
                                            block_name=bus_info.name,
                                            block_type="GPIO_BUS",
                                            msb=bus_info.msb,
                                            lsb=bus_info.lsb)
        elif name in ("EFX_OBUF", "EFX_ODDIO", "EFX_OREG"):
            cmd = ISFCreateOutputGPIOCommand(design_api_handle,
                                             block_name=bus_info.name,
                                             block_type="GPIO_BUS",
                                             msb=bus_info.msb,
                                             lsb=bus_info.lsb)
        elif name in ("EFX_IO_BUF", "EFX_IOREG"):
            cmd = ISFCreateInoutGPIOCommand(design_api_handle,
                                            block_name=bus_info.name,
                                            block_type="GPIO_BUS",
                                            msb=bus_info.msb,
                                            lsb=bus_info.lsb)
        elif name == "EFX_CLKOUT":
            cmd = ISFCreateClkoutGPIOCommand(design_api_handle,
                                             block_name=bus_info.name,
                                             block_type="GPIO_BUS",
                                             msb=bus_info.msb,
                                             lsb=bus_info.lsb)
        else:
            return

        create_commands += [
            cmd
        ]

        for inst in instances:
            # Use the main net name as the isf instance name
            owner = inst.owner
            net = None
            for conn in inst.connections:
                net = conn.net
                if net.bus == bus_info:
                    break

            assert net is not None
            _, prop_cmds = self.process_block(inst, net.name)
            set_prop_commands += prop_cmds

        return create_commands, set_prop_commands

    def build_lock_cmd(self, cmd: ISFSetBlockPropertyCommand) -> ISFLockBlockPropertyCommand:
        return ISFLockBlockPropertyCommand(cmd.design_api_handle, cmd.block_name, cmd.block_type, cmd.prop_name, cmd.prop_value, self.partial_design_name)

    def build_prop_conn_cmds(self, isf_instance_id: str, prop_name: str, net: Net, lock: bool) -> List[ISFAbstractCommand]:
        name = get_connected_user_interface_name(self.ndb, net)
        cmd = ISFSetBlockPropertyCommand(self.design_api_handle, isf_instance_id, 'GPIO', prop_name, name)
        if lock:
            lock_cmd = self.build_lock_cmd(cmd)
            return [cmd, lock_cmd]
        return [cmd]

    def build_simple_prop_cmds(self, isf_instance_id: str, prop_name: str, prop_value: str, lock: bool) -> List[ISFAbstractCommand]:
        cmd = ISFSetBlockPropertyCommand(self.design_api_handle, isf_instance_id, 'GPIO', prop_name, prop_value)
        if lock:
            lock_cmd = self.build_lock_cmd(cmd)
            return [cmd, lock_cmd]
        return [cmd]

    def process_input_buffer(self, inst: Instance, isf_instance_id: str, port_name: str) -> List[ISFAbstractCommand]:
        i_net = inst.get_connected_net(inst.get_port(port_name))
        pt_assert(i_net is not None, f'{port_name} port must be connected to core', ValueError)
        return self.build_prop_conn_cmds(isf_instance_id, 'IN_PIN', i_net, lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'IN_REG', 'BYPASS', lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'PULL_OPTION', remove_double_quote(inst.view.get_param('PULL_OPTION').resolve_value()), lock=False)

    def process_alt_connection(self, inst: Instance, isf_instance_id: str, port_name: str):
        net = inst.get_connected_net(inst.get_port(port_name))
        cmds = []
        for conn in net.connections:
            if conn.instance == inst:
                continue
            if conn.instance == None:
                continue

            conn_inst = conn.instance
            if get_module_name(conn_inst.view) in {'EFX_PLL_V1', 'EFX_PLL_V2', 'EFX_PLL_V3', 'EFX_FPLL_V1'} and \
                    conn.port.name.startswith('CLKIN'):
                cmds += self.build_simple_prop_cmds(isf_instance_id, 'CONN_TYPE', 'PLL_CLKIN', lock=True)
                break
            else:
                gclk_attr = conn.port.get_attr('global_clock')
                if gclk_attr is not None and gclk_attr.value == 'TRUE':
                    cmds += self.build_simple_prop_cmds(isf_instance_id, 'CONN_TYPE', 'GCLK', lock=True)
                    break

        if len(cmds) == 0:
            cmds += self.build_simple_prop_cmds(isf_instance_id, 'CONN_TYPE', 'NONE', lock=False)
        return cmds

    def process_input_register(self, inst: Instance, isf_instance_id: str, port_name: str, clk_port_name: str, clk_invert_param_name: str) -> List[ISFAbstractCommand]:
        i_net = inst.get_connected_net(inst.get_port(port_name))
        pt_assert(i_net is not None, f'{port_name} port must be connected to core', ValueError)
        clk_net = inst.get_connected_net(inst.get_port(clk_port_name))
        pt_assert(clk_net is not None, f'{clk_port_name} port must be connected', ValueError)
        return self.build_simple_prop_cmds(isf_instance_id, 'IN_REG', 'REG', lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'IN_PIN', i_net, lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'IN_CLK_PIN', clk_net, lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'IS_INCLK_INVERTED', '1' if inst.view.get_param(clk_invert_param_name).resolve_value() else '0', lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'PULL_OPTION', remove_double_quote(inst.view.get_param('PULL_OPTION').resolve_value()), lock=False)

    def process_ddio_input(self, inst: Instance, isf_instance_id: str, hi_port_name: str, lo_port_name: str, clk_port_name: str, mode_param_name: str, clk_invert_param_name: str) -> List[ISFAbstractCommand]:
        i0_net = inst.get_connected_net(inst.get_port(hi_port_name))
        i1_net = inst.get_connected_net(inst.get_port(lo_port_name))
        pt_assert(i0_net is not None, f'{hi_port_name} port must be connected to core', ValueError)
        pt_assert(i1_net is not None, f'{lo_port_name} port must be connected to core', ValueError)
        clk_net = inst.get_connected_net(inst.get_port(clk_port_name))
        pt_assert(clk_net is not None, f'Clock port {clk_port_name} must be connected', ValueError)
        return self.build_simple_prop_cmds(isf_instance_id, 'IN_REG',  remove_double_quote(inst.view.get_param(mode_param_name).resolve_value()), lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'IN_HI_PIN', i0_net, lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'IN_LO_PIN',  i1_net, lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'IN_CLK_PIN', clk_net, lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'IS_INCLK_INVERTED', '1' if inst.view.get_param(clk_invert_param_name).resolve_value() else '0', lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'PULL_OPTION', remove_double_quote(inst.view.get_param('PULL_OPTION').resolve_value()), lock=False)

    def process_serial_input(self, inst: Instance, isf_instance_id: str, port_bus_name: str, slow_clk_port_name: str, fast_clk_port_name: str, clk_invert_param_name: str) -> List[ISFAbstractCommand]:
        i_nets = [inst.get_connected_net(inst.get_port(f'{port_bus_name}[{i}]')) for i in range(4)]
        ibus_name = get_connected_user_bus_interface_name(self.ndb, i_nets)
        clk_net = inst.get_connected_net(inst.get_port(slow_clk_port_name))
        pt_assert(clk_net is not None, f'Clock port {slow_clk_port_name} must be connected', ValueError)
        fastclk_net = inst.get_connected_net(inst.get_port(fast_clk_port_name))
        pt_assert(fastclk_net is not None, f'Fast clock port {fast_clk_port_name} must be connected', ValueError)
        return self.build_simple_prop_cmds(isf_instance_id, 'IN_REG', 'SERIAL', lock=True) + \
                self.build_simple_prop_cmds(isf_instance_id, 'IN_PIN', ibus_name, lock=True) + \
                self.build_prop_conn_cmds(isf_instance_id, 'IN_CLK_PIN', clk_net, lock=True) + \
                self.build_prop_conn_cmds(isf_instance_id, 'INFASTCLK_PIN', fastclk_net, lock=True) + \
                self.build_simple_prop_cmds(isf_instance_id, 'IS_INCLK_INVERTED', '1' if inst.view.get_param(clk_invert_param_name).resolve_value() else '0', lock=True) + \
                self.build_simple_prop_cmds(isf_instance_id, 'PULL_OPTION', remove_double_quote(inst.view.get_param('PULL_OPTION').resolve_value()), lock=False)

    def process_output_buffer(self, inst: Instance, isf_instance_id: str, port_name: str) -> List[ISFAbstractCommand]:
        o_out = inst.get_connected_net(inst.get_port(port_name))
        pt_assert(o_out is not None, f'{port_name} port must be connected to core', ValueError)
        return self.build_prop_conn_cmds(isf_instance_id, 'OUT_PIN', o_out, lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'OUT_REG', 'BYPASS', lock=True)

    def format_output_register_cmds(self, isf_instance_id: str, o_net: Net, clk_net: Net, out_reg_value: str, clk_inverted_value: bool) -> List[ISFAbstractCommand]:
        return self.build_simple_prop_cmds(isf_instance_id, 'OUT_REG', out_reg_value, lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'OUT_PIN', o_net, lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'OUT_CLK_PIN', clk_net, lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'IS_OUTCLK_INVERTED',  '1' if clk_inverted_value is True else '0', lock=True)

    def process_output_register(self, inst: Instance, isf_instance_id: str, port_name: str, clk_port_name: str, mode_param_name: str, clk_invert_param_name: str) -> List[ISFAbstractCommand]:
        o_net = inst.get_connected_net(inst.get_port(port_name))
        pt_assert(o_net is not None, f'{port_name} port must be connected to core', ValueError)
        clk_net = inst.get_connected_net(inst.get_port(clk_port_name))
        pt_assert(clk_net is not None, f'{clk_port_name} port must be connected', ValueError)
        out_reg = remove_double_quote(inst.view.get_param(mode_param_name).resolve_value())
        pt_assert(isinstance(out_reg, str), f'{mode_param_name} parameter value must be a string', ValueError)
        clk_inverted = inst.view.get_param(clk_invert_param_name).resolve_value()
        pt_assert(isinstance(clk_inverted, bool), f'{clk_invert_param_name} parameter value must be boolean', ValueError)
        return self.format_output_register_cmds(isf_instance_id=isf_instance_id,
                                                o_net=o_net,
                                                clk_net=clk_net,
                                                out_reg_value=out_reg,
                                                clk_inverted_value=clk_inverted)

    def process_ddio_output(self, inst: Instance, isf_instance_id: str, hi_port_name: str, lo_port_name: str, clk_port_name: str, mode_param_name: str, clk_invert_param_name: str) -> List[ISFAbstractCommand]:
        o0_net = inst.get_connected_net(inst.get_port(hi_port_name))
        o1_net = inst.get_connected_net(inst.get_port(lo_port_name))
        pt_assert(o0_net is not None, f'{hi_port_name} port must be connected to core', ValueError)
        pt_assert(o1_net is not None, f'{lo_port_name} port must be connected to core', ValueError)
        clk_net = inst.get_connected_net(inst.get_port(clk_port_name))
        pt_assert(clk_net is not None, f'Clock port {clk_port_name} must be connected', ValueError)
        return self.build_simple_prop_cmds(isf_instance_id, 'OUT_REG',  remove_double_quote(inst.view.get_param(mode_param_name).resolve_value()), lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'OUT_HI_PIN', o0_net, lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'OUT_LO_PIN',  o1_net, lock=True) + \
               self.build_prop_conn_cmds(isf_instance_id, 'OUT_CLK_PIN', clk_net, lock=True) + \
               self.build_simple_prop_cmds(isf_instance_id, 'IS_OUTCLK_INVERTED', '1' if inst.view.get_param(clk_invert_param_name).resolve_value() else '0', lock=True)

    def process_serial_output(self, inst: Instance, isf_instance_id: str, port_bus_name: str, slow_clk_port_name: str, fast_clk_port_name: str, clk_invert_param_name: str) -> List[ISFAbstractCommand]:
        o_nets = [inst.get_connected_net(inst.get_port(f'{port_bus_name}[{i}]')) for i in range(4)]
        obus_name = get_connected_user_bus_interface_name(self.ndb, o_nets)
        clk_net = inst.get_connected_net(inst.get_port(slow_clk_port_name))
        pt_assert(clk_net is not None, f'Clock port {slow_clk_port_name} must be connected', ValueError)
        fastclk_net = inst.get_connected_net(inst.get_port(fast_clk_port_name))
        pt_assert(fastclk_net is not None, f'Fast clock port {fast_clk_port_name} must be connected', ValueError)
        return  self.build_simple_prop_cmds(isf_instance_id, 'OUT_REG', 'SERIAL', lock=True) + \
                self.build_simple_prop_cmds(isf_instance_id, 'OUT_PIN', obus_name, lock=True) + \
                self.build_prop_conn_cmds(isf_instance_id, 'OUT_CLK_PIN', clk_net, lock=True) + \
                self.build_prop_conn_cmds(isf_instance_id, 'OUTFASTCLK_PIN', fastclk_net, lock=True) + \
                self.build_simple_prop_cmds(isf_instance_id, 'IS_OUTCLK_INVERTED', '1' if inst.view.get_param(clk_invert_param_name).resolve_value() else '0', lock=True)

    def process_oe(self, inst: Instance, isf_instance_id: str, port_name: str) -> List[ISFAbstractCommand]:
        oe_net = inst.get_connected_net(inst.get_port(port_name))
        return self.build_prop_conn_cmds(isf_instance_id, 'OE_PIN', oe_net, lock=True)

    def process_clkout(self, inst: Instance, isf_instance_id: str, port_name: str, clk_invert_param_name: str) -> List[ISFAbstractCommand]:
        clkout_net = inst.get_connected_net(inst.get_port(port_name))
        pt_assert(clkout_net is not None, f'{port_name} port must be connected', ValueError)
        return self.build_prop_conn_cmds(isf_instance_id, 'OUT_CLK_PIN', clkout_net, lock=True) + \
                self.build_simple_prop_cmds(isf_instance_id, 'IS_OUTCLK_INVERTED', '1' if inst.view.get_param(clk_invert_param_name).resolve_value() else '0', lock=True)

    def process_input_gpio_v1(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateInputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))

        in_reg = inst.view.get_param('IN_REG').resolve_value()
        pt_assert(isinstance(in_reg, str), f'IN_REG must be string', ValueError)

        inclk_net = inst.get_connected_net(inst.get_port('INCLK'))
        if in_reg != '"BYPASS"' and inclk_net is None:
            raise ValueError('IN_REG must be BYPASS when INCLK is not connected')

        if inclk_net is not None and not is_constant_net(inclk_net):
            set_cmds += self.process_input_register(inst, isf_instance_id, 'I', 'INCLK', 'IS_INCLK_INVERTED')
        else:
            set_cmds += self.process_input_buffer(inst, isf_instance_id, 'I')
            set_cmds += self.process_alt_connection(inst, isf_instance_id, 'I')

        return create_cmds, set_cmds

    def process_input_gpio_v2(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateInputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))

        in_reg = inst.view.get_param('IN_REG').resolve_value()
        pt_assert(isinstance(in_reg, str), f'IN_REG must be string', ValueError)

        inclk_net = inst.get_connected_net(inst.get_port('INCLK'))
        if in_reg != '"BYPASS"' and inclk_net is None:
            raise ValueError('IN_REG must be BYPASS when INCLK is not connected')

        if inclk_net is not None and not is_constant_net(inclk_net):
            if in_reg == '"REG"':
                set_cmds += self.process_input_register(inst, isf_instance_id, 'I[0]', 'INCLK', 'IS_INCLK_INVERTED')
            elif in_reg == '"DDIO"' or in_reg == '"DDIO_RESYNC"':
                set_cmds += self.process_ddio_input(inst, isf_instance_id, "I[0]", "I[1]", "INCLK", "IN_REG", "IS_INCLK_INVERTED")
            else:
                raise ValueError(f'Unexpected IN_REG value {in_reg} when INCLK is connected to {inclk_net}')
        else:
            set_cmds += self.process_input_buffer(inst, isf_instance_id, 'I[0]')
            set_cmds += self.process_alt_connection(inst, isf_instance_id, 'I[0]')

        return create_cmds, set_cmds

    def process_input_gpio_v3(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateInputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))

        in_reg = inst.view.get_param('IN_REG').resolve_value()
        pt_assert(isinstance(in_reg, str), f'IN_REG must be string', ValueError)

        inclk_net = inst.get_connected_net(inst.get_port('INCLK'))
        if in_reg != '"BYPASS"' and inclk_net is None:
            raise ValueError('IN_REG must be BYPASS when INCLK is not connected')

        if inclk_net is not None and not is_constant_net(inclk_net):
            if in_reg == '"REG"':
                set_cmds += self.process_input_register(inst, isf_instance_id, 'I[0]', 'INCLK', 'IS_INCLK_INVERTED')
            elif in_reg == '"DDIO"' or in_reg == '"DDIO_RESYNC"':
                set_cmds += self.process_ddio_input(inst, isf_instance_id, "I[0]", "I[1]", "INCLK", "IN_REG", "IS_INCLK_INVERTED")
            elif in_reg == '"SERIAL"':
                set_cmds += self.process_serial_input(inst, isf_instance_id, "I", "INCLK", "INFASTCLK", "IS_INCLK_INVERTED")
            else:
                raise ValueError(f'Unexpected IN_REG value {in_reg} when INCLK is connected to {inclk_net}')
        else:
            set_cmds += self.process_input_buffer(inst, isf_instance_id, 'I[0]')
            set_cmds += self.process_alt_connection(inst, isf_instance_id, 'I[0]')

        return create_cmds, set_cmds

    def process_output_gpio_v1(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateOutputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))

        o_net: Net | None = inst.get_connected_net(inst.get_port('O'))
        pt_assert(o_net is not None, f'I port must be connected to core', ValueError)
        set_cmds += self.build_prop_conn_cmds(isf_instance_id, 'OUT_PIN', o_net, lock=True)

        out_reg = inst.view.get_param('OUT_REG').resolve_value()
        pt_assert(isinstance(out_reg, str), f'OUT_REG must be string', ValueError)
        set_cmds += self.build_simple_prop_cmds(isf_instance_id, 'OUT_REG', remove_double_quote(out_reg), lock=True)

        outclk_net = inst.get_connected_net(inst.get_port('OUTCLK'))
        if out_reg != '"BYPASS"' and outclk_net is None:
            raise ValueError('OUT_REG must be BYPASS when OUTCLK is not connected')

        if outclk_net is not None and not is_constant_net(outclk_net):
            if out_reg == '"REG"' or out_reg == '"INVREG"':
                set_cmds += self.process_output_register(inst, isf_instance_id, "O", "OUTCLK", "OUT_REG", "IS_OUTCLK_INVERTED")
            else:
                raise ValueError(f'Unexpected OUT_REG value {out_reg} when OUTCLK is connected to {outclk_net}')
        else:
            set_cmds += self.process_output_buffer(inst, isf_instance_id, "O")

        return create_cmds, set_cmds

    def process_output_gpio_v2(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateOutputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))

        out_reg = inst.view.get_param('OUT_REG').resolve_value()
        pt_assert(isinstance(out_reg, str), f'OUT_REG must be string', ValueError)

        outclk_net = inst.get_connected_net(inst.get_port('OUTCLK'))
        if out_reg != '"BYPASS"' and outclk_net is None:
            raise ValueError('OUT_REG must be BYPASS when OUTCLK is not connected')

        if outclk_net is not None and not is_constant_net(outclk_net):
            if out_reg == '"REG"' or out_reg == '"INVREG"':
                set_cmds += self.process_output_register(inst, isf_instance_id, "O[0]", "OUTCLK", "OUT_REG", "IS_OUTCLK_INVERTED")
            elif out_reg == '"DDIO"' or out_reg == '"DDIO_RESYNC"':
                set_cmds += self.process_ddio_output(inst, isf_instance_id, 'O[0]', 'O[1]', 'OUTCLK', 'OUT_REG', 'IS_OUTCLK_INVERTED')
            else:
                raise ValueError(f'Unexpected OUT_REG value {out_reg} when OUTCLK is connected to {outclk_net}')
        else:
            set_cmds += self.process_output_buffer(inst, isf_instance_id, 'O[0]')

        return create_cmds, set_cmds

    def process_output_gpio_v3(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateOutputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))

        out_reg = inst.view.get_param('OUT_REG').resolve_value()
        pt_assert(isinstance(out_reg, str), f'OUT_REG must be string', ValueError)

        outclk_net = inst.get_connected_net(inst.get_port('OUTCLK'))
        if out_reg != '"BYPASS"' and outclk_net is None:
            raise ValueError('OUT_REG must be BYPASS when OUTCLK is not connected')

        if outclk_net is not None and not is_constant_net(outclk_net):
            if out_reg == '"REG"' or out_reg == '"INVREG"':
                set_cmds += self.process_output_register(inst, isf_instance_id, "O[0]", "OUTCLK", "OUT_REG", "IS_OUTCLK_INVERTED")
            elif out_reg == '"DDIO"' or out_reg == '"DDIO_RESYNC"':
                set_cmds += self.process_ddio_output(inst, isf_instance_id, 'O[0]', 'O[1]', 'OUTCLK', 'OUT_REG', 'IS_OUTCLK_INVERTED')
            elif out_reg == '"SERIAL"':
                set_cmds += self.process_serial_output(inst, isf_instance_id, 'O', 'OUTCLK', 'OUTFASTCLK', 'IS_OUTCLK_INVERTED')
            else:
                raise ValueError(f'Unexpected OUT_REG value {out_reg} when OUTCLK is connected to {outclk_net}')
        else:
            set_cmds += self.process_output_buffer(inst, isf_instance_id, 'O[0]')

        return create_cmds, set_cmds

    def process_inout_gpio_v1(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateInoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
        _, set_in_cmds = self.process_input_gpio_v1(inst, isf_instance_id)
        _, set_out_cmds = self.process_output_gpio_v1(inst, isf_instance_id)
        set_cmds += set_in_cmds + set_out_cmds
        set_cmds += self.process_oe(inst, isf_instance_id, 'OE')
        return create_cmds, set_cmds

    def process_inout_gpio_v2(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateInoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
        _, set_in_cmds = self.process_input_gpio_v2(inst, isf_instance_id)
        _, set_out_cmds = self.process_output_gpio_v2(inst, isf_instance_id)
        set_cmds += set_in_cmds + set_out_cmds
        set_cmds += self.process_oe(inst, isf_instance_id, 'OE')
        return create_cmds, set_cmds

    def process_inout_gpio_v3(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateInoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))

        _, set_in_cmds = self.process_input_gpio_v3(inst, isf_instance_id)
        set_cmds += set_in_cmds

        _, set_out_cmds = self.process_output_gpio_v3(inst, isf_instance_id)
        set_cmds += set_out_cmds

        set_cmds += self.process_oe(inst, isf_instance_id, 'OE')

        return create_cmds, set_cmds

    def process_clkout_gpio_v1(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateClkoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
        set_cmds += self.process_clkout(inst, isf_instance_id, 'OUTCLK', 'IS_OUTCLK_INVERTED')
        return create_cmds, set_cmds

    def process_clkout_gpio_v2(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateClkoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
        set_cmds += self.process_clkout(inst, isf_instance_id, 'OUTCLK', 'IS_OUTCLK_INVERTED')
        return create_cmds, set_cmds

    def process_clkout_gpio_v3(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List]:
        create_cmds = []
        set_cmds = []
        create_cmds.append(ISFCreateClkoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
        set_cmds += self.process_clkout(inst, isf_instance_id, 'OUTCLK', 'IS_OUTCLK_INVERTED')
        return create_cmds, set_cmds

    def process_gpio_v1(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List] | None:
        view = inst.view
        assert view is not None
        name = get_module_name(view)
        assert name == 'EFX_GPIO_V1'

        mode: str = view.get_param('MODE').resolve_value()
        if mode == '"INPUT"':
            return self.process_input_gpio_v1(inst, isf_instance_id)
        elif mode == '"OUTPUT"':
            return self.process_output_gpio_v1(inst, isf_instance_id)
        elif mode == '"INOUT"':
            return self.process_inout_gpio_v1(inst, isf_instance_id)
        elif mode == '"CLKOUT"':
            return self.process_clkout_gpio_v1(inst, isf_instance_id)
        else:
            raise ValueError(f'Unknown GPIO mode {mode}')

    def process_gpio_v2(self, inst: Instance, isf_instance_id: str) -> Tuple[List. List] | None:
        view = inst.view
        assert view is not None
        name = get_module_name(view)
        assert name == 'EFX_GPIO_V2'

        mode: str = view.get_param('MODE').resolve_value()
        if mode == '"INPUT"':
            return self.process_input_gpio_v2(inst, isf_instance_id)
        elif mode == '"OUTPUT"':
            return self.process_output_gpio_v2(inst, isf_instance_id)
        elif mode == '"INOUT"':
            return self.process_inout_gpio_v2(inst, isf_instance_id)
        elif mode == '"CLKOUT"':
            return self.process_clkout_gpio_v2(inst, isf_instance_id)
        else:
            raise ValueError(f'Unknown GPIO mode {mode}')

    def process_gpio_v3(self, inst: Instance, isf_instance_id: str) -> Tuple[List, List] | None:
        view = inst.view
        assert view is not None
        name = get_module_name(view)
        assert name == 'EFX_GPIO_V3'

        mode: str = view.get_param('MODE').resolve_value()
        if mode == '"INPUT"':
            return self.process_input_gpio_v3(inst, isf_instance_id)
        elif mode == '"OUTPUT"':
            return self.process_output_gpio_v3(inst, isf_instance_id)
        elif mode == '"INOUT"':
            return self.process_inout_gpio_v3(inst, isf_instance_id)
        elif mode == '"CLKOUT"':
            return self.process_clkout_gpio_v3(inst, isf_instance_id)
        else:
            raise ValueError(f'Unknown GPIO mode {mode}')

    def process_block(self, inst: Instance, name_overwrite: str = None):
        create_commands = []
        set_prop_commands = []

        view = inst.view
        name = get_module_name(view)

        isf_instance_id = inst.name
        if name_overwrite:
            isf_instance_id = name_overwrite
        elif isf_instance_id.endswith(f'~{name}'):
            isf_instance_id = isf_instance_id.removesuffix(f'~{name}')
        # Special handling for Inferred register
        elif name in {'EFX_IREG', 'EFX_OREG'} and isf_instance_id.endswith(f'~FF'):
            # FIXME: Synthesis would generate a register with name like "reg_name~FF"
            #        but at the same time there maybe another instance call reg_name~EFX_IBUF
            #        which will cause a name conflict.
            new_isf_instance_id = isf_instance_id.removesuffix(f'~FF')
            all_instance_names = set([next(iter(i.name.split('~')), i.name) for i in filter(lambda i: i != inst, inst.owner.instances)])
            if new_isf_instance_id not in all_instance_names:
                isf_instance_id = new_isf_instance_id

        if name == 'EFX_GPIO_V3':
            create_commands, set_prop_commands = self.process_gpio_v3(inst, isf_instance_id)
        elif name == 'EFX_GPIO_V2':
            create_commands, set_prop_commands = self.process_gpio_v2(inst, isf_instance_id)
        elif name == 'EFX_GPIO_V1':
            create_commands, set_prop_commands = self.process_gpio_v1(inst, isf_instance_id)
        elif name == 'EFX_IBUF':
            create_commands.append(ISFCreateInputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            set_prop_commands += self.process_input_buffer(inst, isf_instance_id, 'O')
            set_prop_commands += self.process_alt_connection(inst, isf_instance_id, 'O')
        elif name == 'EFX_IREG':
            create_commands.append(ISFCreateInputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            set_prop_commands += self.process_input_register(inst, isf_instance_id, 'O', 'CLK', 'IS_CLK_INVERTED')
        elif name == 'EFX_IDDIO':
            create_commands.append(ISFCreateInputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            set_prop_commands += self.process_ddio_input(inst, isf_instance_id, 'O_HI', 'O_LO', 'CLK', 'MODE', 'IS_CLK_INVERTED')
        elif name == 'EFX_OBUF':
            create_commands.append(ISFCreateOutputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            set_prop_commands += self.process_output_buffer(inst, isf_instance_id, 'I')
        elif name == 'EFX_OREG':
            create_commands.append(ISFCreateOutputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            o_net = inst.get_connected_net(inst.get_port('I'))
            pt_assert(o_net is not None, 'I port must be connected to core', ValueError)
            clk_net = inst.get_connected_net(inst.get_port("CLK"))
            pt_assert(clk_net is not None, f'CLK port must be connected', ValueError)
            clk_inverted = inst.view.get_param("IS_CLK_INVERTED").resolve_value()
            set_prop_commands += self.format_output_register_cmds(isf_instance_id, o_net, clk_net, "REG", clk_inverted)
        elif name == 'EFX_ODDIO':
            create_commands.append(ISFCreateOutputGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            set_prop_commands += self.process_ddio_output(inst, isf_instance_id, 'I_HI', 'I_LO', 'CLK', 'MODE', 'IS_CLK_INVERTED')
        elif name == 'EFX_IO_BUF':
            create_commands.append(ISFCreateInoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            i_net = inst.get_connected_net(inst.get_port('O'))
            if i_net is not None:
                set_prop_commands += self.process_input_buffer(inst, isf_instance_id, 'O')
            else:
                set_prop_commands += self.build_simple_prop_cmds(isf_instance_id, 'IN_PIN', '', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'IN_REG', 'BYPASS', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'PULL_OPTION', remove_double_quote(inst.view.get_param('PULL_OPTION').resolve_value()), lock=False)

            o_net = inst.get_connected_net(inst.get_port('I'))
            if o_net is not None:
                set_prop_commands += self.process_output_buffer(inst, isf_instance_id, 'I')
            else:
                set_prop_commands += self.build_simple_prop_cmds(isf_instance_id, 'OUT_PIN', '', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'OUT_REG', 'BYPASS', lock=True)

            set_prop_commands += self.process_oe(inst, isf_instance_id, 'OE')
        elif name == 'EFX_CLKOUT':
            create_commands.append(ISFCreateClkoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            set_prop_commands += self.process_clkout(inst, isf_instance_id, 'CLK', 'IS_CLK_INVERTED')

        elif name == 'EFX_IOREG':
            create_commands.append(ISFCreateInoutGPIOCommand(self.design_api_handle, isf_instance_id, 'GPIO'))
            # Input Pin can be optional
            i_net = inst.get_connected_net(inst.get_port('O'))
            i_clk_net = inst.get_connected_net(inst.get_port('OUTCLK'))
            if i_net is not None and i_clk_net is not None and not is_constant_net(i_clk_net):
                set_prop_commands += self.process_input_register(inst, isf_instance_id, 'O', 'OUTCLK', 'IS_OUTCLK_INVERTED')
            else:
                set_prop_commands += self.build_simple_prop_cmds(isf_instance_id, 'IN_REG', 'REG', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'IN_PIN', '', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'IN_CLK_PIN', '', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'IS_INCLK_INVERTED', '1' if inst.view.get_param('IS_OUTCLK_INVERTED').resolve_value() else '0', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'PULL_OPTION', remove_double_quote(inst.view.get_param('PULL_OPTION').resolve_value()), lock=False)

            o_net = inst.get_connected_net(inst.get_port('I'))
            o_clk_net = inst.get_connected_net(inst.get_port("INCLK"))
            clk_inverted = inst.view.get_param("IS_INCLK_INVERTED").resolve_value()
            if o_net is not None and o_clk_net is not None and not is_constant_net(o_clk_net):
                set_prop_commands += self.format_output_register_cmds(isf_instance_id, o_net, o_clk_net, "REG", clk_inverted)
            else:
                set_prop_commands += self.build_simple_prop_cmds(isf_instance_id, 'OUT_REG', 'REG', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'OUT_PIN', '', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'OUT_CLK_PIN', '', lock=True) + \
                                        self.build_simple_prop_cmds(isf_instance_id, 'IS_OUTCLK_INVERTED',  '1' if clk_inverted else '0', lock=True)

            set_prop_commands += self.process_oe(inst, isf_instance_id, "OE")

        return create_commands, set_prop_commands


class OSCISFConverter(AbstractISFConverter):

    @property
    def block_type(self):
        return "OSC"

    def process_block(self, inst: Instance):
        create_commands = []
        set_prop_commands = []
        view = inst.view
        name = get_module_name(view)
        if name not in ("EFX_OSC_V1", "EFX_OSC_V2", "EFX_OSC_V3"):
            return

        design_api_handle = "design"
        create_commands += [
            ISFCreateBlockCommand(design_api_handle, inst.name, "OSC")
        ]

        def build_set_prop_cmd(prop_name, prop_value):
            return ISFSetBlockPropertyCommand(design_api_handle, inst.name, "OSC", prop_name, prop_value)

        for param in view.parameters:
            prop_name = param.name
            prop_value = param.value
            set_prop_commands += [
                build_set_prop_cmd(prop_name, str(prop_value))
            ]

        port_name_to_isf_prop_name = {
            'CLKOUT': 'CLKOUT_PIN',
            'ENA': 'ENA_PIN'
        }

        clkout_net = inst.get_connected_net(inst.get_port('CLKOUT'))
        pt_assert(clkout_net is not None, f'CLKOUT port must be connected', ValueError)

        for port in inst.view.ports:
            net = inst.get_connected_net(port)
            prop_name = port_name_to_isf_prop_name[port.name]
            prop_value = ''
            if net is not None:
                prop_value = get_connected_user_interface_name(self.ndb, net)

            set_prop_commands += self.build_prop_cmds(inst.name, prop_name, prop_value, lock=True)

        return create_commands, set_prop_commands


class JTAGISFConverter(AbstractISFConverter):

    @property
    def block_type(self):
        return "JTAG"

    def process_block(self, inst: Instance) -> Tuple[List[ISFAbstractCommand], List[ISFAbstractCommand]] | None:
        create_cmds = []
        set_prop_cmds = []
        view = inst.view

        name = get_module_name(view)
        if name not in ("EFX_JTAG_V1"):
            return None

        create_cmds += [
            ISFCreateBlockCommand(self.design_api_handle, inst.name, self.block_type)
        ]

        port_name_to_isf_prop_name = {
            "CAPTURE": "CAPTURE",
            "DRCK": "DRCK",
            "RESET": "RESET",
            "RUNTEST": "RUNTEST",
            "SEL": "SEL",
            "SHIFT": "SHIFT",
            "TDI": "TDI",
            "TMS": "TMS",
            "UPDATE": "UPDATE",
            "TDO": "TDO",
            "TCK": "TCK"
        }
        # TCK needed to be handled specically since it is clkout port

        for port in view.ports:
            net = inst.get_connected_net(port)
            prop_name = port_name_to_isf_prop_name[port.name]
            prop_value = ''
            if net is not None:
                prop_value = get_connected_user_interface_name(self.ndb, net)

            set_prop_cmds += self.build_prop_cmds(inst.name, prop_name, prop_value, lock=True)

        resource = view.get_param("RESOURCE")
        if resource is not None and resource.value != '"NONE"':
            resource = remove_double_quote(resource.value)
            set_prop_cmds.append(ISFAssignBlockResourceCommand(self.design_api_handle, inst.name, self.block_type, resource))

        return create_cmds, set_prop_cmds


def is_user_instance(inst: Instance) -> bool:
    return inst.name.endswith('~core~inst') and inst.view.cell_name.endswith('~core')

class PeriNetlistISFBuilder:

    def run(self, ndb: Database, checksum: Optional[str] = None) -> List[str]:
        top = ndb.get_top_module()
        assert top is not None
        cmds = []

        user_inst_name = f'{top.cell_name}~core~inst'

        group_instance: Mapping[NetBus, List[Instance]] = {}
        def add_group(netbus, inst):
            if netbus in group_instance:
                group_instance[netbus].append(inst)
            else:
                group_instance[netbus] = [inst]

        # for netbus in top.netbuses:
        for net in top.nets:
            if not net.bus:
                continue

            for conn in net.connections:
                inst = conn.instance
                name = get_module_name(inst.view)
                port = conn.port
                if name == 'EFX_IBUF' and port.name == 'O':
                    add_group(net.bus, inst)
                    break

                if name == 'EFX_OBUF' and port.name == 'I':
                    add_group(net.bus, inst)
                    break

                if name == 'EFX_IO_BUF' and port.name == 'I':
                    add_group(net.bus, inst)
                    break

                if name == 'EFX_IO_BUF' and port.name == 'O':
                    add_group(net.bus, inst)
                    break

        # Make sure grouped instances having same parameters
        for bus_info, instances in group_instance.items():

            invalid_group = False
            first_inst = instances[0]
            first_view_name = get_module_name(first_inst.view)
            for inst in instances[1:]:
                if first_view_name != get_module_name(inst.view):
                    invalid_group = True
                    break

                if first_inst.view != inst.view:
                    invalid_group = True
                    break

            if invalid_group:
                group_instance.pop(bus_info)

        core_inst = ndb.get_instance(top, user_inst_name)
        assert core_inst is not None

        def can_be_groupped(name: str) -> bool:
            return name in {
                'EFX_IBUF',
                'EFX_OBUF',
                'EFX_IREG',
                'EFX_OREG',
                'EFX_IO_BUF',
                'EFX_IOREG',
                'EFX_GPIO_V1',
                'EFX_GPIO_V2',
                'EFX_GPIO_V3'
            }

        def get_instantiated_insts(container: Netlist, view: Netlist) -> List[Instance]:
            """
            Get the instantiated instance of the view netlist inside the container netlist
            """
            return list(filter(lambda i: i.view_id == view.id, container.instances))

        portbus_groupped_insts = {}
        insts_name_to_portbus = {}
        for netlist in ndb.get_netlists():
            name = get_module_name(netlist)
            if get_peri_netlist(name) is None:
                continue
            can_group = can_be_groupped(name)
            if not can_group:
                continue

            for inst in get_instantiated_insts(top, netlist):
                match name:
                    case 'EFX_IBUF':
                        i_net = inst.get_connected_net(inst.get_port('O'))
                        ports = i_net.get_connected_inst_ports(core_inst)
                        if len(ports) == 1 and ports[0].bus is not None:
                            index = ports[0].index
                            if index != -1:
                                if ports[0].bus not in portbus_groupped_insts:
                                    portbus_groupped_insts[ports[0].bus] = {
                                    }
                                portbus_groupped_insts[ports[0].bus][index] = {
                                    'inst': inst
                                }
                                insts_name_to_portbus[inst.name] = ports[0].bus

                    case 'EFX_OBUF':
                        o_net = inst.get_connected_net(inst.get_port('I'))
                        ports = o_net.get_connected_inst_ports(core_inst)
                        if len(ports) == 1 and ports[0].bus is not None:
                            index = ports[0].index
                            if index != -1:
                                if ports[0].bus not in portbus_groupped_insts:
                                    portbus_groupped_insts[ports[0].bus] = {
                                    }
                                portbus_groupped_insts[ports[0].bus][index] = {
                                    'inst': inst
                                }
                                insts_name_to_portbus[inst.name] = ports[0].bus

                    case 'EFX_IREG':
                        i_net = inst.get_connected_net(inst.get_port('O'))
                        clk = inst.get_connected_net(inst.get_port('CLK'))
                        ports = i_net.get_connected_inst_ports(core_inst)
                        if len(ports) == 1 and ports[0].bus is not None:
                            index = ports[0].index
                            if index != -1:
                                if ports[0].bus not in portbus_groupped_insts:
                                    portbus_groupped_insts[ports[0].bus] = {
                                    }
                                portbus_groupped_insts[ports[0].bus][index] = {
                                    'inst': inst,
                                    'clk': clk.get_source()
                                }
                                insts_name_to_portbus[inst.name] = ports[0].bus
                    case 'EFX_OREG':
                        o_net = inst.get_connected_net(inst.get_port('I'))
                        clk = inst.get_connected_net(inst.get_port('CLK'))
                        ports = o_net.get_connected_inst_ports(core_inst)
                        if len(ports) == 1 and ports[0].bus is not None:
                            index = ports[0].index
                            if index != -1:
                                if ports[0].bus not in portbus_groupped_insts:
                                    portbus_groupped_insts[ports[0].bus] = {
                                    }
                                portbus_groupped_insts[ports[0].bus][index] = {
                                    'inst': inst,
                                    'clk': clk.get_source()
                                }
                                insts_name_to_portbus[inst.name] = ports[0].bus
                    case 'EFX_IO_BUF':
                        # TODO
                        pass

        handled_instances = set()
        gpio_converter = GPIOISFConverter(ndb, checksum)
        for netbus, instances in group_instance.items():
            result = gpio_converter.process_bus_blocks(netbus, instances)
            if result:
                cmds += result[0]
                cmds += result[1]

            for inst in instances:
                handled_instances.add(inst.id)

        gpio_cmds: Dict[Instance, Tuple[List, List]] = {}
        gpio_create_cmds, gpio_set_cmds = [], []
        for inst in top.instances:
            if inst.id in handled_instances:
                continue

            if inst.name != user_inst_name:
                result = gpio_converter.process_block(inst)
                gpio_cmds[inst.id] = result
                if result:
                    if inst.name not in insts_name_to_portbus:
                        cmds += result[0]
                        cmds += result[1]
                    else:
                        gpio_create_cmds += result[0]
                        gpio_set_cmds += result[1]


        for portbus, groupped_result in portbus_groupped_insts.items():
            lsb = min(groupped_result.keys())
            msb = max(groupped_result.keys())

            can_group = True
            if portbus.width != len(groupped_result):
                can_group = False

            # Check all insts have same netlist
            view_name = None
            for index, info in groupped_result.items():
                inst = info['inst']
                if view_name is None:
                    view_name = get_module_name(inst.view)
                else:
                    if view_name != get_module_name(inst.view):
                        can_group = False

            # Check inst name having same pattern
            for index, info in groupped_result.items():
                inst = info['inst']
                if inst.name.startswith(portbus.name) is False:
                    can_group = False

            # Check if having same clk
            clk = None
            for index, info in groupped_result.items():
                if 'clk' in info:
                    if clk is not None:
                        if clk != info['clk']:
                            can_group = False
                            break
                    else:
                        clk = info['clk']

            if can_group:
                create_cmd = None
                for index, info in groupped_result.items():
                    inst = info['inst']
                    if create_cmd is None:
                        create_cmd = next(filter(lambda c: c.block_name.startswith(portbus.name), gpio_create_cmds), None)
                        assert create_cmd is not None
                        create_cmd = copy(create_cmd)
                        create_cmd.block_name = portbus.name
                        create_cmd.msb = portbus.msb
                        create_cmd.lsb = portbus.lsb
                        cmds.append(create_cmd)
            else:
                for index, info in groupped_result.items():
                    inst = info['inst']
                    create_cmd = next(iter(gpio_cmds.get(inst.id)[0]), None)
                    if create_cmd is not None:
                        cmds.append(create_cmd)

        cmds += gpio_set_cmds

        block_converters = [
            PLLISFConverter(ndb, checksum),
            OSCISFConverter(ndb, checksum),
            JTAGISFConverter(ndb, checksum)
        ]
        for converter in block_converters:
            for inst in top.instances:
                if inst.name != user_inst_name:
                    result = converter.process_block(inst)
                    if result:
                        cmds += result[0]
                        cmds += result[1]
        # pprint(cmds)
        return cmds
