from __future__ import annotations
from enum import Enum, auto
from typing import Iterable, List, Optional
from api_service.common.object_db import APIObject
from api_service.internal.int_soc import IntSOCAPI

from api_service.property.gen_prop import AdvPresenter, AdvProp, BaseProp, \
                                         build_api_prop_type, build_prop_option, \
                                         generic_pin_getter, generic_pin_setter, \
                                         generic_param_getter, generic_param_setter, \
                                         generic_pin_attribute_getter, generic_pin_attribute_setter, \
                                         AdvPropData, PropOptionRange, PropOptionSet
from api_service.property.inspector.block_inspect import BlockPropInspector
from api_service.property.soc_prop_id import SOCPropId
from api_service.property.api_options.soc_api_options import enum2api_ops_map
from common_device.property import PropertyMetaData
from design.db import PeriDesign
from tx375_device.soc.design import SOC
from tx375_device.soc.design_param_info import SOCParamInfo


class SOCProp(AdvProp):

    class PropCat(Enum):
        none = auto()

    def __init__(self, block_type=None, db_inst=None, design_db: PeriDesign | None = None, is_build: bool = True):
        self._supported_params: List[Enum] = [
            SOCPropId.axi_master_en,
            SOCPropId.axi_slave_en,
            SOCPropId.custom_instruction_0_en,
            SOCPropId.custom_instruction_1_en,
            SOCPropId.custom_instruction_2_en,
            SOCPropId.custom_instruction_3_en,
            SOCPropId.sys_clk_source,
            SOCPropId.mem_clk_source,
            SOCPropId.pipeline_soc_axi_mem_interface_en,
            SOCPropId.jtag_type,
            SOCPropId.ocr_file_path,
        ]

        self._supported_pins = [
            SOCPropId.axiainterrupt_pn,
            SOCPropId.axia_araddr_pn,
            SOCPropId.axia_arburst_pn,
            SOCPropId.axia_arcache_pn,
            SOCPropId.axia_arlen_pn,
            SOCPropId.axia_arlock_pn,
            SOCPropId.axia_arprot_pn,
            SOCPropId.axia_arqos_pn,
            SOCPropId.axia_arready_pn,
            SOCPropId.axia_arregion_pn,
            SOCPropId.axia_arsize_pn,
            SOCPropId.axia_arvalid_pn,
            SOCPropId.axia_awaddr_pn,
            SOCPropId.axia_awburst_pn,
            SOCPropId.axia_awcache_pn,
            SOCPropId.axia_awlen_pn,
            SOCPropId.axia_awlock_pn,
            SOCPropId.axia_awprot_pn,
            SOCPropId.axia_awqos_pn,
            SOCPropId.axia_awready_pn,
            SOCPropId.axia_awregion_pn,
            SOCPropId.axia_awsize_pn,
            SOCPropId.axia_awvalid_pn,
            SOCPropId.axia_bready_pn,
            SOCPropId.axia_bresp_pn,
            SOCPropId.axia_bvalid_pn,
            SOCPropId.axia_rdata_pn,
            SOCPropId.axia_rlast_pn,
            SOCPropId.axia_rready_pn,
            SOCPropId.axia_rresp_pn,
            SOCPropId.axia_rvalid_pn,
            SOCPropId.axia_wdata_pn,
            SOCPropId.axia_wlast_pn,
            SOCPropId.axia_wready_pn,
            SOCPropId.axia_wstrb_pn,
            SOCPropId.axia_wvalid_pn,
            SOCPropId.cpu0_custominstruction_cmd_ready_pn,
            SOCPropId.cpu0_custominstruction_cmd_valid_pn,
            SOCPropId.cpu0_custominstruction_function_id_pn,
            SOCPropId.cpu0_custominstruction_inputs_0_pn,
            SOCPropId.cpu0_custominstruction_inputs_1_pn,
            SOCPropId.cpu0_custominstruction_outputs_0_pn,
            SOCPropId.cpu0_custominstruction_rsp_ready_pn,
            SOCPropId.cpu0_custominstruction_rsp_valid_pn,
            SOCPropId.cpu1_custominstruction_cmd_ready_pn,
            SOCPropId.cpu1_custominstruction_cmd_valid_pn,
            SOCPropId.cpu1_custominstruction_function_id_pn,
            SOCPropId.cpu1_custominstruction_inputs_0_pn,
            SOCPropId.cpu1_custominstruction_inputs_1_pn,
            SOCPropId.cpu1_custominstruction_outputs_0_pn,
            SOCPropId.cpu1_custominstruction_rsp_ready_pn,
            SOCPropId.cpu1_custominstruction_rsp_valid_pn,
            SOCPropId.cpu2_custominstruction_cmd_ready_pn,
            SOCPropId.cpu2_custominstruction_cmd_valid_pn,
            SOCPropId.cpu2_custominstruction_function_id_pn,
            SOCPropId.cpu2_custominstruction_inputs_0_pn,
            SOCPropId.cpu2_custominstruction_inputs_1_pn,
            SOCPropId.cpu2_custominstruction_outputs_0_pn,
            SOCPropId.cpu2_custominstruction_rsp_ready_pn,
            SOCPropId.cpu2_custominstruction_rsp_valid_pn,
            SOCPropId.cpu3_custominstruction_cmd_ready_pn,
            SOCPropId.cpu3_custominstruction_cmd_valid_pn,
            SOCPropId.cpu3_custominstruction_function_id_pn,
            SOCPropId.cpu3_custominstruction_inputs_0_pn,
            SOCPropId.cpu3_custominstruction_inputs_1_pn,
            SOCPropId.cpu3_custominstruction_outputs_0_pn,
            SOCPropId.cpu3_custominstruction_rsp_ready_pn,
            SOCPropId.cpu3_custominstruction_rsp_valid_pn,
            SOCPropId.io_asyncreset_pn,
            SOCPropId.io_cfuclk_pn,
            SOCPropId.io_cfuclk_invert_en,
            SOCPropId.io_cfureset_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_addr_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_burst_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_cache_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_id_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_len_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_lock_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_prot_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_qos_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_region_pn,
            SOCPropId.io_ddrmasters_0_ar_payload_size_pn,
            SOCPropId.io_ddrmasters_0_ar_ready_pn,
            SOCPropId.io_ddrmasters_0_ar_valid_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_addr_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_allstrb_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_burst_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_cache_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_id_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_len_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_lock_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_prot_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_qos_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_region_pn,
            SOCPropId.io_ddrmasters_0_aw_payload_size_pn,
            SOCPropId.io_ddrmasters_0_aw_ready_pn,
            SOCPropId.io_ddrmasters_0_aw_valid_pn,
            SOCPropId.io_ddrmasters_0_b_payload_id_pn,
            SOCPropId.io_ddrmasters_0_b_payload_resp_pn,
            SOCPropId.io_ddrmasters_0_b_ready_pn,
            SOCPropId.io_ddrmasters_0_b_valid_pn,
            SOCPropId.io_ddrmasters_0_clk_pn,
            SOCPropId.io_ddrmasters_0_clk_invert_en,
            SOCPropId.io_ddrmasters_0_reset_pn,
            SOCPropId.io_ddrmasters_0_r_payload_data_pn,
            SOCPropId.io_ddrmasters_0_r_payload_id_pn,
            SOCPropId.io_ddrmasters_0_r_payload_last_pn,
            SOCPropId.io_ddrmasters_0_r_payload_resp_pn,
            SOCPropId.io_ddrmasters_0_r_ready_pn,
            SOCPropId.io_ddrmasters_0_r_valid_pn,
            SOCPropId.io_ddrmasters_0_w_payload_data_pn,
            SOCPropId.io_ddrmasters_0_w_payload_last_pn,
            SOCPropId.io_ddrmasters_0_w_payload_strb_pn,
            SOCPropId.io_ddrmasters_0_w_ready_pn,
            SOCPropId.io_ddrmasters_0_w_valid_pn,
            SOCPropId.io_jtag_tck_pn,
            SOCPropId.io_jtag_tck_invert_en,
            SOCPropId.io_jtag_tdi_pn,
            SOCPropId.io_jtag_tdo_pn,
            SOCPropId.io_jtag_tms_pn,
            SOCPropId.io_peripheralclk_pn,
            SOCPropId.io_peripheralclk_invert_en,
            SOCPropId.io_peripheralreset_pn,
            SOCPropId.io_systemreset_pn,
            SOCPropId.jtagctrl_capture_pn,
            SOCPropId.jtagctrl_enable_pn,
            SOCPropId.jtagctrl_reset_pn,
            SOCPropId.jtagctrl_shift_pn,
            SOCPropId.jtagctrl_tck_pn,
            SOCPropId.jtagctrl_tck_invert_en,
            SOCPropId.jtagctrl_tdi_pn,
            SOCPropId.jtagctrl_tdo_pn,
            SOCPropId.jtagctrl_update_pn,
            SOCPropId.mem_shutdown_pn,
            SOCPropId.test_pcr_clk_pn,
            SOCPropId.test_pcr_clk_invert_en,
            SOCPropId.t_atpg_nmbist_pn,
            SOCPropId.t_comp_en_pn,
            SOCPropId.t_comp_mask_en_pn,
            SOCPropId.t_comp_mask_ld_pn,
            SOCPropId.t_dft_clock_pn,
            SOCPropId.t_dft_clock_invert_en,
            SOCPropId.t_los_en_pn,
            SOCPropId.t_opcg_enable_pn,
            SOCPropId.t_opcg_load_pn,
            SOCPropId.t_opcg_trigger_pn,
            SOCPropId.t_pmda_done_pn,
            SOCPropId.t_pmda_fail_pn,
            SOCPropId.t_pmda_reset_pn,
            SOCPropId.t_pmda_tck_pn,
            SOCPropId.t_pmda_tck_invert_en,
            SOCPropId.t_pmda_tdi_pn,
            SOCPropId.t_pmda_tdo_pn,
            SOCPropId.t_rstn_pn,
            SOCPropId.t_scanin_pn,
            SOCPropId.t_scanout_pn,
            SOCPropId.t_sck_pn,
            SOCPropId.t_sck_invert_en,
            SOCPropId.t_se_pn,
            SOCPropId.t_se_cg_pn,
            SOCPropId.t_tm_pn,
            SOCPropId.userinterrupta_pn,
            SOCPropId.userinterruptb_pn,
            SOCPropId.userinterruptc_pn,
            SOCPropId.userinterruptd_pn,
            SOCPropId.userinterrupte_pn,
            SOCPropId.userinterruptf_pn,
            SOCPropId.userinterruptg_pn,
            SOCPropId.userinterrupth_pn,
            SOCPropId.userinterrupti_pn,
            SOCPropId.userinterruptj_pn,
            SOCPropId.userinterruptk_pn,
            SOCPropId.userinterruptl_pn,
            SOCPropId.userinterruptm_pn,
            SOCPropId.userinterruptn_pn,
            SOCPropId.userinterrupto_pn,
            SOCPropId.userinterruptp_pn,
            SOCPropId.userinterruptq_pn,
            SOCPropId.userinterruptr_pn,
            SOCPropId.userinterrupts_pn,
            SOCPropId.userinterruptt_pn,
            SOCPropId.userinterruptu_pn,
            SOCPropId.userinterruptv_pn,
            SOCPropId.userinterruptw_pn,
            SOCPropId.userinterruptx_pn,
        ]
        self.__iblock = IntSOCAPI()
        super().__init__(block_type, db_inst, design_db, is_build)

    def _build_db_reader_map(self):
        self._prop_reader_map = {}
        self._prop_reader_map[BaseProp.BasePropId.name] = lambda inst: inst.name
        self._prop_reader_map[BaseProp.BasePropId.resource] = lambda inst: inst.block_def

        for pin_enum in self._supported_pins:
            if self.is_pin_id(pin_enum):
                self._prop_reader_map[pin_enum] = lambda inst, pin_prop_name=pin_enum.value: generic_pin_getter(
                    inst, pin_prop_name)
            elif self.is_pin_invert_enable_id(pin_enum):
                self._prop_reader_map[pin_enum] = lambda inst, pin_prop_name=pin_enum.value, attr_name='is_inverted': generic_pin_attribute_getter(
                    inst, pin_prop_name, attr_name)
        for param_enum in self._supported_params:
            name = param_enum.value
            self._prop_reader_map[param_enum] = lambda inst, param_name=name: generic_param_getter(
                    inst, param_name)

    def _build_db_writer_map(self):
        self._prop_writer_map = {}
        for pin_enum in self._supported_pins:
            if self.is_pin_id(pin_enum):
                self._prop_writer_map[pin_enum] = lambda inst, value, pin_prop_name=pin_enum.value: generic_pin_setter(
                    inst, pin_prop_name, value)
            elif self.is_pin_invert_enable_id(pin_enum):
                self._prop_writer_map[pin_enum] = lambda inst, value, pin_prop_name=pin_enum.value, attr_name='is_inverted': generic_pin_attribute_setter(
                    inst, pin_prop_name, attr_name, value)
        for param_enum in self._supported_params:
            name = param_enum.value
            self._prop_writer_map[param_enum] = lambda inst, value, param_name=name: generic_param_setter(
                    inst, param_name, value)

    def _build_presenter(self):
        self.prs = SOCPresenter()

    def _build_prop_map(self):
        assert self._prop_map is None, "Internal Error: Called more than once"
        self._prop_map = {}
        self._build_param_prop_map(supported_params=self._supported_params, param_info=SOCParamInfo())
        self._build_pin_prop_map(supported_pins=self._supported_pins)

    @staticmethod
    def build_prop(block_type: Optional[APIObject.ObjectType] = None,
                   db_inst: Optional[SOC] = None,
                   design_db: Optional[PeriDesign] = None):
        if design_db is not None:
            if design_db.is_block_supported(PeriDesign.BlockType.soc):
                return SOCProp(block_type, db_inst, design_db)

        else:
            return SOCProp(block_type, db_inst, design_db)

        raise NotImplementedError

    def _build_pin_prop_map(self, supported_pins: List[Enum]):
        assert self._prop_map is not None
        for pin_enum in supported_pins:
            self._insert_prop_map(prop_id=pin_enum,
                                  api_prop_name=pin_enum.value,
                                  api_prop_type=BaseProp.PTypeId.BOOL if pin_enum.name.endswith('invert_en') else BaseProp.PTypeId.STR,
                                  option_list=None,
                                  prop_cat=self.PropCat.none)

    def _build_param_prop_map(self, supported_params: List[Enum], param_info: PropertyMetaData):
        """
        Add Parameter to Property Map

        :support_params: List of supported parameters in PropId Enum
        :param_info: Parameter Definition used to construct the Property Map
        """
        # Make sure the prop map is initialized
        assert self._prop_map is not None

        for param_enum in supported_params:
            prop_id = param_enum
            prop_name = param_enum.value
            prop_info = param_info.get_prop_info_by_name(prop_name)
            assert prop_info is not None
            valid_setting = enum2api_ops_map.get(prop_id.name, None)
            self._insert_prop_map(
                prop_id=prop_id,
                api_prop_name=prop_name,
                api_prop_type=build_api_prop_type(prop_info.data_type),
                option_list=build_prop_option(valid_setting),
                prop_cat=self.PropCat.none)

    def translate(self, prop_id, value, is_db2api=True):
        trans_value = value
        prop_info = self._prop_map.get(prop_id, None)
        if prop_info is not None:
            if is_db2api:
                trans_value = self.prs.translate_db2api(prop_info, value)
            else:
                trans_value = self.prs.translate_api2db(prop_info, value)

        return trans_value

    def build_inspector(self) -> BlockPropInspector:
        from api_service.property.inspector.soc_inspector import SOCPropInspector
        return SOCPropInspector(self, self.__iblock)

    def is_pin_id(self, prop_id: SOCPropId):
        return prop_id.name.endswith('_pn')

    def is_pin_invert_enable_id(self, prop_id: SOCPropId):
        return prop_id.name.endswith('_invert_en')

class SOCPresenter(AdvPresenter):

    pass
