from __future__ import annotations
from enum import Enum
from typing import List, Optional, Dict, Any, Callable
import re

from api_service.common.object_db import APIObject
from api_service.internal.int_raw_serdes import IntRawSerdesAPI

from api_service.property.gen_prop import AdvPresenter, BaseProp, PropOptionSet, AdvPropData
from api_service.property.inspector.block_inspect import BlockPropInspector
from api_service.property.raw_serdes_prop_id import RawSerdesPropId
from api_service.property.api_options.raw_serdes_api_options import enum2api_ops_map
from api_service.property.lane_based_prop import LaneBasedProp

from design.db import PeriDesign

from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonCfgParamInfo
from tx375_device.common_quad.design import QuadLaneCommon

from tx375_device.raw_serdes.design import RawSerdes
from tx375_device.raw_serdes.design_param_info import RawSerdesDesignParamInfo
from tx375_device.raw_serdes.raw_serdes_prop_id import RawSerdesConfigParamInfo
from tx375_device.raw_serdes.quad_param_info import get_hidden_parameters, get_hidden_param_values, get_supported_common_parameters as get_hw_support_cmn_params
from tx375_device.raw_serdes.raw_serdes_pin_dep_graph import get_hidden_ports

from common_device.quad.res_service import QuadType


def convert_preset_str2api_options(preset_str: str):
    api_option_str = ""
    presets = preset_str.split("-")

    if len(presets) != 3:
        return api_option_str

    data_rate, ref_clk_freq, data_width = presets

    def get_check_value(target_str: str, unit: str):
        check_unit = unit.upper()
        return target_str.upper().replace(check_unit, "") if check_unit in target_str.upper() else target_str

    check_data_rate = get_check_value(data_rate, "G")
    check_ref_clk_freq = get_check_value(ref_clk_freq, "MHz")
    check_data_width = get_check_value(data_width, "Bits")

    # Check if valid value
    try:
        float(check_data_rate)
        float(check_ref_clk_freq)
        int(check_data_width)

    except Exception:
        return api_option_str

    return f"{check_data_rate.strip()}G-{check_ref_clk_freq.strip()}MHz-{check_data_width.strip()}Bits"


class RawSerdesProp(LaneBasedProp):
    quad_type = QuadType.raw_serdes

    def build_dummy_instance(self, design_db):
        inst = super().build_dummy_instance(design_db)
        assert isinstance(inst, RawSerdes)

        # PLL config
        reg = design_db.raw_serdes_reg
        assert reg is not None
        inst.pll_cfg = reg.pll_cfg

        return inst

    def get_lane_inst_class(self):
        return RawSerdes

    def get_iblock(self):
        return IntRawSerdesAPI()

    def get_prop_id_class(self):
        return RawSerdesPropId

    def get_hidden_ports(self) -> List[str]:
        return get_hidden_ports()

    def get_hidden_parameters(self) -> List[str]:
        return get_hidden_parameters()

    def get_hidden_param_values(self) -> Dict[str, List[str]]:
        return get_hidden_param_values()

    def get_hw_support_cmn_params(self) -> List[str]:
        return get_hw_support_cmn_params()

    def get_hw_lane_param_id_class(self):
        return RawSerdesDesignParamInfo.Id

    def get_prop_id_str2api_options(self):
        return enum2api_ops_map

    def _build_presenter(self):
        self.prs = RawSerdesPresenter()

    @staticmethod
    def build_prop(block_type: Optional[APIObject.ObjectType] = None,
                   db_inst: Optional[RawSerdes] = None,
                   design_db: Optional[PeriDesign] = None):
        if design_db is not None:
            if design_db.is_block_supported(PeriDesign.BlockType.lane_10g):
                return RawSerdesProp(block_type, db_inst, design_db)

        else:
            return RawSerdesProp(block_type, db_inst, design_db)

        raise NotImplementedError

    def build_inspector(self) -> BlockPropInspector:
        from api_service.property.inspector.raw_serdes_inspector import RawSerdesPropInspector
        return RawSerdesPropInspector(self, self._iblock) # type: ignore

    def _build_param_prop(self, param_enum):
        inst = self.db_inst
        assert isinstance(inst, RawSerdes), f'please build a dummy self.db_inst before calling _build_param_prop'

        if param_enum == RawSerdesPropId.preset:
            self._insert_prop_map(
                prop_id=param_enum,
                api_prop_name=param_enum.value,
                api_prop_type=BaseProp.PTypeId.STR,
                option_list=PropOptionSet(self.__get_preset_api_options(inst)),
                prop_cat=self.PropCat.none
            )
        else:
            return super()._build_param_prop(param_enum)

    def build_param_reader(self, prop_enum):
        self._prop_reader_map: Dict[Enum, Callable[[RawSerdes], str]]
        self._iblock: IntRawSerdesAPI

        if prop_enum == RawSerdesPropId.preset:
            self._prop_reader_map[
            prop_enum] = lambda inst: self._iblock.get_preset_info(inst)[1]
        else:
            return super().build_param_reader(prop_enum)

    def build_param_writer(self, prop_enum):
        self._prop_writer_map: Dict[Enum, Callable[[RawSerdes, Any], None]]
        self._iblock: IntRawSerdesAPI

        if prop_enum == RawSerdesPropId.preset:
            self._prop_writer_map[
            prop_enum] = lambda inst, value: self.set_preset(inst, value)
        else:
            return super().build_param_writer(prop_enum)

    def set_preset(self, inst: RawSerdes, value: str):
        preset_info = value.split("-")

        if len(preset_info) != 3:
            return

        data_rate, ref_clk_freq, data_width = preset_info
        cmn_inst = self.get_common_inst_by_lane(inst)

        if cmn_inst is None:
            return

        inst.set_pll_config_by_preset_key(
            (float(data_rate), float(ref_clk_freq), data_width), cmn_inst, self.is_show_hidden)

    # TODO: check after import all properties from ISF file
    def check_properties(self, inst):
        assert isinstance(inst, RawSerdes)

        err_msg_list: List = []

        return err_msg_list

    def __get_preset_api_options(self, inst: RawSerdes):
        assert inst.pll_cfg is not None
        api_opt_list = []
        for preset in inst.pll_cfg.get_all_preset_names(self.is_show_hidden):
            api_options = convert_preset_str2api_options(preset)
            assert api_options != ""
            api_opt_list.append(api_options)
        return api_opt_list


class RawSerdesPresenter(AdvPresenter):
    def __init__(self):
        super().__init__()

    def translate_api2db(self, prop_info, value: Any) -> Any:
        if isinstance(prop_info, AdvPropData):
            if prop_info.prop_id == RawSerdesPropId.preset:
                api_options = convert_preset_str2api_options(value)
                if api_options == "" or api_options not in prop_info.val_opt:
                    return None

                # Remove all the unit and add " bits"
                db_value = re.sub("[A-Za-z]","",api_options).strip() + " bits"
                return db_value

        return super().translate_api2db(prop_info, value)

    def translate_db2api(self, prop_info, value: Any) -> Optional[str]:
        if isinstance(prop_info, AdvPropData):
            if prop_info.prop_id == RawSerdesPropId.preset:
                api_options = convert_preset_str2api_options(value)
                return api_options

        return super().translate_db2api(prop_info, value)

    def translate_float_api2db(self, prop_info, value) -> Optional[float]:
        value = super().translate_float_api2db(prop_info, value)

        if value is None:
            return value

        precision = self.get_prop_info_precision(prop_info)

        value = round(value, precision)
        return value

    def translate_float_db2api(self, prop_info, value: Any) -> Optional[str]:
        if value is None:
            return None

        precision = self.get_prop_info_precision(prop_info)
        return self.get_float_str(value, decimal_place=precision)

    def get_prop_info_precision(self, prop_info):
        assert prop_info is not None
        precision = None

        assert RawSerdesPropId.has_member(prop_info.name)
        api_id = RawSerdesPropId(prop_info.name)

        if CommonCfgParamInfo.Id.has_key(api_id.name):
            param_id = CommonCfgParamInfo.Id[api_id.name]
            precision = QuadLaneCommon.get_precision(param_id)

        elif RawSerdesConfigParamInfo.Id.has_key(api_id.name):
            param_id = RawSerdesConfigParamInfo.Id[api_id.name]
            precision = RawSerdes.get_precision(param_id)

        # Set default to 3
        if precision is None:
            precision = 3

        return precision
