"""
Copyright (C) 2017-2022 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on September 06, 2022

@author: Shirley Chan
"""

import os
import sys
import abc
from typing import Any, Dict, Optional, Union
from enum import Enum, auto
import networkx as nx
from api_service.excp.prop_excp import PTIllegalValueException
from common_device.mipi.mipi_rx_device_service import MIPIRxService
import device.db_interface as devdb_int

import util.gen_util as gen_util

from common_device.mipi.mipi_design import MIPI, MIPIRegistry, MIPIRx, MIPITx, PhyLane, PhyLaneMap

from api_service.property.gen_prop import BaseProp, AdvProp, AdvPresenter, DynamicPropIdBuilder, PropData, PropOptionRange, PropOptionSet, build_prop_option
from api_service.common.object_db import APIObject
from api_service.internal.int_mipi import IntMIPIAPI

from api_service.property.api_options.mipi_rx_api_options import enum2api_ops_map as rx_api_options_map
from api_service.property.api_options.mipi_tx_api_options import enum2api_ops_map as tx_api_options_map


def generic_pin_attribute_getter(mpd, pin_name: str, attr_name: str):
    pin_group = mpd.gen_pin
    pin = pin_group.get_pin_by_type_name(pin_name)
    assert hasattr(pin, attr_name)
    return getattr(pin, attr_name)

def generic_pin_attribute_setter(mpd, pin_name: str, attr_name: str, value):
    pin_group = mpd.gen_pin
    pin = pin_group.get_pin_by_type_name(pin_name)
    assert hasattr(pin, attr_name)
    return setattr(pin, attr_name, value)


class MIPIProp(AdvProp):
    class PropCat(Enum):
        """
        Property category. This can be internal classification or match with block editor
        """
        none = auto()

    class PropId(Enum):
        """
        Unique ID for each property. ID is auto-generated since the value itself is not important.
        pn = pin name. For this block, each Id name is dynamically generated based on parameter
        and pin in design db.
        """
        pass

    _prop_builder: Optional[DynamicPropIdBuilder] = None  #: Exist only during building, clear after done

    @abc.abstractmethod
    def __init__(self, block_type=None, db_inst=None, design_db=None, is_build=False):
        self._iblock = IntMIPIAPI(design_db)
        self.prs = None
        self._prop_dep_graph = nx.DiGraph()
        self.phylane2pad_map: Dict[PhyLane.PhyLaneIdType, str] = {}

        if design_db is not None and design_db.device_db is not None:
            self.dbi = devdb_int.DeviceDBService(design_db.device_db)

        super().__init__(block_type, db_inst, design_db, is_build)

    def set_phy_lane_map(self):
        if self.db_inst is None:
            return

    def display_prop_id_enum(self):
        for enum_item in self.PropId:
            print(f"{enum_item.name}:{enum_item.value}")

    def translate(self, prop_id, value, is_db2api=True):
        trans_value = value
        prop_info = self._prop_map.get(prop_id, None)
        if prop_info is not None:
            if is_db2api:
                trans_value = self.prs.translate_db2api(prop_info, value)
            else:
                trans_value = self.prs.translate_api2db(prop_info, value)

        return trans_value

    def set_block(self, db_inst):
        super().set_block(db_inst)
        self._build_presenter()

    def get_lane_mapping(self, mipi_info: Union[MIPITx, MIPIRx], phy_lane_id: PhyLane.PhyLaneIdType):
        phy_lane_map = mipi_info.phy_lane_map
        assert phy_lane_map is not None

        phy_lane = phy_lane_map.get_phy_lane(phy_lane_id)
        assert phy_lane is not None
        return phy_lane.logical_lane_id

    def set_lane_mapping(self, mipi_info: Union[MIPITx, MIPIRx], phy_lane_id: PhyLane.PhyLaneIdType, log_lane_id: PhyLane.LogLaneIdType):
        phy_lane_map = mipi_info.phy_lane_map
        assert phy_lane_map is not None
        phy_lane = phy_lane_map.get_phy_lane(phy_lane_id)
        search_phy_lane = phy_lane_map.get_phy_lane_by_log_lane(log_lane_id)

        if search_phy_lane is None:
            raise PTIllegalValueException("ERROR: Invalid lane name")

        # Swap in design
        phy_lane_map.swap_lane_mapping(phy_lane.lane_id, search_phy_lane.lane_id)

    def get_enum_id_by_value(self, phy_lane_value: int, enum_class: Any):
        result_id = None

        for enum_id in enum_class:
            if enum_id.value == phy_lane_value:
                result_id = enum_id
                break

        return result_id


class MIPIRxProp(MIPIProp):
    """
    Handles bi-directional block property translation/mapping between user api and internal api
    for accessing design db.

    .. notes:
       Not meant for heavy design logic, that can be done at the internal api layer
    """

    class PropId(Enum):
        cal_clk_freq = "CAL_CLK_FREQ"
        cal_clk_inverted_en = "CAL_CLK_INVERTED_EN"
        pixel_clk_inverted_en = "PIXEL_CLK_INVERTED_EN"
        rxd0_lane = "RXD0_LANE"
        rxd0_pn_swap = "RXD0_PN_SWAP"
        rxd1_lane = "RXD1_LANE"
        rxd1_pn_swap = "RXD1_PN_SWAP"
        rxd2_lane = "RXD2_LANE"
        rxd2_pn_swap = "RXD2_PN_SWAP"
        rxd3_lane = "RXD3_LANE"
        rxd3_pn_swap = "RXD3_PN_SWAP"
        rxd4_lane = "RXD4_LANE"
        rxd4_pn_swap = "RXD4_PN_SWAP"
        status_en = "STATUS_EN"
        tclk_settle = "TCLK_SETTLE"
        ths_settle = "THS_SETTLE"
        cal_clk_invert_en = "CAL_CLK_INVERT_EN"
        cal_clk_pn = "CAL_CLK_PIN"
        clear_pn = "CLEAR_PIN"
        cnt_pn = "CNT_PIN"
        data_pn = "DATA_PIN"
        dphy_rstn_pn = "DPHY_RSTN_PIN"
        error_pn = "ERROR_PIN"
        hsync_pn = "HSYNC_PIN"
        lanes_pn = "LANES_PIN"
        pixel_clk_invert_en = "PIXEL_CLK_INVERT_EN"
        pixel_clk_pn = "PIXEL_CLK_PIN"
        rstn_pn = "RSTN_PIN"
        type_pn = "TYPE_PIN"
        ulps_pn = "ULPS_PIN"
        ulps_clk_pn = "ULPS_CLK_PIN"
        valid_pn = "VALID_PIN"
        vc_pn = "VC_PIN"
        vc_ena_pn = "VC_ENA_PIN"
        vsync_pn = "VSYNC_PIN"

    def __init__(self, block_type=None, db_inst=None, design_db=None, is_build=False):
        super().__init__(block_type, db_inst, design_db, is_build)
        # Redefined here to link to the class level property
        # Dynamic prop id generation needs access at class object level
        self._prop_map = MIPIRxProp._prop_map
        self.set_phy_lane_map()

    def set_phy_lane_map(self):
        if self.db_inst is None:
            return

        assert self.db_inst.rx_info is not None
        assert isinstance(self.db_inst.rx_info, MIPIRx)
        self.phylane2pad_map = PhyLane.rx_phylane2pad_map

    @staticmethod
    def build_mipi_prop(block_type: APIObject.ObjectType = None, db_inst=None, design_db=None):
        """
        Build the correct MIPI DPhy property based on either design or block instance type.
        Design takes priority.

        :param block_type: API block type
        :param db_inst: MIPI DPhy db instance
        :param design_db: Design db instance
        :return: Prop object depending on context
        """
        gen_util.mark_unused(block_type)
        builder = MIPIRxPropIdBuilder()
        prop = builder.build_prop(db_inst, design_db)
        # prop.display_prop_id_enum()
        return prop

    def _build_presenter(self):
        self.prs = MIPIRxPresenter(self._design_db.mipi_reg)
        # Id is dynamically generated, so it has to be passed here
        self.prs.build(self.PropId, self.db_inst)

    def _build_prop_map(self):

        if self._prop_map is None:
            self._prop_map = {}

            prop_cat = self.PropCat.none

            # Lane mapping
            supported_param = [
                (self.PropId.rxd0_lane, "RXD0_LANE"),
                (self.PropId.rxd1_lane, "RXD1_LANE"),
                (self.PropId.rxd2_lane, "RXD2_LANE"),
                (self.PropId.rxd3_lane, "RXD3_LANE"),
                (self.PropId.rxd4_lane, "RXD4_LANE"),
            ]
            for param_tuple in supported_param:
                api_param_id, api_param_name = param_tuple
                option = self.get_enum2api_ops_map().get(api_param_id.name, None)
                if option is not None:
                    option = build_prop_option(option)
                self._insert_prop_map(api_param_id, api_param_name, BaseProp.PTypeId.STR, option, prop_cat)

            # Assign user property name to each pin to be exposed
            supported_param = [
                (self.PropId.cal_clk_freq,  "CAL_CLK_FREQ", BaseProp.PTypeId.FLOAT),
                (self.PropId.cal_clk_inverted_en, "CAL_CLK_INVERTED_EN", BaseProp.PTypeId.BOOL),
                (self.PropId.pixel_clk_inverted_en, "PIXEL_CLK_INVERTED_EN", BaseProp.PTypeId.BOOL),
                (self.PropId.status_en, "STATUS_EN", BaseProp.PTypeId.BOOL),

                # Timing
                (self.PropId.tclk_settle, "TCLK_SETTLE", BaseProp.PTypeId.INT),
                (self.PropId.ths_settle, "THS_SETTLE", BaseProp.PTypeId.INT),

                # Swap P/N pin
                (self.PropId.rxd0_pn_swap, "RXD0_PN_SWAP", BaseProp.PTypeId.BOOL),
                (self.PropId.rxd1_pn_swap, "RXD1_PN_SWAP", BaseProp.PTypeId.BOOL),
                (self.PropId.rxd2_pn_swap, "RXD2_PN_SWAP", BaseProp.PTypeId.BOOL),
                (self.PropId.rxd3_pn_swap, "RXD3_PN_SWAP", BaseProp.PTypeId.BOOL),
                (self.PropId.rxd4_pn_swap, "RXD4_PN_SWAP", BaseProp.PTypeId.BOOL),
            ]

            for param_tuple in supported_param:
                api_param_id, api_param_name, api_param_type = param_tuple
                option = self.get_enum2api_ops_map().get(api_param_id.name, None)
                if option is not None:
                    option = build_prop_option(option)
                self._insert_prop_map(api_param_id, api_param_name, api_param_type, option, prop_cat)

            # Assign user property name to each pin to be exposed
            supported_pin = [
                (self.PropId.cal_clk_pn, "CAL_CLK_PIN"),
                (self.PropId.pixel_clk_pn, "PIXEL_CLK_PIN"),
                (self.PropId.dphy_rstn_pn, "DPHY_RSTN_PIN"),
                (self.PropId.rstn_pn, "RSTN_PIN"),
                (self.PropId.vc_ena_pn, "VC_ENA_PIN"),
                (self.PropId.lanes_pn, "LANES_PIN"),
                (self.PropId.hsync_pn, "HSYNC_PIN"),
                (self.PropId.vsync_pn, "VSYNC_PIN"),
                (self.PropId.cnt_pn, "CNT_PIN"),
                (self.PropId.valid_pn, "VALID_PIN"),
                (self.PropId.type_pn, "TYPE_PIN"),
                (self.PropId.data_pn, "DATA_PIN"),
                (self.PropId.vc_pn, "VC_PIN"),
                (self.PropId.clear_pn, "CLEAR_PIN"),
                (self.PropId.error_pn, "ERROR_PIN"),
                (self.PropId.ulps_clk_pn, "ULPS_CLK_PIN"),
                (self.PropId.ulps_pn, "ULPS_PIN"),
            ]

            for pin_tuple in supported_pin:
                api_pin_id, api_pin_name = pin_tuple
                self._insert_prop_map(api_pin_id, api_pin_name, BaseProp.PTypeId.STR, None, prop_cat)

    def _build_db_reader_map(self):

        if self._prop_reader_map is None:
            self._prop_reader_map = {}
            self._prop_reader_map[BaseProp.BasePropId.name] = lambda mpd: mpd.name
            self._prop_reader_map[BaseProp.BasePropId.resource] = lambda mpd: mpd.mipi_def

            for param_data in self._prop_builder.param_prop:
                id_name = self._prop_builder.gen_param_name_id(param_data.name)
                param_name = param_data.name
                self._prop_reader_map[self.PropId[id_name]] = lambda mpd, pname=param_name: mpd.rx_info.get_param_group().get_param_value(pname)

            for pin_type in self._prop_builder.pin_type_list:
                id_name = self._prop_builder.gen_pin_name_id(pin_type)
                self._prop_reader_map[self.PropId[id_name]] = lambda mpd, ptype=pin_type: mpd.rx_info.gen_pin.get_pin_name_by_type(ptype)

            self._prop_reader_map[self.PropId.cal_clk_inverted_en] = lambda mpd, pin_name='CAL_CLK', attr_name='is_inverted': generic_pin_attribute_getter(mpd.rx_info, pin_name, attr_name)
            self._prop_reader_map[self.PropId.pixel_clk_inverted_en] = lambda mpd, pin_name='PIXEL_CLK', attr_name='is_inverted': generic_pin_attribute_getter(mpd.rx_info, pin_name, attr_name)
            self._prop_reader_map[self.PropId.status_en] = lambda mpd: mpd.rx_info.is_status_pin_configured()
            self._prop_reader_map[self.PropId.rxd0_lane] = lambda mpd: self.get_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane0)
            self._prop_reader_map[self.PropId.rxd1_lane] = lambda mpd: self.get_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane1)
            self._prop_reader_map[self.PropId.rxd2_lane] = lambda mpd: self.get_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane2)
            self._prop_reader_map[self.PropId.rxd3_lane] = lambda mpd: self.get_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane3)
            self._prop_reader_map[self.PropId.rxd4_lane] = lambda mpd: self.get_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane4)

            self._prop_reader_map[self.PropId.rxd0_pn_swap] = lambda mpd: self.is_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane0)
            self._prop_reader_map[self.PropId.rxd1_pn_swap] = lambda mpd: self.is_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane1)
            self._prop_reader_map[self.PropId.rxd2_pn_swap] = lambda mpd: self.is_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane2)
            self._prop_reader_map[self.PropId.rxd3_pn_swap] = lambda mpd: self.is_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane3)
            self._prop_reader_map[self.PropId.rxd4_pn_swap] = lambda mpd: self.is_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane4)

    def _build_db_writer_map(self):

        if self._prop_writer_map is None:
            self._prop_writer_map = {}

            for param_data in self._prop_builder.param_prop:
                id_name = self._prop_builder.gen_param_name_id(param_data.name)
                param_name = param_data.name
                self._prop_writer_map[self.PropId[id_name]] = lambda mpd, value, pname=param_name: mpd.rx_info.get_param_group().set_param_value(pname, value)

            status_pin = (
                'CLEAR',
                'ERROR',
                'ULPS_CLK',
                'ULPS'
            )
            for pin_type in self._prop_builder.pin_type_list:
                id_name = self._prop_builder.gen_pin_name_id(pin_type)
                self._prop_writer_map[self.PropId[id_name]] = lambda mpd, value, ptype=pin_type: mpd.rx_info.gen_pin.set_pin_name_by_type(ptype, value)

                if pin_type in status_pin:
                    self._prop_writer_map[self.PropId[id_name]] = lambda mpd, value, prop_id=self.PropId[id_name], ptype=pin_type: self.set_pin_value_with_dep(mpd, prop_id, ptype, value)

            self._prop_writer_map[self.PropId.cal_clk_inverted_en] = lambda mpd, value, pin_name='CAL_CLK', attr_name='is_inverted': generic_pin_attribute_setter(mpd.rx_info, pin_name, attr_name, value)
            self._prop_writer_map[self.PropId.pixel_clk_inverted_en] = lambda mpd, value, pin_name='PIXEL_CLK', attr_name='is_inverted': generic_pin_attribute_setter(mpd.rx_info, pin_name, attr_name, value)
            self._prop_writer_map[self.PropId.status_en] = lambda mpd, value: self.set_rx_enable_status(mpd, value)
            self._prop_writer_map[self.PropId.rxd0_lane] = lambda mpd, value: self.set_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane0, value)
            self._prop_writer_map[self.PropId.rxd1_lane] = lambda mpd, value: self.set_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane1, value)
            self._prop_writer_map[self.PropId.rxd2_lane] = lambda mpd, value: self.set_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane2, value)
            self._prop_writer_map[self.PropId.rxd3_lane] = lambda mpd, value: self.set_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane3, value)
            self._prop_writer_map[self.PropId.rxd4_lane] = lambda mpd, value: self.set_lane_mapping(mpd.rx_info, PhyLane.PhyLaneIdType.lane4, value)

            self._prop_writer_map[self.PropId.rxd0_pn_swap] = lambda mpd, value: self.set_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane0, value)
            self._prop_writer_map[self.PropId.rxd1_pn_swap] = lambda mpd, value: self.set_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane1, value)
            self._prop_writer_map[self.PropId.rxd2_pn_swap] = lambda mpd, value: self.set_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane2, value)
            self._prop_writer_map[self.PropId.rxd3_pn_swap] = lambda mpd, value: self.set_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane3, value)
            self._prop_writer_map[self.PropId.rxd4_pn_swap] = lambda mpd, value: self.set_swap_p_n_pin(mpd.rx_info, PhyLane.PhyLaneIdType.lane4, value)

    def get_enum2api_ops_map(self):
        return rx_api_options_map

    def _build_prop_dependency_graph(self):
        self._prop_dep_graph.add_edge(self.PropId.clear_pn, self.PropId.status_en, solver=lambda mpd, val: mpd.rx_info.is_status_pin_configured())
        self._prop_dep_graph.add_edge(self.PropId.error_pn, self.PropId.status_en, solver=lambda mpd, val: mpd.rx_info.is_status_pin_configured())
        self._prop_dep_graph.add_edge(self.PropId.ulps_clk_pn, self.PropId.status_en, solver=lambda mpd, val: mpd.rx_info.is_status_pin_configured())
        self._prop_dep_graph.add_edge(self.PropId.ulps_pn, self.PropId.status_en, solver=lambda mpd, val: mpd.rx_info.is_status_pin_configured())

    def build_inspector(self):
        from api_service.property.inspector.mipi_inspect import MIPIRxPropInspector
        inspector = MIPIRxPropInspector(self, self._iblock)
        return inspector

    def set_pin_value_with_dep(self, mipi: MIPI, prop_id: MIPIProp.PropId, ptype, value):
        self._check_prop_dependency(prop_id, mipi, value)
        mipi.rx_info.gen_pin.set_pin_name_by_type(ptype, value)

    def set_rx_enable_status(self, mipi: MIPI, value: bool):
        mipi_rx = mipi.rx_info
        assert mipi_rx is not None
        mipi_rx.is_status_en = value
        # self.add_updated_prop(self.PropId.status_en)

    def is_swap_p_n_pin(self, mipi_info: MIPIRx, phy_lane_id: PhyLane.PhyLaneIdType):
        phy_lane_map = mipi_info.phy_lane_map
        phy_lane = phy_lane_map.get_phy_lane(phy_lane_id)
        return phy_lane.is_pn_swap

    def set_swap_p_n_pin(self, mipi_info: MIPIRx, phy_lane_id: PhyLane.PhyLaneIdType, is_swap: bool):
        phy_lane_map = mipi_info.phy_lane_map
        phy_lane = phy_lane_map.get_phy_lane(phy_lane_id)
        phy_lane.is_pn_swap = is_swap


class MIPIRxPresenter(AdvPresenter):
    # valid_option_lookup = {}

    def __init__(self, mipi_reg: MIPIRegistry):
        super().__init__()
        self.calculator = mipi_reg.get_timing_calc()
        self.prop_id_enum: Optional[MIPIProp.PropId] = None

    def translate_api2db(self, prop_info: PropData, value: Any) -> Any:
        assert self.prop_id_enum is not None

        if prop_info.prop_id in (self.prop_id_enum.tclk_settle,
                                 self.prop_id_enum.ths_settle):
            if isinstance(value, str):
                try:
                    value = int(value)
                except ValueError:
                    return None
        return super().translate_api2db(prop_info, value)

    def translate_db2api(self, prop_info: PropData, value: Any) -> Optional[str]:
        assert self.prop_id_enum is not None

        result_value = super().translate_db2api(prop_info, value)
        if result_value is not None:
            if prop_info.prop_id == self.prop_id_enum.cal_clk_freq:
                result_value = "{:.2f}".format(value)
        return str(result_value)

    def build(self, prop_id_enum: Any, db_inst: MIPI):
        self.prop_id_enum = prop_id_enum
        self.build_lane_mapping(prop_id_enum)
        self.build_timing_mapping(prop_id_enum, db_inst)

    def build_lane_mapping(self, prop_id_enum: MIPIProp.PropId):
        lane_id_list = [
            prop_id_enum.rxd0_lane,
            prop_id_enum.rxd1_lane,
            prop_id_enum.rxd2_lane,
            prop_id_enum.rxd3_lane,
            prop_id_enum.rxd4_lane
        ]

        for lane_id in lane_id_list:
            self.translator_api2db_map[lane_id] = PhyLane.str2loglane_map
            self.translator_db2api_map[lane_id] = PhyLane.loglane2str_map
            self.valid_option_lookup[lane_id] = list(PhyLane.str2loglane_map.keys())

    def build_timing_mapping(self, prop_id_enum: MIPIProp.PropId, db_inst: MIPI):
        if db_inst is None or db_inst.rx_info is None:
            return

        dphy_calib_clock_freq = db_inst.rx_info.dphy_calib_clock_freq

        if dphy_calib_clock_freq is None:
            return

        self.calculator.update_rx_clk_calib_freq(dphy_calib_clock_freq)

        cal_list2prop_id = (
            (self.calculator.tclk_settle, prop_id_enum.tclk_settle),
            (self.calculator.ths_settle, prop_id_enum.ths_settle),
        )

        for cal_list, prop_id in cal_list2prop_id:
            tclk_db2_api = dict()
            tclk_api2_db = dict()

            opt_index = 1
            for str_delay in cal_list:
                delay = str_delay
                tclk_db2_api[opt_index] = delay
                tclk_api2_db[delay] = opt_index
                opt_index += 1

            self.translator_api2db_map[prop_id] = tclk_api2_db
            self.translator_db2api_map[prop_id] = tclk_db2_api
            self.valid_option_lookup[prop_id] = list(tclk_api2_db.keys())


class MIPIRxPropIdBuilder(DynamicPropIdBuilder):

    def __init__(self):
        super().__init__()

    def build_prop(self, db_inst=None, design_db=None):

        self.prop = MIPIRxProp("MIPI_RX", db_inst, design_db, False)

        block_inst = None
        if design_db is not None and db_inst is None:
            # Create dummy instance to extract param and pin info
            imipi = IntMIPIAPI()
            imipi.set_design(design_db)
            block_inst = imipi.create_rx_block("dummy", False)

        elif db_inst is not None:
            block_inst = db_inst

        if block_inst is not None:
            self.prop._prop_builder = self
            rx_info = block_inst.rx_info
            self.prop.db_inst = block_inst
            self.build_param_id_enum(rx_info)
            self.build_pin_id_enum(rx_info)
            self.prop.PropId = Enum('PropId', self.prop_enum_map)
            self.prop.build()
            self.prop._prop_builder = None

        return self.prop


class MIPITxProp(MIPIProp):
    """
    Handles bi-directional block property translation/mapping between user api and internal api
    for accessing design db.

    .. notes:
       Not meant for heavy design logic, that can be done at the internal api layer
    """

    class PropId(Enum):
        cont_phy_clk_en = "CONT_PHY_CLK_EN"
        esc_clk_freq = "ESC_CLK_FREQ"
        esc_clk_inverted_en = "ESC_CLK_INVERTED_EN"
        phy_freq = "PHY_FREQ"
        pixel_clk_inverted_en = "PIXEL_CLK_INVERTED_EN"
        refclk_freq = "REFCLK_FREQ"
        refclk_inst = "REFCLK_INST"
        refclk_res = "REFCLK_RES"
        tclk_post = "TCLK_POST"
        tclk_pre = "TCLK_PRE"
        tclk_prepare = "TCLK_PREPARE"
        tclk_trail = "TCLK_TRAIL"
        tclk_zero = "TCLK_ZERO"
        ths_prepare = "THS_PREPARE"
        ths_trail = "THS_TRAIL"
        ths_zero = "THS_ZERO"
        txd0_lane = "TXD0_LANE"
        txd1_lane = "TXD1_LANE"
        txd2_lane = "TXD2_LANE"
        txd3_lane = "TXD3_LANE"
        txd4_lane = "TXD4_LANE"
        data_pn = "DATA_PIN"
        dphy_rstn_pn = "DPHY_RSTN_PIN"
        esc_clk_invert_en = "ESC_CLK_INVERT_EN"
        esc_clk_pn = "ESC_CLK_PIN"
        frame_mode_pn = "FRAME_MODE_PIN"
        hres_pn = "HRES_PIN"
        hsync_pn = "HSYNC_PIN"
        lanes_pn = "LANES_PIN"
        pixel_clk_invert_en = "PIXEL_CLK_INVERT_EN"
        pixel_clk_pn = "PIXEL_CLK_PIN"
        rstn_pn = "RSTN_PIN"
        type_pn = "TYPE_PIN"
        ulps_clk_enter_pn = "ULPS_CLK_ENTER_PIN"
        ulps_clk_exit_pn = "ULPS_CLK_EXIT_PIN"
        ulps_enter_pn = "ULPS_ENTER_PIN"
        ulps_exit_pn = "ULPS_EXIT_PIN"
        valid_pn = "VALID_PIN"
        vc_pn = "VC_PIN"
        vsync_pn = "VSYNC_PIN"

    def __init__(self, block_type=None, db_inst=None, design_db=None, is_build=False):
        super().__init__(block_type, db_inst, design_db, is_build)
        # Redefined here to link to the class level property
        # Dynamic prop id generation needs access at class object level
        self._prop_map = MIPITxProp._prop_map
        self.set_phy_lane_map()

    def set_phy_lane_map(self):
        if self.db_inst is None:
            return

        assert self.db_inst.tx_info is not None
        assert isinstance(self.db_inst.tx_info, MIPITx)
        self.phylane2pad_map = PhyLane.tx_phylane2pad_map

    @staticmethod
    def build_mipi_prop(block_type: APIObject.ObjectType = None, db_inst=None, design_db=None):
        """
        Build the correct MIPI DPhy property based on either design or block instance type.
        Design takes priority.

        :param block_type: API block type
        :param db_inst: MIPI DPhy db instance
        :param design_db: Design db instance
        :return: Prop object depending on context
        """
        gen_util.mark_unused(block_type)
        builder = MIPITxPropIdBuilder()
        prop = builder.build_prop(db_inst, design_db)
        # prop.display_prop_id_enum()
        return prop

    def _build_presenter(self):
        self.prs = MIPITxPresenter(self._design_db.mipi_reg)
        # Id is dynamically generated, so it has to be passed here
        self.prs.build(self.PropId, self.db_inst)

    def _build_prop_map(self):

        if self._prop_map is None:
            self._prop_map = {}

            prop_cat = self.PropCat.none

            supported_param = [
                # Lane Mapping
                (self.PropId.txd0_lane, "TXD0_LANE"),
                (self.PropId.txd1_lane, "TXD1_LANE"),
                (self.PropId.txd2_lane, "TXD2_LANE"),
                (self.PropId.txd3_lane, "TXD3_LANE"),
                (self.PropId.txd4_lane, "TXD4_LANE"),
            ]
            for param_tuple in supported_param:
                api_param_id, api_param_name = param_tuple
                option = self.get_enum2api_ops_map().get(api_param_id.name, None)
                if option is not None:
                    option = build_prop_option(option)
                self._insert_prop_map(api_param_id, api_param_name, BaseProp.PTypeId.STR, option, prop_cat)

            # Assign user property name to each pin to be exposed
            supported_param = [
                (self.PropId.esc_clk_freq, "ESC_CLK_FREQ", BaseProp.PTypeId.FLOAT),
                (self.PropId.refclk_freq, "REFCLK_FREQ", BaseProp.PTypeId.FLOAT),

                # Reference Clock
                (self.PropId.refclk_res, "REFCLK_RES", BaseProp.PTypeId.STR),
                (self.PropId.refclk_inst, "REFCLK_INST", BaseProp.PTypeId.STR),

                # PHY Clock
                (self.PropId.phy_freq, "PHY_FREQ", BaseProp.PTypeId.INT),
                (self.PropId.cont_phy_clk_en, "CONT_PHY_CLK_EN", BaseProp.PTypeId.BOOL),

                # Control
                (self.PropId.esc_clk_inverted_en, "ESC_CLK_INVERTED_EN", BaseProp.PTypeId.BOOL),
                (self.PropId.pixel_clk_inverted_en, "PIXEL_CLK_INVERTED_EN", BaseProp.PTypeId.BOOL),

                # Timing
                (self.PropId.tclk_post, "TCLK_POST", BaseProp.PTypeId.INT),
                (self.PropId.tclk_trail, "TCLK_TRAIL", BaseProp.PTypeId.INT),
                (self.PropId.tclk_prepare, "TCLK_PREPARE", BaseProp.PTypeId.INT),
                (self.PropId.tclk_zero, "TCLK_ZERO", BaseProp.PTypeId.INT),
                (self.PropId.tclk_pre, "TCLK_PRE", BaseProp.PTypeId.INT),
                (self.PropId.ths_prepare, "THS_PREPARE", BaseProp.PTypeId.INT),
                (self.PropId.ths_zero, "THS_ZERO", BaseProp.PTypeId.INT),
                (self.PropId.ths_trail, "THS_TRAIL", BaseProp.PTypeId.INT),
            ]

            for param_tuple in supported_param:
                api_param_id, api_param_name, api_param_type = param_tuple
                option = self.get_enum2api_ops_map().get(api_param_id.name, None)
                if option is not None:
                    option = build_prop_option(option)
                self._insert_prop_map(api_param_id, api_param_name, api_param_type, option, prop_cat)

            # Assign user property name to each pin to be exposed
            supported_pin = [
                # (self.PropId.ref_clk_pn, "REF_CLK_PIN"),
                (self.PropId.esc_clk_pn, "ESC_CLK_PIN"),
                (self.PropId.pixel_clk_pn, "PIXEL_CLK_PIN"),
                (self.PropId.dphy_rstn_pn, "DPHY_RSTN_PIN"),
                (self.PropId.rstn_pn, "RSTN_PIN"),
                (self.PropId.lanes_pn, "LANES_PIN"),
                (self.PropId.frame_mode_pn, "FRAME_MODE_PIN"),
                (self.PropId.hres_pn, "HRES_PIN"),
                (self.PropId.hsync_pn, "HSYNC_PIN"),
                (self.PropId.vsync_pn, "VSYNC_PIN"),
                (self.PropId.valid_pn, "VALID_PIN"),
                (self.PropId.type_pn, "TYPE_PIN"),
                (self.PropId.data_pn, "DATA_PIN"),
                (self.PropId.vc_pn, "VC_PIN"),
                (self.PropId.ulps_clk_enter_pn, "ULPS_CLK_ENTER_PIN"),
                (self.PropId.ulps_clk_exit_pn, "ULPS_CLK_EXIT_PIN"),
                (self.PropId.ulps_enter_pn, "ULPS_ENTER_PIN"),
                (self.PropId.ulps_exit_pn, "ULPS_EXIT_PIN"),
            ]

            for pin_tuple in supported_pin:
                api_pin_id, api_pin_name = pin_tuple
                self._insert_prop_map(api_pin_id, api_pin_name, BaseProp.PTypeId.STR, None, prop_cat)

    def _build_db_reader_map(self):

        if self._prop_reader_map is None:
            self._prop_reader_map = {}
            self._prop_reader_map[BaseProp.BasePropId.name] = lambda mpd: mpd.name
            self._prop_reader_map[BaseProp.BasePropId.resource] = lambda mpd: mpd.mipi_def

            for param_data in self._prop_builder.param_prop:
                id_name = self._prop_builder.gen_param_name_id(param_data.name)
                param_name = param_data.name
                self._prop_reader_map[self.PropId[id_name]] = lambda mpd,pname=param_name: mpd.tx_info.get_param_group().get_param_value(pname)

            for pin_type in self._prop_builder.pin_type_list:
                id_name = self._prop_builder.gen_pin_name_id(pin_type)
                self._prop_reader_map[self.PropId[id_name]] = lambda mpd,ptype=pin_type: mpd.tx_info.gen_pin.get_pin_name_by_type(ptype)

            self._prop_reader_map[self.PropId.esc_clk_inverted_en] = lambda mpd, pin_name='ESC_CLK', attr_name='is_inverted': generic_pin_attribute_getter(mpd.tx_info, pin_name, attr_name)
            self._prop_reader_map[self.PropId.pixel_clk_inverted_en] = lambda mpd, pin_name='PIXEL_CLK', attr_name='is_inverted': generic_pin_attribute_getter(mpd.tx_info, pin_name, attr_name)
            self._prop_reader_map[self.PropId.refclk_res] = lambda mpd, info_type='res_name': self.get_refclock_info(mpd, info_type)
            self._prop_reader_map[self.PropId.refclk_inst] = lambda mpd, info_type='inst_name': self.get_refclock_info(mpd, info_type)
            self._prop_reader_map[self.PropId.txd0_lane] = lambda mpd: self.get_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane0)
            self._prop_reader_map[self.PropId.txd1_lane] = lambda mpd: self.get_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane1)
            self._prop_reader_map[self.PropId.txd2_lane] = lambda mpd: self.get_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane2)
            self._prop_reader_map[self.PropId.txd3_lane] = lambda mpd: self.get_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane3)
            self._prop_reader_map[self.PropId.txd4_lane] = lambda mpd: self.get_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane4)

    def _build_db_writer_map(self):
        if self._prop_writer_map is None:
            self._prop_writer_map = {}

            for param_data in self._prop_builder.param_prop:
                id_name = self._prop_builder.gen_param_name_id(param_data.name)
                param_name = param_data.name
                self._prop_writer_map[self.PropId[id_name]] = lambda mpd, value, pname=param_name: mpd.tx_info.get_param_group().set_param_value(pname, value)

            for pin_type in self._prop_builder.pin_type_list:
                id_name = self._prop_builder.gen_pin_name_id(pin_type)
                self._prop_writer_map[self.PropId[id_name]] = lambda mpd, value, ptype=pin_type: mpd.tx_info.gen_pin.set_pin_name_by_type(ptype, value)

            self._prop_writer_map[self.PropId.esc_clk_inverted_en] = lambda mpd, value, pin_name='ESC_CLK', attr_name='is_inverted': generic_pin_attribute_setter(mpd.tx_info, pin_name, attr_name, value)
            self._prop_writer_map[self.PropId.pixel_clk_inverted_en] = lambda mpd, value, pin_name='PIXEL_CLK', attr_name='is_inverted': generic_pin_attribute_setter(mpd.tx_info, pin_name, attr_name, value)
            self._prop_writer_map[self.PropId.txd0_lane] = lambda mpd, value: self.set_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane0, value)
            self._prop_writer_map[self.PropId.txd1_lane] = lambda mpd, value: self.set_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane1, value)
            self._prop_writer_map[self.PropId.txd2_lane] = lambda mpd, value: self.set_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane2, value)
            self._prop_writer_map[self.PropId.txd3_lane] = lambda mpd, value: self.set_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane3, value)
            self._prop_writer_map[self.PropId.txd4_lane] = lambda mpd, value: self.set_lane_mapping(mpd.tx_info, PhyLane.PhyLaneIdType.lane4, value)

            # Remove function for readonly property
            inspector = self.build_inspector()
            for read_only_prop in inspector.readonly_prop:
                self._prop_writer_map[read_only_prop] = None

    def get_enum2api_ops_map(self):
        return tx_api_options_map

    def build_inspector(self):
        from api_service.property.inspector.mipi_inspect import MIPITxPropInspector
        inspector = MIPITxPropInspector(self, self._iblock)
        return inspector

    def get_refclock_info(self, mipi: MIPI, info_type: str):
        assert self._design_db is not None

        tx_info = mipi.tx_info
        if tx_info is None:
            return ""

        mipi_tx_dev_service = self.dbi.get_block_service(devdb_int.DeviceDBService.BlockType.MIPI_TX)
        res_name, inst_name, _ = tx_info.find_refclock_info(
                                    mipi.mipi_def,
                                    mipi_tx_dev_service,
                                    self._design_db.gpio_reg)

        if info_type == "res_name":
            return res_name
        elif info_type == "inst_name":
            return inst_name
        else:
            return "unknown"


class MIPITxPresenter(AdvPresenter):
    valid_option_lookup = {}

    def __init__(self, mipi_reg: MIPIRegistry):
        super().__init__()
        self.calculator = mipi_reg.get_timing_calc()
        self.prop_id_enum: Optional[MIPIProp.PropId] = None

    def translate_api2db(self, prop_info, value: Any) -> Any:
        assert self.prop_id_enum is not None

        if prop_info.prop_id in (
            self.prop_id_enum.tclk_post,
            self.prop_id_enum.tclk_trail,
            self.prop_id_enum.tclk_prepare,
            self.prop_id_enum.tclk_pre,
            self.prop_id_enum.tclk_zero,
            self.prop_id_enum.ths_prepare,
            self.prop_id_enum.ths_zero,
            self.prop_id_enum.ths_trail,
            ):
            if isinstance(value, str):
                try:
                    value = int(value)
                except ValueError:
                    return None
        elif prop_info.prop_id == self.prop_id_enum.phy_freq:
            if isinstance(value, str):
                try:
                    value = float(value)
                except ValueError:
                    return None
        return super().translate_api2db(prop_info, value)

    def translate_db2api(self, prop_info: PropData, value: Any) -> Optional[str]:
        assert self.prop_id_enum is not None

        result_value = super().translate_db2api(prop_info, value)
        if result_value is not None:
            if prop_info.prop_id == self.prop_id_enum.phy_freq:
                result_value = "{:.2f}".format(result_value)
            elif prop_info.prop_id in (self.prop_id_enum.esc_clk_freq, self.prop_id_enum.refclk_freq):
                result_value = "{:.2f}".format(value)
        return str(result_value)

    def build(self, prop_id_enum: MIPIProp.PropId, db_inst: MIPI):
        self.prop_id_enum = prop_id_enum
        self.valid_option_lookup = {
            prop_id_enum.phy_freq: list(MIPITx._phy_freq2code.keys()),
        }
        self.translator_api2db_map = \
            {
                prop_id_enum.phy_freq: MIPITx._phy_freq2code
            }

        self.translator_db2api_map = \
            {
                prop_id_enum.phy_freq: MIPITx._phy_freq_map
            }

        self.build_lane_mapping(prop_id_enum)
        self.build_timing_mapping(prop_id_enum, db_inst)

    def build_lane_mapping(self, prop_id_enum: MIPIProp.PropId):
        lane_id_list = [
            prop_id_enum.txd0_lane,
            prop_id_enum.txd1_lane,
            prop_id_enum.txd2_lane,
            prop_id_enum.txd3_lane,
            prop_id_enum.txd4_lane
        ]

        for lane_id in lane_id_list:
            self.translator_api2db_map[lane_id] = PhyLane.str2loglane_map
            self.translator_db2api_map[lane_id] = PhyLane.loglane2str_map
            self.valid_option_lookup[lane_id] = list(PhyLane.str2loglane_map.keys())

    def build_timing_mapping(self, prop_id_enum: MIPIProp.PropId, db_inst: Optional[MIPI]):
        if db_inst is None or db_inst.tx_info is None:
            return

        phy_tx_freq_code = db_inst.tx_info.phy_tx_freq_code

        if phy_tx_freq_code is None:
            return

        esc_clock_freq = db_inst.tx_info.esc_clock_freq

        self.calculator.update_phy_tx_freq_code(phy_tx_freq_code)
        self.calculator.update_tx_clk_escape_freq(esc_clock_freq)

        cal_list2prop_id = (
            (self.calculator.tclk_post, prop_id_enum.tclk_post),
            (self.calculator.tclk_trail, prop_id_enum.tclk_trail),
            (self.calculator.tclk_prepare, prop_id_enum.tclk_prepare),
            (self.calculator.tclk_pre, prop_id_enum.tclk_pre),
            (self.calculator.tclk_zero, prop_id_enum.tclk_zero),
            (self.calculator.ths_prepare, prop_id_enum.ths_prepare),
            (self.calculator.ths_zero, prop_id_enum.ths_zero),
            (self.calculator.ths_trail, prop_id_enum.ths_trail),
        )

        for cal_list, prop_id in cal_list2prop_id:
            tclk_db2_api = dict()
            tclk_api2_db = dict()

            opt_index = 1
            for _, delay in cal_list:
                tclk_db2_api[opt_index] = delay
                tclk_api2_db[delay] = opt_index
                opt_index += 1

            self.translator_api2db_map[prop_id] = tclk_api2_db
            self.translator_db2api_map[prop_id] = tclk_db2_api
            self.valid_option_lookup[prop_id] = list(tclk_api2_db.keys())


class MIPITxPropIdBuilder(DynamicPropIdBuilder):

    def __init__(self):
        super().__init__()

    def build_prop(self, db_inst=None, design_db=None):

        self.prop = MIPITxProp("MIPI_TX", db_inst, design_db, False)

        block_inst = None
        if design_db is not None and db_inst is None:
            # Create dummy instance to extract param and pin info
            imipi = IntMIPIAPI()
            imipi.set_design(design_db)
            block_inst = imipi.create_tx_block("dummy", False)

        elif db_inst is not None:
            block_inst = db_inst

        if block_inst is not None:
            self.prop._prop_builder = self
            tx_info = block_inst.tx_info
            self.build_param_id_enum(tx_info)
            self.build_pin_id_enum(tx_info)
            self.prop.PropId = Enum('PropId', self.prop_enum_map)
            self.prop.build()
            self.prop._prop_builder = None

        return self.prop


if __name__ == "__main__":
    pass
