from __future__ import annotations
import os
from enum import Enum, auto
from typing import Iterable, List, Dict, Callable, Any, Tuple, Type, TYPE_CHECKING
from abc import abstractmethod

import pt_version

from api_service.internal.int_gen_block import IntBlockAPI

from api_service.property.gen_prop import (
    AdvPresenter,
    AdvProp,
    AdvPropData,
    BaseProp,
    build_api_prop_type,
    build_prop_option,
    PropOptionRange,
    PropOptionSet,
)
from api_service.property.inspector.block_inspect import BlockPropInspector

from design.db import PeriDesign
from tx375_device.common_quad.design_param_info import QuadDesignParamInfo, get_supported_params_by_quad_type

from tx375_device.common_quad.quad_prop_id import CommonQuadConfigParamInfo as CommonCfgParamInfo
from tx375_device.common_quad.design import QuadLaneCommon

from common_device.quad.lane_design import LaneBasedItem
from common_device.quad.res_service import QuadType

if TYPE_CHECKING:
    from common_device.property import ValidSettingTypeAlias

class LaneBasedProp(AdvProp):

    quad_type = QuadType.quad_pcie

    class PropCat(Enum):
        none = auto()


    def __init__(self, block_type=None, db_inst=None, design_db: PeriDesign | None = None, is_build: bool = True):
        self.is_show_hidden = True if pt_version.PT_DEBUG_VERSION == True else False
        supported_cmn_params = get_supported_params_by_quad_type(self.quad_type)
        self.prs: AdvPresenter

        def _is_non_hidden_prop(prop_id: Enum):
            prop_name = prop_id.name.removesuffix('_pn')
            prop_name = prop_name.removeprefix('ID_')

            if QuadDesignParamInfo.Id.has_key(prop_name) and \
                QuadDesignParamInfo.Id[prop_name] not in supported_cmn_params:
                return False
            elif CommonCfgParamInfo.Id.has_key(prop_name) and \
                prop_name not in self.get_hw_support_cmn_params():
                return False

            if self.is_show_hidden:
                return True

            if prop_name in self.get_hidden_parameters():
                return False
            elif prop_name.upper() in self.get_hidden_ports():
                return False

            return True

        self._supported_props: List[Enum] = list(
            prop_id for prop_id in self.get_prop_id_class() if _is_non_hidden_prop(prop_id))

        # Mapping of deprecated prop to tuple of:
        #   - Deprecated message
        #   - If API prop is usable when deprecated
        self._deprecated_prop_id_name2info: Dict[str, Tuple[str, bool]] = {
            "user_phy_reset_n_pn": ("USER_PHY_RESET_N_PIN is deprecated, "\
                "use property PHY_CMN_RESET_N_PIN instead", False),
            "sw_raw_refclk_freq": ("SW_RAW_REFCLK_FREQ is deprecated, "\
                "use property SS_RAW_REFCLK_FREQ instead", False),
        }

        self._iblock = self.get_iblock()
        self.common_quad_reg = None

        if db_inst is None:
            assert design_db is not None
            # Create a dummy instance for constructing prop map
            db_inst = self.build_dummy_instance(design_db)

        if design_db is not None:
            self.common_quad_reg = design_db.common_quad_lane_reg
            self._iblock.set_design(design_db)

        self.db_inst: LaneBasedItem
        super().__init__(block_type, db_inst, design_db, is_build)

    def build_dummy_instance(self, design_db: PeriDesign):
        LANE_CLASS = self.get_lane_inst_class()
        LANE_CLASS.build_port_info(device_db=design_db.device_db)
        db_inst = LANE_CLASS(name='dummy')
        return db_inst

    @abstractmethod
    def get_lane_inst_class(self) -> Type[LaneBasedItem]:
        pass

    @abstractmethod
    def get_iblock(self) -> IntBlockAPI:
        pass

    @abstractmethod
    def get_prop_id_class(self) -> Type[Enum]:
        pass

    @abstractmethod
    def get_hidden_ports(self) -> List[str]:
        pass

    @abstractmethod
    def get_hw_support_cmn_params(self) -> List[str]:
        pass

    @abstractmethod
    def get_hidden_parameters(self) -> List[str]:
        pass

    @abstractmethod
    def get_hidden_param_values(self) -> Dict[str, List[str]]:
        pass

    @abstractmethod
    def get_hw_lane_param_id_class(self) -> Type[Enum]:
        pass

    @abstractmethod
    def get_prop_id_str2api_options(self) -> Dict[str, ValidSettingTypeAlias]:
        pass

    def _build_db_reader_map(self):
        self._prop_reader_map: Dict[Enum, Callable[[LaneBasedItem], Any]] = {}
        self._prop_reader_map[BaseProp.BasePropId.name] = lambda inst: inst.name # type: ignore
        self._prop_reader_map[BaseProp.BasePropId.resource] = lambda inst: inst.get_device()

        for prop_enum in self._supported_props:
            if self.is_pin_id(prop_enum):
                self._prop_reader_map[prop_enum] = lambda inst, pin_prop_name=prop_enum.value: self.generic_pin_getter(
                    inst, pin_prop_name)

            elif self.is_pin_invert_enable_id(prop_enum):
                self._prop_reader_map[prop_enum] = lambda inst, pin_prop_name=prop_enum.value, attr_name='is_inverted': self.generic_pin_attribute_getter(
                    inst, pin_prop_name, attr_name)
            elif prop_enum.name == 'common_inst_name':
                self._prop_reader_map[
                prop_enum] = lambda inst: self.get_common_inst_by_lane(
                    inst).name

            else:
                self.build_param_reader(prop_enum)

    def build_param_reader(self, prop_enum: Enum):
        param_id, _ = self.get_param_id_by_api_id(prop_enum)
        name = param_id.value
        self._prop_reader_map[
            prop_enum] = lambda inst, param_name=name: self.generic_param_getter(
                inst, param_name)

    def _get_pin_type(self, pin_prop_name: str) -> str:
        pin_prop_name = pin_prop_name.removesuffix('_PIN')
        pin_prop_name = pin_prop_name.removesuffix('_INVERT_EN')
        return pin_prop_name

    def get_common_inst_by_lane(self, inst: LaneBasedItem):
        common_inst = None
        if self.common_quad_reg is not None:
            common_inst = self.common_quad_reg.get_inst_by_lane_name(inst.quad_type, inst.name)
        if common_inst is None:
            common_inst = QuadLaneCommon("dummy", "")
        assert common_inst is not None
        return common_inst

    def generic_pin_getter(self, inst: LaneBasedItem, pin_prop_name: str):
        pin_group = inst.gen_pin
        pin_type_name = self._get_pin_type(pin_prop_name)
        pin = pin_group.get_pin_by_type_name(pin_type_name)

        # pin can be from common_quad_lane_reg
        if pin is None:
            common_inst = self.get_common_inst_by_lane(inst)
            assert common_inst is not None
            pin = common_inst.gen_pin.get_pin_by_type_name(pin_type_name)

        if pin is not None:
            return pin.name

        return None

    def generic_pin_setter(self, inst: LaneBasedItem, pin_prop_name: str, value: str):
        pin_group = inst.gen_pin
        pin_type_name = self._get_pin_type(pin_prop_name)
        pin = pin_group.get_pin_by_type_name(pin_type_name)

        # pin can be from common_quad_lane_reg
        if pin is None:
            common_inst = self.get_common_inst_by_lane(inst)
            assert common_inst is not None
            pin = common_inst.gen_pin.get_pin_by_type_name(pin_type_name)

        if pin is not None:
           pin.name = value

    def generic_pin_attribute_getter(self, inst: LaneBasedItem, pin_prop_name: str, attr_name: str):
        pin_group = inst.gen_pin
        pin_type_name = self._get_pin_type(pin_prop_name)
        pin = pin_group.get_pin_by_type_name(pin_type_name)

        # pin can be from common_quad_lane_reg
        if pin is None:
            common_inst = self.get_common_inst_by_lane(inst)
            assert common_inst is not None
            pin = common_inst.gen_pin.get_pin_by_type_name(pin_type_name)

        if pin is not None:
            assert hasattr(pin, attr_name)
            return getattr(pin, attr_name)

        return None

    def generic_pin_attribute_setter(self, inst: LaneBasedItem, pin_prop_name: str, attr_name: str, value):
        pin_group = inst.gen_pin
        pin_type_name = self._get_pin_type(pin_prop_name)
        pin = pin_group.get_pin_by_type_name(pin_type_name)

        # pin can be from common_quad_lane_reg
        if pin is None:
            common_inst = self.get_common_inst_by_lane(inst)
            assert common_inst is not None
            pin = common_inst.gen_pin.get_pin_by_type_name(pin_type_name)

        if pin is not None:
            assert hasattr(pin, attr_name)
            setattr(pin, attr_name, value)

    def generic_param_getter(self, inst: LaneBasedItem, param_name: str) -> Any:
        param_group = inst.param_group
        if param_group.get_param_by_name(param_name) is not None:
            return param_group.get_param_value(param_name)

        # Check if the parameter from Quad lane reg
        common_inst = self.get_common_inst_by_lane(inst)
        assert common_inst is not None
        param_group = common_inst.param_group
        if param_group.get_param_by_name(param_name) is not None:
            return param_group.get_param_value(param_name)

        return None

    def generic_param_setter(self, inst: LaneBasedItem, param_name: str, value: Any):
        param_group = inst.param_group
        if param_group.get_param_by_name(param_name) is not None:
            param_group.set_param_value(param_name, value)

        common_inst = self.get_common_inst_by_lane(inst)
        assert common_inst is not None
        param_group = common_inst.param_group
        if param_group.get_param_by_name(param_name) is not None:
            param_group.set_param_value(param_name, value)

    def _build_db_writer_map(self):
        self._prop_writer_map = {}

        for prop_enum in self._supported_props:
            if self.is_pin_id(prop_enum):
                self._prop_writer_map[prop_enum] = lambda inst, value, pin_prop_name=prop_enum.value: self.generic_pin_setter(
                    inst, pin_prop_name, value)

            elif self.is_pin_invert_enable_id(prop_enum):
                self._prop_writer_map[prop_enum] = lambda inst, value, pin_prop_name=prop_enum.value, attr_name='is_inverted': self.generic_pin_attribute_setter(
                    inst, pin_prop_name, attr_name, value)
            elif prop_enum.name == 'common_inst_name':
                continue

            else:
                self.build_param_writer(prop_enum)

    def build_param_writer(self, prop_enum: Enum):
        param_id, _ = self.get_param_id_by_api_id(prop_enum)
        name = param_id.value
        self._prop_writer_map[
        prop_enum] = lambda inst, value, param_name=name: self.generic_param_setter(
            inst, param_name, value)

    @abstractmethod
    def _build_presenter(self):
        pass

    @abstractmethod
    def build_inspector(self) -> BlockPropInspector:
        pass

    def _build_prop_map(self):
        assert self._prop_map is None, "Internal Error: Called more than once"
        self._prop_map = {}
        self._deprecated_prop_map = {}

        for prop_id in self._supported_props:
            if prop_id.name in self._deprecated_prop_id_name2info:
                msg, is_usable = self._deprecated_prop_id_name2info[prop_id.name]
                self._insert_deprecated_prop(
                    prop_id, prop_id.value, False, msg, is_usable=is_usable)

                if not is_usable:
                    continue

            if self.is_pin_id(prop_id):
                self._build_pin_prop(prop_id)

            elif self.is_pin_invert_enable_id(prop_id):
                self._build_pin_invert_enable_prop(prop_id)

            elif prop_id.name == 'common_inst_name':
                self._insert_prop_map(
                    prop_id=prop_id,
                    api_prop_name=prop_id.value,
                    api_prop_type=BaseProp.PTypeId.STR,
                    option_list=None,
                    prop_cat=self.PropCat.none
                )

            else:
                self._build_param_prop(prop_id)

    def is_pin_id(self, prop_id: Enum):
        return prop_id.name.endswith('_pn')

    def is_pin_invert_enable_id(self, prop_id: Enum):
        return prop_id.name.endswith('_invert_en')

    def _build_pin_prop(self, pin_enum: Enum):
        assert self._prop_map is not None
        self._insert_prop_map(prop_id=pin_enum,
                                api_prop_name=pin_enum.value,
                                api_prop_type=BaseProp.PTypeId.STR,
                                option_list=None,
                                prop_cat=self.PropCat.none)

    def _build_pin_invert_enable_prop(self, invert_enable_enum: Enum):
        assert self._prop_map is not None
        self._insert_prop_map(prop_id=invert_enable_enum,
                                api_prop_name=invert_enable_enum.value,
                                api_prop_type=BaseProp.PTypeId.BOOL,
                                option_list=None,
                                prop_cat=self.PropCat.none)

    def _build_param_prop(self, param_enum: Enum):
        inst = self.db_inst
        assert isinstance(inst, LaneBasedItem), f'please build a dummy self.db_inst before calling _build_param_prop'

        param_id, is_common = self.get_param_id_by_api_id(param_enum)
        if is_common:
            inst = self.get_common_inst_by_lane(inst)
            assert inst is not None
        prop_info = inst.param_info.get_prop_info(param_id)
        assert prop_info is not None, f"failed to find prop_info for param {param_id.name}"
        param_name = param_id.name.removeprefix('ID_')

        if self.is_show_hidden:
            hidden_ops = []
        else:
            hidden_ops = self.get_hidden_param_values().get(param_name, [])

        setting = self.get_prop_id_str2api_options().get(param_enum.name, None)

        if isinstance(setting, list):
            setting = set(setting) - set(hidden_ops)
            setting = list(setting)

        self._insert_prop_map(
            prop_id=param_enum,
            api_prop_name=param_enum.value,
            api_prop_type=build_api_prop_type(prop_info.data_type),
            option_list=build_prop_option(setting, prop_info.data_type),
            prop_cat=self.PropCat.none
        )

    def get_param_id_by_api_id(self, api_id: Enum) -> Tuple[Enum, bool]:
        inst = self.db_inst
        assert inst is not None, f'please build a dummy self.db_inst before calling _build_param_prop'

        assert not self.is_pin_id(api_id)

        prop_name = api_id.name.removeprefix("ID_")
        is_common = False

        hw_param_class = self.get_hw_lane_param_id_class()

        if hasattr(hw_param_class, prop_name):
            param_id = hw_param_class[prop_name]
        elif hasattr(CommonCfgParamInfo.Id, prop_name):
            param_id = CommonCfgParamInfo.Id[prop_name]
            is_common = True
        elif hasattr(QuadDesignParamInfo.Id, prop_name):
            param_id = QuadDesignParamInfo.Id[prop_name]
            is_common = True
        else:
            param_id = inst.param_info.get_prop_id(prop_name)

        assert param_id is not None, f"Failed to find param {prop_name}"

        return param_id, is_common

    def translate(self, prop_id, value, is_db2api=True):
        trans_value = value
        prop_info = self._prop_map.get(prop_id, None)
        if prop_info is not None:
            if is_db2api:
                trans_value = self.prs.translate_db2api(prop_info, value)
            else:
                trans_value = self.prs.translate_api2db(prop_info, value)

        return trans_value

    # TODO: check after import all properties from ISF file
    def check_properties(self, inst):
        assert isinstance(inst, LaneBasedItem)

        err_msg_list: List = []

        return err_msg_list
