"""
Copyright (C) 2017-2021 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on March 25, 2021

@author: yasmin
"""

import os
import sys

from util.gen_util import freeze_it
from api_service.excp.design_excp import PTResUsedException
from util.excp import MsgLevel

from design.db import PeriDesign
from common_device.spi_flash.spi_flash_design import SPIFlash
from tx60_device.hio_res_service import HIOResService
from api_service.internal.int_gen_block import IntBlockAPI

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

@freeze_it
class IntSPIFlashAPI(IntBlockAPI):
    """
    Provides high level SPI Flash API related operation.
    """

    def __init__(self, is_verbose: bool = False):
        super(IntSPIFlashAPI, self).__init__(is_verbose)
        self.inst_name_prefix = "spi_flash"
        self.res_svc = HIOResService()

    def set_design(self, design: PeriDesign):
        super(IntSPIFlashAPI, self).set_design(design)
        self.blk_reg = self.design.get_block_reg(PeriDesign.BlockType.spi_flash)
        if design is not None and self.res_svc is not None:
            self.res_svc.build(design)

    def assign_resource(self, db_inst, res_name: str):
        """
        Override. SPI Flash uses shared dedicated resource.
        """
        if db_inst is not None:
            # Check if target resource is used
            if self.is_resource_used(res_name) and db_inst.get_device() != res_name:
                raise PTResUsedException(f'Resource is used: {res_name}', MsgLevel.warning)

            # Assign jtag resource
            self.blk_reg.assign_inst_device(db_inst, res_name)

    def is_resource_used(self, res_name: str) -> bool:
        """
        Override.
        """
        # Some resource are shared with GPIO. So, if any of the shared
        # resource is used, we consider its used
        if res_name != "" and res_name != "None":
            user_list = self.res_svc.get_spi_flash_reserved_resource_user()

            if len(user_list) > 0:
                # This indicate that the shared resource has been used
                return True
            else:
                inst = self.blk_reg.get_inst_by_device_name(res_name)
                if inst is not None:
                    return True

        return False

    def create_block(self, name, is_register=True):
        """
        Create a new block db instance

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :return: Block instance if successful else None
        ..notes:
          To support is_register, override this function.
        """

        # Overriding this because the base doesn't support is_register False
        # since it is block specific
        if is_register:
            block = super().create_block(name, is_register)

        else:
            block = SPIFlash(name)
            block.set_default_setting(None)

            # Need to have the pins first before we can automate the pin name
            block.generate_pin_name()

        return block