from __future__ import annotations

from typing import Union
from api_service.internal.int_gen_block import IntBlockAPI
from design.db import PeriDesign
from design.db_item import PeriDesignItem
from device.db_interface import DeviceDBService

from api_service.excp.design_excp import (
    PTAPIException,
    PTBlkCreateException,
    PTResUsedException,
    PTNameUsedException,
    PTResInvalidException,
)
from tx180_device.pll_ssc.design import PLLSSC, PLLSSCRegistry
from tx180_device.pll_ssc.calculator import PLLSSCCalculator as Calculator
from tx180_device.mipi_dphy_res_service import MIPIDPHYResService
from util.excp import MsgLevel


class IntPLLSSCAPI(IntBlockAPI):

    def __init__(self, is_verbose: bool=False):
        super().__init__(is_verbose=is_verbose)
        self._blkdev_service = None
        self.res_svc = MIPIDPHYResService()
        self.blk_reg: PLLSSCRegistry

    # Implement
    def set_design(self, design: PeriDesign):
        """
        Set design db

        :param design: design db instance
        """
        self.design = design

        blk_reg = self.design.get_block_reg(PeriDesign.BlockType.pll_ssc)
        assert isinstance(blk_reg, PLLSSCRegistry)
        self.blk_reg = blk_reg

        self.res_svc.build(design)

        if self.design.device_db is not None:
            dbi = DeviceDBService(self.design.device_db)
            self._blkdev_service = dbi.get_block_service(
                DeviceDBService.BlockType.MIPI_TX_ADV_PLL_SSC)

    def assign_resource(self, db_inst: PeriDesignItem, res_name: str):
        # For PLL_SSC, we need to use the ResSVC since the resource
        # is shared with MIPI TX
        assert self.blk_reg is not None

        if db_inst is not None:
            # Able to assign empty resource
            if res_name == "":
                return self.blk_reg.assign_inst_device(db_inst, res_name)

            if self.blk_reg.is_device_valid(res_name) is False:
                raise PTResInvalidException(f"{res_name} is not a valid resource", MsgLevel.error)

            # Check if target resource is used
            if self.res_svc.is_resource_used(res_name) and db_inst.get_device() != res_name:
                raise PTResUsedException(f"{res_name} is already used")

            # Assign block resource
            self.blk_reg.assign_inst_device(db_inst, res_name)

    def create_block(self, name, is_register=True):
        block = None
        # Check name valid or not
        if is_register:
            if self.blk_reg is None:
                raise PTAPIException('Internal Error, should call set_design first')

            block = self.blk_reg.get_inst_by_name(name)
            if block is not None:
                raise PTNameUsedException(f'PLL SSC {name} exists',
                                          MsgLevel.error)

            if name == "":
                inst_name = self.blk_reg.gen_unique_inst_name()
                if inst_name is None:
                    raise PTBlkCreateException(
                        "Fail to auto-generate unique PLL SSC name.",
                        MsgLevel.error)
            else:
                inst_name = name
        else:
            inst_name = name

        # Create block
        if is_register:
            block = self.blk_reg.create_instance(inst_name, auto_pin=True)
        else:
            calc = Calculator(PLLSSCRegistry.csvfile) # type: ignore
            block = PLLSSC(name=inst_name, apply_default=True, calculator=calc)

        return block
