from __future__ import annotations
import os

from typing import Union
from api_service.internal.int_gen_block import IntBlockAPI
from api_service.internal.int_mipi_dphy import PDDItemType
from common_device.mipi_dphy.mipi_dphy_design import MIPIDPhyRx, MIPIDPhyTx
from common_device.mipi.mipi_design import MIPIRegistry, MIPI
from design.db import PeriDesign
from design.db_item import PeriDesignItem
from device.db_interface import DeviceDBService

from api_service.excp.design_excp import PTAPIException, PTBlkCreateException, \
                                PTResUsedException, PTNameUsedException, PTResInvalidException
from tx180_device.mipi_dphy.design import MIPIHardDPHYRx, MIPIHardDPHYTx
from tx180_device.mipi_dphy.timing_calculator import MIPIHardDPHYTxTimingCalculator as TxCalculator
from tx180_device.mipi_dphy_res_service import MIPIDPHYResService
from util.app_setting import AppSetting
from util.excp import MsgLevel


class IntMIPIHardDPHYAPI(IntBlockAPI):

    def __init__(self, is_verbose: bool=False):
        super().__init__(is_verbose=is_verbose)
        self._tx_blkdev_service = None
        self._rx_blkdev_service = None
        self.res_svc = MIPIDPHYResService()

    # Implement
    def set_design(self, design: PeriDesign):
        """
        Set design db

        :param design: design db instance
        """
        self.design = design
        self.blk_reg = self.design.get_block_reg(
            PeriDesign.BlockType.mipi_hard_dphy)
        self.res_svc.build(design)

        if self.design.device_db is not None:
            dbi = DeviceDBService(self.design.device_db)
            self._tx_blkdev_service = dbi.get_block_service(
                DeviceDBService.BlockType.MIPI_TX_ADV)
            self._rx_blkdev_service = dbi.get_block_service(
                DeviceDBService.BlockType.MIPI_RX_ADV)

    def assign_tx_resource(self, db_inst: PeriDesignItem, res_name: str):
        assert self.blk_reg is not None

        if db_inst is not None:
            # Able to assign empty resource
            if res_name == "":
                return self.blk_reg.assign_inst_device(db_inst, res_name)

            if self.blk_reg.is_device_valid(res_name) is False:
                raise PTResInvalidException(f"{res_name} is not a valid resource", MsgLevel.error)

            # Check if target resource is used
            if self.res_svc.is_resource_used(res_name) and db_inst.get_device() != res_name:
                raise PTResUsedException(f"{res_name} is already used")

            # Assign block resource
            self.blk_reg.assign_inst_device(db_inst, res_name)

    def assign_resource(self, db_inst: PeriDesignItem, res_name: str):
        # For MIPI TX, we need to use the ResSVC since the resource
        # is shared with PLL_SSC
        if db_inst.ops_type == MIPI.OpsType.op_tx:
            self.assign_tx_resource(db_inst, res_name)

        else:
            super().assign_resource(db_inst, res_name)

    def create_block(self, name, is_register=True):
        return super().create_block(name, is_register=is_register)

    def __create_directional_block(
            self,
            name: str,
            is_register: bool,
            direction: PDDItemType.OpsType) -> Union[MIPIDPhyTx, MIPIDPhyRx]:
        assert direction in (PDDItemType.OpsType.op_rx, PDDItemType.OpsType.op_tx)

        block = None
        # Check name valid or not
        if is_register:
            if self.blk_reg is None:
                raise PTAPIException('Internal Error, should call set_design first')

            block = self.blk_reg.get_inst_by_name(name)
            if block is not None:
                raise PTNameUsedException(f'MIPI DPHY {name} exists',
                                          MsgLevel.error)

            if name == "":
                dir_type = None
                if direction == PDDItemType.OpsType.op_tx:
                    dir_type = 'tx'
                elif direction == PDDItemType.OpsType.op_rx:
                    dir_type = 'rx'

                inst_name = self.blk_reg.gen_unique_direct_inst_name(dir_type)
                if inst_name is None:
                    raise PTBlkCreateException(
                        "Fail to auto-generate unique MIPI DPHY name.",
                        MsgLevel.error)
            else:
                inst_name = name
        else:
            inst_name = name

        # Create block
        if is_register:
            if direction == PDDItemType.OpsType.op_tx:
                block = self.blk_reg.create_tx_instance(inst_name, auto_pin=True)
            elif direction == PDDItemType.OpsType.op_rx:
                block = self.blk_reg.create_rx_instance(inst_name, auto_pin=True)
        else:
            if direction == PDDItemType.OpsType.op_tx:
                setting = AppSetting()
                efxpt_home = setting.app_path[AppSetting.PathType.install]
                csvfile = os.path.normpath(efxpt_home + "/db/die/block_models/mipi_hard_dphy_tx_timing.csv")
                timing_calc = TxCalculator(csvfile)
                block = MIPIHardDPHYTx(name=inst_name, apply_default=True, timing_calc=timing_calc)

            elif direction == PDDItemType.OpsType.op_rx:
                block = MIPIHardDPHYRx(name=inst_name, apply_default=True)

        return block

    def create_tx_block(self, name: str, is_register: bool = True, **kwargs) -> MIPIDPhyTx:
        block = self.__create_directional_block(name, is_register, direction=PDDItemType.OpsType.op_tx)
        return block

    def create_rx_block(self, name: str, is_register: bool = True, **kwargs) -> MIPIDPhyRx:
        block = self.__create_directional_block(name, is_register, direction=PDDItemType.OpsType.op_rx)
        return block

