"""
 Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

 No portion of this code may be reused, modified or
 distributed in any way without the expressed written
 consent of Efinix Inc.

 Created on October 05, 2020

 @author: yasmin
"""

import os
import sys
import threading
from enum import Enum, auto
from typing import Union, Optional

import util.gui_util as gui_util
import util.gen_util as gen_util
from util.singleton_logger import Logger
from util.gen_util import freeze_it
from util.excp import MsgLevel

from device.db_interface import DeviceDBService

from design.db import PeriDesign
from design.db_item import PeriDesignItem, PeriDesignDirectItem

from common_device.mipi_dphy.mipi_dphy_design import MIPIDPhy, MIPIDPhyTx, MIPIDPhyRx

from tx60_device.hio_res_service import HIOResService

from api_service.excp.design_excp import PTBlkCreateException, PTResUsedException, \
    PTResInvalidException, PTNameUsedException
from api_service.internal.int_gen_block import IntBlockAPI


# Alias type name, for shorter name
PDDItemType=PeriDesignDirectItem

@freeze_it
class IntMIPIDPhyAPI(IntBlockAPI):
    """
    Provides high level MIPI DPHY API related operation.
    """

    def __init__(self, is_verbose: bool = False):
        super(IntMIPIDPhyAPI, self).__init__(is_verbose)
        self.tx_blkdev_service = None
        self.rx_blkdev_service = None
        self.res_svc = HIOResService()

    def set_design(self, design: PeriDesign):
        super(IntMIPIDPhyAPI, self).set_design(design)
        self.res_svc.build(design)
        self.blk_reg = self.design.get_block_reg(PeriDesign.BlockType.mipi_dphy)
        if self.design.device_db is not None:
            dbi = DeviceDBService(self.design.device_db)
            self.rx_blkdev_service = dbi.get_block_service(DeviceDBService.BlockType.HSIO_MIPI_DPHY_RX)
            self.tx_blkdev_service = dbi.get_block_service(DeviceDBService.BlockType.HSIO_MIPI_DPHY_TX)

    def create_block(self, name: str, is_register: bool = True) -> Optional[MIPIDPhy]:
        """
        Override. Create a MIPI DPHY block

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :return: MIPI DPHY if successful else None
        """
        return self.create_direct_block(name, is_register)

    def create_tx_block(self, name: str, is_register: bool = True, **kwargs) -> Optional[MIPIDPhy]:
        """
        Create a TX mipi. Default mode is data.

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :param kwargs: Dynamic parameter, mode = DATA_LANE or CLOCK_LANE
        :return: MIPI DPHY if successful else None
        """

        mode = None
        for key, value in kwargs.items():
            if key == "mode":
                mode = value

        mipi = self.create_direct_block(name, is_register, PDDItemType.OpsType.op_tx)
        if mode is not None:
            tx_info = mipi.tx_info
            if mode == "CLOCK_LANE":
                tx_info.mode = MIPIDPhyTx.ModeType.clock
            elif mode == "DATA_LANE":
                tx_info.mode = MIPIDPhyTx.ModeType.data

        return mipi

    def create_rx_block(self, name: str, is_register: bool = True, **kwargs) -> Optional[MIPIDPhy]:

        mode = None
        conn_type = None
        for key, value in kwargs.items():
            if key == "mode":
                mode = value
            elif key == "conn_type":
                conn_type = value

        mipi = self.create_direct_block(name, is_register, PDDItemType.OpsType.op_rx)
        if mode is not None:
            rx_info = mipi.rx_info
            if mode == "CLOCK_LANE":
                rx_info.mode = MIPIDPhyRx.ModeType.clock
            elif mode == "DATA_LANE":
                rx_info.mode = MIPIDPhyRx.ModeType.data

            if conn_type == "GCLK":
                rx_info.conn_type = MIPIDPhyRx.ConnType.gclk_conn
            elif conn_type == "RCLK":
                rx_info.conn_type = MIPIDPhyRx.ConnType.rclk_conn

            # Reset the static delay based on the selected mode
            if self.design is not None:
                rx_info.set_static_delay_default_setting(self.design.device_db)

        return mipi

    def create_tx_block_auto_name(self, is_register: bool = True) -> Optional[MIPIDPhy]:
        ops_type = PeriDesignDirectItem.OpsType.op_tx
        name = self.blk_reg.gen_unique_direct_inst_name(ops_type, prefix="mipi_tx_ln")
        if name is None:
            self.msg_svc.write(MsgLevel.error, "Fail to auto-generate unique mipi tx lane name")
            return None

        return self.create_direct_block(name, is_register, ops_type)

    def create_rx_block_auto_name(self, is_register: bool = True) -> Optional[MIPIDPhy]:
        ops_type = PeriDesignDirectItem.OpsType.op_rx
        name = self.blk_reg.gen_unique_direct_inst_name(ops_type, prefix="mipi_rx_ln")
        if name is None:
            self.msg_svc.write(MsgLevel.error, "Fail to generate unique mipi rx lane name")
            return None

        return self.create_direct_block(name, is_register, ops_type)

    def create_direct_block(self, name: str, is_register: bool = True,
                            direction: PDDItemType.OpsType = PDDItemType.OpsType.op_tx) -> Optional[MIPIDPhy]:

        mipi = None
        if is_register:
            if self.blk_reg is None:
                return None

            mipi = self.blk_reg.get_inst_by_name(name)
            if mipi is not None:
                msg = "MIPI Lane {} exists.".format(name)
                raise PTNameUsedException(msg, MsgLevel.error)

            if name == "":
                inst_name = self.blk_reg.gen_unique_inst_name()
                if inst_name is None:
                    msg = "Fail to auto-generate unique MIPI Lane name."
                    raise PTBlkCreateException(msg, MsgLevel.error)
            else:
                inst_name = name
        else:
            inst_name = name

        if is_register:
            if direction == PDDItemType.OpsType.op_tx:
                mipi = self.blk_reg.create_tx_instance(inst_name, auto_pin=True)
            elif direction == PDDItemType.OpsType.op_rx:
                mipi = self.blk_reg.create_rx_instance(inst_name, auto_pin=True)
        else:
            mipi = MIPIDPhy(inst_name)
            mipi.update_type(direction, self.design.device_db, auto_pin=True)

        return mipi

    def assign_resource(self, db_inst, res_name:str):
        """
        Override.
        """

        if db_inst is not None:
            if self.res_svc.is_non_gpio_resource(res_name) is False:
                raise PTResInvalidException(f"{res_name} is not a resource name for MIPI Lane", MsgLevel.error)

            # Check if useable resource
            if self.res_svc.is_device_valid(res_name) is False:
                raise PTResInvalidException(f"{res_name} is not a valid resource for MIPI Lane", MsgLevel.error)

            # Check if target resource is used
            if self.is_resource_used(res_name) and db_inst.get_device() != res_name:
                raise PTResUsedException(f"{res_name} is already used")

            # Assign block resource
            # print("IntBlockAPI:assign_resource {} to {}".format(res_name, db_inst.name))
            self.blk_reg.assign_inst_device(db_inst, res_name)

    def is_resource_used(self, res_name: str) -> bool:
        """
        Override. Check if the shared HSIO resource is used by any block type.
        """
        return self.res_svc.is_resourced_used(res_name, True)

