"""
 Copyright (C) 2017-2022 Efinix Inc. All rights reserved.

 No portion of this code may be reused, modified or
 distributed in any way without the expressed written
 consent of Efinix Inc.

 Created on October 06, 2020

 @author: yasmin
"""

from enum import Enum
from typing import Dict, Optional

from util.gen_util import freeze_it
from util.excp import MsgLevel

from device.db_interface import DeviceDBService
from common_device.lvds.lvds_design import LVDS, LVDSRegistry, LVDSRx, LVDSTx
from tx60_device.hio_res_service import HIOResService
from tx60_device.lvds.lvds_design_adv import (
    LVDSAdvance,
    LVDSTxAdvance,
    LVDSRxAdvance,
    LVDSRegistryAdvance
)


from design.db import PeriDesign
from design.db_item import PeriDesignDirectItem, PeriDesignRegistryDirectMixin

from api_service.internal.int_gen_block import IntBlockAPI
from api_service.excp.design_excp import (
    PTBlkCreateException,
    PTNameUsedException,
    PTResUsedException,
    PTResInvalidException
)


@freeze_it
class IntLVDSAPI(IntBlockAPI):
    """
    Provides high level advanced LVDS API related operation.
    """
    api2db_conn_map: Dict[str, Enum] = \
        {
            "NORMAL": LVDSRx.ConnType.normal_conn,
            "PLL_CLKIN": LVDSRx.ConnType.pll_clkin_conn,
        }  #: Mapping of connection type in API string to db enum type

    def __init__(self, is_verbose: bool = False):
        super().__init__(is_verbose)
        self.res_ctrl = None
        self.tx_blkdev_service = None
        self.rx_blkdev_service = None

        self.blk_type = PeriDesign.BlockType.lvds
        self.tx_type = DeviceDBService.BlockType.LVDS_TX
        self.rx_type = DeviceDBService.BlockType.LVDS_RX
        self.direct_item_class = LVDS

    def set_design(self, design: PeriDesign):
        super().set_design(design)
        self.blk_reg: Optional[LVDSRegistry] = self.design.get_block_reg(self.blk_type)
        assert self.blk_reg is not None

        if self.design.device_db is not None:
            dbi = DeviceDBService(self.design.device_db)
            self.rx_blkdev_service = dbi.get_block_service(self.rx_type)
            self.tx_blkdev_service = dbi.get_block_service(self.tx_type)

    def create_pll_input_clock_lvds(self, name: str, is_register: bool = True):
        """
        Create LVDS RX for PLL ref clock.

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :return: Advance LVDS, if successful else None
        """
        lvds = self.create_rx_block(name, is_register, rx_conn_type="PLL_CLKIN")

        return lvds

    def create_block(self, name: str, is_register: bool = True) -> Optional[LVDS]:
        """
        Override. Create LVDS block

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :return: LVDS, if successful else None
        """
        return self.create_direct_block(name, self.direct_item_class.OpsType.op_tx, is_register)

    def create_tx_block(self, name: str, is_register: bool = True, **kwargs) -> Optional[LVDS]:
        """
        Create a TX lvds. Default mode is data output.

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :param kwargs: Dynamic parameter, mode
        :return: LVDS if successful else None
        """

        lvds = self.create_direct_block(name, self.direct_item_class.OpsType.op_tx, is_register)
        if lvds is None:
            return None

        mode = kwargs.get("tx_mode", None)

        if mode is not None:
            self.set_tx_mode_by_str(lvds.tx_info, mode)

        return lvds

    def set_tx_mode_by_str(self, tx_info: LVDSTx, mode: str):
        if mode == "DATA":
            tx_info.mode = tx_info.ModeType.out
        elif mode == "CLKOUT":
            tx_info.mode = tx_info.ModeType.clkout

    def create_rx_block(self, name: str, is_register: bool = True, **kwargs) -> Optional[LVDS]:
        """
        Create a RX lvds. Default connection type is normal.

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :param kwargs: Dynamic parameter, rx_conn_type
        :return: LVDS if successful else None
        """
        lvds = self.create_direct_block(name, self.direct_item_class.OpsType.op_rx, is_register)

        if lvds is None:
            return None

        conn_type = kwargs.get("rx_conn_type", "")
        db_conn_type = self.api2db_conn_map.get(conn_type, None)

        if db_conn_type is not None:
            self.set_rx_conn_settings(lvds, db_conn_type)

        return lvds

    def set_rx_conn_settings(self, lvds: LVDS, db_conn_type):
        rx_info = lvds.rx_info
        assert rx_info is not None
        rx_info.conn_type = db_conn_type

    def create_tx_block_auto_name(self, is_register: bool = True) -> Optional[LVDS]:
        assert self.blk_reg is not None
        ops_type = PeriDesignDirectItem.OpsType.op_tx
        name = self.gen_inst_name("lvds_tx", ops_type)
        if name is None:
            self.msg_svc.write(MsgLevel.error, "Fail to auto-generate unique lvds tx name")
            return None

        return self.create_direct_block(name, ops_type, is_register)

    def create_rx_block_auto_name(self, is_register: bool = True) -> Optional[LVDS]:
        assert self.blk_reg is not None
        ops_type = PeriDesignDirectItem.OpsType.op_rx
        name = self.gen_inst_name("lvds_rx", ops_type)
        if name is None:
            self.msg_svc.write(MsgLevel.error, "Fail to auto-generate unique lvds rx name")
            return None

        return self.create_direct_block(name, ops_type, is_register)

    def gen_inst_name(self, lvds_prefix: str, ops_type: LVDS.OpsType) -> Optional[str]:
        assert self.blk_reg is not None
        return self.blk_reg.gen_unique_inst_name(prefix=lvds_prefix)

    def create_direct_block(self, name: str, direction: LVDS.OpsType,
                            is_register: bool = True):
        assert self.blk_reg is not None

        lvds = None
        if is_register:
            if self.blk_reg is None:
                return None

            lvds = self.blk_reg.get_inst_by_name(name)
            if lvds is not None:
                msg = "LVDS {} exists.".format(name)
                raise PTNameUsedException(msg, MsgLevel.error)

            if name == "":
                inst_name = self.blk_reg.gen_unique_inst_name()
                if inst_name is None:
                    msg = "Fail to auto-generate unique LVDS name."
                    raise PTBlkCreateException(msg, MsgLevel.error)
            else:
                inst_name = name
        else:
            inst_name = name

        if is_register:
            lvds = self.gen_register_lvds_by_direction(inst_name, direction)
        else:
            lvds = self.gen_default_lvds(inst_name, direction)

        return lvds

    def gen_register_lvds_by_direction(self, inst_name: str, direction: LVDS.OpsType):
        """
        Generate lvds using LVDSRegistry

        Args:
            inst_name (str): instance name
            direction (PeriDesignDirectItem.OpsType): Direction of LVDS

        Returns:
            LVDS: LVDS instance
        """
        assert self.blk_reg is not None

        if direction == self.direct_item_class.OpsType.op_tx:
            lvds = self.blk_reg.create_tx_instance(inst_name, auto_pin=True)
        elif direction == self.direct_item_class.OpsType.op_rx:
            lvds = self.blk_reg.create_rx_instance(inst_name, auto_pin=True)
        else:
            lvds = None
        return lvds

    def gen_default_lvds(self, inst_name: str, direction: LVDS.OpsType):
        """
        Generate lvds instance (not in LVDSRegistry)

        Args:
            inst_name (str): instance name
            direction (LVDS.OpsType): Direction of LVDS

        Returns:
            LVDS: LVDS instance
        """
        lvds = LVDS(inst_name)
        lvds.update_type(direction, self.design.device_db, auto_pin=True)
        return lvds

    def is_resource_used(self, res_name: str) -> bool:
        """
        Override. Check if the shared HSIO resource is used by any block type.
        """
        assert self.blk_reg is not None
        return self.blk_reg.is_device_used(res_name, is_full_check=False)


@freeze_it
class IntLVDSAdvAPI(IntLVDSAPI):
    """
    Provides high level advanced LVDS API related operation.
    """
    api2db_conn_map: Dict[str, Enum] = \
        {
            "NORMAL": LVDSRxAdvance.ConnType.normal_conn,
            "PLL_CLKIN": LVDSRxAdvance.ConnType.pll_clkin_conn,
            "PLL_EXTFB": LVDSRxAdvance.ConnType.pll_extfb_conn,
            "GCLK": LVDSRxAdvance.ConnType.gclk_conn,
            "RCLK": LVDSRxAdvance.ConnType.rclk_conn
        }  #: Mapping of connection type in API string to db enum type

    def __init__(self, is_verbose: bool = False):
        self.res_svc = HIOResService()
        super().__init__(is_verbose)

        self.blk_type = PeriDesign.BlockType.adv_lvds
        self.tx_type = DeviceDBService.BlockType.HSIO_LVDS_TX
        self.rx_type = DeviceDBService.BlockType.HSIO_LVDS_RX
        self.direct_item_class = PeriDesignDirectItem

    def set_design(self, design: PeriDesign):
        self.res_svc.build(design)
        super().set_design(design)

    def create_pll_input_clock_lvds(self, name: str, is_register: bool = True, is_bidir: bool=False):
        """
        Create LVDS RX for PLL ref clock. It can based standalone RX or BIDIR.

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :param is_bidir: True, create bidir instance else standalone rx
        :return: Advance LVDS, if successful else None
        """
        if is_bidir:
            lvds = self.create_bidir_block(name, is_register, rx_conn_type="PLL_CLKIN")
        else:
            lvds = super().create_pll_input_clock_lvds(name, is_register)

        return lvds

    def set_rx_conn_settings(self, lvds: LVDSAdvance, db_conn_type):
        rx_info: LVDSRxAdvance = lvds.rx_info
        assert rx_info is not None
        rx_info.conn_type = db_conn_type
        rx_info.sync_conn_type_config()

        # reset the static delay
        if self.design is not None:
            rx_info.set_static_delay_default_setting(self.design.device_db)

    def create_bidir_block(self, name: str, is_register: bool = True, **kwargs) -> Optional[LVDSAdvance]:
        """
        Create a BIDIR lvds. Default connection type is normal.

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :param kwargs: Dynamic parameter, mode, conn_type
        :return: LVDS if successful else None
        """
        lvds = self.create_direct_block(name, PeriDesignDirectItem.OpsType.op_bidir, is_register)

        if lvds is None:
            return None

        mode = kwargs.get("tx_mode", None)

        if mode is not None:
            self.set_tx_mode_by_str(lvds.tx_info, mode)

        conn_type = kwargs.get("rx_conn_type", "")
        db_conn_type = self.api2db_conn_map.get(conn_type, None)

        if db_conn_type is not None:
            self.set_rx_conn_settings(lvds, db_conn_type)

        return lvds

    def create_bidir_block_auto_name(self, is_register: bool = True) -> Optional[LVDSAdvance]:
        ops_type = PeriDesignDirectItem.OpsType.op_bidir
        name = self.gen_inst_name("lvds_bidir", ops_type)
        if name is None:
            self.msg_svc.write(MsgLevel.error, "Fail to auto-generate unique lvds bidir name")
            return None

        return self.create_direct_block(name, ops_type, is_register)

    def gen_inst_name(self, lvds_prefix: str, ops_type: PeriDesignDirectItem.OpsType) -> Optional[str]:
        assert self.blk_reg is not None
        assert isinstance(self.blk_reg, PeriDesignRegistryDirectMixin)
        return self.blk_reg.gen_unique_direct_inst_name(ops_type, prefix=lvds_prefix)

    def gen_register_lvds_by_direction(self, inst_name: str, direction: PeriDesignDirectItem.OpsType):
        """
        Override.
        """
        assert self.blk_reg is not None
        assert isinstance(self.blk_reg, LVDSRegistryAdvance)

        if direction == PeriDesignDirectItem.OpsType.op_tx:
                lvds = self.blk_reg.create_tx_instance(inst_name, auto_pin=True)
        elif direction == PeriDesignDirectItem.OpsType.op_rx:
            lvds = self.blk_reg.create_rx_instance(inst_name, auto_pin=True)
        elif direction == PeriDesignDirectItem.OpsType.op_bidir:
            lvds = self.blk_reg.create_bidir_instance(inst_name, auto_pin=True)
        else:
            lvds = None

        return lvds

    def gen_default_lvds(self, inst_name: str, direction: PeriDesignDirectItem.OpsType):
        """
        Override.
        """
        lvds = LVDSAdvance(inst_name)
        lvds.update_type(direction, self.design.device_db, auto_pin=True)
        return lvds

    def assign_resource(self, db_inst, res_name: str):
        """
        Override.
        """
        assert self.blk_reg is not None

        if db_inst is not None:
            if self.res_svc.is_non_gpio_resource(res_name) is False:
                raise PTResInvalidException(f"{res_name} is not a resource name for LVDS", MsgLevel.error)

            # Check if useable resource
            if self.res_svc.is_device_valid(res_name) is False:
                raise PTResInvalidException(f"{res_name} is not a valid resource for LVDS", MsgLevel.error)

            # Check if target resource is used
            if self.is_resource_used(res_name) and db_inst.get_device() != res_name:
                raise PTResUsedException(f"{res_name} is already used")

            # Assign block resource
            # print("IntBlockAPI:assign_resource {} to {}".format(res_name, db_inst.name))
            self.blk_reg.assign_inst_device(db_inst, res_name)

    def is_resource_used(self, res_name: str) -> bool:
        """
        Override. Check if the shared HSIO resource is used by any block type.
        """
        return self.res_svc.is_resourced_used(res_name, True)
