"""
Copyright (C) 2017-2019 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Oct 21, 2019

@author: yasmin
"""

import os
import sys
import threading
import abc
from enum import Enum
from typing import Union, Optional

import util.gen_util as gen_util

try:
    from enum import auto
except ImportError:
    auto = gen_util.auto_builder()

from util.singleton_logger import Logger
from util.gen_util import freeze_it
from util.excp import MsgLevel

from design.db import PeriDesign
from design.db_item import PeriDesignItem
import common_device.gpio.gpio_design as gpd
import common_device.lvds.lvds_utility as lvds_utility

from common_device.mipi_dphy.mipi_dphy_design import MIPIDPhy
import tx60_device.gpio.gpio_design_comp as comp_gpd
from tx60_device.lvds.lvds_design_adv import LVDSAdvance

from api_service.excp.design_excp import PTBlkCreateException, PTResUsedException, PTNameUsedException


class IntGPIOAPI:
    """
    Provides high level GPIO API related operation.

    The function here, may or may not match directly with user API.
    As much as possible, modules outside from peri_block should use the API
    here and only access raw registry level API when needed.

    Example user of this service are DesignExplorer and GPIO Import/Export.
    """

    class RegType(Enum):
        """
        A special property that capture different type of register.
        Editing this property touches one or more primitive/UI properties.
        """
        bypass = auto(),  #: Unregistered
        reg = auto(),  #: Registered
        inv_reg = auto(),  #: Registered, Inverted for Output and OE
        ddio = auto(),  #: Registered with enabled normal DDIO
        ddio_resync = auto()  #: Registered with enabled resync DDIO

    def __init__(self, design: PeriDesign = None, is_verbose: bool = False):
        self._is_verbose = is_verbose
        self._logger = Logger
        self._design = design
        self._gpio_reg = None
        self._res_ctrl = None

        if self._design is not None:
            self.set_design(self._design)

    @staticmethod
    def build_gpio_api(design: Optional[PeriDesign] = None, is_verbose: bool = False, gpio_block_type=None):
        """
        Build GPIO API based on supported block type. It will be detected automatically based on design.
        If design not available, specify the block type.

        :param design: Design db
        :param is_verbose: True, print detail messaging, False, normal
        :param gpio_block_type: GPIO block type, if design is None
        :return: GPIO API interface
        """
        if design is not None:
            block_type = design.gpio_type
        else:
            block_type = gpio_block_type

        if block_type is None:
            return None

        if block_type == PeriDesign.BlockType.comp_gpio:
            return IntGPIOCompAPI(design, is_verbose)
        else:
            return IntGPIOAPI(design, is_verbose)

    def __build_gpio_obj(self, inst_name):
        if self._design is not None:
            block_type = self._design.gpio_type
            if block_type == PeriDesign.BlockType.comp_gpio:
                return comp_gpd.GPIOComplex(inst_name, "")
            else:
                return gpd.GPIO(inst_name, "")

        return None

    def set_design(self, design: PeriDesign):
        """
        Set design after construction, useful for testing

        :param design: design db
        """
        self._design = design
        self._gpio_reg = self._design.gpio_reg
        self._res_ctrl = IntGPIOResourceAPI(design, self._is_verbose)

    def get_design(self):
        return self._design

    def get_gpio(self, name):
        return self._gpio_reg.get_gpio_by_name(name)

    def get_bus(self, name):
        return self._gpio_reg.get_bus_by_name(name)

    def get_bus_gpio(self, bus_inst):
        if bus_inst is not None:
            return bus_inst.get_member_list()

        return []

    def get_all_gpio_name(self):
        return self._gpio_reg.get_all_gpio_name()

    def get_all_gpio(self):
        return self._gpio_reg.get_all_gpio()

    def create_input_gpio(self, name: str, msb: Optional[int] = None, lsb: Optional[int] = None, is_register: bool = True):
        mode = gpd.GPIO.PadModeType.input
        if msb is None or lsb is None:
            gpio = self.__create_single_gpio(name, mode, is_register)
        else:
            gpio = self.__create_bus_gpio(name, msb, lsb, mode)

        return gpio

    def create_output_gpio(self, name: str, msb: Optional[int] = None, lsb: Optional[int] = None, is_register: bool = True):
        mode = gpd.GPIO.PadModeType.output
        if msb is None or lsb is None:
            gpio = self.__create_single_gpio(name, mode, is_register)
        else:
            gpio = self.__create_bus_gpio(name, msb, lsb, mode)

        return gpio

    def create_inout_gpio(self, name: str, msb: Optional[int] = None, lsb: Optional[int] = None, is_register: bool = True):
        mode = gpd.GPIO.PadModeType.inout
        if msb is None or lsb is None:
            gpio = self.__create_single_gpio(name, mode, is_register)
        else:
            gpio = self.__create_bus_gpio(name, msb, lsb, mode)

        return gpio

    def create_open_drain_output(self, name: str):
        gpio = self.create_inout_gpio(name)
        if gpio is not None:
            input_cfg = gpio.input
            # Input is unused
            input_cfg.name = ""
            input_cfg.name_ddio_lo = ""

            # Output is grounded
            output_cfg = gpio.output
            output_cfg.tied_option = output_cfg.TiedOptType.gnd

        return gpio

    def create_input_clock_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.gclk_conn
            input_cfg.is_register = False
            input_cfg.clock_name = ""
            input_cfg.is_clk_inverted = False

        return gpio

    def create_clockout_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_output_gpio(name, is_register=is_register)
        if gpio is not None:
            gpio.update_mode_config(gpd.GPIO.PadModeType.clkout)

        return gpio

    def create_global_control_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.gctrl_conn

        return gpio

    def create_pll_input_clock_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.pll_clkin_conn

        return gpio

    def create_mipi_input_clock_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.mipi_clkin_conn

        return gpio

    def create_pll_ext_fb_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.pll_extfb_conn

        return gpio

    def create_unused_gpio(self, name: str, is_register: bool = True):
        gpio = self.__create_single_gpio(name, gpd.GPIO.PadModeType.none, is_register)
        if gpio is not None:
            gpio.set_unused_mode_config()

        return gpio

    def __create_single_gpio(self, name: str, mode: gpd.GPIO.PadModeType, is_register: bool = True):
        """
        Create a single bit gpio

        :param name: Unique instance name
        :param mode: GPIO mode
        :param is_register: True, register in GPIORegistry else create standalone GPIO
        :return: GPIO if successful else None
        """

        if is_register:
            if self._gpio_reg is None:
                return None

            gpio = self._gpio_reg.get_gpio_by_name(name)
            if gpio is not None:
                msg = "GPIO {} exists.".format(name)
                raise PTNameUsedException(msg, MsgLevel.error)

            if name == "":
                inst_name = self._gpio_reg.gen_unique_gpio_name()
                if inst_name is None:
                    msg = "Fail to auto-generate unique GPIO name."
                    raise PTBlkCreateException(msg, MsgLevel.error)
            else:
                inst_name = name
        else:
            inst_name = name

        if is_register:
            gpio = self._gpio_reg.create_gpio(inst_name)
        else:
            # Device dependent defaults are only set when created through registry
            gpio = self.__build_gpio_obj(inst_name)
            gpio.io_standard = self._gpio_reg.get_io_standard_default()

        if gpio is not None:
            gpio.update_mode_config(mode)
            gpio.enable_ddio_support(self._design.is_support_ddio())

        return gpio

    def __create_bus_gpio(self, name: str, msb: int, lsb: int, mode: gpd.GPIO.PadModeType):

        if self._gpio_reg is None:
            return None

        if name == "":
            msg = "GPIO bus name is empty"
            raise PTBlkCreateException(msg, MsgLevel.error)
        else:
            inst_name = name

        if msb != lsb:  # Allows a case when it is the same e.g. [0:0]
            is_pass, msg = gpd.GPIOBus.check_bus_config(self._gpio_reg, name, msb, lsb)
            if not is_pass:
                raise PTBlkCreateException(msg, MsgLevel.error)

        # For now, too minor change to override
        if self._design.is_block_supported(PeriDesign.BlockType.comp_gpio):
            gpio_template = comp_gpd.GPIOComplex.build_default_template(name, mode, self._design.is_support_ddio())
        else:
            gpio_template = gpd.GPIO.build_default_template(name, mode, self._design.is_support_ddio())

        bus_inst = self._gpio_reg.create_bus(inst_name, msb, lsb, mode, gpio_template)

        return bus_inst

    def assign_resource(self, db_inst: PeriDesignItem, res_name: str):

        if db_inst is not None and isinstance(db_inst, gpd.GPIO):
            is_used = self._res_ctrl.is_resource_used(res_name)
            if is_used and db_inst.get_device() != res_name:
                msg = "{} is used".format(res_name)
                raise PTResUsedException(msg, MsgLevel.error)

            # Doesn't overwrite
            return self._res_ctrl.assign_to_instance(db_inst.name, res_name)

    def assign_pkg_pin(self, db_inst: PeriDesignItem, pin_name: str):

        if db_inst is not None and isinstance(db_inst, gpd.GPIO):
            is_used = self._res_ctrl.is_pkg_pin_used(pin_name)
            if is_used:
                msg = "{} is used".format(pin_name)
                raise PTResUsedException(msg, MsgLevel.error)

            # Doesn't overwrite
            return self._res_ctrl.assign_to_instance_by_pin(db_inst.name, pin_name)

    def is_resource_used(self, res_name: str):
        return self._res_ctrl.is_resource_used(res_name)

    def is_pkg_pin_used(self, pin_name: str):
        return self._res_ctrl.is_pkg_pin_used(pin_name)

    def get_pkg_pin(self, db_inst):
        if db_inst is None or db_inst.gpio_def == "":
            return ""

        return self._res_ctrl.get_pkg_pin(db_inst.gpio_def)

    def delete_gpio(self, name: str):

        if self._gpio_reg is None:
            return None

        gpio = self._gpio_reg.get_gpio_by_name(name)
        if gpio is not None:
            if gpio.is_lvds_gpio():
                self._design.delete_lvds_gpio(gpio)
            else:
                self._gpio_reg.delete_gpio(gpio.name)

    def delete_bus_gpio(self, name: str):

        if self._gpio_reg is None:
            return None

        gpio_bus = self._gpio_reg.get_bus_by_name(name)
        if gpio_bus is not None:
            if self._design.is_tesseract_design():
                self._gpio_reg.delete_bus(gpio_bus.get_name())
            else:
                self._design.delete_gpio_bus(gpio_bus.get_name())

    def resize_bus(self, gpio_bus: gpd.GPIOBus, msb: int, lsb: int, gpio_temp: gpd.GPIO = None):
        """
        Resize a bus given msb and lsb values

        :param gpio_bus: GPIO bus instance
        :param msb: MSB index
        :param lsb: LSB index
        :param gpio_temp: Configuration template for new gpio member, if bus is expanded
        """
        if gpio_bus is not None:
            if gpio_temp is None:
                gpio_template = gpio_bus.get_first_member()
            else:
                gpio_template = gpio_temp

            gpio_bus.resize(self._design, msb, lsb, gpio_template)

    def _apply_input_register_type(self, igpio, ireg_type):

        gp_input = igpio.input
        if gp_input is None:
            return False

        is_apply = False
        if ireg_type == self.RegType.bypass:
            gp_input.is_register = False
            gp_input.clock_name = ""
            gp_input.is_clk_inverted = False
            gp_input.ddio_type = gp_input.DDIOType.none
            gp_input.name_ddio_lo = ""
            is_apply = True

        elif ireg_type == self.RegType.reg:
            gp_input.is_register = True
            is_apply = True

        elif ireg_type == self.RegType.ddio:
            gp_input.is_register = True
            curr_ddio_type = gp_input.ddio_type
            gp_input.ddio_type = gp_input.DDIOType.normal
            if gp_input.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                igpio.generate_pin_name_input(igpio.name)
            is_apply = True

        elif ireg_type == self.RegType.ddio_resync:
            gp_input.is_register = True
            curr_ddio_type = gp_input.ddio_type
            gp_input.ddio_type = gp_input.DDIOType.resync
            if gp_input.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                igpio.generate_pin_name_input(igpio.name)
            is_apply = True

        return is_apply

    def set_input_register_type(self, gpio: Union[gpd.GPIO, gpd.GPIOBus], reg_type: RegType):

        if gpio is None:
            return

        if isinstance(gpio, gpd.GPIO):
            self._apply_input_register_type(gpio, reg_type)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            member_list = gpio_bus.get_member_list()
            for member in member_list:
                self._apply_input_register_type(member, reg_type)

    def _apply_output_register_type(self, igpio, ireg_type):
        gp_output = igpio.output
        if gp_output is None:
            return

        is_apply = False
        if ireg_type == self.RegType.bypass:
            gp_output.register_option = gp_output.RegOptType.none
            gp_output.clock_name = ""
            gp_output.is_clk_inverted = False
            gp_output.ddio_type = gp_output.DDIOType.none
            gp_output.name_ddio_lo = ""
            is_apply = True

        elif ireg_type == self.RegType.inv_reg:
            gp_output.register_option = gp_output.RegOptType.inv_register
            gp_output.ddio_type = gp_output.DDIOType.none
            gp_output.name_ddio_lo = ""
            is_apply = True

        elif ireg_type == self.RegType.reg:
            gp_output.register_option = gp_output.RegOptType.register
            is_apply = True

        elif ireg_type == self.RegType.ddio:
            gp_output.register_option = gp_output.RegOptType.register
            curr_ddio_type = gp_output.ddio_type
            gp_output.ddio_type = gp_output.DDIOType.normal
            if gp_output.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                igpio.generate_pin_name_output(igpio.name)
            is_apply = True

        elif ireg_type == self.RegType.ddio_resync:
            gp_output.register_option = gp_output.RegOptType.register
            curr_ddio_type = gp_output.ddio_type
            gp_output.ddio_type = gp_output.DDIOType.resync
            if gp_output.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                igpio.generate_pin_name_output(igpio.name)
            is_apply = True

        return is_apply

    def set_output_register_type(self, gpio: Union[gpd.GPIO, gpd.GPIOBus], reg_type: RegType):

        if gpio is None:
            return

        if isinstance(gpio, gpd.GPIO):
            self._apply_output_register_type(gpio, reg_type)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            member_list = gpio_bus.get_member_list()
            for member in member_list:
                self._apply_output_register_type(member, reg_type)

    def _apply_oe_register_type(self, igpio, ireg_type):
        gp_output = igpio.output
        if gp_output is None:
            return

        gp_oe = igpio.output_enable
        if gp_oe is None:
            return

        if ireg_type == self.RegType.bypass:
            gp_oe.is_register = False
            gp_oe.clock_name = ""
            gp_oe.is_clk_inverted = False

        elif ireg_type == self.RegType.reg:
            gp_oe.is_register = True
            gp_oe.clock_name = gp_output.clock_name

    def set_oe_register_type(self, gpio: Union[gpd.GPIO, gpd.GPIOBus], reg_type: RegType):

        if gpio is None:
            return

        if isinstance(gpio, gpd.GPIO):
            self._apply_oe_register_type(gpio, reg_type)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            member_list = gpio_bus.get_member_list()
            for member in member_list:
                self._apply_oe_register_type(member, reg_type)

    def _apply_outpath_clk_pin(self, igpio, clock_name: str):
        gp_output = igpio.output
        if gp_output is not None:
            gp_output.clock_name = clock_name

        gp_oe = igpio.output_enable
        if gp_oe is not None:
            gp_oe.clock_name = clock_name

    def _apply_outpath_clk_inversion(self, igpio, is_clk_inverted: bool):
        gp_output = igpio.output
        if gp_output is not None:
            gp_output.is_clk_inverted = is_clk_inverted

        gp_oe = igpio.output_enable
        if gp_oe is not None:
            gp_oe.is_clk_inverted = is_clk_inverted

    def set_outpath_clock_pin_name(self, gpio: Union[gpd.GPIO, gpd.GPIOBus], clock_name: str):
        if gpio is None:
            return

        if isinstance(gpio, gpd.GPIO):
            self._apply_outpath_clk_pin(gpio, clock_name)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            member_list = gpio_bus.get_member_list()
            for member in member_list:
                self._apply_outpath_clk_pin(member, clock_name)

    def set_output_clock_inverted(self, gpio: Union[gpd.GPIO, gpd.GPIOBus], is_clk_inv: bool):
        if gpio is None:
            return

        if isinstance(gpio, gpd.GPIO):
            self._apply_outpath_clk_inversion(gpio, is_clk_inv)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            member_list = gpio_bus.get_member_list()
            for member in member_list:
                self._apply_outpath_clk_inversion(member, is_clk_inv)

    def db2api_in_reg_type(self, igpio):
        gp_input = igpio.input
        if gp_input is None:
            return None

        if not gp_input.is_register:
            return self.RegType.bypass
        elif gp_input.ddio_type == gp_input.DDIOType.none:
            return self.RegType.reg
        elif gp_input.ddio_type == gp_input.DDIOType.normal:
            return self.RegType.ddio
        elif gp_input.ddio_type == gp_input.DDIOType.resync:
            return self.RegType.ddio_resync

        return None

    def get_input_register_type(self, gpio: Union[gpd.GPIO, gpd.GPIOBus]):

        if gpio is None:
            return None

        if isinstance(gpio, gpd.GPIO):
            return self.db2api_in_reg_type(gpio)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            first_gpio = gpio_bus.get_first_member()
            if first_gpio is not None:
                return self.db2api_in_reg_type(first_gpio)

        return None

    def db2api_out_reg_type(self, igpio):
        gp_output = igpio.output
        if gp_output is None:
            return None

        if gp_output.register_option == gp_output.RegOptType.none:
            return self.RegType.bypass
        elif gp_output.register_option == gp_output.RegOptType.inv_register:
            return self.RegType.inv_reg
        elif gp_output.register_option == gp_output.RegOptType.register:
            if gp_output.ddio_type == gp_output.DDIOType.none:
                return self.RegType.reg
            elif gp_output.ddio_type == gp_output.DDIOType.normal:
                return self.RegType.ddio
            elif gp_output.ddio_type == gp_output.DDIOType.resync:
                return self.RegType.ddio_resync

        return None

    def get_output_register_type(self, gpio: Union[gpd.GPIO, gpd.GPIOBus]):

        if gpio is None:
            return None

        if isinstance(gpio, gpd.GPIO):
            return self.db2api_out_reg_type(gpio)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            first_gpio = gpio_bus.get_first_member()
            if first_gpio is not None:
                return self.db2api_out_reg_type(first_gpio)

        return None

    def get_oe_register_type(self, gpio: Union[gpd.GPIO, gpd.GPIOBus]):

        if gpio is None:
            return None

        def get_register_type(igpio):
            gp_oe = igpio.output_enable
            if gp_oe is None:
                return None

            if not gp_oe.is_register:
                return self.RegType.bypass
            else:
                return self.RegType.reg

        if isinstance(gpio, gpd.GPIO):
            return get_register_type(gpio)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            first_gpio = gpio_bus.get_first_member()
            if first_gpio is not None:
                return get_register_type(first_gpio)

        return None

    def get_outpath_clock_name(self, gpio: Union[gpd.GPIO, gpd.GPIOBus]):
        if gpio is None:
            return None

        # The clock name in both output and oe should be
        # in-sync if they are non-empty.  This is based on
        # the gpio spec that only have one outclk pin.
        # Since GPIOOutput and GPIOutputEnable have their own
        # clock_name, therefore we use the GPIO to figure out
        # what the clock name is since there's no guarantee that
        # both are non-empty when used (oe clock name is saved only
        # when it is in inout mode)
        def get_clock_name(igpio):
            # To consider older design, we check both OE and OUT
            # and take either one if another is empty
            if igpio.output is not None and igpio.output_enable is not None:
                out_clk_name = igpio.output.clock_name
                oe_clk_name = igpio.output_enable.clock_name

                # Use the context with out as default
                if igpio.output.register_option != igpio.output.RegOptType.none:
                    return out_clk_name

                elif igpio.output_enable.is_register:
                    return oe_clk_name

                else:
                    # In the case if there's conflict (older API design),
                    # the rule check will be flagging it.
                    # Just take output
                    return out_clk_name

            elif igpio.output is not None:
                return igpio.output.clock_name
            elif igpio.output_enable is not None:
                return igpio.output_enable.clock_name

            return None

        if isinstance(gpio, gpd.GPIO):
            return get_clock_name(gpio)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            first_gpio = gpio_bus.get_first_member()
            if first_gpio is not None:
                return get_clock_name(first_gpio)

    def get_outpath_clock_inverted(self, gpio: Union[gpd.GPIO, gpd.GPIOBus]):
        if gpio is None:
            return None

        def get_clock_inverted(igpio):
            # To consider older design, we check both OE and OUT
            # and take either one if another is empty
            if igpio.output is not None and igpio.output_enable is not None:
                out_clk_inv = igpio.output.is_clk_inverted
                oe_clk_inv = igpio.output_enable.is_clk_inverted

                if out_clk_inv != oe_clk_inv:
                    # Use the context with out as default
                    if igpio.output.register_option != igpio.output.RegOptType.none:
                        return out_clk_inv

                    elif igpio.output_enable.is_register:
                        return oe_clk_inv

                    # In the case if there's conflict (older API design),
                    # the rule check will be flagging it.
                    # Just take output
                    return out_clk_inv

                else:
                    return out_clk_inv

            elif igpio.output is not None:
                return igpio.output.is_clk_inverted
            elif igpio.output_enable is not None:
                return igpio.output_enable.is_clk_inverted

            return None

        if isinstance(gpio, gpd.GPIO):
            return get_clock_inverted(gpio)
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            first_gpio = gpio_bus.get_first_member()
            if first_gpio is not None:
                return get_clock_inverted(first_gpio)

@freeze_it
class IntGPIOCompAPI(IntGPIOAPI):
    """
    Provides high level complex GPIO API related operation.
    """

    class RegType(Enum):
        """
        Override.
        """
        bypass = auto(),  #: Unregistered
        reg = auto(),  #: Registered
        inv_reg = auto(),  #: Registered, Inverted for Output and OE
        ddio = auto(),  #: Registered with enabled normal DDIO
        ddio_resync = auto()  #: Registered with enabled resync DDIO
        ddio_resync_pipe = auto()  #: Registered with enabled resync DDIO
        serial = auto()  #: Registered with enabled serialization

    def __init__(self, design: PeriDesign = None, is_verbose: bool = False):
        super().__init__(design, is_verbose)

    def set_design(self, design: PeriDesign):
        """
        Set design after construction, useful for testing

        :param design: design db
        """
        self._design = design
        self._gpio_reg = self._design.gpio_reg
        self._res_ctrl = IntHIOResourceAPI(design, self._is_verbose)

    def _apply_input_register_type(self, igpio, ireg_type) -> bool:
        """
        Override
        """

        gp_input = igpio.input
        if gp_input is None:
            return False

        is_apply = super()._apply_input_register_type(igpio, ireg_type)
        if is_apply is False:
            if ireg_type == self.RegType.ddio_resync_pipe:
                gp_input.is_register = True
                gp_input.is_serial = False
                curr_ddio_type = gp_input.ddio_type
                gp_input.ddio_type = gp_input.DDIOType.pipeline
                if gp_input.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                    igpio.generate_pin_name_input(igpio.name)
                is_apply = True

            elif ireg_type == self.RegType.serial:
                gp_input.is_register = True
                gp_input.is_serial = True
                curr_ddio_type = gp_input.ddio_type
                gp_input.ddio_type = gp_input.DDIOType.none
                if gp_input.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                    igpio.generate_pin_name_input(igpio.name)
                is_apply = True

        else:
            gp_input.is_serial = False

        return is_apply

    def _apply_output_register_type(self, igpio, ireg_type) -> bool:
        """
        Override
        """

        gp_output = igpio.output
        if gp_output is None:
            return False

        is_apply = super()._apply_output_register_type(igpio, ireg_type)
        if is_apply is False:
            if ireg_type == self.RegType.ddio_resync_pipe:
                gp_output.register_option = gp_output.RegOptType.register
                gp_output.is_serial = False
                curr_ddio_type = gp_output.ddio_type
                gp_output.ddio_type = gp_output.DDIOType.pipeline
                if gp_output.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                    igpio.generate_pin_name_output(igpio.name)
                is_apply = True

            elif ireg_type == self.RegType.serial:
                gp_output.register_option = gp_output.RegOptType.register
                gp_output.is_serial = True
                curr_ddio_type = gp_output.ddio_type
                gp_output.ddio_type = gp_output.DDIOType.none
                if gp_output.is_ddio_change_require_pin_name_regen(curr_ddio_type):
                    igpio.generate_pin_name_output(igpio.name)
                is_apply = True

        else:
            gp_output.is_serial = False

        return is_apply

    def db2api_in_reg_type(self, igpio):
        """
        Override
        """
        gp_input = igpio.input
        if gp_input is None:
            return None

        if not gp_input.is_register:
            return self.RegType.bypass

        reg_type = super().db2api_in_reg_type(igpio)
        if reg_type == self.RegType.reg:
            # Case of registered but serial is not checked
            if gp_input.is_serial:
                reg_type = self.RegType.serial

        if reg_type is None:  # Case of registered but unknown DDIO type
            if gp_input.ddio_type == gp_input.DDIOType.pipeline:
                reg_type = self.RegType.ddio_resync_pipe

        return reg_type

    def db2api_out_reg_type(self, igpio):
        """
        Override
        """
        gp_output = igpio.output
        if gp_output is None:
            return None

        reg_type = super().db2api_out_reg_type(igpio)
        if reg_type == self.RegType.reg:
            # Case of registered but serial is not checked
            if gp_output.is_serial:
                reg_type = self.RegType.serial

        if reg_type is None:  # Case of registered but unknown DDIO type
            if gp_output.ddio_type == gp_output.DDIOType.pipeline:
                reg_type = self.RegType.ddio_resync_pipe

        return reg_type

    def delete_gpio(self, name: str):

        if self._gpio_reg is None:
            return None

        gpio = self._gpio_reg.get_gpio_by_name(name)
        if gpio is not None:
            self._gpio_reg.delete_gpio(gpio.name)

    def create_vref_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.vref_conn

        return gpio

    def create_quad_pcie_perstn_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.pcie_perstn_conn

        return gpio

    def create_regional_input_clock_gpio(self, name: str, is_register: bool = True):
        gpio = self.create_input_gpio(name, is_register=is_register)
        if gpio is not None:
            input_cfg = gpio.input
            input_cfg.conn_type = input_cfg.ConnType.rclk_conn
            input_cfg.is_register = False
            input_cfg.clock_name = ""
            input_cfg.is_clk_inverted = False

        return gpio

    def is_differential_io(self, gpio: Union[gpd.GPIO, gpd.GPIOBus]):

        if gpio is None:
            return False

        if isinstance(gpio, gpd.GPIO):
            return gpio.is_gpio_differential_stl_io_std()
        elif isinstance(gpio, gpd.GPIOBus):
            gpio_bus = gpio
            first_gpio = gpio_bus.get_first_member()
            if first_gpio is not None:
                return first_gpio.is_gpio_differential_stl_io_std()

        return False


class IntGPIOResourceAPIInterface(metaclass=abc.ABCMeta):

    @abc.abstractmethod
    def __init__(self, design: PeriDesign = None, is_verbose: bool = False):
        pass

    @abc.abstractmethod
    def is_resource_used(self, res_name: str) -> bool:
        """
        Check if resource is used

        :return: True if used, else False
        """
        pass

    @abc.abstractmethod
    def is_pkg_pin_used(self, pin_name: str) -> bool:
        """
        Check if package pin is used

        :return: True if used, else False
        """
        pass

    @abc.abstractmethod
    def get_pkg_pin(self, res_name: str):
        pass

    @abc.abstractmethod
    def clear_resource(self, res_name: str) -> bool:
        """
        Clear resource assignment in design db for the target resource name

        :param res_name: Resource name
        :return: True, if succeed, else False
        """
        pass

    @abc.abstractmethod
    def clear_pin(self, pin_name: str):
        """
        Clear resource assignment in design db for the target pin name

        :param pin_name: Package pin name
        :return: True, if succeed, else False
        """
        pass

    @abc.abstractmethod
    def clear_resource_by_inst_name(self, inst_name: str) -> bool:
        """
        Clear resource assignment in design db for the target instance name

        :param inst_name: Instance name
        :return: True, if succeed, else False
        """
        pass

    @abc.abstractmethod
    def assign_to_resource(self, res_name: str, new_inst_name: str, is_overwrite: bool = False) -> bool:
        """
        Assign instance to target resource in design db

        :param is_overwrite:
        :param res_name: Target resource name
        :param new_inst_name: New instance name
        :return: True, if succeed, else False
        :raise: PTResUsedException, if the resource is used
        """
        pass

    @abc.abstractmethod
    def assign_to_instance(self, inst_name: str, new_res_name: str, is_overwrite: bool = False) -> bool:
        """
        Assign resource to target instance in design db

        :param is_overwrite:
        :param inst_name: Target instance name
        :param new_res_name: New resource name
        :return:True, if succeed else False
        :raise: PTResUsedException
        """
        pass

    @abc.abstractmethod
    def assign_to_instance_by_pin(self, inst_name: str, new_pin_name: str, is_overwrite: bool = False) -> bool:
        """
        Assign resource to target instance in design db

        :param is_overwrite:
        :param inst_name: Target instance name
        :param new_pin_name: New pin name
        :return:True, if succeed else False
        :raises: PTBlkEditException
        """
        pass

    @abc.abstractmethod
    def get_instance_from_resource(self, res_name: str):
        """
        Given a resource name, find its design db instance.
        The instance can be either a gpio instance or lvds instance.

        :param res_name: Target resource name
        :return: Instance tuple(block instance, lvds instance indicator flag)
        """
        pass

    @abc.abstractmethod
    def get_instance(self, inst_name: str):
        """
        Given an instance name, find its design db instance.
        The instance can be either a gpio instance or lvds instance

        :param inst_name: Target instance name
        :return: Instance tuple(block instance, lvds instance indicator flag)
        """
        pass


@freeze_it
class IntGPIOResourceAPI(IntGPIOResourceAPIInterface):
    """
    Resource API for Trion GPIO
    """

    def __init__(self, design: PeriDesign = None, is_verbose: bool = False):
        super().__init__()
        self.__is_verbose = is_verbose
        self.__logger = Logger
        self._write_lock = threading.RLock()  #: To control edit operation which is not reentrant.

        from api_service.internal.int_device import IOResourcePresenter
        self.__io_res = IOResourcePresenter()

        self.__design = design
        if self.__design is None:
            self.__gpio_reg = None
            self.__lvds_reg = None
        else:
            self.__gpio_reg = self.__design.gpio_reg
            self.__lvds_reg = self.__design.lvds_reg
            self.__io_res.build(self.__design)
            self.__io_res.build_pin_res_map()

    def is_resource_used(self, res_name: str):
        """
        Check if resource is used

        :return: True if used, else False
        """
        inst, is_lvds = self.get_instance_from_resource(res_name)
        if inst is not None:
            if not is_lvds:
                gpio = self.__gpio_reg.get_gpio_by_device_name(res_name)
                if gpio is not None:
                    return True
            else:
                lvds_gpio = self.__lvds_reg.get_inst_by_device_name(res_name)
                if lvds_gpio is not None:
                    return True

        return False

    def is_pkg_pin_used(self, pin_name: str):
        """
        Check if package pin is used

        :return: True if used, else False
        """
        res_name = self.__io_res.get_resource_by_pin(pin_name)
        return self.is_resource_used(res_name)

    def get_pkg_pin(self, res_name: str):
        return self.__io_res.get_pin_by_resource(res_name)

    def clear_resource(self, res_name: str):
        """
        Clear resource assignment in design db for the target resource name

        :param res_name: Resource name
        :return: True, if succeed, else False
        """

        with self._write_lock:
            inst, is_lvds = self.get_instance_from_resource(res_name)
            if inst is not None and not is_lvds:
                if inst.is_lvds_gpio():
                    self.__design.reset_lvds_gpio_device(inst)
                    inst.mark_lvds_gpio(False)
                else:
                    self.__gpio_reg.assign_gpio_device(inst, "")

                return True

        return False

    def clear_pin(self, pin_name: str):
        """
        Clear resource assignment in design db for the target pin name

        :param pin_name: Package pin name
        :return: True, if succeed, else False
        """

        with self._write_lock:
            res_name = self.__io_res.get_resource_by_pin(pin_name)
            return self.clear_resource(res_name)

    def clear_resource_by_inst_name(self, inst_name: str):
        """
        Clear resource assignment in design db for the target instance name

        :param inst_name: Instance name
        :return: True, if succeed, else False
        """

        with self._write_lock:
            inst = self.__gpio_reg.get_gpio_by_name(inst_name)
            if inst is not None:
                if inst.is_lvds_gpio():
                    self.__design.reset_lvds_gpio_device(inst)
                    inst.mark_lvds_gpio(False)
                else:
                    self.__gpio_reg.assign_gpio_device(inst, "")

                return True

        return False

    def assign_to_resource(self, res_name: str, new_inst_name: str, is_overwrite: bool = False):
        """
        Assign instance to target resource in design db

        :param is_overwrite:
        :param res_name: Target resource name
        :param new_inst_name: New instance name
        :return: True, if succeed, else False
        :raise: PTResUsedException, if the resource is used
        """
        with self._write_lock:
            is_lvds_res = False
            lvds_type = lvds_utility.get_instance_lvds_type(res_name, self.__design.device_db)
            if lvds_type is not None:
                is_lvds_res = True

            if is_lvds_res:
                is_assigned = self.__apply_lvds_io(res_name, new_inst_name, is_overwrite)
            else:
                is_assigned = self.__apply_gpio(res_name, new_inst_name, is_overwrite)

        return is_assigned

    def assign_to_instance(self, inst_name: str, new_res_name: str, is_overwrite: bool = False):
        """
        Assign resource to target instance in design db

        :param is_overwrite:
        :param inst_name: Target instance name
        :param new_res_name: New resource name
        :return:True, if succeed else False
        :raise: PTResUsedException
        """

        with self._write_lock:
            is_lvds_res = False
            lvds_type = lvds_utility.get_instance_lvds_type(new_res_name, self.__design.device_db)
            if lvds_type is not None:
                is_lvds_res = True

            if is_lvds_res:
                is_assigned = self.__apply_lvds_io_resource_to_instance(inst_name, new_res_name, is_overwrite)
            else:
                is_assigned = self.__apply_gpio_resource_to_instance(inst_name, new_res_name, is_overwrite)

        return is_assigned

    def assign_to_instance_by_pin(self, inst_name: str, new_pin_name: str, is_overwrite: bool = False):
        """
        Assign resource to target instance in design db

        :param is_overwrite:
        :param inst_name: Target instance name
        :param new_pin_name: New pin name
        :return:True, if succeed else False
        :raises: PTBlkEditException
        """

        res_name = self.__io_res.get_resource_by_pin(new_pin_name)
        return self.assign_to_instance(inst_name, res_name, is_overwrite)

    def __apply_gpio_resource_to_instance(self, inst_name: str, assigned_res_name: str, is_overwrite: bool):
        """
        Apply resource to target gpio instance in design db

        :param inst_name: Target instance name
        :param assigned_res_name: Target gpio resource name
        :return: True, assigned, else False, fail to assign
        """

        source_gpio = self.__gpio_reg.get_gpio_by_device_name(assigned_res_name)
        is_same_inst = False
        if source_gpio is not None:
            if source_gpio.name == inst_name:
                is_same_inst = True

            is_used = False
            if not is_same_inst:
                is_used = True
                if is_overwrite:
                    msg = "The resource is already assigned to {}. {}'s resource will be cleared." \
                        .format(source_gpio.name, source_gpio.name)
                    self.__logger.debug(msg)
                else:
                    msg = "The resource is already assigned to {}.".format(source_gpio.name)
                    raise PTResUsedException(msg, MsgLevel.warning)

            # If the device is used, reset the gpio that it is currently
            # assigned to
            if is_used is True:
                # reset the device on source gpio
                # This event should be expected as reaction to primary
                # signal, no broadcast
                self.__gpio_reg.assign_gpio_device(source_gpio, "")

        # Set the new gpio device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(inst_name)
        if new_inst is not None:
            self.clear_resource_by_inst_name(new_inst)
            self.__gpio_reg.assign_gpio_device(new_inst, assigned_res_name)
            new_inst.mark_lvds_gpio(False)

        return True

    # pylint: disable=invalid-name
    def __apply_lvds_io_resource_to_instance(self, inst_name: str, assigned_res_name: str, is_overwrite: bool):
        """
        Apply lvds resource to target gpio instance in design db

        :param inst_name: Target instance name
        :param assigned_res_name: Target lvds resource name
        :return: True, assigned, else False, fail to assign
        """
        if self.__lvds_reg is None:
            return False

        source_inst, is_source_inst_lvds = self.get_instance_from_resource(assigned_res_name)
        is_same_inst = False

        if source_inst is not None:
            is_device_used = True
            if source_inst.name == inst_name:
                is_same_inst = True

        else:
            is_device_used = False

        if is_device_used and not is_same_inst:
            is_used = True

            if is_source_inst_lvds:
                if not is_overwrite:
                    msg = "The resource is already assigned to {} in lvds mode".format(source_inst.name)
                    raise PTResUsedException(msg, MsgLevel.error)

            # If the device is used, reset the gpio that it is currently assigned to
            if is_used is True:

                msg = "The resource is already assigned to {}. {}'s resource will be cleared." \
                    .format(source_inst.name, source_inst.name)
                self.__logger.warning(msg)

                # reset the device on source gpio
                if source_inst is None:
                    if is_source_inst_lvds:
                        self.__logger.warning("Cannot find used lvds in registery : {}".format(assigned_res_name))
                    else:
                        self.__logger.warning("Cannot find used gpio in registery : {}".format(assigned_res_name))

                else:
                    if is_source_inst_lvds:
                        self.__design.reset_lvds_gpio_device(source_inst)
                    else:
                        # This event should be expected as reaction to primary
                        # signal, no broadcast
                        self.__gpio_reg.assign_gpio_device(source_inst, "")

        # Set the new lvds device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(inst_name)
        if new_inst is not None:
            self.clear_resource_by_inst_name(new_inst)
            self.__design.assign_lvds_gpio_device(new_inst, assigned_res_name)
            new_inst.mark_lvds_gpio()

        return True

    def get_instance_from_resource(self, res_name: str):
        """
        Given a resource name, find its design db instance.
        The instance can be either a gpio instance or lvds instance.

        :param res_name: Target resource name
        :return: Instance tuple(block instance, lvds instance indicator flag)
        """
        with self._write_lock:
            if res_name == "":
                return None, False

            is_lvds = False
            inst = self.__gpio_reg.get_gpio_by_device_name(res_name)
            if inst is None:
                # It could be an lvds resource, check if it used in lvds mode
                if self.__lvds_reg is not None and self.__lvds_reg.is_lvds_mode_used(res_name):
                    lvds_mode_base_res_name = self.__design.device_db.get_base_ins_name(res_name)
                    inst = self.__lvds_reg.get_inst_by_device_name(lvds_mode_base_res_name)
                    if inst is not None:
                        is_lvds = True

            return inst, is_lvds

    def get_instance(self, inst_name: str):
        """
        Given an instance name, find its design db instance.
        The instance can be either a gpio instance or lvds instance

        :param inst_name: Target instance name
        :return: Instance tuple(block instance, lvds instance indicator flag)
        """
        with self._write_lock:
            if inst_name == "":
                return None, False

            is_lvds = False
            inst = self.__gpio_reg.get_gpio_by_name(inst_name)
            if inst is None:
                # It could be an lvds resource, check if it used in lvds mode
                if self.__lvds_reg is not None:
                    inst = self.__lvds_reg.get_inst_by_name(inst_name)
                    if inst is not None:
                        is_lvds = True

            return inst, is_lvds

    def __apply_gpio(self, res_name: str, assigned_inst_name: str, is_overwrite: bool = False):
        """
        Apply instance to target gpio resource in design db

        :param res_name: Target gpio resource name
        :param assigned_inst_name: Target instance name
        :return: True, assigned, else False, fail to assign
        """

        # Check current assigned instance to prevent multiple signals
        source_gpio = self.__gpio_reg.get_gpio_by_device_name(res_name)
        is_same_inst = False
        if source_gpio is not None:
            if source_gpio.name == assigned_inst_name:
                is_same_inst = True

        # Check db rather than check UI for status
        if self.__gpio_reg.is_gpio_device_used(res_name) and not is_same_inst:
            is_used = True
            if not is_overwrite:
                msg = "The resource is already assigned to {}. {}'s resource will be cleared." \
                    .format(source_gpio.name, source_gpio.name)
                raise PTResUsedException(msg, MsgLevel.warning)

            # If the device is used, reset the gpio that it is currently assigned to
            if is_used is True:

                # reset the device on source gpio
                if source_gpio is None:
                    self.__logger.warning("Cannot find used gpio in registery : {}".format(res_name))
                else:
                    # This event should be expected as reaction to primary
                    # signal, no broadcast
                    self.__gpio_reg.assign_gpio_device(source_gpio, "")

        # Set the new gpio device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(assigned_inst_name)
        if new_inst is not None:
            self.__gpio_reg.assign_gpio_device(new_inst, res_name)
            new_inst.mark_lvds_gpio(False)

        return True

    def __apply_lvds_io(self, res_name: str, assigned_inst_name: str, is_overwrite: bool = False):
        """
        Apply instance to target lvds resource in design db

        :param res_name: Target lvds resource name
        :param assigned_inst_name: Target instance name
        :return: True, assigned, else False, fail to assign
        """

        if self.__lvds_reg is None:
            return False

        # Check current assigned instance to prevent multiple signals
        # Instance can be gpio or lvds instance
        source_inst, is_source_inst_lvds = self.get_instance_from_resource(res_name)
        is_same_inst = False
        if source_inst is not None:
            is_device_used = True
            if source_inst.name == assigned_inst_name:
                is_same_inst = True
        else:
            is_device_used = False

        if is_device_used and not is_same_inst:
            is_used = True
            if not is_overwrite:
                msg = "The resource is already assigned to {} in lvds mode. {}'s resource will be cleared." \
                    .format(source_inst.name, source_inst.name)
                raise PTResUsedException(msg, MsgLevel.warning)

            # If the device is used, reset the gpio that it is currently assigned to
            if is_used is True:

                # reset the device on source gpio
                if source_inst is None:
                    if is_source_inst_lvds:
                        self.__logger.warning("Cannot find used lvds in registery : {}".format(res_name))
                    else:
                        self.__logger.warning("Cannot find used gpio in registery : {}".format(res_name))

                else:
                    # This event should be expected as reaction to primary
                    # signal, no broadcast
                    if is_source_inst_lvds:
                        self.__design.reset_lvds_gpio_device(source_inst)
                    else:
                        self.__gpio_reg.assign_gpio_device(source_inst, "")

        # Set the new lvds device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(assigned_inst_name)
        self.__design.assign_lvds_gpio_device(new_inst, res_name)
        new_inst.mark_lvds_gpio()

        return True


class IntHIOResourceAPI(IntGPIOResourceAPIInterface):
    """
    Resource API for Titanium GPIO
    """

    def __init__(self, design: PeriDesign = None, is_verbose: bool = False):
        super().__init__()

        self.__is_verbose = is_verbose
        self._write_lock = threading.RLock()  #: To control edit operation which is not reentrant.

        from tx60_device.hio_res_service import HIOResService
        self.__io_res = HIOResService()
        self.__pin_res_svc = None

        self.__design = design
        if self.__design is None:
            self.__gpio_reg = None
        else:
            self.__gpio_reg = self.__design.gpio_reg

            self.__io_res.build(self.__design)
            self.__io_res.build_pin_res_svc()
            self.__pin_res_svc = self.__io_res.pin_res_svc

    def is_resource_used(self, res_name: str):
        """
        Check if resource is used

        :return: True if used, else False
        """
        return self.__io_res.is_resourced_used(res_name, True)

    def is_pkg_pin_used(self, pin_name: str):
        """
        Check if package pin is used

        :return: True if used, else False
        """
        res_name = self.__pin_res_svc.get_resource_by_pin(pin_name)
        if res_name == "" and self.__is_verbose:
            print(f"Fail to find GPIO resource for package pin = {pin_name}")

        return self.is_resource_used(res_name)

    def get_pkg_pin(self, res_name: str):
        return self.__pin_res_svc.get_pin_by_resource(res_name)

    def clear_resource(self, res_name: str):
        """
        Clear resource assignment in design db for the target resource name

        :param res_name: Resource name
        :return: True, if succeed, else False
        """

        with self._write_lock:
            inst, is_non_gpio = self.get_instance_from_resource(res_name)
            if inst is not None and is_non_gpio is False:
                self.__gpio_reg.assign_gpio_device(inst, "")
                inst.mark_hsio_gpio(False)
                return True

            if inst is None and self.__is_verbose:
                print(f"Fail to clear GPIO resource, unknown resource = {res_name}")

        return False

    def clear_pin(self, pin_name: str):
        """
        Clear resource assignment in design db for the target pin name

        :param pin_name: Package pin name
        :return: True, if succeed, else False
        """

        with self._write_lock:
            res_name = self.__pin_res_svc.get_resource_by_pin(pin_name)
            return self.clear_resource(res_name)

    def clear_resource_by_inst_name(self, inst_name: str):
        """
        Clear resource assignment in design db for the target instance name

        :param inst_name: Instance name
        :return: True, if succeed, else False
        """

        with self._write_lock:
            inst = self.__gpio_reg.get_gpio_by_name(inst_name)
            if inst is not None:
                return self.clear_resource(inst.gpio_def)
            elif self.__is_verbose:
                print(f"Fail to clear GPIO resource, unknown GPIO instance = {inst_name}")

        return False

    def assign_to_resource(self, res_name: str, new_inst_name: str, is_overwrite: bool = False):
        """
        Assign instance to target resource in design db

        :param is_overwrite:
        :param res_name: Target resource name
        :param new_inst_name: New instance name
        :return: True, if succeed, else False
        :raise: PTResUsedException, if the resource is used
        """
        with self._write_lock:
            if self.__io_res.is_hsio_gpio_resource(res_name):
                is_assigned = self.__apply_hsio_io(res_name, new_inst_name, is_overwrite)
            else:
                is_assigned = self.__apply_gpio(res_name, new_inst_name, is_overwrite)

        return is_assigned

    def assign_to_instance(self, inst_name: str, new_res_name: str, is_overwrite: bool = False):
        """
        Assign resource to target instance in design db

        :param is_overwrite:
        :param inst_name: Target instance name
        :param new_res_name: New resource name
        :return:True, if succeed else False
        :raise: PTResUsedException
        """
        with self._write_lock:
            if self.__io_res.is_hsio_gpio_resource(new_res_name):
                is_assigned = self.__apply_hsio_io_resource_to_instance(inst_name, new_res_name, is_overwrite)
            else:
                is_assigned = self.__apply_gpio_resource_to_instance(inst_name, new_res_name, is_overwrite)

        return is_assigned

    def assign_to_instance_by_pin(self, inst_name: str, new_pin_name: str, is_overwrite: bool = False):
        """
        Assign resource to target instance in design db

        :param is_overwrite:
        :param inst_name: Target instance name
        :param new_pin_name: New pin name
        :return:True, if succeed else False
        :raises: PTBlkEditException
        """
        res_name = self.__pin_res_svc.get_resource_by_pin(new_pin_name)
        return self.assign_to_instance(inst_name, res_name, is_overwrite)

    def get_instance_from_resource(self, res_name: str):
        """
        Given a resource name, find its design db instance.
        The instance can be either a gpio instance or lvds instance.

        :param res_name: Target resource name
        :return: Instance tuple(block instance, non-gpio instance indicator flag)
        """
        with self._write_lock:
            if res_name == "":
                return None, False

            inst = None
            is_non_gpio = False
            user_list = self.__io_res.find_resource_user(res_name)
            if len(user_list) > 0:
                inst = user_list[0]
                if isinstance(inst, LVDSAdvance) or isinstance(inst, MIPIDPhy):
                    is_non_gpio = True
                elif isinstance(inst, comp_gpd.GPIOComplex):
                    for user_inst in user_list:
                        if user_inst.gpio_def == res_name:
                            return user_inst, False

                    return None, False

            return inst, is_non_gpio


    def get_instance(self, inst_name: str):
        """
        Given an instance name, find its design db instance.
        The instance can be either a gpio instance or lvds instance or mipi dphy instance

        :param inst_name: Target instance name
        :return: Instance tuple(block instance, non-gpio instance indicator flag)
        """
        with self._write_lock:
            if inst_name == "":
                return None, False

            is_non_gpio = False
            inst = self.__gpio_reg.get_gpio_by_name(inst_name)
            if inst is None:
                non_gpio_reg = [self.__io_res.lvds_reg, self.__io_res.mipi_dph_reg]
                for reg in non_gpio_reg:
                    inst = reg.get_inst_by_name(inst_name)
                    if inst is not None:
                        is_non_gpio = True
                        break

            return inst, is_non_gpio

    def __apply_gpio(self, res_name: str, assigned_inst_name: str, is_overwrite: bool = False):
        """
        Apply instance to target gpio (hvio) resource in design db

        :param res_name: Target gpio resource name
        :param assigned_inst_name: Target instance name
        :return: True, assigned, else False, fail to assign
        """

        # Check current assigned instance to prevent multiple signals
        source_gpio = self.__gpio_reg.get_gpio_by_device_name(res_name)
        is_same_inst = False
        if source_gpio is not None and source_gpio.name == assigned_inst_name:
                is_same_inst = True

        # Check db rather than check UI for status
        if self.__gpio_reg.is_gpio_device_used(res_name) and not is_same_inst:
            is_used = True
            if not is_overwrite:
                msg = "The GPIO resource is already assigned to {}. {}'s resource will be cleared." \
                    .format(source_gpio.name, source_gpio.name)
                raise PTResUsedException(msg, MsgLevel.warning)

            # If the device is used, reset the gpio that it is currently assigned to
            if is_used is True:

                # reset the device on source gpio
                if source_gpio is None:
                    if self.__is_verbose:
                        msg = f"Cannot find used GPIO in registry : {res_name}"
                        print(msg)
                else:
                    # This event should be expected as reaction to primary
                    # signal, no broadcast
                    self.__gpio_reg.assign_gpio_device(source_gpio, "")

        # Set the new gpio device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(assigned_inst_name)
        if new_inst is not None:
            self.__gpio_reg.assign_gpio_device(new_inst, res_name)
            new_inst.mark_hsio_gpio(False)

        return True

    def __apply_hsio_io(self, res_name: str, assigned_inst_name: str, is_overwrite: bool = False):
        """
        Apply instance to target hsio resource in design db

        :param res_name: Target lvds resource name
        :param assigned_inst_name: Target instance name
        :return: True, assigned, else False, fail to assign
        """

        lvds_reg = self.__io_res.lvds_reg
        if lvds_reg is None:
            return False

        # Check current assigned instance to prevent multiple signals
        # Instance can be gpio or lvds instance or mipi dphy
        source_inst, is_source_non_gpio = self.get_instance_from_resource(res_name)
        is_same_inst = False
        if source_inst is not None:
            is_device_used = True
            if source_inst.name == assigned_inst_name:
                is_same_inst = True
        else:
            is_device_used = False

        if is_device_used and not is_same_inst:
            is_used = True
            if not is_overwrite:
                msg = "The resource is already assigned to {}. {}'s resource will be cleared." \
                    .format(source_inst.name, source_inst.name)
                raise PTResUsedException(msg, MsgLevel.warning)

            # If the device is used, reset the gpio that it is currently assigned to
            if is_used is True:

                # reset the device on source gpio
                if source_inst is None:
                    if self.__is_verbose:
                        if is_source_non_gpio:
                            print(f"Cannot find used LVDS or MIPI Lane in registry : {res_name}")
                        else:
                            print("Cannot find used GPIO in registry : {}".format(res_name))

                else:
                    # This event should be expected as reaction to primary
                    # signal, no broadcast
                    self.__gpio_reg.assign_gpio_device(source_inst, "")

        # Set the new hsio device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(assigned_inst_name)
        self.__design.assign_gpio_device(new_inst, res_name)
        new_inst.mark_hsio_gpio()

        return True

    def __apply_gpio_resource_to_instance(self, inst_name: str, assigned_res_name: str, is_overwrite: bool):
        """
        Apply gpio (hvio) resource to target gpio instance in design db

        :param inst_name: Target instance name
        :param assigned_res_name: Target gpio resource name
        :return: True, assigned, else False, fail to assign
        """

        source_gpio = self.__gpio_reg.get_gpio_by_device_name(assigned_res_name)
        is_same_inst = False
        if source_gpio is not None:
            if source_gpio.name == inst_name:
                is_same_inst = True

            is_used = False
            if not is_same_inst:
                is_used = True
                if is_overwrite:
                    if self.__is_verbose:
                        msg = "The GPIO resource is already assigned to {}. {}'s resource will be cleared." \
                            .format(source_gpio.name, source_gpio.name)
                        print(msg)
                else:
                    msg = "The GPIO resource is already assigned to {}.".format(source_gpio.name)
                    raise PTResUsedException(msg, MsgLevel.warning)

            # If the device is used, reset the gpio that it is currently
            # assigned to
            if is_used is True:
                # reset the device on source gpio
                # This event should be expected as reaction to primary
                # signal, no broadcast
                self.__gpio_reg.assign_gpio_device(source_gpio, "")

        # Set the new gpio device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(inst_name)
        if new_inst is not None:
            self.__gpio_reg.assign_gpio_device(new_inst, assigned_res_name)
            new_inst.mark_hsio_gpio(False)

        return True

    # pylint: disable=invalid-name
    def __apply_hsio_io_resource_to_instance(self, inst_name: str, assigned_res_name: str, is_overwrite: bool):
        """
        Apply hsio resource to target gpio instance in design db

        :param inst_name: Target instance name
        :param assigned_res_name: Target lvds resource name
        :return: True, assigned, else False, fail to assign
        """
        source_inst, is_source_non_gpio = self.get_instance_from_resource(assigned_res_name)
        is_same_inst = False

        if source_inst is not None:
            is_device_used = True
            if source_inst.name == inst_name:
                is_same_inst = True

        else:
            is_device_used = False

        if is_device_used and not is_same_inst:
            is_used = True

            if is_source_non_gpio:
                if not is_overwrite:
                    msg = "The resource is already assigned to {}".format(source_inst.name)
                    raise PTResUsedException(msg, MsgLevel.error)

            # If the device is used, reset the gpio that it is currently assigned to
            if is_used is True:
                if self.__is_verbose:
                    msg = "The resource is already assigned to {}. {}'s resource will be cleared." \
                        .format(source_inst.name, source_inst.name)
                    print(msg)

                # reset the device on source gpio
                if source_inst is None:
                    if self.__is_verbose:
                        if is_source_non_gpio:
                            print("Cannot find used LVDS or MIPI Lane in registry : {}".format(assigned_res_name))
                        else:
                            print("Cannot find used GPIO in registry : {}".format(assigned_res_name))

                else:
                    # This event should be expected as reaction to primary
                    # signal, no broadcast
                    self.__gpio_reg.assign_gpio_device(source_inst, "")

        # Set the new lvds device to current gpio in config
        new_inst = self.__gpio_reg.get_gpio_by_name(inst_name)
        if new_inst is not None:
            self.__gpio_reg.assign_gpio_device(new_inst, assigned_res_name)
            new_inst.mark_lvds_gpio()

        return True


if __name__ == "__main__":
    pass
