"""
Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on July 13, 2020

@author: yasmin
"""

from __future__ import annotations
import abc
from typing import Optional, List, TYPE_CHECKING

from util.singleton_logger import Logger
from util.excp import MsgLevel

from design.db import PeriDesign
from design.db_item import PeriDesignItem, PeriDesignRegistry

from common_gui.validator import InputValidator

from api_service.common.design_session import DesignSession
from api_service.excp.design_excp import PTResInvalidException, PTResUsedException


if TYPE_CHECKING:
    from common_device.quad.lane_design import LaneBaseRegistry


class IntBlockAPI:
    """
    Provides high level block API related operation.

    The function here, may or may not match directly with user API.
    As much as possible, modules outside from peri_block should use the API
    here and only access raw registry level API when needed.

    The operations in the class is the db interaction performs in DesignExplorer.
    When it make sense, DesignExplorer should be refactored to use this block API

    """
    def __init__(self, is_verbose=False):
        self.is_verbose = is_verbose
        self.logger = Logger
        self.design = None
        self.blk_reg: Optional[PeriDesignRegistry] = None
        self.inst_name_prefix = ""
        self.session = DesignSession()
        from api_service.common.infra import InfraService
        self.infra_svc = InfraService.get_instance()
        assert self.infra_svc is not None
        self.msg_svc = self.infra_svc.get_message_service()
        self.validator = InputValidator()

    @abc.abstractmethod
    def set_design(self, design: PeriDesign):
        """
        Set design db. Override this to set the correct registry and block device service.

        :param design: design db instance
        :param design_block_type: design db block type
        """
        self.design = design

    def get_block(self, name: str) -> Optional[PeriDesignItem]:
        """
        Given a block name, get its block db instance.

        :param name: Block name
        :return: Block instance
        """
        assert self.blk_reg is not None
        return self.blk_reg.get_inst_by_name(name)

    def get_all_block_name(self) -> List[str]:
        """
        Get a list of instance names

        :return: A list of names. If zero instance, empty list
        """
        assert self.blk_reg is not None

        all_name = []
        for inst in self.blk_reg.get_all_inst():
            assert hasattr(inst, 'name')
            name = getattr(inst, 'name')
            all_name.append(name)

        return all_name

    def get_all_block(self)->List[PeriDesignItem]:
        """
        Get a list of all block db instance

        :return: A list of block instance
        """
        assert self.blk_reg is not None
        return self.blk_reg.get_all_inst()

    def create_block(self, name: str, is_register: bool = True):
        """
        Create a new block db instance

        :param name: Unique instance name
        :param is_register: True, register in registry else create standalone block
        :return: Block instance if successful else None
        ..notes:
          To support is_register, override this function.
        """
        if self.blk_reg is None:
            return None

        block = self.blk_reg.get_inst_by_name(name)

        if block is None:
            inst_name = self.check_instance_name(name)
            if inst_name is None:
                return None

            block = self.blk_reg.create_instance(inst_name, True, True)

            if block is None:
                self.logger.debug(f"Fail to create instance: {inst_name}")
                return block

            # Need to have the pins first before we can automate the pin name
            block.build_generic_pin()
            block.generate_pin_name()

        return block

    def check_instance_name(self, name: str):
        if self.blk_reg is None:
            return None

        if name == "":
            inst_name = self.blk_reg.gen_unique_inst_name(self.inst_name_prefix)
            if inst_name is None:
                if self.is_verbose:
                    self.msg_svc.write(MsgLevel.warning, "Fail to create block, fail auto-gen unique instance name")
                return None

        elif not self.validator.is_name_valid(name):
            inst_name = self.blk_reg.gen_unique_inst_name(self.inst_name_prefix)
            if inst_name is None:
                return None
            self.msg_svc.write(MsgLevel.warning, f"Invalid instance name {name}, auto-generate a new name {inst_name}")

        else:
            inst_name = name

        return inst_name

    def delete_block(self, name):
        """
        Delete block db instance, given its name

        :param name: Block instance name
        """

        if self.blk_reg is None:
            return None

        self.blk_reg.delete_inst(name)

    def assign_resource(self, db_inst: PeriDesignItem, res_name:str):
        """
        Assign resource to a block

        :param db_inst: Block instance
        :param res_name: Resource name
        """
        assert self.blk_reg is not None

        if db_inst is not None:
            # Able to assign empty resource
            if res_name == "":
                return self.blk_reg.assign_inst_device(db_inst, res_name)

            if self.blk_reg.is_device_valid(res_name) is False:
                raise PTResInvalidException(f"{res_name} is not a valid resource", MsgLevel.error)

            # Check if target resource is used
            if self.is_resource_used(res_name) and db_inst.get_device() != res_name:
                if self.is_verbose:
                    raise PTResUsedException(f"Resource is used: {res_name}", MsgLevel.warning)

            # Assign block resource
            self.blk_reg.assign_inst_device(db_inst, res_name)

    def is_resource_used(self, res_name: str) -> bool:
        """
        Check if a resource has been used.

        :param res_name: Resource name
        :return: True, if used else False
        ..notes:
          Override this to manage shared resource.
        """
        assert self.blk_reg is not None
        return self.blk_reg.is_device_used(res_name)

class IntLaneBasedAPI(IntBlockAPI):
    def __init__(self, is_verbose=False):
        super().__init__(is_verbose)
        self.blk_reg: LaneBaseRegistry

    def create_block(self, name: str, is_register=True, **kwargs):
        blk = super().create_block(name, is_register)

        if blk is None:
            return blk

        new_cmn_name = kwargs.get("cmn_name", "")
        if new_cmn_name != "":
            new_cmn_name = self.check_instance_name(new_cmn_name)
            if new_cmn_name is None:
                return blk

            self._update_cmn_inst_name(blk, new_cmn_name)

        return blk

    def _update_cmn_inst_name(self, blk, new_cmn_name: str):
        cmn_inst = self.blk_reg.get_cmn_inst_by_lane_name(blk.name)
        if self.blk_reg.common_quad_reg is None or cmn_inst is None:
            self.logger.debug(
                "Internal error: common quad reg is empty/ Fail to found common instance"\
                    " when calling IntLaneBasedAPI.create_block")
            return

        try:
            self.blk_reg.common_quad_reg.rename_inst(cmn_inst.name, new_cmn_name, auto_pin=True)

        except Exception:
            self.msg_svc.write(MsgLevel.warning,
                f"Common instance name {new_cmn_name} is already used.  Auto-generate another name {cmn_inst.name}")



if __name__ == "__main__":
    pass
