"""
Copyright (C) 2017-2019 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Oct 6, 2019

@author: yasmin
"""

from __future__ import annotations
import os
import sys
import pathlib
from typing import Optional, Union, Dict, List
import traceback

from util.singleton_logger import Logger
from util.gen_util import freeze_it
from util.excp import PTException, MsgLevel
from util.app_setting import AppSetting
import util.gui_util as gui_util

from design.service import DesignService
from device.service import DeviceService
from design.rule_service import RuleCheckService
import design.excp as db_excp
from design.db import PeriDesign

import api_service.excp.design_excp as pte
from api_service.common.design_session import DesignSession
from api_service.common.infra import InfraService
from api_service.common.object_db import APIObject
from api_service.api_info import APIVersion

from api_service.internal.int_device_setting import IntDeviceSettingAPI
from api_service.internal.int_jtag import IntJTAGAPI, IntBlockAPI
from api_service.internal.int_osc import IntOSCAPI
from api_service.internal.int_pll import IntPLLAPI
from api_service.internal.int_mipi import IntMIPIAPI
from api_service.internal.int_mipi_dphy import IntMIPIDPhyAPI
from api_service.internal.int_lvds import IntLVDSAPI, IntLVDSAdvAPI
from api_service.internal.int_hyper_ram import IntHyperRAMAPI
from api_service.internal.int_spi_flash import IntSPIFlashAPI
from api_service.internal.int_mipi_hard_dphy import IntMIPIHardDPHYAPI
from api_service.internal.int_pll_ssc import IntPLLSSCAPI
from api_service.internal.int_ddr import IntDDRAPI
from api_service.internal.int_quad import IntQuadAPI
from api_service.internal.int_quad_pcie import IntQuadPCIEAPI
from api_service.internal.int_soc import IntSOCAPI
from api_service.internal.int_lane10g import IntLane10GAPI
from api_service.internal.int_lane1g import IntLane1GAPI
from api_service.internal.int_raw_serdes import IntRawSerdesAPI


@freeze_it
class IntDesignAPI:
    """
    Provides high level design API related operation.

    The function here, may or may not match directly with user API.
    As much as possible, modules outside from peri_block should use the API
    here and only access raw registry level API when needed.

    Example user of this service are DesignExplorer and GPIO Import/Export.

    .. note::
       For developer, make sure any low level exception is caught here and
       translated to public facing API exception defined in
       api_service.excp.design_excp
    """

    def __init__(self, design=None, is_verbose=False):
        self.__app_setting = AppSetting()
        self.__is_verbose = is_verbose
        self.__logger = Logger
        self.__design = design
        self.__idev_setting = None
        self.__ijtag = None
        self.__ipll = None
        self.__iosc = None
        self.__imipi = None
        self.__imipi_dphy = None
        self.__ilvds = None
        self.__ihyper_ram = None
        self.__ispi_flash = None
        self.__imipi_hard_dphy = None
        self.__iddr = None
        self.__ipll_ssc = None
        self.__isoc = None
        self.__iquad = None
        self.__iquad_pcie = None
        self.__ilane_10g = None
        self.__ilane_1g = None
        self.__iraw_serdes = None
        self.__session = DesignSession()
        self.__infra_svc = InfraService.get_instance()
        self.__msg_svc = self.__infra_svc.get_message_service()

        self.__blktype_db2handler = {}  #: A map of design db block type to its internal API handler
        self.__blktype_api2handler: Dict[str, IntBlockAPI] = {}  #: A map of API block type name to its internal API handler

        # Turn off default plugin so it will read from the release plugin file
        DesignService._is_plugin_mockup = False

    def is_block_supported(self, api_block_type: Union[APIObject.ObjectType, str]) -> bool:
        """
        Check if a block type supported in API

        :param api_block_type: API block type, either a name or the enum type
        :return: True, supported else not
        ..notes:
          Check here rather than APIObject because it depends on device family too
        """

        if api_block_type == "GPIO":
            return True

        block_type_name = None
        if isinstance(api_block_type, APIObject.ObjectType):
            block_type_name = APIObject.otype2str_map.get(api_block_type, None)
        else:

            if APIObject.str2otype_map.get(api_block_type, None) is not None:
                block_type_name = api_block_type

        api_handler = self.__blktype_api2handler.get(block_type_name, None)
        if api_handler is not None:
            return True

        if self.__idev_setting.is_setting_supported(api_block_type):
            return True

        return False

    def load(self, design_file, check_efx=True, is_migration=False):
        """
        See :func:`DesignAPI.load`

        :param design_file: See :func:`DesignAPI.load`
        :param check_efx: Check data sync between Efinity project and periphery design
        """

        try:
            self.__session.check_env()

            service = DesignService()
            if os.path.exists(design_file):
                msg = "Load design..."
                if self.__is_verbose:
                    self.__msg_svc.write(MsgLevel.info, msg)

                # Make sure the design file is parseable
                is_good = service.check_design_file(design_file)
                if not is_good:
                    msg = "Design file has error : {}".format(design_file)
                    if self.__is_verbose:
                        self.__msg_svc.write(MsgLevel.error, msg)
                    raise pte.PTDsgLoadException(msg, MsgLevel.error)

                self.__logger.info("Passed file check.")

                # Load it
                service.is_check_device = False
                self.__design = service.load_design(design_file, ignore_device_proj_setting=is_migration)
                if self.__design is not None:
                    self.set_design(self.__design)

                    # Check if device valid
                    dev_service = DeviceService()
                    if not dev_service.is_device_exists(self.__design.device_def):
                        raise pte.PTDsgLoadException(
                            "Fail to load design, device {} is not supported".format(self.__design.device_def),
                            MsgLevel.error)

                    # Check Efinity project and Interface Designer design sync
                    if check_efx:
                        self.check_efx_project_sync()

                    # Check for auto-update that make sure design is valid
                    self.check_iobank_setting()
                    self.__check_design_auto_update()

                    if self.__is_verbose:
                        self.__msg_svc.write(MsgLevel.info, msg + "done.")
            else:
                msg = "Fail to load design from {}".format(design_file)
                raise pte.PTDsgLoadException(msg, MsgLevel.error)

        except PTException as excp:
            msg = "Fail to load design from {}".format(design_file)
            raise pte.PTDsgLoadException(msg, MsgLevel.error, excp)

    def migrate_design(self, design_file, update_design):
        """
        This is used for migrating design for non-virtual migration type
        :param design_file: The design file
        :param update_design: Bool flag indicating that the migration is to update design instead
                        of create a new design
        :return: True if it was successful
        """
        valid = True

        try:
            # Purposedly skip checking the design project and device name differences
            # The project name resolution is expected to be done when opening the UI
            self.load(design_file, check_efx=False, is_migration=True)

            if self.__design is not None:
                service = DesignService()
                service.is_check_device = False

                device_stat = service.check_device_status(self.__design)
                new_dev_name = device_stat[0]

                self.__logger.info("Applying new device {}".format(new_dev_name))
                final_design, _ = service.apply_new_device(self.__design, new_dev_name, update_design)
                self.__logger.info("Design has been migrated. Run checks...")

                self.set_design(self.__design)
                self.clean_output()

                if update_design:
                    self.check_design(False)

        except Exception as excp:
            self.__logger.error("Unexpected error: {}".format(excp))
            self.__logger.error(traceback.format_exc())
            valid = False

        return valid

    def check_efx_project_sync(self):
        """
        Check if setting in Efinity project matches with periphery design.
        If project file doesn't exists, checks will be skipped.

        :raises: PTDsgLoadException
        """

        dsg_svc = DesignService()
        try:
            project_name = dsg_svc.check_design_name_status(self.__design)
            if project_name is not None:
                msg = "Design name mismatched detected, Efinity project name is {}, periphery design name is {}" \
                    .format(project_name, self.__design.name)
                raise pte.PTDsgLoadException(msg, MsgLevel.error)

            project_device = dsg_svc.check_device_status(self.__design)
            if project_device is not None:
                msg = "Device name mismatched detected, Efinity project device is {}, periphery design device is {}" \
                    .format(project_device, self.__design.device_def)
                raise pte.PTDsgLoadException(msg, MsgLevel.error)

        except db_excp.CheckDeviceException:
            # Cannot find project or cannot read project, assume everything is good
            if self.__is_verbose:
                msg = "Unable to locate Efinity project. Skip project synchronization checks"
                self.__msg_svc.write(MsgLevel.warning, msg)

        if self.__is_verbose:
            msg = "Periphery design is in sync with Efinity project"
            self.__msg_svc.write(MsgLevel.info, msg)

    def check_iobank_setting(self):
        """
        Check if I/O bank setting has been updated due to change in device.

        :return: A tuple of(update status, affected iobank names).
                 Status is True, if updated, else False
                 Names returns a list of names or empty list

        .. note::
           Similar to the code in engine, see :func`Engine.verify_io_bank`
        """
        if self.__design is not None:
            if self.__design.device_setting is not None:
                iobank_reg = self.__design.device_setting.iobank_reg
                if iobank_reg is not None:
                    if iobank_reg.is_iobank_reset_when_load():
                        iobank_names = iobank_reg.get_iobank_reset()
                        msg = "The following I/O Bank was updated or reset to default" \
                              " due to invalid/deprecated setting:\n {}".format(",".join(iobank_names))
                        self.__msg_svc.write(MsgLevel.info, msg)
                        return True, iobank_names

        return False, []

    def __check_design_auto_update(self):
        """
        Check if key design component has been updated due to product change.
        This can happen if an old design is opened.

        :return: True, if updated, else False
        """

        is_updated = False

        dsg_svc = DesignService()
        if dsg_svc.is_resource_name_migration:
            msg = "Instance resource name was migrated. Please regenerate constraint and report files."
            self.__msg_svc.write(MsgLevel.warning, msg)
            is_updated = True

        if dsg_svc.is_bus_member_pin_name_regen:
            msg = "GPIO instance bus pin name was regenerated for synchronization. Please revisit bus configuration."
            self.__msg_svc.write(MsgLevel.warning, msg)
            is_updated = True

        return is_updated

    def create(self, design_name, device_name, path=".", auto_save=False, overwrite=False):
        """
        See :func:`DesignAPI.create`
        """

        full_design_file = "{}/{}.peri.xml".format(path, design_name)
        try:

            msg = "Create design..."
            if self.__is_verbose:
                self.__msg_svc.write(MsgLevel.info, msg)

            self.__session.check_env()

            if design_name == "":
                msg = "Design name is empty : {}".format(design_name)
                if self.__is_verbose:
                    self.__msg_svc.write(MsgLevel.error, msg)
                raise pte.PTDsgCreateException(msg, MsgLevel.error)

            if device_name == "":
                msg = "Device name is empty : {}".format(device_name)
                if self.__is_verbose:
                    self.__msg_svc.write(MsgLevel.error, msg)
                raise pte.PTDsgCreateException(msg, MsgLevel.error)

            design_path = os.path.dirname(full_design_file)
            if auto_save:
                if os.path.exists(full_design_file):
                    if not overwrite:
                        msg = "Design file exists : {}".format(full_design_file)
                        if self.__is_verbose:
                            self.__msg_svc.write(MsgLevel.error, msg)
                        raise pte.PTDsgCreateException(msg, MsgLevel.error)

                    else:
                        # Delete it so it will be recreated
                        os.remove(full_design_file)
                        msg = "Overwrite design file : {}".format(full_design_file)
                        if self.__is_verbose:
                            self.__msg_svc.write(MsgLevel.warning, msg)

            service = DesignService()
            service.is_check_device = False

            self.__design = service.create_design(design_name, device_name, design_path, is_save=auto_save)
            if self.__design is not None:
                self.set_design(self.__design)
                if self.__is_verbose:
                    self.__msg_svc.write(MsgLevel.info, "Create design...done.")

        except PTException as excp:
            msg = "Fail to create design, {}".format(full_design_file)
            raise pte.PTDsgCreateException(msg, MsgLevel.error, excp)

    def save(self):
        """
        See :func:`DesignAPI.save`
        """

        if self.__design is not None:
            try:
                service = DesignService()
                msg = "Save design..."
                if self.__is_verbose:
                    self.__msg_svc.write(MsgLevel.info, msg)
                service.save_design(self.__design)
                if self.__is_verbose:
                    self.__msg_svc.write(MsgLevel.info, msg + "done.")

            except PTException as excp:
                msg = "Fail to save design, {}".format(self.__design.design_file)
                raise pte.PTDsgSaveException(msg, MsgLevel.error, excp)
        else:
            msg = "Design object is None."
            raise pte.PTDsgSaveException(msg, MsgLevel.error)

    def save_as(self, design_file, overwrite=False):
        """
        See :func:`DesignAPI.save_as`
        """

        if self.__design is not None:
            msg = "Save as design..."
            if self.__is_verbose:
                self.__msg_svc.write(MsgLevel.info, msg)

            tmp_file = design_file + ".tmp"
            if overwrite:
                if os.path.exists(design_file):
                    os.rename(design_file, tmp_file)

            service = DesignService()
            try:
                service.save_design(self.__design, design_file)
                if overwrite:
                    if os.path.exists(tmp_file):
                        os.remove(tmp_file)

                if self.__is_verbose:
                    self.__msg_svc.write(MsgLevel.info, msg + "done.")

            except PTException as excp:
                if overwrite:
                    if os.path.exists(tmp_file):
                        os.rename(tmp_file, design_file)

                msg = "Fail to save as design, {}".format(design_file)
                raise pte.PTDsgSaveException(msg, MsgLevel.error, excp)
        else:
            msg = "Design object is None."
            raise pte.PTDsgSaveException(msg, MsgLevel.error)

    def set_design(self, design: Optional[PeriDesign]):
        """
        Set design database object

        :param design: Design database
        :return: None
        ..note::
          This function does not need to verify or assume correct design structure and supported block types.
          That is handled by design db. Here it just needs to query for block type and create appropriate
          api mgr if block type exists.
        """
        if design is not None:
            self.__design = design
            self.__idev_setting = IntDeviceSettingAPI()

        if self.__design is not None:
            self.__infra_svc.set_design(self.__design)
            self.__session.setup_design(self.__design, self.__design.design_file)
            self.__idev_setting.setup_design(design)

            self.__ijtag = IntJTAGAPI(self.__design, self.__is_verbose)
            self.__blktype_db2handler[PeriDesign.BlockType.jtag] = self.__ijtag
            self.__blktype_api2handler["JTAG"] = self.__ijtag

            self.__setup_pll()
            self.__setup_osc()
            self.__setup_mipi()
            self.__setup_lvds()

            if self.__design.is_block_supported(PeriDesign.BlockType.hyper_ram):
                self.__ihyper_ram = IntHyperRAMAPI(self.__is_verbose)
                self.__ihyper_ram.set_design(self.__design)
                self.__blktype_db2handler[PeriDesign.BlockType.hyper_ram] = self.__ihyper_ram
                self.__blktype_api2handler["HYPERRAM"] = self.__ihyper_ram

            if self.__design.is_block_supported(PeriDesign.BlockType.spi_flash):
                self.__ispi_flash = IntSPIFlashAPI(self.__is_verbose)
                self.__ispi_flash.set_design(self.__design)
                self.__blktype_db2handler[PeriDesign.BlockType.spi_flash] = self.__ispi_flash
                self.__blktype_api2handler["SPI_FLASH"] = self.__ispi_flash

            self.__setup_ddr()
            self.__setup_pll_ssc()
            self.__setup_soc()
            self.__setup_quad()
            self.__setup_serdes()

    def __setup_serdes(self):
        self.__setup_quad_pcie()
        self.__setup_lane_10g()
        self.__setup_lane_1g()
        self.__setup_raw_serdes()

    def __setup_osc(self):
        """
        Build oscillator support layer. Not a primary block, build as needed.
        """
        self.__iosc = None
        osc_block_type = None
        if self.__design.is_block_supported(PeriDesign.BlockType.hposc):
            osc_block_type = PeriDesign.BlockType.hposc
        elif self.__design.is_block_supported(PeriDesign.BlockType.adv_osc):
            osc_block_type = PeriDesign.BlockType.adv_osc
        elif self.__design.is_block_supported(PeriDesign.BlockType.osc):
            osc_block_type = PeriDesign.BlockType.osc

        if osc_block_type is not None:
            self.__iosc = IntOSCAPI.build_osc_api(self.__design, self.__is_verbose, osc_block_type)
            if self.__iosc is not None:
                self.__iosc.set_design(self.__design)
                self.__blktype_db2handler[osc_block_type] = self.__iosc
                self.__blktype_api2handler["OSC"] = self.__iosc

    def __setup_pll(self):
        """
        Build pll support layer. A primary block, always build.
        """
        self.__ipll = None
        if self.__design.is_block_supported(PeriDesign.BlockType.adv_pll):
            pll_block_type = PeriDesign.BlockType.adv_pll
        elif self.__design.is_block_supported(PeriDesign.BlockType.comp_pll):
            pll_block_type = PeriDesign.BlockType.comp_pll
        elif self.__design.is_block_supported(PeriDesign.BlockType.efx_pll_v3_comp):
            pll_block_type = PeriDesign.BlockType.efx_pll_v3_comp
        elif self.__design.is_block_supported(PeriDesign.BlockType.efx_fpll_v1):
            pll_block_type = PeriDesign.BlockType.efx_fpll_v1
        else:
            pll_block_type = PeriDesign.BlockType.pll

        if pll_block_type is not None:
            self.__ipll = IntPLLAPI.build_pll_api(is_verbose=self.__is_verbose, pll_block_type=pll_block_type)
            if self.__ipll is not None:
                self.__ipll.set_design(self.__design)
                self.__blktype_db2handler[pll_block_type] = self.__ipll
                self.__blktype_api2handler["PLL"] = self.__ipll

    def __setup_mipi(self):
        if self.__design.is_block_supported(PeriDesign.BlockType.mipi):
                self.__imipi = IntMIPIAPI(self.__is_verbose)
                self.__imipi.set_design(self.__design)
                self.__blktype_db2handler[PeriDesign.BlockType.mipi] = self.__imipi
                self.__blktype_api2handler["MIPI_TX"] = self.__imipi
                self.__blktype_api2handler["MIPI_RX"] = self.__imipi

        if self.__design.is_block_supported(PeriDesign.BlockType.mipi_dphy):
                self.__imipi_dphy = IntMIPIDPhyAPI(self.__is_verbose)
                self.__imipi_dphy.set_design(self.__design)
                self.__blktype_db2handler[PeriDesign.BlockType.mipi_dphy] = self.__imipi_dphy
                self.__blktype_api2handler["MIPI_TX_LANE"] = self.__imipi_dphy
                self.__blktype_api2handler["MIPI_RX_LANE"] = self.__imipi_dphy

        if self.__design.is_block_supported(PeriDesign.BlockType.mipi_hard_dphy):
            self.__imipi_hard_dphy = IntMIPIHardDPHYAPI(self.__is_verbose)
            self.__imipi_hard_dphy.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.mipi_hard_dphy] = self.__imipi_hard_dphy
            assert self.__design.device_db is not None
            if self.__design.is_device_block_tx_rx_supported(PeriDesign.BlockType.mipi_hard_dphy, False):
                self.__blktype_api2handler["MIPI_DPHY_RX"] = self.__imipi_hard_dphy
            if self.__design.is_device_block_tx_rx_supported(PeriDesign.BlockType.mipi_hard_dphy, True):
                self.__blktype_api2handler["MIPI_DPHY_TX"] = self.__imipi_hard_dphy

    def __setup_lvds(self):
        if self.__design.is_block_supported(PeriDesign.BlockType.lvds):
                self.__ilvds = IntLVDSAPI(self.__is_verbose)
                self.__ilvds.set_design(self.__design)
                self.__blktype_db2handler[PeriDesign.BlockType.lvds] = self.__ilvds
                self.__blktype_api2handler["LVDS_TX"] = self.__ilvds
                self.__blktype_api2handler["LVDS_RX"] = self.__ilvds

        elif self.__design.is_block_supported(PeriDesign.BlockType.adv_lvds):
                self.__ilvds = IntLVDSAdvAPI(self.__is_verbose)
                self.__ilvds.set_design(self.__design)
                self.__blktype_db2handler[PeriDesign.BlockType.adv_lvds] = self.__ilvds
                self.__blktype_api2handler["LVDS_TX"] = self.__ilvds
                self.__blktype_api2handler["LVDS_RX"] = self.__ilvds
                self.__blktype_api2handler["LVDS_BIDIR"] = self.__ilvds

    def __setup_ddr(self):
        self.__iddr = None
        if self.__design.is_block_supported(PeriDesign.BlockType.ddr):
            ddr_block_type = PeriDesign.BlockType.ddr
        elif self.__design.is_block_supported(PeriDesign.BlockType.adv_ddr):
            ddr_block_type = PeriDesign.BlockType.ddr
        else:
            return

        self.__iddr = IntDDRAPI.build_ddr_api(is_verbose=self.__is_verbose, ddr_block_type=ddr_block_type)
        if self.__iddr is not None:
            self.__iddr.set_design(self.__design)
            self.__blktype_db2handler[ddr_block_type] = self.__iddr
            self.__blktype_api2handler["DDR"] = self.__iddr

    def __setup_pll_ssc(self):
        assert self.__design is not None
        self.__ipll_ssc = None

        if self.__design.is_block_supported(PeriDesign.BlockType.pll_ssc):
            self.__ipll_ssc = IntPLLSSCAPI(self.__is_verbose)
            self.__ipll_ssc.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.pll_ssc] = self.__ipll_ssc
            self.__blktype_api2handler["PLL_SSC"] = self.__ipll_ssc

    def __setup_soc(self):
        assert self.__design is not None
        self.__isoc = None

        if self.__design.is_block_supported(PeriDesign.BlockType.soc):
            self.__isoc = IntSOCAPI(self.__is_verbose)
            self.__isoc.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.soc] = self.__isoc
            self.__blktype_api2handler["SOC"] = self.__isoc

    def __setup_quad(self):
        # "QUAD" is for internal use only
        assert self.__design is not None
        self.__iquad = None

        if self.__design.is_block_supported(PeriDesign.BlockType.quad):
            self.__iquad = IntQuadAPI(self.__is_verbose)
            self.__iquad.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.quad] = self.__iquad
            self.__blktype_api2handler["QUAD"] = self.__iquad

    def __setup_quad_pcie(self):
        assert self.__design is not None
        self.__iquad_pcie = None

        if self.__design.is_block_supported(PeriDesign.BlockType.quad_pcie):
            self.__iquad_pcie = IntQuadPCIEAPI(self.__is_verbose)
            self.__iquad_pcie.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.quad_pcie] = self.__iquad_pcie
            self.__blktype_api2handler["QUAD_PCIE"] = self.__iquad_pcie

    def __setup_lane_10g(self):
        assert self.__design is not None
        self.__ilane_10g = None

        if self.__design.is_block_supported(PeriDesign.BlockType.lane_10g):
            self.__ilane_10g = IntLane10GAPI(self.__is_verbose)
            self.__ilane_10g.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.lane_10g] = self.__ilane_10g
            self.__blktype_api2handler["10GBASE_KR"] = self.__ilane_10g

    def __setup_lane_1g(self):
        assert self.__design is not None
        self.__ilane_1g = None

        if self.__design.is_block_supported(PeriDesign.BlockType.lane_1g):
            self.__ilane_1g = IntLane1GAPI(self.__is_verbose)
            self.__ilane_1g.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.lane_1g] = self.__ilane_1g
            self.__blktype_api2handler["SGMII"] = self.__ilane_1g

    def __setup_raw_serdes(self):
        assert self.__design is not None
        self.__iraw_serdes = None

        if self.__design.is_block_supported(PeriDesign.BlockType.raw_serdes):
            self.__iraw_serdes = IntRawSerdesAPI(self.__is_verbose)
            self.__iraw_serdes.set_design(self.__design)
            self.__blktype_db2handler[PeriDesign.BlockType.raw_serdes] = self.__iraw_serdes
            self.__blktype_api2handler["PMA_DIRECT"] = self.__iraw_serdes

    def get_design(self) -> Optional[PeriDesign]:
        """
        Get design database object

        :return: Design database
        """
        return self.__design

    def get_idevice_setting(self)->IntDeviceSettingAPI:
        """
        Get device setting interface
        """
        return self.__idev_setting

    def get_block_type(self, is_filter=True) -> List[str]:
        """
        See :func:`DesignAPI.get_block_type`
        """
        block_list: List[str] = []
        if self.__design is not None:
            app_block_list = self.__design.get_supported_block_name_list()
            # Filter only the block type supported by API
            if is_filter:
                version = APIVersion()
                curr_ver_info = version.get_current_version_info()
                api_block = curr_ver_info.get_block()
                for block in self.__blktype_api2handler:
                    if block in api_block:
                        block_list.append(block)
                block_list.append("GPIO")
            else:
                block_list = app_block_list

            gui_util.natural_order_sort_in_place(block_list)

        return block_list

    def create_block(self, name, block_type: str, **kwargs):
        """
        See :func:`DesignAPI.create_block
        """
        if self.__design is not None:
            try:
                if block_type == "JTAG":
                    return self.__ijtag.create_block(name)
                elif block_type == "PLL":
                    return self.__ipll.create_block(name)
                elif block_type == "OSC":
                    return self.__iosc.create_block(name)
                elif block_type == "MIPI_TX":
                    return self.__imipi.create_tx_block(name, True, **kwargs)
                elif block_type == "MIPI_RX":
                    return self.__imipi.create_rx_block(name, True, **kwargs)
                elif block_type == "MIPI_TX_LANE":
                    return self.__imipi_dphy.create_tx_block(name, True, **kwargs)
                elif block_type == "MIPI_RX_LANE":
                    return self.__imipi_dphy.create_rx_block(name, True, **kwargs)
                elif block_type == "LVDS_TX":
                    return self.__ilvds.create_tx_block(name, True, **kwargs)
                elif block_type == "LVDS_RX":
                    return self.__ilvds.create_rx_block(name, True, **kwargs)
                elif block_type == "LVDS_BIDIR":
                    return self.__ilvds.create_bidir_block(name, True, **kwargs)
                elif block_type == "MIPI_DPHY_TX":
                    return self.__imipi_hard_dphy.create_tx_block(name, True, **kwargs)
                elif block_type == "MIPI_DPHY_RX":
                    return self.__imipi_hard_dphy.create_rx_block(name, True, **kwargs)
                elif block_type == "HYPERRAM":
                    return self.__ihyper_ram.create_block(name, True, **kwargs)
                elif block_type == "SPI_FLASH":
                    return self.__ispi_flash.create_block(name, True, **kwargs)
                elif block_type == "DDR":
                    return self.__iddr.create_block(name, True, **kwargs)
                elif block_type == "PLL_SSC":
                    assert self.__ipll_ssc is not None
                    return self.__ipll_ssc.create_block(name, True, **kwargs)
                elif block_type == "SOC":
                    assert self.__isoc is not None
                    return self.__isoc.create_block(name, True, **kwargs)
                elif block_type == "QUAD":
                    assert self.__iquad is not None
                    return self.__iquad.create_block(name, True, **kwargs)
                elif block_type == "QUAD_PCIE":
                    assert self.__iquad_pcie is not None
                    return self.__iquad_pcie.create_block(name, True, **kwargs)
                elif block_type == "10GBASE_KR":
                    assert self.__ilane_10g is not None
                    return self.__ilane_10g.create_block(name, True, **kwargs)
                elif block_type == "PMA_DIRECT":
                    assert self.__iraw_serdes is not None
                    return self.__iraw_serdes.create_block(name, True, **kwargs)
                elif block_type == "SGMII":
                    assert self.__ilane_1g is not None
                    return self.__ilane_1g.create_block(name, True, **kwargs)
                else:
                    msg = "Fail to create block, unknown block type {}".format(block_type)
                    raise pte.PTBlkCreateException(msg, MsgLevel.error)

            except PTException as excp:
                if excp.msg == "":
                    msg = "Fail to create block"
                    raise pte.PTBlkCreateException(msg, MsgLevel.error, excp)
                else:
                    raise pte.PTBlkCreateException(excp.msg, MsgLevel.error)

        return None

    def delete_block(self, name: str, block_type: str):
        """
        See :func:`DesignAPI.delete_block
        """

        if self.__design is not None:
            try:
                if block_type == "JTAG":
                    return self.__ijtag.delete_block(name)
                else:
                    block_api = self.__blktype_api2handler.get(block_type, None)
                    if block_api is not None:
                        return block_api.delete_block(name)
                    else:
                        msg = "Fail to delete block, unknown block type {}".format(block_type)
                        raise pte.PTBlkDeleteException(msg, MsgLevel.error)

            except PTException as excp:
                msg = "Fail to delete block"
                raise pte.PTBlkDeleteException(msg, MsgLevel.error, excp)

    def clear_design(self, block_types: List[str]):
        if self.__design is None:
            return

        try:
            db_blk_types = self.convert_api_types2db_types(block_types)
            self.__design.clear_design(db_blk_types)

        except PTException as excp:
            msg = "Fail to clear design"
            raise pte.PTBlkDeleteException(msg, MsgLevel.error, excp)

    def convert_api_types2db_types(self, api_blk_types: List[str]):
        """
        Convert API block type string to match PeriDesign.blktype2str_map
        for LVDS/MIPI, need to specify TX/RX at last

        :param api_blk_types: API block types
        :type api_blk_types: List[str]
        :return: design block types
        :rtype: List[str]
        """
        db_types = []

        for blk_type in api_blk_types:
            api_type = APIObject.str2otype_map.get(blk_type)
            if api_type is None:
                continue

            db_type = blk_type

            match blk_type:
                # MIPI DPHY
                case "MIPI_TX_LANE":
                    db_type = "MIPI_DPHY_TX"
                case "MIPI_RX_LANE":
                    db_type = "MIPI_DPHY_RX"

                # MIPI Hard DPHY
                case "MIPI_DPHY_TX":
                    db_type = "MIPI_TX"
                case "MIPI_DPHY_RX":
                    db_type = "MIPI_RX"

                # Hyper RAM
                case "HYPERRAM":
                    db_type = "HYPER_RAM"

                case _:
                    db_type = blk_type

            db_types.append(db_type)

        return db_types

    def get_all_block_name(self, block_type: str):
        """
        See :func:`DesignAPI.get_all_block_name`
        """
        if self.__design is not None:
            try:
                if block_type == "JTAG":
                    return self.__ijtag.get_all_block_name()
                else:
                    block_api = self.__blktype_api2handler.get(block_type, None)
                    if block_api is not None:
                        return block_api.get_all_block_name()
                    else:
                        msg = "Fail to get block name, unknown block type {}".format(block_type)
                        raise pte.PTBlkReadException(msg, MsgLevel.error)

            except PTException as excp:
                msg = "Fail to get block name"
                raise pte.PTBlkReadException(msg, MsgLevel.error, excp)

        return []

    def get_block(self, name: str, block_type: str):
        """
        See :func:`DesignAPI.get_block`
        """
        if self.__design is not None:
            try:
                if block_type == "JTAG":
                    return self.__ijtag.get_block(name)

                else:
                    block_api: Optional[IntBlockAPI] = self.__blktype_api2handler.get(block_type, None)

                    if block_api is None:
                        msg = "Fail to get block, unknown block type {}".format(block_type)
                        raise pte.PTBlkReadException(msg, MsgLevel.error)
                    else:
                        return block_api.get_block(name)

            except PTException as excp:
                msg = "Fail to get block"
                raise pte.PTBlkReadException(msg, MsgLevel.error, excp)

        return None

    def assign_resource(self, obj_inst, res_name, block_type):
        """
        See :func:`DesignAPI.assign_resource`

        :param obj_inst: Design db block instance
        :param res_name: Resource name
        :param block_type: Target block type
        """
        if self.__design is not None:
            try:
                block_api = self.__blktype_api2handler.get(block_type, None)
                if block_api is not None:
                    block_api.assign_resource(obj_inst, res_name)
                else:
                    msg = "Fail to assign resource, unknown block type {}".format(block_type)
                    raise pte.PTBlkEditException(msg, MsgLevel.error)

            except PTException as excp:
                msg = "Fail to assign resource"
                raise pte.PTBlkEditException(msg, MsgLevel.error, excp)

    def check_design(self, get_details=False):
        """
        See :func:`DesignAPI.check_design`

        :param get_details: True, get issue detail if check fails, else no
        """

        issue_list = []
        try:
            outflow_dir = self.get_outflow_dir()
            if os.path.exists(outflow_dir) is False:
                pathlib.Path(outflow_dir).mkdir(parents=True, exist_ok=True)

            msg = "Check design..."
            if self.__is_verbose:
                self.__msg_svc.write(MsgLevel.info, msg)

            service = RuleCheckService(self.__design)
            is_pass = service.run(RuleCheckService.RunModeType.COMP)
            issue_reg = self.__design.issue_reg
            if issue_reg is not None:
                issue_reg.save_to_file(self.get_checker_result_file())
                if get_details:
                    issue_list = issue_reg.get_all_issue()

            # TODO: After running check design which runs clock mux routing
            # we then refresh in case there was clock rotation

            msg = "Check design...done."
            if self.__is_verbose:
                self.__msg_svc.write(MsgLevel.info, msg)

        except Exception as exc:
            raise pte.PTDsgCheckException("Fail to check design.", MsgLevel.warning, exc)

        return is_pass, issue_list

    def export_design_check_issue(self):
        """
        Export design check issue to a csv file
        """

        # Create the ouflow directory first if it doesnt exists
        outflow_dir = self.get_outflow_dir()

        if not os.path.exists(outflow_dir) and \
                (self.__design.issue_reg is not None or self.__design.export_import_issue_reg is not None):
            pathlib.Path(outflow_dir).mkdir(parents=True, exist_ok=True)

        if self.__design.issue_reg is not None:
            self.__design.issue_reg.write_to_csv_file(
                self.get_checker_result_export_file())

    def get_outflow_dir(self):
        """
        Get outflow directory. It is always a subdirectory of where design file is.

        :return: Outflow full path name
        """
        # TODO : Need to update when supporting multiple periphery design
        if self.__design is None:
            return ""

        return self.__design.location + "/" + self.__app_setting.output_dir

    def get_checker_result_file(self):
        """
        Get design check output file name

        :return: File name
        """
        return self.get_outflow_dir() + "/design_issue.xml"

    def get_checker_result_export_file(self):
        """
        Get design check output export file name

        :return: File name
        """
        return self.get_outflow_dir() + "/design_issue.csv"

    def gen_constraint(self, enable_bitstream, outdir=None):
        """
        See :func:`DesignAPI.gen_constraint`
        """
        if outdir is None:
            outflow_dir = self.get_outflow_dir()
        else:
            outflow_dir = outdir

        if os.path.exists(outflow_dir) is False:
            pathlib.Path(outflow_dir).mkdir(parents=True, exist_ok=True)

        # PT-390: We should not generate PCR if there's any issue in generating
        # any of the constraints. However, if any of the other files failed to
        # generate, we can still generate the non-lpf files
        gen_error = False
        error_msg = "Fail to generate "
        error_files = []
        error_excp = None
        used_ports = None
        input_to_loc_map = None

        try:
            used_ports, input_to_loc_map = self.gen_interface(outflow_dir)
        except Exception:
            gen_error = True
            error_files.append("Interface")

        try:
            int_error = False
            if "Interface" in error_files:
                int_error = True

            self.gen_timing(outflow_dir, input_to_loc_map, int_error)
        except Exception:
            gen_error = True
            error_files.append("Timing")

        try:
            self.gen_verilog_template(outflow_dir, used_ports)
        except Exception:
            gen_error = True
            error_files.append("Verilog template")

        try:
            self.gen_option_register(outflow_dir)
        except Exception:
            gen_error = True
            error_files.append("Option register")

        if self.is_post_config_bsdl_supported():
            try:
                self.gen_post_config_bsdl(outflow_dir)
            except Exception:
                # BSDL file is minor file, shouldn't raise error
                # gen_error = True
                # error_files.append("Post-Configured BSDL file")
                pass

        if not gen_error:
            if enable_bitstream:
                # Generate LPF only if Efinity device supports bitstream
                # generation
                self.gen_pcr(outflow_dir)
        else:
            if len(error_files) > 1:
                error_msg = "Fail to generate {} files".format(
                    ",".join(error_files))
            else:
                error_msg = "Fail to generate {} file".format(
                    ",".join(error_files))

            raise pte.PTDsgGenConstException(
                error_msg, MsgLevel.warning, error_excp)

    def gen_report(self, outdir=None):
        """
        See :func:`DesignAPI.gen_report`
        """

        if outdir is None:
            outflow_dir = self.get_outflow_dir()
        else:
            outflow_dir = outdir

        if os.path.exists(outflow_dir) is False:
            pathlib.Path(outflow_dir).mkdir(parents=True, exist_ok=True)

        # If summary reporting fails, exception is thrown and it won't
        # generate the pinout
        self.gen_summary_report(outflow_dir)
        self.gen_pinout_report(outflow_dir)

    def gen_pcr(self, outflow_dir):
        """
        Generate PCR for full design

        :param outflow_dir: Output directory to store file
        """

        if self.__design is None:
            raise pte.PTDsgGenConstException("Design is empty. Fail to generate PCR file", MsgLevel.warning)

        try:
            from writer.logical_periphery import LogicalPeriphery
            lpf_gen = LogicalPeriphery(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
            lpf_gen.write()

        except Exception as exc:
            raise pte.PTDsgGenConstException("Fail to generate PCR file", MsgLevel.warning, exc)

    def gen_interface(self, outflow_dir):
        """
        Generate interface configuration file

        :param outflow_dir: Output directory to store file
        """
        used_ports = None
        input_to_loc_map = None

        if self.__design is None:
            raise pte.PTDsgGenConstException("Design is empty. Fail to generate interface configuration file",
                                             MsgLevel.warning)

        try:
            from writer.interface_config import InterfaceConfig
            icfg_gen = InterfaceConfig(self.__design.device_db, self.__design, outflow_dir, self.__design.name)

            icfg_gen.write()
            used_ports = icfg_gen.get_populated_used_ports()
            input_to_loc_map = icfg_gen.get_input_interface_location_map()

        except Exception as exc:
            raise pte.PTDsgGenConstException("Fail to generate interface configuration file", MsgLevel.warning, exc)

        return used_ports, input_to_loc_map

    def gen_timing(self, outflow_dir, input_to_loc_map=None,
                   is_interface_error=False):
        """
        Generate timing related files

        :param outflow_dir: Output directory to store file
        """

        if self.__design is None:
            raise pte.PTDsgGenConstException("Design is empty. Fail to generate timing related files", MsgLevel.warning)

        try:
            from writer.timing_report import TimingReport
            tgen = TimingReport(self.__design.device_db, self.__design,
                                outflow_dir, self.__design.name)
            if input_to_loc_map is not None:
                tgen.set_pre_populated_input_loc_map(input_to_loc_map)

            if not is_interface_error:
                tgen.write()
            else:
                tgen.writer_pre_requisite_fail()

        except Exception as exc:
            raise pte.PTDsgGenConstException("Fail to generate timing related files", MsgLevel.warning, exc)

    def gen_verilog_template(self, outflow_dir, used_ports=None):
        """
        Generate verilog template file

        :param outflow_dir: Output directory to store file
        """

        if self.__design is None:
            raise pte.PTDsgGenConstException(
                "Design is empty. Fail to generate verilog template file", MsgLevel.warning)

        try:
            from writer.verilog_template import VerilogTemplate
            vgen = VerilogTemplate(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
            if used_ports is not None:
                vgen.set_pre_populated_ports(used_ports)
            vgen.write()

        except Exception as exc:
            raise pte.PTDsgGenConstException(
                "Fail to generate verilog template file", MsgLevel.warning, exc)

    def is_post_config_bsdl_supported(self) -> bool:
        from writer.bsdl_writer import BSDLWriter
        bsdl_writer = BSDLWriter(self.__design.device_db, self.__design)
        return bsdl_writer.is_supported()

    def gen_post_config_bsdl(self, outflow_dir):
        """
        Generate boundary scan description file

        :param outflow_dir: Output directory to store file
        """

        if self.__design is None:
            raise pte.PTDsgGenConstException(
                "Design is empty. Fail to generate boundary scan description file", MsgLevel.warning)

        try:
            from writer.bsdl_writer import BSDLWriter
            bsdl_writer = BSDLWriter(self.__design.device_db, self.__design)
            bsdl_writer.write(self.__design.name, outflow_dir)

        except Exception as exc:
            raise pte.PTDsgGenConstException(
                "Fail to generate boundary scan description file", MsgLevel.warning, exc)

    def gen_summary_report(self, outflow_dir):
        """
        Generate report summary

        :param outflow_dir: Output directory to store file
        """

        if self.__design is None:
            raise pte.PTDsgGenReportException(
                "Design is empty. Fail to generate summary report file", MsgLevel.warning)

        try:
            from writer.summary_report import SummaryReport
            rpt_gen = SummaryReport(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
            rpt_gen.write()

        except Exception as exc:
            raise pte.PTDsgGenReportException("Fail to generate summary report file", MsgLevel.warning, exc)

    def gen_pinout_report(self, outflow_dir):
        """
        Generate pinout report

        :param outflow_dir: Output directory to store file
        """

        if self.__design is None:
            raise pte.PTDsgGenReportException("Design is empty. Fail to generate pinout report file", MsgLevel.warning)

        try:
            from writer.pinout import PinoutReport
            rpt_gen = PinoutReport(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
            rpt_gen.write()

        except Exception as exc:
            raise pte.PTDsgGenReportException(
                "Fail to generate pinout report file", MsgLevel.warning, exc)

    def gen_option_register(self, outflow_dir):
        """
        Generate option register file

        :param outflow_dir: Output directory to store file
        """

        if self.__design is None:
            raise pte.PTDsgGenConstException("Design is empty. Fail to generate option register file",
                                             MsgLevel.warning)

        try:
            from writer.option_register import OptionRegister
            optreg_gen = OptionRegister(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
            optreg_gen.write()

        except Exception as exc:
            raise pte.PTDsgGenConstException("Fail to generate option register file", MsgLevel.warning, exc)

    def clean_output(self):
        """
        Clean up outflow. See :func:`DesignAPI.clean_output`
        """

        outflow_dir = self.get_outflow_dir()
        # Nothing to remove if the directory does not exists
        if not os.path.exists(outflow_dir):
            return

        self.clear_checker_file(outflow_dir)
        self.clear_constraint_file(outflow_dir)
        self.clear_report_file(outflow_dir)

    def clear_checker_file(self, outflow_dir):
        """
        Delete checker result file. Keep the user exported version.

        :param outflow_dir: Output directory where files are located
        """
        if self.__design is None:
            return

        checker_file = self.get_checker_result_file()
        if os.path.exists(checker_file) is True:
            os.remove(checker_file)

    def clear_report_file(self, outflow_dir):
        """
        Delete all physical report files

        :param outflow_dir: Output directory where files are located
        """
        if self.__design is None:
            return

        # Note : Constructor for writer is lightweight so we can afford to instantiate
        # it again to get the filename

        # Clear summary report
        from writer.summary_report import SummaryReport
        rpt_gen = SummaryReport(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        report_filename = rpt_gen.get_file_name()

        if os.path.exists(report_filename) is True:
            os.remove(report_filename)

        xml_summary_filename = rpt_gen.get_xml_file_name()

        if os.path.exists(xml_summary_filename) is True:
            os.remove(xml_summary_filename)

        # Clear pinout csv report
        from writer.pinout import PinoutReport
        rpt_gen = PinoutReport(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        csv_report_filename = rpt_gen.get_file_name()

        if os.path.exists(csv_report_filename) is True:
            os.remove(csv_report_filename)

        # Clear the pinout report
        report_filename = rpt_gen.get_readable_report_file_name()
        if os.path.exists(report_filename) is True:
            os.remove(report_filename)

    def clear_constraint_file(self, outflow_dir):
        """
        Delete all physical constraint files

        :param outflow_dir: Output directory where files are located
        """
        if self.__design is None:
            return

        # Note : Constructor for writer is lightweight so we can afford to instantiate
        # it again to get the filename

        # Clear io
        from writer.io_placement import IOPlacement
        place_gen = IOPlacement(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        const_filename = place_gen.get_file_name()

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

        # Clear pcr
        from writer.logical_periphery import LogicalPeriphery
        lpf_gen = LogicalPeriphery(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        const_filename = lpf_gen.get_file_name()

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

        # Check for DDR Trion .c file if exists
        ddr_c_filename = lpf_gen.get_ddr_c_file()
        if os.path.exists(ddr_c_filename) is True:
            os.remove(ddr_c_filename)

        # PT-1676: Check for DDR Titanium pcr_write_pattern.v files
        verilog_pattern_filename = lpf_gen.get_verilog_pattern_file()
        if os.path.exists(verilog_pattern_filename) is True:
            os.remove(verilog_pattern_filename)

        # Clear interface
        from writer.interface_config import InterfaceConfig
        icfg_gen = InterfaceConfig(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        const_filename = icfg_gen.get_file_name()

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

        # Clear timing report
        from writer.timing_report import TimingReport
        timing_gen = TimingReport(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        const_filename = timing_gen.get_report_file_name()

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

        # Clear timing constraint
        const_filename = timing_gen.get_sdc_file_name()

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

        # Clear verilog template
        from writer.verilog_template import VerilogTemplate
        verilog_gen = VerilogTemplate(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        const_filename = verilog_gen.get_file_name()

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

        # Clear the option register
        from writer.option_register import OptionRegister
        optreg_gen = OptionRegister(self.__design.device_db, self.__design, outflow_dir, self.__design.name)
        const_filename = optreg_gen.get_file_name()

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

        from writer.bsdl_writer import BSDLWriter
        bsdl_writer = BSDLWriter(self.__design.device_db, self.__design)
        const_filename = bsdl_writer.get_file_name(self.__design.name, outflow_dir)

        if os.path.exists(const_filename) is True:
            os.remove(const_filename)

    def get_block_apimgr(self, block_type: str):
        """
        Get access to block API manager. To be used when specific block
        operation is required.

        :param block_type: Block type name
        :return: Block API manager
        """

        if self.__design is None:
            return None

        return self.__blktype_api2handler.get(block_type, None)

    def get_block_apimgr_by_db_type(self, block_type: PeriDesign.BlockType):
        """
        Similar to get_block_apimgr. This one useful for internal use.

        :param block_type: Design block type
        :return: Block API manager
        """

        if self.__design is None:
            return None

        return self.__blktype_db2handler.get(block_type, None)

    def reset(self):
        self.__blktype_api2handler = {}
        self.__blktype_db2handler = {}

if __name__ == "__main__":
    pass
