"""
 Copyright (C) 2017-2020 Efinix Inc. All rights reserved.

 No portion of this code may be reused, modified or
 distributed in any way without the expressed written
 consent of Efinix Inc.

 Created on Mar 26, 2020

 @author: yasmin
"""

import os
import sys
from typing import List, Dict, Any, Optional
from abc import ABC, abstractmethod
from enum import Enum, auto
from api_service.internal.int_gen_block import IntBlockAPI
from api_service.internal.int_mipi_hard_dphy import IntMIPIHardDPHYAPI
from api_service.property.gen_prop import BaseProp
from design.db_item import PeriDesignRegistry
from tx180_device.ddr.ddr_design_adv import DDRAdvance

import util.gen_util as gen_util
from util.excp import MsgLevel

import common_device.gpio.gpio_design as gd
import common_device.osc.osc_design as oscd
from common_device.mipi.mipi_design import MIPI, PhyLane
from common_device.mipi_dphy.mipi_dphy_design import MIPIDPhy, MIPIDPhyTx, MIPIDPhyRx
from tx180_device.mipi_dphy.design import MIPIHardDPHYRx, MIPIHardDPHYTx
from tx60_device.osc.osc_design_adv import OSCAdvance
from tx60_device.hposc.hposc_design import HPOSC
from tx60_device.gpio.gpio_design_comp import GPIOComplex
from common_device.lvds.lvds_design import LVDS, LVDSRx, LVDSTx
from tx60_device.lvds.lvds_design_adv import LVDSAdvance
from tx60_device.clock_mux.clkmux_design_adv import ClockMuxRegistryAdvance
from tx180_device.clock_mux.clkmux_design_comp import ClockMuxRegistryComplex

from design.db import PeriDesign
from design.design_export_import import ExportGPIODesign

from api_service.internal.int_gpio import IntGPIOAPI
from api_service.internal.int_osc import IntOSCAPI
from api_service.internal.int_pll import IntPLLAPI
from api_service.internal.int_jtag import IntJTAGAPI
from api_service.internal.int_mipi import IntMIPIAPI
from api_service.internal.int_mipi_dphy import IntMIPIDPhyAPI
from api_service.internal.int_lvds import IntLVDSAPI, IntLVDSAdvAPI
from api_service.internal.int_iobank import IntIOBankAPI
from api_service.internal.int_seu import IntSEUAPI
from api_service.internal.int_ctrl import IntCtrlAPI
from api_service.internal.int_hyper_ram import IntHyperRAMAPI
from api_service.internal.int_spi_flash import IntSPIFlashAPI
from api_service.internal.int_ddr import IntDDRAPI, IntDDRAdvAPI
from api_service.internal.int_ext_flash import IntExtFlashAPI
from api_service.internal.int_clkmux import IntClockMuxAPI
from api_service.internal.int_pll_ssc import IntPLLSSCAPI
from api_service.internal.int_soc import IntSOCAPI
from api_service.internal.int_quad_pcie import IntQuadPCIEAPI
from api_service.internal.int_lane10g import IntLane10GAPI
from api_service.internal.int_lane1g import IntLane1GAPI
from api_service.internal.int_raw_serdes import IntRawSerdesAPI

from api_service.property.gpio_prop import GPIOProp
from api_service.property.osc_prop import OSCProp
from api_service.property.pll_prop import PLLProp
from api_service.property.jtag_prop import JTAGProp
from api_service.property.mipi_prop import MIPITxProp, MIPIRxProp
from api_service.property.mipi_dphy_prop import MIPIDPhyTxProp, MIPIDPhyRxProp
from api_service.property.mipi_hard_dphy_prop import MIPIHardDPHYRxProp, MIPIHardDPHYTxProp
from api_service.property.lvds_prop import LVDSRxBasePresenter, LVDSRxPresenter, LVDSRxProp, LVDSTxAdvPresenter, LVDSTxAdvProp, LVDSRxAdvPresenter, LVDSBidirAdvProp, LVDSTxBasePresenter, LVDSTxPresenter, LVDSTxProp
from api_service.property.iobank_prop import IOBankProp
from api_service.property.seu_prop import SEUProp
from api_service.property.ctrl_prop import CtrlProp
from api_service.property.hyper_ram_prop import HyperRAMProp
from api_service.property.spi_flash_prop import SPIFlashProp
from api_service.property.ddr_prop import DDRProp, DDRAdvProp
from api_service.property.ext_flash_prop import ExtFlashProp
from api_service.property.clkmux_prop import ClockMuxProp
from api_service.property.pll_ssc_prop import PLLSSCProp
from api_service.property.soc_prop import SOCProp
from api_service.property.quad_pcie_prop import QuadPCIEProp
from api_service.property.lane10g_prop import Lane10GProp
from api_service.property.lane1g_prop import Lane1GProp
from api_service.property.raw_serdes_prop import RawSerdesProp

from api_service.common.object_db import APIObject


@gen_util.freeze_it
class DesignISFExporter:
    """
    Export design db into an Interface Scripting File (ISF)
    """
    design_api_handle = "design"  #: design api variabe in legal python script

    def __init__(self, is_verbose=False):
        self.is_verbose = is_verbose
        self.design = None  #: Design db instance
        self.isf_filename = ""  #: ISF file name
        self.command_list = []  #: Command list that captures design db content
        self.block_inst = []  #: Selected instances to export
        self.iobank_builder = None  #: ISF builder for IOBank
        self.seu_builder = None  #: ISF builder for SEU Detector
        self.ctrl_builder = None  #: ISF builder for Remote Update
        self.gpio_builder = None  #: ISF builder for GPIO
        self.osc_builder = None  #: ISF builder for OSC
        self.jtag_builder = None  #: ISF builder for JTAG
        self.pll_builder = None  #: ISF builder for PLL
        self.mipi_tx_builder = None  #: ISF builder for MIPI TX
        self.mipi_rx_builder = None  #: ISF builder for MIPI RX
        self.mipi_dphy_tx_builder = None  #: ISF builder for MIPI DPhy TX
        self.mipi_dphy_rx_builder = None  #: ISF builder for MIPI DPhy RX
        self.lvds_tx_builder = None  #: ISF builder for LVDS TX
        self.lvds_rx_builder = None  #: ISF builder for LVDS RX
        self.lvds_bidir_builder = None  #: ISF builder for LVDS Bidir
        self.mipi_hard_dphy_tx_builder: Optional[MIPIHardDPHYTxISFBuilder] = None
        self.mipi_hard_dphy_rx_builder: Optional[MIPIHardDPHYRxISFBuilder] = None
        self.pll_ssc_builder: Optional[PLLSSCISFBuilder] = None
        self.hyper_ram_builder = None
        self.spi_flash_builder = None
        self.ddr_builder = None  #: ISF builder for DDR
        self.ext_flash_builder = None
        self.clockmux_builder = None
        self.soc_builder: Optional[SOCISFBuilder] = None
        self.pcie_builder: Optional[QuadPCIEISFBuilder] = None
        self.lane_exporter: Optional[LaneBaseISFExporter] = None
        self.lane10g_builder: Optional[Lane10GISFBuilder] = None
        self.lane1g_builder: Optional[Lane1GISFBuilder] = None
        self.raw_serdes_builder: Optional[RawSerdesISFBuilder] = None
        self.is_gen_header = True  #: True, generate info header else no
        self.supp_block = []

        from api_service.common.infra import InfraService
        infra_svc = InfraService.get_instance()
        assert infra_svc is not None
        self.msg_svc = infra_svc.get_message_service()

    def build_supported_block_type(self):
        """
        Identify and keep supported API block for ISF export.
        This can differ with API support and it also take into account supported block in device.
        """
        self.supp_block.clear()
        self.supp_block = ["GPIO", "PLL", "IOBANK", "JTAG"]

        if self.design is not None:
            if self.design.get_block_reg(PeriDesign.BlockType.osc) is not None:
                self.supp_block.append("OSC")

            if self.design.is_block_supported(PeriDesign.BlockType.mipi):
                self.supp_block.append("MIPI_TX")
                self.supp_block.append("MIPI_RX")

            if self.design.is_block_supported(PeriDesign.BlockType.mipi_hard_dphy):
                self.supp_block.append("MIPI_DPHY_TX")
                self.supp_block.append("MIPI_DPHY_RX")

            if self.design.is_tesseract_design():
                self.supp_block.append("MIPI_RX_LANE")
                self.supp_block.append("MIPI_TX_LANE")
                self.supp_block.append("LVDS_RX")
                self.supp_block.append("LVDS_TX")
                self.supp_block.append("LVDS_BIDIR")
                self.supp_block.append("SEU")

            if self.design.is_block_supported(PeriDesign.BlockType.lvds):
                self.supp_block.append("LVDS_RX")
                self.supp_block.append("LVDS_TX")

            if self.design.device_setting is not None and \
                self.design.device_setting.ctrl_reg is not None:
                self.supp_block.append("RU")

            if self.design.is_block_supported(PeriDesign.BlockType.hyper_ram):
                self.supp_block.append("HYPERRAM")

            if self.design.is_block_supported(PeriDesign.BlockType.spi_flash):
                self.supp_block.append("SPI_FLASH")

            if self.design.is_block_supported(PeriDesign.BlockType.ddr) or self.design.is_block_supported(PeriDesign.BlockType.adv_ddr):
                self.supp_block.append("DDR")

            if self.design.is_block_supported(PeriDesign.BlockType.pll_ssc):
                self.supp_block.append("PLL_SSC")

            if self.design.device_setting is not None and \
                self.design.device_setting.ext_flash_reg is not None:
                self.supp_block.append("EXT_FLASH")

            if self.design.device_setting is not None and \
                self.design.device_setting.clkmux_reg is not None and \
                self.design.is_tesseract_design():
                self.supp_block.append("CLKMUX")

            if self.design.is_block_supported(PeriDesign.BlockType.soc):
                self.supp_block.append("SOC")

            if self.design.is_block_supported(PeriDesign.BlockType.quad_pcie):
                self.supp_block.append("QUAD_PCIE")

            if self.design.is_block_supported(PeriDesign.BlockType.lane_10g):
                self.supp_block.append("10GBASE_KR")

            if self.design.is_block_supported(PeriDesign.BlockType.lane_1g):
                self.supp_block.append("SGMII")

            if self.design.is_block_supported(PeriDesign.BlockType.raw_serdes):
                self.supp_block.append("PMA_DIRECT")

    def build_block_builder_list(self) -> List:
        """
        Get a list of block builder. The sequence determines writing order and it needs
        to match design db xml registry order.

        :return: A list of block builder object.
        ..note:
          This has to be dynamically generated because list stores the object value.
        """
        return [self.gpio_builder, self.pll_builder, self.osc_builder, self.jtag_builder,
                self.mipi_dphy_rx_builder, self.mipi_dphy_tx_builder,
                self.lvds_rx_builder, self.lvds_tx_builder, self.lvds_bidir_builder,
                self.mipi_hard_dphy_tx_builder, self.mipi_hard_dphy_rx_builder,
                self.hyper_ram_builder, self.spi_flash_builder, self.ddr_builder,
                self.mipi_rx_builder, self.mipi_tx_builder, self.pll_ssc_builder,
                self.soc_builder, self.pcie_builder, self.lane10g_builder, self.lane1g_builder,
                self.raw_serdes_builder]

    def get_supported_block_type(self) -> List[str]:
        """
        Get API block type name supported for ISF export.

        :return: A list of block type name
        """
        self.build_supported_block_type()
        return self.supp_block

    def is_block_supported(self, block_type: str) -> bool:
        """
        Check if block type is supported for ISF export.

        :param block_type: API block type name
        :return: True, if supported, else not
        """
        return block_type in self.get_supported_block_type()

    def export_isf(self, design: PeriDesign, block_type: Optional[list] = None, block_inst: Optional[list] = None,
                   isf_file: str = "", export_all_pin: bool = False):
        """
        Export design db to ISF.

        :param design: Design db instance
        :param block_type: See :func:`DesignAPI.export_isf`
        :param block_inst: See :func:`DesignAPI.export_isf`
        :param isf_file: See :func:`DesignAPI.export_isf`
        :param export_all_pin: See :func:`DesignAPI.export_all_pin`
        """

        if block_inst is None:
            block_inst = []

        self.design = design
        self.block_inst = block_inst
        self.build_supported_block_type()
        if block_type is None:
            # ISF support level may differ from API support level
            selected_block_type = self.get_supported_block_type()
        else:
            selected_block_type = block_type

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Export blocks : {selected_block_type}")

        if "IOBANK" in selected_block_type:
            self.iobank_builder = IOBankISFBuilder(design, self.design_api_handle, export_all_pin, self.is_verbose)

        if "SEU" in selected_block_type:
            self.seu_builder = SEUISFBuilder(design, self.design_api_handle, export_all_pin, self.is_verbose)

        if "RU" in selected_block_type:
            self.ctrl_builder = CtrlISFBuilder(design, self.design_api_handle, export_all_pin, self.is_verbose)

        if "GPIO" in selected_block_type:
            self.gpio_builder = GPIOISFBuilder(design, self.design_api_handle, self.block_inst,
                                               export_all_pin, self.is_verbose)

        if "PLL" in selected_block_type:
            if design.is_block_supported(PeriDesign.BlockType.adv_pll):
                self.pll_builder = PLLAdvISFBuilder(design, self.design_api_handle, self.block_inst,
                                                    export_all_pin, self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.pll):
                self.pll_builder = PLLISFBuilder(design, self.design_api_handle, self.block_inst,
                                                 export_all_pin, self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.comp_pll):
                self.pll_builder = PLLCompISFBuilder(design, self.design_api_handle, self.block_inst,
                                                     export_all_pin, self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.efx_pll_v3_comp):
                self.pll_builder = PLLV3ComplexISFBuilder(design=design,
                                                   design_api_handle=self.design_api_handle,
                                                   block_inst_name=self.block_inst,
                                                   dump_all_pin=export_all_pin,
                                                   is_verbose=self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.efx_fpll_v1):
                self.pll_builder = EfxFpllV1ISFBuilder(design=design,
                                                   design_api_handle=self.design_api_handle,
                                                   block_inst_name=self.block_inst,
                                                   dump_all_pin=export_all_pin,
                                                   is_verbose=self.is_verbose)

        # We build only if the OSC registry exists
        if "OSC" in selected_block_type:
            if design.is_block_supported(PeriDesign.BlockType.hposc):
                self.osc_builder = HPOSCISFBuilder(design, self.design_api_handle, self.block_inst,
                                                   export_all_pin, self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.adv_osc):
                self.osc_builder = OSCAdvISFBuilder(design, self.design_api_handle, self.block_inst,
                                                    export_all_pin, self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.osc):
                self.osc_builder = OSCISFBuilder(design, self.design_api_handle, self.block_inst,
                                                 export_all_pin, self.is_verbose)

        if "JTAG" in selected_block_type:
            self.jtag_builder = JTAGISFBuilder(design, self.design_api_handle, self.block_inst,
                                               export_all_pin, self.is_verbose)

        if "MIPI_TX" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.mipi):
            self.mipi_tx_builder = MIPITxISFBuilder(design, self.design_api_handle, self.block_inst,
                                                             export_all_pin, self.is_verbose)

        if "MIPI_RX" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.mipi):
            self.mipi_rx_builder = MIPIRxISFBuilder(design, self.design_api_handle, self.block_inst,
                                                             export_all_pin, self.is_verbose)

        if "MIPI_TX_LANE" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.mipi_dphy):
            self.mipi_dphy_tx_builder = MIPIDPhyTxISFBuilder(design, self.design_api_handle, self.block_inst,
                                                             export_all_pin, self.is_verbose)

        if "MIPI_RX_LANE" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.mipi_dphy):
            self.mipi_dphy_rx_builder = MIPIDPhyRxISFBuilder(design, self.design_api_handle, self.block_inst,
                                                             export_all_pin, self.is_verbose)

        if "MIPI_DPHY_TX" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.mipi_hard_dphy):
            self.mipi_hard_dphy_tx_builder = MIPIHardDPHYTxISFBuilder(design=design,
                                                                      design_api_handle=self.design_api_handle,
                                                                      block_inst_name=self.block_inst,
                                                                      dump_all_pin=export_all_pin,
                                                                      is_verbose=self.is_verbose)

        if "MIPI_DPHY_RX" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.mipi_hard_dphy):
            self.mipi_hard_dphy_rx_builder = MIPIHardDPHYRxISFBuilder(design=design,
                                                                      design_api_handle=self.design_api_handle,
                                                                      block_inst_name=self.block_inst,
                                                                      dump_all_pin=export_all_pin,
                                                                      is_verbose=self.is_verbose)

        if "LVDS_TX" in selected_block_type:
            if design.is_block_supported(PeriDesign.BlockType.adv_lvds):
                self.lvds_tx_builder = LVDSTxAdvISFBuilder(design, self.design_api_handle, self.block_inst,
                                                        export_all_pin, self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.lvds):
                self.lvds_tx_builder = LVDSTxISFBuilder(design, self.design_api_handle, self.block_inst,
                                                        export_all_pin, self.is_verbose)

        if "LVDS_RX" in selected_block_type:
            if design.is_block_supported(PeriDesign.BlockType.adv_lvds):
                self.lvds_rx_builder = LVDSRxAdvISFBuilder(design, self.design_api_handle, self.block_inst,
                                                        export_all_pin, self.is_verbose)
            elif design.is_block_supported(PeriDesign.BlockType.lvds):
                self.lvds_rx_builder = LVDSRxISFBuilder(design, self.design_api_handle, self.block_inst,
                                                        export_all_pin, self.is_verbose)

        if design.is_block_supported(PeriDesign.BlockType.adv_lvds):
            if "LVDS_BIDIR" in selected_block_type:
                self.lvds_bidir_builder = LVDSBidirAdvISFBuilder(design, self.design_api_handle, self.block_inst,
                                                                    export_all_pin, self.is_verbose)

        if "HYPERRAM" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.hyper_ram):
            self.hyper_ram_builder = HyperRAMISFBuilder(design=design,
                                                      design_api_handle=self.design_api_handle,
                                                      block_inst_name=self.block_inst,
                                                      dump_all_pin=export_all_pin,
                                                      is_verbose=self.is_verbose)

        if "SPI_FLASH" in selected_block_type and design.is_block_supported(PeriDesign.BlockType.spi_flash):
            self.spi_flash_builder = SPIFlashISFBuilder(design=design,
                                                      design_api_handle=self.design_api_handle,
                                                      block_inst_name=self.block_inst,
                                                      dump_all_pin=export_all_pin,
                                                      is_verbose=self.is_verbose)

        if "DDR" in selected_block_type:
            if design.is_block_supported(PeriDesign.BlockType.ddr):
                self.ddr_builder = DDRISFBuilder(design=design,
                                                design_api_handle=self.design_api_handle,
                                                block_inst_name=self.block_inst,
                                                dump_all_pin=export_all_pin,
                                                is_verbose=self.is_verbose)

            elif design.is_block_supported(PeriDesign.BlockType.adv_ddr):
                self.ddr_builder = DDRAdvISFBuilder(design=design,
                                                design_api_handle=self.design_api_handle,
                                                block_inst_name=self.block_inst,
                                                dump_all_pin=export_all_pin,
                                                is_verbose=self.is_verbose)
        if "EXT_FLASH" in selected_block_type:
            self.ext_flash_builder = ExtFlashISFBuilder(design, self.design_api_handle, export_all_pin, self.is_verbose)

        if "CLKMUX" in selected_block_type:
            self.clockmux_builder = ClockMuxISFBuilder(design, self.design_api_handle, export_all_pin, self.is_verbose)

        if "PLL_SSC" in selected_block_type:
            self.pll_ssc_builder = PLLSSCISFBuilder(
                                        design=design,
                                        design_api_handle=self.design_api_handle,
                                        block_inst_name=self.block_inst,
                                        dump_all_pin=export_all_pin,
                                        is_verbose=self.is_verbose)

        if "SOC" in selected_block_type:
            self.soc_builder = SOCISFBuilder(design=design,
                                             design_api_handle=self.design_api_handle,
                                             block_inst_name=self.block_inst,
                                             dump_all_pin=export_all_pin,
                                             is_verbose=self.is_verbose)

        if "QUAD_PCIE" in selected_block_type:
            self.pcie_builder = QuadPCIEISFBuilder(design=design,
                                             design_api_handle=self.design_api_handle,
                                             block_inst_name=self.block_inst,
                                             dump_all_pin=export_all_pin,
                                             is_verbose=self.is_verbose)

        if "10GBASE_KR" in selected_block_type or "SGMII" in selected_block_type or \
                "PMA_DIRECT" in selected_block_type:
            self.lane_exporter = LaneBaseISFExporter()

            if "10GBASE_KR" in selected_block_type:
                self.lane10g_builder = Lane10GISFBuilder(design=design,
                                                design_api_handle=self.design_api_handle,
                                                block_inst_name=self.block_inst,
                                                dump_all_pin=export_all_pin,
                                                is_verbose=self.is_verbose,
                                                lane_exporter=self.lane_exporter)

            if "SGMII" in selected_block_type:
                self.lane1g_builder = Lane1GISFBuilder(design=design,
                                                design_api_handle=self.design_api_handle,
                                                block_inst_name=self.block_inst,
                                                dump_all_pin=export_all_pin,
                                                is_verbose=self.is_verbose,
                                                lane_exporter=self.lane_exporter)

            if "PMA_DIRECT" in selected_block_type:
                self.raw_serdes_builder = RawSerdesISFBuilder(design=design,
                                                design_api_handle=self.design_api_handle,
                                                block_inst_name=self.block_inst,
                                                dump_all_pin=export_all_pin,
                                                is_verbose=self.is_verbose,
                                                lane_exporter=self.lane_exporter)


        if isf_file == "":
            if design.location == "":
                self.isf_filename = f"./{design.name}.isf"
            else:
                self.isf_filename = f"{design.location}/{design.name}.isf"
        else:
            self.isf_filename = isf_file

        self.command_list.clear()

        # The export file follows these specific sequence. Build all the commands and then write it to the file.
        self.build_set_device_command()
        self.build_create_command()
        self.build_set_property_command()
        self.build_resasg_command()
        self.save()

        self.iobank_builder = None
        self.gpio_builder = None
        self.pll_builder = None
        self.osc_builder = None
        self.jtag_builder = None
        self.mipi_dphy_tx_builder = None
        self.mipi_dphy_rx_builder = None
        self.mipi_hard_dphy_tx_builder = None
        self.mipi_hard_dphy_rx_builder = None
        self.lvds_tx_builder = None
        self.lvds_rx_builder = None
        self.lvds_bidir_builder = None
        self.hyper_ram_builder = None
        self.spi_flash_builder = None
        self.pll_ssc_builder = None
        self.ddr_builder = None
        self.seu_builder = None
        self.ctrl_builder = None
        self.ext_flash_builder = None
        self.clockmux_builder = None

    def build_set_device_command(self):
        if self.iobank_builder is None and self.seu_builder is None and \
            self.ctrl_builder is None and self.ext_flash_builder is None and\
                self.clockmux_builder is None:
            return

        if self.is_gen_header:
                self.command_list.append("\n")
        self.command_list.append("# Device setting")

        if self.iobank_builder is not None:
            cmd_list = self.iobank_builder.build_set_property_command()
            if len(cmd_list) > 0:
                self.command_list += cmd_list

        if self.seu_builder is not None:
            cmd_list = self.seu_builder.build_set_property_command()
            if len(cmd_list) > 0:
                self.command_list += cmd_list

        if self.ctrl_builder is not None:
            cmd_list = self.ctrl_builder.build_set_property_command()
            if len(cmd_list) > 0:
                self.command_list += cmd_list

        if self.ext_flash_builder is not None:
            cmd_list = self.ext_flash_builder.build_set_property_command()
            if len(cmd_list) > 0:
                self.command_list += cmd_list

        if self.clockmux_builder is not None:
            cmd_list = self.clockmux_builder.build_set_property_command()
            if len(cmd_list) > 0:
                self.command_list += cmd_list

    def build_create_command(self):
        """
        Build a series of create instance command
        """

        # TODO : Clean up
        # if self.is_gen_header:
        #     self.command_list.append("\n")

        self.command_list.append("\n# Create instance")

        supp_builder = self.build_block_builder_list()

        for builder in supp_builder:
            if builder is not None:
                cmd_list = builder.build_create_command()
                if len(cmd_list) > 0:
                    self.command_list += cmd_list

    def build_set_property_command(self):
        """
        Build a series of set property command
        """
        self.command_list.append("\n# Set property, non-defaults")

        supp_builder = self.build_block_builder_list()

        for builder in supp_builder:
            if builder is not None:
                cmd_list = builder.build_set_property_command()
                if len(cmd_list) > 0:
                    self.command_list += cmd_list

    def build_resasg_command(self):
        """
        Build a series of resource assignment command
        """
        self.command_list.append("\n# Set resource assignment")

        supp_builder = self.build_block_builder_list()

        for builder in supp_builder:
            if builder is not None:
                cmd_list = builder.build_resasg_command()
                if len(cmd_list) > 0:
                    self.command_list += cmd_list

    def save(self):
        """
        Write commands to ISF
        """
        header_gen = None
        if self.is_gen_header:
            header_gen = ExportGPIODesign(self.design, self.isf_filename)
            header_gen.header_title = "Efinity Interface Configuration"

        with open(self.isf_filename, "w", encoding='UTF-8') as isf_file:
            if header_gen is not None:
                header_gen.write_header(isf_file)

            for cmd in self.command_list:
                isf_file.write(f"{cmd}\n")


@gen_util.freeze_it
class GPIOISFBuilder:
    """
    Build GPIO ISF commands.

    ..notes:
      Supports all GPIO types since there are small differences. If it gets complicated, refactor to separate builder.
      Minimal impact since only exporter use this.
    """

    class CommandType(Enum):
        """
        GPIO command id to be used in ISF generation
        """
        gpio_input = auto()
        gpio_output = auto()
        gpio_inout = auto()
        gpio_clkout = auto()
        gpio_glb_clk = auto()
        gpio_glb_ctrl = auto()
        gpio_pll_clkin = auto()
        gpio_pll_extfb = auto()
        gpio_mipi_clkin = auto()
        gpio_pcie_perstn = auto()
        gpio_unused = auto()
        set_prop = auto()
        assign_pin = auto()
        # GPIOComplex
        gpio_reg_clk = auto()
        gpio_reg_ctrl = auto()
        gpio_vref = auto()

    cmd2str_map = {
        CommandType.gpio_input: "create_input_gpio",
        CommandType.gpio_output: "create_output_gpio",
        CommandType.gpio_inout: "create_inout_gpio",
        CommandType.gpio_clkout: "create_clockout_gpio",
        CommandType.gpio_glb_clk: "create_input_clock_gpio",
        CommandType.gpio_glb_ctrl: "create_global_control_gpio",
        CommandType.gpio_pll_clkin: "create_pll_input_clock_gpio",
        CommandType.gpio_pll_extfb: "create_pll_ext_fb_gpio",
        CommandType.gpio_mipi_clkin: "create_mipi_input_clock_gpio",
        CommandType.gpio_pcie_perstn: "create_pcie_perstn_gpio",
        CommandType.gpio_unused: "create_unused_gpio",
        CommandType.set_prop: "set_property",
        CommandType.assign_pin: "assign_pkg_pin",
        CommandType.gpio_reg_clk: "create_regional_input_clock_gpio",
        #CommandType.gpio_reg_ctrl: "create_regional_control_gpio",
        CommandType.gpio_vref: "create_vref_gpio"
    }  #: Map ISF command to its API string representation. The API must matched with the one in DesignAPI.

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):

        if block_inst_name is None:
            block_inst_name = []

        self.igpio = IntGPIOAPI.build_gpio_api(design)
        self.design_api_handle = design_api_handle
        self.design = design
        self.is_dump_all_pin = dump_all_pin
        self.is_verbose = is_verbose

        from api_service.common.infra import InfraService
        infra_svc = InfraService.get_instance()
        self.msg_svc = infra_svc.get_message_service()

        if self.design is not None:
            self.gpio_reg = self.design.gpio_reg

            if block_inst_name is None or len(block_inst_name) == 0:
                # Export all
                self.gpio_bus_list = self.gpio_reg.get_all_bus()
                self.scalar_gpio_list = self.gpio_reg.get_all_scalar_gpio()
            else:
                # Export selected instances
                self.gpio_bus_list = []
                self.scalar_gpio_list = []
                for name in block_inst_name:
                    inst = self.gpio_reg.get_bus_by_name(name)
                    if inst is not None:
                        self.gpio_bus_list.append(inst)

                for name in block_inst_name:
                    inst = self.gpio_reg.get_gpio_by_name(name)
                    if inst is not None:
                        self.scalar_gpio_list.append(inst)

            def get_name(db_inst):
                return db_inst.name

            # Sorted by instance name
            self.gpio_bus_list.sort(key=get_name)
            self.scalar_gpio_list.sort(key=get_name)

        else:
            self.gpio_reg = None
            self.gpio_bus_list = None
            self.scalar_gpio_list = None

    def build_create_command(self) -> list:
        """
        Build a series of create bus and scalar gpio command

        :return: A list of commands
        """
        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, "Generate GPIO build commands")

        for bus in self.gpio_bus_list:
            cmd = self._build_bus_create_command(bus)

            if cmd == "":
                continue

            command_list.append(cmd)

        for gpio in self.scalar_gpio_list:
            cmd = self._build_gpio_create_command(gpio)

            if cmd == "":
                continue

            command_list.append(cmd)

        return command_list

    def build_set_property_command(self) -> list:
        """
        Build a series of gpio set property command

        :return: A list of commands
        """

        command_list = []
        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, "Generate GPIO bus configure commands")

        for bus in self.gpio_bus_list:

            # Bus property, using designated template gpio.
            # Doesn't include base bus property itself since it is handled as part of create instance.
            temp_gpio = bus.get_first_member()
            cmd_list = self._build_bus_property_command(temp_gpio, bus.name)
            if len(cmd_list) > 0:
                command_list += cmd_list

            # The rest of the bus member, in case it is changed against the template gpio
            gpio_bus_member = bus.get_member_list()
            for gpio in gpio_bus_member:
                if gpio == temp_gpio:
                    continue
                cmd_list = self._build_bus_member_property_command(gpio, temp_gpio)
                if len(cmd_list) > 0:
                    command_list += cmd_list

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, "Generate GPIO bus configure commands")

        gpio_prop = None
        for gpio in self.scalar_gpio_list:
            if gpio_prop is None:
                gpio_prop = GPIOProp.build_gpio_prop(db_inst=gpio, design_db=self.design)
            else:
                gpio_prop.set_gpio(gpio)

            cmd_list = self._build_gpio_property_command(gpio, gpio_prop)
            if len(cmd_list) > 0:
                command_list += cmd_list

        return command_list

    def build_resasg_command(self) -> list:
        """
        Build a series of gpio resource assignment commands. Using package pin command rather than resource.

        :return: A list of commands
        """

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, "Generate GPIO assign pin commands")

        command_list = []
        for bus in self.gpio_bus_list:
            gpio_bus_member = bus.get_member_list()
            for gpio in gpio_bus_member:
                cmd_list = self._build_gpio_resasg_command(gpio)
                if len(cmd_list) > 0:
                    command_list += cmd_list

        for gpio in self.scalar_gpio_list:
            cmd_list = self._build_gpio_resasg_command(gpio)
            if len(cmd_list) > 0:
                command_list += cmd_list

        return command_list

    def _build_gpio_create_command(self, gpio: gd.GPIO) -> str:
        """
        Build create gpio command. Must match with function signature in DesignAPI.

        :param gpio: Target gpio
        :return: A DesignAPI command
        """

        parameter = f"\"{gpio.name}\""
        command = ""

        is_complex = False
        if isinstance(gpio, GPIOComplex):
            is_complex = True

        if gpio.mode == gd.GPIO.PadModeType.input:
            gp_input = gpio.input
            assert gp_input is not None
            if gp_input.conn_type == gp_input.ConnType.gclk_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_glb_clk]}({parameter})"
            elif gp_input.conn_type == gp_input.ConnType.gctrl_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_glb_ctrl]}({parameter})"
            elif gp_input.conn_type == gp_input.ConnType.pll_clkin_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_pll_clkin]}({parameter})"
            elif gp_input.conn_type == gp_input.ConnType.pll_extfb_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_pll_extfb]}({parameter})"
            elif gp_input.conn_type == gp_input.ConnType.mipi_clkin_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_mipi_clkin]}({parameter})"
            elif is_complex and gp_input.conn_type == gp_input.ConnType.pcie_perstn_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_pcie_perstn]}({parameter})"
            elif is_complex and gp_input.conn_type == gp_input.ConnType.rclk_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_reg_clk]}({parameter})"
            elif is_complex and gp_input.conn_type == gp_input.ConnType.rctrl_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_reg_ctrl]}({parameter})"
            elif is_complex and gp_input.conn_type == gp_input.ConnType.vref_conn:
                command = f"{self.cmd2str_map[self.CommandType.gpio_vref]}({parameter})"
            else:
                command = f"{self.cmd2str_map[self.CommandType.gpio_input]}({parameter})"

        elif gpio.mode == gd.GPIO.PadModeType.output:
            command = f"{self.cmd2str_map[self.CommandType.gpio_output]}({parameter})"
        elif gpio.mode == gd.GPIO.PadModeType.inout:
            command = f"{self.cmd2str_map[self.CommandType.gpio_inout]}({parameter})"
        elif gpio.mode == gd.GPIO.PadModeType.clkout:
            command = f"{self.cmd2str_map[self.CommandType.gpio_clkout]}({parameter})"
        elif gpio.mode == gd.GPIO.PadModeType.none:
            command = f"{self.cmd2str_map[self.CommandType.gpio_unused]}({parameter})"

        if command is not None:
            command = f"{self.design_api_handle}.{command}"

        return command

    def _build_bus_create_command(self, bus: gd.GPIOBus) -> str:
        """
        Build create bus command. Must match with function signature in DesignAPI.

        :param gpio: Target bus
        :return: A DesignAPI command
        """

        lsb = bus.get_lsb_index()
        msb = bus.get_msb_index()
        mode = bus.get_mode()
        command = ""

        parameter = f"\"{bus.name}\",{msb},{lsb}"
        if mode == gd.GPIO.PadModeType.input:
            command = f"{self.cmd2str_map[self.CommandType.gpio_input]}({parameter})"
        elif mode == gd.GPIO.PadModeType.output:
            command = f"{self.cmd2str_map[self.CommandType.gpio_output]}({parameter})"
        elif mode == gd.GPIO.PadModeType.inout:
            command = f"{self.cmd2str_map[self.CommandType.gpio_inout]}({parameter})"

        if command is not None:
            command = f"{self.design_api_handle}.{command}"

        return command

    def _build_bus_property_command(self, gpio: gd.GPIO, bus_name) -> list:
        """
        Build bus property setting command. Must match with function signature in DesignAPI.

        :param gpio: Target bus reference gpio for bus level property setting
        :param bus_name: Target bus name
        :return: A list of DesignAPI command
        """

        command_list = []
        gpio_prop = GPIOProp.build_gpio_prop(db_inst=gpio, design_db=self.design)
        prop_inspector = gpio_prop.build_inspector()
        edited_prop = prop_inspector.get_edited_prop(gpio, register_all_pin=self.is_dump_all_pin)

        # Pin name is set at bus member level
        pin_prop = ["IN_PIN", "OUT_PIN", "OE_PIN", "OEN_PIN", "IN_LO_PIN", "IN_HI_PIN", "OUT_LO_PIN", "OUT_HI_PIN"]
        pin_command = []

        for prop, val in edited_prop.items():
            if prop in pin_prop:
                # Ref gpio prop, should be first member
                # Same regardless if OEN_PIN or not
                parameter = f"\"{gpio.name}\",\"{prop}\",\"{val}\""
                command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
                pin_command.append(command)

            else:
                parameter = f"\"{bus_name}\",\"{prop}\",\"{val}\""
                command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
                command_list.append(command)

        command_list += pin_command

        return command_list

    def _build_bus_member_property_command(self, gpio: gd.GPIO, ref_gpio) -> list:
        """
        Build bus member property setting command. Must match with function signature in DesignAPI.

        :param gpio: Target gpio, a bus member
        :param ref_gpio: Reference gpio for the bus
        :return: A list of DesignAPI command
        """

        command_list = []
        gpio_prop = GPIOProp.build_gpio_prop(db_inst=gpio, design_db=self.design)
        prop_inspector = gpio_prop.build_inspector()
        edited_prop = prop_inspector.get_edited_prop(gpio, ref_gpio, register_all_pin=self.is_dump_all_pin)

        for prop, val in edited_prop.items():
            parameter = f"\"{gpio.name}\",\"{prop}\",\"{val}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        return command_list

    def _build_gpio_property_command(self, gpio: gd.GPIO, gpio_prop) -> list:
        """
        Build gpio property setting command. Must match with function signature in DesignAPI.

        :param gpio: Target gpio
        :param gpio_prop: GPIO property object
        :return: A list of DesignAPI command
        """

        command_list = []
        prop_inspector = gpio_prop.build_inspector()
        edited_prop = prop_inspector.get_edited_prop(gpio, register_all_pin=self.is_dump_all_pin)

        for prop, val in edited_prop.items():
            parameter = f"\"{gpio.name}\",\"{prop}\",\"{val}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        return command_list

    def _build_gpio_resasg_command(self, gpio: gd.GPIO) -> list:
        """
        Build gpio resource assignment command using package pin.
        Must match with function signature in DesignAPI.

        :param gpio: Target gpio
        :return: A list of DesignAPI command
        """

        command_list = []
        if gpio.gpio_def != "":
            pin = self.igpio.get_pkg_pin(gpio)
            parameter = f"\"{gpio.name}\",\"{pin}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.assign_pin]}({parameter})"
            command_list.append(command)

        return command_list


class OSCISFBuilder:
    """
    Build OSC ISF commands
    """

    class CommandType(Enum):
        """
        OSC command id to be used in ISF generation
        """
        create_inst = auto()
        set_prop = auto()
        assign_resource = auto()

    cmd2str_map = {
        CommandType.create_inst: "create_block",
        CommandType.set_prop: "set_property",
        CommandType.assign_resource: "assign_resource"
    }  #: Map ISF command to its API string representation. The API must matched with the one in DesignAPI.

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):

        if block_inst_name is None:
            block_inst_name = []

        self.block_reg = design.osc_reg
        self.db_block_type = PeriDesign.BlockType.osc
        self.iosc = IntOSCAPI.build_osc_api(design, osc_block_type=design.osc_type)
        self.design_api_handle = design_api_handle
        self.design = design
        self.is_dump_all_pin = dump_all_pin
        self.is_verbose = is_verbose
        self.is_edited = False

        from api_service.common.infra import InfraService
        infra_svc = InfraService.get_instance()
        self.msg_svc = infra_svc.get_message_service()

        if self.design is not None and (design.is_block_supported(design.osc_type)):
            self.osc_reg = self.design.osc_reg

            if block_inst_name is None or len(block_inst_name) == 0:
                # Export all
                self.osc_list = self.osc_reg.get_all_inst()
            else:
                # Export selected instances
                self.osc_list = []
                for name in block_inst_name:
                    inst = self.osc_reg.get_inst_by_name(name)
                    if inst is not None:
                        self.osc_list.append(inst)

            def get_name(db_inst):
                return db_inst.name

            # Sorted by instance name
            self.osc_list.sort(key=get_name)

        else:
            self.osc_reg = None
            self.osc_list = []

    def build_create_command(self) -> list:
        """
        Build a series of create osc command

        :return: A list of commands
        """
        command_list = []
        block_type = APIObject.otype2str_map.get(APIObject.ObjectType.osc, "")

        if block_type != "":

            if self.is_verbose:
                self.msg_svc.write(MsgLevel.info, "Generate OSC build commands")

            for osc in self.osc_list:
                if osc is not None:
                    parameter = f"\"{osc.name}\",\"{block_type}\""
                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"

                    command_list.append(command)

        return command_list

    def build_set_property_command(self) -> list:
        """
        Build a series of osc set property command

        :return: A list of commands
        """

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, "Generate OSC configure commands")

        osc_prop = None
        for osc in self.osc_list:
            if osc_prop is None:
                osc_prop = OSCProp.build_osc_prop(
                    block_type=APIObject.ObjectType.osc, db_inst=osc, design_db=self.design)

            else:
                osc_prop.set_osc(osc)

            cmd_list = self._build_osc_property_command(osc, osc_prop)
            if len(cmd_list) > 0:
                command_list += cmd_list

        return command_list

    def _build_osc_property_command(self, osc: oscd.OSC, osc_prop) -> list:
        """
        Build osc property setting command. Must match with function signature in DesignAPI.

        :param osc: Target osc
        :param osc_prop: OSC property object
        :return: A list of DesignAPI command
        """

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, "Generate OSC configure commands")

        command_list = []
        prop_inspector = osc_prop.build_inspector()

        # PT-1470: Make OSC to always export all by default
        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(osc, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(osc)

        block_type = APIObject.otype2str_map.get(APIObject.ObjectType.osc, "")

        for prop, val in export_prop.items():
            parameter = f"\"{osc.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        return command_list

    def build_resasg_command(self) -> list:
        """
        Build a series of osc resource assignment commands.

        :return: A list of commands
        """

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, "Generate OSC assign resource commands")

        command_list = []
        block_type = APIObject.otype2str_map.get(APIObject.ObjectType.osc, "")

        if block_type != "":
            for osc in self.osc_list:
                if osc is not None and osc.osc_def != "":
                    parameter = f"\"{osc.name}\",\"{osc.osc_def}\",\"{block_type}\""
                    command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.assign_resource]}({parameter})"
                    command_list.append(command)

        return command_list


class OSCAdvISFBuilder(OSCISFBuilder):
    """
    ISF commands builder for OSC Advance
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.osc_reg
        self.db_block_type = PeriDesign.BlockType.adv_osc

    def _build_osc_property_command(self, osc: OSCAdvance, osc_prop) -> list:
        """
        Build osc property setting command. Must match with function signature in DesignAPI.

        :param osc: Target osc
        :param osc_prop: OSC property object
        :return: A list of DesignAPI command
        """
        return super()._build_osc_property_command(osc, osc_prop)


@gen_util.freeze_it
class HPOSCISFBuilder(OSCISFBuilder):
    """
    ISF commands builder for High Precision Oscillator
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.osc_reg
        self.db_block_type = PeriDesign.BlockType.hposc

    def _build_osc_property_command(self, osc: HPOSC, osc_prop) -> list:
        """
        Build osc property setting command. Must match with function signature in DesignAPI.

        :param osc: Target osc
        :param osc_prop: OSC property object
        :return: A list of DesignAPI command
        """
        return super()._build_osc_property_command(osc, osc_prop)


class BlockISFBuilder(ABC):
    """
    Base ISF commands builder for block
    """

    class CommandType(Enum):
        """
        Block command id to be used in ISF generation
        """
        create_inst = auto()
        set_prop = auto()
        assign_resource = auto()

    cmd2str_map = {
        CommandType.create_inst: "create_block",
        CommandType.set_prop: "set_property",
        CommandType.assign_resource: "assign_resource"
    }  #: Map ISF command to its API string representation. The API must matched with the one in DesignAPI.

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: List = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        """
        Remarks:
        block_inst_name = None means to export all blocks
        """
        self.is_edited = False
        self.iblock = self.build_block_api(design)  #: API block type
        self.design_api_handle = design_api_handle
        self.design = design
        self.is_dump_all_pin = dump_all_pin
        self.block_reg: Optional[PeriDesignRegistry] = None
        self.api_block_type: Optional[APIObject.ObjectType] = None
        self.db_block_type: Optional[PeriDesign.BlockType] = None
        self.block_list: List[Any] = []  #: A list of design db block instance
        self.block_inst_name = block_inst_name

        self.is_verbose = is_verbose

        from api_service.common.infra import InfraService
        infra_svc = InfraService.get_instance()
        self.msg_svc = infra_svc.get_message_service()

        # Cannot build block list since parameter is not set.

    @abstractmethod
    def build_block_api(self, design: PeriDesign) -> Any:
        """
        Build block internal API handler

        :param design: Design db
        :return: API handler
        """

    @abstractmethod
    def build_block_prop(self, design: PeriDesign) -> Any:
        """
        Build block property handler

        :param design: Design db
        :return: Block property handler
        """

    def build_all_prop(self) -> Dict:
        """
        Generate a map of property name and value.

        :return: A property map
        """
        block_prop = self.build_block_prop(self.design)
        if block_prop is not None:
            return block_prop.get_all_property()

        return {}

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        """
        Build a list of design db block instances.
        If instance name not specified, it will retrieve all.

        :param block_inst_name: A list of instance name
        """
        if self.design is not None and self.design.is_block_supported(self.db_block_type):

            self.block_reg = self.design.get_block_reg(self.db_block_type)
            if block_inst_name is None or len(block_inst_name) == 0:
                # Export all
                self.block_list = self.block_reg.get_all_inst()
            else:
                # Export selected instances
                self.block_list = []
                for name in block_inst_name:
                    inst = self.block_reg.get_inst_by_name(name)
                    if inst is not None:
                        self.block_list.append(inst)

            def get_name(db_inst):
                return db_inst.name

            # Sorted by instance name
            self.block_list.sort(key=get_name)

        else:
            self.block_reg = None
            self.block_list = []

    def build_create_command(self) -> List[str]:
        """
        Build a series of create block command

        :return: A list of commands
        """
        assert self.api_block_type is not None

        self.build_block_list(self.block_inst_name)

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} build commands")

        command_list = []
        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if self.api_block_type != "":
            for block in self.block_list:
                if block is not None:
                    parameter = f"\"{block.name}\",\"{block_type}\""
                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"

                    command_list.append(command)

        return command_list

    def build_set_property_command(self) -> List[str]:
        """
        Build a series of block set property command

        :return: A list of commands
        """

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} configure commands")

        block_prop = self.build_block_prop(self.design)
        for block_inst in self.block_list:
            block_prop.set_block(block_inst)
            cmd_list = self._build_block_property_command(block_inst, block_prop)
            if len(cmd_list) > 0:
                command_list += cmd_list

        return command_list

    def _build_block_property_command(self, block_inst, block_prop) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        for prop, val in export_prop.items():
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        return command_list

    def build_resasg_command(self) -> List[str]:
        """
        Build a series of block resource assignment commands.

        :return: A list of commands
        """
        assert self.api_block_type is not None

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} assign resource commands")

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if block_type != "":
            for block in self.block_list:
                if block is not None and block.get_device() != "":
                    parameter = f"\"{block.name}\",\"{block.get_device()}\",\"{block_type}\""
                    command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.assign_resource]}({parameter})"
                    command_list.append(command)

        return command_list


class PLLISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for PLL
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.pll_reg
        self.db_block_type = PeriDesign.BlockType.pll
        self.api_block_type = APIObject.ObjectType.pll

    def build_block_api(self, design):
        return IntPLLAPI.build_pll_api(design)

    def build_block_prop(self, design):
        return PLLProp.build_pll_prop(design_db=design)


class PLLAdvISFBuilder(PLLISFBuilder):
    """
    ISF commands builder for PLL Advance
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.pll_reg
        self.db_block_type = PeriDesign.BlockType.adv_pll


class PLLCompISFBuilder(PLLAdvISFBuilder):
    """
    ISF commands builder for PLL Complex
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.pll_reg
        self.db_block_type = PeriDesign.BlockType.comp_pll

class PLLV3ComplexISFBuilder(PLLCompISFBuilder):
    """
    ISF commands builder for V3Complex (Ti180)
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.pll_reg
        self.db_block_type = PeriDesign.BlockType.efx_pll_v3_comp

class EfxFpllV1ISFBuilder(PLLV3ComplexISFBuilder):
    """
    ISF commands builder for EFX_FPLL_V1 (Ti375)
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.pll_reg
        self.db_block_type = PeriDesign.BlockType.efx_fpll_v1

class JTAGISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for JTAG
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.jtag_reg
        self.db_block_type = PeriDesign.BlockType.jtag
        self.api_block_type = APIObject.ObjectType.jtag

    def build_block_api(self, design: PeriDesign):
        return IntJTAGAPI.build_jtag_api(design)

    def build_block_prop(self, design: PeriDesign):
        return JTAGProp.build_jtag_prop(design_db=design)


class MIPITxISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for MIPI Tx
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: Optional[List[str]] = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.mipi_reg
        self.db_block_type = PeriDesign.BlockType.mipi
        self.api_block_type = APIObject.ObjectType.mipi_tx

    def build_block_api(self, design):
        mipi_api = IntMIPIAPI()
        mipi_api.set_design(design)
        return mipi_api

    def build_block_prop(self, design):
        return MIPITxProp.build_mipi_prop(design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)

        def filter_tx(block_inst: MIPI):
            if block_inst.ops_type == MIPI.OpsType.op_tx:
                return True
            return False

        self.block_list: List[MIPI] = list(filter(filter_tx, self.block_list))

    def _build_block_property_command(self, block_inst: MIPI, block_prop: MIPITxProp) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        block_prop.db_inst = block_inst
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        set_front = ("PHY_FREQ","ESC_CLK_FREQ")

        for prop_name in set_front:
            val = export_prop.get(prop_name)
            _set_command(prop_name, val)

        for prop, val in export_prop.items():
            if prop in set_front:
                continue
            _set_command(prop, val)

        return command_list


class MIPIRxISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for MIPI Rx
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.mipi_reg
        self.db_block_type = PeriDesign.BlockType.mipi
        self.api_block_type = APIObject.ObjectType.mipi_rx

    def build_block_api(self, design):
        mipi_api = IntMIPIAPI()
        mipi_api.set_design(design)
        return mipi_api

    def build_block_prop(self, design):
        return MIPIRxProp.build_mipi_prop(design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)

        def filter_rx(block_inst: MIPI):
            if block_inst.ops_type == MIPI.OpsType.op_rx:
                return True
            return False

        self.block_list: List[MIPI] = list(filter(filter_rx, self.block_list))

    def _build_block_property_command(self, block_inst, block_prop) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        block_prop.db_inst = block_inst
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)


        set_front = ("STATUS_EN", "CAL_CLK_FREQ")

        status_enable = export_prop.get("STATUS_EN", None)
        status_pin_prop = ("CLEAR_PIN", "ERROR_PIN", "ULPS_CLK_PIN", "ULPS_PIN")

        for prop_name in set_front:
            val = export_prop.get(prop_name)
            _set_command(prop_name, val)

        for prop, val in export_prop.items():
            if prop in set_front:
                continue
            elif status_enable == '0' and prop in status_pin_prop:
                continue
            _set_command(prop, val)

        return command_list


class MIPIDPhyTxISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for MIPI DPhy Tx
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.mipi_dphy_reg
        self.db_block_type = PeriDesign.BlockType.mipi_dphy
        self.api_block_type = APIObject.ObjectType.mipi_dphy_tx

    def build_block_api(self, design):
        mipi_api = IntMIPIDPhyAPI()
        mipi_api.set_design(design)
        return mipi_api

    def build_block_prop(self, design):
        return MIPIDPhyTxProp.build_mipi_dphy_prop(design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)

        def filter_tx(block_inst):
            if block_inst.ops_type == MIPIDPhy.OpsType.op_tx:
                return True
            return False

        self.block_list = list(filter(filter_tx, self.block_list))

    def build_create_command(self) -> List[str]:
        assert self.block_inst_name is not None
        assert self.api_block_type is not None

        self.build_block_list(self.block_inst_name)

        command_list = []
        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} build commands")

        if self.api_block_type != "":
            for block in self.block_list:
                if block is not None:
                    tx_info = block.tx_info
                    if tx_info.mode == MIPIDPhyTx.ModeType.clock:
                        extra_param = "mode=\"CLOCK_LANE\""
                    else:
                        extra_param = "mode=\"DATA_LANE\""

                    parameter = f"\"{block.name}\",\"{block_type}\",{extra_param}"
                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"

                    command_list.append(command)

        return command_list

    def _build_block_property_command(self, block_inst: MIPIDPhyRx, block_prop: MIPIDPhyRxProp) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        block_prop.db_inst = block_inst
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        for prop, val in export_prop.items():
            # Set in create command
            if prop in ('MODE'):
                continue
            _set_command(prop, val)

        return command_list


class MIPIDPhyRxISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for MIPI DPhy Rx
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.mipi_dphy_reg
        self.db_block_type = PeriDesign.BlockType.mipi_dphy
        self.api_block_type = APIObject.ObjectType.mipi_dphy_rx

    def build_block_api(self, design):
        mipi_api = IntMIPIDPhyAPI()
        mipi_api.set_design(design)
        return mipi_api

    def build_block_prop(self, design):
        return MIPIDPhyRxProp.build_mipi_dphy_prop(design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)

        def filter_rx(block_inst):
            if block_inst.ops_type == MIPIDPhy.OpsType.op_rx:
                return True
            return False

        self.block_list = list(filter(filter_rx, self.block_list))

    def build_create_command(self) -> List[str]:
        assert self.block_inst_name is not None
        assert self.api_block_type is not None

        self.build_block_list(self.block_inst_name)

        command_list = []
        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} build commands")

        if self.api_block_type != "":
            for block in self.block_list:
                if block is not None:
                    rx_info = block.rx_info
                    if rx_info.mode == MIPIDPhyRx.ModeType.clock:
                        if rx_info.conn_type == MIPIDPhyRx.ConnType.rclk_conn:
                            extra_param = "mode=\"CLOCK_LANE\",conn_type=\"RCLK\""
                        else:
                            extra_param = "mode=\"CLOCK_LANE\",conn_type=\"GCLK\""
                    else:
                        extra_param = "mode=\"DATA_LANE\""

                    parameter = f"\"{block.name}\",\"{block_type}\",{extra_param}"

                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"

                    command_list.append(command)

        return command_list

    def _build_block_property_command(self, block_inst: MIPIDPhyRx, block_prop: MIPIDPhyRxProp) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        block_prop.db_inst = block_inst
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        for prop, val in export_prop.items():
            # Set in create command
            if prop in ('CONN_TYPE', 'MODE'):
                continue
            _set_command(prop, val)

        return command_list


class MIPIHardDPHYTxISFBuilder(BlockISFBuilder):

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: List = None, dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name=block_inst_name, dump_all_pin=dump_all_pin, is_verbose=is_verbose)
        self.block_reg = self.design.mipi_hard_dphy_reg
        self.api_block_type = APIObject.ObjectType.mipi_hard_dphy_tx
        self.db_block_type = PeriDesign.BlockType.mipi_hard_dphy

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntMIPIHardDPHYAPI(design)
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return MIPIHardDPHYTxProp(block_type=self.api_block_type, design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)
        self.block_list = list(filter(lambda inst: isinstance(inst, MIPIHardDPHYTx), self.block_list))


class MIPIHardDPHYRxISFBuilder(BlockISFBuilder):

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: List = None, dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name=block_inst_name, dump_all_pin=dump_all_pin, is_verbose=is_verbose)
        self.block_reg = self.design.mipi_hard_dphy_reg
        self.api_block_type = APIObject.ObjectType.mipi_hard_dphy_rx
        self.db_block_type = PeriDesign.BlockType.mipi_hard_dphy

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntMIPIHardDPHYAPI(design)
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return MIPIHardDPHYRxProp(block_type=self.api_block_type, design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)
        self.block_list = list(filter(lambda inst: isinstance(inst, MIPIHardDPHYRx), self.block_list))


class PLLSSCISFBuilder(BlockISFBuilder):

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: Optional[List] = None, dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name=block_inst_name, dump_all_pin=dump_all_pin, is_verbose=is_verbose) # type: ignore
        self.block_reg = self.design.pll_ssc_reg
        self.api_block_type = APIObject.ObjectType.pll_ssc
        self.db_block_type = PeriDesign.BlockType.pll_ssc

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntPLLSSCAPI()
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return PLLSSCProp(block_type=self.api_block_type, design_db=design)


class SOCISFBuilder(BlockISFBuilder):
    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: Optional[List] = None, dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name=block_inst_name, dump_all_pin=dump_all_pin, is_verbose=is_verbose) # type: ignore
        self.block_reg = self.design.pll_ssc_reg
        self.api_block_type = APIObject.ObjectType.soc
        self.db_block_type = PeriDesign.BlockType.soc

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntSOCAPI()
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return SOCProp(block_type=self.api_block_type, design_db=design)

class QuadPCIEISFBuilder(BlockISFBuilder):
    def __init__(self, design: PeriDesign, design_api_handle: str,
                 block_inst_name: Optional[List] = None, dump_all_pin: bool = False,
                 is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name=block_inst_name, dump_all_pin=dump_all_pin, is_verbose=is_verbose) # type: ignore

        self.block_reg = self.design.quad_pcie_reg
        self.api_block_type = APIObject.ObjectType.quad_pcie
        self.db_block_type = PeriDesign.BlockType.quad_pcie

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntQuadPCIEAPI()
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return QuadPCIEProp(block_type=self.api_block_type, design_db=design)

class LaneBaseISFExporter:
    def __init__(self) -> None:
        self.used_cmn_inst = set()


class LaneBaseISFBuilder(BlockISFBuilder):
    def __init__(self, design: PeriDesign, design_api_handle: str, lane_exporter: LaneBaseISFExporter,
                 block_inst_name: Optional[List] = None, dump_all_pin: bool = False,
                 is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose) # type: ignore
        self.lane_exporter = lane_exporter
        self.api_block_type = self.get_api_blk_type()
        self.db_block_type = self.get_db_blk_type()
        self.block_reg = self.get_blk_reg()

    def get_blk_reg(self):
        return self.design.get_block_reg(self.db_block_type)

    @abstractmethod
    def get_api_blk_type(self) -> APIObject.ObjectType:
        pass

    @abstractmethod
    def get_db_blk_type(self) -> PeriDesign.BlockType:
        pass

    def build_create_command(self) -> List[str]:
        assert self.block_inst_name is not None
        assert self.api_block_type is not None

        self.build_block_list(self.block_inst_name)

        command_list = []
        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} build commands")

        if self.api_block_type != "":
            for block in self.block_list:
                if block is not None:
                    extra_param = self.gen_extra_create_parameter(block)
                    parameter = f"\"{block.name}\",\"{block_type}\""
                    if extra_param != "":
                        parameter += f",{extra_param}"
                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"

                    command_list.append(command)

        return command_list

    def gen_extra_create_parameter(self, inst):
        extra_param_list = []
        if self.block_reg is None:
            return extra_param_list
        cmn_inst = self.block_reg.get_cmn_inst_by_lane_name(inst.name) # type: ignore

        if cmn_inst is not None and cmn_inst.name not in self.lane_exporter.used_cmn_inst:
            extra_param_list.append(f"cmn_name=\"{cmn_inst.name}\"")
            self.lane_exporter.used_cmn_inst.add(cmn_inst.name)

        extra_param = ", ".join(extra_param_list)
        return extra_param


class Lane10GISFBuilder(LaneBaseISFBuilder):
    def get_api_blk_type(self) -> APIObject.ObjectType:
        return APIObject.ObjectType.lane_10g

    def get_db_blk_type(self) -> PeriDesign.BlockType:
        return PeriDesign.BlockType.lane_10g

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntLane10GAPI()
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return Lane10GProp(block_type=self.api_block_type, design_db=design)


class Lane1GISFBuilder(LaneBaseISFBuilder):
    def get_api_blk_type(self) -> APIObject.ObjectType:
        return APIObject.ObjectType.sgmii

    def get_db_blk_type(self) -> PeriDesign.BlockType:
        return PeriDesign.BlockType.lane_1g

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntLane1GAPI()
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return Lane1GProp(block_type=self.api_block_type, design_db=design)


class RawSerdesISFBuilder(LaneBaseISFBuilder):
    def get_api_blk_type(self) -> APIObject.ObjectType:
        return APIObject.ObjectType.raw_serdes

    def get_db_blk_type(self) -> PeriDesign.BlockType:
        return PeriDesign.BlockType.raw_serdes

    def build_block_api(self, design: PeriDesign) -> IntBlockAPI:
        iapi = IntRawSerdesAPI()
        iapi.set_design(design)
        return iapi

    def build_block_prop(self, design: PeriDesign) -> BaseProp:
        return RawSerdesProp(block_type=self.api_block_type, design_db=design)

class LVDSTxISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for LVDS Tx
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.lvds_reg
        self.db_block_type = PeriDesign.BlockType.lvds
        self.api_block_type = APIObject.ObjectType.lvds_tx
        self.presenter: Any = LVDSTxPresenter

    def build_block_api(self, design):
        lvds_api = IntLVDSAPI()
        lvds_api.set_design(design)
        return lvds_api

    def build_block_prop(self, design):
        return LVDSTxProp.build_lvds_prop(design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)

        def filter_tx(block_inst: LVDS):
            if block_inst.ops_type == block_inst.OpsType.op_tx:
                return True
            return False

        self.block_list: List[LVDS] = list(filter(filter_tx, self.block_list))

    def build_create_command(self) -> List[str]:

        assert self.block_inst_name is not None
        assert self.api_block_type is not None

        self.build_block_list(self.block_inst_name)

        command_list = []
        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} build commands")

        if self.api_block_type != "":
            for block in self.block_list:
                if block is not None:
                    tx_info = block.tx_info
                    extra_param = self.gen_extra_create_parameter(tx_info, self.presenter)

                    parameter = f"\"{block.name}\",\"{block_type}\",{extra_param}"
                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"

                    command_list.append(command)

        return command_list

    @staticmethod
    def gen_extra_create_parameter(tx_info: LVDSTx, presenter: LVDSTxBasePresenter):
        extra_param_list = []

        api_mode = presenter.mode_db2api_map.get(tx_info.mode, None)

        if api_mode is not None:
            extra_param_list.append(f"tx_mode=\"{api_mode}\"")

        extra_param = ", ".join(extra_param_list)
        return extra_param

    def _build_block_property_command(self, block_inst: LVDS, block_prop: LVDSTxProp) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        block_prop.db_inst = block_inst
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        for prop, val in export_prop.items():
            # Invalid option (Disabled serialization)
            if prop == "TX_SER" and val == "1":
                continue
            _set_command(prop, val)

        return command_list


class LVDSRxISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for LVDS Rx
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.lvds_reg
        self.db_block_type = PeriDesign.BlockType.lvds
        self.api_block_type = APIObject.ObjectType.lvds_rx
        self.presenter: Any = LVDSRxPresenter

    def build_block_api(self, design):
        lvds_api = IntLVDSAPI()
        lvds_api.set_design(design)
        return lvds_api

    def build_block_prop(self, design):
        return LVDSRxProp.build_lvds_prop(design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)

        def filter_rx(block_inst: LVDS):
            if block_inst.ops_type == block_inst.OpsType.op_rx:
                return True
            return False

        self.block_list = list(filter(filter_rx, self.block_list))

    def build_create_command(self) -> List[str]:

        assert self.block_inst_name is not None
        assert self.api_block_type is not None

        self.build_block_list(self.block_inst_name)

        command_list = []
        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} build commands")

        if self.api_block_type != "":
            for block in self.block_list:
                if block is not None:
                    rx_info = block.rx_info
                    extra_param = self.gen_extra_create_parameter(rx_info, self.presenter)

                    if extra_param == "":
                        parameter = f"\"{block.name}\",\"{block_type}\""
                    else:
                        parameter = f"\"{block.name}\",\"{block_type}\",{extra_param}"

                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"

                    command_list.append(command)

        return command_list

    @staticmethod
    def gen_extra_create_parameter(rx_info: LVDSRx, presenter: LVDSRxBasePresenter):
        extra_param_list = []

        api_mode = presenter.conn_type_db2api_map.get(rx_info.conn_type, None)

        if api_mode is not None:
            extra_param_list.append(f"rx_conn_type=\"{api_mode}\"")

        extra_param = ", ".join(extra_param_list)
        return extra_param

    def _build_block_property_command(self, block_inst: LVDS, block_prop: LVDSRxProp) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        block_prop.db_inst = block_inst
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        for prop, val in export_prop.items():
            # Invalid option (Disabled serialization)
            if prop == "RX_DESER" and val == "1":
                continue
            _set_command(prop, val)

        return command_list


class LVDSTxAdvISFBuilder(LVDSTxISFBuilder):
    """
    ISF commands builder for LVDS Tx Advance
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.lvds_reg
        self.db_block_type = PeriDesign.BlockType.adv_lvds
        self.api_block_type = APIObject.ObjectType.lvds_tx
        self.presenter = LVDSTxAdvPresenter

    def build_block_api(self, design):
        lvds_api = IntLVDSAdvAPI()
        lvds_api.set_design(design)
        return lvds_api

    def build_block_prop(self, design):
        return LVDSTxAdvProp.build_lvds_prop(design_db=design)


class LVDSRxAdvISFBuilder(LVDSRxISFBuilder):
    """
    ISF commands builder for LVDS Rx Advance
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.lvds_reg
        self.db_block_type = PeriDesign.BlockType.adv_lvds
        self.api_block_type = APIObject.ObjectType.lvds_rx
        self.presenter = LVDSRxAdvPresenter

    def build_block_api(self, design):
        lvds_api = IntLVDSAdvAPI()
        lvds_api.set_design(design)
        return lvds_api


class LVDSBidirAdvISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for LVDS Bidir Advance
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.lvds_reg
        self.db_block_type = PeriDesign.BlockType.adv_lvds
        self.api_block_type = APIObject.ObjectType.lvds_bidir

    def build_block_api(self, design):
        lvds_api = IntLVDSAdvAPI()
        lvds_api.set_design(design)
        return lvds_api

    def build_block_prop(self, design):
        return LVDSBidirAdvProp.build_lvds_prop(design_db=design)

    def build_block_list(self, block_inst_name: Optional[List[str]]):
        super().build_block_list(block_inst_name)

        def filter_bidir(block_inst):
            if block_inst.ops_type == LVDSAdvance.OpsType.op_bidir:
                return True
            return False

        self.block_list = list(filter(filter_bidir, self.block_list))

    def build_create_command(self) -> List[str]:

        assert self.block_inst_name is not None
        assert self.api_block_type is not None

        self.build_block_list(self.block_inst_name)

        command_list = []
        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} build commands")

        if self.api_block_type != "":
            for block in self.block_list:
                if block is not None:

                    rx_info = block.rx_info
                    rx_extra_param = LVDSRxAdvISFBuilder.gen_extra_create_parameter(rx_info, LVDSRxAdvPresenter) # type: ignore

                    tx_info = block.tx_info
                    tx_extra_param = LVDSTxAdvISFBuilder.gen_extra_create_parameter(tx_info, LVDSTxAdvPresenter) # type: ignore

                    extra_param = f"{tx_extra_param},{rx_extra_param}"
                    if extra_param == "":
                        parameter = f"\"{block.name}\",\"{block_type}\""
                    else:
                        parameter = f"\"{block.name}\",\"{block_type}\",{extra_param}"

                    sub_command = f"{self.cmd2str_map[self.CommandType.create_inst]}({parameter})"
                    command = f"{self.design_api_handle}.{sub_command}"
                    command_list.append(command)

        return command_list


class HyperRAMISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for HyperRAM
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.mipi_dphy_reg
        self.db_block_type = PeriDesign.BlockType.hyper_ram
        self.api_block_type = APIObject.ObjectType.hyper_ram

    def build_block_api(self, design):
        hram_api = IntHyperRAMAPI()
        hram_api.set_design(design)
        return hram_api

    def build_block_prop(self, design):
        return HyperRAMProp.build_prop(design_db=design)

class SPIFlashISFBuilder(BlockISFBuilder):
    """
    ISF commands builder for SPIFlash
    """

    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.mipi_dphy_reg
        self.db_block_type = PeriDesign.BlockType.spi_flash
        self.api_block_type = APIObject.ObjectType.spi_flash

    def build_block_api(self, design):
        sflash_api = IntSPIFlashAPI()
        sflash_api.set_design(design)
        return sflash_api

    def build_block_prop(self, design):
        return SPIFlashProp.build_prop(design_db=design)


class DDRISFBuilder(BlockISFBuilder):
    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.ddr_reg
        self.db_block_type = PeriDesign.BlockType.ddr
        self.api_block_type = APIObject.ObjectType.ddr

    def build_block_api(self, design):
        return IntDDRAPI.build_ddr_api(design)

    def build_block_prop(self, design):
        return DDRProp.build_ddr_prop(design_db=design)

    def _build_block_property_command(self, block_inst, block_prop) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """

        assert self.api_block_type is not None

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        set_front = ("MEMORY_TYPE","TARGET0_EN", "TARGET1_EN",'GDELAY_OVERRIDE_EN', 'CONTROL_TYPE')
        mem_type = export_prop.get("MEMORY_TYPE", None)

        exclude_mem_set = self.get_excluded_mem_prop(mem_type)

        adv_density_enable = export_prop.get("ADV_DENSITY_EN", None)
        axi0_enable = export_prop.get("TARGET0_EN", None)
        axi1_enable = export_prop.get("TARGET1_EN", None)
        gate_delay_enable = export_prop.get("GDELAY_OVERRIDE_EN", None)
        ctrl_type = export_prop.get('CONTROL_TYPE', None)
        exclude_ctrl_set = self.get_excluded_ctrl_prop(ctrl_type)

        for prop_name in set_front:
            val = export_prop.get(prop_name)
            _set_command(prop_name, val)

        for prop, val in export_prop.items():
            if prop in set_front:
                continue
            elif axi0_enable == '0' and 'AXI0' in prop:
                continue
            elif axi1_enable == '0' and 'AXI1' in prop:
                continue
            elif gate_delay_enable == '0' and prop in ('GCOARSE_DELAY','GFINE_DELAY'):
                continue
            elif prop in exclude_mem_set:
                continue
            elif prop in exclude_ctrl_set:
                continue
            # PT-2480 PRESET should not be exported,
            # since user can update other properties after using preset
            elif prop == "PRESET":
                continue
            elif adv_density_enable == '0' and \
                    prop in ('ROW_WIDTH', "COLUMN_WIDTH"):
                continue

            _set_command(prop, val)

        return command_list

    @staticmethod
    def get_excluded_mem_prop(mem_type):
        ddr3_spec = {'BL', 'PRECHARGE_PD', 'ASR', 'CL',
                     'CWL', 'RTT_WR', 'RTT_NOM', 'MEM_OTERM', 'READ_BURST_TYPE',
                     'SRT'
        }

        lpddr2_spec = {
            'BL', 'READ_BURST_TYPE', 'OUT_DRIVE_STR', 'RL/WL'
        }

        lpddr3_spec = {'OUT_DRIVE_STR', 'RL/WL', 'DQ_ODT'}

        if mem_type == 'DDR3':
            exclude_set = (lpddr2_spec | lpddr3_spec) - ddr3_spec
        elif mem_type == 'LPDDR2':
            exclude_set = (ddr3_spec | lpddr3_spec) - lpddr2_spec
        elif mem_type == 'LPDDR3':
            exclude_set = (ddr3_spec | lpddr2_spec) - lpddr3_spec
        else:
            exclude_set = ddr3_spec | lpddr2_spec | lpddr3_spec

        return exclude_set

    @staticmethod
    def get_excluded_ctrl_prop(ctrl_type):
        excluded_list = {}
        user_reset = {
            'RESET_PIN', 'SEQ_RESET_PIN', 'SEQ_START_PIN'
        }
        calibration = {
            'SCL_IN_PIN', 'SDA_IN_PIN', 'SDA_OUT_PIN'
        }
        reset_calibration = {
            'RESET_PIN', 'SCL_IN_PIN', 'SDA_IN_PIN', 'SDA_OUT_PIN'
        }

        if ctrl_type == 'User Reset':
            excluded_list = (calibration | reset_calibration) - user_reset
        elif ctrl_type == 'Calibration':
            excluded_list = (user_reset | reset_calibration) - calibration
        elif ctrl_type == 'Reset and Calibration':
            excluded_list = (user_reset | calibration) - reset_calibration
        else:
            excluded_list = user_reset | calibration | reset_calibration

        return excluded_list


class DDRAdvISFBuilder(DDRISFBuilder):
    def __init__(self, design: PeriDesign, design_api_handle: str, block_inst_name: list = None,
                 dump_all_pin: bool = False, is_verbose: bool = False):
        super().__init__(design, design_api_handle, block_inst_name, dump_all_pin, is_verbose)
        self.block_reg = self.design.ddr_reg
        self.db_block_type = PeriDesign.BlockType.adv_ddr

    def build_block_api(self, design):
        return IntDDRAdvAPI.build_ddr_api(design)

    def build_block_prop(self, design):
        return DDRAdvProp.build_ddr_prop(design_db=design)

    def _build_block_property_command(self, block_inst: DDRAdvance, block_prop: DDRAdvProp) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        # Set Font
        set_font_prop_list = block_prop.get_dependency_param_list()
        set_front = self.convert_prop_id_list2name_list(block_prop, set_font_prop_list)

        # Dependency Param
        # Soft Config/ Debug
        # is_reg_en = block_inst.ctrl_reg_inf.is_reg_ena
        # ctrl_reg_param_list = block_prop.get_all_ctrl_reg_inf_param_str() + \
        #     [f"{pin.lower()}_pn" for pin in block_prop.get_all_ctrl_reg_inf_pin_type()]
        # ctrl_reg_prop_list = self.convert_prop_id_list2name_list(block_prop, ctrl_reg_param_list)

        is_axi0_enable = block_inst.axi_target0.is_enabled
        is_axi1_enable = block_inst.axi_target1.is_enabled

        physical_rank = block_inst.get_physical_rank()
        excl_phy_rank_prop_id_list = self.get_excluded_phy_rank_prop(physical_rank)
        exclude_phy_rank_name_list = self.convert_prop_id_list2name_list(block_prop, excl_phy_rank_prop_id_list)

        # internal_prop_id_list = block_prop.get_read_write_param_str()
        # internal_prop_name_list = self.convert_prop_id_list2name_list(block_prop, internal_prop_id_list)

        # Set command
        for prop_name in set_front:
            val = export_prop.get(prop_name)
            _set_command(prop_name, val)

        for prop, val in export_prop.items():
            if prop in set_front:
                continue

            elif not is_axi0_enable and 'AXI0' in prop:
                continue

            elif not is_axi1_enable and 'AXI1' in prop:
                continue

            elif prop in exclude_phy_rank_name_list: # + internal_prop_name_list:
                continue

            # elif not is_reg_en and prop in ctrl_reg_prop_list:
            #     continue

            _set_command(prop, val)

        return command_list

    def convert_prop_id_list2name_list(self, block_prop: DDRAdvProp, prop_id_list: List[str]):
        name_list = []

        for prop_id in prop_id_list:
            name = block_prop.prop_id_to_name(block_prop.PropId[prop_id])
            name_list.append(name)

        return name_list

    @staticmethod
    def get_excluded_phy_rank_prop(phy_rank: int) -> List[str]:
        all_phy_prop = set(DDRAdvProp.get_mem_phy_rank_param_list(None))
        exclude_prop = all_phy_prop - set(DDRAdvProp.get_mem_phy_rank_param_list(phy_rank))
        return list(exclude_prop)


class DeviceSettingISFBuilder():
    """
    ISF commands builder for Device Setting
    """

    class CommandType(Enum):
        """
        Block command id to be used in ISF generation
        """
        set_prop = auto()

    cmd2str_map = {
        CommandType.set_prop: "set_device_property",
    }  #: Map ISF command to its API string representation. The API must matched with the one in DesignAPI.

    def __init__(self, design: PeriDesign, design_api_handle: str, dump_all_pin: bool = False,
                 is_verbose: bool = False):

        self.is_edited = False
        self.iblock: Optional[Any] = None
        self.design_api_handle = design_api_handle
        self.design = design
        self.is_dump_all_pin = dump_all_pin

        #self.iobank_reg = None
        self.db_reg = None
        self.db_block_type = None
        self.api_block_type: Optional[APIObject.ObjectType] = None
        self.db_inst: Optional[Any] = None
        self.block_prop: Optional[BaseProp] = None


        self.is_verbose = is_verbose

        from api_service.common.infra import InfraService
        infra_svc = InfraService.get_instance()
        self.msg_svc = infra_svc.get_message_service()

    def build_all_prop(self, block_prop) -> Dict:
        """
        Generate a map of property name and value.

        :return: A property map
        """
        self.block_prop = block_prop
        if self.block_prop is not None:
            return self.block_prop.get_all_property()

        return {}

    def _build_block_property_command(self, block_inst, block_prop: BaseProp, set_last = None) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :param set_last: Set the property must be the last one
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        for prop, val in export_prop.items():
            if set_last is not None and prop in set_last:
                continue
            _set_command(prop, val)

        if set_last is not None:
            if isinstance(set_last, List):
                for prop_name in set_last:
                    val = export_prop.get(prop_name)
                    _set_command(prop_name, val)
            else:
                val = export_prop.get(set_last)
                _set_command(set_last, val)

        return command_list


class IOBankISFBuilder(DeviceSettingISFBuilder):

    class CommandType(Enum):
        """
        Block command id to be used in ISF generation
        """
        set_prop = "set_device_property"
        set_mode_sel = "set_mode_sel_name"

    def __init__(self, design: PeriDesign, design_api_handle: str, dump_all_pin: bool = False,
                 is_verbose: bool = False):
        super().__init__(design, design_api_handle, dump_all_pin, is_verbose)
        self.iblock = IntIOBankAPI(is_verbose)
        self.api_block_type = APIObject.ObjectType.iobank
        self.iobank_inst_list: List = []

    def build_all_prop(self):
        """
        Generate a map of property name and value.

        :return: A property map
        """
        return super().build_all_prop(IOBankProp.build_iobank_prop(design_db=self.design))

    def build_iobank_list(self):
        """
        Build a list of design db iobank instances.
        If instance name not specified, it will retrieve all.

        :param block_inst_name: A list of instance name
        """
        if self.design is not None and self.design.device_setting is not None:
            self.db_reg = self.design.device_setting.iobank_reg
            # Export all
            self.iobank_inst_list = self.db_reg.get_all_inst()
            def get_name(db_inst):
                return db_inst.name

            # Sorted by instance name
            self.iobank_inst_list.sort(key=get_name)

        else:
            self.db_reg = None
            self.iobank_inst_list = []

    def build_set_property_command(self) -> List[str]:
        """
        Build a series of I/O Bank set property command

        :return: A list of commands
        """

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} configure commands")

        self.build_all_prop()
        self.build_iobank_list()

        assert self.block_prop is not None

        for iobank in self.iobank_inst_list:
            self.block_prop.set_block(iobank)
            cmd_list = self._build_block_property_command(iobank, self.block_prop)
            if len(cmd_list) > 0:
                command_list += cmd_list

        return command_list

    def _build_block_property_command(self, block_inst, block_prop: BaseProp) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :param set_last: Set the property must be the last one
        :return: A list of DesignAPI command
        """
        assert self.api_block_type is not None

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.CommandType.set_prop.value}({parameter})"
            command_list.append(command)

        def _set_mode_sel_command():
            for bonded_bank, pin_name in block_inst.bank2mode_sel_name.items():
                parameter = f"\"{block_inst.name}\",\"{pin_name}\",\"{bonded_bank}\""
                command = f"{self.design_api_handle}.{self.CommandType.set_mode_sel.value}({parameter})"
                command_list.append(command)

        for prop, val in export_prop.items():
            if prop == "MODE_SEL_PIN":
                _set_mode_sel_command()
                continue
            _set_command(prop, val)


        return command_list


class SEUISFBuilder(DeviceSettingISFBuilder):
    """
    ISF commands builder for SEU Detect
    """
    def __init__(self, design: PeriDesign, design_api_handle: str, dump_all_pin: bool = False,
                 is_verbose: bool = False):
        super().__init__(design, design_api_handle, dump_all_pin, is_verbose)
        self.iblock = IntSEUAPI(is_verbose)
        self.api_block_type = APIObject.ObjectType.seu

    def build_all_prop(self):
        """
        Generate a map of property name and value.

        :return: A property map
        """
        return super().build_all_prop(SEUProp.build_seu_prop(design_db=self.design))

    def build_seu_list(self):
        """
        Build a list of design db iobank instances.
        If instance name not specified, it will retrieve all.

        :param block_inst_name: A list of instance name
        """
        if self.design is not None and self.design.device_setting is not None:
            self.db_reg = self.design.device_setting.seu_reg
            self.db_inst = self.db_reg.get_seu()
        else:
            self.db_reg = None

    def build_set_property_command(self) -> List[str]:
        """
        Build a series of SEU Detection set property command

        :return: A list of commands
        """

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} configure commands")

        self.build_all_prop()
        self.build_seu_list()

        assert self.block_prop is not None

        self.block_prop.set_block(self.db_inst)
        cmd_list = self._build_block_property_command(self.db_inst, self.block_prop, 'ENA_DETECT')
        if len(cmd_list) > 0:
            command_list += cmd_list

        return command_list

    def _build_block_property_command(self, block_inst, block_prop, set_last = None) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :param set_last: Set the property must be the last one
        :return: A list of DesignAPI command
        """

        assert self.api_block_type is not None

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        is_enable = export_prop.get("ENA_DETECT", None)

        if is_enable == "1":
            mode = export_prop.get("MODE", None)

            for prop, val in export_prop.items():

                if mode == "AUTO" and prop == "START_PIN":
                    continue
                elif mode == "MANUAL" and prop == "WAIT_INTERVAL":
                    continue

                if set_last is not None and prop in set_last:
                    continue
                _set_command(prop, val)

        if set_last is not None:
            if isinstance(set_last, List):
                for prop_name in set_last:
                    val = export_prop.get(prop_name)
                    _set_command(prop_name, val)
            else:
                val = export_prop.get(set_last)
                _set_command(set_last, val)

        return command_list


class CtrlISFBuilder(DeviceSettingISFBuilder):
    """
    ISF commands builder for Remote Update
    """
    def __init__(self, design: PeriDesign, design_api_handle: str, dump_all_pin: bool = False,
                 is_verbose: bool = False):
        super().__init__(design, design_api_handle, dump_all_pin, is_verbose)
        self.iblock = IntCtrlAPI(is_verbose)
        self.api_block_type = APIObject.ObjectType.rcu

    def build_all_prop(self):
        """
        Generate a map of property name and value.

        :return: A property map
        """
        super().build_all_prop(CtrlProp.build_ctrl_prop(design_db=self.design))

    def build_ctrl_list(self):
        """
        Build a list of design db iobank instances.
        If instance name not specified, it will retrieve all.

        :param block_inst_name: A list of instance name
        """
        if self.design is not None and self.design.device_setting is not None:
            self.db_reg = self.design.device_setting.ctrl_reg
            self.db_inst = self.db_reg.get_ctrl()
        else:
            self.db_reg = None

    def build_set_property_command(self) -> List[str]:
        """
        Build a series of Remote Update set property command

        :return: A list of commands
        """

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} configure commands")

        self.build_all_prop()
        self.build_ctrl_list()

        assert self.block_prop is not None

        self.block_prop.set_block(self.db_inst)
        cmd_list = self._build_block_property_command(self.db_inst, self.block_prop, 'RECONFIG_EN')
        if len(cmd_list) > 0:
            command_list += cmd_list

        return command_list

    def _build_block_property_command(self, block_inst, block_prop, set_last = None, is_enable = True) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :param set_last: Set the property must be the last one
        :return: A list of DesignAPI command
        """

        assert self.api_block_type is not None

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _set_command(prop, val):
            parameter = f"\"{block_inst.name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        for prop, val in export_prop.items():
            _set_command(prop, val)

        return command_list

class ExtFlashISFBuilder(DeviceSettingISFBuilder):
    """
    ISF commands builder for Remote Update
    """
    def __init__(self, design: PeriDesign, design_api_handle: str, dump_all_pin: bool = False,
                 is_verbose: bool = False):
        super().__init__(design, design_api_handle, dump_all_pin, is_verbose)
        self.iblock = IntExtFlashAPI(is_verbose)
        self.api_block_type = APIObject.ObjectType.ext_flash

    def build_all_prop(self) -> Dict:
        """
        Generate a map of property name and value.

        :return: A property map
        """
        super().build_all_prop(ExtFlashProp.build_ext_flash_prop(design_db=self.design))

    def build_ext_flash_list(self):
        """
        Build a list of design db iobank instances.
        If instance name not specified, it will retrieve all.

        :param block_inst_name: A list of instance name
        """
        if self.design is not None and self.design.device_setting is not None:
            self.db_reg = self.design.device_setting.ext_flash_reg
            self.db_inst = self.db_reg.get_ext_flash()
        else:
            self.db_reg = None

    def build_set_property_command(self) -> List[str]:
        """
        Build a series of Remote Update set property command

        :return: A list of commands
        """

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} configure commands")

        self.build_all_prop()
        self.build_ext_flash_list()
        self.block_prop.set_block(self.db_inst)
        cmd_list = self._build_block_property_command(self.db_inst, self.block_prop)
        if len(cmd_list) > 0:
            command_list += cmd_list

        return command_list

class ClockMuxISFBuilder(DeviceSettingISFBuilder):
    """
    ISF commands builder for Remote Update
    """
    def __init__(self, design: PeriDesign, design_api_handle: str, dump_all_pin: bool = False,
                 is_verbose: bool = False):
        super().__init__(design, design_api_handle, dump_all_pin, is_verbose)
        self.iblock = IntClockMuxAPI(is_verbose)
        self.api_block_type = APIObject.ObjectType.clkmux
        self.block_inst_list = []

    def build_all_prop(self) -> Dict:
        """
        Generate a map of property name and value.

        :return: A property map
        """
        super().build_all_prop(ClockMuxProp.build_clockmux_prop(design_db=self.design))

    def is_clockmux_api_supported(self):
        if self.design is not None and self.design.is_tesseract_design() and\
            self.design.device_setting is not None and \
            self.design.device_setting.clkmux_reg is not None:
            return True

        return False

    def build_clockmux_list(self, block_inst_name: List[str]):
        """
        Build a list of design db clockmux instances.
        If instance name not specified, it will retrieve all.

        :param block_inst_name: A list of instance name
        """
        if self.design is not None and self.design.device_setting is not None and\
                self.is_clockmux_api_supported():
            self.db_reg = self.design.device_setting.clkmux_reg
            if block_inst_name is None or len(block_inst_name) == 0:
                # Export all
                self.block_inst_list = self.db_reg.get_all_inst()
            else:
                # Export selected instances
                self.block_inst_list = []
                for name in block_inst_name:
                    inst = self.db_reg.get_clock_mux_by_name(name)
                    if inst is not None:
                        self.block_inst_list.append(inst)

            def get_name(db_inst):
                return db_inst.name

            # Sorted by instance name
            self.block_inst_list.sort(key=get_name)

        else:
            self.db_reg = None
            self.block_inst_list = []

    def build_set_property_command(self) -> List[str]:
        """
        Build a series of Remote Update set property command

        :return: A list of commands
        """

        command_list = []

        if self.is_verbose:
            self.msg_svc.write(MsgLevel.info, f"Generate {self.api_block_type} configure commands")

        self.build_all_prop()
        self.build_clockmux_list([])

        for clkmux in self.block_inst_list:
            if clkmux is not None:
                self.block_prop.set_block(clkmux)

                cmd_list = self._build_block_property_command(clkmux, self.block_prop)
                if len(cmd_list) > 0:
                    command_list += cmd_list

        return command_list

    def _build_block_property_command(self, block_inst, block_prop, set_last = None) -> List[str]:
        """
        Build block property setting command. Must match with function signature in DesignAPI.

        :param block_inst: Target block
        :param block_prop: Block property object
        :param set_last: Set the property must be the last one
        :return: A list of DesignAPI command
        """

        command_list = []
        prop_inspector = block_prop.build_inspector()

        if self.is_edited:
            export_prop = prop_inspector.get_edited_prop(block_inst, register_all_pin=self.is_dump_all_pin)
        else:
            export_prop = prop_inspector.get_all_config_prop(block_inst)

        block_type = APIObject.otype2str_map.get(self.api_block_type, "")

        def _get_ins_name(block_inst):
            # Get the display name for API usage
            ins_name = block_inst.name
            if block_inst.name in ClockMuxRegistryComplex.clk_def_to_user_name:
                # V4 inherits from Complex and the mapping doesn't change
                ins_name = ClockMuxRegistryComplex.clk_def_to_user_name[block_inst.name]
            elif block_inst.name in ClockMuxRegistryAdvance.clk_def_to_user_name:
                ins_name = ClockMuxRegistryAdvance.clk_def_to_user_name[block_inst.name]

            return ins_name

        def _set_command(ins_name, prop, val):
            parameter = f"\"{ins_name}\",\"{prop}\",\"{val}\",\"{block_type}\""
            command = f"{self.design_api_handle}.{self.cmd2str_map[self.CommandType.set_prop]}({parameter})"
            command_list.append(command)

        api_ins_name = _get_ins_name(block_inst)

        is_dmux0_enable = export_prop.get("DYN_MUX0_EN", False)
        is_dmux7_enable = export_prop.get("DYN_MUX7_EN", False)

        for prop, val in export_prop.items():
            if prop.startswith("DYN_MUX0") and prop != "DYN_MUX0_EN":
                if is_dmux0_enable == "1":
                    _set_command(api_ins_name, prop, val)
                else:
                    continue

            elif prop.startswith("DYN_MUX7") and prop != "DYN_MUX7_EN":
                if is_dmux7_enable == "1":
                    _set_command(api_ins_name, prop, val)
                else:
                    continue

            else:
                _set_command(api_ins_name, prop, val)

        if set_last is not None:
            if isinstance(set_last, List):
                for prop_name in set_last:
                    val = export_prop.get(prop_name)
                    _set_command(api_ins_name, prop_name, val)
            else:
                val = export_prop.get(set_last)
                _set_command(api_ins_name, set_last, val)

        return command_list

if __name__ == "__main__":
    pass
