"""
Copyright (C) 2017-2019 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Oct 7, 2019

@author: yasmin
"""

from __future__ import annotations
import os
from pathlib import Path
from pprint import pprint
import sys
from typing import TYPE_CHECKING, Dict, List, Optional, Tuple, Union, Any

from api_service.property.gen_prop import AdvProp
from api_service.property.gpio_prop import GPIOProp
from api_service.property.jtag_prop import JTAGProp
from api_service.property.osc_prop import OSCProp
from common_device.gpio.gpio_design import GPIO
from common_device.pll.pll_design import PLL
from common_device.osc.osc_design import OSC
from tx60_device.hposc.hposc_design import HPOSC

from design.db import PeriDesign
from design.db_item import PeriDesignItem, PeriDesignItemNetlistConfig, PeriDesignRegistry

import util.gen_util as gen_util
from util.excp import PTException, MsgLevel

from util.singleton_logger import LoggerManager

import api_service.excp.design_excp as pte

from api_service.internal.int_design import IntDesignAPI
from api_service.internal.int_gpio import IntGPIOAPI, IntGPIOCompAPI

from api_service.common.infra import InfraService, PLLProp
from api_service.common.object_db import APIObject

from api_service.device import DeviceAPI

from api_service.exim.isf_importer import DesignISFImporter
from api_service.exim.isf_exporter import DesignISFExporter

from api_service.exim.isf_command import ISFAbstractCommand, ISFAbstractCreateGPIOCommand, ISFAssignBlockResourceCommand, ISFFinishPartialDesignCommand, ISFLockBlockPropertyCommand, ISFSetBlockPropertyCommand, ISFSetupPartialDesignCommand, get_gpio_type, is_creation_command
from api_service.resource import DesignAutoResourcePlanner
from api_service.unified_netlist.isf_converter import PeriNetlistISFBuilder
from api_service.unified_netlist.peri_netlist_builder import PeriNetlistBuilder, PeriNetlistDBLoader
from api_service.unified_netlist.peri_timing_netlist_builder import PeriTimingNetlistBuilder
from api_service.unified_netlist.sim_netlist_writer import PeripherySimVerilogWriter

if TYPE_CHECKING:
    from design.db import DesignIssue
    from netlistdb import Database

# It can be str for the block name, or the object id handle
BlockInstanceType = Union[str, int]

@gen_util.freeze_it
class DesignAPI:
    """
    Includes the majority of API functions relating to the Interface Designer.
    """

    def __init__(self, is_verbose: bool=False):
        """
        Design API initializer.

        Turn on the verbose message to see design check issues.

        :param is_verbose: True, verbose message will be generated, else standard message
        """
        self.__is_verbose = is_verbose
        self.__logger_manager = LoggerManager()
        self.__logger_manager.set_console_logging(False)
        self.__idesign = IntDesignAPI(is_verbose=self.__is_verbose)
        self.__igpio: Optional[IntGPIOAPI] = None

        self.__infra_svc = InfraService.get_instance()
        assert self.__infra_svc is not None

        self.__msg_svc = self.__infra_svc.get_message_service()
        self.__obj_reg = self.__infra_svc.get_obj_reg()
        self.__last_design_issue_list: Optional[List] = None

        self.__netlist_builder = PeriNetlistBuilder()

    def _setup_design(self, design_db: PeriDesign):
        self.__obj_reg.reset()
        self.__idesign.set_design(design_db)
        self.__igpio = IntGPIOAPI.build_gpio_api(design_db, self.__is_verbose)

    def get_device_api(self) -> DeviceAPI:
        """
        Get DeviceAPI for current design.

        :return: A device object
        """
        return DeviceAPI(self.__is_verbose)

    def load(self, design_file: str):
        """
        Load the interface design from the specified file

        :param design_file: The design filename including the path, enclose in double quotes.
        :raises PTDsgLoadException: Exception when loading a design from a file
        """
        self.__obj_reg.reset()
        self.__idesign.load(design_file)
        self.__igpio = IntGPIOAPI.build_gpio_api(self.__idesign.get_design(), self.__is_verbose)

    def create(self, design_name: str, device_name: str, path: str=".", auto_save: bool=False, overwrite: bool=False):
        """
        Create a new interface design. After you create the design, call the ``save()`` function to
        save the file to disk.

        :param design_name: Efinity project name, enclose in double quotes.

        :param device_name: Name and package for the target FPGA, enclose in double quotes. For \
        example, T20F256. This must be the same name you specified for the project in the Efinity \
        Project Editor

        :param path: Location to store the design file, enclose in double quotes.

        :param auto_save: Boolean ``True`` or ``False`` (default). If set to ``True``, the design writes the file at \
        the specified location

        :param overwrite: Boolean ``True`` or ``False`` (default). If ``auto_save`` is ``True``, overwrites the file if \
        it exists at the specified location.
        """
        self.__obj_reg.reset()
        self.__idesign.reset()
        self.__idesign.create(design_name, device_name, path, auto_save, overwrite)
        self.__igpio = IntGPIOAPI.build_gpio_api(self.__idesign.get_design(), self.__is_verbose)

    def save(self):
        """
        Save the interface to the file specified with the ``create()`` command.

        :raises PTDsgSaveException: Exception when saving a design
        """
        self.__idesign.save()

    def save_as(self, design_file: str, overwrite: bool=False):
        """
        Save the interface design to a new file

        :param design_file: The design filename including the path, enclose in double quotes.
        :param overwrite: Boolean ``True`` or ``False`` (default). If ``auto_save`` is ``True``, overwrites the file if \
        it exists at the specified location.
        :raises PTDsgSaveException: Exception when saving a design
        """
        self.__idesign.save_as(design_file, overwrite)

    def get_block_type(self) -> List[str]:
        """
        Gets a list of the available block types. See See Block Types and Device Settings on page 38.

        :return: A list of block type name
        """
        return self.__idesign.get_block_type()

    def create_block(self, name: str, block_type: str, **kwargs) -> Optional[int]:
        r"""
        Create a new block instance of the specified type.

        Do not use ``create_block()`` for GPIO, instead use one of the ``create_<type>_gpio()``
        blocks.

        :param name: Name of the block, enclose in double quotes.
        :param block_type: The block type, enclose in double quotes. Refer to Block Types and Device \
                            Settings on page 38 for the supported blocks
        :param kwargs: \**kwargs are parameters that are used for specific block types

        :Keyword Arguments:
            * *mode* --
                Defines a clock lane (``CLOCK_LANE``) or data lane (``DATA_LANE``). If you do not specify
                the parameter, the software defaults to ``DATA_LANE``.

            * *conn_type* --
                For the MIPI Lane RX, indicate whether you want to connect to the global
                network (``GCLK``) or regional network (``RCLK``). If you do not specify the parameter, the
                software defaults to ``GCLK``.

            * *tx_mode* --
                For LVDS_TX blocks and LVDS_BIDIR blocks, specify whether the block is data
                (DATA) or clock out (CLKOUT). If you do not specify the parameter, the software defaults
                to DATA.

            * *rx_conn_type* --
                For LVDS_RX blocks and LVDS_BIDIR blocks, specify the connection type
                for the LVDS RX. ``NORMAL`` (default), ``GCLK``, ``RCLK``, ``PLL_CLKIN``, ``PLL_EXTFB``.

        :return: The object ID of the new block.

        :raises PTBlkCreateException: Exception when creating a block
        """

        try:
            if self.__idesign.is_block_supported(block_type) is False:
                msg = f"Fail to create block, block type {block_type} is not supported"
                raise pte.PTBlkCreateException(msg, MsgLevel.error)

            blk_obj = self.__idesign.create_block(name, block_type, **kwargs)
            blk_obj_type_id = APIObject.name2otype(block_type)

            if blk_obj_type_id is None:
                msg = f"Fail to create block, unknown block type {block_type}"
                raise pte.PTBlkCreateException(msg, MsgLevel.error)

            # Register in object registry
            if blk_obj is not None:
                handle_id = self.__obj_reg.register_block_instance(blk_obj, blk_obj_type_id)
                return handle_id

        except PTException as excp:
            # Filter out exactly the same message
            if excp.msg != "Fail to create block" and excp.msg != "":
                msg = "Fail to create block: {}".format(excp.msg)
            else:
                msg = "Fail to create block"
            raise pte.PTBlkCreateException(msg, MsgLevel.error, excp)

        return None

    def delete_block(self, inst: BlockInstanceType, block_type: str):
        """
        Delete the specified block instance. if you use a name for inst, also specify the block
        type. For GPIO, use ``delete_gpio()``.

        :param inst: Block object ID or name. If you are deleting the block by name, enclose the name \
        in double quotes

        :param block_type: The block type; enclose in double quotes.

        :raises PTBlkDeleteException: Exception when deleting a block
        """

        if self.__idesign.is_block_supported(block_type) is False:
            msg = f"Fail to delete block, block type {block_type} is not supported"
            raise pte.PTBlkDeleteException(msg, MsgLevel.error)

        if isinstance(inst, str):
            # Used block is in registry
            object_id = self.__obj_reg.get_object_id_by_inst_name(inst)

            # In case where the block hasn't been used
            if object_id is None:
                object_id = self.get_block(inst, block_type)
        else:
            object_id = inst

        if object_id is None:
            msg = f"Fail to delete block. Unknown instance {inst}"
            raise pte.PTBlkDeleteException(msg, MsgLevel.error)

        blk_obj = self.__obj_reg.get_object(object_id)
        if blk_obj is not None:

            # Remove from design db
            db_inst = blk_obj.get_instance()
            if db_inst is not None:
                blk_obj_type_name = blk_obj.get_object_type_name()
                if blk_obj_type_name is not None:
                    self.__idesign.delete_block(db_inst.name, blk_obj_type_name)
                else:
                    msg = f"Fail to delete block, unknown block type {blk_obj.get_object_type()}"
                    raise pte.PTBlkDeleteException(msg, MsgLevel.error)

            # Remove from object registry
            self.__obj_reg.remove_object(blk_obj.get_id())

    def delete_gpio(self, inst: BlockInstanceType):
        """
        Delete a GPIO block or GPIO bus block

        :param inst: GPIO or bus object ID or name. If you are deleting the GPIO by name, enclose the \
        name in double quotes. The API searches for GPIO first.

        :raises PTBlkDeleteException: Exception when deleting a block
        """
        if isinstance(inst, str):
            object_id = self.__obj_reg.get_object_id_by_inst_name(inst)

            # In case where the GPIO hasn't been used
            if object_id is None:
                object_id = self.get_gpio(inst)

                # Could be a bus
                if object_id is None:
                    object_id = self.get_bus(inst)

        else:
            object_id = inst

        if object_id is None:
            msg = f"Fail to delete gpio. Unknown instance {inst}"
            raise pte.PTBlkDeleteException(msg, MsgLevel.error)

        gpio_obj = self.__obj_reg.get_object(object_id)
        if gpio_obj is not None:
            # Remove from design db
            db_inst = gpio_obj.get_instance()
            if db_inst is not None and self.__igpio is not None:
                gpio_obj_type_id = gpio_obj.get_object_type()
                if gpio_obj_type_id == APIObject.ObjectType.gpio:
                    self.__igpio.delete_gpio(db_inst.name)
                elif gpio_obj_type_id == APIObject.ObjectType.gpio_bus:
                    self.__igpio.delete_bus_gpio(db_inst.name)

            # Remove from object registry
            self.__obj_reg.remove_object(gpio_obj.get_id())

    def get_all_block_name(self, block_type: str) -> List[str]:
        """
        Gets the names of all block instances for the specified block type.

        :param block_type: The block type; enclose in double quotes. the default is ``GPIO``.
        :return: A list of the block names.
        """
        if self.__idesign.is_block_supported(block_type) is False:
            self.__msg_svc.write(MsgLevel.warning, f"Fail to get block names, block type {block_type} is not supported")
            return []

        return self.__idesign.get_all_block_name(block_type)

    def get_block(self, name: str, block_type: str):
        """
        Get the object ID of the named block. For GPIO, use ``get_gpio()``.

        :param name: Name of the block, enclose in double quotes.
        :param block_type: The block type, enclose in double quotes
        :return: Block object ID
        :raises PTBlkReadException: Exception when querying for a block property
        """

        if self.__idesign.is_block_supported(block_type) is False:
            self.__msg_svc.write(MsgLevel.warning, f"Fail to get block, block type {block_type} is not supported")
            return None

        blk_obj_type_id = APIObject.name2otype(block_type)
        if blk_obj_type_id is None:
            msg = f"Fail to get block, unknown block type {block_type}"
            self.__msg_svc.write(MsgLevel.warning, msg)
            return None

        blk_obj = self.__idesign.get_block(name, block_type)
        if blk_obj is not None:
            handle_id = self.__obj_reg.register_block_instance(blk_obj, blk_obj_type_id)
            return handle_id

        if self.__is_verbose:
            self.__msg_svc.write(MsgLevel.warning,
                                 f"Fail to get block <{name}> with type <{block_type}>, it doesn't exists.")

        return None

    def get_all_property(self, block_type:Optional[str]=None, object_id:Optional[int]=None):
        """
        Displays all of the properties and values for the specified block type. If you also specify an
        ``object_id``, this command only shows the properties for the block type and object. For
        example:

        ``block_type="JTAG"``, displays all possible properties.

        ``block_type="GPIO"``, displays all possible properties for all modes.

        ``block_type="GPIO", obj_id=input_gpio``, displays all properties for input GPIO

        :param block_type: Type of block; enclose in double quotes. The default is None.
        :param object_id: Block object ID. The default is None
        :return: A dict of property names to values.
        :raises PTBlkReadException: Exception when querying for a block property
        """
        prop_map = {}

        try:
            if block_type is None:
                blk_type = ""
            else:
                if self.__idesign.is_block_supported(block_type) is False:
                    msg = f"Fail to get all property, block type {block_type} is not supported"
                    raise pte.PTBlkReadException(msg, MsgLevel.error)
                blk_type = block_type

            block_type_id = APIObject.str2otype_map.get(blk_type, None)

            assert self.__infra_svc is not None
            prop_svc = self.__infra_svc.get_prop_service(block_type_id, object_id)
            if prop_svc is not None:
                prop_map = prop_svc.get_all_property()
        except PTException as excp:
            msg = "Fail to get all property"
            raise pte.PTBlkReadException(msg, MsgLevel.error, excp)

        return prop_map

    def get_property(self, inst: BlockInstanceType, prop_name: Union[str, List[str]], block_type: str="GPIO") -> Dict[str, str]:
        """
        Display the value of the specified property for the instance and block type. You can
        specify one or more property names. See GPIO Property Reference on page 39

        :param inst: The block's object ID or instance name. Enclose instance names in double quotes.

        :param prop_name: Name of the property or a comma-separated list of properties. Enclose \
        names in double quotes.

        :param block_type: Type of block, enclose in double quotes. If ``inst`` is an instance name, \
        indicate the block type. The default is ``GPIO``.

        :return: A dict of property name to value.
        :raises PTBlkReadException: Exception when querying for a block property
        """

        try:
            if self.__idesign.is_block_supported(block_type) is False:
                msg = f"Fail to get property, block type {block_type} is not supported"
                raise pte.PTBlkReadException(msg, MsgLevel.error)

            object_id = self.__get_object_id(inst, block_type)
            if object_id is None:
                msg = f"Fail to get property. Unknown instance {inst}"
                raise pte.PTBlkReadException(msg, MsgLevel.error)

            reg_obj = self.__obj_reg.get_object(object_id)
            if reg_obj is not None:
                reg_obj_type = reg_obj.get_object_type()
                if reg_obj_type == APIObject.ObjectType.gpio_bus:
                    return self.__get_bus_property(object_id, prop_name)
                else:
                    return self.__get_simple_property(object_id, prop_name)

        except PTException as excp:
            msg = "Fail to get property"
            raise pte.PTBlkReadException(msg, MsgLevel.error, excp)

        return {}

    def get_device_property(self, inst: str, prop_name: str, devset_type: str="IOBANK") -> Dict[str, str]:
        """
        Get the value of the property for a specific device setting type

        :param inst: The instance name
        :type inst: str

        :param prop_name: Property name
        :type prop_name: str

        :param devset_type: Device setting type, default to "IOBANK", supported types are "IOBANK" / "SEU" / "RU" / "EXT_FLASH"
        :type devset_type: str

        :raises PTDevSetReadException: Exception when querying for a device setting property

        :return: A value if assigned, otherwise, an empty string.
        :rtype: Dict[str, str]
        """

        try:
            # if self.__idesign.is_block_supported(devset_type) is False:
            #     msg = f"Fail to get device setting property, device setting type {devset_type} is not supported"
            #     raise pte.PTDevSetReadException(msg, MsgLevel.error)

            object_id = self.__get_object_id(inst, devset_type)
            if object_id is None:
                msg = f"Fail to get device setting property. Unknown device setting instance {inst}"
                raise pte.PTDevSetReadException(msg, MsgLevel.error)

            reg_obj = self.__obj_reg.get_object(object_id)
            if reg_obj is not None:
                return self.__get_simple_property(object_id, prop_name)

        except PTException as excp:
            msg = "Fail to get device setting property"
            raise pte.PTDevSetReadException(msg, MsgLevel.error, excp)

        return {}

    def set_property(self, inst: BlockInstanceType, prop_name: Union[str, Dict[str, str]], prop_value: Optional[str]=None, block_type: str="GPIO"):
        """
        Set the value of the specified property for the specified block. If ``inst`` is an instance name,
        indicate the block type. See GPIO Property Reference on page 39

        Do not use ``set_property()`` for resources or pins, instead use ``assign_resource()``
        and ``assign_pkg_pin()``, respectively

        :param inst: The block's object ID or instance name, enclose instance names in double quotes.

        :param prop_name: Name of the property, or property-value dictionary

        :param prop_value: The property value, enclose in double quotes. The default is None.

        :param block_type: Type of block, enclose in double quotes. If inst is an instance name, \
        indicate the block type. The default is GPIO.

        :raises PTBlkEditException: Exception when editing a block property
        """

        try:
            if self.__idesign.is_block_supported(block_type) is False:
                msg = f"Fail to set property, block type {block_type} is not supported"
                raise pte.PTBlkEditException(msg, MsgLevel.error)

            object_id = self.__get_object_id(inst, block_type)
            if object_id is None:
                msg = f"Fail to set property. Unknown instance {inst}"
                raise pte.PTBlkEditException(msg, MsgLevel.error)

            obj_inst = self.__obj_reg.get_block_instance(object_id)
            if obj_inst is not None:
                assert self.__infra_svc is not None
                prop_svc = self.__infra_svc.get_prop_service(object_id=object_id)
                if prop_svc is not None:
                    if isinstance(prop_name, dict):
                        for name, value in prop_name.items():
                            prop_svc.set_property(name, value)
                    else:
                        if prop_value is None:
                            msg = "Fail to set property, property value is None."
                            raise pte.PTBlkEditException(msg, MsgLevel.error)
                        else:
                            prop_svc.set_property(prop_name, prop_value)
            else:
                msg = f"Fail to set property. Fail to find db object for id<{object_id}>"
                raise pte.PTBlkEditException(msg, MsgLevel.error)

        except PTException as excp:
            msg = "Fail to set property"
            raise pte.PTBlkEditException(msg, MsgLevel.error, excp)

    def set_device_property(self, inst: str, prop_name: Union[str, Dict[str, str]], prop_value: Optional[str]=None, devset_type: str="IOBANK"):
        """
        Set the property of the specified device setting.

        To set a single property, pass a property name and its value.

        To set multiple properties, pass a property dictionary that contains property name to value mapping.

        :param inst: The instance name
        :type inst: str

        :param prop_name: Property name or a property name-value dictionary
        :type prop_name: Union[str, Dict[str, str]]

        :param prop_value: Property value to be set in single property setting case, defaults to None
        :type prop_value: str, optional

        :param devset_type: Device setting type, defaults to "IOBANK", supported types are "IOBANK" / "SEU" / "RU" / "EXT_FLASH"
        :type devset_type: str, optional

        :raises PTDevSetEditException: Exception when editing a device setting property
        """

        try:
            if inst.strip() == "":
                msg = "Fail to set device property, instance name is empty"
                raise pte.PTDevSetEditException(msg, MsgLevel.error)

            idevset = self.__idesign.get_idevice_setting()
            if idevset.is_setting_supported(devset_type) is False:
                msg = f"Fail to set device property, setting {devset_type} is not supported"
                raise pte.PTDevSetEditException(msg, MsgLevel.error)

            object_id = self.__get_object_id(inst, devset_type)
            if object_id is None:
                msg = f"Fail to set device property. Unknown instance {inst}"
                raise pte.PTDevSetEditException(msg, MsgLevel.error)

            obj_inst = self.__obj_reg.get_block_instance(object_id)
            if obj_inst is not None:
                assert self.__infra_svc is not None
                prop_svc = self.__infra_svc.get_prop_service(object_id=object_id)
                if prop_svc is not None:
                    if isinstance(prop_name, dict):
                        for name, value in prop_name.items():
                            prop_svc.set_property(name, value)
                    else:
                        if prop_value is None:
                            msg = "Fail to set device property, property value is None."
                            raise pte.PTDevSetEditException(msg, MsgLevel.error)
                        else:
                            prop_svc.set_property(prop_name, prop_value)
            else:
                msg = f"Fail to set device property. Fail to find db object for id<{object_id}>"
                raise pte.PTDevSetEditException(msg, MsgLevel.error)

        except PTException as excp:
            msg = "Fail to set property"
            raise pte.PTDevSetEditException(msg, MsgLevel.error, excp)

    def get_all_iobank_name(self) -> List[str]:
        """
        Get the names of all the I/O banks.

        :return: List of I/O banks
        :raises PTDevSetReadException: Exception when querying for a device setting property
        """
        try:
            iiobank = self.__idesign.get_idevice_setting().get_iobank_svc()

            if iiobank is None:
                msg = "Fail to get IO Bank names"
                raise pte.PTDevSetReadException(msg, MsgLevel.error)

            return iiobank.get_all_iobank_name()

        except PTException as excp:
            msg = "Fail to get IO Bank names"
            raise pte.PTDevSetReadException(msg, MsgLevel.error, excp)

    def set_iobank_voltage(self, bank_name: Union[str, Dict[str, str]], voltage: Optional[str]=None):
        """
        Set the voltage of the specified I/O bank.

        You can set values for multiple banks by defining a map. Refer to the **iobank_setting.py**
        example script.

        :param bank_name: I/O Bank name or a I/O Bank name-voltage dictionary
        :param voltage: The voltage to set, for single bank's voltage setting. \
                        Must set to None for multi bank voltage setting case.
        """
        if isinstance(bank_name, dict):
            assert voltage is None
            for bank, volt in bank_name.items():
                self.set_device_property(bank, "VOLTAGE", volt, "IOBANK")
        else:
            self.set_device_property(bank_name, "VOLTAGE", voltage, "IOBANK")

    def get_iobank_voltage(self, bank_name: Optional[str]=None) -> Optional[Union[str, Dict[str, str]]]:
        """
        Get the voltage set for the specified I/O bank.

        :param bank_name: The I/O bank; enclose in double quotes. The default is ``None``
        :return: If bank name is provided, return the I/O Bank voltage assigned; otherwise \
                 a I/O Bank name-volatage dictionary
        """
        if bank_name is None:
            iobank_map = {}
            iobank_name_list = self.get_all_iobank_name()
            for iobank in iobank_name_list:
                volt_prop = self.get_device_property(iobank, "VOLTAGE", "IOBANK")
                iobank_map[iobank] = volt_prop["VOLTAGE"]

            return iobank_map
        else:
            volt_prop = self.get_device_property(bank_name, "VOLTAGE", "IOBANK")
            if volt_prop is not None:
                return volt_prop["VOLTAGE"]

        return None

    def get_mode_sel_name(self, bank_name: str, bonded_bank: Optional[str]=None):
        """
        Get mode selection pin name(s) by I/O bank name.

        :param bank_name: The I/O bank; enclose in double quotes
        :type bank_name: str
        :param bonded_bank: The I/O bank; enclose in double quotes.\
            The default is ``None``
        :type bonded_bank: Optional[str]

        :return: A value if assigned, otherwise, an empty string.
        :rtype: Dict[str, str]

        :raises PTDevSetReadException: Exception when querying for a device setting property
        """
        try:
            iiobank = self.__idesign.get_idevice_setting().get_iobank_svc()

            if iiobank is None:
                msg = "Fail to get mode selection pin name"
                raise pte.PTDevSetReadException(msg, MsgLevel.error)

            get_name_func = None

            if bonded_bank is None:
                get_name_func = getattr(iiobank, 'get_all_mode_sel_name', None)
                if get_name_func is not None:
                    return get_name_func()
            else:
                get_name_func = getattr(iiobank, 'get_mode_sel_name', None)
                if get_name_func is not None:
                    return {
                        bonded_bank: get_name_func(bank_name, bonded_bank)
                    }

            if get_name_func is None:
                msg = f"Mode selection pin name is not supported"
                raise pte.PTDevSetEditException(msg, MsgLevel.error)

        except PTException as excp:
            msg = "Fail to get mode selection pin name"
            raise pte.PTDevSetReadException(excp.msg if excp.msg != "" else msg, MsgLevel.error, excp)

    def get_bonded_bank(self, bank_name: str) -> List[str]:
        """
        Get all bonded I/O bank name by bank name. Some I/O bank may have 1 or more bonded
        bank name. e.g. BR1_BR2_BR3_BR4 have 2 bonded I/O banks BR3 and BR4.

        :param bank_name: I/O bank name
        :type bank_name: str
        :return: List of bonded I/O bank name
        :rtype: List[str]

        :raises PTDevSetReadException: Exception when querying for a device setting property
        """
        try:
            iiobank = self.__idesign.get_idevice_setting().get_iobank_svc()

            if iiobank is None:
                msg = "Fail to get bonded I/O bank"
                raise pte.PTDevSetReadException(msg, MsgLevel.error)

            get_bond_bank = getattr(iiobank, "get_bonded_bank", None)

            if get_bond_bank is None:
                msg = f"Mode selection pin name is not supported"
                raise pte.PTDevSetEditException(msg, MsgLevel.error)

            return get_bond_bank(bank_name)

        except PTException as excp:
            msg = "Fail to get bonded I/O bank"
            raise pte.PTDevSetReadException(excp.msg if excp.msg != "" else msg, MsgLevel.error, excp)

    def set_mode_sel_name(self, bank_name: str, value: str, bonded_bank: Optional[str]=None):
        """
        Set mode selection pin name(s) by I/O bank name.

        :param bank_name: The I/O bank; enclose in double quotes
        :type bank_name: str
        :param value: Mode select pin name
        :type value: str
        :param bonded_bank: The I/O bank; enclose in double quotes.\
            The default is ``None``
        :type bonded_bank: Optional[str]

        :raises PTDevSetReadException: Exception when querying for a device setting property
        """
        try:
            iiobank = self.__idesign.get_idevice_setting().get_iobank_svc()

            if iiobank is None:
                msg = "Fail to set mode selection pin name"
                raise pte.PTDevSetReadException(msg, MsgLevel.error)

            if bonded_bank is None:
                bonded_bank = bank_name

            set_name = getattr(iiobank, "set_mode_sel_name", None)

            if set_name is None:
                msg = f"Mode selection pin name is not supported"
                raise pte.PTDevSetEditException(msg, MsgLevel.error)

            set_name(bank_name, bonded_bank, value)

        except PTException as excp:
            msg = "Fail to set mode selection pin name"
            raise pte.PTDevSetReadException(excp.msg if excp.msg != "" else msg, MsgLevel.error, excp)

    def assign_resource(self, inst: BlockInstanceType, res_name: str, block_type: str="GPIO"):
        """
        Assign a resource to a GPIO or other block

        :param inst: The block's object ID or instance name
        :param res_name: Resource name, enclose in double quotes.
        :param block_type: Type of block. If inst is an instance name, indicate the block type. The \
        default is ``GPIO``.

        :raises PTBlkEditException: Exception when editing a block property
        :raises PTDsgLoadException: Exception when loading a design from a file
        """

        if self.__idesign.is_block_supported(block_type) is False:
            msg = f"Fail to assign resource, block type {block_type} is not supported"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        object_id = self.__get_object_id(inst, block_type)
        if object_id is None:
            msg = f"Fail to assign resource. Unknown instance {inst}"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        obj_inst = self.__obj_reg.get_block_instance(object_id)
        blk_obj = self.__obj_reg.get_object(object_id)
        is_assign = False

        if obj_inst is not None and blk_obj is not None:
            obj_type = blk_obj.get_object_type()

            if obj_type == APIObject.ObjectType.gpio:
                if self.__igpio is None:
                    msg = "Fail to load Internal GPIO"
                    raise pte.PTDsgLoadException(msg, MsgLevel.error)

                is_assign = self.__igpio.assign_resource(obj_inst, res_name)
            else:
                blk_type = APIObject.otype2name(obj_type)
                self.__idesign.assign_resource(obj_inst, res_name, blk_type)
                is_assign = True

        if not is_assign:
            msg = "Fail to assign resource"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

    def assign_pkg_pin(self, inst: BlockInstanceType, pin_name: str):
        """
        Assign the specified GPIO to the resource associated with the specified package pin.

        :param inst: The block's object ID or instance name.
        :param pin_name: GPIO pin name
        :raises PTBlkEditException: Exception when editing a block property
        """

        object_id = self.__get_object_id(inst, "GPIO")
        if object_id is None:
            msg = f"Fail to assign package pin. Unknown instance {inst}"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        obj_inst = self.__obj_reg.get_block_instance(object_id)
        is_assign = False
        if obj_inst is not None and self.__igpio is not None:
            self.__igpio.assign_pkg_pin(obj_inst, pin_name)
            is_assign = True

        if not is_assign:
            msg = "Fail to assign package pin"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

    def is_resource_used(self, res_name: str, block_type: str="GPIO") -> bool:
        """
        Check if a resource already has an assignment.

        :param res_name: Resource name, enclose in double quotes.
        :param block_type:  Indicates the block type, enclose in double quotes. The default is ``GPIO``.
        :return: Boolean ``True`` if used or ``False`` if not.
        """

        if self.__idesign.is_block_supported(block_type) is False:
            self.__msg_svc.write(MsgLevel.warning, f"Fail to check resource, block type {block_type} is not supported")
            return False

        obj_type_id = APIObject.str2otype_map.get(block_type, None)

        if obj_type_id == APIObject.ObjectType.gpio:
            block_api = self.__igpio
        else:
            block_api = self.__idesign.get_block_apimgr(block_type)

        if block_api is not None:
            return block_api.is_resource_used(res_name)

        return False

    def is_pkg_pin_used(self, pin_name: str) -> bool:
        """
        Check if a package pin already has an assignment.

        :param pin_name: Package pin name, enclose in double quotes
        :return: Boolean ``True`` if used or ``False`` if not

        :raises PTDsgLoadException: Exception when loading a design from a file
        """
        if self.__igpio is None:
            msg = "Fail to load Internal GPIO"
            raise pte.PTDsgLoadException(msg, MsgLevel.error)
        return self.__igpio.is_pkg_pin_used(pin_name)

    def get_resource(self, inst: BlockInstanceType, block_type: str="GPIO") -> str:
        """
        Get the name of the resource assigned to the instance of the specified block type.

        :param inst: The block's object ID or instance name. Enclose instance names in double quotes.
        :param block_type: Indicates the block type, enclose in double quotes. The default is ``GPIO``.
        :return: If assigned, returns a string with the name; otherwise an empty string

        :raises PTBlkEditException: Exception when editing a block property
        """

        if self.__idesign.is_block_supported(block_type) is False:
            self.__msg_svc.write(MsgLevel.warning, f"Fail to get resource, block type {block_type} is not supported")
            return ""

        object_id = self.__get_object_id(inst, block_type)

        if object_id is None:
            msg = f"Fail to assign resource. Unknown instance {inst}"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        prop_map = self.get_property(object_id, "RESOURCE")
        return prop_map.get("RESOURCE", "")

    def get_pkg_pin(self, inst: BlockInstanceType) -> str:
        """
        Get the package pin assigned to the GPIO instance

        :param inst: GPIO object ID or instance name. Enclose instance names in double quotes.
        :return: If assigned, returns the pin name; otherwise an empty string.
        """

        object_id = self.__get_object_id(inst, "GPIO")
        if object_id is None or self.__igpio is None:
            return ""

        blk_inst = self.__obj_reg.get_block_instance(object_id)
        return self.__igpio.get_pkg_pin(blk_inst)

    def get_all_gpio_name(self) -> List[str]:
        """
        Gets the names of all GPIO and buses used.

        :return: A list of GPIO and bus names.
        """
        if self.__igpio is None:
            return []
        return self.__igpio.get_all_gpio_name()

    def get_gpio(self, name: str) -> Optional[int]:
        """
        Get the object ID of a GPIO block by name. To get a bus, use ``get_bus()``.

        :param name: The GPIO name, enclose in double quotes.
        :return: If found, returns the GPIO object ID.

        :raises PTDsgLoadException: Exception when loading a design from a file
        """
        if self.__igpio is None:
            msg = "Fail to load Internal GPIO"
            raise pte.PTDsgLoadException(msg, MsgLevel.error)

        gpio_obj = self.__igpio.get_gpio(name)
        if gpio_obj is not None:
            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def get_bus(self, name: str) -> Optional[int]:
        """
        Get the object ID of a GPIO bus block by name. To get a single GPIO block, use
        ``get_gpio()``.

        :param name: GPIO bus name, enclose in double quotes.
        :return: If found, returns the GPIO bus object ID.

        :raises PTDsgLoadException: Exception when loading a design from a file
        """
        if self.__igpio is None:
            msg = "Fail to load Internal GPIO"
            raise pte.PTDsgLoadException(msg, MsgLevel.error)

        bus_obj = self.__igpio.get_bus(name)
        if bus_obj is not None:
            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(bus_obj, APIObject.ObjectType.gpio_bus)
            return handle_id

        return None

    def get_bus_gpio(self, bus_object_id: int) -> List[int]:
        """
        Get a list of the GPIO blocks in the specified bus.

        :param bus_object_id: Object ID of the GPIO bus.
        :return: A list of GPIO or an empty list.

        :raises PTDsgLoadException: Exception when loading a design from a file
        """
        if self.__igpio is None:
            msg = "Fail to load Internal GPIO"
            raise pte.PTDsgLoadException(msg, MsgLevel.error)

        gpio_id_list = []

        bus_object = self.__obj_reg.get_block_instance(bus_object_id)
        gpio_obj_list = self.__igpio.get_bus_gpio(bus_object)
        if gpio_obj_list is not None:
            for gpio_obj in gpio_obj_list:
                # Register in object registry
                handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
                gpio_id_list.append(handle_id)

        return gpio_id_list

    def get_all_gpio(self) -> List[int]:
        """
        Gets a list of all GPIO and buses used

        :return: A list of GPIO or an empty list.

        :raises PTDsgLoadException: Exception when loading a design from a file
        """
        if self.__igpio is None:
            msg = "Fail to load Internal GPIO"
            raise pte.PTDsgLoadException(msg, MsgLevel.error)

        gpio_id_list = []
        gpio_obj_list = self.__igpio.get_all_gpio()
        for gpio_obj in gpio_obj_list:
            obj_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            if obj_id is not None:
                gpio_id_list.append(obj_id)

        return gpio_id_list

    def get_object_block_type(self, inst: Union[int, str]) -> str:
        """
        Given design element instance, either object id or based on its name,
        detects the block type.

        :param inst: The block's object ID or instance name, enclose instance names in double quotes.
        :return: Type of block if supported; otherwise an empty string.
        """
        blk_type = ""

        if isinstance(inst, str):
            obj_id = self.__obj_reg.get_object_id_by_inst_name(inst)
        else:
            obj_id = inst

        if obj_id is None:
            return blk_type

        api_obj = self.__obj_reg.get_object(obj_id)

        if api_obj is None:
            return blk_type

        blk_type = api_obj.get_object_type_name()

        if blk_type is None:
            blk_type = ""

        return blk_type

    def create_input_gpio(self, name: str, msb: Optional[int]=None, lsb: Optional[int]=None) -> Optional[int]:
        """
        Create an input GPIO. To create a bus, specify values for the MSB and LSB bits.

        :param name: The GPIO name, enclose in double quotes., enclose in double quotes.
        :param msb: Integer. If creating a bus, specify the MSB. The default is ``None``.
        :param lsb: Integer. If creating a bus, specify the LSB. The default is ``None``.
        :return: The object ID of the GPIO or bus.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_input_gpio(name, msb, lsb)

            # Register in object registry
            if msb == lsb:
                api_obj_type = APIObject.ObjectType.gpio
            else:
                api_obj_type = APIObject.ObjectType.gpio_bus

            handle_id = self.__obj_reg.register_block_instance(gpio_obj, api_obj_type)
            return handle_id

        return None

    def create_output_gpio(self, name: str, msb: Optional[int]=None, lsb: Optional[int]=None) -> Optional[int]:
        """
        Create an output GPIO. To create a bus, specify values for the MSB and LSB bits.

        :param name: The GPIO name, enclose in double quotes.
        :param msb: Integer. If creating a bus, specify the MSB. The default is ``None``.
        :param lsb: Integer. If creating a bus, specify the LSB. The default is ``None``.
        :return: The object ID of the GPIO or bus
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_output_gpio(name, msb, lsb)

            # Register in object registry
            if msb == lsb:
                api_obj_type = APIObject.ObjectType.gpio
            else:
                api_obj_type = APIObject.ObjectType.gpio_bus

            handle_id = self.__obj_reg.register_block_instance(gpio_obj, api_obj_type)
            return handle_id

        return None

    def create_inout_gpio(self, name: str, msb: Optional[int]=None, lsb: Optional[int]=None) -> Optional[int]:
        """
        Create an inout GPIO. To create a bus, specify values for the MSB and LSB bits.

        :param name: The GPIO name, enclose in double quotes., enclose in double quotes.
        :param msb: Integer. If creating a bus, specify the MSB. The default is ``None``.
        :param lsb: Integer. If creating a bus, specify the LSB. The default is ``None``
        :return: The object ID of the GPIO or bus
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_inout_gpio(name, msb, lsb)

            # Register in object registry
            if msb == lsb:
                api_obj_type = APIObject.ObjectType.gpio
            else:
                api_obj_type = APIObject.ObjectType.gpio_bus

            handle_id = self.__obj_reg.register_block_instance(gpio_obj, api_obj_type)
            return handle_id

        return None

    def create_open_drain_output_gpio(self, name: str) -> Optional[int]:
        """
        Create an inout GPIO in open drain output configuration; the output is grounded.

        Constant output is set to 0 (GND).

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """
        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_open_drain_output(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_vref_gpio(self, name: str) -> Optional[int]:
        """
        Create a VREF GPIO.

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """
        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None and isinstance(self.__igpio, IntGPIOCompAPI):
            # Create using internal API
            gpio_obj = self.__igpio.create_vref_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_input_clock_gpio(self, name: str) -> Optional[int]:
        """
        Create an input clock GPIO with connection type ``GCLK``.

        :param name: The GPIO name, enclose in double quotes
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_input_clock_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_regional_input_clock_gpio(self, name: str) -> Optional[int]:
        """
        Create a regional input clock GPIO with connection type ``RCLK``

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None and isinstance(self.__igpio, IntGPIOCompAPI):
            # Create using internal API
            gpio_obj = self.__igpio.create_regional_input_clock_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_pll_input_clock_gpio(self, name: str) -> Optional[int]:
        """
        Create an input clock for the PLL (``pll_clkin``) with connection type ``PLL_CLKIN``.

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_pll_input_clock_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_pll_ext_fb_gpio(self, name: str) -> Optional[int]:
        """
        Create a PLL external feedback input GPIO (``pll_extfb_conn``) with connection type
        ``PLL_EXTFB``.

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_pll_ext_fb_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_mipi_input_clock_gpio(self, name: str) -> Optional[int]:
        """
        Create MIPI input clock GPIO (``mipi_clkin``) with connection type ``MIPI_CLKIN``. This
        input clock is used with the Trion hard MIPI blocks.

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_mipi_input_clock_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_pcie_perstn_gpio(self, name: str) -> Optional[int]:
        """
        Create PCIe input clock GPIO (``pcie_perstn``) with connection type ``PCIE_PERSTN``. This
        input clock is used with the Titanium PCIe blocks.

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None and isinstance(self.__igpio, IntGPIOCompAPI):
            # Create using internal API
            gpio_obj = self.__igpio.create_quad_pcie_perstn_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_clockout_gpio(self, name: str) -> Optional[int]:
        """
        Create an output clock GPIO (``clkout``).

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_clockout_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_global_control_gpio(self, name: str) -> Optional[int]:
        """
        Create a global control GPIO with connection type ``GCTRL``

        :param name: The GPIO name enclose in double quotes.
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_global_control_gpio(name)

            # Register in object registry
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def create_unused_gpio(self, name: str) -> Optional[int]:
        """
        Create an unused GPIO.

        :param name: The GPIO name, enclose in double quotes.
        :return: The GPIO object ID.
        """

        design = self.__idesign.get_design()
        if design is not None and self.__igpio is not None:
            # Create using internal API
            gpio_obj = self.__igpio.create_unused_gpio(name)
            handle_id = self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)
            return handle_id

        return None

    def generate(self, enable_bitstream: bool, outdir: Optional[os.PathLike]=None):
        """
        Performs design checks and generates constraint and report files. Optionally generate a
        bitstream file.

        :param enable_bitstream: Boolean ``True`` or ``False``. If ``True``, generate the bitstream file

        :param outdir: Specify an optional output directory with path. The default, None, saves files in
        the project's **outflow** directory. Enclose in double quotes.

        :raises PTDsgGenConstException: Exception when generating design constraint files
        :raises PTDsgGenReportException: Exception when generating design report file
        """
        self.clean_output()
        if self.check_design():
            self.__gen_report(outdir)
            self.__gen_constraint(enable_bitstream, outdir)

    def clean_output(self):
        """
        Cleans the generated output files in the project's **outflow** directory.
        """
        self.__idesign.clean_output()

    def check_design(self) -> bool:
        """
        Performs a design rule check. For Python to print errors, set ``is_verbose()`` to ``True``. See
        Elements of a Python Script on page 11.

        :return: Boolean ``True`` (if passes) or ``False`` (if fails).
        :raises PTDsgCheckException: Exception when running design check
        """
        self.__last_design_issue_list = []
        is_pass, issue_list = self.__idesign.check_design(get_details=True)
        for issue in issue_list:
            self.__msg_svc.write(MsgLevel.info, f"Design check issue : {issue}")
        self.__last_design_issue_list = issue_list

        return is_pass

    def export_design(self, block_type: Optional[List[str]]=None, block_inst: Optional[List[BlockInstanceType]]=None, isf_file: str="", export_all_pin: bool=False):
        """
        Exports the interface design settings for the specified block type(s) or block instance(s) to
        an **.isf** file in the project directory. If you specify both ``block_type`` and ``block_inst``, the
        block type takes precedence.

        :param block_type: Optional. A comma-separated list of block types, enclose names in double \
        quotes. The default is ``None``

        :param block_inst: Optional. A comma-separated list of instance names, enclose names in \
        double quotes. For GPIO, you can specify a list of bus instances. The default is ``None``.

        :param isf_file: : Optional. User-specified path and filename. If you do not specify a name, the \
        script creates the file <design>**.isf** in the project directory.

        :param export_all_pin: Optional. Export all pin names, including default auto-generated \
        names. The default is ``False``.

        :raises PTDsgExportException: Exception when exporting design
        """

        if block_inst is None:
            block_inst = []

        if block_type is not None:
            unsupported_block = []
            for blk_type in block_type:
                if self.__idesign.is_block_supported(blk_type) is False:
                    unsupported_block.append(blk_type)
            if len(unsupported_block) != 0:
                msg = f"Fail to export design, these block types are not supported.\n{unsupported_block}"
                raise pte.PTDsgExportException(msg, MsgLevel.error)

        try:
            exporter = DesignISFExporter(self.__is_verbose)
            design = self._get_design_db()
            assert design is not None

            exporter.export_isf(design, block_type, block_inst, isf_file, export_all_pin)
        except Exception as excp:
            msg = "Fail to export design"
            raise pte.PTDsgExportException(msg, MsgLevel.error, excp)

    def import_design(self, design_isf_file: str, gen_issue_csv: bool=False, path: str="") -> Tuple[bool, List[DesignIssue]]:
        """
        Imports the Interface Design from the specified **.isf**

        :param design_isf_file: Design filename with **.isf** extension, enclose in double quotes.

        :param gen_issue_csv: Boolean ``True`` or ``False`` (default). When ``True``, the command exports
        any design issues to a **.csv** file.

        :param path: Specify an optional path to save the issues **.csv** file. If you do not specify a path, it is
        saved into the same location as the **.isf**.

        :return:
            is_pass: Boolean indicate if the file is imported successfully
            issues: List of DesignIssue (warning / error) found during the import
        :raises PTDsgImportException: Exception when importing the design
        """

        try:
            importer = DesignISFImporter(self.__is_verbose)
            design = self._get_design_db()
            assert design is not None

            is_pass = importer.import_isf(design, design_isf_file)

            if gen_issue_csv:
                if path == "":
                    csv_file = design_isf_file.replace(".isf", "_import_issue.csv")
                else:
                    isf_filename = os.path.basename(design_isf_file)
                    csv_file = f"{path}/{isf_filename}_import_issue.csv"

                importer.export_issue_to_csv(csv_file)

        except Exception as excp:
            msg = "Fail to import design"
            raise pte.PTDsgImportException(msg, MsgLevel.error, excp)

        return is_pass, importer.issue_reg.get_all_issue()

    def setup_partial_design(self, name: str) -> DesignAPI:
        """
        Setup the data structure for partial design migration
        """
        design = self._get_design_db()

        # Detach the design instances related to this partial design,
        # They will be attached back in finish_partial_design()
        pd = dict()
        for blk_type in design.get_supported_block():
            reg = design.get_block_reg(blk_type)
            if reg is None:
                continue
            bus_insts = set()
            for inst in reg.get_all_inst():
                if inst.netlist_config is not None and inst.netlist_config.netlist_checksum == name:
                    pd[inst.name] = {
                        'blk_type': blk_type,
                        'reg': reg,
                        'inst': inst,
                    }
                    if isinstance(inst, GPIO):
                        if inst.bus_name != "":
                            bus_insts.add(inst.bus_name)
                        # I like the way that we need special handling for GPIO
                        # because of the genius implementation
                        self.__igpio.delete_gpio(inst.name)
                    else:
                        reg.delete_inst(inst.name)

            if len(bus_insts) > 0:
                for bus_name in bus_insts:
                    self.__igpio.delete_bus_gpio(bus_name)

        if design.find_partial_design(name) is None or len(design.find_partial_design(name)) == 0:
            design.add_partial_design(name, pd)

    def finish_partial_design(self, name: str):
        """
        Perform migration steps of the partial design with given name
        where
            it migrates the volatile setting for instances belongs to this partial design to the current design
            volatile settings means the settings that are not locked by the partial design and user can modify them via
            GUI / ISF
        """
        design = self._get_design_db()
        pd = design.find_partial_design(name)
        for _, store in pd.items():
            blk_type: PeriDesign.BlockType = store['blk_type']
            reg: PeriDesignRegistry = store['reg']
            inst: PeriDesignItem = store['inst']

            new_inst = reg.get_inst_by_name(inst.name)
            if new_inst is None:
                # This mean the inst was deleted by the new partial design, report it
                print(f"Instance {inst.name} is deleted in the migrated partial design")
            else:
                # Mirgate the old settings to the partial design
                prop = self.build_api_prop(blk_type, inst, self._get_design_db())
                inspector = prop.build_inspector()
                all_props = inspector.get_all_config_prop(inst)

                modified_props = {}
                locked_props = {k:v for k, v in inst.netlist_config.parameters}
                for k, v in all_props.items():
                    if k not in locked_props:
                        modified_props[k] = v

                for k, v in modified_props.items():
                    print(f'Restore original setting in partial design for instance = {inst.name}, {k} = {v}')
                    self.set_property(inst.name, k, v, block_type=self.db_blk_type_to_api_block_type_str(blk_type))

                if inst.get_device() != "" and inst.get_device() != new_inst.get_device():
                    print(f'Restore original setting in partial design for instance = {inst.name}: {inst.get_device()}')
                    self.assign_resource(inst.name, inst.get_device(), block_type=self.db_blk_type_to_api_block_type_str(blk_type))

    def db_blk_type_to_api_block_type_str(self, blk_type: PeriDesign.BlockType) -> str:
        """
        Utility function to convert Design BlockType to API BlockType string
        """
        match blk_type:
            case PeriDesign.BlockType.gpio | PeriDesign.BlockType.comp_gpio | PeriDesign.BlockType.adv_l2_gpio:
                return "GPIO"
            case PeriDesign.BlockType.pll | PeriDesign.BlockType.adv_pll | PeriDesign.BlockType.comp_pll | \
                    PeriDesign.BlockType.efx_pll_v3_comp | PeriDesign.BlockType.efx_fpll_v1:
                return "PLL"
            case PeriDesign.BlockType.jtag:
                return "JTAG"
            case PeriDesign.BlockType.osc | PeriDesign.BlockType.adv_osc:
                return "OSC"
            case _:
                raise NotImplementedError(f"Unhandled blk_type = {blk_type}")

    def build_api_prop(self, blk_type: PeriDesign.BlockType, db_inst: PeriDesignItem, design_db: PeriDesign) -> AdvProp:
        """
        Utility function to build the AdvProp for a given block type and instance.
        """
        match blk_type:
            case PeriDesign.BlockType.gpio | PeriDesign.BlockType.comp_gpio | PeriDesign.BlockType.adv_l2_gpio:
                return GPIOProp.build_gpio_prop(db_inst=db_inst, design_db=design_db)
            case PeriDesign.BlockType.pll | PeriDesign.BlockType.adv_pll | PeriDesign.BlockType.comp_pll | \
                    PeriDesign.BlockType.efx_pll_v3_comp | PeriDesign.BlockType.efx_fpll_v1:
                return PLLProp.build_pll_prop(db_inst=db_inst, design_db=design_db)
            case PeriDesign.BlockType.jtag:
                return JTAGProp.build_jtag_prop(db_inst=db_inst, design_db=design_db)
            case PeriDesign.BlockType.osc | PeriDesign.BlockType.adv_osc:
                return OSCProp.build_osc_prop(db_inst=db_inst, design_db=design_db)
            case _:
                raise NotImplementedError(f'Unhandled blk_type = {blk_type}, db_inst = {db_inst}, design_db = {design_db}')

    def convert_periphery_netlist_to_isf_commands(self, peri_netlist_file: os.PathLike) -> List[ISFAbstractCommand]:
        """
        Convert periphery netlist file to ISF Commands

        :param peri_netlist_file: The path of **.db** file

        :raises PTDsgImportException: Exception when importing the design
        """
        peri_netlist_file_path = Path(peri_netlist_file)
        if not peri_netlist_file_path.exists():
            msg = f'Cannot find {peri_netlist_file_path}'
            raise pte.PTDsgImportException(msg, MsgLevel.error)

        netlist_checksum = peri_netlist_file_path.name

        loader = PeriNetlistDBLoader()
        ndb = loader.load_netlistdb(peri_netlist_file_path)

        if ndb is None:
            msg = "Failed to load periphery netlist"
            raise pte.PTDsgImportException(msg, MsgLevel.error)

        try:
            loader.validate(ndb)
        except AssertionError as exc:
            raise pte.PTDsgImportException(str(exc), MsgLevel.error) from exc

        isf_builder = PeriNetlistISFBuilder()
        cmds = isf_builder.run(ndb, netlist_checksum)

        cmds.insert(0, ISFSetupPartialDesignCommand(design_api_handle="design", pd_name=peri_netlist_file_path.name))
        cmds.append(ISFFinishPartialDesignCommand(design_api_handle="design", pd_name=peri_netlist_file_path.name))

        return cmds

    def convert_periphery_netlist_to_isf_file(self, peri_netlist_file: os.PathLike, design_isf_file: os.PathLike):
        """
        Convert periphery netlist file to ISF File

        :param peri_netlist_file: Netlist file path with .db extension, enclose in double quotes
        :param design_isf_file: Design file path with .isf extension, enclose in double quotes
        """

        cmds = self.convert_periphery_netlist_to_isf_commands(peri_netlist_file)
        design_isf_file_path = Path(design_isf_file)
        if design_isf_file_path.exists():
            print(f'Warning, {design_isf_file_path} exist and will be overwritten')

        with open(design_isf_file, mode='w', encoding='utf-8') as file_handle:
            for cmd in cmds:
                file_handle.write(f'{cmd.format_str()}\n')

    def auto_assign_resource(self) -> List[ISFAbstractCommand]:
        """
        Auto assign resource based on the design item property

        Return a List of ISF Command executed for resource related assignment
        """
        planner = DesignAutoResourcePlanner()
        design_db = self.__idesign.get_design()
        assignment_cmds = planner(design_db)

        set_prop_cmds = list(filter(lambda c: isinstance(c, ISFSetBlockPropertyCommand), assignment_cmds))
        if len(set_prop_cmds) > 0:
            importer = DesignISFImporter(self.__is_verbose)
            importer.import_isf_commands(design_db, set_prop_cmds)
        return assignment_cmds

    def export_isf_commands_file(self, commands: List[ISFAbstractCommand], out_isf_file: os.PathLike):
        out_isf_file = Path(out_isf_file)
        if out_isf_file.exists():
            print(f"Wanring, {out_isf_file} exist and will be overwritten")

        with open(out_isf_file, mode='w', encoding='utf-8') as fp:
            for cmd in commands:
                fp.write(f'{cmd.format_str()}\n')


    def export_auto_assignment_isf_file(self, assignments: Dict[str, Tuple[PeriDesignItem, str]], out_isf_file: os.PathLike):
        out_isf_file = Path(out_isf_file)
        if out_isf_file.exists():
            print(f"Wanring, {out_isf_file} exist and will be overwritten")

        def design_type_to_command_block_type(inst: PeriDesignItem) -> str:
            if isinstance(inst, GPIO):
                return "GPIO"
            elif isinstance(inst, PLL):
                return "PLL"
            elif isinstance(inst, (OSC, HPOSC)):
                return "OSC"
            else:
                raise NotImplementedError(f"TODO: unsupported type: {type(inst)}")

        with open(out_isf_file, mode='w', encoding='utf-8') as fp:
            for inst, resource in assignments.values():
                cmd = ISFAssignBlockResourceCommand(design_api_handle='design',
                                                    block_name=inst.name,
                                                    block_type=design_type_to_command_block_type(inst),
                                                    resource_name=resource)
                fp.write(f"{cmd.format_str()}\n")

    def is_periphery_sim_supported_for_all_blocks(self) -> bool:
        """
        Return True when all the design instances support periphery sim
        Require design loaded
        """
        design_db = self.__idesign.get_design()
        if design_db is None:
            return False

        is_empty_design = True
        for reg in design_db.get_all_block_reg():
            if reg is None:
                continue
            is_empty_design &= reg.get_inst_count() == 0

        if is_empty_design:
            return False

        netlist_builder = self.__netlist_builder
        for reg in design_db.get_all_block_reg():
            if reg is None:
                continue
            for block in reg.get_all_inst():
                is_supported = netlist_builder.is_block_supported(block)
                if not is_supported:
                    return False

        return True

    def export_periphery_sim_netlist(self, top_module_name: str, out_rtl_sim_netlist: os.PathLike, out_pt_sim_netlist: os.PathLike, rtl_ndb: Optional[Database] = None, language: str = 'verilog'):
        """
        Export Design to simulation netlist that used for periphery model simulation

        rtl_sim_netlist used for run periphery model simulation with blocks defined in RTL
        pt_sim_netlist used fro run periphery simulation with blocks defined in RTL + blocks defined in peri.xml

        :param top_module_name: Top Module name of the design
        :type top_module_name: str
        :param out_rtl_sim_netlist: Output file path for sim netlist used for rtl level postmap sim
        :type out_rtl_sim_netlist: os.PathLike
        :param out_pt_sim_netlist: Output file path for sim netlist used for full design level postmap sim
        :type out_pt_sim_netlist: os.PathLike
        :param rtl_ndb: Periphery NetlistDB generated by synthesis
        :type rtl_ndb: Optional[Database]
        :param language: Format of the sim netlist, defaults to 'verilog'
        :type language: str, optional
        """
        assert language in ('verilog')

        design_db = self.__idesign.get_design()
        assert design_db is not None

        is_empty_design = True
        for reg in design_db.get_all_block_reg():
            if reg is None:
                continue
            is_empty_design &= reg.get_inst_count() == 0

        if is_empty_design:
            msg = "Fail to export simulation netlist, the design doesn't contain periphery block\n"
            raise pte.PTDsgExportException(msg, MsgLevel.error)

        netlist_builder = self.__netlist_builder
        for reg in design_db.get_all_block_reg():
            if reg is None:
                continue
            for block in reg.get_all_inst():
                is_supported = netlist_builder.is_block_supported(block)
                if not is_supported:
                    msg = f"Fail to export simulation netlist, the design contains supported block: {type(block)}(name={block.name})\n"
                    raise pte.PTDsgExportException(msg, MsgLevel.error)

        if not self.check_design():
            msg = "Fail to export design simulation netlist, the design rule check failed.\n"
            raise pte.PTDsgExportException(msg, MsgLevel.error)

        ndb = netlist_builder.run(design_db, top_module_name, rtl_ndb)

        writer = PeripherySimVerilogWriter()
        # Write rtl sim
        rtl_sim_netlist = ndb.get_netlist(cell_name=top_module_name)
        ndb.set_top_module(rtl_sim_netlist)

        with open(out_rtl_sim_netlist, encoding='utf-8', mode='w') as fp:
            writer.write(ndb, fp)

        # Write pt sim
        fc_sim_netlist = ndb.get_netlist(cell_name=f'{top_module_name}~chip')
        ndb.set_top_module(fc_sim_netlist)

        with open(out_pt_sim_netlist, encoding='utf-8', mode='w') as fp:
            writer.write(ndb, fp)

    def export_periphery_timing_netlist(self, timing_netlist_file: os.PathLike,
            timing_fixed_delay_file: os.PathLike,
            timing_cell_site_instance_file: os.PathLike):

        from netlistdb import NetlistDBPBNWriter

        design_db = self.__idesign.get_design()
        assert design_db is not None

        is_empty_design = True
        for reg in design_db.get_all_block_reg():
            if reg is None:
                continue
            is_empty_design &= reg.get_inst_count() == 0

        if is_empty_design:
            msg = "Fail to export timing netlist, the design doesn't contain periphery block\n"
            raise pte.PTDsgExportException(msg, MsgLevel.error)

        netlist_builder = PeriTimingNetlistBuilder()
        for reg in design_db.get_all_block_reg():
            if reg is None:
                continue
            for block in reg.get_all_inst():
                is_supported = netlist_builder.is_block_supported(block)
                if not is_supported:
                    msg = f"Fail to export timing netlist, the design contains supported block: {type(block)}(name={block.name})\n"
                    raise pte.PTDsgExportException(msg, MsgLevel.error)

        if not self.check_design():
            msg = "Fail to export timing netlist, the design rule check failed.\n"
            raise pte.PTDsgExportException(msg, MsgLevel.error)

        ndb = netlist_builder.run(design_db, f'{design_db.name}~chip')

        writer = NetlistDBPBNWriter()

        assert writer is not None
        with open(timing_netlist_file, encoding='utf-8', mode='w') as fp:
            writer.write(ndb, fp)

        with open(timing_fixed_delay_file, encoding='utf-8', mode='w') as fp:
            writer.write_qdelay(ndb, fp)

        with open(timing_cell_site_instance_file, encoding='utf-8', mode='w') as fp:
            writer.write_csi(ndb, fp)

    def lock_property(self, inst: BlockInstanceType, prop_name: str, prop_value: str, block_type: str, netlist_checksum: str):
        try:
            # if self.__idesign.is_block_supported(block_type) is False:
            #     msg = f"Fail to lock property, block type {block_type} is not supported"
            #     raise pte.PTBlkReadException(msg, MsgLevel.error)

            object_id = self.__get_object_id(inst, block_type)
            if object_id is None:
                msg = f"Fail to lock property. Unknown instance {inst}"
                print(msg)
                return None
                # raise pte.PTBlkReadException(msg, MsgLevel.error)

            reg_obj = self.__obj_reg.get_object(object_id)
            if reg_obj is not None:
                obj_inst: Optional[PeriDesignItem] = self.__obj_reg.get_block_instance(object_id)
                if obj_inst is None:
                    msg = "Fail to lock property"
                    raise pte.PTBlkEditException(msg, MsgLevel.error)

                netlist_config = obj_inst.netlist_config
                if netlist_config is None:
                    netlist_config = PeriDesignItemNetlistConfig(netlist_checksum)
                    obj_inst.netlist_config = netlist_config


                netlist_config.add_parameter(prop_name, prop_value)
        except PTException as excp:
            msg = "Fail to lock property"
            raise pte.PTBlkReadException(msg, MsgLevel.error, excp)

        return

    def get_design_check_issue(self, raw: bool = False) -> List[str | DesignIssue]:
        """
        Get a list of design check issues.

        :return: A list of messages.
        :return: A list of DesignIssue when "raw" is True
        """
        design_issues: List[str | DesignIssue] = []
        if self.__last_design_issue_list is None:
            return design_issues

        for issue in self.__last_design_issue_list:
            if raw:
                design_issues.append(issue)
            else:
                issue_desc = f"Id:{issue.issue_id},Inst:{issue.instance_name},Type:{issue.instance_type},Sev:{issue.severity},Rule:{issue.rule_name},Msg:{issue.msg}"
                design_issues.append(issue_desc)

        return design_issues

    def calc_pll_clock(self, inst: BlockInstanceType) -> Dict[str, str]:
        """
        Calculates the VCO frequency, PLL frequency, and all enabled output clock frequencies.
        You should configure the M, N, and O counters and output dividers before using this
        function

        :param inst: A PLL instance block object id or an instance name.
        :return: Calculated frequencies property map. For example

        :raises PTBlkEditException: Exception when editing a block property
        """

        object_id = self.__get_object_id(inst, "PLL")
        if object_id is None:
            msg = f"Fail to calculate PLL clocks. Unknown instance {inst}"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        obj_inst = self.__obj_reg.get_block_instance(object_id)
        blk_obj = self.__obj_reg.get_object(object_id)

        if obj_inst is not None and blk_obj is not None:
            obj_type = blk_obj.get_object_type()
            if obj_type == APIObject.ObjectType.pll:
                ipll = self.__idesign.get_block_apimgr("PLL")
                if ipll is not None:
                    ipll.calc_clock(obj_inst)
                    assert self.__infra_svc is not None
                    prop_svc = self.__infra_svc.get_prop_service(object_id=object_id)
                    if prop_svc is not None:
                        assert isinstance(prop_svc, PLLProp)
                        prop_map = prop_svc.get_frequency_property(obj_inst)
                        return prop_map

        return {}

    def get_all_preset_info(self, name: str, block_type: str = "DDR") -> List[Tuple[str, str]]:
        """
        Get all preset options according to the provided instance name.

        :param name: The instance name. Enclose instance names in double quotes.
        :return: Returns all available preset of the provided instance name
        :raises PTBlkReadException: Exception when querying for a block property
        """
        if block_type not in ["DDR", "PMA_DIRECT"]:
            raise pte.PTBlkReadException(f"Unsupported block type {block_type}", MsgLevel.error)

        elif self.__idesign.is_block_supported(block_type) is False:
            msg = f"Fail to get all preset info, block type {block_type} is not supported"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        inst: Any = self.__idesign.get_block(name, block_type)

        if inst is None:
            msg = f"Fail to get all preset options. Unknown instance {inst}"
            raise pte.PTBlkReadException(msg, MsgLevel.error)

        iblk = self.__idesign.get_block_apimgr(block_type)

        if iblk is None:
            msg = f"Fail to get all preset options, block type {block_type} is not supported"
            raise pte.PTBlkReadException(msg, MsgLevel.error)

        return iblk.get_all_preset_info(inst)

    def set_preset(self, name: str, preset_id: str, block_type: str = "DDR") -> Tuple[str, str]:
        """
        Set preset ID and define which preset is used.

        :param name: The instance name. Enclose instance names in double quotes.
        :param preset_id: The preset ID. If block_type is `PMA_DIRECT`, please use preset description
        :param block_type: The type of the instance. Can be `DDR` or `PMA_DIRECT`.

        :return: Returns current preset ID and description.
        """
        if block_type not in ["DDR", "PMA_DIRECT"]:
            raise pte.PTBlkReadException(f"Unsupported block type {block_type}", MsgLevel.error)

        self.set_property(name, "PRESET", prop_value=preset_id, block_type=block_type)

        return self.get_preset(name, block_type)

    def get_preset(self, name: str, block_type: str = "DDR") -> Tuple[str, str]:
        """
        Get preset ID and preset description of current DDR instance.

        :param name: The instance name. Enclose instance names in double quotes.
        :return: Returns current preset ID and description.

        :raises PTBlkReadException: Exception when querying for a block property
        """
        if block_type not in ["DDR", "PMA_DIRECT"]:
            raise pte.PTBlkReadException(f"Unsupported block type {block_type}", MsgLevel.error)

        elif self.__idesign.is_block_supported(block_type) is False:
            msg = f"Fail to get all preset options, block type {block_type} is not supported"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        preset_info = ("", "")
        inst: Any = self.__idesign.get_block(name, block_type)

        if inst is None:
            return preset_info

        iblk: Optional[Any] = self.__idesign.get_block_apimgr(block_type)

        if iblk is None:
            msg = f"Fail to get all preset options, block type {block_type} is not supported"
            raise pte.PTBlkReadException(msg, MsgLevel.error)

        preset_info = iblk.get_preset_info(inst)
        return preset_info

    def trace_ref_clock(self, inst: BlockInstanceType, block_type: str="PLL") -> Optional[List[Dict]]:
        """
        Trace the clock source for the target instance. If the instance is a PLL, the clock source is a
        GPIO instance (PLL V1 and V2) or a GPIO or LVDS instance (PLL V3). In v2021.2, only the
        PLL block is supported.

        :param inst: A PLL instance block object id or an instance name.
        :param block_type: Block type of the instance if it is specified by name.
        :return: A list of one or more clock source property maps.

        :raises PTBlkEditException: Exception when editing a block property
        """

        if block_type != "PLL":
            self.__msg_svc.write(MsgLevel.warning, "Trace reference clock fails. Only PLL block type is supported")
            return None

        object_id = self.__get_object_id(inst, block_type)
        if object_id is None:
            msg = f"Fail to get clock source. Unknown instance {inst}"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        obj_inst = self.__obj_reg.get_block_instance(object_id)
        blk_obj = self.__obj_reg.get_object(object_id)

        if obj_inst is not None and blk_obj is not None:
            obj_type = blk_obj.get_object_type()
            if obj_type == APIObject.ObjectType.pll:
                ipll = self.__idesign.get_block_apimgr("PLL")
                if ipll is not None:
                    clk_source_info = ipll.trace_ref_clock(obj_inst)
                    if clk_source_info is not None:
                        clk_src_list = []
                        for clk_src in clk_source_info:
                            clk_src_map = {}
                            gpio_name, gpio_res, gpio_pin = clk_src
                            clk_src_map["NAME"] = gpio_name
                            clk_src_map["RESOURCE"] = gpio_res
                            clk_src_map["IN_PIN"] = gpio_pin
                            clk_src_list.append(clk_src_map)

                        return clk_src_list
            else:
                self.__msg_svc.write(MsgLevel.info, "Only PLL block type is supported")

        return None

    def gen_pll_ref_clock(self, inst: BlockInstanceType, **kwargs):
        r"""
        Generate a PLL reference clock. Note: the software automatically detects the GPIO
        resource. Refer to Example Scripts on page 14 for examples.

        :param inst: PLL block
        :param kwargs: \**kwargs are parameters that are used for specific block types

        :Keyword Arguments:
            * *pll_res* --
                PLL resource name, will overwrite existing resource if it exists.

            * *refclk_name* --
                GPIO instance name for reference clock.

        :raises PTBlkEditException: Exception when editing a block property
        """

        object_id = self.__get_object_id(inst, "PLL")
        if object_id is None:
            msg = f"Fail to generate PLL clock source. Unknown instance {inst}"
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        obj_inst = self.__obj_reg.get_block_instance(object_id)
        blk_obj = self.__obj_reg.get_object(object_id)

        try:
            if obj_inst is not None and blk_obj is not None:
                obj_type = blk_obj.get_object_type()
                if obj_type == APIObject.ObjectType.pll:
                    ipll = self.__idesign.get_block_apimgr("PLL")
                    if ipll is not None:
                        ipll.gen_ref_clock(obj_inst, **kwargs)
        except Exception as excp:
            msg = "Fail to generate PLL clock source"
            raise pte.PTBlkEditException(msg, MsgLevel.error, excp)

    def auto_calc_pll_clock(self, inst: BlockInstanceType, target_freq: Optional[Dict[str, str]]=None, apply_optimal: bool=True, result_len: Optional[int] = None) -> List[Dict]:
        """
        Given desired output clock frequency in MHz (and/or phase in degree), this function sets
        the required counters and output divider.

        :param inst: A PLL instance block object id or an instance name.

        :param target_freq: A map of output frequency parameters in MHz. The parameters depend \
                            on the PLL you are using. If you do not include a parameter, the software uses the default \
                            setting.

        :param apply_optimal: When True, apply optimal result counters to PLL, when False, does not
        :param result_len: Number of results. 5 results will be given by default. Get all results if 0

        :returns: A list of PLL setting options, a list of property maps.
        :raises PTBlkEditException: Exception when editing a block property
        """

        object_id = self.__get_object_id(inst, "PLL")
        if object_id is None:
            msg = f"Fail to auto calculate PLL clocks. Unknown instance {inst}."
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        if target_freq is None:
            msg = "Fail to auto calculate PLL clocks. Need to specify desired frequency."
            raise pte.PTBlkEditException(msg, MsgLevel.error)

        obj_inst = self.__obj_reg.get_block_instance(object_id)
        blk_obj = self.__obj_reg.get_object(object_id)
        if obj_inst is not None and blk_obj is not None:
            obj_type = blk_obj.get_object_type()
            if obj_type == APIObject.ObjectType.pll:
                ipll = self.__idesign.get_block_apimgr("PLL")
                if ipll is not None:
                    return ipll.run_clock_auto_calc(obj_inst, target_freq, apply_optimal, result_len)

        return []

    def get_all_global_mux_name(self) -> List[str]:
        """
        :return: a list of string which indicates the names for the global
                clock mux
        """
        if self.__idesign.get_idevice_setting().is_setting_supported("CLKMUX") is False:
            self.__msg_svc.write(MsgLevel.warning, f"Fail to get device setting, CLKMUX is not supported")
            return []

        try:
            iclkmux = self.__idesign.get_idevice_setting().clkmux
            return iclkmux.get_all_clockmux_name()

        except PTException as excp:
            msg = "Fail to get Global clock mux names"
            raise pte.PTDevSetReadException(msg, MsgLevel.error, excp)

    def get_regional_buffer_info(self, clkmux_name: str, regional_buf: Optional[str]=None) -> List[Tuple]:
        """
        Get the information associated to the regional buffer resource being queried.
        If the resource has been properly configured for regional clock usage, then
        a valid information will be returned.

        :param clkmux_name: The name of the sides
        :param regional_buf: The RBUF# string being queried
        :return: A list of tuple of the following information:
            resource name: The block resource name
            pin name: If it has been configured correctly for regional buffer usage
            connected to global: True/False indicating if it is connected to global buffer
        """
        if self.__idesign.get_idevice_setting().is_setting_supported("CLKMUX") is False:
            self.__msg_svc.write(MsgLevel.warning, f"Fail to get device setting, CLKMUX is not supported")
            return []

        try:
            iclkmux = self.__idesign.get_idevice_setting().clkmux

            # Get the regional clockmux info
            return iclkmux.get_regional_buffer_info(clkmux_name, regional_buf)

        except PTException as excp:
            msg = "Fail to get Regional Buffer information"
            raise pte.PTDevSetReadException(msg, MsgLevel.error, excp)

    def get_global_dynamic_mux_input_info(self, clkmux_name: str, dynamic_mux: str, clock_input: str,
                                      index: Optional[str]=None) -> List[Tuple]:
        """
        Get the information associated to the input at the specific Mux input index
        of the provided Dynamix mux.  If the index is not provided, it will return
        all the information to all possible index of the specified mux input.

        :param clkmux_name: The name of the sides
        :param dynamic_mux: The dynamic mux index to query (0 or 7)
        :param clock_input: The Clock input index on the mux to query (0-3)
        :param index: The index of the input (device dependent 0-3 for Ti60 and 0-5 for Ti180)
        :return: A list of tuple of the following information:
            index: The index on the mux input options list
            resource name:  The block resource name
            instance name: The name of the instance that has been correctly configured
                    to connect to the global mux.
        """
        if self.__idesign.get_idevice_setting().is_setting_supported("CLKMUX") is False:
            self.__msg_svc.write(MsgLevel.warning, f"Fail to get device setting, CLKMUX is not supported")
            return []

        try:
            iclkmux = self.__idesign.get_idevice_setting().clkmux

            # Get the regional clockmux info
            return iclkmux.get_global_dynamic_mux_input_info(clkmux_name, dynamic_mux,
                                                         clock_input, index)

        except PTException as excp:
            msg = "Fail to get Regional Buffer information"
            raise pte.PTDevSetReadException(msg, MsgLevel.error, excp)

    def reset_device_settings(self, devset_types: Optional[List[str]] = None):
        idevset = self.__idesign.get_idevice_setting()

        try:
            if devset_types is None:
                idevset.reset_all_setting()

            else:
                for devset_type in devset_types:
                    if idevset.is_setting_supported(devset_type) is False:
                        msg = f"Fail to reset device property, setting {devset_type} is not supported"
                        raise pte.PTDevSetEditException(msg, MsgLevel.error)

                    idevset.reset_setting(devset_type)

        except PTException as excp:
            msg = "Fail to reset device settings"
            raise pte.PTDevSetEditException(msg, MsgLevel.error, excp)

    def clear_design(self, block_types: Optional[List[str]]=None):
        """
        Clear all the existing design

        :param block_types: A list of block types. If not provided, all design will be clear
        """
        if block_types is None:
            block_types = self.__idesign.get_block_type()

        try:
            # Delete all instances in reg
            for block_type in block_types:
                if self.__idesign.is_block_supported(block_type) is False:
                    msg = f"Fail to set property, block type {block_type} is not supported"
                    raise pte.PTBlkEditException(msg, MsgLevel.error)

            self.__idesign.clear_design(block_types)

        except PTException as excp:
            msg = "Fail to clear design"
            raise pte.PTDevSetEditException(msg, MsgLevel.error, excp)

    def __get_bus_property(self, object_id: int, prop_name: Union[str, List[str]]) -> Dict[str, str]:
        """
        Get bus and its member properties

        :param object_id: Bus object id
        :param prop_name: Property name or a list of property name
        :return: A map, dict of a property name to its value
        :raises PTBlkReadException: Exception when querying for a block property
        """

        def get_bprop(pname):
            # Try bus property
            pmap = self.__get_simple_property(object_id, pname)
            if pmap is None or len(pmap) == 0:  # Not bus property
                gpio_obj = self.__get_bus_first_gpio(object_id)
                return self.__get_simple_property(gpio_obj, pname)

            return pmap

        if isinstance(prop_name, list):
            all_prop_map = {}
            for name in prop_name:
                prop_map = get_bprop(name)
                all_prop_map[name] = prop_map.get(name, "")

            return all_prop_map
        else:
            return get_bprop(prop_name)

    def __get_simple_property(self, object_id, prop_name) -> Dict[str, str]:
        """
        Get direct value for target design object property

        :param object_id: Design object id
        :param prop_name: Property name or a list of property name
        :return: A map, dict of a property name to its value
        :raises PTBlkReadException: Exception when querying for a block property
        """
        prop_map = {}

        obj_inst = self.__obj_reg.get_block_instance(object_id)
        if obj_inst is not None and self.__infra_svc is not None:
            prop_svc = self.__infra_svc.get_prop_service(object_id=object_id)
            if prop_svc is not None:
                if isinstance(prop_name, list):
                    all_prop_map = {}
                    for name in prop_name:
                        prop_map = prop_svc.get_property(name)
                        all_prop_map[name] = prop_map.get(name, "")

                    return all_prop_map
                else:
                    return prop_svc.get_property(prop_name)
        else:
            msg = f"Fail to get property. Fail to find db object for id<{object_id}>"
            raise pte.PTBlkReadException(msg, MsgLevel.error)

        return prop_map


    def __get_bus_first_gpio(self, bus_object_id):
        """
        Get first bus member

        :param bus_object_id: GPIO Bus object id
        :return: First gpio
        """

        bus_object = self.__obj_reg.get_block_instance(bus_object_id)
        if bus_object is None:
            return None

        gpio_obj = bus_object.get_first_member()

        if gpio_obj is not None:
            # Register in object registry
            return self.__obj_reg.register_block_instance(gpio_obj, APIObject.ObjectType.gpio)

        return None

    def __gen_constraint(self, enable_bitstream, outdir=None):
        """
        Generate all constraint files including verilog template file.
        The constraints are
        - pcr
        - core interface
        - timing

        :param enable_bitstream: True, generate PCR else no
        :param outdir: When None, use default outflow directory for project, else use the specified dir
        :raises PTDsgGenConstException: Exception when generating design constraint files
        """
        self.__idesign.gen_constraint(enable_bitstream, outdir)

    def __gen_report(self, outdir=None):
        """
        Generate all report files.
        The reports are
        - summary
        - pinout

        For pinout, an Efinity project file must be available at the periphery design file
        location.

        :param outdir: When None, use default outflow directory for project, else use the specified dir
        :raises PTDsgGenReporttException: Exception when generating design report file
        """
        self.__idesign.gen_report(outdir)

    def __get_object_id(self, inst, block_type):
        """
        Given an instance, find its object

        :param inst: Instance name
        :param block_type: Block type name to search for
        :return: An object if found, else None
        """

        if isinstance(inst, str):
            block_type_id = APIObject.str2otype_map.get(block_type, None)
            if block_type_id is None:
                return None

            if block_type_id == APIObject.ObjectType.gpio:
                object_id = self.get_gpio(inst)
                if object_id is None:
                    object_id = self.get_bus(inst)

            elif block_type_id == APIObject.ObjectType.gpio_bus:
                object_id = self.get_bus(inst)
            elif block_type_id == APIObject.ObjectType.iobank:
                object_id = self.__get_iobank(inst)
            elif block_type_id == APIObject.ObjectType.seu:
                object_id = self.__get_seu()
            elif block_type_id == APIObject.ObjectType.rcu:
                object_id = self.__get_ctrl()
            elif block_type_id == APIObject.ObjectType.ext_flash:
                object_id = self.__get_ext_flash()
            elif block_type_id == APIObject.ObjectType.clkmux:
                object_id = self.__get_clockmux(inst)
            else:
                object_id = self.get_block(inst, block_type)
        else:
            object_id = inst

        return object_id

    def __get_iobank(self, bank_name):
        iiobank = self.__idesign.get_idevice_setting().get_iobank_svc()

        if iiobank is None:
            return None

        blk_obj = iiobank.get_iobank(bank_name)
        if blk_obj is not None:
            handle_id = self.__obj_reg.register_block_instance(blk_obj, APIObject.ObjectType.iobank)
            return handle_id

        if self.__is_verbose:
            self.__msg_svc.write(MsgLevel.warning,
                                 f"Fail to get IOBank <{bank_name}>, it doesn't exists.")

        return None

    def __get_seu(self):
        seu = self.__idesign.get_idevice_setting().seu

        if seu is None:
            return None

        blk_obj = seu.get_seu()
        if blk_obj is not None:
            handle_id = self.__obj_reg.register_block_instance(blk_obj, APIObject.ObjectType.seu)
            return handle_id

        if self.__is_verbose:
            self.__msg_svc.write(MsgLevel.warning,
                                 "Fail to get SEU, it doesn't exists.")

        return None

    def __get_ctrl(self):
        ctrl = self.__idesign.get_idevice_setting().ctrl

        if ctrl is None:
            return None

        blk_obj = ctrl.get_ctrl()
        if blk_obj is not None:
            handle_id = self.__obj_reg.register_block_instance(blk_obj, APIObject.ObjectType.rcu)
            return handle_id

        if self.__is_verbose:
            self.__msg_svc.write(MsgLevel.warning,
                                 "Fail to get RU, it doesn't exists.")

        return None

    def __get_ext_flash(self):
        ext_flash = self.__idesign.get_idevice_setting().ext_flash

        if ext_flash is None:
            return None

        blk_obj = ext_flash.get_ext_flash()
        if blk_obj is not None:
            handle_id = self.__obj_reg.register_block_instance(blk_obj, APIObject.ObjectType.ext_flash)
            return handle_id

        if self.__is_verbose:
            self.__msg_svc.write(MsgLevel.warning,
                                 "Fail to get EXT_FLASH, it doesn't exists.")

        return None

    def __get_clockmux(self, clkmux_name: str):
        clkmux = self.__idesign.get_idevice_setting().clkmux
        blk_obj = clkmux.get_clockmux_based_on_display_name(clkmux_name)
        if blk_obj is not None:
            handle_id = self.__obj_reg.register_block_instance(blk_obj, APIObject.ObjectType.clkmux)
            return handle_id

        if self.__is_verbose:
            self.__msg_svc.write(MsgLevel.warning,
                                 f"Fail to get CLKMUX <{clkmux_name}>, it doesn't exists.")

        return None

    def _get_design_db(self) -> Optional[PeriDesign]:
        return self.__idesign.get_design()

    def _get_internal_design(self) -> IntDesignAPI:
        return self.__idesign


if __name__ == "__main__":
    import doctest

    doctest.testmod()
