"""
Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Aug 10, 2018

@author: yasmin
"""

from __future__ import annotations
import re
from typing import TYPE_CHECKING

import util.gen_util
import device.db_interface as devdb_int

import common_device.rule as base_rule
import common_device.pll.pll_rule as an08_pllrule

from common_device.gpio.gpio_design import GPIOInput
from common_device.lvds.lvds_design import LVDSRx

from tx60_device.gpio.gpio_design_comp import GPIORegistryComplex, GPIOInputComplex
from tx60_device.lvds.lvds_design_adv import LVDSRxAdvance

if TYPE_CHECKING:
    from an20_device.pll.pll_design_adv import PLLAdvance, PLLRegistryAdvance


@util.gen_util.freeze_it
class PLLCheckerAdv(an08_pllrule.PLLChecker):
    def __init__(self, design):
        super().__init__(design)
        self.reg: PLLRegistryAdvance

    def _build_rules(self):
        """
        Build pll rule database
        """
        self._add_rule(an08_pllrule.PLLRuleInstanceName())
        self._add_rule(an08_pllrule.PLLRuleInputFreq())
        self._add_rule(an08_pllrule.PLLRuleMultiplier())
        self._add_rule(an08_pllrule.PLLRulePreDivider())
        self._add_rule(an08_pllrule.PLLRulePostDivider())
        self._add_rule(PLLRuleOutputDivider())
        self._add_rule(an08_pllrule.PLLRuleOutputClock())
        self._add_rule(an08_pllrule.PLLRuleOutputNumber())
        self._add_rule(an08_pllrule.PLLRuleResource())
        self._add_rule(PLLRuleVCOFreq())
        self._add_rule(PLLRuleOutputClockFreq())
        self._add_rule(an08_pllrule.PLLRuleOutputName())
        self._add_rule(PLLRuleRefClk())
        self._add_rule(PLLRuleFeedback())
        self._add_rule(PLLRulePhaseShiftDiv())
        # PT-1604: Disabling this for Trion since extfb is never supported
        #self._add_rule(PLLRuleFeedbackResource())
        self._add_rule(an08_pllrule.PLLRulePinEmptyName())
        self._add_rule(an08_pllrule.PLLRulePinInvalidName())
        self._add_rule(an08_pllrule.PLLRuleInputFreqLimit())
        self._add_rule(PLLRuleNonInternalFeedbackFreq())
        self._add_rule(PLLRulePhaseShiftPostDiv())
        self._add_rule(PLLRuleClockSelectorPinName())
        self._add_rule(an08_pllrule.PLLRuleDividerRelationWhenCascade())
        self._add_rule(PLLRulePostDividerMultipleOutputClock())
        self._add_rule(PLLRuleNonInternalVCOLowerBound())
        self._add_rule(PLLRulePostVCOFrequency())
        self._add_rule(PLLRuleMultiplierLockTime())

    def check_depended_resources(self, gpio_reg, lvds_reg, dev_res_list,
                                 expected_conn_type, lvds_expected_conn_type):
        '''
        Function use to check if the depended resources
        are configured correctly.

        :param gpio_reg: GPIO Design Registry
        :param lvds_reg: LVDS Design Registry
        :param dev_res_list: List of device instance
                names that is going to be investigated
        :param expected_conn_type: The expected connection type to check
        :return: a tuple of:
            1) count: The number of resource that was configured
                    based on the list
            2) non_global: Indication that the resource was not set
                    to alternate connection type
            3) invalid_gpio: a string of gpio resource names that
                    are invalid
        '''
        count = 0
        non_global = False
        gpio_name = ""

        for gpio_res in dev_res_list:
            tmp_name = ""

            gpio = gpio_reg.get_gpio_by_device_name(gpio_res)
            if gpio is None:
                # Check if it is configured as LVDS
                if lvds_reg is not None:
                    lvds = lvds_reg.get_inst_by_device_name(gpio_res)
                    if lvds is not None:
                        # Check that it was configured as alternate type
                        if lvds.ops_type == lvds.OpsType.op_rx and\
                                lvds.rx_info is not None:
                            lvds_rx = lvds.rx_info
                            if lvds_rx.conn_type == lvds_expected_conn_type:
                                count += 1

                            else:
                                non_global = True
                                tmp_name = gpio_res

            # Check the validity of the CLKIN Resource
            else:
                # If there is dependent instances, then we
                # check its usage. This could be an lvds gpio
                # as well.
                if gpio.mode == gpio.PadModeType.input:
                    gpio_input = gpio.input
                    if gpio_input is not None:
                        if gpio_input.conn_type == expected_conn_type:
                            count += 1

                        else:
                            non_global = True
                            tmp_name = gpio_res

            if tmp_name != "":
                if gpio_name != "":
                    gpio_name += "," + tmp_name
                else:
                    gpio_name = tmp_name

        return count, non_global, gpio_name

    def check_outclk_driving_ddr(self, pll_inst):
        """
        Check if the specified instance is driving any DRR pin
        that is being configured.
        :param pll_inst: The PLL design instance object to check
        :return A tuple of :
            1) True if the PLL has output clock that is dependent
                by a DDR instance
            2) The ddr instance object that depends on this pll (None if
                not drivint a DDR)
        """
        is_conn_to_ddr = False
        found_ddr_inst = None

        if self.design is None:
            return is_conn_to_ddr, found_ddr_inst

        # If the design does not have any DDR, then return False
        ddr_reg = self.design.ddr_reg
        if ddr_reg is None:
            return is_conn_to_ddr, found_ddr_inst

        # Get the list of configured DDR Instance in the device
        all_ddr_inst = ddr_reg.get_all_inst()

        if all_ddr_inst and self.design.device_db is not None:
            # Get the DDR device service
            dbi = devdb_int.DeviceDBService(self.design.device_db)
            ddrdev_service = dbi.get_block_service(
                devdb_int.DeviceDBService.BlockType.DDR)

            # Check the dependency list of DDR configured design
            # instance
            for ddr_inst in all_ddr_inst:
                if ddr_inst is not None and ddr_inst.ddr_def != "":

                    result = ddr_inst.find_phy_clock_info(
                        ddrdev_service, self.reg)

                    if result is not None:
                        (_, des_ins_name, _) = result

                        if des_ins_name == pll_inst.name:
                            is_conn_to_ddr = True
                            found_ddr_inst = ddr_inst
                            break

        return is_conn_to_ddr, found_ddr_inst


class PLLRuleOutputDivider(an08_pllrule.PLLRuleOutputDivider):
    """
    Check if output divider is valid
    """

    def check(self, checker, design_block: PLLAdvance):

        util.gen_util.mark_unused(checker)

        pll = design_block

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        for clk in out_clock_list:
            if clk.out_divider < 1 or clk.out_divider > 256:
                self.error("Output divider for {} is invalid. Valid values are between 1-256." \
                    .format(clk.name))


class PLLRuleRefClk(base_rule.Rule):
    """
    Checks the validity of the Reference clock
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_refclk"

    def get_resource_names_on_ext_clock(self, ext_clk_info_list):
        '''
        :param ext_clk_info_list: a list of tuple returned from the PLLAdvance
                    find_external_clock_info
        :return a string of resource name associated to the specified list.
                    If more than one resource is found, the string will contain
                    ',' as separator
        '''

        resource_names = ""

        if ext_clk_info_list:
            for info_tup in ext_clk_info_list:
                res_name, _, _ = info_tup

                if resource_names == "":
                    resource_names = res_name
                else:
                    resource_names = resource_names + "," + res_name

        return resource_names

    def check_external(self, checker, design_block: PLLAdvance):

        pll = design_block

        valid = True

        gpio_reg = checker.design.gpio_reg
        if gpio_reg is None:
            checker.logger.warning(
                "{}: gpio reg is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return valid

        lvds_reg = checker.design.lvds_reg

        is_found_pin1 = False
        if pll.get_refclk2_pin_name(checker.design) != "":
            is_found_pin1 = True

        is_found_pin2 = False
        if pll.get_refclk3_pin_name(checker.design) != "":
            is_found_pin2 = True

        # Don't need to check the name matches or not because
        # it is auto populated based on resource config for external
        index_list = []

        if pll.ref_clock_mode == pll.RefClockModeType.external:
            device_db = checker.design.device_db

            # Either refclk2 or refclk3 is set
            if pll.ext_ref_clock_id is None or \
                (pll.ext_ref_clock_id != pll.RefClockIDType.ext_clock_0 and
                 pll.ext_ref_clock_id != pll.RefClockIDType.ext_clock_1):

                self.error('External refclk pin has to be set in external mode')
                return False

            if pll.ext_ref_clock_id == pll.RefClockIDType.ext_clock_0 and\
                    is_found_pin1 is False:

                ext_clk0_info_list = pll.find_external_clock_info(pll.RefClockIDType.ext_clock_0,
                                                                  checker.plldev_service, lvds_reg, gpio_reg, device_db)
                if ext_clk0_info_list:
                    # PT-800: Provide the External resource information
                    res_names = self.get_resource_names_on_ext_clock(
                        ext_clk0_info_list)
                    #self.msg = 'Selected external clock 0 pin name has to be specified'
                    self.msg = 'Reference clock at {} connected to external clock pin 0 has not been defined'.format(
                        res_names)

                else:
                    # PT-744: Use a different message if it was an instance not
                    # bonded out
                    self.msg = 'Invalid external clock 0 resource selected: Resource Unbonded'

                self.error()
                return False

            elif pll.ext_ref_clock_id == pll.RefClockIDType.ext_clock_1 and\
                    is_found_pin2 is False:

                ext_clk1_info_list = pll.find_external_clock_info(pll.RefClockIDType.ext_clock_1,
                                                                  checker.plldev_service, lvds_reg, gpio_reg, device_db)

                if ext_clk1_info_list:
                    # PT-800: Provide the External resource information
                    res_names = self.get_resource_names_on_ext_clock(
                        ext_clk1_info_list)
                    #self.msg = 'Selected external clock 1 pin name has to be specified'
                    self.msg = 'Reference clock at {} connected to external clock pin 1 has not been defined'.format(
                        res_names)

                else:
                    # PT-744: Use a different message if it was an instance not
                    # bonded out
                    self.msg = 'Invalid external clock 1 resource selected: Resource Unbonded'

                self.error()
                return False

            # Get the pin name that corresponds to the chosen resource
            index_list.append(int(pll.ext_ref_clock_id))

        elif pll.ref_clock_mode == pll.RefClockModeType.dynamic:
            # Both bonded out pin name needs to be specified in dynamic
            ext_pin1_bonded = True
            ext_pin2_bonded = True

            if is_found_pin1 is False or is_found_pin2 is False:
                device_db = checker.design.device_db

                # PT-882: Flag only if the missing pin is not the unbonded ones
                is_missing = False

                if is_found_pin1 is False:
                    ext_clk0_info_list = pll.find_external_clock_info(pll.RefClockIDType.ext_clock_0,
                                                                      checker.plldev_service, lvds_reg, gpio_reg,
                                                                      device_db)

                    if ext_clk0_info_list:
                        is_missing = True
                    else:
                        ext_pin1_bonded = False

                if is_found_pin2 is False:
                    ext_clk1_info_list = pll.find_external_clock_info(pll.RefClockIDType.ext_clock_1,
                                                                      checker.plldev_service, lvds_reg, gpio_reg,
                                                                      device_db)

                    if ext_clk1_info_list:
                        is_missing = True
                    else:
                        ext_pin2_bonded = False

                if is_missing:
                    # Both needs to be specified in dynamic
                    self.error('Bonded external reference clock pin has to be specified in dynamic mode')
                    return False

            if ext_pin1_bonded:
                index_list.append(2)
            if ext_pin2_bonded:
                index_list.append(3)

        # For some it would be an LVDS instance
        for index in index_list:
            gpio_res_list = checker.plldev_service.get_all_resource_on_ref_clk_pin(
                pll.pll_def, index)

            # If it is a list of multiple items, it means that
            # either one is required
            count, non_global, invalid_gpio = checker.check_depended_resources(
                gpio_reg, lvds_reg, gpio_res_list, GPIOInput.ConnType.pll_clkin_conn,
                LVDSRx.ConnType.pll_clkin_conn)

            # By right there should only be one driver
            if count > 1:
                self.error("There can only be one configured resource for CLKIN[{}]".format(index))
                valid = False
                break

            elif count == 0:
                if non_global:
                    # If there's a GPIO resource but it wasn't set as alternate
                    self.error('External reference clock resource {} is not ' \
                        'configured as pll_clkin connection'.format(invalid_gpio))
                    valid = False
                    break

                else:
                    self.error("The resource for CLKIN[{}] is not configured".format(index))
                    valid = False
                    break

        return valid

    def check_core(self, pll, checker):
        util.gen_util.mark_unused(checker)

        if pll.ref_clock_mode == pll.RefClockModeType.core:
            # When in core mode, it is expected that it is hardcoded to refclk0
            if pll.ref_clock_name == "":
                self.error('Core refclk pin has to be specified in core mode')
                return False

        elif pll.ref_clock_name == "" or pll.ref_clock1_name == "":
            # Both should be specified
            self.error('Both core refclk pins have to be specified in dynamic mode')
            return False

        return True

    def check(self, checker, design_block: PLLAdvance):
        pll = design_block
        valid = True

        if pll.ref_clock_mode == pll.RefClockModeType.core or\
                pll.ref_clock_mode == pll.RefClockModeType.dynamic:
            valid = self.check_core(pll, checker)

        # Check next one only if core is valid
        if pll.ref_clock_mode == pll.RefClockModeType.external or \
                pll.ref_clock_mode == pll.RefClockModeType.dynamic:
            if valid:
                self.check_external(checker, pll)


class PLLRuleFeedback(base_rule.Rule):
    """
    Check the validity of the feedback clock in core and external modes
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_feedback_clock"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore
        util.gen_util.mark_unused(checker)

        pll = design_block

        if pll.fb_mode == pll.FeedbackModeType.core or\
                pll.fb_mode == pll.FeedbackModeType.external or \
                pll.fb_mode == pll.FeedbackModeType.local:

            if self.check_name_empty_or_invalid(pll.fb_clock_name,
                'Feedback clock name is required with non-internal feedback'
            ):
                return

            fb_clk_data = checker.reg.find_output_clock(pll.fb_clock_name)

            if fb_clk_data is not None:
                fb_pll_ins, fb_out_no = fb_clk_data

                # It should be of the same instance
                # TODO: remove this if we support clkout from other PLL
                if pll != fb_pll_ins:
                    self.error('Feedback clock name {} is not from the same PLL'.format(pll.fb_clock_name))
                    return

                # Get the output clock
                out_clk = fb_pll_ins.get_output_clock_by_number(fb_out_no)

                if pll.fb_mode == pll.FeedbackModeType.local:
                    if fb_out_no != 0:
                        self.error('Feedback clock in local mode has to connect to output clock 0')
                        return

                # Check the values:
                # DIVX * DIVO * DIVQFB <= 255
                fb_div = pll.multiplier * pll.post_divider * out_clk.out_divider
                if fb_div > 255:
                    self.error('Total non-internal feedback division factor {} is out of range. Max=255'
                                .format(fb_div))
                    return

                # Also check that the multiplier value is <= 128 (PT-463)
                if pll.multiplier > 128:
                    self.error('Non-internal feedback multiplier {} is out of range. Max=128'
                                .format(pll.multiplier))
                    return

                if out_clk.phase_shift != 0:
                    self.info('Feedback clock phase shift is not 0-degree, check that the feedback clock is in-phase with the reference clock.')
                    return

            else:
                self.error('Feedback clock {} is not connected to pll clkout'
                            .format(pll.fb_clock_name))
                return

class PLLRulePhaseShiftPostDiv(base_rule.Rule):
    """
    PT-1288: Check that Post Divider is 1 when Phase Shift is not 0
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_phase_shift_post_divider"

    def check(self, checker, design_block: PLLAdvance):
        util.gen_util.mark_unused(checker)

        pll = design_block

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        found_non_zero_pshift = False

        for clk in out_clock_list:
            # PT-1288: Check that if the phase shift is non-zero, then
            # Post divider is 0
            if clk is not None and clk.phase_shift != 0:
                found_non_zero_pshift = True
                break

        if found_non_zero_pshift:
            if pll.post_divider == 1:
                self.warning('Post-divider should be greater than 1 when '\
                        'there is an output clock with non-zero phase shift')

class PLLRulePhaseShiftDiv(base_rule.Rule):
    """
    Check that Output Divider is 2 when Phase Shift is not 0
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_phase_shift"

    def check(self, checker, design_block: PLLAdvance):
        util.gen_util.mark_unused(checker)

        pll = design_block

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        for clk in out_clock_list:
            # Check that if the phase shift is non-zero, then
            # DIVQ == 2 (change in PT-463)
            if clk.phase_shift != 0:
                if clk.phase_shift == 45 or clk.phase_shift == 135:
                    if clk.out_divider != 4:
                        self.error("Output divider of clock {} has to be 4 when phase shift is 45 or 135." \
                            .format(clk.name))

                # PT-836: allow output divider to be 2 or 4 when phase shift is
                # 90
                elif clk.phase_shift == 90:
                    # PT-1695: Support outdiv = 2,4,6 for 90 degree
                    if clk.out_divider not in (2, 4, 6):
                        self.error("Output divider of clock {} has to be either 2, 4 or 6 when phase shift is 90." \
                            .format(clk.name))

                elif clk.out_divider != 2:
                    self.error("Output divider of clock {} has to be 2 when phase shift is any of this values: 180,270." \
                        .format(clk.name))


class PLLRuleVCOFreq(base_rule.Rule):
    """
    Check if VCO frequency is valid.
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_vco_freq"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore[override]

        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        # Yasmin:
        # Calling this function for AN20 ensure feedback clock is determine correctly for calc.
        # Long term, we should not re-calc the frequency in UI (unless user edit), rule and writer
        # but instead read the value directly from design.
        # The frequency in design is pre-calculated during design construction and updated when user edit.
        # I don't actually have time to vigorously test that so for now,
        # recalculate.
        is_pass_calc = checker.reg.calc_single_instance_frequency(pll)
        if is_pass_calc:
            min_freq, max_freq = checker.plldev_service.get_min_max_vco_freq(
                checker.design, pll)
            if pll.vco_freq < min_freq or pll.vco_freq > max_freq:
                self.error("VCO frequency is out of range. Freq={} Min={}MHz Max={}MHz." \
                    .format(pll.get_vco_freq_str(), min_freq, max_freq))
        else:
            # If user see this, it means we have coding error not config error
            checker.logger.warning(
                "{}: Fail to calculate VCO frequency in order to check for valid range.".format(self.name))
            self.severity = self.SeverityType.unknown


class PLLRuleOutputClockFreq(base_rule.Rule):
    """
    Check that the output clock frequency is within valid range.
    The difference between this and the normal PLL is the dependency
    with DDR instance. If the output clock is driving DDR, the
    check should not be executed on the max freq.
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_output_freq"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore

        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        min_out, max_out = checker.plldev_service.get_min_max_output_clock_freq()

        for clk in out_clock_list:
            if clk.out_clock_freq is not None:
                # Check if the range exists
                if min_out is not None:
                    if clk.out_clock_freq < min_out:
                        self.error()
                        if max_out is not None:
                            self.msg = "Output frequency {}MHz is out of range. Min={}MHz Max={}MHz." \
                                .format(clk.get_output_freq_str(), min_out, max_out)
                        else:
                            self.msg = "Output frequency {}MHz is out of range. Min={}MHz." \
                                .format(clk.get_output_freq_str(), min_out)
                        return

                # Skip max check if this PLL is driving DDR and it is clkout 0
                # Changed due to PT-710 which allows other clkout1,2 to be used
                # for other purpose.
                is_driving_ddr, ddr_inst = checker.check_outclk_driving_ddr(
                    pll)

                if not is_driving_ddr or \
                        (is_driving_ddr and clk.number != 0):

                    if max_out is not None:
                        if clk.out_clock_freq > max_out:
                            self.error()
                            if min_out is not None:
                                self.msg = "Output frequency {}MHz is out of range. Min={}MHz Max={}MHz." \
                                    .format(clk.get_output_freq_str(), min_out, max_out)
                            else:
                                self.msg = "Output frequency {}MHz is out of range. Max={}MHz." \
                                    .format(clk.get_output_freq_str(), max_out)
                            return

                # PT-907: There is a max limit if driving the DDR, which is DDR
                # speedgrade/2
                elif is_driving_ddr and clk.number == 0 and ddr_inst is not None:
                    # Get the DDR instance and its speedgrade value (without
                    # the letters)
                    if ddr_inst.config is not None:
                        speedbin = ddr_inst.config.cs_speedbin

                        if speedbin != "":
                            match_obj = re.match(r'^([0-9.]+)$', speedbin)
                            max_freq = None

                            if match_obj:
                                max_freq = float(match_obj.group(1)) / 2
                            else:
                                # Get the numbers in front if it contains
                                # letter
                                match_obj = re.match(
                                    r'^([0-9.]+)[A-Za-z]+$', speedbin)
                                if match_obj:
                                    speed_value = match_obj.group(1)
                                    max_freq = float(speed_value) / 2

                            # Only check if we can get a number from the
                            # speedgrade
                            if max_freq is not None:
                                # Add 1MHz tolerance so that it isn't too
                                # strict
                                if clk.out_clock_freq >= max_freq + 1:
                                    self.error()
                                    if min_out is not None:
                                        self.msg = "Output frequency {}MHz driving DDR is out of range. Min={}MHz Max={}MHz." \
                                            .format(clk.get_output_freq_str(), min_out, max_freq)
                                    else:
                                        self.msg = "Output frequency {}MHz driving DDR is out of range. Max={}MHz." \
                                            .format(clk.get_output_freq_str(), max_freq)
                                    return


class PLLRuleFeedbackResource(base_rule.Rule):
    """
    Check if External IO feedback is used, the resource is valid.
    It could either be a GPIO or LVDS
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_extfb_resource"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore

        pll = design_block

        # fb_clock_name nonempty is checked in the PLLRuleFeedback
        if pll.fb_mode != pll.FeedbackModeType.external or\
                pll.fb_clock_name == "":
            return

        # Check that the right resource is set to alternate connection
        # type.
        gpio_reg = checker.design.gpio_reg
        if gpio_reg is None:
            checker.logger.warning(
                "{}: gpio reg is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        lvds_reg = checker.design.lvds_reg

        # Find the resource
        assert checker.plldev_service is not None
        iofb_res_list = checker.plldev_service.get_resource_on_io_fb_pin(
            pll.pll_def)
        if iofb_res_list:

            # Check if it is an lvds resource
            # Depending on the registry type, we are going to call different
            # GPIO class name
            if isinstance(gpio_reg, GPIORegistryComplex):
                count, non_global, invalid_gpio = checker.check_depended_resources(
                    gpio_reg, lvds_reg, iofb_res_list, GPIOInputComplex.ConnType.pll_extfb_conn,
                    LVDSRxAdvance.ConnType.pll_extfb_conn)

            else:
                count, non_global, invalid_gpio = checker.check_depended_resources(
                    gpio_reg, lvds_reg, iofb_res_list, GPIOInput.ConnType.pll_extfb_conn,
                    LVDSRx.ConnType.pll_extfb_conn)

            # By right there should only be one driver
            if count > 1:
                self.error("There can only be one configured resource for external IO feedback")

            elif count == 0:
                if non_global:
                    # If there's a GPIO resource but it wasn't set as alternate
                    self.error('External IO feedback resource {} is not ' \
                        'configured as pll_extfb connection.'.format(invalid_gpio))

                else:
                    self.error("The resource for external IO feedback is not configured.")

        else:
            # This is when user set it to extfb but we don't find any bonded out
            # resources
            self.error("External IO feedback resource is unbonded")

# TODO: Enable this check when we support cascade feature


class PLLRuleCascadeOutput(base_rule.Rule):
    """
    Check that Output cascade is set correctly
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_cascade_output"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore
        pll = design_block

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        cascade_count = 0
        for clk in out_clock_list:
            if clk.is_cascade:
                cascade_count += 1

        assert checker.plldev_service is not None
        max_count = checker.plldev_service.get_max_output_clock()

        if cascade_count >= max_count:
            self.error("All output clock cannot be cascaded")

class PLLRuleNonInternalFeedbackFreq(base_rule.Rule):
    """
    Check that the feedback frequency is within valid range
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_fb_freq"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore

        pll = design_block
        assert checker.plldev_service is not None

        # Check if the non-internal feedback mode is used
        if pll.fb_mode == pll.FeedbackModeType.local or\
                pll.fb_mode == pll.FeedbackModeType.core or\
                pll.fb_mode == pll.FeedbackModeType.external:

            min_fb, max_fb = checker.plldev_service.get_min_max_external_fb_freq(pll)
            if min_fb is not None and max_fb is not None:

                # Find the feedback clockout
                if pll.fb_clock_name != "":
                    fb_clk_data = checker.reg.find_output_clock(
                        pll.fb_clock_name)

                    if fb_clk_data is not None:
                        fb_pll_ins, fb_out_no = fb_clk_data
                        # Get the output clock
                        out_clk = fb_pll_ins.get_output_clock_by_number(
                            fb_out_no)

                        if out_clk is not None and out_clk.out_clock_freq is not None:

                            if out_clk.out_clock_freq < min_fb or out_clk.out_clock_freq > max_fb:
                                self.error("Feedback frequency {}MHz is out of range. Min={}MHz Max={}MHz." \
                                    .format(out_clk.get_output_freq_str(), min_fb, max_fb))


class PLLRuleClockSelectorPinName(base_rule.Rule):
    """
    Check if clock selector pin name is valid
    It can be empty for now
    """
    def __init__(self):
        super().__init__()
        self.name = "pll_rule_clksel_pin"

    def check(self, checker, design_block: PLLAdvance):

        util.gen_util.mark_unused(checker)

        pll = design_block

        if pll is not None and pll.ref_clock_mode == pll.RefClockModeType.dynamic:
            if not self.is_name_empty(pll.clock_sel_name) and self.is_name_pattern_invalid(pll.clock_sel_name):
                self.error("Valid characters are alphanumeric characters with dash and underscore only.")
                return


class PLLRulePostDividerMultipleOutputClock(base_rule.Rule):
    """
    PT-1695: Check post divider must be 2 or greater when having multiple output clocks
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_post_div_multi_out_clk"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore
        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        out_clock_list = pll.get_output_clock_list()
        count = len(out_clock_list)
        if out_clock_list is None or count == 0:
            # Empty output clock is trapped in another rule
            return

        if count == 1:
            # Single output clock, No checking required
            return

        if not isinstance(pll.post_divider, int):
            # Invalid value type, should be handled by another rule already
            return

        if pll.post_divider <= 1:
            self.error('Post-divider should be greater than 1 when ' \
                        'there are multiple output clocks.')

class PLLRuleNonInternalVCOLowerBound(base_rule.Rule):
    """
    Warning when VCO frequency is lower than 1.6GHZ in core / local feedback mode
    """

    def __init__(self):
        super().__init__()
        self.name = 'pll_rule_non_internal_vco_lower_bound'

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore
        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        is_pass_calc = checker.reg.calc_single_instance_frequency(pll)
        if not is_pass_calc:
            # Internal coding error
            return

        if pll.fb_mode not in (design_block.FeedbackModeType.core, design_block.FeedbackModeType.local):
            # Not in core/local mode, skip
            return

        assert pll.vco_freq is not None
        if pll.vco_freq <= 1600:
            msg = 'VCO frequency should be greater than 1.6GHz in local/core feedback mode'
            self.warning(msg)


class PLLRulePostVCOFrequency(base_rule.Rule):
    """
    Check the Post VCO Frequency (PLL frequency) is valid
    """

    def __init__(self):
        super().__init__()
        self.name = 'pll_rule_pll_freq'

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance): # type: ignore
        pll = design_block

        if checker.plldev_service is None:
            checker.logger.warning(
                "{}: device db is empty. Skipped.".format(self.name))
            self.severity = self.SeverityType.unknown
            return

        # Follow PLLRuleVCOFreq
        is_pass_calc = checker.reg.calc_single_instance_frequency(pll)
        if is_pass_calc:
            min_freq, max_freq = checker.plldev_service.get_min_max_pll_freq(checker.design, pll)

            if pll.pll_freq < min_freq or pll.pll_freq > max_freq:
                self.severity = self.SeverityType.error
                self.pass_status = False
                self.msg = f"PLL Frequency is out of range, Freq={pll.get_pll_freq_str()} Min={min_freq}MHz Max={max_freq}MHz"
        else:
            checker.logger.warning(
                "{}: Fail to calculate PLL frequency in order to check for valid range.".format(self.name))
            self.severity = self.SeverityType.unknown


class PLLRuleMultiplierLockTime(base_rule.Rule):
    """
    PT-2124 Trion PLL advance has limitation where M=1 cause longer lock time
    """

    def __init__(self):
        super().__init__()
        self.name = "pll_rule_pll_multipler_lock_time"

    def check(self, checker: PLLCheckerAdv, design_block: PLLAdvance):
        pll = design_block

        if pll.multiplier == 1:
            self.warning("M should be greater than 1 for optimized performance")

