"""

Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 26, 2019

@author: yasmin
"""

from __future__ import annotations
from typing import Optional, TYPE_CHECKING

import util.gen_util as gen_util
from util.widget_formatter import WidgetFormatter

import device.db_interface as devdb_int
from device.block_definition import Port as DevicePort

import design.db as dsg_db

import common_device.mipi.mipi_design as mipid
import an20_device.mipi.gui.presenter as prs
from an20_device.mipi.gui.config import ENABLE_TIMING

from common_gui.base_viewer import BaseViewer
from common_gui.base_viewer import BaseViewerModel


if TYPE_CHECKING:
    from common_device.mipi.timing_calculator import MIPITimingCalculator


class Viewer(BaseViewer):
    """
    Override.

    Controller for instance info table.

    For each instance, it display the info related to assigned resource.
    This include read-only info from device database.
    """

    def __init__(self, parent, mipi_config=None):
        """
        Constructor

        :param parent: MIPI main window
        :mipi_config: MIPI config page
        """
        BaseViewer.__init__(self, parent, mipi_config)

    def build(self, block_name, design):
        """
        Override
        """
        self.block_name = block_name
        self.design = design

        self.model = ViewerModel()  # Using the local model
        self.model.load(design, block_name)
        self.tv_viewer.setModel(self.model)

        formatter = WidgetFormatter()
        formatter.refresh_table_view_format(self.tv_viewer)


class ViewerModel(BaseViewerModel):
    """
    UI data model that drives the info table viewer
    """

    def __init__(self):
        """
        Constructor
        """
        super().__init__()

        self.presenter = prs.MIPIConfigUIPresenter()
        self.calculator: Optional[MIPITimingCalculator] = None

    def load(self, design, block_name):
        """
        Override.
        Load instance information from design and display in the viewer UI

        :param design: A periphery design
        :param block_name: A mipi instance name
        """

        self.design = design
        mipi_reg = self.design.get_block_reg(dsg_db.PeriDesign.BlockType.mipi)
        if mipi_reg is None:
            return

        self.calculator = mipi_reg.get_timing_calc()

        mipi = mipi_reg.get_inst_by_name(block_name)
        if mipi is not None:
            self.add_row_item("Instance Name", mipi.name)
            self.add_row_item("MIPI Resource", mipi.mipi_def)
            self.add_row_item("MIPI Resource Type", mipid.MIPI.opstype2str_map.get(mipi.ops_type, ""))

            if mipi.ops_type == mipid.MIPI.OpsType.op_tx:
                self.load_tx_info(mipi)
            elif mipi.ops_type == mipid.MIPI.OpsType.op_rx:
                self.load_rx_info(mipi)
        else:
            self.logger.warning("MIPI {} does not exists".format(block_name))

    def load_tx_info(self, mipi):
        tx_info = mipi.tx_info
        if tx_info is not None:
            self.add_row_item("PHY Bandwidth", "{}".format(tx_info.get_phy_freq_str()))
            self.add_row_item("Enable Continuous PHY Clocking",
                              "{}".format(gen_util.bool2str(tx_info.is_cont_phy_clocking)))
            self.load_tx_ref_clock_info(tx_info, mipi.mipi_def)
            self.add_row_item("Control", "", True)
            pin_list = tx_info.get_pin_by_class(DevicePort.CLASS_MIPI_CONTROL)
            self.add_simple_pin(tx_info, pin_list, False, True)

            self.add_row_item("Video", "", True)
            pin_list = tx_info.get_pin_by_class(DevicePort.CLASS_MIPI_VIDEO)
            video_pin_list = list(filter(lambda pin: not pin.type_name.startswith("ULPS"), pin_list))
            self.add_simple_pin(tx_info, video_pin_list, False, False)

            self.add_row_item("Video - ULPS Mode", "", True)
            ulps_video_pin_list = list(filter(lambda pin: pin.type_name.startswith("ULPS"), pin_list))
            self.add_simple_pin(tx_info, ulps_video_pin_list, False, False)

            self.load_phy_lane_info(tx_info, False)

            if ENABLE_TIMING:
                self.load_tx_timer_info(tx_info)

    def load_tx_ref_clock_info(self, tx_info, mipi_def):
        """
        Load MIPI Tx reference clock information

        :param mipi_def:
        :param tx_info: MIPI design tx info
        """

        def get_display_name(name):
            if name == "":
                return "Unknown"
            return name

        self.add_row_item("Reference Clock", "", True)
        self.add_row_item("Frequency", tx_info.get_ref_clock_freq_str())

        dbi = devdb_int.DeviceDBService(self.design.device_db)
        mipi_tx_dev_service = dbi.get_block_service(devdb_int.DeviceDBService.BlockType.MIPI_TX)

        res_name, inst_name, pin_name = tx_info.find_refclock_info(mipi_def, mipi_tx_dev_service,
                                                                   self.design.gpio_reg)

        self.add_row_item("Resource", get_display_name(res_name))
        self.add_row_item("Instance", get_display_name(inst_name))
        self.add_row_item("Pin Name", get_display_name(pin_name))

    def load_tx_timer_info(self, mipi_info):

        if mipi_info is None or self.calculator is None:
            return

        self.calculator.update_phy_tx_freq_code(mipi_info.phy_tx_freq_code)
        self.calculator.update_tx_clk_escape_freq(mipi_info.esc_clock_freq)

        self.add_row_item("Timing", "", True)
        self.add_row_item("Timing - Clock Timer", "", True)

        if mipi_info.t_clk_post is not None:
            self.add_row_item("T_clk_post", "{}".format(self.calculator.tclk_post[mipi_info.t_clk_post - 1][1]))
        else:
            self.add_row_item("T_clk_post", "{}".format(mipi_info.t_clk_post))

        if mipi_info.t_clk_trail is not None:
            self.add_row_item("T_clk_trail", "{}".format(self.calculator.tclk_trail[mipi_info.t_clk_trail - 1][1]))
        else:
            self.add_row_item("T_clk_trail", "{}".format(mipi_info.t_clk_trail))

        if mipi_info.t_clk_prepare is not None:
            self.add_row_item("T_clk_prepare", "{}".format(
                self.calculator.tclk_prepare[mipi_info.t_clk_prepare - 1][1]))
        else:
            self.add_row_item("T_clk_prepare", "{}".format(mipi_info.t_clk_prepare))

        if mipi_info.t_clk_zero is not None:
            self.add_row_item("T_clk_zero", "{}".format(self.calculator.tclk_zero[mipi_info.t_clk_zero - 1][1]))
        else:
            self.add_row_item("T_clk_zero", "{}".format(mipi_info.t_clk_zero))

        self.add_row_item("Escape Clock Frequency", "{}".format(mipi_info.get_esc_clock_freq_str()))

        if mipi_info.t_clk_pre is not None:
            self.add_row_item("T_clk_pre", "{}".format(self.calculator.tclk_pre[mipi_info.t_clk_pre - 1][1]))
        else:
            self.add_row_item("T_clk_pre", "{}".format(mipi_info.t_clk_pre))

        self.add_row_item("Timing - Data Timer", "", True)

        if mipi_info.t_hs_prepare is not None:
            self.add_row_item("T_hs_prepare", "{}".format(self.calculator.ths_prepare[mipi_info.t_hs_prepare - 1][1]))
        else:
            self.add_row_item("T_hs_prepare", "{}".format(mipi_info.t_hs_prepare))

        if mipi_info.t_hs_zero is not None:
            self.add_row_item("T_hs_zero", "{}".format(self.calculator.ths_zero[mipi_info.t_hs_zero - 1][1]))
        else:
            self.add_row_item("T_hs_zero", "{}".format(mipi_info.t_hs_zero))

        if mipi_info.t_hs_trail is not None:
            self.add_row_item("T_hs_trail", "{}".format(self.calculator.ths_trail[mipi_info.t_hs_trail - 1][1]))
        else:
            self.add_row_item("T_hs_trail", "{}".format(mipi_info.t_hs_trail))

    def load_phy_lane_info(self, mipi_info, is_rx):
        """
        Load physical lane info. Pin swap is only fro mipi rx.

        :param mipi_info: mipi tx or rx info
        :param is_rx: True, if mipi rx else False
        """

        if mipi_info is None:
            return

        def get_lane_id(phy_lane):
            return phy_lane.lane_id

        phy_lane_map = mipi_info.phy_lane_map
        if phy_lane_map is not None:
            self.add_row_item("Lane Mapping", "", True)
            phy_lane_list = phy_lane_map.get_phy_lane_list()
            phy_lane_list = sorted(phy_lane_list, key=get_lane_id)
            for lane in phy_lane_list:
                if is_rx:
                    phy_lane_name = "{}".format(mipid.PhyLane.rx_phylane2pad_map.get(lane.lane_id))
                else:
                    phy_lane_name = "{}".format(mipid.PhyLane.tx_phylane2pad_map.get(lane.lane_id))

                prop_name = "{} - Lane".format(phy_lane_name)
                self.add_row_item(prop_name, mipid.PhyLane.loglane2str_map[lane.logical_lane_id])
                if is_rx:
                    prop_name = "{} - P&N Pin Swap".format(phy_lane_name)
                    self.add_row_item(prop_name, gen_util.bool2str(lane.is_pn_swap))

    def load_rx_info(self, mipi):
        """
        Load mipi rx info

        :param mipi: mipi instance
        """
        rx_info = mipi.rx_info
        if rx_info is not None:
            self.add_row_item("Control", "", True)
            pin_list = rx_info.get_pin_by_class(DevicePort.CLASS_MIPI_CONTROL)
            self.add_simple_pin(rx_info, pin_list, True, True)

            self.add_row_item("Video", "", True)
            pin_list = rx_info.get_pin_by_class(DevicePort.CLASS_MIPI_VIDEO)
            self.add_simple_pin(rx_info, pin_list, True, False)

            self.add_row_item("Status", "", True)
            self.add_row_item("Enable Status", "{}".format(gen_util.bool2str(rx_info.is_status_en)))
            pin_list = rx_info.get_pin_by_class(DevicePort.CLASS_MIPI_STATUS)
            self.add_simple_pin(rx_info, pin_list, True, False)

            self.load_phy_lane_info(rx_info, True)

            if ENABLE_TIMING:
                self.load_rx_timer_info(rx_info)

    def load_rx_timer_info(self, mipi_info):

        if mipi_info is None or self.calculator is None:
            return

        self.calculator.update_rx_clk_calib_freq(mipi_info.dphy_calib_clock_freq)

        self.add_row_item("Timing", "", True)
        self.add_row_item("DPHY Calibration Clock Frequency", mipi_info.get_dphy_calib_clock_freq_str())
        self.add_row_item("Timing - Clock Timer", "", True)

        if mipi_info.t_clk_settle is not None:
            self.add_row_item("T_clk_settle", "{}".format(self.calculator.tclk_settle[mipi_info.t_clk_settle - 1]))
        else:
            self.add_row_item("T_clk_settle", "{}".format(mipi_info.t_clk_settle))

        self.add_row_item("Timing - Data Timer", "", True)

        if mipi_info.t_hs_settle is not None:
            self.add_row_item("T_hs_settle", "{}".format(self.calculator.ths_settle[mipi_info.t_hs_settle - 1]))
        else:
            self.add_row_item("T_hs_settle", "{}".format(mipi_info.t_hs_settle))

    def add_simple_pin(self, mipi_info, pin_list, is_rx, is_control):
        """
        Add simple pin property display based on the input pin list.

        :param is_control: True, control pins else others
        :param is_rx: True, mipi rx else tx
        :param mipi_info: Target instance
        :param pin_list: The instance pin list

        .. note::
           By default the pins are sorted based on property name using alphabetical order.
           Except for control pins which have specific sequence defined.
           The sorting scheme must matched with the scheme applied in editor.
        """

        is_sorted = False
        sorted_pin_list = None
        if is_rx:
            if is_control:
                sorted_pin_list = \
                    sorted(pin_list,
                           key=lambda this_pin: (self.presenter.rx_control_pin_seqno.get(this_pin.type_name, 0)))
                is_sorted = True
        else:
            if is_control:
                sorted_pin_list = \
                    sorted(pin_list,
                           key=lambda this_pin: (self.presenter.tx_control_pin_seqno.get(this_pin.type_name, 0)))
                is_sorted = True

        if not is_sorted:
            # Default sorting by name
            sorted_pin_list = sorted(pin_list,
                                     key=lambda this_pin: mipi_info.get_pin_property_name(this_pin.type_name))

        if sorted_pin_list is not None:
            for pin in sorted_pin_list:
                prop_name = mipi_info.get_pin_property_name(pin.type_name)
                self.add_row_item(prop_name, pin.name)

                # Clock pin is only in control and it can be inverted
                if is_control and mipi_info.is_pin_type_clock(pin.type_name):
                    clock_name = prop_name.replace(" Pin Name", "")
                    self.add_row_item("{} Inverted".format(clock_name), gen_util.bool2str(pin.is_inverted))


if __name__ == "__main__":
    pass
