"""

Copyright (C) 2017-2018 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Nov 26, 2018

@author: yasmin
"""

import enum
import threading
from typing import Optional
from collections import OrderedDict

from PyQt5 import QtCore, QtWidgets

import design.db as dsg_db

import util.gen_util as gen_util
import util.gui_util as gui_util

from device.block_definition import Port as DevicePort

from common_device.mipi.mipi_design import MIPI, PhyLane, MIPIRx, MIPITx

from common_gui.base_config import BaseConfig
from common_gui.builder.smart_widget import GenericPinWidgetBuilder, ClockPinWidgetGroup

from an20_device.mipi.gui.presenter import MIPIConfigUIPresenter
from an20_device.mipi.gui.phy_freq_editor import PhyFreqEditor

ENABLE_TIMING = True


@gen_util.freeze_it
class MIPIConfig(BaseConfig):
    """
    Controller for specific instance configuration
    """

    class PropIndexType(enum.IntEnum):
        """
        Property stack widget index
        """
        tx_prop = 0
        rx_prop = 1

    def __init__(self, parent):
        """
        Constructor
        """

        self.sw_config = parent.sw_config

        self.phy_editor = PhyFreqEditor()
        self.pb_select_freq = parent.pb_select_freq

        # MIPI Tx
        self.tw_config_tx = parent.tw_config_tx
        self.le_tx_name = parent.le_tx_name
        self.cb_tx_device = parent.cb_tx_device
        self.lbl_tx_phy_freq = parent.lbl_tx_phy_freq
        self.lbl_tx_refclk_res = parent.lbl_tx_refclk_res
        self.lbl_tx_refclk_gpio_inst = parent.lbl_tx_refclk_gpio_inst
        self.cb_tx_cont_clocking = parent.cb_tx_cont_clocking
        self.cb_tx_refclk_freq = parent.cb_tx_refclk_freq
        self.tx_tab_base = parent.tx_tab_base
        self.tx_tab_control = parent.tx_tab_control
        self.tx_vl_tab_control = parent.tx_vl_tab_control
        self.tx_tab_video = parent.tx_tab_video
        self.tx_vl_tab_video = parent.tx_vl_tab_video
        self.tx_tab_video_ulps = parent.tx_tab_video_ulps
        self.tx_vl_tab_video_ulps = parent.tx_vl_tab_video_ulps
        self.lbl_tx_tmg_phy_freq = parent.lbl_tx_tmg_phy_freq
        self.dsb_tx_esc_clk_freq = parent.dsb_tx_esc_clk_freq

        self.tx_tab_timing = parent.tx_tab_timing
        self.cb_tx_clk_post = parent.cb_tx_clk_post
        self.cb_tx_clk_trail = parent.cb_tx_clk_trail
        self.cb_tx_clk_prepare = parent.cb_tx_clk_prepare
        self.cb_tx_clk_zero = parent.cb_tx_clk_zero
        self.cb_tx_clk_pre = parent.cb_tx_clk_pre
        self.cb_tx_hs_prepare = parent.cb_tx_hs_prepare
        self.cb_tx_hs_zero = parent.cb_tx_hs_zero
        self.cb_tx_hs_trail = parent.cb_tx_hs_trail

        # Sample object name in UI, cb_tx_phy0_log_lane
        self.cb_tx_phy_log_lane = OrderedDict()
        self.cb_tx_phy_log_lane[parent.cb_tx_phy0_log_lane.objectName()] = parent.cb_tx_phy0_log_lane
        self.cb_tx_phy_log_lane[parent.cb_tx_phy1_log_lane.objectName()] = parent.cb_tx_phy1_log_lane
        self.cb_tx_phy_log_lane[parent.cb_tx_phy2_log_lane.objectName()] = parent.cb_tx_phy2_log_lane
        self.cb_tx_phy_log_lane[parent.cb_tx_phy3_log_lane.objectName()] = parent.cb_tx_phy3_log_lane
        self.cb_tx_phy_log_lane[parent.cb_tx_phy4_log_lane.objectName()] = parent.cb_tx_phy4_log_lane
        self.tx_pin_widget_map = None
        self.is_build_tx_gen_pin = False

        # MIPI Rx
        self.tw_config_rx = parent.tw_config_rx
        self.le_rx_name = parent.le_rx_name
        self.cb_rx_device = parent.cb_rx_device
        self.rx_tab_base = parent.rx_tab_base
        self.rx_tab_control = parent.rx_tab_control
        self.rx_vl_tab_control = parent.rx_vl_tab_control
        self.rx_tab_video = parent.rx_tab_video
        self.rx_vl_tab_video = parent.rx_vl_tab_video
        self.rx_tab_status = parent.rx_tab_status
        self.rx_vl_tab_status = parent.rx_vl_tab_status
        self.dsb_rx_calib_clock_freq = parent.dsb_rx_calib_clock_freq

        self.rx_tab_timing = parent.rx_tab_timing
        self.cb_rx_en_status = parent.cb_rx_en_status
        self.cb_rx_clk_settle = parent.cb_rx_clk_settle
        self.cb_rx_hs_settle = parent.cb_rx_hs_settle

        self.cb_rx_phy_log_lane = OrderedDict()
        self.cb_rx_phy_log_lane[parent.cb_rx_phy0_log_lane.objectName()] = parent.cb_rx_phy0_log_lane
        self.cb_rx_phy_log_lane[parent.cb_rx_phy1_log_lane.objectName()] = parent.cb_rx_phy1_log_lane
        self.cb_rx_phy_log_lane[parent.cb_rx_phy2_log_lane.objectName()] = parent.cb_rx_phy2_log_lane
        self.cb_rx_phy_log_lane[parent.cb_rx_phy3_log_lane.objectName()] = parent.cb_rx_phy3_log_lane
        self.cb_rx_phy_log_lane[parent.cb_rx_phy4_log_lane.objectName()] = parent.cb_rx_phy4_log_lane

        self.cb_rx_phy_pn_swap = OrderedDict()
        self.cb_rx_phy_pn_swap[parent.cb_rx_phy0_pn_swap.objectName()] = parent.cb_rx_phy0_pn_swap
        self.cb_rx_phy_pn_swap[parent.cb_rx_phy1_pn_swap.objectName()] = parent.cb_rx_phy1_pn_swap
        self.cb_rx_phy_pn_swap[parent.cb_rx_phy2_pn_swap.objectName()] = parent.cb_rx_phy2_pn_swap
        self.cb_rx_phy_pn_swap[parent.cb_rx_phy3_pn_swap.objectName()] = parent.cb_rx_phy3_pn_swap
        self.cb_rx_phy_pn_swap[parent.cb_rx_phy4_pn_swap.objectName()] = parent.cb_rx_phy4_pn_swap

        self.rx_pin_widget_map = None
        self.is_build_rx_gen_pin = False
        self.rx_status_pin_widget_map = None

        self._write_lock = threading.RLock()  #: To control edit operation which is not reentrant.

        self.block_inst: MIPI
        BaseConfig.__init__(self, parent)

        self.presenter = MIPIConfigUIPresenter()

    def disable_mouse_wheel_scroll(self):
        """
        For each QComboBox and QSpinBox widgets apply event filter
        to prevent auto-scroll.
        """
        self.cb_tx_device.installEventFilter(self)
        self.cb_tx_refclk_freq.installEventFilter(self)

        for lane_widget in self.cb_tx_phy_log_lane.values():
            lane_widget.installEventFilter(self)

        for lane_widget in self.cb_rx_phy_log_lane.values():
            lane_widget.installEventFilter(self)

    def build_tx_simple_pin_widgets(self, mipi):
        """
        Build widgets for simple pin configuration for MIPI Tx. This need to be called only once.
        Repeated showing of the editor just requires data loading from MIPI design.
        MIPI instance is needed to get a list of generic pin definition which is common across all MIPI.

        :param mipi: MIPI instance
        """

        if self.is_build_tx_gen_pin:
            return

        tx_info = mipi.tx_info

        pin_builder = GenericPinWidgetBuilder()

        #
        # Control pin
        #
        pin_builder.clear_pin()

        # Build simple pin widget in UI
        control_pin_list = tx_info.get_pin_by_class(DevicePort.CLASS_MIPI_CONTROL)

        for pin in control_pin_list:
            seq_no = self.presenter.tx_control_pin_seqno.get(pin.type_name, 0)

            # Only control has clock pins
            if tx_info.is_pin_type_clock(pin.type_name):
                pin_builder.add_clock_pin(pin.type_name, tx_info.get_pin_property_name(pin.type_name), seq_no)
            else:
                pin_builder.add_pin(pin.type_name, tx_info.get_pin_property_name(pin.type_name), seq_no)

        # Tab in QTabWidget already has a layout
        control_pin_widget_map = pin_builder.build(self.tx_vl_tab_control, is_sort=True, is_sort_seqno=True)

        #
        # Video pin
        #
        pin_builder.clear_pin()

        # Build simple pin widget in UI
        video_pin_list = tx_info.get_pin_by_class(DevicePort.CLASS_MIPI_VIDEO)
        ulps_video_pin_list = []
        for pin in video_pin_list:
            if pin.type_name.startswith("ULPS"):
                ulps_video_pin_list.append(pin)
            else:
                pin_builder.add_pin(pin.type_name, tx_info.get_pin_property_name(pin.type_name))

        # Tab in QTabWidget already has a layout
        video_pin_widget_map = pin_builder.build(self.tx_vl_tab_video, is_sort=True, is_sort_seqno=False)

        #
        # ULPS Video pin
        #
        pin_builder.clear_pin()

        for pin in ulps_video_pin_list:
            pin_builder.add_pin(pin.type_name, tx_info.get_pin_property_name(pin.type_name))

        # Tab in QTabWidget already has a layout
        ulps_video_pin_widget_map = pin_builder.build(self.tx_vl_tab_video_ulps, is_sort=True, is_sort_seqno=False)

        # Merge all pin maps so that we can do dynamic monitoring
        self.tx_pin_widget_map = {**control_pin_widget_map, **video_pin_widget_map, **ulps_video_pin_widget_map}

        self.is_build_tx_gen_pin = True

    def build_rx_simple_pin_widgets(self, mipi):
        """
        Build widgets for simple pin configuration for MIPI Rx. This need to be called only once.
        Repeated showing of the editor just requires data loading from MIPI design.
        MIPI instance is needed to get a list of generic pin definition which is common across all MIPI.

        :param mipi: MIPI instance
        """

        if self.is_build_rx_gen_pin:
            return

        rx_info = mipi.rx_info

        pin_builder = GenericPinWidgetBuilder()

        #
        # Control pin
        #
        pin_builder.clear_pin()

        # Build simple pin widget in UI
        control_pin_list = rx_info.get_pin_by_class(DevicePort.CLASS_MIPI_CONTROL)
        for pin in control_pin_list:
            seq_no = self.presenter.rx_control_pin_seqno.get(pin.type_name, 0)

            # Only control has clock pins
            if rx_info.is_pin_type_clock(pin.type_name):
                pin_builder.add_clock_pin(pin.type_name, rx_info.get_pin_property_name(pin.type_name), seq_no)
            else:
                pin_builder.add_pin(pin.type_name, rx_info.get_pin_property_name(pin.type_name), seq_no)

        # Tab in QTabWidget already has a layout
        control_pin_widget_map = pin_builder.build(self.rx_vl_tab_control, is_sort=True, is_sort_seqno=True)

        #
        # Video pin
        #
        pin_builder.clear_pin()

        # Build simple pin widget in UI
        video_pin_list = rx_info.get_pin_by_class(DevicePort.CLASS_MIPI_VIDEO)
        for pin in video_pin_list:
            pin_builder.add_pin(pin.type_name, rx_info.get_pin_property_name(pin.type_name))

        # Tab in QTabWidget already has a layout
        video_pin_widget_map = pin_builder.build(self.rx_vl_tab_video, is_sort=True, is_sort_seqno=False)

        #
        # Status pin
        #
        pin_builder.clear_pin()

        # Build simple pin widget in UI
        status_pin_list = rx_info.get_pin_by_class(DevicePort.CLASS_MIPI_STATUS)
        for pin in status_pin_list:
            pin_builder.add_pin(pin.type_name, rx_info.get_pin_property_name(pin.type_name))

        # Tab in QTabWidget already has a layout
        self.rx_status_pin_widget_map = pin_builder.build(self.rx_vl_tab_status, is_sort=True, is_sort_seqno=False)

        # Merge all pin maps so that we can do dynamic monitoring
        self.rx_pin_widget_map = {**control_pin_widget_map, **video_pin_widget_map, **self.rx_status_pin_widget_map}

        self.is_build_rx_gen_pin = True

    def build(self, block_name, design):
        """
        Update editor content based on currently selected block instance
        """
        super().build(block_name, design)

        if not ENABLE_TIMING:
            self.tw_config_tx.removeTab(5)
            self.tw_config_rx.removeTab(5)

        self.block_reg = self.design.get_block_reg(dsg_db.PeriDesign.BlockType.mipi)
        self.presenter.setup_design(self.design)

        if self.is_monitor is True:
            self.stop_monitor_changes()

        mipi = self.block_reg.get_inst_by_name(block_name)
        if mipi is not None:
            self.block_inst = mipi

            self.phy_editor.build(mipi)

            if mipi.ops_type == MIPI.OpsType.op_tx:
                self.sw_config.setCurrentIndex(self.PropIndexType.tx_prop)
                self.build_tx_simple_pin_widgets(mipi)
                self.presenter.load_base_property(mipi, self.le_tx_name, self.cb_tx_device)
                self.presenter.load_tx_property(mipi, self.lbl_tx_phy_freq, self.cb_tx_cont_clocking,
                                                self.dsb_tx_esc_clk_freq,
                                                self.lbl_tx_tmg_phy_freq)
                self.presenter.load_tx_clock_timer_property(mipi.tx_info, self.cb_tx_clk_post, self.cb_tx_clk_trail,
                                                            self.cb_tx_clk_prepare, self.cb_tx_clk_zero,
                                                            self.cb_tx_clk_pre)
                self.presenter.load_tx_data_timer_property(mipi.tx_info, self.cb_tx_hs_prepare, self.cb_tx_hs_zero,
                                                           self.cb_tx_hs_trail)
                self.presenter.load_tx_refclk_property(mipi, self.cb_tx_refclk_freq,
                                                       self.lbl_tx_refclk_res, self.lbl_tx_refclk_gpio_inst)
                self.presenter.load_tx_phy_lane_property(self.block_inst.tx_info.phy_lane_map,
                                                         list(self.cb_tx_phy_log_lane.values()))

                # Load pins, display pin name from design into pin widget
                tx_info = self.block_inst.tx_info
                all_pin_list = tx_info.gen_pin.get_all_pin()
                for pin in all_pin_list:
                    pin_widget = self.tx_pin_widget_map.get(pin.type_name, None)
                    if pin_widget is not None:
                        gui_util.set_config_widget(pin_widget.le_pin, pin.name, pin.name)
                        if tx_info.is_pin_type_clock(pin.type_name):
                            gui_util.set_config_widget(pin_widget.cb_is_inverted, pin.is_inverted, "")

            else:
                self.sw_config.setCurrentIndex(self.PropIndexType.rx_prop)
                self.build_rx_simple_pin_widgets(mipi)
                self.presenter.load_base_property(mipi, self.le_rx_name, self.cb_rx_device)
                self.presenter.load_rx_property(mipi, self.dsb_rx_calib_clock_freq)
                self.presenter.load_rx_timer_property(mipi.rx_info, self.cb_rx_clk_settle, self.cb_rx_hs_settle)
                self.presenter.load_rx_phy_lane_property(self.block_inst.rx_info.phy_lane_map,
                                                         list(self.cb_rx_phy_log_lane.values()),
                                                         list(self.cb_rx_phy_pn_swap.values()))

                # Load pins, display pin name from design into pin widget
                rx_info = self.block_inst.rx_info
                all_pin_list = rx_info.gen_pin.get_all_pin()
                for pin in all_pin_list:
                    pin_widget = self.rx_pin_widget_map.get(pin.type_name, None)
                    if pin_widget is not None:
                        gui_util.set_config_widget(pin_widget.le_pin, pin.name, pin.name)
                        if rx_info.is_pin_type_clock(pin.type_name):
                            gui_util.set_config_widget(pin_widget.cb_is_inverted, pin.is_inverted, "")

                is_enable_status = mipi.rx_info.is_status_pin_configured()
                self.cb_rx_en_status.setChecked(is_enable_status)
                self.load_status_pin(mipi.rx_info, is_enable_status)

            gui_util.set_child_combo_spinbox_filter(self)
            # To avoid initial changes notification, monitor after setup
            # finished
            self.monitor_changes()

    def monitor_changes(self):
        """
        Start monitoring widgets for user actions
        """
        if not self.is_monitor:
            if self.block_inst.ops_type == MIPI.OpsType.op_tx:
                le_name = self.le_tx_name
                cb_device = self.cb_tx_device
                self._monitor_simple_pin_widgets(self.tx_pin_widget_map)
                self.cb_tx_refclk_freq.currentIndexChanged.connect(self.on_tx_refclk_freq_changed)
                self.cb_tx_cont_clocking.stateChanged.connect(self.on_tx_cont_phy_clocking_changed)
                self.pb_select_freq.clicked.connect(self.on_show_phy_editor)
                self.tw_config_tx.currentChanged.connect(self.on_tx_tab_changed)
                self.dsb_tx_esc_clk_freq.valueChanged.connect(self.on_tx_esc_clk_freq_changed)

                self._monitor_tx_timer_widgets()

                for lane_widget in self.cb_tx_phy_log_lane.values():
                    lane_widget.currentIndexChanged.connect(self.on_phylane_log_map_changed)

            else:
                le_name = self.le_rx_name
                cb_device = self.cb_rx_device
                self._monitor_simple_pin_widgets(self.rx_pin_widget_map)
                self.cb_rx_en_status.stateChanged.connect(self.on_rx_enable_status_changed)
                self.dsb_rx_calib_clock_freq.valueChanged.connect(self.on_rx_calib_clk_freq_changed)

                self._monitor_rx_timer_widgets()

                for lane_widget in self.cb_rx_phy_log_lane.values():
                    lane_widget.currentIndexChanged.connect(self.on_phylane_log_map_changed)

                for lane_widget in self.cb_rx_phy_pn_swap.values():
                    lane_widget.stateChanged.connect(self.on_phylane_pn_swap_changed)

            le_name.editingFinished.connect(self.on_name_changed)
            cb_device.currentIndexChanged.connect(self.on_device_changed)

            self.is_monitor = True

    def stop_monitor_changes(self):
        """
        Stop monitoring widgets
        """
        if self.is_monitor:

            if self.block_inst.ops_type == MIPI.OpsType.op_tx:
                le_name = self.le_tx_name
                cb_device = self.cb_tx_device
                self._stop_monitoring_simple_pin_widgets(self.tx_pin_widget_map)
                self.cb_tx_refclk_freq.currentIndexChanged.disconnect(self.on_tx_refclk_freq_changed)
                self.cb_tx_cont_clocking.stateChanged.disconnect(self.on_tx_cont_phy_clocking_changed)
                self.pb_select_freq.clicked.disconnect(self.on_show_phy_editor)
                self.tw_config_tx.currentChanged.disconnect(self.on_tx_tab_changed)
                self.dsb_tx_esc_clk_freq.valueChanged.disconnect(self.on_tx_esc_clk_freq_changed)

                self._stop_monitor_tx_timer_widgets()

                for lane_widget in self.cb_tx_phy_log_lane.values():
                    lane_widget.currentIndexChanged.disconnect(self.on_phylane_log_map_changed)

            else:
                le_name = self.le_rx_name
                cb_device = self.cb_rx_device
                self._stop_monitoring_simple_pin_widgets(self.rx_pin_widget_map)
                self.cb_rx_en_status.stateChanged.disconnect(self.on_rx_enable_status_changed)
                self.dsb_rx_calib_clock_freq.valueChanged.disconnect(self.on_rx_calib_clk_freq_changed)

                self._stop_monitor_rx_timer_widgets()

                for lane_widget in self.cb_rx_phy_log_lane.values():
                    lane_widget.currentIndexChanged.disconnect(self.on_phylane_log_map_changed)

                for lane_widget in self.cb_rx_phy_pn_swap.values():
                    lane_widget.stateChanged.disconnect(self.on_phylane_pn_swap_changed)

            le_name.editingFinished.disconnect(self.on_name_changed)
            cb_device.currentIndexChanged.disconnect(self.on_device_changed)

            self.is_monitor = False

    def _monitor_simple_pin_widgets(self, pin_widget_map):
        if pin_widget_map is not None:
            for pin in pin_widget_map.values():
                pin_widget = pin.le_pin
                pin_widget.editingFinished.connect(self.on_simple_pin_name_changed)

                if isinstance(pin, ClockPinWidgetGroup):
                    pin_widget = pin.cb_is_inverted
                    pin_widget.stateChanged.connect(self.on_clock_pin_invert_changed)

    def _stop_monitoring_simple_pin_widgets(self, pin_widget_map):
        if pin_widget_map is not None:
            for pin in pin_widget_map.values():
                pin_widget = pin.le_pin
                pin_widget.editingFinished.disconnect(self.on_simple_pin_name_changed)

                if isinstance(pin, ClockPinWidgetGroup):
                    pin_widget = pin.cb_is_inverted
                    pin_widget.stateChanged.disconnect(self.on_clock_pin_invert_changed)

    def _monitor_tx_timer_widgets(self):

        self.cb_tx_clk_post.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_clk_post.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

        self.cb_tx_clk_trail.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_clk_trail.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

        self.cb_tx_clk_prepare.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_clk_prepare.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

        self.cb_tx_clk_zero.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_clk_zero.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

        self.cb_tx_clk_pre.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_clk_pre.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

        self.cb_tx_hs_prepare.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_hs_prepare.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

        self.cb_tx_hs_zero.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_hs_zero.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

        self.cb_tx_hs_trail.currentIndexChanged.connect(self.on_tx_timer_changed)
        self.cb_tx_hs_trail.lineEdit().editingFinished.connect(self.on_tx_timer_edited)

    def _stop_monitor_tx_timer_widgets(self):

        self.cb_tx_clk_post.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_clk_post.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

        self.cb_tx_clk_trail.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_clk_trail.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

        self.cb_tx_clk_prepare.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_clk_prepare.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

        self.cb_tx_clk_zero.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_clk_zero.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

        self.cb_tx_clk_pre.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_clk_pre.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

        self.cb_tx_hs_prepare.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_hs_prepare.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

        self.cb_tx_hs_zero.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_hs_zero.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

        self.cb_tx_hs_trail.currentIndexChanged.disconnect(self.on_tx_timer_changed)
        self.cb_tx_hs_trail.lineEdit().editingFinished.disconnect(self.on_tx_timer_edited)

    def _monitor_rx_timer_widgets(self):

        self.cb_rx_clk_settle.currentIndexChanged.connect(self.on_rx_timer_changed)
        self.cb_rx_clk_settle.lineEdit().editingFinished.connect(self.on_rx_timer_edited)

        self.cb_rx_hs_settle.currentIndexChanged.connect(self.on_rx_timer_changed)
        self.cb_rx_hs_settle.lineEdit().editingFinished.connect(self.on_rx_timer_edited)

    def _stop_monitor_rx_timer_widgets(self):

        self.cb_rx_clk_settle.currentIndexChanged.disconnect(self.on_rx_timer_changed)
        self.cb_rx_clk_settle.lineEdit().editingFinished.disconnect(self.on_rx_timer_edited)

        self.cb_rx_hs_settle.currentIndexChanged.disconnect(self.on_rx_timer_changed)
        self.cb_rx_hs_settle.lineEdit().editingFinished.disconnect(self.on_rx_timer_edited)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(int)
    def on_tx_tab_changed(self, tab_index):

        # If change to timing tab, refresh the phy freq display
        if tab_index == 5:
            self.lbl_tx_tmg_phy_freq.setText(self.block_inst.tx_info.get_phy_freq_str())

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_device_changed(self):

        if self.block_inst is None:
            return

        if self.block_inst.ops_type == MIPI.OpsType.op_tx:
            cb_device = self.cb_tx_device
        else:
            cb_device = self.cb_rx_device

        new_device = cb_device.currentText()
        if new_device != "":
            if new_device == self.block_inst.mipi_def:
                return

            if new_device == "None":
                self.block_reg.reset_inst_device(self.block_inst)
                if self.block_inst.ops_type == MIPI.OpsType.op_tx:
                    self.lbl_tx_refclk_gpio_inst.setText("unknown")
                    self.lbl_tx_refclk_res.setText("unknown")
            else:
                if self.block_reg.is_device_used(new_device):
                    msg = "MIPI resource {} is already used. Please choose another.".format(new_device)
                    QtWidgets.QMessageBox.warning(self.parent, "Update mipi resource", msg)
                    self.logger.error("Fail to assign resource, it has been used")

                    # Revert to original name before edit
                    if self.block_inst.mipi_def == "":
                        cb_device.setCurrentText("None")
                    else:
                        cb_device.setCurrentText(self.block_inst.mipi_def)
                    return

                self.block_reg.assign_inst_device(self.block_inst, new_device)
                self.presenter.load_tx_refclk_property(self.block_inst, None,
                                                       self.lbl_tx_refclk_res, self.lbl_tx_refclk_gpio_inst)

            self.sig_prop_updated.emit()
            self.sig_resource_updated.emit(new_device)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_simple_pin_name_changed(self):
        """
        Generic change handler for simple pins
        """

        if self.block_inst is None:
            return

        pin_widget = self.sender()
        if pin_widget is not None:
            name = pin_widget.text()
            pin_type_name = pin_widget.objectName()
            self.logger.info("Pin widget sender : " + pin_type_name)

            # Remove the le_ prefix
            pin_type_name = pin_type_name[3:]

            if self.block_inst.ops_type == MIPI.OpsType.op_tx:
                mipi_info = self.block_inst.tx_info
            else:
                mipi_info = self.block_inst.rx_info

            gen_pin = mipi_info.gen_pin
            pin_design = gen_pin.get_pin_by_type_name(pin_type_name)
            if pin_design is not None:
                pin_design.name = name
                self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(int)
    def on_clock_pin_invert_changed(self, state):
        """
        Generic change handler for clock pin invert status
        """

        gen_util.mark_unused(state)

        if self.block_inst is None:
            return

        pin_widget = self.sender()
        if pin_widget is not None:

            is_check = pin_widget.isChecked()
            pin_type_name = pin_widget.objectName()
            self.logger.info("Pin widget sender : " + pin_type_name)

            # Remove the cb_ prefix
            pin_type_name = pin_type_name[3:]

            if self.block_inst.ops_type == MIPI.OpsType.op_tx:
                mipi_info = self.block_inst.tx_info
            else:
                mipi_info = self.block_inst.rx_info

            gen_pin = mipi_info.gen_pin
            pin_design = gen_pin.get_pin_by_type_name(pin_type_name)
            if pin_design is not None:
                pin_design.is_inverted = is_check
                self.sig_prop_updated.emit()

    def sync_pin_display(self):
        """
        Refresh any widget that identifies pin name with design data
        """
        if self.block_inst.ops_type == MIPI.OpsType.op_tx:
            mipi_info = self.block_inst.tx_info
            pin_widget_map = self.tx_pin_widget_map
        else:
            mipi_info = self.block_inst.rx_info
            pin_widget_map = self.rx_pin_widget_map

        if mipi_info is None or pin_widget_map is None:
            return

        all_pin_list = mipi_info.gen_pin.get_all_pin()
        for pin in all_pin_list:
            pin_widget = pin_widget_map.get(pin.type_name, None)
            if pin_widget is not None:
                gui_util.set_config_widget(pin_widget.le_pin, pin.name, pin.name)

    def get_inst_resource(self):
        """
        Get resource name of instance
        """
        return self.block_inst.mipi_def

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_tx_refclk_freq_changed(self):

        if self.block_inst is None or self.block_inst.tx_info is None:
            return

        index = self.cb_tx_refclk_freq.currentIndex()
        if index != -1:
            self.presenter.save_tx_refclk_freq(self.block_inst, self.cb_tx_refclk_freq)
            self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_phylane_log_map_changed(self):
        """
        Handles changes on logical lane setting.

        This is a generic function for all logical lane changes. It relies on object name
        to know which lane is being changed.
        """

        if self.block_inst is None:
            return

        log_lane_widget = self.sender()
        if log_lane_widget is not None:
            obj_name = log_lane_widget.objectName()
            self.logger.info("Logical lane widget sender : " + obj_name)

            is_tx = True
            phy_lane = None
            if self.block_inst.ops_type == MIPI.OpsType.op_tx:
                phy_lane_map = self.block_inst.tx_info.phy_lane_map

                if "tx_phy0" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane0)
                elif "tx_phy1" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane1)
                elif "tx_phy2" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane2)
                elif "tx_phy3" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane3)
                elif "tx_phy4" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane4)

            else:
                is_tx = False
                phy_lane_map = self.block_inst.rx_info.phy_lane_map

                if "rx_phy0" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane0)
                elif "rx_phy1" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane1)
                elif "rx_phy2" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane2)
                elif "rx_phy3" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane3)
                elif "rx_phy4" in obj_name:
                    phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane4)

            if phy_lane is not None:
                index = log_lane_widget.currentIndex()
                if index != -1:
                    log_lane_str = log_lane_widget.currentText()
                    log_lane_id = PhyLane.str2loglane_map.get(log_lane_str, PhyLane.LogLaneIdType.unused)
                    if log_lane_id != PhyLane.LogLaneIdType.unused:
                        search_phy_lane = phy_lane_map.get_phy_lane_by_log_lane(log_lane_id)
                        if search_phy_lane is not None:

                            # Swap in design
                            phy_lane_map.swap_lane_mapping(phy_lane.lane_id, search_phy_lane.lane_id)

                            # Refresh the whole map
                            self.refresh_lane_mapping_view(is_tx)

                        else:
                            self.presenter.save_phylane_log_map(phy_lane, log_lane_widget)

                    self.sig_prop_updated.emit()

    def refresh_lane_mapping_view(self, is_tx):
        """
        Reload lane mapping from design db

        :param is_tx: True, mipi tx else mipi rx

        ..  note::
            Need to disable the widget monitoring before reload to avoid infinite recursion
            when the widget is changed through code (rather than user action).
        """

        if is_tx:
            for lane_widget in self.cb_tx_phy_log_lane.values():
                lane_widget.currentIndexChanged.disconnect(self.on_phylane_log_map_changed)

            self.presenter.load_tx_phy_lane_property(self.block_inst.tx_info.phy_lane_map,
                                                     list(self.cb_tx_phy_log_lane.values()))

            for lane_widget in self.cb_tx_phy_log_lane.values():
                lane_widget.currentIndexChanged.connect(self.on_phylane_log_map_changed)

        else:
            for lane_widget in self.cb_rx_phy_log_lane.values():
                lane_widget.currentIndexChanged.disconnect(self.on_phylane_log_map_changed)

            self.presenter.load_rx_phy_lane_property(self.block_inst.rx_info.phy_lane_map,
                                                     list(self.cb_rx_phy_log_lane.values()),
                                                     list(self.cb_rx_phy_pn_swap.values()))

            for lane_widget in self.cb_rx_phy_log_lane.values():
                lane_widget.currentIndexChanged.connect(self.on_phylane_log_map_changed)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(int)
    def on_tx_cont_phy_clocking_changed(self, state):

        if self.block_inst is None:
            return

        tx_info = self.block_inst.tx_info
        if tx_info is None:
            return

        if state == QtCore.Qt.Checked:
            is_checked = True
        else:
            is_checked = False

        tx_info.is_cont_phy_clocking = is_checked
        self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(int)
    def on_phylane_pn_swap_changed(self, state):
        """
        Handles changes on pn pin swap setting.

        This is a generic function for all lane pin swap. It relies on object name
        to know which lane is being changed.

        :param state: P&N Pin Swap check/uncheck state
        """

        if self.block_inst is None:
            return

        if state == QtCore.Qt.Checked:
            is_checked = True
        else:
            is_checked = False

        rx_info = self.block_inst.rx_info
        if rx_info is None:
            return

        pn_swap_widget = self.sender()
        if pn_swap_widget is not None:
            obj_name = pn_swap_widget.objectName()
            self.logger.info("P&N Swap widget sender : " + obj_name)

            phy_lane_map = rx_info.phy_lane_map
            phy_lane = None
            if "phy0" in obj_name:
                phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane0)
            elif "phy1" in obj_name:
                phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane1)
            elif "phy2" in obj_name:
                phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane2)
            elif "phy3" in obj_name:
                phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane3)
            elif "phy4" in obj_name:
                phy_lane = phy_lane_map.get_phy_lane(PhyLane.PhyLaneIdType.lane4)

            if phy_lane is not None:
                phy_lane.is_pn_swap = is_checked
                self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(bool)
    def on_show_phy_editor(self, is_clicked):
        gen_util.mark_unused(is_clicked)

        if self.block_inst is None:
            return

        tx_info = self.block_inst.tx_info
        if tx_info is None:
            return

        self.phy_editor.show()

        result = self.phy_editor.get_selected_freq()
        if result is None:
            return

        freq_str, freq_code = result
        if tx_info.phy_tx_freq_code != freq_code:
            self.lbl_tx_phy_freq.setText(freq_str)
            tx_info.phy_tx_freq_code = freq_code
            self.presenter.save_phy_tx_freq_code(freq_code)
            self.presenter.load_tx_clock_timer_property(tx_info, self.cb_tx_clk_post, self.cb_tx_clk_trail,
                                                        self.cb_tx_clk_prepare, self.cb_tx_clk_zero, self.cb_tx_clk_pre)
            self.presenter.load_tx_data_timer_property(tx_info, self.cb_tx_hs_prepare, self.cb_tx_hs_zero,
                                                       self.cb_tx_hs_trail)
            self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(int)
    def on_rx_enable_status_changed(self, state):

        if self.block_inst is None:
            return

        rx_info = self.block_inst.rx_info
        if rx_info is None:
            return

        self.set_rx_enable_status(rx_info, state == QtCore.Qt.Checked)

        self.sig_prop_updated.emit()

    def set_rx_enable_status(self, rx_info: Optional[MIPIRx], is_enable):
        """
        Enable/disable Status pin without broadcast.

        When enable, enable all status pin widgets and auto-gen the pin name.
        When disable, clear all status pin name, its widget and disabled it.

        :param rx_info: rx info
        :param is_enable: True, enable, else False
        """

        if rx_info is None:
            return

        rx_info.is_status_en = is_enable

        if is_enable:
            rx_info.generate_status_pin_name_from_inst(self.block_inst.name)
        else:
            rx_info.clear_all_status_pin()

        self.load_status_pin(rx_info, is_enable)

    def load_status_pin(self, rx_info: MIPIRx, is_enable: bool):
        if self.rx_status_pin_widget_map is None:
            return

        for pin_widget in self.rx_status_pin_widget_map.values():
            pin_name = ""

            if is_enable:
                pin_type_name = pin_widget.get_pin_type_name()
                pin_design = rx_info.gen_pin.get_pin_by_type_name(pin_type_name)
                assert pin_design is not None
                pin_name = pin_design.name

            pin_widget.set_current_data(pin_name)
            pin_widget.set_enable(is_enable)

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(float)
    def on_tx_esc_clk_freq_changed(self, value):

        gen_util.mark_unused(value)

        if self.block_inst is None or self.block_inst.tx_info is None:
            return

        self.presenter.save_tx_esc_clock_freq(self.block_inst, self.dsb_tx_esc_clk_freq)
        self.presenter.refresh_tx_clk_pre(self.block_inst.tx_info, self.cb_tx_clk_pre)
        self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot(float)
    def on_rx_calib_clk_freq_changed(self, value):

        gen_util.mark_unused(value)

        if self.block_inst is None or self.block_inst.rx_info is None:
            return

        self.presenter.save_rx_calib_clock_freq(self.block_inst, self.dsb_rx_calib_clock_freq)
        self.presenter.load_rx_timer_property(self.block_inst.rx_info, self.cb_rx_clk_settle, self.cb_rx_hs_settle)
        self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_tx_timer_changed(self):
        """
        Generic change handler for Tx timer
        """

        if self.block_inst is None:
            return

        with self._write_lock:
            timer_widget = self.sender()
            if timer_widget is not None:
                timer_str = timer_widget.currentText()
                timer_type_name = timer_widget.objectName()
                self.logger.debug("Tx Timer widget sender : " + timer_type_name)

                # Remove the cb_tx_ prefix
                timer_type_name = timer_type_name[6:]

                timer_type = None
                if "clk_post" in timer_type_name:
                    timer_type = MIPITx.TimerType.clk_post
                elif "clk_prepare" in timer_type_name:
                    timer_type = MIPITx.TimerType.clk_prepare
                elif "clk_pre" in timer_type_name:
                    timer_type = MIPITx.TimerType.clk_pre
                elif "clk_trail" in timer_type_name:
                    timer_type = MIPITx.TimerType.clk_trail
                elif "clk_zero" in timer_type_name:
                    timer_type = MIPITx.TimerType.clk_zero
                elif "hs_prepare" in timer_type_name:
                    timer_type = MIPITx.TimerType.hs_prepare
                elif "hs_trail" in timer_type_name:
                    timer_type = MIPITx.TimerType.hs_trail
                elif "hs_zero" in timer_type_name:
                    timer_type = MIPITx.TimerType.hs_zero

                if timer_type is not None:
                    self.presenter.save_tx_timer(self.block_inst, timer_type, timer_str)
                    self.sig_prop_updated.emit()

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_tx_timer_edited(self):
        """
        Generic change handler for handling invalid user input for Tx timer.
        It just reverts to first valid value.
        """

        if self.block_inst is None:
            return

        with self._write_lock:
            timer_widget = self.sender()
            if timer_widget is not None:
                parent_widget = timer_widget.parent()
                timer_str = timer_widget.text()
                timer_type_name = parent_widget.objectName()
                self.logger.debug("Tx Timer parent widget sender : " + timer_type_name)

                # Remove the cb_tx_ prefix
                timer_type_name = timer_type_name[6:]

                if "clk_post" in timer_type_name:
                    if self.cb_tx_clk_post.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_clk_post.setCurrentIndex(0)

                elif "clk_prepare" in timer_type_name:
                    if self.cb_tx_clk_prepare.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_clk_prepare.setCurrentIndex(0)

                elif "clk_pre" in timer_type_name:
                    if self.cb_tx_clk_pre.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_clk_pre.setCurrentIndex(0)

                elif "clk_trail" in timer_type_name:
                    if self.cb_tx_clk_trail.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_clk_trail.setCurrentIndex(0)

                elif "clk_zero" in timer_type_name:
                    if self.cb_tx_clk_zero.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_clk_zero.setCurrentIndex(0)

                elif "hs_prepare" in timer_type_name:
                    if self.cb_tx_hs_prepare.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_hs_prepare.setCurrentIndex(0)

                elif "hs_trail" in timer_type_name:
                    if self.cb_tx_hs_trail.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_hs_trail.setCurrentIndex(0)

                elif "hs_zero" in timer_type_name:
                    if self.cb_tx_hs_zero.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_tx_hs_zero.setCurrentIndex(0)

    @QtCore.pyqtSlot()
    def on_rx_timer_changed(self):
        """
        Generic change handler for Rx timer
        """

        if self.block_inst is None:
            return

        with self._write_lock:
            timer_widget = self.sender()
            if timer_widget is not None:
                timer_str = timer_widget.currentText()
                timer_type_name = timer_widget.objectName()
                self.logger.debug("Rx Timer widget sender : " + timer_type_name)

                # Remove the cb_rx_ prefix
                timer_type_name = timer_type_name[6:]

                timer_type = None
                if "clk_settle" == timer_type_name:
                    timer_type = MIPIRx.TimerType.clk_settle
                elif "hs_settle" == timer_type_name:
                    timer_type = MIPIRx.TimerType.hs_settle

                if timer_type is not None:
                    self.presenter.save_rx_timer(self.block_inst, timer_type, timer_str)
                    self.sig_prop_updated.emit()

    @QtCore.pyqtSlot()
    def on_rx_timer_edited(self):
        """
        Generic change handler for handling invalid user input for Rx timer.
        It just reverts to first valid value.
        """

        if self.block_inst is None:
            return

        with self._write_lock:

            timer_widget = self.sender()
            if timer_widget is not None:
                parent_widget = timer_widget.parent()
                timer_str = timer_widget.text()
                timer_type_name = parent_widget.objectName()
                self.logger.debug("Rx Timer parent widget sender : " + timer_type_name)

                # Remove the cb_rx_ prefix
                timer_type_name = timer_type_name[6:]

                if "clk_settle" == timer_type_name:
                    if self.cb_rx_clk_settle.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_rx_clk_settle.setCurrentIndex(0)

                elif "hs_settle" == timer_type_name:
                    if self.cb_rx_hs_settle.findText(timer_str, QtCore.Qt.MatchExactly) == -1:
                        self.cb_rx_hs_settle.setCurrentIndex(0)


if __name__ == "__main__":
    pass
