"""
Copyright (C) 2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on Apr 17, 2020

@author: dklim
"""

import os
import sys
from cmd2 import with_argparser, argparse_completer, with_category, Cmd2ArgumentParser
from argparse import ArgumentParser
import colorama
from io import StringIO
from pathlib import Path

from array import array as Array
from typing import List
from binascii import hexlify
from pyftdi.spi import SpiController
from pyftdi.bits import BitSequence
from pyftdi.ftdi import Ftdi, FtdiError
from pyftdi.jtag import JtagEngine, JtagController
from pyftdi.gpio import GpioController
from pyftdi.usbtools import UsbToolsError

from spiflash.serialflash import SerialFlashManager
from spiflash.serialflash import SerialFlashUnknownJedec
from spiflash.serialflash import _SpiFlashDevice, _Gen25FlashDevice

from base import BaseExt
from utils import Utils
import shared

sys.path.append(os.path.join(os.environ['EFINITY_HOME'], 'pgm', 'bin'))

from efx_pgm.console import Console
from efx_dbg.flash import FlashJtagManager, FlashLController

import efx_pgm.efinix_flash as efinix_flash
import efx_pgm.ftdi_program as ftdi_program
import efx_pgm.programmer as programmer
from efx_pgm.efx_hw_common.boards import EfxHwBoardProfile, EfxHwBoardProfileSelector, PinoutProfile
from efx_pgm.efx_hw_common.manager import EfxFTDIHwConnectionManager
from efx_pgm.usb_resolver import UsbResolver

class ExtFtdi(BaseExt):
    # find_devices
    find_devices_parser = ArgumentParser(
        prog='find_devices',
        description='Find USB target')

    @with_argparser(find_devices_parser)
    def do_find_devices(self, arg):
        shared.resolved_usb = FlashJtagManager.list_usb_devices()

        result = [(x[4], x[2]) for x in shared.resolved_usb]

        for i in range(len(result)):
            print(colorama.Fore.GREEN + str(i +1) + ". " + str(result[i]))

    def sernum_completer(text: str, line: str, begidx: int, endidx: int) -> List[str]:
        if (shared.resolved_usb == None or len(shared.resolved_usb) == 0):
            shared.resolved_usb = FlashJtagManager.list_usb_devices()

        if (shared.resolved_usb == None or len(shared.resolved_usb) == 0):
            match_against = []
        else:
            match_against = [x[2] for x in shared.resolved_usb]
        
        return cmd2.Cmd.basic_complete(text, line, begidx, endidx, match_against)

    # config
    config_parser = Cmd2ArgumentParser(
        prog='config',
        description='Configure the SPI/JTAG spec on a USB device')

    config_parser.add_argument(
        'sernum',
        completer=sernum_completer,
        help='USB device\'s serial number. Supports tab-completion.')

    config_parser.add_argument('-p',
                        type=int,
                        choices=[1, 2],
                        default=1,
                        help='Port number to use on the USB device.')

    config_parser.add_argument('-t',
                        type=str,
                        choices=['spi','jtag'],
                        default='spi',
                        help='Specify type of conneciton.')

    config_parser.add_argument('-f',
                        type=int,
                        default=3E06,
                        help='Frequency, in Python format. Default: 4E06')

    @with_argparser(config_parser)
    def do_config(self, arg):
        if (shared.resolved_usb == None or len(shared.resolved_usb) == 0):
            shared.resolved_usb = FlashJtagManager.list_usb_devices()

            if (shared.resolved_usb == None or len(shared.resolved_usb) == 0):
                print(colorama.Fore.RED + "No devices found!")
                return
                
        try:
            if arg.t == 'spi':
                self.unconfig()
                self.spi_configuration(arg.sernum ,arg.p)
            elif arg.t =='jtag':
                self.unconfig()
                self.jtag_configuration(arg.sernum, arg.p)
        except Exception as e:
            print(e)

    # unconfig
    unconfig_parser = ArgumentParser(
        prog='unconfig',
        description='Close the configured SPI/JTAG port')

    @with_argparser(unconfig_parser)
    def do_unconfig(self, arg):
        self.unconfig()


    #auto_config
    auto_config_parser = ArgumentParser(
        prog='auto_config',
        description='Auto-config USB device, will select the first device.')

    auto_config_parser.add_argument('-p',
                        type=int,
                        choices=[1, 2],
                        default=1,
                        help='Port number to use on the USB device.')

    auto_config_parser.add_argument('-t',
                        type=str,
                        choices=['spi','jtag'],
                        default='spi',
                        help='Specify type of conneciton.')

    auto_config_parser.add_argument('-d',
                        type=int,
                        choices=range(1, 20),
                        metavar="{1-20}",
                        default=1,
                        help='Specify which device to use. Default is 1. Use \'find_devices\' command to see device number.')

    @with_argparser(auto_config_parser)
    def do_auto_config(self, arg):
        try:
            fnameobj = programmer.FilenameObj("")
            shared.prog = ftdi_program.FtdiProgram(args = fnameobj, console=Utils.OutputConsole())
            
            resolver = UsbResolver(console=None, mixed_backend=True)
            resolved_usb = resolver.get_usb_connections()
            if (resolved_usb == None or len(resolved_usb) == 0):
                print(colorama.Fore.RED + "No devices found!")
                return
            # sort resolved_usb by  serialnumber(_sn) in usb object
            #resolved_usb.sort(key=lambda x: x._sn)

            prog_usb_target = resolved_usb[arg.d - 1]
            url_list = prog_usb_target.URLS
            prog_urls = [None, None, None, None] # Hard-code it to be 4 that support FT4232
            if len(url_list) == 1:
                prog_urls[0] = url_list[0]
                prog_urls[1] = url_list[0]
            elif len(url_list) >= 2:
                # hard-coding this for now, per Kee-Chiew's spec 10/26/2018
                prog_urls[0] = url_list[0]
                prog_urls[1] = url_list[1]
            
            shared.prog.urls = prog_urls

            board_profile_selector = EfxHwBoardProfileSelector([ Path(os.environ["EFXPGM_HOME"], 'bin', 'efx_pgm', 'efx_hw_common', 'boards')])
            best_profile = board_profile_selector.get_best_profile(prog_usb_target)
            shared.prog.pinout = best_profile.pinout

            sernum = prog_usb_target._sn
            if arg.t == 'spi':
                self.unconfig()
                shared.shell.prompt = sernum + ' > '
                shared.url, cs_count, cs_port = shared.prog.get_spi_info_from_pinout()
                #print("spi_url=%s, cs_count=%s, cs_port=%s" % ( shared.url, cs_count, cs_port))


                shared.ctrl = EfxFTDIHwConnectionManager.get_controller(shared.url, EfxFTDIHwConnectionManager.ConnectionType.SPI)
                shared.ctrl.configure(shared.url, cs_count=cs_count, debug=False)
                shared.spi = shared.ctrl.get_port(cs=cs_port, freq=6E6, mode=0)

                shared.flash = shared.prog.setup_flash_device(shared.spi, shared.ctrl)
                print(colorama.Fore.GREEN + str(shared.flash))
                #with shared.prog.get_flash_context_manager(shared.spi, shared.ctrl) as shared.flash:
                #    print(colorama.Fore.GREEN + str(shared.flash))

                jedec = SerialFlashManager.read_jedec_id(shared.spi)
                manufacturer_id, device_id, capacity_id = jedec[0:3]
                print(colorama.Fore.GREEN + 'JEDEC id: 0x%02X%02X%02X' % (manufacturer_id, device_id, capacity_id))
            elif arg.t =='jtag':
                self.unconfig()
                shared.shell.prompt = sernum + ' > '
                shared.url = prog_urls[1]

                f = FlashJtagManager()
                tap = f.detect_jtag_tap(url=shared.url)
                f.config_jtag(url=shared.url, freq=6E06, tap=tap)
                configured_list = f.view_configured_jtag()
                url = next(iter(configured_list))
                io = StringIO("""\n{\n    "debug_cores": [\n        {\n            "name": "FL_0",
                        "type": "fl"\n        }\n    ]\n}\n""")
                f.add_debug_profile(io, url)
                f.select_core(url, 'FL_0')

                obj = f._get_debugger_obj(shared.url, 'FL_0', FlashLController)
                obj.console = Utils.OutputConsole()
                shared.jtag_flash_ctrl = obj
                shared.jtag = f

                obj.handshake()
                obj.request_access()
                obj.release_power_down()
                manufacturing_id, device_id, capacity_id = obj.read_manufacturer_id()
                jedec = Array('B', (manufacturing_id, device_id, capacity_id))
                print('JEDEC id: 0x%02X%02X%02X' % (manufacturing_id, device_id, capacity_id))
                #self.jtag_configuration(sernum, arg.p)
        except Exception as e:
            print(e)

    def find_urls_and_profile(self, sernum, port):
        try:
            resolver = UsbResolver(console=None, mixed_backend=True)
            resolved_usb = resolver.get_usb_connections()
            if (resolved_usb == None or len(resolved_usb) == 0):
                print(colorama.Fore.RED + "No devices found!")
                raise Exception("No devices found!")

            targets = list(filter(lambda d: d.serial_number == sernum, resolved_usb))

            if( len(targets) <= 0 ):
                raise Exception("No matching device!")

            prog_usb_target = targets[0]    # Select the first matching device

            url_list = prog_usb_target.URLS
            prog_urls = [None, None, None, None] # Hard-code it to be 4 that support FT4232
            if len(url_list) == 1:
                prog_urls[0] = url_list[0]
                prog_urls[1] = url_list[0]
            elif len(url_list) >= 2:
                # hard-coding this for now, per Kee-Chiew's spec 10/26/2018
                prog_urls[0] = url_list[0]
                prog_urls[1] = url_list[1]
            
            #shared.prog.urls = prog_urls

            board_profile_selector = EfxHwBoardProfileSelector([ Path(os.environ["EFXPGM_HOME"], 'bin', 'efx_pgm', 'efx_hw_common', 'boards')])
            best_profile = board_profile_selector.get_best_profile(prog_usb_target)
            #shared.prog.pinout = best_profile.pinout

            return prog_urls, best_profile

        except Exception as e:
            print(e)

    def spi_configuration(self, sernum, port):
        devices = [((v, p), s) for v, p, s, _, _ in shared.resolved_usb]
        dev_vpd, dev_sernum = list(zip(*devices)) if devices else ([], [])
        if sernum not in dev_sernum:
            msg = 'config: Unrecognized USB device s/n "{}".'.format(sernum)
            raise UnknownUsbSernumError(msg)
        vid, pid = dev_vpd[dev_sernum.index(sernum)]
        shared.url = 'ftdi://0x{:04x}:0x{:04x}:{}/{}'.format(vid, pid, sernum, port)

        shared.shell.prompt = sernum + ' > '

        shared.ctrl = SpiController()
        shared.ctrl.configure(shared.url, cs_count=1, debug=False)
        shared.spi = shared.ctrl.get_port(cs=0, freq=6E0, mode=0)

        fnameobj = programmer.FilenameObj("")
        shared.prog = ftdi_program.FtdiProgram(args = fnameobj, console=Utils.OutputConsole())
        shared.prog.url = shared.url

        prog_urls, best_profile = self.find_urls_and_profile(sernum, port)
        shared.prog.urls = prog_urls
        shared.prog.pinout = best_profile.pinout

        shared.flash = shared.prog.setup_flash_device(shared.spi, shared.ctrl)
        num_bytes = len(shared.flash)
        print(colorama.Fore.GREEN + str(shared.flash))

        jedec = SerialFlashManager.read_jedec_id(shared.spi)
        manufacturer_id, device_id, capacity_id = jedec[0:3]
        print(colorama.Fore.GREEN + 'JEDEC id: 0x%02X%02X%02X' % (manufacturer_id, device_id, capacity_id))

    def jtag_configuration(self, sernum, port):
        shared.ctrl = None
        devices = [((v, p), s) for v, p, s, _, _ in shared.resolved_usb]
        dev_vpd, dev_sernum = list(zip(*devices)) if devices else ([], [])
        if sernum not in dev_sernum:
            msg = 'config: Unrecognized USB device s/n "{}".'.format(sernum)
            raise UnknownUsbSernumError(msg)
        vid, pid = dev_vpd[dev_sernum.index(sernum)]
        shared.url = 'ftdi://0x{:04x}:0x{:04x}:{}/{}'.format(vid, pid, sernum, port)

        shared.shell.prompt = sernum + ' > '

        fnameobj = programmer.FilenameObj("")
        shared.prog = ftdi_program.FtdiProgram(args = fnameobj, console=None)
        shared.prog.url = shared.url

        f = FlashJtagManager()
        tap = f.detect_jtag_tap(url=shared.url)
        f.config_jtag(url=shared.url, freq=6E06, tap=tap)
        io = StringIO("""\n{\n    "debug_cores": [\n        {\n            "name": "FL_0",
                "type": "fl"\n        }\n    ]\n}\n""")
        f.add_debug_profile(io, shared.url)
        f.select_core(shared.url, 'FL_0')

        obj = f._get_debugger_obj(shared.url, 'FL_0', FlashLController)
        obj.console = Utils.OutputConsole()
        shared.jtag_flash_ctrl = obj
        shared.jtag = f

        obj.handshake()
        obj.request_access()
        manufacturing_id, device_id, capacity_id = obj.read_manufacturer_id()
        jedec = Array('B', (manufacturing_id, device_id, capacity_id))
        print('JEDEC id: 0x%02X%02X%02X' % (manufacturing_id, device_id, capacity_id))  

    def unconfig(self):
        if(shared.spi):
            shared.ctrl.terminate()
            shared.spi = None
            shared.ctrl = None
        elif(shared.jtag_flash_ctrl):
            shared.jtag.unconfig_jtag(url=shared.url)
            shared.jtag_flash_ctrl._engine.close()
            shared.jtag_flash_ctrl = None

        shared.shell.prompt = '> '
