"""
Copyright (C) 2020 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Updated on Nov 20, 2020

@author: kyle

"""

import sys
import os
import csv
# import random #For testing

from PyQt5 import QtWidgets, QtCore, QtGui
from PyQt5.QtCore import Qt

from pyftdi.bits import BitSequence

EFXDBG_HOME = os.environ["EFXDBG_HOME"]
EFXPGM_HOME = os.environ["EFXPGM_HOME"]
sys.path.append(os.path.join(EFXDBG_HOME,'bin'))
sys.path.append(os.path.join(EFXPGM_HOME,'bin'))

from efx_pgm.efx_hw_common.manager import EfxFTDIHwConnectionManager

# import version
from efx_pgm._version import __version__
from efx_pgm.Ui_ti_jtag_status import Ui_Dialog as Ui_TiJtagStatusDialog
from efx_dbg.jcf import JtagChainEngine


ti60es_errors = [3,4,5,6,11,13]
ti60mp_errors = [3,5,6,11,13]


class TiJTAGStatusDialog(QtCore.QObject):

    def __init__(self, console, parent):
        """
        Constructor
        """

        super().__init__(parent)
        self.dialog = QtWidgets.QDialog(parent)
        self.ui_ti_jtag_status_dialog = Ui_TiJtagStatusDialog()
        self.dialog.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui_ti_jtag_status_dialog.setupUi(self.dialog)
        self.console = console

        #Bits to be checked for GUI
        self.check_reg = None
        #Bits read from status registers
        self.status_bits = None
        #Bits to be checked for errors
        self.errors = None
        #Bits read from USERCODE instruction
        self.usercode = None

        #Data for setting up JTAG connection, passed in from MainWindow()
        self.url = None
        self.device_id = None
        self.devices = None
        self.target_num = None
        self.jtag_engine = None

        self.is_monitor = False

        self.monitor_changes()


    def show(self, url, device_id, devices=None, target_num=1):

        self.url = url
        self.devices = devices
        self.target_num = target_num
        self.device_db = self.dialog.parentWidget().device_db
        
        #Initialize table with correct entries if device changes
        if device_id != self.device_id:
            try:
                self.init_table(device_id)
            except ValueError:
                self.console.write_error_msg('ERROR: Device is not compatible with the Advanced Device Status widget')
                return
        
        self.read_reg()
        self.update_table()
        self.show_usercode()

        self.dialog.exec()


    def monitor_changes(self):
        """
        Monitor changes in the dialog
        """
        self.ui_ti_jtag_status_dialog.pb_update.clicked.connect(self.on_update_clicked)

        self.is_monitor = True

    # noinspection PyArgumentList
    @QtCore.pyqtSlot()
    def on_update_clicked(self):
        """
        Whenever update button is clicked
        """
        
        try:
            self.read_reg()
        except Exception as exc:
            self.console.write_error_msg("ERROR: Unable to read from JTAG status registers, closing widget!")
            self.console.write_error_msg(str(exc))
            self.dialog.done(1)
        else:
            self.update_table()
            self.show_usercode()
            self.console.write("Updated advanced device status register readings")


    def update_table(self):
        """
        Update table with new values from status_bits, change corresponding colors, set user mode
        """

        #FOR TESTING
        # ~ self.status_bits = [random.randint(0,1) for i in range(14)]

        row_count = 0
        for idx in self.check_reg.keys():
            self.ui_ti_jtag_status_dialog.tw_jtag_status.item(row_count,1).setText(str(self.status_bits[int(idx)]))
            if int(idx) in self.errors and self.status_bits[int(idx)] == 1:
                self.ui_ti_jtag_status_dialog.tw_jtag_status.item(row_count,1).setBackground(QtGui.QColor(255,0,0,155))
            elif self.check_reg[idx] in ["AES256_PASS", "RSA_PASS"] and self.check_reg[idx] == 0:
                self.ui_ti_jtag_status_dialog.tw_jtag_status.item(row_count,1).setBackground(QtGui.QColor(255,0,0,155))
            else:
                self.ui_ti_jtag_status_dialog.tw_jtag_status.item(row_count,1).setBackground(QtGui.QColor(0,0,0,0))
            row_count += 1

    def init_table(self, device_id):
        """
        Populate table with appropriate entries
        """
        die_name = None
        csv_file = None
        for die in self.device_db.findall('die'):
            die_name = die.get('name')
            idcode = die.find('idcode').get('value')
            if idcode == device_id:
                csv_file = die.find('csv').get('value')
                break
        
        if csv_file == None:
            raise ValueError('Die (%s) is not supported' % die_name)
        
        #Get path for .csv file depending on device id
        entries_path = os.path.join(EFXPGM_HOME,'bin','efx_pgm',csv_file)
        if die_name == 'Ti60':
            self.errors = ti60es_errors
        else:
            self.errors = ti60mp_errors
            
        self.device_id = device_id
        
        self.check_reg = {}
        #Populate table with entries from .csv file
        #Start from empty table
        self.ui_ti_jtag_status_dialog.tw_jtag_status.clearContents()
        with open(entries_path) as entries:
            csv_reader = csv.DictReader(entries, fieldnames=["idx","name"])
            for row in csv_reader:
                self.check_reg[row["idx"]] = row["name"]

            self.ui_ti_jtag_status_dialog.tw_jtag_status.setRowCount(len(self.check_reg))
            self.ui_ti_jtag_status_dialog.tw_jtag_status.setColumnCount(2)

            row_count = 0
            for name in self.check_reg.values():
                table_entry = QtWidgets.QTableWidgetItem(name)
                self.ui_ti_jtag_status_dialog.tw_jtag_status.setItem(row_count,0,table_entry)
                self.ui_ti_jtag_status_dialog.tw_jtag_status.setItem(row_count,1,QtWidgets.QTableWidgetItem("0"))
                row_count += 1

            self.ui_ti_jtag_status_dialog.tw_jtag_status.setVerticalHeaderLabels([str(i) for i in range(len(self.check_reg))])

    def read_reg(self):
        """
        Setup JTAG interface and send in Ti instruction to scan status registers
        """
        engine_provider = EfxFTDIHwConnectionManager.get_controller(self.url, EfxFTDIHwConnectionManager.ConnectionType.JTAG)
        self.jtag_engine = JtagChainEngine(devices=self.devices, target_num=self.target_num, engine_provider=engine_provider, tap='efx_ti')
        self.jtag_engine.configure(url=self.url)
        self.jtag_engine.reset()

        instr_bits = '01100'
        bseq = BitSequence(value=instr_bits, msb=True, length=len(instr_bits))
        self.jtag_engine.write_ir(bseq)
        self.status_bits = self.jtag_engine.read_dr(32)
        
        # `define USERCODE 5'b01101
        instr_bits = '01101'
        bseq = BitSequence(value=instr_bits, msb=True, length=len(instr_bits))
        self.jtag_engine.write_ir(bseq)
        self.usercode = self.jtag_engine.read_dr(32)

        self.jtag_engine.close()

    def show_usercode(self):
        """
        Get JTAG USERCODE using instraction and display in widget
        """
        
        usercode_hex = '0x{:08X}'.format(int(self.usercode))
        self.ui_ti_jtag_status_dialog.lbl_jtag_usercode.setText('JTAG USERCODE: ' + usercode_hex)


if __name__ == "__main__":
    '''
    Read Advanced JTAG Status register from command line, no JTAG chain supported yet
    '''
    from efx_pgm.usb_resolver import UsbResolver
    
    _devices = None
    _target_num = 1

    resolver = UsbResolver()
    usb_connects = resolver.get_usb_connections()
    if usb_connects == []:
        print("No valid USB targets found")
        sys.exit(0)
    else:
        _url = usb_connects[0].URLS[1]

    _engine_provider = EfxFTDIHwConnectionManager.get_controller(_url, EfxFTDIHwConnectionManager.ConnectionType.JTAG)
    _jtag_engine = JtagChainEngine(devices=_devices, target_num=_target_num, engine_provider=_engine_provider)
    _jtag_engine.configure(url=_url)
    _jtag_engine.reset()
    
    id_length = 32
    num_devices = 1
    jtagid = super(JtagChainEngine,_jtag_engine).read_dr(id_length*num_devices) # one IDCODE is 32 bits
    offset = (num_devices-_target_num) * id_length
    single_id = jtagid[offset:offset+id_length]
    device_id = '0x{:08X}'.format(int(single_id))

    _instr_bits = '01100'
    _bseq = BitSequence(value=_instr_bits, msb=True, length=len(_instr_bits))
    _jtag_engine.write_ir(_bseq)
    _status_bits = _jtag_engine.read_dr(32)
    
    _instr_bits = '01101'
    _bseq = BitSequence(value=_instr_bits, msb=True, length=len(_instr_bits))
    _jtag_engine.write_ir(_bseq)
    _usercode = _jtag_engine.read_dr(32)

    _jtag_engine.close()

    ti60es_id = "0x00360A79"

    ti60mp_jtag_ids = [
        "0x10660A79",   # ti60 production
        "0x10661A79",   # ti35 production
        "0x1067FA79",   # ti60mut production
        "0x10668A79",   # tz50 production  
        "0x10678A79"    # TP50 production
    ]

    other_ti_jtag_ids = [
        "0x00680A79",   # ti180 rev A/A1
        "0x00684A79",   # ti180l all revs
        "0x0068CA79",   # ti120l all revs
        "0x00688A79",   # ti90l all revs
        "0x00690A79",   # ti180 rev B
        "0x00692A79",   # ti120 rev B
        "0x00691A79",   # ti90 rev B
        "0x006A0A79",   # ti375
        "0x006A0EF3",   # tj375
        "0x006A2A79",   # ti240
        "0x006A2EF3",   # tj240
        "0x006A1A79",   # ti165
        "0x006A1EF3",   # tj165
        "0x206C0A79",   # ti135es
        "0x006C0A79",   # ti135
        "0x006C0EF3",   # tj135
        "0x006C2A79",   # ti85
        "0x006C2EF3",   # tj85
        "0x00698A79",   # tz110
        "0x00699A79",   # tz170
        "0x006A8A79",   # tz200
        "0x006A9A79",   # tz325
        "0x006C8A79",   # tz75
        "0x006C9A79",   # tz100
        "0x0069CA79",   # TP110
        "0x0069DA79",   # TP170
        "0x006B8EF3",   # TP200
        "0x006B9EF3",   # TP325
        "0x006D8EF3",   # TP75
        "0x006D9EF3"    # TP100
    ]
    
    #Get path for .csv file depending on device id
    if device_id == ti60es_id:
        entries_path = os.path.join(EFXPGM_HOME,'bin','efx_pgm','ti_jtag_status_entries_ti60es.csv')
    elif device_id in ti60mp_jtag_ids:
        entries_path = os.path.join(EFXPGM_HOME,'bin','efx_pgm','ti_jtag_status_entries_ti60mp.csv')
    elif device_id in other_ti_jtag_ids:
        entries_path = os.path.join(EFXPGM_HOME,'bin','efx_pgm','ti_jtag_status_entries_ti180.csv')
    else:
        raise ValueError('Device ID (%s) is not supported' % device_id)

    _check_reg = {}
    with open(entries_path) as entries:
        _csv_reader = csv.DictReader(entries, fieldnames=["idx","name"])
        for row in _csv_reader:
            _check_reg[row["idx"]] = row["name"]

    print("\nAdvanced Device Status Registers")
    print("--------------------------------")
    for i in _check_reg.keys():
        print(_check_reg[i].ljust(20) + ":".ljust(5) + str(_status_bits[int(i)]))
    print('JTAG USERCODE: ' + '0x{:08X}'.format(int(_usercode)))
    
    print('\nSuccessfully run advanced device status!')
