import sys
import time
from array import array as Array
from typing import Tuple, Union, Any
from pyftdi.misc import pretty_size
from spiflash.serialflash import SerialFlashTimeout, SerialFlashUnknownJedec, SerialFlash, SerialFlashRequestError
from efx_pgm.jtag2SpiFlashSvf import _Gen25FlashDeviceSvf, N25QFlashDeviceSvf, W25XFlashDeviceSvf, W25QFlashDeviceSvf, Mx25lFlashDeviceSvf, M25PxFlashDeviceSvf, W25QJVFlashDeviceSvf, convertByteArray2SvfByteStr

#------------------------------------------------------------------------------------------------------
# Puyu
class P25xFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Puya P25Q flash device implementation"""
    CMD_READ_STATUS_REG_2  = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31

    JEDEC_ID = 0x85
    DEVICES = {0x60: 'P25Q', 0x40: 'P25Q', 0x42: 'P25Q'}
    SIZES = {
            0x10: 4 << 14,
            0x11: 4 << 15,
            0x12: 4 << 16,
            0x13: 4 << 17,
            0x14: 4 << 18,
            0x15: 4 << 19,
            0x16: 4 << 20,
            0x17: 4 << 21,
            0x18: 4 << 22
            }
    MEMORY_DENSITY = {
        0x60 : {
            0x10: "05",  # 512Kb
            0x11: "10",  # 1Mb
            0x12: "20",  # 2Mb
            0x13: "40",  # 4Mb
            0x14: "80",  # 8Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
        },
        0x40: {
            0x10: "06",  # 512Kb
            0x11: "11",  # 1Mb
            0x12: "21",  # 2Mb
            0x13: "40",  # 4Mb
            0x14: "80",  # 8Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
        },
        0x42: {
            0x10: "06",  # 512Kb
            0x11: "11",  # 1Mb
            0x12: "21",  # 2Mb
            0x13: "40",  # 4Mb
            0x14: "80",  # 8Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    CMD_READ_UID = 0x4B
    UID_LEN = 0x16  # 128 bits
    READ_UID_WIDTH = 4  # 4 dummy bytes
    TIMINGS = {'page': (0.002, 0.003),  # 8/20 ms       # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.008, 0.012),
               'subsector': (0.016, 0.030),  # 8/20 ms
               'sector': (0.016, 0.030),  # 8/20 ms
               'bulk': (0.016, 0.030),  # 8/20 ms
               'lock': (0.05, 0.1)}  # 50/100 ms

    FEATURES = SerialFlash.FEAT_SECTERASE | SerialFlash.FEAT_SUBSECTERASE

    def __init__(self, freq, jedec):
        super(P25xFlashDeviceSvf, self).__init__(freq=freq)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = P25xFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

        self.UNLOCK_DELAY = self.TIMINGS['lock'][1]*freq
        self.PAGE_PROGRAM_DELAY = int(self.TIMINGS['page'][0]*freq)
        self.ERASE_SECTOR_DELAY = int(self.TIMINGS['sector'][0]*freq)
        self.ERASE_SUBSECTOR_DELAY = int(self.TIMINGS['subsector'][0]*freq)
        self.WRITE_STATUS_DELAY = int(self.TIMINGS['write_status_register'][1]*freq)

    def __str__(self):
        return 'Puya %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    #def _read_status2(self, sync, f):
    #    cmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
    #    cmdstr = convertByteArray2SvfByteStr(cmd)
    #    cmdstr_bitlength = len(cmdstr)*4
    #    print('// Begin Read Status', file=f)
    #    print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
    #    print('STATE IDLE;', file=f)
    #    print('// End Read Status', file=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _write_status2(self, val, f):
        # write enable
        self._enable_write(f)

        # write status reg
        cmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b00000010, f)
        print('// End Enable QE', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], P25xFlashDeviceSvf.__name__, P25xFlashDeviceSvf)

#------------------------------------------------------------------------------------------------------
# Gigadevice GD25x
class GD25xFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Gigadevice GD25 flash device implementation"""

    CMD_READ_STATUS_REG_2  = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31

    JEDEC_ID = 0xC8
    DEVICES = {0x40: 'GD25Q', 0x65: 'GD25WQ'}
    SIZES = {
            0x12: 4 << 16,  # 2Mb
            0x14: 4 << 18,  # 8Mb
            0x15: 4 << 19,  # 16Mb
            0x16: 4 << 20,  # 32Mb
            0x17: 4 << 21,  # 64Mb
            0x18: 4 << 22,  # 128Mb
            0x19: 4 << 23   # 256Mb
            }
    MEMORY_DENSITY = {
        0x40 : {
            0x12: "2",  # 2Mb
            0x14: "8",  # 2Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
            0x19: "256", # 256Mb
        },
        0x65: {
            0x12: "2",  # 2Mb
            0x14: "8",  # 2Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
            0x19: "256", # 256Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.0006, 0.0024),  # 0.6/2.4 ms       # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.03),
               'sector': (0.045, 0.300),  # 45/300 ms
               'lock': (0.005, 0.03)}  # 5/30 ms
    FEATURES = SerialFlash.FEAT_SECTERASE

    def __init__(self, freq, jedec):
        super(GD25xFlashDeviceSvf, self).__init__(freq)
        if not GD25xFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = GD25xFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

        self.UNLOCK_DELAY = int(self.TIMINGS['lock'][1]*freq)
        self.PAGE_PROGRAM_DELAY = int(self.TIMINGS['page'][0]*freq)
        self.ERASE_SECTOR_DELAY = int(self.TIMINGS['sector'][0]*freq)
        self.WRITE_STATUS_DELAY = int(self.TIMINGS['write_status_register'][1]*freq)

    def __str__(self):
        return 'Gigadevice %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _write_status2(self, val, f):
        # write enable
        self._enable_write(f)

        # write status reg
        cmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b00000010, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], GD25xFlashDeviceSvf.__name__, GD25xFlashDeviceSvf)		# Add GD25xFlashDevice to spiflash.serialflash module

#------------------------------------------------------------------------------------------------------
# Gigadevice GD25lq
class GD25lqFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Gigadevice GD25LQ flash device implementation"""
    CMD_READ_STATUS_REG_2  = 0x35

    FEAT_WRITE_BOTH_STATUS_REG = True

    JEDEC_ID = 0xC8
    DEVICES = {0x60: 'GD25LQ'}
    SIZES = {
            0x16: 4 << 20   # 32Mb
            }
    MEMORY_DENSITY = {
        0x60 : {
            0x16: "32"   # 32Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page'     : (0.0007, 0.004),  # 0.7/4 ms      # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.035),
               'subsector': (0.09, 0.60),    # 90/600 ms      # 4k erase time
               'hsector'  : (0.30, 1.6),     # 0.30/1.6 s     # 32k erase time
               'sector'   : (0.45, 3.0),     # 0.45/3.0 s     # 64k erase time
               'lock'     : (0.005, 0.035)}  # 5/35 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(GD25lqFlashDeviceSvf, self).__init__(freq)
        if not GD25lqFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = GD25lqFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

        self.UNLOCK_DELAY = int(self.TIMINGS['lock'][1]*freq)
        self.PAGE_PROGRAM_DELAY = int(self.TIMINGS['page'][0]*freq)
        self.ERASE_HSECTOR_DELAY = int(self.TIMINGS['hsector'][0]*freq)
        self.ERASE_SECTOR_DELAY = int(self.TIMINGS['sector'][0]*freq)
        self.ERASE_SUBSECTOR_DELAY = int(self.TIMINGS['subsector'][0]*freq)
        self.WRITE_STATUS_DELAY = int(self.TIMINGS['write_status_register'][1]*freq)

    def __str__(self):
        return 'Gigadevice %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    def unlock(self, f):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        self._enable_write(f)
        cmd = Array('B', (GD25lqFlashDeviceSvf.CMD_WRSR,
                               _Gen25FlashDeviceSvf.SR_WEL |
                               _Gen25FlashDeviceSvf.SR_PROTECT_NONE |
                               _Gen25FlashDeviceSvf.SR_UNLOCK_PROTECT))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Unlock Flash', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Unlock Flash', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._read_status(expected=0, mask=_Gen25FlashDeviceSvf.SR_PROTECT_ALL, f=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def _write_status(self, val1, val2, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val1, val2))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b00000000, 0b00000010, f)
        print('// End Enable QE', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], GD25lqFlashDeviceSvf.__name__, GD25lqFlashDeviceSvf)		# Add GD25lqFlashDevice to spiflash.serialflash module

#------------------------------------------------------------------------------------------------------
# Gigadevice GD25lb
class GD25lbFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Gigadevice GD25LB flash device implementation"""
    FEAT_WRITE_BOTH_STATUS_REG = False

    JEDEC_ID = 0xC8
    DEVICES = {0x67: 'GD25LB'}
    SIZES = {
            0x16: 4 << 20,  # 32Mb
            0x1A: 4 << 24   # 512Mb
            }
    MEMORY_DENSITY = {
        0x60 : {
            0x16: "32"   # 32Mb
        },
        0x67 : {
            0x1A: "512"   # 512Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page'     : (0.0007, 0.004),  # 0.7/4 ms      # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.035),
               'subsector': (0.09, 0.60),    # 90/600 ms      # 4k erase time
               'hsector'  : (0.30, 1.6),     # 0.30/1.6 s     # 32k erase time
               'sector'   : (0.45, 3.0),     # 0.45/3.0 s     # 64k erase time
               'lock'     : (0.005, 0.035)}  # 5/35 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(GD25lbFlashDeviceSvf, self).__init__(freq)
        if not GD25lbFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = GD25lbFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

        self.UNLOCK_DELAY = int(self.TIMINGS['lock'][1]*freq)
        self.PAGE_PROGRAM_DELAY = int(self.TIMINGS['page'][0]*freq)
        self.ERASE_HSECTOR_DELAY = int(self.TIMINGS['hsector'][0]*freq)
        self.ERASE_SECTOR_DELAY = int(self.TIMINGS['sector'][0]*freq)
        self.ERASE_SUBSECTOR_DELAY = int(self.TIMINGS['subsector'][0]*freq)
        self.WRITE_STATUS_DELAY = int(self.TIMINGS['write_status_register'][1]*freq)

    def __str__(self):
        return 'Gigadevice %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    def unlock(self, f):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        self._enable_write(f)
        cmd = Array('B', (GD25lbFlashDeviceSvf.CMD_WRSR,
                               _Gen25FlashDeviceSvf.SR_WEL |
                               _Gen25FlashDeviceSvf.SR_PROTECT_NONE |
                               _Gen25FlashDeviceSvf.SR_UNLOCK_PROTECT))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Unlock Flash', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Unlock Flash', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._read_status(expected=0, mask=_Gen25FlashDeviceSvf.SR_PROTECT_ALL, f=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def _write_status(self, val1, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val1))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], GD25lbFlashDeviceSvf.__name__, GD25lbFlashDeviceSvf)		# Add GD25lbFlashDevice to spiflash.serialflash module


#------------------------------------------------------------------------------------------------------
# Macronix mx25l

# TODO: no chip on hand
class Mx25lFlashDeviceSvfExt(Mx25lFlashDeviceSvf):
    """Extend the implementation of original Mx25lFlashDevice"""

    CMD_READ_ELECTRONIC_ID = 0xAB
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    def __init__(self, freq, jedec):
        super(Mx25lFlashDeviceSvfExt, self).__init__(freq, jedec)
        #if self._spi is not None:
        # Read the RES (Electronic ID)
        #self._electronic_id = self._read_electronic_id(sync=False)
        #if(self._electronic_id == 0x14 and self._device == 'MX25L'):
        #    self._device = 'MX25V'
        #else:
        #    self._electronic_id = None

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    #def _read_electronic_id(self, sync):
    #    rcmd = Array('B', (self.CMD_READ_ELECTRONIC_ID, 0xFF, 0xFF, 0xFF))
    #    rcmd = reverseBytes(bytes(rcmd))
    #    self._jtag_engine.write_dr_bytes(rcmd)
    #    data = self._jtag_engine.read_dr2(8)
    #    self._jtag_engine.go_idle2(sync=sync)
    #    data = data.tobytes(True)

    #    if len(data) != 1:
    #        raise SerialFlashTimeout("Unable to read electronic id")
    #    return data[0]

    def enable_quad_spi(self, f):
        # MX25V doesn't support quad
        if self._device == 'MX25V':
            return
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b01000000, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], Mx25lFlashDeviceSvfExt.__name__, Mx25lFlashDeviceSvfExt)

# no chip on hand
class Mx25uFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Macronix MX25U flash device implementation"""

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    JEDEC_ID = 0xC2
    DEVICES = {0x25: 'MX25U',
               0x95: 'MX25U'}
    SIZES = {
            0x34: 4 << 18,   # 8Mb
            0x39: 32 << 20,  # 256Mb
            0x3A: 64 << 20  # 512Mb
            }
    MEMORY_DENSITY = {
        0x25 : {
            0x34: "8035",    # 8Mb
            0x39: "25645G", # 256Mb
            0x3A: "512G"  # 512Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.00015, 0.00075),  # 1.5/3 ms       # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.04, 0.04),
               'subsector': (0.025, 0.4),  # 300/300 ms
               'hsector': (0.15, 1),  # 2/2 s
               'sector': (0.22, 1.3),  # 2/2 s
               'lock': (0.0015, 0.003)}  # 1.5/3 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(Mx25uFlashDeviceSvf, self).__init__(freq)
        if not Mx25uFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = Mx25uFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

        self.UNLOCK_DELAY = int(self.TIMINGS['lock'][1]*freq)
        self.PAGE_PROGRAM_DELAY = int(self.TIMINGS['page'][0]*freq)
        self.ERASE_HSECTOR_DELAY = int(self.TIMINGS['hsector'][0]*freq)
        self.ERASE_SECTOR_DELAY = int(self.TIMINGS['sector'][0]*freq)
        self.ERASE_SUBSECTOR_DELAY = int(self.TIMINGS['subsector'][0]*freq)
        self.WRITE_STATUS_DELAY = int(self.TIMINGS['write_status_register'][1]*freq)

    def __str__(self):
        return 'Macronix %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        print('// Begin Enable QE', file=f)
        self._write_status(0b01000000, f)
        print('// End Enable QE', file=f)

    def disable_quad_spi(self, f):
        print('// Begin Disable QE', file=f)
        self._write_status(0b00000000, f)
        print('// End Disable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], Mx25uFlashDeviceSvf.__name__, Mx25uFlashDeviceSvf)

#------------------------------------------------------------------------------------------------------
# Micron
class N25QFlashDeviceSvfExt(N25QFlashDeviceSvf):
    """Extend the implementation of original N25QFlashDevice"""

    READ_NV_CONF_REG = 0xB5
    WRITE_NV_CONF_REG = 0xB1

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    # Change the name from N25Q to MT25Q
    DEVICES = {0xBA: 'MT25Q', 0xBB: 'MT25Q'}

    # Add new device capacity table
    SIZES = {
                0x15: 2 << 20,
                0x16: 4 << 20,
                0x17: 8 << 20,
                0x18: 16 << 20,
                0x19: 32 << 20,
                0x20: 64 << 20,
                0x21: 128 << 20,
                0x22: 256 << 20
            }

    TIMINGS = {'page': (0.00012, 0.005),  # 0.5/5 ms
               'write_status_register': (0.0013, 0.008),
               'subsector': (0.05, 3.0),  # 300/3000 ms
               'sector': (0.15, 3.0),  # 700/3000 ms
               'bulk': (60, 120)}  # seconds

    def __init__(self, freq, jedec):
        try:
            super(N25QFlashDeviceSvfExt, self).__init__(freq, jedec)
        # This is expected, as the base class doesn't aware of the newly added SIZES
        except SerialFlashUnknownJedec as exc:
            if not self.match(jedec):
                raise SerialFlashUnknownJedec(jedec) from exc
            device, capacity = jedec[1:]
            self._size = self.SIZES[capacity]
            self._device = self.DEVICES[device]

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _write_nv_conf_ref(self, data, f):
        # write enable
        cmd = Array('B', (self.CMD_WRITE_ENABLE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Enable', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Enable', file=f)

        # write nonvolative conf reg
        cmd = Array('B', (self.WRITE_NV_CONF_REG, data[0], data[1]))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Nonvolatile configuration Register', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Nonvolatile configuration Register', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    # Micron Flash doesn't need set anything for for Dual/Quad operation.
    # Beware the differences between dual-I/O and dual operation. Likewise for qual-I/O and quad operation.
    #def enable_quad_spi(self):
    #    nv_conf_reg = self._read_nv_conf_ref()
    #    nv_conf_reg[0] = nv_conf_reg[0] & 0b11110111    # bit-3 Quad protocol [0 = enable]
    #    nv_conf_reg[0] = nv_conf_reg[0] | 0b00000100    # Disable Dual by setting bit-2 to 1
    #    self._write_nv_conf_ref(nv_conf_reg)

    #def enable_dual_spi(self):
    #    nv_conf_reg = self._read_nv_conf_ref()
    #    nv_conf_reg[0] = nv_conf_reg[0] & 0b11111011    # bit-2 Dual protocol [0 = enable]
    #    nv_conf_reg[0] = nv_conf_reg[0] | 0b00001000    # Disable quad by setting bit-3 to 1
    #    self._write_nv_conf_ref(nv_conf_reg)

    #def is_address_mode_3bytes(self):

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], N25QFlashDeviceSvfExt.__name__, N25QFlashDeviceSvfExt)

class W25XFlashDeviceSvfExt(W25XFlashDeviceSvf):
    """Extend the implementation of original W25xFlashDevice"""

    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    def __init__(self, freq, jedec):
        super(W25XFlashDeviceSvfExt, self).__init__(freq, jedec)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _write_status2(self, val, f):
        # write enable
        self._enable_write(f)

        # write status reg
        cmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b00000010, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], W25XFlashDeviceSvfExt.__name__, W25XFlashDeviceSvfExt)

class W25QFlashDeviceSvfExt(W25QFlashDeviceSvf):
    """Extend the implementation of original W25xFlashDevice"""

    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    DEVICES = {0x40: 'W25Q', 0x60: 'W25Q', 0x80: 'W25Q'}

    def __init__(self, freq, jedec):
        super(W25QFlashDeviceSvfExt, self).__init__(freq, jedec)

    #def _read_status2(self, sync):
    #    pass

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _write_status2(self, val, f):
        # write enable
        self._enable_write(f)

        # write status reg
        cmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        print('// Begin Enable QE', file=f)
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        self._write_status(0b00000010, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], W25QFlashDeviceSvfExt.__name__, W25QFlashDeviceSvfExt)

class W25QJVFlashDeviceSvfExt(W25QJVFlashDeviceSvf):
    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    def __init__(self, freq, jedec):
        super(W25QJVFlashDeviceSvfExt, self).__init__(freq, jedec)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _write_status2(self, val, f):
        # write enable
        self._enable_write(f)

        # write status reg
        cmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b00000010, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], W25QJVFlashDeviceSvfExt.__name__, W25QJVFlashDeviceSvfExt)


class M25PxFlashDeviceSvfExt(M25PxFlashDeviceSvf):
    """Extend the implementation of original M25PxFlashDevice"""

    #JEDEC_ID = 0x20
    #DEVICES = {0x71: 'M25PX', 0x20: 'M25P'}
    #SIZES = {0x14: 1 << 20, 0x15: 2 << 20, 0x16: 4 << 20, 0x17: 8 << 20, 0x18: 16 << 20}
    #SPI_FREQ_MAX = 75  # MHz (P series only)
    #TIMINGS = {'page': (0.0015, 0.003),  # 1.5/3 ms
    #           'write_status_register': (1, 1),
    #           'subsector': (0.150, 0.150),  # 150/150 ms
    #           'sector': (0.8, 0.8),  # 0.8/3 s
    #           'bulk': (32, 64),  # seconds
    #           'lock': (0.0015, 0.003)}  # 1.5/3 ms
    #FEATURES = SerialFlash.FEAT_SECTERASE

    def __init__(self, freq, jedec):
        super(M25PxFlashDeviceSvfExt, self).__init__(freq, jedec)

    def __str__(self):
        return 'Micron %s%d %s' % \
            (self._device, len(self) >> 17,
             pretty_size(self._size, lim_m=1 << 20))

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], M25PxFlashDeviceSvfExt.__name__, M25PxFlashDeviceSvfExt)

class XT25xFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """XTX XT25F flash device implementation"""

    CMD_READ_STATUS_REG_2 = 0x35

    FEAT_WRITE_BOTH_STATUS_REG = True

    JEDEC_ID = 0x0B
    DEVICES = {0x40: 'XT25F'}
    SIZES = {
            0x14: 4 << 18,
            0x15: 2 << 20,
            }
    MEMORY_DENSITY = {
        0x40: {
            0x14: "08B",  # 8Mb
            0x15: "16B",  #16Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz
    CMD_READ_UID = 0x4B
    UID_LEN = 0x16  # 128 bits
    READ_UID_WIDTH = 4  # 4 dummy bytes
    TIMINGS = {'page': (0.0004, 0.003),  # 0.4/0.7 ms        # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.001, 0.02),
               'subsector': (0.08, 0.80),  # 70/800 ms      # 4k erase time
               'hsector': (0.15, 1.2),  # 0.15/1.2 s        # 32k erase time
               'sector': (0.25, 1.6),  # 0.25/1.6 s         # 64k erase time
               'lock': (0.07, 0.8)}  # 70/800 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(XT25xFlashDeviceSvf, self).__init__(freq)
        if not XT25xFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = XT25xFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'XTX %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))


    def unlock(self, f):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        self._enable_write(f)
        cmd = Array('B', (XT25xFlashDeviceSvf.CMD_WRSR,
                               _Gen25FlashDeviceSvf.SR_WEL |
                               _Gen25FlashDeviceSvf.SR_PROTECT_NONE |
                               _Gen25FlashDeviceSvf.SR_UNLOCK_PROTECT))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Unlock Flash', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Unlock Flash', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._read_status(expected=0, mask=_Gen25FlashDeviceSvf.SR_PROTECT_ALL, f=f)

    def _write_status(self, val1, val2, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val1, val2))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status

        print('// Begin Enable QE', file=f)
        self._write_status(0b00000000, 0b00000010, f)
        print('// End Enable QE', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], XT25xFlashDeviceSvf.__name__, XT25xFlashDeviceSvf)		# Add XT25xFlashDevice to spiflash.serialflash module

class Mx75lFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Macronix MX75L flash device implementation"""

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    JEDEC_ID = 0xC2
    DEVICES = {0x26: 'MX75L'}
    SIZES = {
            0x19: 4 << 23   # 256Mb
            }
    MEMORY_DENSITY = {
        0x26 : {
            0x19: "25690"
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.0005, 0.003),  # 0.5/3 ms         # no datasheet found, this value find thru tne
               'write_status_register': (0.04, 1),
               'subsector': (0.025, 0.300),  # 25/300 ms    # 4k erase time
               'hsector': (0.140, 1.0),  # 0.14/1 s         # 32k erase time
               'sector': (0.25, 2.0), # 0.25/2.0 s          # 64k erase time
               'lock': (0.0015, 0.04)}  # 1.5/40 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(Mx75lFlashDeviceSvf, self).__init__(freq)
        if not Mx75lFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = Mx75lFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Macronix %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    def unlock(self, f):
        self._enable_write(f)
        cmd = Array('B', (_Gen25FlashDeviceSvf.CMD_WRSR,
                               _Gen25FlashDeviceSvf.SR_WEL |
                               _Gen25FlashDeviceSvf.SR_PROTECT_NONE |
                               _Gen25FlashDeviceSvf.SR_UNLOCK_PROTECT))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Unlock Flash', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Unlock Flash', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._read_status(expected=0, mask=_Gen25FlashDeviceSvf.SR_PROTECT_ALL, f=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b01000000, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], Mx75lFlashDeviceSvf.__name__, Mx75lFlashDeviceSvf)

class Mx75uFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Macronix MX75U flash device implementation"""

    CMD_WRITE_STATUS_REG = 0x01

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    JEDEC_ID = 0xC2
    DEVICES = {0x29: 'MX75U'}
    SIZES = {
            0x39: 4 << 23   # 256Mb
            }
    MEMORY_DENSITY = {
        0x29 : {
            0x39: "25690"
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.0005, 0.003),  # 0.5/3 ms         # no datasheet found, this value find thru tne
               'write_status_register': (0.04, 1),
               'subsector': (0.025, 0.300),  # 25/300 ms    # 4k erase time
               'hsector': (0.150, 1.25),  # 0.14/1 s         # 32k erase time
               'sector': (0.28, 2.5), # 0.25/2.0 s          # 64k erase time
               'lock': (0.0015, 0.04)}  # 1.5/40 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(Mx75uFlashDeviceSvf, self).__init__(freq)
        if not Mx75uFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = Mx75uFlashDeviceSvf.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Macronix %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def unlock(self, f):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        self._enable_write(f)
        cmd = Array('B', (_Gen25FlashDeviceSvf.CMD_WRSR,
                               _Gen25FlashDeviceSvf.SR_WEL |
                               _Gen25FlashDeviceSvf.SR_PROTECT_NONE |
                               _Gen25FlashDeviceSvf.SR_UNLOCK_PROTECT))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Unlock Flash', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Unlock Flash', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._read_status(expected=0, mask=_Gen25FlashDeviceSvf.SR_PROTECT_ALL, f=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b01000000, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)


setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], Mx75uFlashDeviceSvf.__name__, Mx75uFlashDeviceSvf)

class At25DLFlashDeviceSvf(_Gen25FlashDeviceSvf):
    JEDEC_ID = 0x1F
    DEVICES = { 0x45: 'AT25DL' }
    SIZES = { 0x02: 1 << 20 }

    SPI_FREQ_MAX = 85  # MHz

    TIMINGS = {'page': (0.001, 0.003),      # 0.7/5 ms          # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.002, 0.005),
               'subsector': (0.05, 0.2),  # 60/300 ms         # 4k erase time
               'hsector': (0.25, 0.6),        # 0.3/1.3 s         # 32k erase time
               'sector': (0.55, 0.95),           # 0.5/3 s           # 64k erase time
               'lock': (0.015, 0.015)}       # 15/15 ms          #

    MEMORY_DENSITY = {
        0x45 : {
            0x02: "081"
        }
    }

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(At25DLFlashDeviceSvf, self).__init__(freq)
        if not At25DLFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = self.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Adesto %s%s %s' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20))

    def enable_quad_spi(self, sync):
        raise Exception('Adesto At25DL not support quad IO')

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], At25DLFlashDeviceSvf.__name__, At25DLFlashDeviceSvf)

class At25sfRenFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Renesas AT25SF flash device implementation"""

    CMD_WRITE_STATUS_REG = 0x01
    CMD_READ_STATUS_REG_2 = 0x35
    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31

    JEDEC_ID = 0x1F
    # For adesto flash, the 2nd Byte of JEDEC_ID has the infomation of flash capacity
    # while 3rd Byte of JEDEC_ID has only the product version information.
    DEVICES = { 0x88: 'AT25SF' }

    # Add new device capacity table
    SIZES = { 0x88 : 1 << 23}

    MEMORY_DENSITY = {
        0x88 : {
            0x01 : "641"
        }
    }

    SPI_FREQ_MAX = 55  # MHz    #Need to confirm

    TIMINGS = {'page': (0.0004, 0.003),     # 0.4/3 ms          # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.03),
               'subsector': (0.065, 0.25),  # 65/250 ms         # 4k erase time
               'hsector': (0.15, 0.5),      # 0.15/0.5 s         # 32k erase time
               'sector': (0.24, 0.9),       # 0.24/0.9 s           # 64k erase time
               'lock': (0.005, 0.03)}       # 5/30 ms          #

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(At25sfRenFlashDeviceSvf, self).__init__(freq)
        if not At25sfRenFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, capacity, ver = jedec[0:3]
        self._size = At25sfRenFlashDeviceSvf.SIZES[capacity]
        self._device = 'AT25SF'
        self._density_code = At25sfRenFlashDeviceSvf.MEMORY_DENSITY[capacity][ver]

    def __str__(self):
        return 'Renesas %s%s %s' % \
            (self._device, self._density_code, pretty_size(self._size, lim_m=1 << 20))

    def unlock(self, f):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        self._enable_write(f)
        cmd = Array('B', (At25sfFlashDeviceSvf.CMD_WRITE_STATUS_REG,
                               _Gen25FlashDeviceSvf.SR_WEL |
                               _Gen25FlashDeviceSvf.SR_PROTECT_NONE |
                               _Gen25FlashDeviceSvf.SR_UNLOCK_PROTECT))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Unlock Flash', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Unlock Flash', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._read_status(expected=0, mask=_Gen25FlashDeviceSvf.SR_PROTECT_ALL, f=f)

    #def _read_status2(self, sync):
    #    pass

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def _write_status2(self, val, f):
        # write enable
        self._enable_write(f)

        # write status reg
        cmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status2(0b00000010, f)
        print('// End Enable QE', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], At25sfRenFlashDeviceSvf.__name__, At25sfRenFlashDeviceSvf)

class At25sfFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Atmel/Adesto AT25SF(rev A) flash device implementation"""

    CMD_WRITE_STATUS_REG = 0x01
    CMD_READ_STATUS_REG_2 = 0x35

    FEAT_WRITE_BOTH_STATUS_REG = True

    JEDEC_ID = 0x1F
    # For adesto flash, the 2nd Byte of JEDEC_ID has the infomation of flash capacity
    # while 3rd Byte of JEDEC_ID has only the product version information.
    DEVICES = { 0x85: 'AT25SF' }

    # Add new device capacity table
    SIZES = { 0x85 : 1 << 20}

    MEMORY_DENSITY = {
        0x85 : {
            0x01 : "081"
        }
    }

    SPI_FREQ_MAX = 55  # MHz    #Need to confirm

    TIMINGS = {'page': (0.0007, 0.005),      # 0.7/5 ms          # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.015, 0.03),
               'subsector': (0.060, 0.300),  # 60/300 ms         # 4k erase time
               'hsector': (0.3, 1.3),        # 0.3/1.3 s         # 32k erase time
               'sector': (0.5, 3),           # 0.5/3 s           # 64k erase time
               'lock': (0.015, 0.015)}       # 15/15 ms          #

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, freq, jedec):
        super(At25sfFlashDeviceSvf, self).__init__(freq)
        if not At25sfFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, capacity, ver = jedec[0:3]
        self._size = At25sfFlashDeviceSvf.SIZES[capacity]
        self._device = 'AT25SF'
        self._density_code = At25sfFlashDeviceSvf.MEMORY_DENSITY[capacity][ver]

    def __str__(self):
        return 'Adesto %s%s %s' % \
            (self._device, self._density_code, pretty_size(self._size, lim_m=1 << 20))

    def unlock(self, f):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        self._enable_write(f)
        cmd = Array('B', (At25sfFlashDeviceSvf.CMD_WRITE_STATUS_REG,
                               _Gen25FlashDeviceSvf.SR_WEL |
                               _Gen25FlashDeviceSvf.SR_PROTECT_NONE |
                               _Gen25FlashDeviceSvf.SR_UNLOCK_PROTECT))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Unlock Flash', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Unlock Flash', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._read_status(expected=0, mask=_Gen25FlashDeviceSvf.SR_PROTECT_ALL, f=f)

    #def _read_status2(self, sync):
    #    rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
    #    rcmd = reverseBytes(bytes(rcmd))
    #    self._jtag_engine.write_dr_bytes(rcmd)
    #    data = self._jtag_engine.read_dr(8)
    #    self._jtag_engine.go_idle2(sync=sync)
    #    data = data.tobytes(True)

    #    if len(data) != 1:
    #        raise SerialFlashTimeout("Unable to read status register-2")
    #    return data[0]

    def _write_status(self, val1, val2, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val1, val2))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b00000010, 0b00000010, f)
        print('// End Enable QE', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], At25sfFlashDeviceSvf.__name__, At25sfFlashDeviceSvf)

class IS25LPWPFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """ISSI IS25LP flash device implementation"""
    # IS25LP and IS25WP only differ in voltage supply(2.3V to 3.6V to 1.65V to 1.95V)

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0x29

    JEDEC_ID = 0x9D
    DEVICES = { 0x40: 'IS25LP', 0x60: 'IS25LP', 0x70: 'IS25WP'}
    MEMORY_DENSITY = {
        0x40 : {
            0x11 : "1",
            0x12 : "2",
            0x13 : "4",
        },
        0x60 : {
            0x14 : "8",
            0x15 : "16",
            0x16 : "32",
            0x17 : "64",
            0x18 : "128",
            0x19 : "256",
            0x1A : "512",
        },
        0x70 : {
            0x11 : "1",
            0x12 : "2",
            0x13 : "4",
            0x14 : "8",
            0x15 : "16",
            0x16 : "32",
            0x17 : "64",
            0x18 : "128",
            0x19 : "256",
            0x1A : "512",
        }
    }
    SIZES = {0x11: 4 << 15,
             0x12: 4 << 16,
             0x13: 4 << 17,
             0x14: 4 << 18,
             0x15: 4 << 19,
             0x16: 4 << 20,
             0x17: 4 << 21,
             0x18: 4 << 22,
             0x19: 4 << 23,
             0x1A: 4 << 24}

    SPI_FREQ_MAX = 55  # MHz

    TIMINGS = {'page': (0.0002, 0.0008),    # 0.2/0.8 ms        # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.002, 0.015),
               # subsector min is 0.07(70ms) in json tho, however found 0.05(50ms) in datasheet
               'subsector': (0.05, 0.3),    # 50/300 ms         # 4k erase time
               'hsector': (0.1, 0.5),       # 0.1/0.5 s         # 32k erase time
               'sector': (0.15, 1),         # 0.15s/1 s         # 64k erase time
               'lock': (0.002, 0.0008)}     # 0.2/0.8 ms        # not found in datasheet(to be confirmed)
               #'chip': (80, 270)}

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)
                #SerialFlash.FEAT_CHIPERASE)

    def __init__(self, freq, jedec):
        super(IS25LPWPFlashDeviceSvf, self).__init__(freq)
        if not IS25LPWPFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._size = IS25LPWPFlashDeviceSvf.SIZES[capacity]
        self._device = IS25LPWPFlashDeviceSvf.DEVICES[device]
        self._density_code = IS25LPWPFlashDeviceSvf.MEMORY_DENSITY[device][capacity]

    def __str__(self):
        return 'ISSI %s%s %s' % \
            (self._device, self._density_code, pretty_size(self._size, lim_m=1 << 20))

    def _soft_reset(self, f):
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b01000000, f)
        print('// End Enable QE', file=f)

    def enter_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Enable 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Enable 4byte Addr Mode', file=f)

    def exit_4byte_addr_mode(self, f):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Exit 4byte Addr Mode', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Exit 4byte Addr Mode', file=f)


setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], IS25LPWPFlashDeviceSvf.__name__, IS25LPWPFlashDeviceSvf)

# XMC
class XM25QFlashDeviceSvf(_Gen25FlashDeviceSvf):
    """Winbond W25X flash device implementation"""

    JEDEC_ID = 0x20
    DEVICES = {0x40: 'XM25QH'}
    SIZES = {0x17: 8 << 20}
    SPI_FREQ_MAX = 104  # MHz
    CMD_READ_UID = 0x4B
    UID_LEN = 0x8  # 64 bits
    READ_UID_WIDTH = 4  # 4 dummy bytes
    TIMINGS = {'page': (0.0005, 0.003),  # 1.5/3 ms
               'subsector': (0.04, 0.4),  # 200/200 ms
               'hsector'  : (0.12, 0.9),     # 0.30/1.6 s     # 32k erase time
               'sector': (0.25, 2),  # 1/1 s
               'write_status_register': (0.001, 0.05),
               'lock': (0.05, 0.1)  # 50/100 ms
              }
    # TODO 32KB erase should be supported according to datasheet
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)
                #SerialFlash.FEAT_CHIPERASE)

    MEMORY_DENSITY = {
        0x40 : {
            0x17 : "64"}
    }

    def __init__(self, freq, jedec):
        super(XM25QFlashDeviceSvf, self).__init__(freq)
        if not XM25QFlashDeviceSvf.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        device, capacity = jedec[1:3]
        self._device = self.DEVICES[device]
        self._size = self.SIZES[capacity]

    def __str__(self):
        return 'XMC %s%d %s' % \
            (self._device, len(self) >> 17,
             pretty_size(self._size, lim_m=1 << 20))

    #def _read_status2(self, sync):
    #    rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
    #    rcmd = reverseBytes(bytes(rcmd))
    #    self._jtag_engine.write_dr_bytes(rcmd)
    #    data = self._jtag_engine.read_dr2(8)
    #    self._jtag_engine.go_idle2(sync=sync)
    #    data = data.tobytes(True)

    #    if len(data) != 1:
    #        raise SerialFlashTimeout("Unable to read status register-2")
    #    return data[0]

    def _write_status(self, val, f):
        self._enable_write(f)

        # write nonvolative conf reg
        cmd = Array('B', (self.CMD_WRSR, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)

        # soft reset
        self._soft_reset(f)

    def _write_status2(self, val, f):
        # write enable
        self._enable_write(f)

        # write status reg
        cmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Write Status', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Write Status', file=f)

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._wait_for_completion(self.WRITE_STATUS_DELAY, f)

        self._read_status(expected=0, mask=self.SR_WIP, f=f)
        self._soft_reset(f)

    def enable_quad_spi(self, f):
        #status_reg_2 = self._read_status2(f)
        # FIXME; this way will overwrite others status
        print('// Begin Enable QE', file=f)
        self._write_status(0b00000010, f)
        print('// End Enable QE', file=f)

    def _soft_reset(self, f):
        # soft reset
        cmd = Array('B', (0x66,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print('// Begin Soft Reset', file=f)
        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)

        cmd = Array('B', (0x99,))
        cmdstr = convertByteArray2SvfByteStr(cmd)
        cmdstr_bitlength = len(cmdstr)*4

        print(f'SDR {cmdstr_bitlength} TDI ({cmdstr});', file=f)
        print('STATE IDLE;', file=f)
        print('// End Soft Reset', file=f)

setattr(sys.modules['efx_pgm.jtag2SpiFlashSvf'], XM25QFlashDeviceSvf.__name__, XM25QFlashDeviceSvf)


class GenericEfinixSupportedFlashSvf(_Gen25FlashDeviceSvf):
    TIMINGS = {'page': (0.0015, 0.003),  # 1.5/3 ms
               'subsector': (0.1, 0.4),  # 200/200 ms
               'sector': (0.15, 2),  # 1/1 s
               'write_status_register': (0.01, 0.03),
               'bulk': (32, 64),  # seconds
               'lock': (0.05, 0.1)}  # 50/100 ms
               #'chip': (20, 100)}
    FEATURES = SerialFlash.FEAT_SECTERASE | SerialFlash.FEAT_SUBSECTERASE

    def __init__(self, freq, jedec):
        super(GenericEfinixSupportedFlashSvf, self).__init__(freq)
        self.manufacturer_id, self.device_id, self.capacity_id = jedec[0:3]
        self._device = 'Unknown'
        self._size = 16 << 20	# Assume 128Mbit device

    def __str__(self):
        return 'Generic Flash Profile. JEDEC id: 0x%02X%02X%02X' % (self.manufacturer_id, self.device_id, self.capacity_id)
