import sys
import time
from array import array as Array
from typing import Tuple, Union, Any

from pyftdi.misc import pretty_size
from spiflash.serialflash import SerialFlashTimeout, SerialFlashUnknownJedec, SerialFlash, SerialFlashRequestError

from efx_pgm.jtag2SpiFlash import _Gen25FlashDevice, N25QFlashDevice, W25XFlashDevice, W25QFlashDevice,Mx25lFlashDevice, M25PxFlashDevice, W25QJVFlashDevice
from efx_pgm.jtag2SpiDriver import Jtag2SpiChainEngine


def reverseBytes(_bytes: bytes):
    def _reverseByte(_byte):
        return int('{:08b}'.format(_byte)[::-1], 2)
    return bytes(tuple(map(lambda x: _reverseByte(x), _bytes)))

#------------------------------------------------------------------------------------------------------
# Puyu
class P25xFlashDevice(_Gen25FlashDevice):
    """Puya P25Q flash device implementation"""
    CMD_READ_STATUS_REG_2  = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31

    JEDEC_ID = 0x85
    DEVICES = {0x60: 'P25Q', 0x40: 'P25Q', 0x42: 'P25Q'}
    SIZES = {
            0x10: 4 << 14,
            0x11: 4 << 15,
            0x12: 4 << 16,
            0x13: 4 << 17,
            0x14: 4 << 18,
            0x15: 4 << 19,
            0x16: 4 << 20,
            0x17: 4 << 21,
            0x18: 4 << 22
            }
    MEMORY_DENSITY = {
        0x60 : {
            0x10: "05",  # 512Kb
            0x11: "10",  # 1Mb
            0x12: "20",  # 2Mb
            0x13: "40",  # 4Mb
            0x14: "80",  # 8Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
        },
        0x40: {
            0x10: "06",  # 512Kb
            0x11: "11",  # 1Mb
            0x12: "21",  # 2Mb
            0x13: "40",  # 4Mb
            0x14: "80",  # 8Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
        },
        0x42: {
            0x10: "06",  # 512Kb
            0x11: "11",  # 1Mb
            0x12: "21",  # 2Mb
            0x13: "40",  # 4Mb
            0x14: "80",  # 8Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    CMD_READ_UID = 0x4B
    UID_LEN = 0x16  # 128 bits
    READ_UID_WIDTH = 4  # 4 dummy bytes
    TIMINGS = {'page': (0.002, 0.003),  # 8/20 ms       # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.008, 0.012),
               'subsector': (0.016, 0.030),  # 8/20 ms
               'sector': (0.016, 0.030),  # 8/20 ms
               'bulk': (0.016, 0.030),  # 8/20 ms
               'lock': (0.05, 0.1)}  # 50/100 ms

    FEATURES = SerialFlash.FEAT_SECTERASE | SerialFlash.FEAT_SUBSECTERASE

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(P25xFlashDevice, self).__init__(jtag_engine, jedec)
        if not P25xFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = P25xFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Puya %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(P25xFlashDevice, self).__str__())

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _write_status2(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        self._jtag_engine.go_idle2(sync=sync)

    def enable_quad_spi(self, sync):
        status_reg_2 = self._read_status2(sync=False)
        self._write_status(status_reg_2 | 0b00000010, sync=sync)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], P25xFlashDevice.__name__, P25xFlashDevice)


#------------------------------------------------------------------------------------------------------
# Gigadevice GD25x
class GD25xFlashDevice(_Gen25FlashDevice):
    """Gigadevice GD25 flash device implementation"""

    CMD_READ_STATUS_REG_2  = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31

    JEDEC_ID = 0xC8
    DEVICES = {0x40: 'GD25Q', 0x65: 'GD25WQ'}
    SIZES = {
            0x12: 4 << 16,  # 2Mb
            0x14: 4 << 18,  # 8Mb
            0x15: 4 << 19,  # 16Mb
            0x16: 4 << 20,  # 32Mb
            0x17: 4 << 21,  # 64Mb
            0x18: 4 << 22,  # 128Mb
            0x19: 4 << 23   # 256Mb
            }
    MEMORY_DENSITY = {
        0x40 : {
            0x12: "2",  # 2Mb
            0x14: "8",  # 2Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
            0x19: "256", # 256Mb
        },
        0x65: {
            0x12: "2",  # 2Mb
            0x14: "8",  # 2Mb
            0x15: "16",  # 16Mb
            0x16: "32",  # 32Mb
            0x17: "64",  # 64Mb
            0x18: "128", # 128Mb
            0x19: "256", # 256Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.0006, 0.0024),  # 0.6/2.4 ms       # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.03),
               'sector': (0.045, 0.300),  # 45/300 ms
               'lock': (0.005, 0.03)}  # 5/30 ms
    FEATURES = SerialFlash.FEAT_SECTERASE

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(GD25xFlashDevice, self).__init__(jtag_engine, jedec)
        if not GD25xFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = GD25xFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Gigadevice %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(GD25xFlashDevice, self).__str__())

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def _write_status(self, val, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _write_status2(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)

    def enable_quad_spi(self, sync):
        status_reg_2 = self._read_status2(sync=False)
        self._write_status2(status_reg_2 | 0b00000010, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], GD25xFlashDevice.__name__, GD25xFlashDevice)		# Add GD25xFlashDevice to spiflash.serialflash module


#------------------------------------------------------------------------------------------------------
# Gigadevice GD25lq
class GD25lqFlashDevice(_Gen25FlashDevice):
    """Gigadevice GD25LQ flash device implementation"""
    CMD_READ_STATUS_REG_2  = 0x35
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    FEAT_WRITE_BOTH_STATUS_REG = True

    JEDEC_ID = 0xC8
    DEVICES = {0x60: 'GD25LQ'}
    SIZES = {
            0x16: 4 << 20,  # 32Mb
            0x1A: 4 << 24   # 512Mb
            }
    MEMORY_DENSITY = {
        0x60 : {
            0x16: "32"   # 32Mb
        },
        0x67 : {
            0x1A: "512"   # 512Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page'     : (0.0007, 0.004),  # 0.7/4 ms      # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.035),
               'subsector': (0.09, 0.60),    # 90/600 ms      # 4k erase time
               'hsector'  : (0.30, 1.6),     # 0.30/1.6 s     # 32k erase time
               'sector'   : (0.45, 3.0),     # 0.45/3.0 s     # 64k erase time
               'lock'     : (0.005, 0.035)}  # 5/35 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(GD25lqFlashDevice, self).__init__(jtag_engine, jedec)
        if not GD25lqFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = GD25lqFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Gigadevice %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(GD25lqFlashDevice, self).__str__())

    def unlock(self, sync):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        status_reg_2 = self._read_status_reg_2(sync=False)

        self._enable_write(sync=False)
        wcmd = Array('B', (GD25lqFlashDevice.CMD_WRSR,
                               _Gen25FlashDevice.SR_WEL |
                               _Gen25FlashDevice.SR_PROTECT_NONE |
                               _Gen25FlashDevice.SR_UNLOCK_PROTECT,
                               status_reg_2))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()
        duration = self.get_timings('lock')
        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)
        status = self._read_status(sync=sync)
        if status & GD25lqFlashDevice.SR_PROTECT_ALL:
            raise SerialFlashRequestError("Cannot unprotect flash device")

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    # a dedicated write status register impl
    def _write_status(self, val1, val2, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val1, val2))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        # Wait for the write to complete
        # start = time.time()
        # NOTE: why wait for write enable after write?
        # while self._read_status(False) & 2:
        #    if time.time() - start >= 1:
        #        raise SerialFlashTimeout("Timed out while setting status register-1.")

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)

    def enable_quad_spi(self, sync):
        status_reg = self._read_status(sync=False)
        status_reg_2 = self._read_status2(sync=False)
        self._write_status(status_reg, status_reg_2 | 0b00000010, sync=sync)

        #status_reg = self._read_status_reg()
        #status_reg_2 = self._read_status_reg_2()

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], GD25lqFlashDevice.__name__, GD25lqFlashDevice)		# Add GD25lqFlashDevice to spiflash.serialflash module

#------------------------------------------------------------------------------------------------------
# Gigadevice GD25lb
class GD25lbFlashDevice(_Gen25FlashDevice):
    """Gigadevice GD25LB flash device implementation"""
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    FEAT_WRITE_BOTH_STATUS_REG = False

    JEDEC_ID = 0xC8
    DEVICES = {0x67: 'GD25LB'}
    SIZES = {
            0x16: 4 << 20,  # 32Mb
            0x1A: 4 << 24   # 512Mb
            }
    MEMORY_DENSITY = {
        0x60 : {
            0x16: "32"   # 32Mb
        },
        0x67 : {
            0x1A: "512"   # 512Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page'     : (0.0007, 0.004),  # 0.7/4 ms      # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.035),
               'subsector': (0.09, 0.60),    # 90/600 ms      # 4k erase time
               'hsector'  : (0.30, 1.6),     # 0.30/1.6 s     # 32k erase time
               'sector'   : (0.45, 3.0),     # 0.45/3.0 s     # 64k erase time
               'lock'     : (0.005, 0.035)}  # 5/35 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(GD25lbFlashDevice, self).__init__(jtag_engine, jedec)
        if not GD25lbFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = GD25lbFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Gigadevice %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(GD25lbFlashDevice, self).__str__())

    def unlock(self, sync):
        '''
        Unlock the flash for write. For GD25LQ flash, we need to write both SR-1 and SR-2 together
        '''
        self._enable_write(sync=False)
        wcmd = Array('B', (GD25lbFlashDevice.CMD_WRSR,
                            _Gen25FlashDevice.SR_WEL |
                            _Gen25FlashDevice.SR_PROTECT_NONE |
                            _Gen25FlashDevice.SR_UNLOCK_PROTECT))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()
        duration = self.get_timings('lock')
        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)
        status = self._read_status(sync=sync)
        if status & GD25lbFlashDevice.SR_PROTECT_ALL:
            raise SerialFlashRequestError("Cannot unprotect flash device")

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    # a dedicated write status register impl
    def _write_status(self, val1, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val1))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        # Wait for the write to complete
        # start = time.time()
        # NOTE: why wait for write enable after write?
        # while self._read_status(False) & 2:
        #    if time.time() - start >= 1:
        #        raise SerialFlashTimeout("Timed out while setting status register-1.")

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], GD25lbFlashDevice.__name__, GD25lbFlashDevice)		# Add GD25lbFlashDevice to spiflash.serialflash module

#------------------------------------------------------------------------------------------------------
# Macronix mx25l

# TODO: no chip on hand
class Mx25lFlashDeviceExt(Mx25lFlashDevice):
    """Extend the implementation of original Mx25lFlashDevice"""

    CMD_READ_ELECTRONIC_ID = 0xAB
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(Mx25lFlashDeviceExt, self).__init__(jtag_engine, jedec)

        #if self._spi is not None:
        # Read the RES (Electronic ID)
        self._electronic_id = self._read_electronic_id(sync=False)
        if(self._electronic_id == 0x14 and self._device == 'MX25L'):
            self._device = 'MX25V'
        #else:
        #    self._electronic_id = None

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _read_electronic_id(self, sync):
        rcmd = Array('B', (self.CMD_READ_ELECTRONIC_ID, 0xFF, 0xFF, 0xFF))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read electronic id")
        return data[0]

    def enable_quad_spi(self, sync):
        # MX25V doesn't support quad
        if self._device == 'MX25V':
            return

        status_reg = self._read_status(sync=False)
        self._write_status_reg(status_reg | 0b01000000, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], Mx25lFlashDeviceExt.__name__, Mx25lFlashDeviceExt)

# no chip on hand
class Mx25uFlashDevice(_Gen25FlashDevice):
    """Macronix MX25U flash device implementation"""

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    JEDEC_ID = 0xC2
    DEVICES = {0x25: 'MX25U',
               0x95: 'MX25U'}
    SIZES = {
            0x34: 4 << 18,   # 8Mb
            0x39: 32 << 20,  # 256Mb
            0x3A: 64 << 20  # 512Mb
            }
    MEMORY_DENSITY = {
        0x25 : {
            0x34: "8035",    # 8Mb
            0x39: "25645G", # 256Mb
            0x3A: "512G"  # 512Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.00015, 0.00075),  # 1.5/3 ms       # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.04, 0.04),
               'subsector': (0.025, 0.4),  # 300/300 ms
               'hsector': (0.15, 1),  # 2/2 s
               'sector': (0.22, 1.3),  # 2/2 s
               'lock': (0.0015, 0.003)}  # 1.5/3 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(Mx25uFlashDevice, self).__init__(jtag_engine, jedec)
        if not Mx25uFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = Mx25uFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Macronix %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(Mx25uFlashDevice, self).__str__())

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg = self._read_status(sync=False)
        self._write_status(status_reg | 0b01000000, sync=sync)

    def disable_quad_spi(self, sync):
        status_reg = self._read_status(sync=False)
        self._write_status(status_reg & 0b10111111, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], Mx25uFlashDevice.__name__, Mx25uFlashDevice)

#------------------------------------------------------------------------------------------------------
# Micron
class N25QFlashDeviceExt(N25QFlashDevice):
    """Extend the implementation of original N25QFlashDevice"""

    READ_NV_CONF_REG = 0xB5
    WRITE_NV_CONF_REG = 0xB1

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    # Change the name from N25Q to MT25Q
    DEVICES = {0xBA: 'MT25Q', 0xBB: 'MT25Q'}

    # Add new device capacity table
    SIZES = {
                0x15: 2 << 20,
                0x16: 4 << 20,
                0x17: 8 << 20,
                0x18: 16 << 20,
                0x19: 32 << 20,
                0x20: 64 << 20,
                0x21: 128 << 20,
                0x22: 256 << 20
            }

    TIMINGS = {'page': (0.00012, 0.005),  # 0.5/5 ms
               'write_status_register': (0.0013, 0.008),
               'subsector': (0.05, 3.0),  # 300/3000 ms
               'sector': (0.15, 3.0),  # 700/3000 ms
               'bulk': (60, 120)}  # seconds

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        try:
            super(N25QFlashDeviceExt, self).__init__(jtag_engine, jedec)
        # This is expected, as the base class doesn't aware of the newly added SIZES
        except SerialFlashUnknownJedec as exc:
            if not self.match(jedec):
                raise SerialFlashUnknownJedec(jedec) from exc
            device, capacity = jedec[1:]
            self._size = self.SIZES[capacity]
            self._device = self.DEVICES[device]

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _read_nv_conf_ref(self, sync):
        rcmd = Array('B', (self.READ_NV_CONF_REG,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(16)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 2:
            raise SerialFlashTimeout("Unable to retrieve nonvolative configuration status")
        return data

    def _write_nv_conf_ref(self, data):
        # write enable
        wcmd = Array('B', (self.CMD_WRITE_ENABLE,))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle2(False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.WRITE_NV_CONF_REG, data[0], data[1]))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)

        # soft reset
        self._soft_reset(sync=True)
        time.sleep(1)

    # Micron Flash doesn't need set anything for for Dual/Quad operation.
    # Beware the differences between dual-I/O and dual operation. Likewise for qual-I/O and quad operation.
    #def enable_quad_spi(self):
    #    nv_conf_reg = self._read_nv_conf_ref()
    #    nv_conf_reg[0] = nv_conf_reg[0] & 0b11110111    # bit-3 Quad protocol [0 = enable]
    #    nv_conf_reg[0] = nv_conf_reg[0] | 0b00000100    # Disable Dual by setting bit-2 to 1
    #    self._write_nv_conf_ref(nv_conf_reg)

    #def enable_dual_spi(self):
    #    nv_conf_reg = self._read_nv_conf_ref()
    #    nv_conf_reg[0] = nv_conf_reg[0] & 0b11111011    # bit-2 Dual protocol [0 = enable]
    #    nv_conf_reg[0] = nv_conf_reg[0] | 0b00001000    # Disable quad by setting bit-3 to 1
    #    self._write_nv_conf_ref(nv_conf_reg)

    #def is_address_mode_3bytes(self):

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)


setattr(sys.modules['efx_pgm.jtag2SpiFlash'], N25QFlashDeviceExt.__name__, N25QFlashDeviceExt)



class W25XFlashDeviceExt(W25XFlashDevice):
    """Extend the implementation of original W25xFlashDevice"""

    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(W25XFlashDeviceExt, self).__init__(jtag_engine, jedec)

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(True)
        time.sleep(0.05)

    def _write_status2(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg_2 = self._read_status2(sync=False)
        self._write_status2(status_reg_2 | 0b00000010, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], W25XFlashDeviceExt.__name__, W25XFlashDeviceExt)

class W25QFlashDeviceExt(W25QFlashDevice):
    """Extend the implementation of original W25xFlashDevice"""

    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    DEVICES = {0x40: 'W25Q', 0x60: 'W25Q', 0x80: 'W25Q'}

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(W25QFlashDeviceExt, self).__init__(jtag_engine, jedec)

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _write_status2(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)

    def enable_quad_spi(self, sync):
        status_reg_2 = self._read_status2(sync=False)
        self._write_status2(status_reg_2 | 0b00000010, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], W25QFlashDeviceExt.__name__, W25QFlashDeviceExt)

class W25QJVFlashDeviceExt(W25QJVFlashDevice):
    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31
    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(W25QJVFlashDeviceExt, self).__init__(jtag_engine, jedec)

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _write_status2(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)

    def enable_quad_spi(self, sync):
        status_reg_2 = self._read_status2(sync=False)
        self._write_status2(status_reg_2 | 0b00000010, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], W25QJVFlashDeviceExt.__name__, W25QJVFlashDeviceExt)


class M25PxFlashDeviceExt(M25PxFlashDevice):
    """Extend the implementation of original M25PxFlashDevice"""

    #JEDEC_ID = 0x20
    #DEVICES = {0x71: 'M25PX', 0x20: 'M25P'}
    #SIZES = {0x14: 1 << 20, 0x15: 2 << 20, 0x16: 4 << 20, 0x17: 8 << 20, 0x18: 16 << 20}
    #SPI_FREQ_MAX = 75  # MHz (P series only)
    #TIMINGS = {'page': (0.0015, 0.003),  # 1.5/3 ms
    #           'write_status_register': (1, 1),
    #           'subsector': (0.150, 0.150),  # 150/150 ms
    #           'sector': (0.8, 0.8),  # 0.8/3 s
    #           'bulk': (32, 64),  # seconds
    #           'lock': (0.0015, 0.003)}  # 1.5/3 ms
    #FEATURES = SerialFlash.FEAT_SECTERASE

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(M25PxFlashDeviceExt, self).__init__(jtag_engine, jedec)

    def __str__(self):
        return 'Micron %s%d %s (%s)' % \
            (self._device, len(self) >> 17,
             pretty_size(self._size, lim_m=1 << 20),
             super(M25PxFlashDeviceExt, self).__str__())

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)


setattr(sys.modules['efx_pgm.jtag2SpiFlash'], M25PxFlashDeviceExt.__name__, M25PxFlashDeviceExt)

class XT25xFlashDevice(_Gen25FlashDevice):
    """XTX XT25F flash device implementation"""

    CMD_READ_STATUS_REG_2 = 0x35

    FEAT_WRITE_BOTH_STATUS_REG = True

    JEDEC_ID = 0x0B
    DEVICES = {0x40: 'XT25F'}
    SIZES = {
            0x14: 4 << 18,
            0x15: 2 << 20,
            }
    MEMORY_DENSITY = {
        0x40: {
            0x14: "08B",  # 8Mb
            0x15: "16B",  #16Mb
        }
    }
    SPI_FREQ_MAX = 55  # MHz
    CMD_READ_UID = 0x4B
    UID_LEN = 0x16  # 128 bits
    READ_UID_WIDTH = 4  # 4 dummy bytes
    TIMINGS = {'page': (0.0004, 0.003),  # 0.4/0.7 ms        # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.001, 0.02),
               'subsector': (0.08, 0.80),  # 70/800 ms      # 4k erase time
               'hsector': (0.15, 1.2),  # 0.15/1.2 s        # 32k erase time
               'sector': (0.25, 1.6),  # 0.25/1.6 s         # 64k erase time
               'lock': (0.07, 0.8)}  # 70/800 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(XT25xFlashDevice, self).__init__(jtag_engine, jedec)
        if not XT25xFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = XT25xFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'XTX %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(XT25xFlashDevice, self).__str__())

    def unlock(self, sync):
        '''
        Unlock the flash for write. For XTX flash, we need to write both SR-1 and SR-2 together
        '''
        status_reg_2 = self._read_status2(False)

        self._enable_write(sync=False)
        wcmd = Array('B', (XT25xFlashDevice.CMD_WRSR,
                               _Gen25FlashDevice.SR_WEL |
                               _Gen25FlashDevice.SR_PROTECT_NONE |
                               _Gen25FlashDevice.SR_UNLOCK_PROTECT,
                               status_reg_2))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()
        duration = self.get_timings('lock')
        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)
        status = self._read_status(sync=sync)
        if status & _Gen25FlashDevice.SR_PROTECT_ALL:
            raise SerialFlashRequestError("Cannot unprotect flash device")

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr(8)
        self._jtag_engine.go_idle2(sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val1, val2, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val1, val2))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg = self._read_status(sync=False)
        status_reg_2 = self._read_status2(sync=False)
        self._write_status(status_reg, status_reg_2 | 0b00000010, sync)

        #status_reg = self._read_status_reg()
        #status_reg_2 = self._read_status_reg_2()

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], XT25xFlashDevice.__name__, XT25xFlashDevice)		# Add XT25xFlashDevice to spiflash.serialflash module

class Mx75lFlashDevice(_Gen25FlashDevice):
    """Macronix MX75L flash device implementation"""

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    JEDEC_ID = 0xC2
    DEVICES = {0x26: 'MX75L'}
    SIZES = {
            0x19: 4 << 23   # 256Mb
            }
    MEMORY_DENSITY = {
        0x26 : {
            0x19: "25690"
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.0005, 0.003),  # 0.5/3 ms         # no datasheet found, this value find thru tne
               'write_status_register': (0.04, 1),
               'subsector': (0.025, 0.300),  # 25/300 ms    # 4k erase time
               'hsector': (0.140, 1.0),  # 0.14/1 s         # 32k erase time
               'sector': (0.25, 2.0), # 0.25/2.0 s          # 64k erase time
               'lock': (0.0015, 0.04)}  # 1.5/40 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(Mx75lFlashDevice, self).__init__(jtag_engine, jedec)
        if not Mx75lFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = Mx75lFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Macronix %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(Mx75lFlashDevice, self).__str__())

    def unlock(self, sync):
        '''
        Unlock the flash for write.
        '''
        status_reg = self._read_status(sync=False)
        status_reg = (status_reg |
                      _Gen25FlashDevice.SR_WEL |
                      _Gen25FlashDevice.SR_PROTECT_NONE |
                      _Gen25FlashDevice.SR_UNLOCK_PROTECT)

        self._enable_write(False)
        wcmd = Array('B', (_Gen25FlashDevice.CMD_WRSR, status_reg))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()
        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)
        status = self._read_status(sync)
        if status & _Gen25FlashDevice.SR_PROTECT_ALL:
            raise SerialFlashRequestError("Cannot unprotect flash device")

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg = self._read_status(sync=False)
        self._write_status(status_reg | 0b01000000, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], Mx75lFlashDevice.__name__, Mx75lFlashDevice)

class Mx75uFlashDevice(_Gen25FlashDevice):
    """Macronix MX75U flash device implementation"""

    CMD_WRITE_STATUS_REG = 0x01

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0xE9

    JEDEC_ID = 0xC2
    DEVICES = {0x29: 'MX75U'}
    SIZES = {
            0x39: 4 << 23   # 256Mb
            }
    MEMORY_DENSITY = {
        0x29 : {
            0x39: "25690"
        }
    }
    SPI_FREQ_MAX = 55  # MHz    #Need to confirm
    TIMINGS = {'page': (0.0005, 0.003),  # 0.5/3 ms         # no datasheet found, this value find thru tne
               'write_status_register': (0.04, 1),
               'subsector': (0.025, 0.300),  # 25/300 ms    # 4k erase time
               'hsector': (0.150, 1.25),  # 0.14/1 s         # 32k erase time
               'sector': (0.28, 2.5), # 0.25/2.0 s          # 64k erase time
               'lock': (0.0015, 0.04)}  # 1.5/40 ms
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(Mx75uFlashDevice, self).__init__(jtag_engine, jedec)
        if not Mx75uFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = Mx75uFlashDevice.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Macronix %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(Mx75uFlashDevice, self).__str__())

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def unlock(self, sync):
        '''
        Unlock the flash for write.
        '''
        status_reg = self._read_status(sync=False)
        status_reg = (status_reg |
                      _Gen25FlashDevice.SR_WEL |
                      _Gen25FlashDevice.SR_PROTECT_NONE |
                      _Gen25FlashDevice.SR_UNLOCK_PROTECT)

        self._enable_write(False)
        wcmd = Array('B', (_Gen25FlashDevice.CMD_WRSR, status_reg))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()
        duration = self.get_timings('lock')

        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)
        status = self._read_status(sync=sync)
        if status & _Gen25FlashDevice.SR_PROTECT_ALL:
            raise SerialFlashRequestError("Cannot unprotect flash device")

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg = self._read_status(sync=False)
        self._write_status(status_reg | 0b01000000, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)


setattr(sys.modules['efx_pgm.jtag2SpiFlash'], Mx75uFlashDevice.__name__, Mx75uFlashDevice)

class At25DLFlashDevice(_Gen25FlashDevice):
    JEDEC_ID = 0x1F
    DEVICES = { 0x45: 'AT25DL' }
    SIZES = { 0x02: 1 << 20 }

    SPI_FREQ_MAX = 85  # MHz

    TIMINGS = {'page': (0.001, 0.003),      # 0.7/5 ms          # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.002, 0.005),
               'subsector': (0.05, 0.2),  # 60/300 ms         # 4k erase time
               'hsector': (0.25, 0.6),        # 0.3/1.3 s         # 32k erase time
               'sector': (0.55, 0.95),           # 0.5/3 s           # 64k erase time
               'lock': (0.015, 0.015)}       # 15/15 ms          #

    MEMORY_DENSITY = {
        0x45 : {
            0x02: "081"
        }
    }

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(At25DLFlashDevice, self).__init__(jtag_engine, jedec)
        if not At25DLFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._device = self.DEVICES[device]
        self._size = self.SIZES[capacity]
        self._manufacturer_id = manufacturer
        self._device_id = device
        self._capacity_id = capacity

    def __str__(self):
        return 'Adesto %s%s %s (%s)' % \
            (self._device, self.MEMORY_DENSITY[self._device_id][self._capacity_id],
             pretty_size(self._size, lim_m=1 << 20),
             super(At25DLFlashDevice, self).__str__())

    def enable_quad_spi(self, sync):
        raise Exception('Adesto At25DL not support quad IO')

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], At25DLFlashDevice.__name__, At25DLFlashDevice)

class At25sfRenFlashDevice(_Gen25FlashDevice):
    """Renesas AT25SF flash device implementation"""

    CMD_WRITE_STATUS_REG = 0x01
    CMD_READ_STATUS_REG_2 = 0x35
    CMD_READ_STATUS_REG_2 = 0x35
    CMD_WRITE_STATUS_REG_2 = 0x31

    JEDEC_ID = 0x1F
    # For adesto flash, the 2nd Byte of JEDEC_ID has the infomation of flash capacity
    # while 3rd Byte of JEDEC_ID has only the product version information.
    DEVICES = { 0x88: 'AT25SF' }

    # Add new device capacity table
    SIZES = { 0x88 : 1 << 23}

    MEMORY_DENSITY = {
        0x88 : {
            0x01 : "641"
        }
    }

    SPI_FREQ_MAX = 55  # MHz    #Need to confirm

    TIMINGS = {'page': (0.0004, 0.003),     # 0.4/3 ms          # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.005, 0.03),
               'subsector': (0.065, 0.25),  # 65/250 ms         # 4k erase time
               'hsector': (0.15, 0.5),      # 0.15/0.5 s         # 32k erase time
               'sector': (0.24, 0.9),       # 0.24/0.9 s           # 64k erase time
               'lock': (0.005, 0.03)}       # 5/30 ms          #

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(At25sfRenFlashDevice, self).__init__(jtag_engine, jedec)
        if not At25sfRenFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, capacity, ver = jedec[0:3]
        self._size = At25sfRenFlashDevice.SIZES[capacity]
        self._device = 'AT25SF'
        self._density_code = At25sfRenFlashDevice.MEMORY_DENSITY[capacity][ver]

    def __str__(self):
        return 'Renesas %s%s %s (%s)' % \
            (self._device, self._density_code, pretty_size(self._size, lim_m=1 << 20), super(At25sfRenFlashDevice, self).__str__())

    def unlock(self, sync):
        '''
        Unlock the flash for write. For XTX flash, we need to write both SR-1 and SR-2 together
        '''
        status_reg = self._read_status(sync=False)

        self._enable_write(sync=False)
        wcmd = Array('B', (At25sfFlashDevice.CMD_WRITE_STATUS_REG,
                               _Gen25FlashDevice.SR_WEL |
                               _Gen25FlashDevice.SR_PROTECT_NONE |
                               _Gen25FlashDevice.SR_UNLOCK_PROTECT,
                               status_reg))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()
        duration = self.get_timings('lock')
        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)
        status = self._read_status(sync=sync)
        if status & _Gen25FlashDevice.SR_PROTECT_ALL:
            raise SerialFlashRequestError("Cannot unprotect flash device")

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(True)
        time.sleep(0.05)

    def _write_status2(self, val, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(True)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg_2 = self._read_status2(sync=False)
        self._write_status2(status_reg_2 | 0b00000010, sync=sync)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], At25sfRenFlashDevice.__name__, At25sfRenFlashDevice)

class At25sfFlashDevice(_Gen25FlashDevice):
    """Atmel/Adesto AT25SF(rev A) flash device implementation"""

    CMD_WRITE_STATUS_REG = 0x01
    CMD_READ_STATUS_REG_2 = 0x35

    FEAT_WRITE_BOTH_STATUS_REG = True

    JEDEC_ID = 0x1F
    # For adesto flash, the 2nd Byte of JEDEC_ID has the infomation of flash capacity
    # while 3rd Byte of JEDEC_ID has only the product version information.
    DEVICES = { 0x85: 'AT25SF' }

    # Add new device capacity table
    SIZES = { 0x85 : 1 << 20}

    MEMORY_DENSITY = {
        0x85 : {
            0x01 : "081"
        }
    }

    SPI_FREQ_MAX = 55  # MHz    #Need to confirm

    TIMINGS = {'page': (0.0007, 0.005),      # 0.7/5 ms          # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.015, 0.03),
               'subsector': (0.060, 0.300),  # 60/300 ms         # 4k erase time
               'hsector': (0.3, 1.3),        # 0.3/1.3 s         # 32k erase time
               'sector': (0.5, 3),           # 0.5/3 s           # 64k erase time
               'lock': (0.015, 0.015)}       # 15/15 ms          #

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(At25sfFlashDevice, self).__init__(jtag_engine, jedec)
        if not At25sfFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, capacity, ver = jedec[0:3]
        self._size = At25sfFlashDevice.SIZES[capacity]
        self._device = 'AT25SF'
        self._density_code = At25sfFlashDevice.MEMORY_DENSITY[capacity][ver]

    def __str__(self):
        return 'Adesto %s%s %s (%s)' % \
            (self._device, self._density_code, pretty_size(self._size, lim_m=1 << 20), super(At25sfFlashDevice, self).__str__())

    def unlock(self, sync):
        '''
        Unlock the flash for write. For XTX flash, we need to write both SR-1 and SR-2 together
        '''
        status_reg_2 = self._read_status2(sync=False)

        self._enable_write(sync=False)
        wcmd = Array('B', (At25sfFlashDevice.CMD_WRITE_STATUS_REG,
                               _Gen25FlashDevice.SR_WEL |
                               _Gen25FlashDevice.SR_PROTECT_NONE |
                               _Gen25FlashDevice.SR_UNLOCK_PROTECT,
                               status_reg_2))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()
        duration = self.get_timings('lock')
        if any(duration):
            self._poll_for_completion(duration, self.SR_WIP)
        status = self._read_status(sync=sync)
        if status & _Gen25FlashDevice.SR_PROTECT_ALL:
            raise SerialFlashRequestError("Cannot unprotect flash device")

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val1, val2, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val1, val2))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(True)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg = self._read_status(sync=False)
        status_reg_2 = self._read_status2(sync=False)
        self._write_status(status_reg, status_reg_2 | 0b00000010, sync=sync)

        #status_reg = self._read_status_reg()
        #status_reg_2 = self._read_status_reg_2()

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], At25sfFlashDevice.__name__, At25sfFlashDevice)

class IS25LPWPFlashDevice(_Gen25FlashDevice):
    """ISSI IS25LP flash device implementation"""
    # IS25LP and IS25WP only differ in voltage supply(2.3V to 3.6V to 1.65V to 1.95V)

    CMD_ENTER_4BYTE = 0xB7
    CMD_EXIT_4BYTE = 0x29

    JEDEC_ID = 0x9D
    DEVICES = { 0x40: 'IS25LP', 0x60: 'IS25LP', 0x70: 'IS25WP'}
    MEMORY_DENSITY = {
        0x40 : {
            0x11 : "1",
            0x12 : "2",
            0x13 : "4",
        },
        0x60 : {
            0x14 : "8",
            0x15 : "16",
            0x16 : "32",
            0x17 : "64",
            0x18 : "128",
            0x19 : "256",
            0x1A : "512",
        },
        0x70 : {
            0x11 : "1",
            0x12 : "2",
            0x13 : "4",
            0x14 : "8",
            0x15 : "16",
            0x16 : "32",
            0x17 : "64",
            0x18 : "128",
            0x19 : "256",
            0x1A : "512",
        }
    }
    SIZES = {0x11: 4 << 15,
             0x12: 4 << 16,
             0x13: 4 << 17,
             0x14: 4 << 18,
             0x15: 4 << 19,
             0x16: 4 << 20,
             0x17: 4 << 21,
             0x18: 4 << 22,
             0x19: 4 << 23,
             0x1A: 4 << 24}

    SPI_FREQ_MAX = 55  # MHz

    TIMINGS = {'page': (0.0002, 0.0008),    # 0.2/0.8 ms        # Refer datasheet AC Characteristics for Program and Erase
               'write_status_register': (0.002, 0.015),
               # subsector min is 0.07(70ms) in json tho, however found 0.05(50ms) in datasheet
               'subsector': (0.05, 0.3),    # 50/300 ms         # 4k erase time
               'hsector': (0.1, 0.5),       # 0.1/0.5 s         # 32k erase time
               'sector': (0.15, 1),         # 0.15s/1 s         # 64k erase time
               'lock': (0.002, 0.0008)}     # 0.2/0.8 ms        # not found in datasheet(to be confirmed)
               #'chip': (80, 270)}

    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)
                #SerialFlash.FEAT_CHIPERASE)

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(IS25LPWPFlashDevice, self).__init__(jtag_engine, jedec)
        if not IS25LPWPFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        manufacturer, device, capacity = jedec[0:3]
        self._size = IS25LPWPFlashDevice.SIZES[capacity]
        self._device = IS25LPWPFlashDevice.DEVICES[device]
        self._density_code = IS25LPWPFlashDevice.MEMORY_DENSITY[device][capacity]

    def __str__(self):
        return 'ISSI %s%s %s (%s)' % \
            (self._device, self._density_code, pretty_size(self._size, lim_m=1 << 20), super(IS25LPWPFlashDevice, self).__str__())

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def _write_status(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def enable_quad_spi(self, sync):
        status_reg = self._read_status(sync=sync)
        self._write_status(status_reg | 0b01000000, sync=sync)

    def enter_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_ENTER_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync=sync)

    def exit_4byte_addr_mode(self, sync):
        cmd = Array('B', (self.CMD_EXIT_4BYTE,))
        cmd = reverseBytes(bytes(cmd))
        self._jtag_engine.write_dr_bytes(cmd)
        self._jtag_engine.go_idle2(sync)

    #def _erase_chip(self, command: int, times: Tuple[float, float]) -> None:
    #    """Erase an entire chip"""
    #    self._enable_write(sync=False)
    #    cmd: bytes  = bytes((command, 0))
    #    cmd = reverseBytes(bytes(cmd), 1)
    #    self._jtag_engine.write_dr_bytes(cmd)
    #    self._jtag_engine.go_idle()
    #    self._poll_for_completion(times, self.SR_WIP)
setattr(sys.modules['efx_pgm.jtag2SpiFlash'], IS25LPWPFlashDevice.__name__, IS25LPWPFlashDevice)


# XMC
class XM25QFlashDevice(_Gen25FlashDevice):
    """Winbond W25X flash device implementation"""

    JEDEC_ID = 0x20
    DEVICES = {0x40: 'XM25QH'}
    SIZES = {0x17: 8 << 20}
    SPI_FREQ_MAX = 104  # MHz
    CMD_READ_UID = 0x4B
    UID_LEN = 0x8  # 64 bits
    READ_UID_WIDTH = 4  # 4 dummy bytes
    TIMINGS = {'page': (0.0005, 0.003),  # 1.5/3 ms
               'subsector': (0.04, 0.4),  # 200/200 ms
               'hsector'  : (0.12, 0.9),     # 0.30/1.6 s     # 32k erase time
               'sector': (0.25, 2),  # 1/1 s
               'write_status_register': (0.001, 0.05),
               'lock': (0.05, 0.1)  # 50/100 ms
              }
    # TODO 32KB erase should be supported according to datasheet
    FEATURES = (SerialFlash.FEAT_SECTERASE |
                SerialFlash.FEAT_HSECTERASE |
                SerialFlash.FEAT_SUBSECTERASE)
                #SerialFlash.FEAT_CHIPERASE)

    MEMORY_DENSITY = {
        0x40 : {
            0x17 : "64"}
    }

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(XM25QFlashDevice, self).__init__(jtag_engine, jedec)
        if not XM25QFlashDevice.match(jedec):
            raise SerialFlashUnknownJedec(jedec)
        device, capacity = jedec[1:3]
        self._device = self.DEVICES[device]
        self._size = self.SIZES[capacity]

    def __str__(self):
        return 'XMC %s%d %s (%s)' % \
            (self._device, len(self) >> 17,
             pretty_size(self._size, lim_m=1 << 20),
             super(XM25QFlashDevice, self).__str__())

    def _read_status2(self, sync):
        rcmd = Array('B', (self.CMD_READ_STATUS_REG_2,))
        rcmd = reverseBytes(bytes(rcmd))
        self._jtag_engine.write_dr_bytes(rcmd)
        data = self._jtag_engine.read_dr2(8)
        self._jtag_engine.go_idle2(sync=sync)
        data = data.tobytes(True)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register-2")
        return data[0]

    def _write_status(self, val, sync):
        self._enable_write(sync=False)

        # write nonvolative conf reg
        wcmd = Array('B', (self.CMD_WRSR, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)
        time.sleep(0.05)

    def _write_status2(self, val, sync):
        # write enable
        self._enable_write(sync=False)

        # write status reg
        wcmd = Array('B', (self.CMD_WRITE_STATUS_REG_2, val))
        wcmd = reverseBytes(bytes(wcmd))
        self._jtag_engine.write_dr_bytes(wcmd)
        self._jtag_engine.go_idle()

        duration = self.get_timings('write_status_register')
        if any(duration):
            self._poll_for_completion(duration, 2)

        # soft reset
        self._soft_reset(sync)

    def enable_quad_spi(self, sync):
        status_reg_2 = self._read_status2(sync=False)
        self._write_status2(status_reg_2 | 0b00000010, sync=sync)

    def _soft_reset(self, sync):
        # soft reset
        sreset_cmd = Array('B', (0x66,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=False)

        sreset_cmd = Array('B', (0x99,))
        sreset_cmd = reverseBytes(bytes(sreset_cmd))
        self._jtag_engine.write_dr_bytes(sreset_cmd)
        self._jtag_engine.go_idle2(sync=sync)

setattr(sys.modules['efx_pgm.jtag2SpiFlash'], XM25QFlashDevice.__name__, XM25QFlashDevice)


class GenericEfinixSupportedFlash(_Gen25FlashDevice):
    TIMINGS = {'page': (0.0015, 0.003),  # 1.5/3 ms
               'subsector': (0.1, 0.4),  # 200/200 ms
               'sector': (0.15, 2),  # 1/1 s
               'write_status_register': (0.01, 0.03),
               'bulk': (32, 64),  # seconds
               'lock': (0.05, 0.1)}  # 50/100 ms
               #'chip': (20, 100)}
    FEATURES = SerialFlash.FEAT_SECTERASE | SerialFlash.FEAT_SUBSECTERASE

    def __init__(self, jtag_engine: Jtag2SpiChainEngine, jedec):
        super(GenericEfinixSupportedFlash, self).__init__(jtag_engine, jedec)
        self.manufacturer_id, self.device_id, self.capacity_id = jedec[0:3]
        self._device = 'Unknown'
        self._size = 16 << 20	# Assume 128Mbit device

    def __str__(self):
        return 'Generic Flash Profile. JEDEC ID: 0x%02X%02X%02X' % (self.manufacturer_id, self.device_id, self.capacity_id)
