#!/usr/bin/env python3

# Copyright (C) 2019 Efinix Inc. All rights reserved.
#
# No portion of this code may be reused, modified or
# distributed in any way without the expressed written
# consent of Efinix Inc.
#
"""
Programs HEX file to Efinix device, over FTDI-based cable (FTDI USB to MPSSE Serial cable)

Supported modes:

	-- active
		-- this connects to SPI bus and programs SPI Flash device
		-- SPI MASTER:  FTDI cable
		-- SPI SLAVE:   SPI Flash Device
		-- flash address must be on erase sector boundary.  Usually just use 0x00
		-- optional:  erase entire flash or just minimum space needed for image
	-- passive
		-- this connects to SPI bus and programs Efinix device directly
		-- SPI MASTER:  FTDI cable
		-- SPI SLAVE:   Efinix Device
		-- optional, may connect additional FTDI pins for CRESET, CDONE, SS
			-- pinout
				-- SCLK   - ADBUS0 - ORANGE wire
				-- MOSI   - ADBUS1 - YELLOW wire
				-- MISO   - ADBUS2 - GREEN wire (unused)
				-- CS     - ADBUS3 - BROWN wire
				-- CRESET - ADBUS4 - GRAY wire
				-- CDONE  - ADBUS5 - PURPLE wire
	-- jtag
		-- this connects to 4-pin JTAG interface on Efinix Device (no TRST)
		-- IDCODE of Efinix Device read
		-- then, issue PROGRAM instruction and program device over JTAG interface
		-- lastly, issue final instruction to enter user mode


This was tested using C232HM-DDHSL-0 programming cable,
however other cables based on FTDI FT232H may also work.  Other FTDI chipsets that
support USB to MPSSE may also work, although you may need to update the args.url below.


T20F256 engineering board pinout:
GND      - BLACK
TCK      - ORANGE
TDI      - YELLOW
TDO      - GREEN
TMS      - BROWN
CRESET   - GRAY
CDONE    - PURPLE
HOLD     - WHITE
PORT_SEL - BLUE


SW requirements:
	Besides python3, this script needs pyspiflash python library installed.
	To do this:

		pip3 install pyspiflash

	dependent python libraries that will also be installed:
		pyusb
		pyserial
		pyftdi

	More documentation may be found at
		https://eblot.github.io/pyftdi/index.html
		https://github.com/eblot/pyspiflash


Linux specific requirements:

	Requires libusb-1.0 to be installed.  On Ubuntu / Debian distributions:

		sudo apt-get install libusb-1.0

	On Red Hat / CentOS distributions:

		sudo yum install libusb-1.0


	Also, FTDI cable will need to be enabled via UDEV system.  Example setup for FTDI FT232H based:

		#1)  sudo gedit /etc/udev/rules.d/80-efx-pgm.rules
		#2)  paste the following into your new file (or append to existing file), save, and close:
SUBSYSTEM=="usb", ATTRS{idVendor}=="0403", ATTRS{idProduct}=="6014", MODE="666", GROUP="plugdev"
		#3)  sudo udevadm control --reload-rules

		The efinity script "install_usb_driver.sh" also does this.

Windows specific requirements:

	To install USB driver for FTDI cable:

		#1)  Download and install zadig tool from https://zadig.akeo.ie/
		#2)  Run Zadig.
				USB ID should match that for FTDI device.  Example for FT232H:  0403 6014
				Driver selection should be "libusbk"
				Once everything set up correctly, click on "Replace Driver"



python library licensing:
    pyftdi:  ftdi.py licensed under LGPLv2, the rest MIT
    pyusb:   Apache 2.0
    pyserial:  MIT-ish
    pyspiflash:  MIT





Script tested 09/24/2018

"""

import sys
import os
import argparse
import binascii
import re
from io import StringIO
import json
from typing import Tuple, Union, Any
from time import sleep
from time import time as now
from array import array as Array
from hashlib import sha1
from pathlib import Path, PurePath
from configparser import ParsingError
from functools import reduce
import itertools
import tempfile
import shutil

from pyftdi.misc import pretty_size
from pyftdi.bits import BitSequence
from pyftdi.spi import SpiPort, SpiController
from pyftdi.usbtools import UsbToolsError
from spiflash.serialflash import SerialFlashManager
from spiflash.serialflash import SerialFlashUnknownJedec, SerialFlashTimeout, SerialFlashError

sys.path.append(os.path.join(os.environ['EFINITY_HOME'], 'debugger', 'bin'))
from efx_dbg.define import DebugProfileBuilder
from efx_dbg.jcf import JcfParser, JtagChainEngine
from efx_dbg.jtag import JtagManager, UnknownUrlError
from efx_dbg.flash import CORRECT_FL_JSON, FlashJtagManager, HandshakeError

import efx_pgm.util.bitstream_util as bitstream_util
from efx_pgm.header_processor import HeaderProcessor, UnknownWidthError, OutofRangeError

import efx_pgm.efinix_flash as efinix_flash
from efx_pgm.efx_hw_common.boards import PinoutProfile, EfxHwBoardProfileSelector
from efx_pgm.efx_hw_common.manager import EfxFTDIHwConnectionManager
from efx_pgm.usb_resolver import UsbResolver, RemoteDeviceResolver
from efx_pgm.efx_hw_common.device import EfxFTDIDevice
from efx_pgm.efx_hw_client.remote.jtag import JtagRemoteEngine
from efx_pgm.efx_hw_client.remote.spi import RemoteSpiController
from efx_pgm.util.config_util import HwToolsConfigUtil

from efx_pgm.jtag2SpiFlash import SerialFlashDetector, reverseBytes
import efx_pgm.jtag2SpiEfinixFlash as jtag2spi_efinix_flash

class FtdiProgramError(Exception):
    pass

class FtdiProgram:

    def __init__(self,
                 args,
                 jtag_program_freq=3E6,
                 spi_direct_program_freq=3E6,
                 spi_flash_program_freq=6E6,
                 console=None):
        self.input_file = args.input_file
        self.mode = args.mode
        self.output_file = args.output_file
        self.urls = args.urls
        self.xml = args.xml
        self.chip_num = int(args.num)
        self.address = int(args.address) if hasattr(args, 'address') else 0
        self.num_bytes = int(args.num_bytes) if hasattr(args, 'num_bytes') else -1
        self.burst_size = int(args.burst_size) if hasattr(args, 'burst_size')  else -1
        self.jtag_bridge_mode = args.jtag_bridge_mode if hasattr(args, 'jtag_bridge_mode') else 'all'
        self.jtag_program_freq = int(args.jtag_clock_freq) if hasattr(args, 'jtag_clock_freq') else int(jtag_program_freq)
        self.spi_direct_program_freq = spi_direct_program_freq
        self.spi_flash_program_freq = spi_flash_program_freq
        self.creset_before_jtag_config = False
        self.jtag_tap = 'efx'

        # (ASSERT_CRESET_DURING_FLASH_PROGRAMMING, CONTROL_HOLD_PIN, TOGGLE_PORT_SEL_PIN)
        self.toggles = [True, True, True]
        # global message handler
        self.console = console
        if hasattr(args, 'pinout'):
            self.pinout = args.pinout
        else:
            self.pinout = PinoutProfile()

    # if console is registered, send output there instead of cmd line
    def pgm_print(self, msg, error=False):
        if self.console:
            if error:
                self.console.write_error_msg(msg)
            else:
                self.console.write(msg)
        else:
            print(msg)

    # common steps to setup flash device on ftdi
    def setup_flash_device(self, spi: SpiPort, ctrl: SpiController) -> Union[Any, efinix_flash.GenericEfinixSupportedFlash]:
        try:
            if any(self.toggles):

                configure = {
                    "A": {
                        "used": False,
                        "pins": 0x00,
                        "direction": 0x00,
                        "init_val": 0x00,
                    },
                    "B": {
                        "used": False,
                        "pins": 0x00,
                        "direction": 0x00,
                        "init_val": 0x00,
                    },
                    "C": {
                        "used": False,
                        "pins": 0x00,
                        "direction": 0x00,
                        "init_val": 0x00,
                    },
                    "D": {
                        "used": False,
                        "pins": 0x00,
                        "direction": 0x00,
                        "init_val": 0x00,
                    }
                }

                def set_configure(default_bus: str, default_idx: int, init_val: int, pin: str = '', use_default: bool = False):
                    """Closure to modify `configure` dict above"""
                    if use_default:
                        bus, idx = default_bus, default_idx
                    else:
                        bus, idx = FtdiProgram.parse_pinout_to_bus_and_idx(pin)
                        if bus is None or idx is None:
                            bus, idx = default_bus, default_idx
                    configure[bus]["used"] = True
                    configure[bus]["pins"] |= 0x1 << idx
                    configure[bus]["direction"] |= 0x1 << idx
                    configure[bus]["init_val"] |= init_val << idx

                # configure pin that connect to CRESET_N as output; Default: ADBUS4 CRESETN
                if self.toggles[0]:
                    set_configure("A", 4, 0x0, pin=self.pinout.CRESET_N)

                # configure pin that connect to HOLD as GPIO Output; Default: ADBUS6 HOLD
                if self.toggles[1]:
                    set_configure("A", 6, 0x1, pin=self.pinout.HOLD) # init to high to disable hold to flash

                # configure ADBUS7 as GPIO Output (PORT_SEL control ... T20 internal board)
                if self.toggles[2]: # PORT_SEL Pin, only for some boards, keep it
                    set_configure("A", 7, 0x0, use_default=True) # set to low to enable SPI

                # spi flash voltage translator enable, for TJ180A484S devkit
                set_configure("A", 5, 0x0, pin=self.pinout.FL_ENA)

                for bus, payload in configure.items():
                    if not payload["used"]:
                        continue

                    if bus == "A":
                        ctrl.set_gpio_direction(payload["pins"],
                                                payload["direction"])
                        sleep(0.001)

                        # initial value
                        ctrl.write_gpio(payload["init_val"])
                        sleep(0.002)
                    elif bus == "B" or bus == "C" or bus == "D":
                        url = self.get_url_from_bus(bus)
                        gpio = EfxFTDIHwConnectionManager.get_controller(url, EfxFTDIHwConnectionManager.ConnectionType.GPIO)
                        gpio.open_from_url(self.get_url_from_bus(bus),
                                           direction=payload["direction"])
                        sleep(0.001)

                        gpio.write_port(payload["init_val"])
                        sleep(0.002)

            # Some devices need extra delay, so we just add a small 0.5 delay here.
            sleep(0.5)
            # soft reset
            wren_cmd = Array('B', (0x66,))
            spi.exchange(wren_cmd)

            wren_cmd = Array('B', (0x99,))
            spi.exchange(wren_cmd)
            try:
                flash = SerialFlashManager.get_flash_device(spi)
            except TypeError as exc:
                self.pgm_print(
                    'ERROR: Unknown error trying to read flash device, aborting.', error=True
                )
                return None

            return flash
        except SerialFlashUnknownJedec as exc:
            jedec = binascii.unhexlify(str(exc)[-7:-1])
            flash = efinix_flash.EfinixSupportedFlash(spi, jedec)
            if flash.__str__() != "":
                self.pgm_print('Found matching Flash profile from Efinix Flash JSON Database.')
                flash.set_spi_frequency(spi.frequency)
                return flash
            else:
                self.pgm_print(
                    'Unrecognized Flash device. Will use Generic Flash profile. Please contact support if you face any problem.'
                )
                flash = efinix_flash.GenericEfinixSupportedFlash(spi, jedec)
                flash.set_spi_frequency(spi.frequency)
                return flash

    # common steps to terminate connection to flash device on ftdi
    def close_flash_device(self, spi: SpiPort, ctrl: SpiController):

        # close spi
        ctrl.terminate()

        # release any GPIO's, configure as inputs
        pins = 0x00
        reset_val = 0x00
        release_val = 0x00
        if any(self.toggles):
            # TODO: Better way to control pins in different bus without affecting each other...
            configure = {
                "A": {
                    "used": False,
                    "pins": 0x00,
                    "direction": 0x00,
                    "init_val": 0x00,
                    "step_1_val": 0x00
                },
                "B": {
                    "used": False,
                    "pins": 0x00,
                    "direction": 0x00,
                    "init_val": 0x00,
                    "step_1_val": 0x00
                },
                "C": {
                    "used": False,
                    "pins": 0x00,
                    "direction": 0x00,
                    "init_val": 0x00,
                    "step_1_val": 0x00
                },
                "D": {
                    "used": False,
                    "pins": 0x00,
                    "direction": 0x00,
                    "init_val": 0x00,
                    "step_1_val": 0x00
                }
            }

            # We don't need this, as ctrl.terminate() will automatically set the CRESET back to high, and FPGA will reset and load program
            if self.toggles[0]:
                # Release CRESETN
                bus, idx = FtdiProgram.parse_pinout_to_bus_and_idx(
                    self.pinout.CRESET_N)
                if bus is None or idx is None:
                    bus, idx = "A", 4  # Default: ADBUS4

                configure[bus]["used"] = True
                configure[bus]["pins"] |= 0x1 << idx
                configure[bus]["direction"] |= 0x1 << idx  # Output
                configure[bus]["init_val"] |= 0x0 << idx  # LOW, reset
                configure[bus][
                    "step_1_val"] |= 0x1 << idx  # HIGH, release reset

                for bus, payload in configure.items():
                    if not payload["used"]:
                        continue

                    gpio = EfxFTDIHwConnectionManager.get_controller(self.get_url_from_bus(bus), EfxFTDIHwConnectionManager.ConnectionType.GPIO)
                    gpio.open_from_url(self.get_url_from_bus(bus),
                                       direction=payload["direction"])

                    gpio.write_port(payload["init_val"])  # Reset
                    sleep(0.003)

                    gpio.write_port(payload["step_1_val"])  # Release
                    sleep(0.5)

                    # PROG-366 configure all GPIOs as input before exit
                    gpio.set_direction(0x00, 0x00)
                    gpio.close()

    def get_flash_context_manager(self, spi: SpiPort, ctrl: SpiController):
        class FlashContextManager:
            """
            Helper context manager for flash
            Ensures flash are closed upon exit
            """
            def __init__(self, parent: FtdiProgram, spi: SpiPort, ctrl: SpiController):
                self.parent = parent
                self.spi = spi
                self.ctrl = ctrl

            def __enter__(self) -> Union[Any, efinix_flash.GenericEfinixSupportedFlash]:
                return self.parent.setup_flash_device(self.spi, self.ctrl)

            def __exit__(self, exc_type, exc_value, exc_traceback):
                self.parent.close_flash_device(self.spi, self.ctrl)

        return FlashContextManager(self, spi, ctrl)

    # command to read flash contents and write to file (HEX format)
    def spi_read_flash(self, start_address, read_length, output_file):

        spi_url, cs_count, cs_port = self.get_spi_info_from_pinout()
        if not spi_url:
            self.pgm_print(
                "Cannot get SPI url. Please check your board profile configuration.", error=True
            )
            raise FtdiProgramError('could not get SPI url')

        ctrl = EfxFTDIHwConnectionManager.get_controller(spi_url, EfxFTDIHwConnectionManager.ConnectionType.SPI)
        ctrl.configure(spi_url, cs_count=cs_count, debug=False)
        spi = ctrl.get_port(cs=cs_port, freq=self.spi_flash_program_freq, mode=0)

        outfile = output_file
        if not outfile:
            self.pgm_print(
                'No output file specified for read flash contents, aborting', error=True
            )
            raise FtdiProgramError('no output file specified')

        with self.get_flash_context_manager(spi, ctrl) as flash:

            if not flash:
                self.pgm_print('Aborting flash programming', error=True)
                raise FtdiProgramError('could not get flash device')

            self.pgm_print('Flash device: %s @ SPI freq %0.1f MHz' %
                        (flash, flash.spi_frequency / 1E6))

            mode_4byte = 0
            is_flash_larger_than_16MB = 0
            enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
            exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

            try:
                final_address = start_address + read_length
                mode_4byte = final_address > 16 * 1024 * 1024
                if '_size' in flash.__dict__:
                    if flash._size / (1<<20) > 16 :
                        is_flash_larger_than_16MB = 1
            except OutofRangeError:
                #For when 4byte mode flag return value other than 0 or 1
                pass
            except Exception as exc:
                self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

            if mode_4byte:
                if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                    flash.ADDRESS_MODE_4BYTE = True
                    self.pgm_print('Enable 4-byte address mode = True')
                    flash.enter_4byte_addr_mode()
                else:
                    self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                    self.pgm_print('Aborting flash programming', error=True)
                    raise FtdiProgramError("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB")

            jedec = SerialFlashManager.read_jedec_id(spi)
            nv_conf_reg = None
            need_restore_nv_conf_reg = False

            # Micon MT25Q only
            if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                nv_conf_reg = self.spi_read_nonvolatile_conf_reg(spi)

                # If the address mode is 4-bytes, change it to 3-bytes
                lsb = int(nv_conf_reg[0])
                if lsb & 1 == 0:
                    self.pgm_print("Overwrite Address mode to 3-bytes")
                    need_restore_nv_conf_reg = True
                    self.spi_write_nonvolatile_conf_reg(spi, [0xFF, 0xFF])

            num_bytes = read_length
            flash_address = start_address

            # Read from flash into buffer
            self.pgm_print(
                'Reading %s from flash @ %s ...' %
                (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
            delta = now()
            readdata = flash.read(flash_address, num_bytes)
            delta = now() - delta
            self.pgm_print('Finished read in %d seconds' % int(delta))

            readbytes = readdata.tobytes() if hasattr(readdata, 'tobytes') else readdata

            # write out data to file
            with open(outfile, 'w', newline='\n') as file:
                for curr_byte in readbytes:
                    print('%02X' % curr_byte, file=file)

            # Restore nv configuration reg
            # Micon MT25Q only
            if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                if need_restore_nv_conf_reg:
                    self.pgm_print("Restore Non-volative configuration register.")
                    self.spi_write_nonvolatile_conf_reg(spi, nv_conf_reg)

            if mode_4byte:
                if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                    self.pgm_print('Exit 4-byte address mode')
                    flash.exit_4byte_addr_mode()


    # Command to erase flash, with specific range
    def spi_erase_flash(self, start, length):
        spi_url, cs_count, cs_port = self.get_spi_info_from_pinout()
        if not spi_url:
            self.pgm_print(
                "Cannot get SPI url. Please check your board profile configuration.", error=True
            )
            raise FtdiProgramError('could not get SPI url')

        ctrl = EfxFTDIHwConnectionManager.get_controller(spi_url, EfxFTDIHwConnectionManager.ConnectionType.SPI)
        ctrl.configure(spi_url, cs_count=cs_count, debug=False)
        spi = ctrl.get_port(cs=cs_port,
                            freq=self.spi_flash_program_freq,
                            mode=0)

        with self.get_flash_context_manager(spi, ctrl) as flash:

            if not flash:
                self.pgm_print('Aborting flash programming', error=True)
                raise FtdiProgramError('could not get flash device')

            mode_4byte = 0
            is_flash_larger_than_16MB = 0
            enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
            exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

            try:
                final_address = start + length
                mode_4byte = final_address > 16 * 1024 * 1024
                if '_size' in flash.__dict__:
                    if flash._size / (1<<20) > 16 :
                        is_flash_larger_than_16MB = 1
            except OutofRangeError:
                #For when 4byte mode flag return value other than 0 or 1
                pass
            except Exception as exc:
                self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

            if mode_4byte:
                if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                    flash.ADDRESS_MODE_4BYTE = True
                    self.pgm_print('Enable 4-byte address mode = True')
                    flash.enter_4byte_addr_mode()
                else:
                    self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                    self.pgm_print('Aborting flash programming', error=True)
                    raise FtdiProgramError("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB")

            # need to figure out erase size... adjust flash space to erase as needed
            erase_min_size = flash.get_erase_size()

            if (length % erase_min_size) == 0:
                erase_length  = length
            else:
                erase_length = (int(length / erase_min_size) + 1) * erase_min_size

            # starting flash address for erase should start on sector boundary
            if (start % erase_min_size) != 0:
                ok_fa_low = int(start / erase_min_size) * erase_min_size
                ok_fa_high = ok_fa_low + erase_min_size
                raise ValueError(
                    'cannot erase address %s, not on sector boundary.  Nearest valid addresses: %s, %s'
                    % ("0x{:08x}".format(start), "0x{:08x}".format(ok_fa_low),
                    "0x{:08x}".format(ok_fa_high)))

            # length should be multiple of erase_min_size
            if (erase_length % erase_min_size) != 0:
                raise ValueError('erase length should be multiple of %s' %
                                ("0x{:08x}".format(erase_min_size)))

            status_reg = None
            need_restore_status_reg = False
            status_reg = self.spi_read_status_reg(spi)
            if status_reg & 0b11111100 != 0:
                # For Macronix flash, they use status register-6 for QE bit.
                # So even we fall into here, we need to make sure whether we really need to do unlocking
                if( (isinstance(flash, efinix_flash.Mx75uFlashDevice) or isinstance(flash, efinix_flash.Mx75lFlashDevice)) and
                    (status_reg & 0b00111100 == 0) ):
                    pass
                else:
                    self.pgm_print("Unlock all sectors for write.")
                    need_restore_status_reg = True

                    if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                        status_reg_2 = flash._read_status_reg_2()
                        flash._write_status_reg(0, status_reg_2)
                    else:
                        self.spi_write_status_reg(spi, 0)

            jedec = SerialFlashManager.read_jedec_id(spi)
            nv_conf_reg = None
            need_restore_nv_conf_reg = False

            # Micon MT25Q only
            if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                nv_conf_reg = self.spi_read_nonvolatile_conf_reg(spi)

                # If the address mode is 4-bytes, change it to 3-bytes
                lsb = int(nv_conf_reg[0])
                if lsb & 1 == 0:
                    self.pgm_print("Overwrite Address mode to 3-bytes")
                    need_restore_nv_conf_reg = True
                    self.spi_write_nonvolatile_conf_reg(spi, [0xFF, 0xFF])

            #erase_length = length
            start_erase = start

            # unlock flash
            flash.unlock()

            # erase flash
            self.pgm_print(
                'Erasing %s from flash @ %s (may take a while...)' %
                (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
            delta = now()
            flash.erase(address=start_erase, length=erase_length, verify=True)
            delta = now() - delta
            self.pgm_print('Finished erase in %d seconds' % int(delta))

            # Restore write-protect status reg value
            if need_restore_status_reg:
                self.pgm_print("Restore Write-protect status register.")

                if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                    status_reg_2 = flash._read_status_reg_2()
                    flash._write_status_reg(status_reg, status_reg_2)
                else:
                    self.spi_write_status_reg(spi, status_reg)

            # Micon MT25Q only
            if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                # Restore nv configuration reg
                if need_restore_nv_conf_reg:
                    self.pgm_print("Restore Non-volative configuration register.")
                    self.spi_write_nonvolatile_conf_reg(spi, nv_conf_reg)

            if mode_4byte:
                if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                    self.pgm_print('Exit 4-byte address mode')
                    flash.exit_4byte_addr_mode()

    def get_spi_info_from_pinout(self):
        """
        Return url, cs_count, cs_port for pyftdi API

        Refer to for the usage of those parameter
        https://eblot.github.io/pyftdi/api/spi.html
        """
        # Default value when no pinout profile provided
        url = self.urls[0]
        cs_count = 1
        cs_port = 0

        if self.pinout:
            if self.pinout.CCK == "FTDI_ADBUS0" and \
                self.pinout.CDI0 == "FTDI_ADBUS1" and \
                self.pinout.CDI1 == "FTDI_ADBUS2" and \
                "FTDI_ADBUS3" in self.pinout.SS:
                url = self.get_url_from_bus("A")
                cs_port = 0
            elif self.pinout.CCK == "FTDI_BDBUS0" and \
                self.pinout.CDI0 == "FTDI_BDBUS1" and \
                self.pinout.CDI1 == "FTDI_BDBUS2" and \
                "FTDI_BDBUS3" in self.pinout.SS:
                url = self.get_url_from_bus("B")
                cs_port = 0

        return url, cs_count, cs_port

    @staticmethod
    def parse_pinout_to_bus_and_idx(
            val: str) -> Tuple[Union[str, None], Union[int, None]]:
        """
        Return:
        bus: str in ["A", "B", "C", "D"] or None
        idx: int or None
        """
        pattern_str = r"FTDI_([ABC])DBUS(\d)"
        pattern = re.compile(pattern_str)
        match = re.search(pattern, val)

        bus, idx = None, None
        if match:
            bus = match.group(1)
            idx = int(match.group(2))
        return bus, idx

    def get_url_from_bus(self, bus: str) -> Union[str, None]:
        if bus == "A" and len(self.urls) > 0:
            return self.urls[0]
        elif bus == "B" and len(self.urls) > 1:
            return self.urls[1]
        elif bus == "C" and len(self.urls) > 2:
            return self.urls[2]
        elif bus == "D" and len(self.urls) > 3:
            return self.urls[3]
        return None

    def get_jtag_bus_from_pinout(self) -> str:
        bus = None
        if self.pinout:
            bus, _ = FtdiProgram.parse_pinout_to_bus_and_idx(self.pinout.TCK)

        if bus is None:
            bus = "B"  # Default connected to BDBUS
        return bus

    def get_spi_bus_from_pinout(self) -> str:
        bus = None
        if self.pinout:
            bus, _ = FtdiProgram.parse_pinout_to_bus_and_idx(self.pinout.CCK)

        if bus is None:
            bus = "A"  # Default connected to ADBUS
        return bus

    # main routine for active programming
    # call this if programming directly to FLASH device over SPI bus
    #   ie  FTDI - SPI master
    #       SPI FLASH - SPI slave
    #
    # much of this code was based on sample found at
    # https://github.com/eblot/pyspiflash/blob/master/spiflash/tests/serialflash.py
    #
    def spi_flash_program(self,
                          flash_address,
                          erase_entire_flash,
                          verify_after_write=True,
                          erase_before_write=True):

        # open SPI port, flash device
        url, cs_count, cs_port = self.get_spi_info_from_pinout()
        ctrl = EfxFTDIHwConnectionManager.get_controller(url, EfxFTDIHwConnectionManager.ConnectionType.SPI)
        self.pgm_print(f'SPI Active Programming on {url}')
        ctrl.configure(url, cs_count=cs_count, debug=False)
        spi = ctrl.get_port(cs=cs_port,
                            freq=self.spi_flash_program_freq,
                            mode=0)

        with self.get_flash_context_manager(spi, ctrl) as flash:
            if not flash:
                self.pgm_print('Aborting flash programming', error=True)
                raise FtdiProgramError('could not get flash device')

            self.pgm_print('Flash device: %s @ SPI freq %0.1f MHz' %
                        (flash, flash.spi_frequency / 1E6))

            # Process bitstream header
            is_bitstream = 0
            try:
                device_db = bitstream_util.get_device_db()
                hp = HeaderProcessor(self.input_file, None, device_db, None)
                hp.process_header()
                is_bitstream = 1
            except UnicodeDecodeError:
                #For when header contains non-ASCII displayable characters
                pass
            except Exception as exc:
                self.pgm_print("Warning: Detected general exception while processing bitstream header - %s" % exc)

            mode_4byte = 0
            is_flash_larger_than_16MB = 0
            enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
            exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

            try:
                num_bytes_in_bitstream = bitstream_util.get_bitstream_num_bytes(self.input_file)
                final_address = flash_address + num_bytes_in_bitstream
                mode_4byte = final_address > 16 * 1024 * 1024
                if '_size' in flash.__dict__:
                    if flash._size / (1<<20) > 16 :
                        is_flash_larger_than_16MB = 1
            except OutofRangeError:
                #For when 4byte mode flag return value other than 0 or 1
                pass
            except Exception as exc:
                self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

            if mode_4byte:
                if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                    flash.ADDRESS_MODE_4BYTE = True
                    self.pgm_print('Enable 4-byte address mode = True')
                    flash.enter_4byte_addr_mode()
                else:
                    self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                    self.pgm_print('Aborting flash programming', error=True)
                    raise FtdiProgramError("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB")

            if is_bitstream:
                # Determine if need to enable QE bit
                # If input hex file's prog width is 4, then we enable QE bit
                try:
                    prog_width = hp.get_prog_width()
                except UnknownWidthError:
                    #For when width cannot be determined from header
                    pass
                except Exception as exc:
                    self.pgm_print("Warning: Detected general exception '{}' while enabling QE bit".format(exc))
                else:
                    if prog_width == 4:
                        enable_quad_spi = getattr(flash, "enable_quad_spi", None)
                        if callable(enable_quad_spi):
                            flash.enable_quad_spi()
                    elif prog_width == 2:
                        enable_dual_spi = getattr(flash, "enable_dual_spi", None)
                        if callable(enable_dual_spi):
                            flash.enable_dual_spi()

                '''
                # Set 4byte address mode
                # get_4byte_mode should return 0 if 4byte flag not found in bitstream/non-bitstream.
                # So there is no need to gate with device I think. Or is it not?
                mode_4byte = 0
                is_flash_larger_than_16MB = 0
                try:
                    mode_4byte = hp.get_4byte_mode()
                    if '_size' in flash.__dict__:
                        if flash._size / (1<<20) > 16 :
                            is_flash_larger_than_16MB = 1
                except OutofRangeError:
                    #For when 4byte mode flag return value other than 0 or 1
                    pass
                except Exception as exc:
                    self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)
                else:
                    # ADDRESS_MODE_4BYTE default value is False if not set
                    if mode_4byte:
                        if is_flash_larger_than_16MB:
                            self.pgm_print("Detected 4Byte flag in bitstream. Will use 4byte operation to load")
                            flash.ADDRESS_MODE_4BYTE = True
                            self.pgm_print('Enable 4-byte address mode = True')
                        else:
                            self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                            self.pgm_print('Aborting flash programming', error=True)
                            return
                '''

            #end_if is_bitstream

            status_reg = None
            need_restore_status_reg = False
            status_reg = self.spi_read_status_reg(spi)
            if status_reg & 0b11111100 != 0:
                # For Macronix flash, they use status register-6 for QE bit.
                # So even we fall into here, we need to make sure whether we really need to do unlocking
                if( (isinstance(flash, efinix_flash.Mx75uFlashDevice) or isinstance(flash, efinix_flash.Mx75lFlashDevice)) and (status_reg & 0b00111100 == 0) ):
                    pass
                else:
                    self.pgm_print("Unlock all sectors for write.")
                    need_restore_status_reg = True

                    if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                        status_reg_2 = flash._read_status_reg_2()
                        flash._write_status_reg(0, status_reg_2)
                    else:
                        self.spi_write_status_reg(spi, 0)

            jedec = SerialFlashManager.read_jedec_id(spi)
            nv_conf_reg = None
            need_restore_nv_conf_reg = False

            # Micon MT25Q only
            if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                nv_conf_reg = self.spi_read_nonvolatile_conf_reg(spi)

                # If the address mode is 4-bytes, change it to 3-bytes
                lsb = int(nv_conf_reg[0])
                if lsb & 1 == 0:
                    self.pgm_print("Overwrite Address mode to 3-bytes")
                    need_restore_nv_conf_reg = True
                    self.spi_write_nonvolatile_conf_reg(spi, [0xFF, 0xFF])

            num_bytes = bitstream_util.get_bitstream_num_bytes(self.input_file)

            flash_size = len(flash)
            # need to figure out erase size... adjust flash space to erase as needed
            erase_min_size = flash.get_erase_size()
            if (num_bytes % erase_min_size) == 0:
                num_erase_bytes = num_bytes
            else:
                num_erase_bytes = (int(num_bytes / erase_min_size) + 1) * erase_min_size

            # starting flash address for new FPGA image should start on sector boundary as well
            if (flash_address % erase_min_size) != 0:
                ok_fa_low = int(flash_address / erase_min_size) * erase_min_size
                ok_fa_high = ok_fa_low + erase_min_size
                raise ValueError(
                    'cannot write image to address %s, not on sector boundary.  Nearest valid addresses: %s, %s'
                    % ("0x{:08x}".format(flash_address),
                    "0x{:08x}".format(ok_fa_low), "0x{:08x}".format(ok_fa_high)))

            # determine how much flash to erase (either entire flash or just as needed for this image)
            if erase_entire_flash:
                erase_length = flash_size
                start_erase = 0x00
            else:
                erase_length = num_erase_bytes
                start_erase = flash_address

            if erase_length > flash_size:
                raise ValueError(
                    'Specified image file (requires %d bytes in flash memory) is too large for flash device, which is limited to %d bytes'
                    % (erase_length, flash_size))

            # unlock flash
            flash.unlock()

            # erase flash
            if erase_before_write:
                self.pgm_print(
                    'Erasing %s from flash @ %s (may take a while...)' %
                    (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
                delta = now()
                flash.erase(address=start_erase, length=erase_length, verify=False)
                delta = now() - delta
                self.pgm_print('Finished erase in %d seconds' % int(delta))

            # main buffer
            hexbytes = Array('B')

            # open HEX file
            with open(self.input_file, 'r') as file:

                # read file by line
                line = file.readline()
                while line:

                    # convert line to binary, append to buffer
                    val = int(line[0:2], 16)
                    hexbytes.append(val)
                    line = file.readline()

            bufstr = hexbytes.tobytes()

            # write string buffer to flash
            self.pgm_print(
                'Writing %s to flash @ %s ...' %
                (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
            delta = now()
            flash.write(flash_address, bufstr)
            delta = now() - delta
            self.pgm_print('Finished write in %d seconds' % int(delta))

            # Verify after write
            if verify_after_write:
                # compute hash from original file contents (for flash verify later)
                wmd = sha1()
                wmd.update(bufstr)
                refdigest = wmd.hexdigest()

                # Read from flash into buffer
                self.pgm_print(
                    'Reading %s from flash @ %s ...' %
                    (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
                delta = now()
                readdata = flash.read(flash_address, num_bytes)
                delta = now() - delta
                self.pgm_print('Finished read in %d seconds' % int(delta))

                # compute hash from read data contents
                rmd = sha1()
                rmd.update(readdata.tobytes() if hasattr(readdata, 'tobytes') else readdata)
                newdigest = rmd.hexdigest()

                # do verification
                if refdigest == newdigest:
                    self.pgm_print('Flash verify successful')
                else:
                    self.pgm_print('ERROR: Flash verify unsuccessful... mismatch found',
                                error=True)

            # Restore write-protect status reg value
            if need_restore_status_reg:
                self.pgm_print("Restore Write-protect status register.")

                if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                    status_reg_2 = flash._read_status_reg_2()
                    flash._write_status_reg(status_reg, status_reg_2)
                else:
                    self.spi_write_status_reg(spi, status_reg)

            # Micon MT25Q only
            if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                # Restore nv configuration reg
                if need_restore_nv_conf_reg:
                    self.pgm_print("Restore Non-volative configuration register.")
                    self.spi_write_nonvolatile_conf_reg(spi, nv_conf_reg)

            if mode_4byte:
                if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                    self.pgm_print('Exit 4-byte address mode')
                    flash.exit_4byte_addr_mode()

            # Print finish msg
            self.pgm_print('\"SPI active\" programming...done')


    # main routine for passive programming
    # call this if programming directly to Trion over SPI bus
    #   ie  FTDI - SPI master
    #       Trion - SPI slave
    def spi_direct_program(self):

        spi_bus = self.get_spi_bus_from_pinout()
        spi_url = self.get_url_from_bus(spi_bus)

        if not spi_url:
            self.pgm_print(
                'Cannot get SPI url, Please check your board profile configuration', error=True
            )
            raise FtdiProgramError('could not get SPI url')
        self.pgm_print(f'SPI passive programming on {spi_url}')

        url, cs_count, cs_port = self.get_spi_info_from_pinout()
        self.pgm_print(url)
        # open SPI port
        spi = EfxFTDIHwConnectionManager.get_controller(url, EfxFTDIHwConnectionManager.ConnectionType.SPI)
        spi.configure(url, cs_count=cs_count, debug=False)
        slave = spi.get_port(cs=cs_port,
                             freq=self.spi_direct_program_freq,
                             mode=0)

        # Get HW fifo size, for efficiency
        # HACK: FIXME when update pyftdi
        fifo_size = 1024
        try:
            if isinstance(spi, RemoteSpiController):
                pass
            else:
                fifo_size = spi.ftdi.fifo_sizes[0]
        except AttributeError:
            # pyftdi version 0.29.2 hacks
            fifo_size = spi._ftdi.fifo_sizes[0]

        # configure ADBUS5 as GPIO Input (CDONE status)

        configure = {
            "A": {
                "used": False,
                "pins": 0x00,
                "direction": 0x00,
                "init_val": 0x00,
            },
            "B": {
                "used": False,
                "pins": 0x00,
                "direction": 0x00,
                "init_val": 0x00,
            },
            "C": {
                "used": False,
                "pins": 0x00,
                "direction": 0x00,
                "init_val": 0x00,
            },
            "D": {
                "used": False,
                "pins": 0x00,
                "direction": 0x00,
                "init_val": 0x00,
            }
        }

        # configure CRESETN
        bus, idx = FtdiProgram.parse_pinout_to_bus_and_idx(self.pinout.CRESET_N)
        if bus is None or idx is None:
            bus, idx = "A", 4  # Default ADBUS4 as GPIO Output (CRESET control)

        configure[bus]["used"] = True
        configure[bus]["pins"] |= 0x1 << idx
        configure[bus]["direction"] |= 0x1 << idx  # Output
        configure[bus]["init_val"] |= 0x1 << idx

        # configure CDONE
        bus, cdone_idx = FtdiProgram.parse_pinout_to_bus_and_idx(
            self.pinout.CONDONE)
        if bus is None or idx is None:
            bus, cdone_idx = "A", 5  # Default ADBUS5 as GPIO Input (CDONE status)

        configure[bus]["used"] = True
        configure[bus]["pins"] |= 0x1 << cdone_idx
        configure[bus]["direction"] |= 0x0 << cdone_idx  # Input
        configure[bus]["init_val"] |= 0x0 << cdone_idx

        if self.toggles[2]:  # PORT_SEL Pin, only for some boards, keep it
            bus, idx = "A", 7  # Default ADBUS7

            configure[bus]["used"] = True
            configure[bus]["pins"] |= 0x1 << idx
            configure[bus]["direction"] |= 0x1 << idx  # Output
            configure[bus]["init_val"] |= 0x0 << idx

        for bus, payload in configure.items():
            if not payload["used"]:
                continue

            if bus == "A":
                spi.set_gpio_direction(payload["pins"], payload["direction"])
                sleep(0.005)

                # initial value
                spi.write_gpio(payload["init_val"])
                sleep(0.005)
            elif bus == "B" or bus == "C" or bus == "D":
                url = self.get_url_from_bus(bus)
                gpio = EfxFTDIHwConnectionManager.get_controller(url, EfxFTDIHwConnectionManager.ConnectionType.GPIO)
                gpio.open_from_url(url,
                                   direction=payload["direction"])
                sleep(0.001)

                gpio.write_port(payload["init_val"])
                sleep(0.002)

        def read_pin(bus, idx) -> bool:
            if bus == "A":
                return spi.read_gpio() & (0x1 << idx)
            elif bus == "B" or bus == "C" or bus == "D":
                url = self.get_url_from_bus(bus)
                gpio = EfxFTDIHwConnectionManager.get_controller(url, EfxFTDIHwConnectionManager.ConnectionType.GPIO)
                gpio.open_from_url(url,
                                   direction=(0x0 << idx))
                sleep(0.001)
                return gpio.read_port() & (0x1 << idx)

        def toggle_pin(bus, idx, high: bool):
            val = 0x0 << idx
            if high:
                val = 0x1 << idx

            if bus == "A":
                spi.write_gpio(val)
            elif bus == "B" or bus == "C" or bus == "D":
                url = self.get_url_from_bus(bus)
                gpio = EfxFTDIHwConnectionManager.get_controller(url, EfxFTDIHwConnectionManager.ConnectionType.GPIO)
                gpio.open_from_url(url,
                                   direction=(0x1 << idx))
                sleep(0.001)
                gpio.write_port(val)

        def toggle_cresetn(high: bool):
            bus, idx = FtdiProgram.parse_pinout_to_bus_and_idx(
                self.pinout.CRESET_N)
            if bus is None or idx is None:
                bus, idx = "A", 4  # Default ADBUS4 as GPIO Output (CRESET control)
            toggle_pin(bus, idx, high)

        def read_cdone():
            bus, idx = FtdiProgram.parse_pinout_to_bus_and_idx(
                self.pinout.CONDONE)
            if bus is None or idx is None:
                bus, idx = "A", 5  # Default ADBUS5 as GPIO Input (CDONE status)
            return read_pin(bus, idx)

        # Measure current CDONE before programming
        cdone_val = read_cdone()
        # self.pgm_print('CDONE status before programming: %d' % cdone_val)
        self.pgm_print('Programming direct on SPI at freq %0.1f MHz' %
                       (self.spi_direct_program_freq / 1E6))

        '''
        Allow user to use define CRESET pulse width using ENV variable
        '''
        env = os.environ.get('SPI_PASSIVE_CRESETN_PULSE_WIDTH')
        if env is None:
            pulse_low = 0.3
            pulse_high = 0.15
        else:
            try:
                pulse_low = float(env)
                pulse_high = float(env)
            except (ValueError, TypeError):
                pulse_low = 0.3
                pulse_high = 0.15

        spibuf = bytearray()

        # open HEX file
        with open(self.input_file, 'r') as file:

            # assert CRESET, wait a few milliseconds
            toggle_cresetn(False)

            sleep(pulse_low)

            # start SPI transaction
            # This forces CSb low
            slave.write([0x00], start=True, stop=False)
            sleep(0.05)

            # release CRESET
            # With CSb in low state, Trion device will be in
            # passive programming mode
            toggle_cresetn(True)
            sleep(pulse_high)

            # read file by line
            line = file.readline()
            while line:

                # convert line to binary, append to buffer
                val = int(line[0:2], 16)
                spibuf.append(val)

                # once buffer has reached fifo_size
                # send a SPI write transaction
                # CSb remains low for entirety of programming operation
                if len(spibuf) == fifo_size:
                    slave.write(spibuf, start=False, stop=False)
                    spibuf = bytearray()
                line = file.readline()

            # write out remaining bits, if any
            if len(spibuf) > 0:
                slave.write(spibuf, start=False, stop=False)

            # send extra zero's (extra clock edges) to finish programming
            spibuf = bytearray()
            for i in range(0, 10000):
                spibuf.append(0)
                if len(spibuf) == fifo_size:
                    slave.write(spibuf, start=False, stop=False)
                    spibuf = bytearray()

            # stop SPI transaction
            # This will pull CSb back high, and finish SPI programming
            slave.write([0x00], start=False, stop=True)

        # check that CDONE is successful
        sleep(0.05)
        cdone_val = read_cdone()
        # self.pgm_print('CDONE status after programming: %d' % cdone_val)

        self.pgm_print('Finished SPI direct program')
        # close spi port
        spi.terminate()

    def spi_read_status_reg(self, spi):
        wren_cmd = Array('B', (0x05,))
        data = spi.exchange(wren_cmd, 1)

        if len(data) != 1:
            raise SerialFlashTimeout("Unable to read status register")
        return data[0]

    def spi_write_status_reg(self, spi, reg_val):
        # write enable
        wren_cmd = Array('B', (0x06,))
        spi.exchange(wren_cmd)

        # write nonvolative conf reg
        wrsr_cmd = Array('B', (0x01, reg_val))
        spi.exchange(wrsr_cmd)

        # Wait for the write to complete
        start = now()
        while self.spi_read_status_reg(spi) & 2:
            if now() - start >= 1:
                self.pgm_print(
                    'spi_write_status_reg: Timed out while setting status register.'
                )

        # soft reset
        wren_cmd = Array('B', (0x66,))
        spi.exchange(wren_cmd)

        wren_cmd = Array('B', (0x99,))
        spi.exchange(wren_cmd)

        sleep(0.05)

    def spi_read_nonvolatile_conf_reg(self, spi):
        wren_cmd = Array('B', (0xb5,))
        data = spi.exchange(wren_cmd, 2)

        if len(data) != 2:
            raise SerialFlashTimeout(
                "Unable to retrieve nonvolative configuration status")
        return data

    # data should only contains 2 bytes
    def spi_write_nonvolatile_conf_reg(self, spi, data):
        # write enable
        wren_cmd = Array('B', (0x06,))
        spi.exchange(wren_cmd)

        # write nonvolative conf reg
        wrsr_cmd = Array('B', (0xb1, data[0], data[1]))
        spi.exchange(wrsr_cmd)

        # Wait for the write to complete
        start = now()
        while self.spi_read_status_reg(spi) & 2:
            if now() - start >= 1:
                self.pgm_print(
                    'spi_write_nonvolatile_conf_reg: Timed out while setting nonvolatile config register.'
                )

        # soft reset
        wren_cmd = Array('B', (0x66,))
        spi.exchange(wren_cmd)

        wren_cmd = Array('B', (0x99,))
        spi.exchange(wren_cmd)

        sleep(1)

    # use JTAG API, program through JTAG pins
    def jtag_program(self, chain=False):
        self.pgm_print("jtag programming started!")

        jtag_bus = self.get_jtag_bus_from_pinout()
        jtag_url = self.get_url_from_bus(jtag_bus)
        if not jtag_url:
            self.pgm_print(
                'Cannot get JTAG url, Please check your board profile configuration', error=True
            )
            raise FtdiProgramError('could not get JTAG url')

        self.jtag_tap = 'efx'
        try:  # Fix for HD-26, get engine if debugger is already running with same url
            jtag, _, _ = JtagManager().get_configured_jtag(jtag_url)
            if jtag:
                self.pgm_print(
                    'JTAG (url={}) already in use! '
                    'Try stopping debuggers before programming. Skipping...'.
                    format(jtag_url), error=True)
                raise FtdiProgramError('jtag url already in use')
        except UnknownUrlError:
            pass  # Expecting error normally, go on

        jtag = None
        if chain:
            jcfs = JcfParser(self.xml).all_devices()
            chip_num = self.chip_num
            self.pgm_print(f'JTAG Chain Programming on {jtag_url}, target: {chip_num}')
        else:
            self.pgm_print(f'JTAG Programming on {jtag_url}')
            jcfs, chip_num = None, 1

        engine_provider = EfxFTDIHwConnectionManager.get_controller(jtag_url, EfxFTDIHwConnectionManager.ConnectionType.JTAG, trst=False, frequency=self.jtag_program_freq)
        jtag = JtagChainEngine(devices=jcfs,
                               target_num=chip_num,
                               engine_provider=engine_provider)

        self.jtag_tap = JtagManager().detect_jtag_tap(jtag_url, jcfs=jcfs, chip_num=chip_num, freq=self.jtag_program_freq)
        JTAG_INSTR = JtagManager()._tap_to_instr(self.jtag_tap)

        jtag.configure(jtag_url)

        # Get HW fifo size, for efficiency
        # HACK: FIXME when update pyftdi
        fifo_size = 1024
        try:
            if isinstance(engine_provider, JtagRemoteEngine):
                pass
            else:
                fifo_size = engine_provider._ctrl.ftdi.fifo_sizes[0]
        except AttributeError:
            # pyftdi version 0.29.2 hacks
            fifo_size = engine_provider._ctrl._ftdi.fifo_sizes[0]

        EfxFTDIHwConnectionManager.set_jtagcontroller_pipe_len(engine_provider, fifo_size)

        # start from reset
        jtag.reset()

        # read IDCODE
        jtag.write_ir(JTAG_INSTR['IDCODE'])
        idcode = jtag.read_dr(32)
        jtag.go_idle()
        sleep(0.1)

        # close JTAG connection, will re-open a new one later
        jtag.close()
        sleep(0.01)

        #fifo_size = 1024
        #pgm_print('fifo size: %d' % fifo_size)
        #pgm_print('current URL: %s' % args.url)
        self.pgm_print('Programming \'%s\' via JTAG at freq %0.1f MHz' %
                       (self.input_file, self.jtag_program_freq / 1E6))

        gpio = None
        configure = {
            bus: {
                "used": False,
                "pins": 0x00,
                "direction": 0x00,
                "init_vals": [0x00, 0x00, 0x00],
                "gpio": None
            } for bus in ["A", "B", "C", "D"]
        }

        # PROG-440 Xyloni fix: Xyloni board uses ADBUS7 as FT_SPI_ENA signal
        # PROG-452: always enable ADBUS7 control regardless of AN08/AN20 or .ini setting
        port_sel_bus, port_sel_idx = "A", 7  # Default: ADBUS7
        configure[port_sel_bus]["used"] = True
        configure[port_sel_bus]["pins"] |= 0x1 << port_sel_idx
        configure[port_sel_bus]["direction"] |= 0x1 << port_sel_idx  # Output
        configure[port_sel_bus]["init_vals"][0] |= 0x0 << port_sel_idx  # PORT_SEL to SPI
        configure[port_sel_bus]["init_vals"][1] |= 0x0 << port_sel_idx
        configure[port_sel_bus]["init_vals"][2] |= 0x1 << port_sel_idx  # PORT_SEL to JTAG

        # Only do this if the device is T4/T8/T13/T20, which idcode ix 0x0 or 0x210a79, or .ini setting is set
        if( (int(idcode) == 0) or int(idcode) == 2165369 or self.creset_before_jtag_config is True):

            # configure CRESETN
            cresetn_bus, cresetn_idx = FtdiProgram.parse_pinout_to_bus_and_idx(
                self.pinout.CRESET_N)
            if cresetn_bus is None or cresetn_idx is None:
                cresetn_bus, cresetn_idx = "A", 4  # Default ADBUS4

            configure[cresetn_bus]["used"] = True
            configure[cresetn_bus]["pins"] |= 0x1 << cresetn_idx
            configure[cresetn_bus]["direction"] |= 0x1 << cresetn_idx  # Output
            configure[cresetn_bus]["init_vals"][0] |= 0x0 << cresetn_idx  # CRESETN low
            configure[cresetn_bus]["init_vals"][1] |= 0x1 << cresetn_idx  # release CRESETN
            configure[cresetn_bus]["init_vals"][2] |= 0x1 << cresetn_idx

            # configure SS
            # use SS_JTAG as workaround for FT232 cable profile. PROG-395
            if self.pinout.SS_JTAG != "":
                ss_pins = self.pinout.SS_JTAG.split("/")
            else:
                # this split only seems relevant to isx_prog_cable.json board profile
                # which uses "SS": "FTDI_BDBUS5/FTDI_ADBUS3"
                ss_pins = self.pinout.SS.split("/")
            for pin in ss_pins:
                ss_bus, ss_idx = FtdiProgram.parse_pinout_to_bus_and_idx(
                    pin)
                if ss_bus is None or ss_idx is None:
                    ss_bus, ss_idx = "A", 3  # Default ADBUS3

                configure[ss_bus]["used"] = True
                configure[ss_bus]["pins"] |= 0x1 << ss_idx
                configure[ss_bus]["direction"] |= 0x1 << ss_idx  # Output
                configure[ss_bus]["init_vals"][0] |= 0x0 << ss_idx  # SS always LOW
                configure[ss_bus]["init_vals"][1] |= 0x0 << ss_idx
                configure[ss_bus]["init_vals"][2] |= 0x0 << ss_idx

        # PROG-444 fix: only add to used_busses if url/drivers are present
        used_busses = set()
        for bus in ["A","B","C","D"]:
            if configure[bus]["used"]:
                url = self.get_url_from_bus(bus)
                if url is None:
                    continue
                used_busses.add(bus)
                configure[bus]["gpio"] = EfxFTDIHwConnectionManager.get_controller(url, EfxFTDIHwConnectionManager.ConnectionType.GPIO)
                configure[bus]["gpio"].open_from_url(url, direction=configure[bus]["direction"])

        # Configure all channels together for timing performance,
        # partically for case when CRESET_N and SS_N is located at different channel. (Eg, ISX cable with RSV4 as SS_N)
        # Refer to PROG-383 for details
        for step in range(4):
            sleep(0.003)
            for bus in used_busses:
                if step == 3:
                    configure[bus]["gpio"].close()
                    configure[bus]["gpio"] = None
                    continue
                configure[bus]["gpio"].write_port(configure[bus]["init_vals"][step])

        # create new jtag
        jtag = None
        engine_provider = EfxFTDIHwConnectionManager.get_controller(jtag_url, EfxFTDIHwConnectionManager.ConnectionType.JTAG, trst=False, frequency=self.jtag_program_freq)
        jtag = JtagChainEngine(devices=jcfs,
                               target_num=self.chip_num,
                               engine_provider=engine_provider)
        jtag.configure(jtag_url)
        jtag.reset()

        # go to idle state
        jtag.go_idle()
        sleep(0.1)

        # it's embarrassing to print the T8/T4 idcode...
        if int(idcode) != 0:
            self.pgm_print("Device ID read from JTAG: 0x%08X" % int(idcode))

        # Check that idcode is consistent with the JCF file
        if chain and JcfParser(self.xml).get_device(chip_num)['id_code'].upper(
        ) != ('0x%08X' % int(idcode)).upper():
            raise AssertionError(
                'Read IDCODE does not match JTAG chain file IDCODE')

        # Enter programming mode
        jtag.write_ir(JTAG_INSTR['PROGRAM'])

        # PROG-236
        # Issue PROGRAM instruction twice for certain Titanium devices
        if (int(idcode) == 0x10660A79) or (int(idcode) == 0x10661A79) or (int(idcode) == 0x1067FA79):
            sleep(0.01)
            jtag.reset()
            jtag.write_ir(JTAG_INSTR['PROGRAM'])

        # open HEX file
        with open(self.input_file, 'r') as file:

            # shift JTAG state to SHIFT_DR and don't let it change for duration of
            # programming
            jtag.change_state('shift_dr')

            # read file by line
            line = file.readline()

            msb_val = False
            use_last = False
            is_first = True

            buf = ''
            # pad front of bitstream with some extra bytes to give control
            # block plenty of time to initialize
            for i in range(0, 20):
                buf += ('0' * 8)

            while line:

                # convert line to binary, append to buffer
                buf += '{0:08b}'.format(int(line[0:2], 16))
                buflen = len(buf)
                if (buflen % 8) != 0:
                    raise ValueError('incorrect JTAG bit sequencing')

                # once buffer has reached fifo_size, send data
                if int(buflen / 8) >= fifo_size:
                    jtag.write(BitSequence(buf, length=len(buf), msb=msb_val),
                               first=is_first,
                               last=False,
                               use_last=use_last)
                    is_first = False
                    buf = ''
                line = file.readline()

            # write out remaining bits, if any
            buflen = len(buf)
            bytelen = int(buflen / 8)
            #pgm_print('final bytes length = %d, fifo_size = %d' % (bytelen, fifo_size))
            for i in range(bytelen, fifo_size):
                buf += ('0' * 8)
            jtag.write(BitSequence(buf, length=len(buf), msb=msb_val),
                       first=is_first,
                       last=False,
                       use_last=use_last)

        # Send to user mode
        sleep(0.1)
        jtag.write_ir(JTAG_INSTR['ENTERUSER'])
        jtag.write_dr(
            BitSequence('0' * 100, length=100,
                        msb=True))  # extra clock ticks after ENTERUSER as well
        jtag.write_ir(JTAG_INSTR['IDCODE'])

        # wait a bit for all operations to flush, then close the jtag channel
        jtag.reset()
        sleep(0.2)
        jtag.close()

        if gpio:
            gpio.close()
            gpio = None

        self.pgm_print('... finished with JTAG programming')

        return self.jtag_tap


    # use JTAG API, read JTAG IDCode
    def jtag_read_idcode(self, jtag_chain_devices, jtag_chain_target_num):

        jtag_bus = self.get_jtag_bus_from_pinout()
        jtag_url = self.get_url_from_bus(jtag_bus)
        if not jtag_url:
            self.pgm_print(
                'Cannot get JTAG url, Please check your board profile configuration', error=True
            )
            return

        tap = JtagManager().detect_jtag_tap(jtag_url, jcfs=jtag_chain_devices, chip_num=jtag_chain_target_num, freq=self.jtag_program_freq)
        engine_provider = EfxFTDIHwConnectionManager.get_controller(jtag_url, EfxFTDIHwConnectionManager.ConnectionType.JTAG, frequency=self.jtag_program_freq)
        jtag = JtagChainEngine(devices=jtag_chain_devices,
                        target_num=jtag_chain_target_num, engine_provider=engine_provider, tap=tap)
        jtag.configure(jtag_url)
        jtag.reset()
        jtag.write_ir(JtagManager()._tap_to_instr(tap=tap)['IDCODE'])
        jid = jtag.read_dr(32)
        jtag.close()
        return jid

    def jtag_bridge(self):
        # If this is a "read/erase" mode, ignore the input_file argument
        # This is because we want to avaoid passed in num_user_bytes to be overwritten
        if( self.jtag_bridge_mode == 'read' or self.jtag_bridge_mode == 'erase' ):
            self.input_file = ''

        jtag_bus = self.get_jtag_bus_from_pinout()
        jtag_url = self.get_url_from_bus(jtag_bus)

        jtag_session = None
        flash_session = None

        try:
            fjtag = FlashJtagManager()
            if not self.xml:
                tap = fjtag.detect_jtag_tap(url=jtag_url,
                                            freq=self.jtag_program_freq)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = fjtag.config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap)
            else:
                jcfs = JcfParser(self.xml).all_devices()

                tap = fjtag.detect_jtag_tap(url=jtag_url,
                                        jcfs=jcfs,
                                        chip_num=self.chip_num,
                                        freq=self.jtag_program_freq)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = fjtag.config_jtag(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num, freq=self.jtag_program_freq, tap=tap)

            flash_loader_profile = DebugProfileBuilder.from_str(CORRECT_FL_JSON)
            flash_session = fjtag.config_debug(jtag_session, flash_loader_profile)

            if self.jtag_bridge_mode == 'read':
                fjtag.fl_flash_read(dbg_sesh=flash_session, core_name='FL_0', output_file=self.output_file, address=self.address,
                                    num_user_bytes=self.num_bytes, burst_size=self.burst_size)
            elif self.jtag_bridge_mode == 'erase':
                fjtag.fl_flash_erase(dbg_sesh=flash_session, core_name='FL_0', address=self.address,
                                    num_user_bytes=self.num_bytes)
            else:
                fjtag.fl_flash_load(dbg_sesh=flash_session, core_name='FL_0', input_file=self.input_file, output_file=self.output_file, address=self.address,
                                    num_user_bytes=self.num_bytes, burst_size=self.burst_size, mode=self.jtag_bridge_mode)
        except HandshakeError:
            self.pgm_print('ERROR: Flash loader soft core not working/nonexistent', error=True)
            raise FtdiProgramError('Flash loader soft core not working/nonexistent')
        except (SerialFlashUnknownJedec, TypeError, SerialFlashError) as exc:
            self.pgm_print('ERROR: FTDI communication with SPI flash is corrupted', error=True)
            raise FtdiProgramError('FTDI communication with SPI flash is corrupted')
        finally:
            if flash_session:
                fjtag.unconfig_debug(flash_session)
                flash_session = None

            if jtag_session:
                fjtag.unconfig_jtag(jtag_session)
                jtag_session = None

    def jtag_bridge_x8(self):
        # If this is a "read/erase" mode, ignore the input_file argument
        # This is because we want to avaoid passed in num_user_bytes to be overwritten
        if( self.jtag_bridge_mode == 'read' or self.jtag_bridge_mode == 'erase' ):
            self.input_file = ''

        jtag_bus = self.get_jtag_bus_from_pinout()
        jtag_url = self.get_url_from_bus(jtag_bus)

        jtag_session = None
        flash_session = None

        try:
            fjtag = FlashJtagManager()
            if not self.xml:
                tap = fjtag.detect_jtag_tap(url=jtag_url,
                                                  freq=self.jtag_program_freq)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = fjtag.config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap)
            else:
                jcfs = JcfParser(self.xml).all_devices()

                tap = fjtag.detect_jtag_tap(url=jtag_url,
                                        jcfs=jcfs,
                                        chip_num=self.chip_num,
                                        freq=self.jtag_program_freq)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = fjtag.config_jtag(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num, freq=self.jtag_program_freq, tap=tap)

            flash_loader_profile = DebugProfileBuilder.from_str(CORRECT_FL_JSON)
            flash_session = fjtag.config_debug(jtag_session, flash_loader_profile)

            if self.jtag_bridge_mode == 'read':
                fjtag.fl_flash_read_x8(dbg_sesh=flash_session, core_name='FL_0', output_file=self.output_file, address=self.address,
                                        num_user_bytes=self.num_bytes, burst_size=self.burst_size)
            elif self.jtag_bridge_mode == 'erase':
                fjtag.fl_flash_erase_x8(dbg_sesh=flash_session, core_name='FL_0', address=self.address,
                                        num_user_bytes=self.num_bytes)
            else:
                fjtag.fl_flash_load_x8(dbg_sesh=flash_session, core_name='FL_0', input_file=self.input_file, output_file=self.output_file, address=self.address,
                                        num_user_bytes=self.num_bytes, burst_size=self.burst_size, mode=self.jtag_bridge_mode)
        except HandshakeError:
            self.pgm_print('ERROR: Flash loader soft core not working/nonexistent', error=True)
            raise FtdiProgramError('Flash loader soft core not working/nonexistent')
        except (SerialFlashUnknownJedec, TypeError, SerialFlashError) as exc:
            self.pgm_print('ERROR: FTDI communication with SPI flash is corrupted', error=True)
            raise FtdiProgramError('FTDI communication with SPI flash is corrupted')
        finally:
            if flash_session:
                fjtag.unconfig_debug(flash_session)
                flash_session = None

            if jtag_session:
                fjtag.unconfig_jtag(jtag_session)
                jtag_session = None

    ###########################################################################################################################################################
    # jtag2spi

    # top-level control for when jtag_bridge_new is selected
    def jtag2spi(self):

        jtag_bus = self.get_jtag_bus_from_pinout()
        jtag_url = self.get_url_from_bus(jtag_bus)
        if self.xml:
            jcfs = JcfParser(self.xml).all_devices()
            tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
        else:
            tap = JtagManager().detect_jtag_tap(url=jtag_url)

        if self.jtag_bridge_mode == 'read':
            self.jtag2spi_read_flash(start_address=self.address,
                                     read_length=self.num_bytes,
                                     output_file=self.output_file,
                                     setup=True,
                                     tap=tap)
        elif self.jtag_bridge_mode == 'erase':
            self.jtag2spi_erase_flash(start=self.address,
                                      length=self.num_bytes,
                                      setup=True,
                                      tap=tap)
        elif self.jtag_bridge_mode == 'write':
            self.jtag2spi_program(setup=True,
                                  flash_address=self.address,
                                  erase_entire_flash=False,
                                  tap=tap,
                                  verify_after_write=False,
                                  erase_before_write=False)
        elif self.jtag_bridge_mode == 'erase_and_write':
            self.jtag2spi_program(setup=True,
                                  flash_address=self.address,
                                  erase_entire_flash=False,
                                  tap=tap,
                                  verify_after_write=False,
                                  erase_before_write=True)
        elif self.jtag_bridge_mode == 'all_no_erase':
            self.jtag2spi_program(setup=True,
                                  flash_address=self.address,
                                  erase_entire_flash=False,
                                  tap=tap,
                                  verify_after_write=True,
                                  erase_before_write=False)
        else: # 'all'
            self.jtag2spi_program(setup=True,
                                  flash_address=self.address,
                                  erase_entire_flash=False,
                                  tap=tap,
                                  verify_after_write=True,
                                  erase_before_write=True)

    # TODO provide gui option to tristate spi signal
    def jtag2spi_set_spiclk_tristate(tristate: bool, setup):
        pass

    def jtag2spi_set_lower_flash_tristate():
        pass

    def jtag2spi_set_upper_flash_tristate():
        pass

    def jtag2spi_read_flash(self, start_address, read_length, output_file, setup, tap='efx', selected_flash='lower', is_dual=False):
        jtag_session = None
        jedec = None
        error = None
        try:
            jtag_bus = self.get_jtag_bus_from_pinout()
            jtag_url = self.get_url_from_bus(jtag_bus)

            jcfs, chip_num = None, 1
            if self.xml:
                jcfs = JcfParser(self.xml).all_devices()
                #tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, jcfs=jcfs, chip_num=self.chip_num, use_custom_engine=True)
            else:
                #tap = JtagManager().detect_jtag_tap(url=jtag_url)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, use_custom_engine=True)

            with jtag_session.get_engine() as jtag_engine:
                if setup == True:
                    jtag_engine.setup()
                else:
                    # even selected not to do setup, we just do it anyway if read jedec id failed
                    jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                    if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)) or ((jedec[0] == 0x00) and (jedec[1] == 0x00) and (jedec[2] == 0x00)):
                        jtag_engine.setup()

                if selected_flash == 'lower':
                    jtag_engine.switch_flash(0)
                elif selected_flash == 'upper':
                    jtag_engine.switch_flash(1)

                jtag_engine.tristate_spi_clk(True)
                jtag_engine.tristate_lower_flash(True)
                if is_dual:
                    jtag_engine.tristate_upper_flash(True)
                else:
                    jtag_engine.tristate_upper_flash(False)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec)

                outfile = output_file
                if not outfile:
                    self.pgm_print('No output file specified for read flash contents, aborting', error=True)
                    raise FtdiProgramError('no output file specified')

                try:
                    flash = SerialFlashDetector.get_flash_device(jtag_engine)
                except SerialFlashUnknownJedec as exc:
                    jedec = binascii.unhexlify(str(exc)[-7:-1])
                    self.pgm_print('Unrecognized Flash device. Will use Generic Flash profile. Please contact support if you face any problem.')
                    flash = jtag2spi_efinix_flash.GenericEfinixSupportedFlash(jtag_engine, jedec)
                self.pgm_print('Flash device: %s @ JTAG freq %0.1f MHz' % (flash, self.jtag_program_freq / 1E6))

                mode_4byte = 0
                is_flash_larger_than_16MB = 0
                enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
                exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

                try:
                    final_address = start_address + read_length
                    mode_4byte = final_address > 16 * 1024 * 1024
                    if '_size' in flash.__dict__:
                        if flash._size / (1<<20) > 16 :
                            is_flash_larger_than_16MB = 1
                except OutofRangeError:
                    #For when 4byte mode flag return value other than 0 or 1
                    pass
                except Exception as exc:
                    self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

                if mode_4byte:
                    if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                        flash.ADDRESS_MODE_4BYTE = True
                        self.pgm_print('Enable 4-byte address mode = True')
                        flash.enter_4byte_addr_mode(sync=False)
                    else:
                        self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                        self.pgm_print('Aborting flash programming', error=True)
                        raise FtdiProgramError('Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB')

                nv_conf_reg = None
                need_restore_nv_conf_reg = False

                # Micon MT25Q only
                if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                    nv_conf_reg = flash._read_nv_conf_ref(sync=False)

                    # If the address mode is 4-bytes, change it to 3-bytes
                    lsb = int(nv_conf_reg[0])
                    if lsb & 1 == 0:
                        self.pgm_print("Overwrite Address mode to 3-bytes")
                        need_restore_nv_conf_reg = True
                        flash._write_nv_conf_ref([0xFF, 0xFF])

                num_bytes = read_length
                flash_address = start_address

                # Read from flash into buffer
                self.pgm_print(
                    'Reading %s from flash @ %s ...' %
                    (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
                delta = now()
                readdata = flash.read(flash_address, num_bytes, sync=False)
                delta = now() - delta
                self.pgm_print('Finished read in %d seconds' % int(delta))

                #readbytes = readdata.tobytes() if hasattr(readdata, 'tobytes') else readdata

                # Restore nv configuration reg
                # Micon MT25Q only
                if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                    if need_restore_nv_conf_reg:
                        self.pgm_print("Restore Non-volative configuration register.")
                        flash._write_nv_conf_ref(nv_conf_reg)

                if mode_4byte:
                    if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                        self.pgm_print('Exit 4-byte address mode')
                        flash.exit_4byte_addr_mode(sync=True)

                # write out data to file
                with open(outfile, 'w', newline='\n') as file:
                    for curr_byte in readdata:
                        print('%02X' % curr_byte, file=file)

        except Exception as e:
            self.pgm_print(f'{e}, aborting jtag2spi read', error=True)
            error = True
        finally:
            if jtag_session:
                JtagManager().unconfig_jtag(jtag_session)
                jtag_session = None
            if error:
                raise FtdiProgramError
            else:
                return jedec


    def jtag2spi_erase_flash(self, start, length, setup, tap='efx', selected_flash='lower', is_dual=False):
        jtag_session = None
        jedec = None
        error = None
        try:
            jtag_bus = self.get_jtag_bus_from_pinout()
            jtag_url = self.get_url_from_bus(jtag_bus)

            jcfs, chip_num = None, 1
            if self.xml:
                jcfs = JcfParser(self.xml).all_devices()
                #tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, jcfs=jcfs, chip_num=self.chip_num, use_custom_engine=True)
            else:
                #tap = JtagManager().detect_jtag_tap(url=jtag_url)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, use_custom_engine=True)

            with jtag_session.get_engine() as jtag_engine:
                if setup == True:
                    jtag_engine.setup()
                else:
                    # even selected not to do setup, we just do it anyway if read jedec id failed
                    jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                    if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)) or ((jedec[0] == 0x00) and (jedec[1] == 0x00) and (jedec[2] == 0x00)):
                        jtag_engine.setup()

                jtag_engine.tristate_spi_clk(True)
                jtag_engine.tristate_lower_flash(True)
                if is_dual:
                    jtag_engine.tristate_upper_flash(True)
                else:
                    jtag_engine.tristate_upper_flash(False)

                if selected_flash == 'lower':
                    jtag_engine.switch_flash(0)
                elif selected_flash == 'upper':
                    jtag_engine.switch_flash(1)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec)

                try:
                    flash = SerialFlashDetector.get_flash_device(jtag_engine)
                except SerialFlashUnknownJedec as exc:
                    jedec = binascii.unhexlify(str(exc)[-7:-1])
                    self.pgm_print('Unrecognized Flash device. Will use Generic Flash profile. Please contact support if you face any problem.')
                    flash = jtag2spi_efinix_flash.GenericEfinixSupportedFlash(jtag_engine, jedec)
                self.pgm_print('Flash device: %s @ JTAG freq %0.1f MHz' % (flash, self.jtag_program_freq / 1E6))

                erase_min_size: int  = flash.get_erase_size()
                if (length % erase_min_size) == 0:
                    erase_length  = length
                else:
                    erase_length = (int(length / erase_min_size) + 1) * erase_min_size

                mode_4byte = 0
                is_flash_larger_than_16MB = 0
                enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
                exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

                try:
                    final_address = start + erase_length
                    mode_4byte = final_address > 16 * 1024 * 1024
                    if '_size' in flash.__dict__:
                        if flash._size / (1<<20) > 16 :
                            is_flash_larger_than_16MB = 1
                except OutofRangeError:
                    #For when 4byte mode flag return value other than 0 or 1
                    pass
                except Exception as exc:
                    self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

                if mode_4byte:
                    if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                        flash.ADDRESS_MODE_4BYTE = True
                        self.pgm_print('Enable 4-byte address mode = True')
                        flash.enter_4byte_addr_mode(sync=False)
                    else:
                        self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                        self.pgm_print('Aborting flash programming', error=True)
                        raise FtdiProgramError('Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB')

                # need to figure out erase size... adjust flash space to erase as needed
                erase_min_size = flash.get_erase_size()

                # starting flash address for erase should start on sector boundary
                if (start % erase_min_size) != 0:
                    ok_fa_low = int(start / erase_min_size) * erase_min_size
                    ok_fa_high = ok_fa_low + erase_min_size
                    raise ValueError(
                        'cannot erase address %s, not on sector boundary.  Nearest valid addresses: %s, %s'
                        % ("0x{:08x}".format(start), "0x{:08x}".format(ok_fa_low),
                        "0x{:08x}".format(ok_fa_high)))

                # length should be multiple of erase_min_size
                if (erase_length % erase_min_size) != 0:
                    raise ValueError('erase length should be multiple of %s' %
                                    ("0x{:08x}".format(erase_min_size)))

                status_reg = None
                need_restore_status_reg = False
                status_reg = flash._read_status(sync=False)
                #if status_reg & 0b11111100 != 0:
                # For Macronix flash, they use status register-6 for QE bit.
                # So even we fall into here, we need to make sure whether we really need to do unlocking
                #if( (isinstance(flash, jtag2spi_efinix_flash.Mx75uFlashDevice) or isinstance(flash, jtag2spi_efinix_flash.Mx75lFlashDevice)) and
                #    (status_reg & 0b00111100 == 0) ):
                #    pass
                #else:
                self.pgm_print("Unlock all sectors for write.")
                need_restore_status_reg = True
                
                if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                    status_reg_2 = flash._read_status2(sync=False)
                    flash._write_status(0, status_reg_2, sync=False)
                else:
                    flash._write_status(0, sync=False)

                nv_conf_reg = None
                need_restore_nv_conf_reg = False

                # Micon MT25Q only
                if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                    nv_conf_reg = flash._read_nv_conf_ref(sync=False)

                    # If the address mode is 4-bytes, change it to 3-bytes
                    lsb = int(nv_conf_reg[0])
                    if lsb & 1 == 0:
                        self.pgm_print("Overwrite Address mode to 3-bytes")
                        need_restore_nv_conf_reg = True
                        flash._write_nv_conf_ref([0xFF, 0xFF])

                #erase_length = length
                start_erase = start

                # unlock flash
                flash.unlock(sync=False)

                # erase flash
                self.pgm_print('Erasing %s from flash address @ %s (may take a while...)' % (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
                delta = now()
                flash.erase(address=start_erase, length=erase_length, verify=True, sync=False)
                delta = now() - delta
                self.pgm_print('Finished erase in %d seconds' % int(delta))

                # Restore write-protect status reg value
                if need_restore_status_reg:
                    self.pgm_print("Restore Write-protect status register.")

                    if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                        status_reg_2 = flash._read_status2(sync=False)
                        flash._write_status(status_reg, status_reg_2, sync=False)
                    else:
                        flash._write_status(status_reg, sync=False)

                # Micon MT25Q only
                if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                    # Restore nv configuration reg
                    if need_restore_nv_conf_reg:
                        self.pgm_print("Restore Non-volative configuration register.")
                        flash._write_nv_conf_ref(nv_conf_reg)

                if mode_4byte:
                    if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                        self.pgm_print('Exit 4-byte address mode')
                        flash.exit_4byte_addr_mode(sync=True)
                else:
                    jtag_engine.go_idle2(sync=True)

        except Exception as e:
            self.pgm_print(f'{e}, aborting jtag2spi erase', error=True)
            error = True
        finally:
            if jtag_session:
                JtagManager().unconfig_jtag(jtag_session)
                jtag_session = None
            if error:
                raise FtdiProgramError
            else:
                return jedec


    def jtag2spi_program(self,
                         setup,
                         flash_address,
                         erase_entire_flash,
                         tap='efx',
                         verify_after_write=True,
                         erase_before_write=True,
                         selected_flash='lower',
                         is_dual=False):

        jtag_session = None
        jedec = None
        error = None
        try:
            jtag_bus = self.get_jtag_bus_from_pinout()
            jtag_url = self.get_url_from_bus(jtag_bus)
            JTAG_INSTR = JtagManager()._tap_to_instr(self.jtag_tap)

            jcfs, chip_num = None, 1
            if self.xml:
                jcfs = JcfParser(self.xml).all_devices()
                #tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, jcfs=jcfs, chip_num=self.chip_num, use_custom_engine=True)
            else:
                #tap = JtagManager().detect_jtag_tap(url=jtag_url)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, use_custom_engine=True)

            with jtag_session.get_engine() as jtag_engine:
                if setup == True:
                    jtag_engine.setup()
                else:
                    # even selected not to do setup, we just do it anyway if read jedec id failed
                    jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                    if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)) or ((jedec[0] == 0x00) and (jedec[1] == 0x00) and (jedec[2] == 0x00)):
                        jtag_engine.setup()

                if selected_flash == 'lower':
                    jtag_engine.switch_flash(0)
                elif selected_flash == 'upper':
                    jtag_engine.switch_flash(1)

                jtag_engine.tristate_spi_clk(True)
                jtag_engine.tristate_lower_flash(True)
                if is_dual:
                    jtag_engine.tristate_upper_flash(True)
                else:
                    jtag_engine.tristate_upper_flash(False)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec)

                # Process bitstream header
                is_bitstream = 0
                try:
                    device_db = bitstream_util.get_device_db()
                    hp = HeaderProcessor(self.input_file, None, device_db, None)
                    hp.process_header()
                    is_bitstream = 1
                except UnicodeDecodeError:
                    #For when header contains non-ASCII displayable characters
                    pass
                except Exception as exc:
                    self.pgm_print("Warning: Detected general exception while processing bitstream header - %s" % exc)

                # main buffer
                hexbytes = Array('B')
                # open HEX file
                with open(self.input_file, 'r') as file:
                    # read file by line
                    line = file.readline()
                    while line:
                        # convert line to binary, append to buffer
                        val = int(line[0:2], 16)
                        hexbytes.append(val)
                        line = file.readline()
                bufstr = hexbytes.tobytes()

                try:
                    flash = SerialFlashDetector.get_flash_device(jtag_engine)
                except SerialFlashUnknownJedec as exc:
                    jedec = binascii.unhexlify(str(exc)[-7:-1])
                    self.pgm_print('Unrecognized Flash device. Will use Generic Flash profile. Please contact support if you face any problem.')
                    flash = jtag2spi_efinix_flash.GenericEfinixSupportedFlash(jtag_engine, jedec)

                self.pgm_print('Flash device: %s @ JTAG freq %0.1f MHz' % (flash, self.jtag_program_freq / 1E6))
                mode_4byte = 0
                is_flash_larger_than_16MB = 0
                enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
                exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

                try:
                    num_bytes_in_bitstream = bitstream_util.get_bitstream_num_bytes(self.input_file)
                    final_address = flash_address + num_bytes_in_bitstream
                    mode_4byte = final_address > 16 * 1024 * 1024
                    if '_size' in flash.__dict__:
                        if flash._size / (1<<20) > 16 :
                            is_flash_larger_than_16MB = 1
                except OutofRangeError:
                    #For when 4byte mode flag return value other than 0 or 1
                    pass
                except Exception as exc:
                    self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

                if mode_4byte:
                    if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                        flash.ADDRESS_MODE_4BYTE = True
                        self.pgm_print('Enable 4-byte address mode = True')
                        flash.enter_4byte_addr_mode(sync=False)
                    else:
                        self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                        self.pgm_print('Aborting flash programming', error=True)
                        raise FtdiProgramError('Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB')

                enable_quad_spi = False
                disable_quad_spi = False
                enable_dual_spi = False
                if is_bitstream:
                    # Determine if need to enable QE bit
                    # If input hex file's prog width is 4, then we enable QE bit
                    try:
                        prog_width = hp.get_prog_width()
                    except UnknownWidthError:
                        #For when width cannot be determined from header
                        pass
                    except Exception as exc:
                        self.pgm_print("Warning: Detected general exception '{}' while enabling QE bit".format(exc))
                    else:
                        if prog_width == 4:
                            enable_quad_spi = getattr(flash, "enable_quad_spi", None)
                            if callable(enable_quad_spi):
                                enable_quad_spi = True
                                #flash.enable_quad_spi(sync=False)
                        elif prog_width == 2:
                            enable_dual_spi = getattr(flash, "enable_dual_spi", None)
                            if callable(enable_dual_spi):
                                enable_dual_spi = True
                                #flash.enable_dual_spi(sync=False)
                        elif prog_width == 1:
                            disable_quad_spi = getattr(flash, "disable_quad_spi", None)
                            if callable(disable_quad_spi):
                                # debug code only -- keep disabled production
                                #flash.disable_quad_spi(sync=False)
                                #disable_quad_spi = True
                                pass

                status_reg = None
                need_restore_status_reg = False
                status_reg = flash._read_status(sync=False)
                #if status_reg & 0b11111100 != 0:
                # For Macronix flash, they use status register-6 for QE bit.
                # So even we fall into here, we need to make sure whether we really need to do unlocking
                #if( (isinstance(flash, jtag2spi_efinix_flash.Mx75uFlashDevice) or isinstance(flash, jtag2spi_efinix_flash.Mx75lFlashDevice)) and
                #   (status_reg & 0b00111100 == 0) ):
                #    pass
                #else:
                self.pgm_print("Unlock all sectors for write.")
                need_restore_status_reg = True

                if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                    status_reg_2 = flash._read_status2(sync=False)
                    flash._write_status(0, status_reg_2, sync=False)
                else:
                    flash._write_status(0, sync=False)
                    #flash._write_status_reg(0, status_reg_2)

                nv_conf_reg = None
                need_restore_nv_conf_reg = False

                # Micon MT25Q only
                if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                    nv_conf_reg = flash._read_nv_conf_ref(sync=False)

                    # If the address mode is 4-bytes, change it to 3-bytes
                    lsb = int(nv_conf_reg[0])
                    if lsb & 1 == 0:
                        self.pgm_print("Overwrite Address mode to 3-bytes")
                        need_restore_nv_conf_reg = True
                        flash._write_nv_conf_ref([0xFF, 0xFF])

                num_bytes = bitstream_util.get_bitstream_num_bytes(self.input_file)

                flash_size = len(flash)
                # need to figure out erase size... adjust flash space to erase as needed
                erase_min_size = flash.get_erase_size()
                if (num_bytes % erase_min_size) == 0:
                    num_erase_bytes = num_bytes
                else:
                    num_erase_bytes = (int(num_bytes / erase_min_size) + 1) * erase_min_size

                # starting flash address for new FPGA image should start on sector boundary as well
                if (flash_address % erase_min_size) != 0:
                    ok_fa_low = int(flash_address / erase_min_size) * erase_min_size
                    ok_fa_high = ok_fa_low + erase_min_size
                    raise ValueError(
                        'cannot write image to address %s, not on sector boundary.  Nearest valid addresses: %s, %s'
                        % ("0x{:08x}".format(flash_address),
                        "0x{:08x}".format(ok_fa_low), "0x{:08x}".format(ok_fa_high)))

                # determine how much flash to erase (either entire flash or just as needed for this image)
                if erase_entire_flash:
                    erase_length = flash_size
                    start_erase = 0x00
                else:
                    erase_length = num_erase_bytes
                    start_erase = flash_address

                if erase_length > flash_size:
                    raise ValueError(
                        'Specified image file (requires %d bytes in flash memory) is too large for flash device, which is limited to %d bytes'
                        % (erase_length, flash_size))

                # unlock flash
                flash.unlock(sync=False)

                # erase flash
                if erase_before_write:
                    self.pgm_print(
                        'Erasing %s from flash @ %s (may take a while...)' %
                        (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
                    delta = now()
                    flash.erase(address=start_erase, length=erase_length, verify=False, sync=False)
                    delta = now() - delta
                    self.pgm_print('Finished erase in %d seconds' % int(delta))

                # write string buffer to flash
                self.pgm_print(
                    'Writing %s to flash @ %s ...' %
                    (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
                delta = now()
                flash.write(flash_address, bufstr)
                delta = now() - delta
                self.pgm_print('Finished write in %d seconds' % int(delta))

                # Verify after write
                if verify_after_write:
                    # compute hash from original file contents (for flash verify later)
                    wmd = sha1()
                    wmd.update(bufstr)
                    refdigest = wmd.hexdigest()

                    # Read from flash into buffer
                    self.pgm_print(
                        'Reading %s from flash @ %s ...' %
                        (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
                    delta = now()
                    readdata = flash.read(flash_address, num_bytes, sync=False)
                    delta = now() - delta
                    self.pgm_print('Finished read in %d seconds' % int(delta))

                    # compute hash from read data contents
                    rmd = sha1()
                    rmd.update(readdata)
                    newdigest = rmd.hexdigest()

                    # do verification
                    if refdigest == newdigest:
                        self.pgm_print('Flash verify successful')
                    else:
                        self.pgm_print('ERROR: Flash verify unsuccessful... mismatch found', error=True)
                        self.pgm_print('retry program using max clk ...', error=True)

                        flash.change_wait_time(False)

                        # erase flash
                        if erase_before_write:
                            self.pgm_print(
                                'Erasing %s from flash @ %s (may take a while...)' %
                                (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
                            delta = now()
                            flash.erase(address=start_erase, length=erase_length, verify=False, sync=False)
                            delta = now() - delta
                            self.pgm_print('Finished erase in %d seconds' % int(delta))

                        # write string buffer to flash
                        self.pgm_print(
                            'Writing %s to flash @ %s ...' %
                            (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
                        delta = now()
                        flash.write(flash_address, bufstr)
                        delta = now() - delta
                        self.pgm_print('Finished write in %d seconds' % int(delta))

                        # Read from flash into buffer
                        self.pgm_print(
                            'Reading %s from flash @ %s ...' %
                            (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
                        delta = now()
                        readdata = flash.read(flash_address, num_bytes, sync=False)
                        delta = now() - delta
                        self.pgm_print('Finished read in %d seconds' % int(delta))

                        # compute hash from read data contents
                        rmd = sha1()
                        rmd.update(readdata)
                        newdigest = rmd.hexdigest()

                        if refdigest == newdigest:
                            self.pgm_print('Flash verify successful')
                        else:
                            self.pgm_print('ERROR: Flash verify unsuccessful... mismatch found', error=True)

                # Restore write-protect status reg value
                if need_restore_status_reg:
                    self.pgm_print("Restore Write-protect status register.")

                    if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                        status_reg_2 = flash._read_status2(sync=False)
                        flash._write_status(status_reg, status_reg_2, sync=False)
                    else:
                        flash._write_status(status_reg, sync=False)

                # Micon MT25Q only
                if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                    # Restore nv configuration reg
                    if need_restore_nv_conf_reg:
                        self.pgm_print("Restore Non-volative configuration register.")
                        flash._write_nv_conf_ref(nv_conf_reg)

                if mode_4byte:
                    if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                        self.pgm_print('Exit 4-byte address mode')
                        flash.exit_4byte_addr_mode(sync=True)
                    else:
                        jtag_engine.go_idle2(sync=True)

                # need to enable quad/dual SPI AFTER status register restoration
                # PROG-579
                if enable_quad_spi:
                    flash.enable_quad_spi(sync=False)
                if enable_dual_spi:
                    flash.enable_dual_spi(sync=False)
                if disable_quad_spi:
                    # debug only, not for production
                    self.pgm_print('DEBUG ONLY: disabling quad spi mode')
                    flash.disable_quad_spi(sync=False)

                # Print finish msg
                self.pgm_print('JTAG2SPI programming...done')

                ## FIXME: still cant reset board
                ## Send to user mode
                #sleep(0.1)
                #jtag_engine.write_ir(JTAG_INSTR['ENTERUSER'])
                #jtag_engine.write_dr(
                #    BitSequence('0' * 100, length=100,
                #                msb=True))  # extra clock ticks after ENTERUSER as well
                #jtag_engine.write_ir(JTAG_INSTR['IDCODE'])

                ## wait a bit for all operations to flush, then close the jtag channel
                #jtag_engine.reset()
                #sleep(0.2)

        except Exception as e:
            self.pgm_print(f'{e}, aborting flash programming', error=True)
            error = True
        finally:
            if jtag_session:
                JtagManager().unconfig_jtag(jtag_session)
                jtag_session = None
            if error:
                raise FtdiProgramError
            else:
                return jedec

    ###########################################################################################################################################################
    # dual flash
    def repartition(self, bufstr):
        # slice byte array into 2 list
        bufstr0 = bufstr[0:len(bufstr):2]
        bufstr1 = bufstr[1:len(bufstr):2]

        if len(bufstr0) - len(bufstr1) == 1:
            bufstr1.append(0xFF)

        # turn byte in bytearray into 4 bit tuple pair
        bufstr = list(map(lambda a: [((a[1] & 0b00001111) << 4) | (bufstr1[a[0]] & 0b00001111), (a[1] & 0b11110000) | ((bufstr1[a[0]] & 0b11110000) >> 4)], enumerate(bufstr0)))
        bufstr = tuple(zip(*bufstr))
        return bufstr

    def restore_partition(self, bufstr):
        # turn byte in bytearray into 4 bit tuple pair
        bufstr = list(map(lambda a:
                      [(bufstr[1][a[0]] & 0b11110000) | ((a[1] & 0b11110000) >> 4), ((bufstr[1][a[0]] & 0b00001111) << 4) | (a[1] & 0b00001111)], enumerate(bufstr[0])))
        bufstr = list(itertools.chain.from_iterable(bufstr))
        return bufstr


    # top-level control for when jtag_bridge_x8_new is selected
    def jtag2spi_x8(self):

        jtag_bus = self.get_jtag_bus_from_pinout()
        jtag_url = self.get_url_from_bus(jtag_bus)
        if self.xml:
            jcfs = JcfParser(self.xml).all_devices()
            tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
        else:
            tap = JtagManager().detect_jtag_tap(url=jtag_url)

        if self.jtag_bridge_mode == 'read':
            self.jtag2spi_read_flash_dual_flash(start_address=self.address,
                                                read_length=self.num_bytes,
                                                output_file=self.output_file,
                                                setup=False,
                                                tap=tap)
        elif self.jtag_bridge_mode == 'erase':
            self.jtag2spi_erase_flash_dual_flash(start=self.address,
                                                length=self.num_bytes,
                                                setup=False,
                                                tap=tap)
        elif self.jtag_bridge_mode == 'write':
            self.jtag2spi_program_dual_flash(setup=False,
                                            flash_address=self.address,
                                            erase_entire_flash=False,
                                            tap=tap,
                                            verify_after_write=False,
                                            erase_before_write=False)
        elif self.jtag_bridge_mode == 'erase_and_write':
            self.jtag2spi_program_dual_flash(setup=False,
                                            flash_address=self.address,
                                            erase_entire_flash=False,
                                            tap=tap,
                                            verify_after_write=False,
                                            erase_before_write=True)
        elif self.jtag_bridge_mode == 'all_no_erase':
            self.jtag2spi_program_dual_flash(setup=False,
                                            flash_address=self.address,
                                            erase_entire_flash=False,
                                            tap=tap,
                                            verify_after_write=True,
                                            erase_before_write=False)
        else: # 'all'
            self.jtag2spi_program_dual_flash(setup=False,
                                            flash_address=self.address,
                                            erase_entire_flash=False,
                                            tap=tap,
                                            verify_after_write=True,
                                            erase_before_write=True)

    def jtag2spi_read_flash_dual_flash(self, start_address, read_length, output_file, setup, tap='efx'):
        jtag_session = None
        jedec = None
        error = None

        # should be true for all bitstreams
        SYNC_POS = 256
        WORST_CASE_SPLIT_POINT = SYNC_POS + 168

        # need to read extra if doing bitstream splicing
        # NOTE: if not a proper x8 active bitstream, need to reverse
        # this later
        updated_read_length = read_length + WORST_CASE_SPLIT_POINT

        try:
            jtag_bus = self.get_jtag_bus_from_pinout()
            jtag_url = self.get_url_from_bus(jtag_bus)

            jcfs, chip_num = None, 1
            if self.xml:
                jcfs = JcfParser(self.xml).all_devices()
                #tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, jcfs=jcfs, chip_num=self.chip_num, use_custom_engine=True)
            else:
                #tap = JtagManager().detect_jtag_tap(url=jtag_url)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, use_custom_engine=True)

            with jtag_session.get_engine() as jtag_engine:
                if setup == True:
                    jtag_engine.setup()
                else:
                    # even selected not to do setup, we just do it anyway if read jedec id failed
                    jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                    if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)) or ((jedec[0] == 0x00) and (jedec[1] == 0x00) and (jedec[2] == 0x00)):
                        jtag_engine.setup()

                jtag_engine.tristate_spi_clk(True)
                jtag_engine.tristate_lower_flash(True)
                jtag_engine.tristate_upper_flash(True)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec = SerialFlashDetector.read_jedec_id(jtag_engine)

                jtag_engine.switch_flash(1)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec1 = SerialFlashDetector.read_jedec_id(jtag_engine)

                if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec)
                if (not jedec1) or ((jedec1[0] == 0xFF) and (jedec1[1] == 0xFF) and (jedec1[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec1)

                if jedec != jedec1:
                    self.pgm_print('We will only support same flash type for x8 Programming!')
                    raise FtdiProgramError('flash mismatch detected, dual flash operations only supported for same flash type')

                outfile = output_file
                if not outfile:
                    self.pgm_print('No output file specified for read flash contents, aborting', error=True)
                    raise FtdiProgramError('no output file specified')

                readdata = [(), ()]

                jtag_engine.switch_flash(0)
                for index, rdata in enumerate(readdata):
                    try:
                        flash = SerialFlashDetector.get_flash_device(jtag_engine)
                    except SerialFlashUnknownJedec as exc:
                        jedec = binascii.unhexlify(str(exc)[-7:-1])
                        self.pgm_print('Unrecognized Flash device. Will use Generic Flash profile. Please contact support if you face any problem.')
                        flash = jtag2spi_efinix_flash.GenericEfinixSupportedFlash(jtag_engine, jedec)
                    self.pgm_print('Flash device: %s @ JTAG freq %0.1f MHz' % (flash, self.jtag_program_freq / 1E6))

                    mode_4byte = 0
                    is_flash_larger_than_16MB = 0
                    enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
                    exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

                    try:
                        final_address = start_address + updated_read_length/2
                        mode_4byte = final_address > 16 * 1024 * 1024
                        if '_size' in flash.__dict__:
                            if flash._size / (1<<20) > 16 :
                                is_flash_larger_than_16MB = 1
                    except OutofRangeError:
                        #For when 4byte mode flag return value other than 0 or 1
                        pass
                    except Exception as exc:
                        self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

                    if mode_4byte:
                        if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                            flash.ADDRESS_MODE_4BYTE = True
                            self.pgm_print('Enable 4-byte address mode = True')
                            flash.enter_4byte_addr_mode(sync=False)
                        else:
                            self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                            self.pgm_print('Aborting flash programming', error=True)
                            raise FtdiProgramError('Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB')

                    nv_conf_reg = None
                    need_restore_nv_conf_reg = False

                    # Micon MT25Q only
                    if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                        nv_conf_reg = flash._read_nv_conf_ref(sync=False)

                        # If the address mode is 4-bytes, change it to 3-bytes
                        lsb = int(nv_conf_reg[0])
                        if lsb & 1 == 0:
                            self.pgm_print("Overwrite Address mode to 3-bytes")
                            need_restore_nv_conf_reg = True
                            flash._write_nv_conf_ref([0xFF, 0xFF])

                    num_bytes = int(read_length/2)
                    updated_num_bytes = int(updated_read_length/2)
                    flash_address = start_address

                    # Read from flash into buffer
                    self.pgm_print(
                        'Reading %s from flash @ %s ...' %
                        (pretty_size(num_bytes), "0x{:08x}".format(flash_address)))
                    delta = now()
                    readdata[index] = tuple(flash.read(flash_address, updated_num_bytes, sync=False))
                    delta = now() - delta
                    self.pgm_print('Finished read in %d seconds' % int(delta))

                    #readbytes = readdata.tobytes() if hasattr(readdata, 'tobytes') else readdata

                    # Restore nv configuration reg
                    # Micon MT25Q only
                    if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                        if need_restore_nv_conf_reg:
                            self.pgm_print("Restore Non-volative configuration register.")
                            flash._write_nv_conf_ref(nv_conf_reg)

                    if mode_4byte:
                        if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                            self.pgm_print('Exit 4-byte address mode')
                            flash.exit_4byte_addr_mode(sync=False)

                    if index == 0:
                        jtag_engine.switch_flash(1)
                    else:
                        jtag_engine.go_idle2(sync=True)

            # realign data if x8 bitstream as required
            self.realign_x8_bitstream_read_data(readdata, SYNC_POS, WORST_CASE_SPLIT_POINT)


            readdata = bytes(self.restore_partition(readdata))
            # write out data to file
            with open(outfile, 'w', newline='\n') as file:
                for curr_byte in readdata:
                    print('%02X' % curr_byte, file=file)

        except Exception as e:
            self.pgm_print(f'{e}, aborting jtag2spi read', error=True)
            error = True
        finally:
            if jtag_session:
                JtagManager().unconfig_jtag(jtag_session)
                jtag_session = None
            if error:
                raise FtdiProgramError
            else:
                return jedec

    def jtag2spi_erase_flash_dual_flash(self, start, length, setup, tap='efx'):
        jtag_session = None
        jedec = None
        error = None
        try:
            jtag_bus = self.get_jtag_bus_from_pinout()
            jtag_url = self.get_url_from_bus(jtag_bus)

            jcfs, chip_num = None, 1
            if self.xml:
                jcfs = JcfParser(self.xml).all_devices()
                #tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, jcfs=jcfs, chip_num=self.chip_num, use_custom_engine=True)
            else:
                #tap = JtagManager().detect_jtag_tap(url=jtag_url)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, use_custom_engine=True)

            with jtag_session.get_engine() as jtag_engine:
                if setup == True:
                    jtag_engine.setup()
                else:
                    # even selected not to do setup, we just do it anyway if read jedec id failed
                    jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                    if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)) or ((jedec[0] == 0x00) and (jedec[1] == 0x00) and (jedec[2] == 0x00)):
                        jtag_engine.setup()

                jtag_engine.tristate_spi_clk(True)
                jtag_engine.tristate_lower_flash(True)
                jtag_engine.tristate_upper_flash(True)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec = SerialFlashDetector.read_jedec_id(jtag_engine)

                jtag_engine.switch_flash(1)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec1 = SerialFlashDetector.read_jedec_id(jtag_engine)

                if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(0xFF)

                if (not jedec1) or ((jedec1[0] == 0xFF) and (jedec1[1] == 0xFF) and (jedec1[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec1)

                if jedec != jedec1:
                    self.pgm_print('We will only support same flash type for x8 Programming!')
                    raise FtdiProgramError('flash mismatch detected, dual flash operations only supported for same flash type')

                jtag_engine.switch_flash(0)
                for index, _ in enumerate(range(2)):
                    try:
                        flash = SerialFlashDetector.get_flash_device(jtag_engine)
                    except SerialFlashUnknownJedec as exc:
                        jedec = binascii.unhexlify(str(exc)[-7:-1])
                        self.pgm_print('Unrecognized Flash device. Will use Generic Flash profile. Please contact support if you face any problem.')
                        flash = jtag2spi_efinix_flash.GenericEfinixSupportedFlash(jtag_engine, jedec)
                    self.pgm_print('Flash device: %s @ JTAG freq %0.1f MHz' % (flash, self.jtag_program_freq / 1E6))

                    mode_4byte = 0
                    is_flash_larger_than_16MB = 0
                    enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
                    exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

                    try:
                        final_address = start + int(length/2)
                        mode_4byte = final_address > 16 * 1024 * 1024
                        if '_size' in flash.__dict__:
                            if flash._size / (1<<20) > 16 :
                                is_flash_larger_than_16MB = 1
                    except OutofRangeError:
                        #For when 4byte mode flag return value other than 0 or 1
                        pass
                    except Exception as exc:
                        self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

                    if mode_4byte:
                        if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                            flash.ADDRESS_MODE_4BYTE = True
                            self.pgm_print('Enable 4-byte address mode = True')
                            flash.enter_4byte_addr_mode(sync=False)
                        else:
                            self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                            self.pgm_print('Aborting flash programming', error=True)
                            raise FtdiProgramError('Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB')

                    status_reg = None
                    need_restore_status_reg = False
                    status_reg = flash._read_status(sync=False)
                    #if status_reg & 0b11111100 != 0:
                    # For Macronix flash, they use status register-6 for QE bit.
                    # So even we fall into here, we need to make sure whether we really need to do unlocking
                    #if( (isinstance(flash, jtag2spi_efinix_flash.Mx75uFlashDevice) or isinstance(flash, jtag2spi_efinix_flash.Mx75lFlashDevice)) and
                    #    (status_reg & 0b00111100 == 0) ):
                    #    pass
                    #else:
                    self.pgm_print("Unlock all sectors for write.")
                    need_restore_status_reg = True

                    if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                        status_reg_2 = flash._read_status2(sync=False)
                        flash._write_status(0, status_reg_2, sync=False)
                    else:
                        flash._write_status(0, sync=False)

                    nv_conf_reg = None
                    need_restore_nv_conf_reg = False

                    # Micon MT25Q only
                    if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                        nv_conf_reg = flash._read_nv_conf_ref(sync=False)

                        # If the address mode is 4-bytes, change it to 3-bytes
                        lsb = int(nv_conf_reg[0])
                        if lsb & 1 == 0:
                            self.pgm_print("Overwrite Address mode to 3-bytes")
                            need_restore_nv_conf_reg = True
                            flash._write_nv_conf_ref([0xFF, 0xFF])

                    erase_length = int(length/2)
                    erase_min_size  = flash.get_erase_size()
                    if (erase_length % erase_min_size) != 0:
                        erase_length = (int(erase_length / erase_min_size) + 1) * erase_min_size

                    # need to figure out erase size... adjust flash space to erase as needed
                    # starting flash address for erase should start on sector boundary
                    if (start % erase_min_size) != 0:
                        ok_fa_low = int(start / erase_min_size) * erase_min_size
                        ok_fa_high = ok_fa_low + erase_min_size
                        raise ValueError(
                            'cannot erase address %s, not on sector boundary.  Nearest valid addresses: %s, %s'
                            % ("0x{:08x}".format(start), "0x{:08x}".format(ok_fa_low),
                            "0x{:08x}".format(ok_fa_high)))

                    # length should be multiple of erase_min_size
                    if (erase_length % erase_min_size) != 0:
                        raise ValueError('erase length should be multiple of %s' %
                                        ("0x{:08x}".format(erase_min_size)))

                    start_erase = start

                    # unlock flash
                    flash.unlock(sync=False)

                    # erase flash
                    self.pgm_print('Erasing %s from flash address @ %s (may take a while...)' % (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
                    delta = now()
                    flash.erase(address=start_erase, length=erase_length, verify=True, sync=False)
                    delta = now() - delta
                    self.pgm_print('Finished erase in %d seconds' % int(delta))

                    # Restore write-protect status reg value
                    if need_restore_status_reg:
                        self.pgm_print("Restore Write-protect status register.")

                        if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                            status_reg_2 = flash._read_status2(sync=False)
                            flash._write_status(status_reg, status_reg_2, sync=False)
                        else:
                            flash._write_status(status_reg, sync=False)

                    # Micon MT25Q only
                    if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                        # Restore nv configuration reg
                        if need_restore_nv_conf_reg:
                            self.pgm_print("Restore Non-volative configuration register.")
                            flash._write_nv_conf_ref(nv_conf_reg)

                    if mode_4byte:
                        if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                            self.pgm_print('Exit 4-byte address mode')
                            flash.exit_4byte_addr_mode(sync=False)

                    if index == 0:
                        jtag_engine.switch_flash(1)
                    else:
                        jtag_engine.go_idle2(sync=True)

        except Exception as e:
            self.pgm_print(f'{e}, aborting jtag2spi erase', error=True)
            error = True
        finally:
            if jtag_session:
                JtagManager().unconfig_jtag(jtag_session)
                jtag_session = None
            if error:
                raise FtdiProgramError
            else:
                return jedec

    def jtag2spi_program_dual_flash(self,
                         setup,
                         flash_address,
                         erase_entire_flash,
                         tap='efx',
                         verify_after_write=True,
                         erase_before_write=True,
                         idcode=''):

        jtag_session = None
        jedec = None
        error = None
        try:
            jtag_bus = self.get_jtag_bus_from_pinout()
            jtag_url = self.get_url_from_bus(jtag_bus)
            #JTAG_INSTR = JtagManager()._tap_to_instr(self.jtag_tap)

            jcfs, chip_num = None, 1
            if self.xml:
                jcfs = JcfParser(self.xml).all_devices()
                #tap = JtagManager().detect_jtag_tap(url=jtag_url, jcfs=jcfs, chip_num=self.chip_num)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, jcfs=jcfs, chip_num=self.chip_num, use_custom_engine=True)
            else:
                #tap = JtagManager().detect_jtag_tap(url=jtag_url)
                self.pgm_print('Connecting to JTAG_TAP: {}'.format(tap))
                jtag_session = JtagManager().config_jtag(url=jtag_url, freq=self.jtag_program_freq, tap=tap, use_custom_engine=True)

            with jtag_session.get_engine() as jtag_engine:
                if setup == True:
                    jtag_engine.setup()
                else:
                    # even selected not to do setup, we just do it anyway if read jedec id failed
                    jedec = SerialFlashDetector.read_jedec_id(jtag_engine)
                    if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)) or ((jedec[0] == 0x00) and (jedec[1] == 0x00) and (jedec[2] == 0x00)):
                        jtag_engine.setup()

                jtag_engine.tristate_spi_clk(True)
                jtag_engine.tristate_lower_flash(True)
                jtag_engine.tristate_upper_flash(True)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec = SerialFlashDetector.read_jedec_id(jtag_engine)

                jtag_engine.switch_flash(1)

                sleep(0.001)
                SerialFlashDetector.release_powerdown(jtag_engine)
                sleep(0.001)

                jedec1 = SerialFlashDetector.read_jedec_id(jtag_engine)

                if (not jedec) or ((jedec[0] == 0xFF) and (jedec[1] == 0xFF) and (jedec[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec)
                if (not jedec1) or ((jedec1[0] == 0xFF) and (jedec1[1] == 0xFF) and (jedec1[2] == 0xFF)):
                    raise SerialFlashUnknownJedec(jedec1)

                if jedec != jedec1:
                    self.pgm_print('We will only support same flash type for x8 Programming!')
                    raise FtdiProgramError('flash mismatch detected, dual flash operations only supported for same flash type')

                input_file_ori = self.input_file

                # let's ignore ES device and do this in all cases
                # Create a "virtual" input file, so that we can do all the bitstream manipulation for x8
                temp_outfile = tempfile.NamedTemporaryFile(mode="w+t", suffix=".hex")
                input_file = temp_outfile.name
                temp_outfile.close()
                bitstream_util.generate_x8_hex_from_input(input_file=input_file_ori, output_file=input_file, idcode='')

                mode_4byte = 0
                num_bytes_in_bitstream = bitstream_util.get_bitstream_num_bytes(input_file)
                final_address = flash_address + num_bytes_in_bitstream

                mode_4byte = final_address > 16 * 1024 * 1024

                # Process bitstream header
                is_bitstream = 0
                try:
                    device_db = bitstream_util.get_device_db()
                    hp = HeaderProcessor(input_file_ori, None, device_db, None)
                    hp.process_header()
                    is_bitstream = 1
                except UnicodeDecodeError:
                    #For when header contains non-ASCII displayable characters
                    pass
                except Exception as exc:
                    self.pgm_print("Warning: Detected general exception while processing bitstream header - %s" % exc)

                #num_bytes = bitstream_util.get_bitstream_num_bytes(self.input_file)

                # main buffer
                hexbytes = Array('B')
                # open HEX file
                with open(input_file, 'r') as file:
                    # read file by line
                    line = file.readline()
                    while line:
                        # convert line to binary, append to buffer
                        val = int(line[0:2], 16)
                        hexbytes.append(val)
                        line = file.readline()
                bufstr = hexbytes

                #div_num_bytes = num_bytes//2

                (bufstr0, bufstr1) = self.repartition(bufstr)

                numbytes_length = (len(bufstr0), len(bufstr1))
                (bufstr0, bufstr1) = (bytes(bufstr0), bytes(bufstr1))

                jtag_engine.switch_flash(0)
                for index, bufstr in enumerate((bufstr0, bufstr1)):
                    try:
                        flash = SerialFlashDetector.get_flash_device(jtag_engine)
                    except SerialFlashUnknownJedec as exc:
                        jedec = binascii.unhexlify(str(exc)[-7:-1])
                        self.pgm_print('Unrecognized Flash device. Will use Generic Flash profile. Please contact support if you face any problem.')
                        flash = jtag2spi_efinix_flash.GenericEfinixSupportedFlash(jtag_engine, jedec)
                    self.pgm_print('Flash device: %s @ JTAG freq %0.1f MHz' % (flash, self.jtag_program_freq / 1E6))

                    is_flash_larger_than_16MB = 0
                    enter_4byte_addr_mode = getattr(flash, "enter_4byte_addr_mode", None)
                    exit_4byte_addr_mode = getattr(flash, "exit_4byte_addr_mode", None)

                    #flash.enable_quad_spi(sync=False)

                    try:
                        if '_size' in flash.__dict__:
                            if flash._size / (1<<20) > 16 :
                                is_flash_larger_than_16MB = 1
                    except OutofRangeError:
                        #For when 4byte mode flag return value other than 0 or 1
                        pass
                    except Exception as exc:
                        self.pgm_print("Warning: Detected general exception while checking 4byte address mode flag - %s" % exc)

                    if mode_4byte:
                        if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                            flash.ADDRESS_MODE_4BYTE = True
                            self.pgm_print('Enable 4-byte address mode = True')
                            flash.enter_4byte_addr_mode(sync=False)
                        else:
                            self.pgm_print("Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB", error=True)
                            self.pgm_print('Aborting flash programming', error=True)
                            raise FtdiProgramError('Detected 4Byte flag in bitstream but flash is smaller or equal to 16MiB')

                    enable_quad_spi = False
                    disable_quad_spi = False
                    enable_dual_spi = False
                    if is_bitstream:
                        # Determine if need to enable QE bit
                        # If input hex file's prog width is 4, then we enable QE bit
                        try:
                            prog_width = hp.get_prog_width()
                        except UnknownWidthError:
                            #For when width cannot be determined from header
                            pass
                        except Exception as exc:
                            self.pgm_print("Warning: Detected general exception '{}' while enabling QE bit".format(exc))
                        else:
                            if (prog_width == 4) or (prog_width == 8):
                                enable_quad_spi = getattr(flash, "enable_quad_spi", None)
                                if callable(enable_quad_spi):
                                    #flash.enable_quad_spi(sync=False)
                                    enable_quad_spi = True
                            elif prog_width == 2:
                                enable_dual_spi = getattr(flash, "enable_dual_spi", None)
                                if callable(enable_dual_spi):
                                    #flash.enable_dual_spi(sync=False)
                                    enable_dual_spi = True
                            elif prog_width == 1:
                                disable_quad_spi = getattr(flash, "disable_quad_spi", None)
                                if callable(disable_quad_spi):
                                    # debug code only -- keep disabled production
                                    #flash.disable_quad_spi(sync=False)
                                    #disable_quad_spi = True
                                    pass

                    status_reg = None
                    need_restore_status_reg = False
                    status_reg = flash._read_status(sync=False)
                    #if status_reg & 0b11111100 != 0:
                    # For Macronix flash, they use status register-6 for QE bit.
                    # So even we fall into here, we need to make sure whether we really need to do unlocking
                    #if( (isinstance(flash, jtag2spi_efinix_flash.Mx75uFlashDevice) or isinstance(flash, jtag2spi_efinix_flash.Mx75lFlashDevice)) and
                    #   (status_reg & 0b00111100 == 0) ):
                    #    pass
                    #else:
                    self.pgm_print("Unlock all sectors for write.")
                    need_restore_status_reg = True

                    if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                        status_reg_2 = flash._read_status2(sync=False)
                        flash._write_status(0, status_reg_2, sync=False)
                    else:
                        flash._write_status(0, sync=False)
                        #flash._write_status_reg(0, status_reg_2)

                    nv_conf_reg = None
                    need_restore_nv_conf_reg = False

                    # Micon MT25Q only
                    if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB)):
                        nv_conf_reg = flash._read_nv_conf_ref(sync=False)

                        # If the address mode is 4-bytes, change it to 3-bytes
                        lsb = int(nv_conf_reg[0])
                        if lsb & 1 == 0:
                            self.pgm_print("Overwrite Address mode to 3-bytes")
                            need_restore_nv_conf_reg = True
                            flash._write_nv_conf_ref([0xFF, 0xFF])

                    flash_size = len(flash)
                    # need to figure out erase size... adjust flash space to erase as needed
                    erase_min_size = flash.get_erase_size()
                    if (numbytes_length[index] % erase_min_size) == 0:
                        num_erase_bytes = numbytes_length[index]
                    else:
                        num_erase_bytes = (int(numbytes_length[index] / erase_min_size) + 1) * erase_min_size

                    # starting flash address for new FPGA image should start on sector boundary as well
                    if (flash_address % erase_min_size) != 0:
                        ok_fa_low = int(flash_address / erase_min_size) * erase_min_size
                        ok_fa_high = ok_fa_low + erase_min_size
                        raise ValueError(
                            'cannot write image to address %s, not on sector boundary.  Nearest valid addresses: %s, %s'
                            % ("0x{:08x}".format(flash_address),
                            "0x{:08x}".format(ok_fa_low), "0x{:08x}".format(ok_fa_high)))

                    # determine how much flash to erase (either entire flash or just as needed for this image)
                    if erase_entire_flash:
                        erase_length = flash_size
                        start_erase = 0x00
                    else:
                        erase_length = num_erase_bytes
                        start_erase = flash_address

                    if erase_length > flash_size:
                        raise ValueError(
                            'Specified image file (requires %d bytes in flash memory) is too large for flash device, which is limited to %d bytes'
                            % (erase_length, flash_size))

                    # unlock flash
                    flash.unlock(sync=False)

                    # erase flash
                    if erase_before_write:
                        self.pgm_print(
                            'Erasing %s from flash @ %s (may take a while...)' %
                            (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
                        delta = now()
                        flash.erase(address=start_erase, length=erase_length, verify=False, sync=False)
                        delta = now() - delta
                        self.pgm_print('Finished erase in %d seconds' % int(delta))

                    # write string buffer to flash
                    self.pgm_print(
                        'Writing %s to flash @ %s ...' %
                        (pretty_size(numbytes_length[index]), "0x{:08x}".format(flash_address)))
                    delta = now()
                    flash.write(flash_address, bufstr)
                    delta = now() - delta
                    self.pgm_print('Finished write in %d seconds' % int(delta))

                    # Verify after write
                    if verify_after_write:
                        # compute hash from original file contents (for flash verify later)
                        wmd = sha1()
                        wmd.update(bufstr)
                        refdigest = wmd.hexdigest()

                        # Read from flash into buffer
                        self.pgm_print(
                            'Reading %s from flash @ %s ...' %
                            (pretty_size(numbytes_length[index]), "0x{:08x}".format(flash_address)))
                        delta = now()
                        readdata = flash.read(flash_address, numbytes_length[index], sync=False)
                        delta = now() - delta
                        self.pgm_print('Finished read in %d seconds' % int(delta))

                        # compute hash from read data contents
                        rmd = sha1()
                        rmd.update(readdata)
                        newdigest = rmd.hexdigest()

                        # do verification
                        if refdigest == newdigest:
                            self.pgm_print('Flash verify successful')
                        else:
                            self.pgm_print('ERROR: Flash verify unsuccessful... mismatch found', error=True)
                            self.pgm_print('retry program using max clk ...', error=True)

                            flash.change_wait_time(False)

                            # erase flash
                            if erase_before_write:
                                self.pgm_print(
                                    'Erasing %s from flash @ %s (may take a while...)' %
                                    (pretty_size(erase_length), "0x{:08x}".format(start_erase)))
                                delta = now()
                                flash.erase(address=start_erase, length=erase_length, verify=False, sync=False)
                                delta = now() - delta
                                self.pgm_print('Finished erase in %d seconds' % int(delta))

                            # write string buffer to flash
                            self.pgm_print(
                                'Writing %s to flash @ %s ...' %
                                (pretty_size(numbytes_length[index]), "0x{:08x}".format(flash_address)))
                            delta = now()
                            flash.write(flash_address, bufstr)
                            delta = now() - delta
                            self.pgm_print('Finished write in %d seconds' % int(delta))

                            # Read from flash into buffer
                            self.pgm_print(
                                'Reading %s from flash @ %s ...' %
                                (pretty_size(numbytes_length[index]), "0x{:08x}".format(flash_address)))
                            delta = now()
                            readdata = flash.read(flash_address, numbytes_length[index], sync=False)
                            delta = now() - delta
                            self.pgm_print('Finished read in %d seconds' % int(delta))

                            # compute hash from read data contents
                            rmd = sha1()
                            rmd.update(readdata)
                            newdigest = rmd.hexdigest()

                            if refdigest == newdigest:
                                self.pgm_print('Flash verify successful')
                            else:
                                self.pgm_print('ERROR: Flash verify unsuccessful... mismatch found', error=True)

                    # Restore write-protect status reg value
                    if need_restore_status_reg:
                        self.pgm_print("Restore Write-protect status register.")

                        if(getattr(flash, "FEAT_WRITE_BOTH_STATUS_REG", False)):
                            status_reg_2 = flash._read_status2(sync=False)
                            flash._write_status(status_reg, status_reg_2, sync=False)
                        else:
                            flash._write_status(status_reg, sync=False)

                    # Micon MT25Q only
                    if (jedec[0] == 0x20 and (jedec[1] == 0xBA or jedec[1] == 0xBB) and (jedec[2] in [0x19, 0x20])):
                        # Restore nv configuration reg
                        if need_restore_nv_conf_reg:
                            self.pgm_print("Restore Non-volative configuration register.")
                            flash._write_nv_conf_ref(nv_conf_reg)

                    if mode_4byte:
                        if is_flash_larger_than_16MB and (callable(enter_4byte_addr_mode) and callable(exit_4byte_addr_mode)):
                            self.pgm_print('Exit 4-byte address mode')
                            flash.exit_4byte_addr_mode(sync=False)

                    # need to enable quad/dual SPI AFTER status register restoration
                    # PROG-579
                    if enable_quad_spi:
                        flash.enable_quad_spi(sync=False)
                    if enable_dual_spi:
                        flash.enable_dual_spi(sync=False)
                    if disable_quad_spi:
                        # debug only, not for production
                        self.pgm_print('DEBUG ONLY: disabling quad spi mode')
                        flash.disable_quad_spi(sync=False)

                    if index == 0:
                        jtag_engine.switch_flash(1)

                # Print finish msg
                self.pgm_print('JTAG2SPI dual flash programming...done')

                ## FIXME: still cant reset board
                ## Send to user mode
                #sleep(0.1)
                #jtag_engine.write_ir(JTAG_INSTR['ENTERUSER'])
                #jtag_engine.write_dr(
                #    BitSequence('0' * 100, length=100,
                #                msb=True))  # extra clock ticks after ENTERUSER as well
                #jtag_engine.write_ir(JTAG_INSTR['IDCODE'])

                ## wait a bit for all operations to flush, then close the jtag channel
                #jtag_engine.reset()
                #sleep(0.2)
        except Exception as e:
            self.pgm_print(f'{e}, aborting flash programming', error=True)
            error = True
        finally:
            if jtag_session:
                JtagManager().unconfig_jtag(jtag_session)
                jtag_session = None
            if error:
                raise FtdiProgramError
            else:
                return jedec

    def realign_x8_bitstream_read_data(self, readdata: list[Tuple[int, ...]], SYNC_POS: int, WORST_CASE_SPLIT_POINT: int):

        # for x8 read specifically
        # since BITSTREAM early data is stored in each flash asymmetrically
        # in other words, first portion of bitstream is ONLY in lower flash, with upper flash just padding

        # to reconstruct original bitstream, we need to throw out the upper flash padding, spread the lower flash
        # init contents across both fields.

        # for sanity, double-confirm existence of sync pattern where it should be

        # if anything is amiss, just revert the extra read data and return

        if len(readdata) != 2:
            self.pgm_print('ERROR: Flash read for dual flash found incorrectly formatted raw data... ', error=True)
        if len(readdata[0]) != len(readdata[1]):
            self.pgm_print('ERROR: Flash read for dual flash found incorrectly formatted raw data... ', error=True)

        # sync pattern check...   0x168a2236
        expected_sync = [0x16, 0x8a, 0x22, 0x36]
        read_sync = [x for x in readdata[0][SYNC_POS:SYNC_POS+4]]

        # utility to remove excess read data from end of tuples
        def remove_bytes_from_tail(num_bytes: int):
           half_num_bytes = num_bytes // 2
           new_len = len(readdata[0]) - half_num_bytes
           readdata[0] = readdata[0][:new_len]
           readdata[1] = readdata[1][:new_len]

        # since we are wildly guessing if this is bitstream (or not), if sync pattern not found, just print
        # info message and return
        if read_sync != expected_sync:
            read_sync_str = ''
            expected_sync_str = ''
            for x in read_sync: read_sync_str += '{:02X}'.format(x)
            for x in expected_sync: expected_sync_str += '{:02X}'.format(x)
            self.pgm_print('sync pattern check: found %s but expected %s... assuming read data is not a programmed bitstream' % (read_sync_str, expected_sync_str))
            remove_bytes_from_tail(WORST_CASE_SPLIT_POINT)
            return

        # convert 256-byte pre-header to utf-8, search for device info in string contents
        preheader = ''.join(x.to_bytes(1).decode('utf-8') for x in readdata[0][0:SYNC_POS])
        #print(preheader)

        # get device_db DIE object via processing preheader text
        die = None
        device_db = bitstream_util.get_device_db()
        hp = HeaderProcessor(None, None, device_db, None)
        die_name = hp.get_die_name(preheader)
        if not die_name:
            return
        for curr_die in device_db.findall('die'):
            if curr_die.get('name') == die_name:
                die = curr_die
                break
        else:
            remove_bytes_from_tail(WORST_CASE_SPLIT_POINT)
            return

        # double check this is x8 width bitstream
        prog_width = hp.get_prog_width(preheader)
        if prog_width != 8:
            self.pgm_print('WARNING: expected x8 spi width bitstream, found x%d width instead' % prog_width)
            remove_bytes_from_tail(WORST_CASE_SPLIT_POINT)
            return

        # get opt reg length
        opt_reg = die.find('option_register')
        opt_bit_length = int(opt_reg.get('length'))
        opt_byte_length = opt_bit_length // 8

        # set SPLIT_POINT accordingly
        SPLIT_POINT = SYNC_POS + 4 + opt_byte_length


        if len(readdata[0]) <= SPLIT_POINT:
            remove_bytes_from_tail(WORST_CASE_SPLIT_POINT)
            return

        header_lower = []
        header_upper = []
        for i in range(0, SPLIT_POINT, 2):
            header_lower.append((readdata[0][i+1] & 0b00001111) | ((readdata[0][i] & 0b00001111) << 4))
            header_upper.append(((readdata[0][i+1] & 0b11110000) >> 4) | (readdata[0][i] & 0b11110000))
        #junk_pad = [x for x in readdata[1][0:SPLIT_POINT]]

        # earlier read was assuming worst case option register length
        # need to account for that here
        end_pt = len(readdata[0]) - ((WORST_CASE_SPLIT_POINT - SPLIT_POINT) // 2)

        #print(str(header_lower))
        #print(str(header_upper))
        #print(str(junk_pad))

        readdata[0] = tuple(header_lower + list(readdata[0][SPLIT_POINT:end_pt]))
        readdata[1] = tuple(header_upper + list(readdata[1][SPLIT_POINT:end_pt]))











###########################################################################################################################################################

clock_speeds = (
    # ("30 MHz", 0x0000, 30E6),
    ("15 MHz", 0x0000, 15E6),
    ("10 MHz", 0x0001, 10E6),
    ("7.5 MHz", 0x0002, 7.5E6),
    ("6 MHz", 0x0003, 6E6),
    # ("5 MHz", 0x0005, 5E5),
    # ("4.286 MHz", 0x0006, 4285714.285714285),
    # ("3.75 MHz", 0x0007, 3.75E6),
    # ("3.333 MHz", 0x0008, 3.333E6),
    # ("3 MHz", 0x0009, 3E6)
)

ti_clock_speeds = (
    # ("30 MHz", 0x0000, 30E6),
    #("15 MHz", 0x0000, 15E6),
    ("10 MHz", 0x0000, 10E6),
    ("7.5 MHz", 0x0001, 7.5E6),
    ("6 MHz", 0x0002, 6E6),
    # ("5 MHz", 0x0005, 5E5),
    # ("4.286 MHz", 0x0006, 4285714.285714285),
    # ("3.75 MHz", 0x0007, 3.75E6),
    # ("3.333 MHz", 0x0008, 3.333E6),
    # ("3 MHz", 0x0009, 3E6)
)


def get_val_from_freq(freq):
    val = (30E6 / freq) - 1
    return int(val)


def get_freq_str_from_val(val):
    '''
	val_high ,val_low = divmod(val, 0x100)
	freq = 60E6 / ((1 + ((val_high * 256) + val_low)) * 2)
	return freq
	'''
    for i in range(len(clock_speeds)):
        if val == clock_speeds[i][2]:
            return clock_speeds[i][0]


def get_freq_val_from_str(string):
    for i in range(len(clock_speeds)):
        if string == clock_speeds[i][0]:
            return clock_speeds[i][2]


def get_freq_index_from_str(string):
    for i in range(len(clock_speeds)):
        if string == clock_speeds[i][0]:
            return clock_speeds[i][1]
        continue
    return 4


def get_freq_index_from_val(val):
    for i in range(len(clock_speeds)):
        if val == clock_speeds[i][2]:
            return clock_speeds[i][1]
        continue
    return 4

def get_ti_freq_index_from_val(val):
    for i in range(len(ti_clock_speeds)):
        if val == ti_clock_speeds[i][2]:
            return ti_clock_speeds[i][1]
        continue
    return 3

def get_ti_freq_index_from_str(string):
    for i in range(len(ti_clock_speeds)):
        if string == ti_clock_speeds[i][0]:
            return ti_clock_speeds[i][1]
        continue
    return 3

def check_flash_if_supported(jedec_id):
    jedec =  bytearray.fromhex(jedec_id)
    try:
        # TODO, I cannot use __str__() at here for At45FlashDevice because At45FlashDevice class has some handshake with
        # actual hardware during init, and it will return error if we didnt connect to actual hardware.
        #
        #flash = SerialFlashManager._get_flash("", jedec)
        devices = []
        match = 0
        contents = sys.modules['spiflash.serialflash'].__dict__
        for name in contents:
            if name.endswith('FlashDevice') and not name.startswith('_'):
                devices.append(contents[name])
            # Here, we extend our implementation over original FlashDevice classes
            if name.endswith('FlashDeviceExt') and not name.startswith('_'):
                original_device = name[:-3]
                devices.remove(contents[original_device])
                devices.append(contents[name])
        for device in devices:
            if device.match(jedec):
                match = 1
                break
        if not match:
            raise SerialFlashUnknownJedec(jedec)
        else:
            print("Found matched in Efinix Flash INTERNAL Database")
            print('JEDEC ID   : %s' %  jedec_id)
            print("CLASS      : %s" % device.__name__)
            #print('Found matched JEDEC_ID=%s in Efinix Flash INTERNAL Database.' %  jedec_id)
            if device.__name__ == "At45FlashDevice":
                print("EXTRA INFO : NA")
            else:
                print("EXTRA INFO : %s" % device(None,jedec).__str__())
    except SerialFlashUnknownJedec as exc :
        flash = efinix_flash.EfinixSupportedFlash(None, jedec)
        if flash.__str__() != "":
            print("Found matched in Efinix Flash JSON Database")
            print('JEDEC ID   : %s' %  jedec_id)
            print("CLASS      : %s" % flash.__class__.__name__)
            #print('Found matched JEDEC_ID=%s in Efinix Flash JSON Database.' %  jedec_id)
            print("EXTRA INFO : %s" % flash.__str__())
        else:
            print('Unrecognized flash device, JEDEC_ID=%s.' %  jedec_id)
    except Exception as exc:
        print("Something not right - %s" % exc)
        raise Exception(str(exc))

def list_usb(connections):
    print("Available USB targets:")
    for target in connections:
        print(f"Usb Target: {target.description}, {target.USB_ID}")
        for idx, url in enumerate(target.URLS):
            print(f"\turl[{idx}]: {url}")

def set_up_pinout_from_profile(profile):
    selector = EfxHwBoardProfileSelector([
            Path(os.environ["EFXPGM_HOME"], 'bin', 'efx_pgm', 'efx_hw_common', 'boards')
        ])
    print("Using board profile: '{}' for pinout connection".format(profile))
    profile = list(filter(lambda p: p.name == profile, selector.get_available_profiles()))[0]
    return profile.pinout

def set_up_urls(ftdi: FtdiProgram, connections: list, url: str, aurl: str) -> list[str]:
    '''
    Set up URLs to be used during programming operations
    '''
    # determine correct set of urls
    usb_targets = list(filter(lambda target: isinstance(target, EfxFTDIDevice), connections))
    if url:
        for target in usb_targets:
            if url in target.URLS:
                url_list = target.URLS
                break
        else:
            raise ValueError('Specified FTDI URL could not be matched to any connected USB targets')
    else:
        url_list = usb_targets[0].URLS

    # 4232 interface supports 4 busses
    bus_to_url_idx = {
        'A': 0,
        'B': 1,
        'C': 2,
        'D': 3
    }
    # determine SPI and JTAG urls
    jtag_url_idx = bus_to_url_idx[ftdi.get_jtag_bus_from_pinout()]
    spi_url_idx = bus_to_url_idx[ftdi.get_spi_bus_from_pinout()]
    if 'jtag' in ftdi.mode.lower():
        jtag_url = url if url else url_list[jtag_url_idx]
        spi_url = aurl if aurl else url_list[spi_url_idx]
    else:
        jtag_url = aurl if aurl else url_list[jtag_url_idx]
        spi_url = url if url else url_list[spi_url_idx]

    # set SPI and JTAG urls at correct indices
    ftdi_urls = [None for _ in bus_to_url_idx.keys()]
    ftdi_urls[jtag_url_idx] = jtag_url
    ftdi_urls[spi_url_idx] = spi_url
    return ftdi_urls

def validate_address_and_bytes(address, num_bytes):
    # Make sure the "address" and "num_bytes" are in correct format (int), convert hex to int if passed value start with 0x
    addr_validator = r'^(0x[0-9A-F]{1,8})|([0-9]{1,10})$'

    if re.fullmatch(addr_validator, address, re.IGNORECASE):
        if address.startswith('0x'):
            address = int(address, 16)
        else:
            address = int(address, 10)

    if re.fullmatch(addr_validator, num_bytes, re.IGNORECASE):
        if num_bytes.startswith('0x'):
            num_bytes = int(num_bytes, 16)
        else:
            num_bytes = int(num_bytes, 10)

    return address, num_bytes

def check_mode_file_compatibility(prog_mode, input_file):
    allowed_modes = {
                            'jtag' : {'display_name':'JTAG', 'allowed':'.bit'},
                            'jtag_chain' : {'display_name':'JTAG Chain', 'allowed':'.bit'},
                            'passive' : {'display_name':'SPI Passive', 'allowed':'.hex'},
                            'active': {'display_name':'SPI Active', 'allowed':'.hex'},
                            'jtag_bridge': {'display_name':'SPI Active using JTAG Bridge (legacy)', 'allowed':'.hex'},
                            'jtag_bridge_x8': {'display_name':'SPI Active x8 using JTAG Bridge (legacy)', 'allowed':'.hex'},
                            'jtag_bridge_new': {'display_name':'SPI Active using JTAG Bridge (new)', 'allowed':'.hex'},
                            'jtag_bridge_x8_new': {'display_name':'SPI Active x8 using JTAG Bridge (new)', 'allowed':'.hex'}
                        }
    file_ext = PurePath(input_file).suffix
    if file_ext not in allowed_modes[prog_mode]['allowed']:
        print('ERROR: Incompatible file extension for programming mode, '\
                'please use %s file for %s programming' % (allowed_modes[prog_mode]['allowed'], allowed_modes[prog_mode]['display_name']))
        return False
    else:
        return True

# cmd-line parser
def parse_cmdline():

    parser = argparse.ArgumentParser(
        description='Efinity programmer for FTDI cable')
    parser.add_argument('input_file', nargs='?', default='', help='HEX file generated from efx_pgm')
    parser.add_argument('-m',
                        '--mode',
                        choices=[
                            'passive', 'active', 'jtag', 'jtag_chain','erase_flash', 'read_flash',
                            'jtag_bridge', 'jtag_bridge_x8', 'jtag_bridge_new', 'jtag_bridge_x8_new'
                        ],
                        default='passive',
                        help='programming mode')
    parser.add_argument('-o',
                        '--output_file',
                        help='output file used for read_flash mode')

    # URL is based on which exact FTDI chipset used in cable
    # see 'https://eblot.github.io/pyftdi/urlscheme.html' for more details
    parser.add_argument('-u',
                        '--url',
                        default='',
                        help='FTDI URL')
    parser.add_argument('-a', '--aurl', default='', help='(DEPRECATED) ALT URL')
    parser.add_argument('-x',
                        '--xml',
                        default='',
                        help='XML file for JTAG programming')
    parser.add_argument('-n',
                        '--num',
                        default='1',
                        help='Chip target number for JTAG chain programming')

    selector = EfxHwBoardProfileSelector([
        Path(os.environ["EFXPGM_HOME"], 'bin', 'efx_pgm', 'efx_hw_common', 'boards')
    ])
    parser.add_argument('-b',
                        '--board_profile',
                        choices=[
                            f"{p.name}" for p in selector.get_available_profiles()
                        ],
                        default=None,
                        help='Name of the board profile used')

    parser.add_argument('--address',
                        default='0',
                        help='Starting flash address for flash read/write operations')

    parser.add_argument('--num_bytes',
                        default='-1',
                        help='Number of bytes to erase or read (for modes \"erase\" and \"read\" only).')

    parser.add_argument('--burst_size',
                        default=-1,
                        help='Individual read/write burst size (in multiples of 256 bytes). For legacy JTAG Bridge modes only (i.e. ONLY "jtag_bridge" and "jtag_bridge_x8").')

    parser.add_argument('--jtag_bridge_mode',
                        choices=['erase', 'write', 'erase_and_write', 'read', 'all_no_erase', 'all'],
                        default='all',
                        help='jtag bridge programming mode')

    parser.add_argument('--jtag_clock_freq',
                        default='6000000',
                        help='Set JTAG clock frequency')

    parser.add_argument('--check_flash_if_supported',
                        default=None,
                        help="Check if flash is supported using JEDEC_ID_HEX_STRING. Eg: C84012")

    parser.add_argument('-l', '--list_usb',  action='store_true', help="List available usb target's URL")

    # run parser
    args = parser.parse_args()

    return args

def apply_hwtools_config():
    '''
    Apply settings defined in hw_tools.ini for command-line programming
    '''
    try:
        hwtconfig_util = HwToolsConfigUtil()
    except ParsingError:
        print("WARNING: Unable to read hw_tools.ini, using default settings")
        legacy_mode = False
        fast_spi_active_mode = False
        os.environ['FAST_SPI_ACTIVE_MODE'] = str(False)
        creset_before_jtag_config = False

    try:
        hwtconfig_util = HwToolsConfigUtil()
    except json.JSONDecodeError:
        print("WARNING: Unable to decode hw_tools.json, using default settings")
        legacy_mode = False
        fast_spi_active_mode = False
        os.environ['FAST_SPI_ACTIVE_MODE'] = str(False)
        creset_before_jtag_config = False
    else:

        if hwtconfig_util.migrated_from_ini():
            print("Migrated hw_tools.ini to hw_tools.json")
            hwtconfig_util.remove_migrate_flag()

        legacy_mode = hwtconfig_util.read_setting(sect='config', opt='legacy_mode')

        fast_spi_active_mode = hwtconfig_util.read_setting(sect='config', opt='fast_spi_active_mode')
        os.environ['FAST_SPI_ACTIVE_MODE'] = str(fast_spi_active_mode)

        creset_before_jtag_config = hwtconfig_util.read_setting(sect='config', opt='creset_before_jtag_config')


    return legacy_mode, fast_spi_active_mode, creset_before_jtag_config


# script entry point
if __name__ == '__main__':
    legacy_mode, _, creset_before_jtag_config = apply_hwtools_config()
    args = parse_cmdline()

    if args.check_flash_if_supported:
        try:
            check_flash_if_supported(args.check_flash_if_supported)
        except Exception as exc:
            print('Error: exception encountered - ', str(exc))
            sys.exit(1)
        else:
            sys.exit(0)

    usb_connects = UsbResolver().get_usb_connections()
    if len(usb_connects) == 0:
        print('ERROR: No USB target detected, aborting!')
        sys.exit(1)

    if args.list_usb:
        list_usb(usb_connects)
        sys.exit(0)

    if len(usb_connects) > 1 and args.url == '':
        print('ERROR: More than one USB target detected, FTDI URL must be specified, aborting!')
        sys.exit(1)

    if args.board_profile:
        args.pinout = set_up_pinout_from_profile(args.board_profile)
    args.address, args.num_bytes = validate_address_and_bytes(args.address, args.num_bytes)
    args.urls = []

    prog = FtdiProgram(args)
    try:
        prog.urls = set_up_urls(prog, usb_connects, args.url, args.aurl)
    except ValueError as exc:
        print('ERROR: %s' % str(exc))
        sys.exit(1)
    prog.creset_before_jtag_config = creset_before_jtag_config

    prog_modes = ['passive', 'active', 'jtag', 'jtag_chain', 'jtag_bridge', 'jtag_bridge_new', 'jtag_bridge_x8', 'jtag_bridge_x8_new']
    if legacy_mode == False and args.mode in prog_modes:
        if not check_mode_file_compatibility(args.mode, args.input_file):
            sys.exit(1)

    try:
        if prog.mode == 'passive':
            prog.spi_direct_program()
        elif prog.mode == 'active':
            prog.spi_flash_program(flash_address=args.address,
                                erase_entire_flash=False)
            #spi_flash_program(args, flash_address=0x00080000, erase_entire_flash=True)
        elif prog.mode == 'jtag':
            prog.jtag_program()
        elif prog.mode == 'jtag_chain':
            prog.jtag_program(chain=True)
        elif prog.mode == 'erase_flash':
            prog.spi_erase_flash(start=args.address,
                                length=args.num_bytes)
        elif prog.mode == 'read_flash':
            #def spi_read_flash(self, start_address, read_length, output_file):
            prog.spi_read_flash(start_address=args.address,
                                read_length=args.num_bytes,
                                output_file=args.output_file)
        elif prog.mode == 'jtag_bridge':
            prog.jtag_bridge()
        elif prog.mode == 'jtag_bridge_x8':
            prog.jtag_bridge_x8()
        elif prog.mode == 'jtag_bridge_new':
            prog.jtag2spi()
        elif prog.mode == 'jtag_bridge_x8_new':
            prog.jtag2spi_x8()
        else:
            print('unknown programming mode: %s' % prog.mode)
            sys.exit(1)
    except (FtdiProgramError, ValueError, AssertionError):
        print('ERROR: problem occurred during {} programming, aborting!'.format(args.mode))
        sys.exit(1)
    sys.exit(0)
'''
 Copyright (C) 2013-2020 Efinix Inc. All rights reserved.

 This   document  contains  proprietary information  which   is
 protected by  copyright. All rights  are reserved.  This notice
 refers to original work by Efinix, Inc. which may be derivitive
 of other work distributed under license of the authors.  In the
 case of derivative work, nothing in this notice overrides the
 original author's license agreement.  Where applicable, the
 original license agreement is included in it's original
 unmodified form immediately below this header.

 WARRANTY DISCLAIMER.
     THE  DESIGN, CODE, OR INFORMATION ARE PROVIDED “AS IS” AND
     EFINIX MAKES NO WARRANTIES, EXPRESS OR IMPLIED WITH
     RESPECT THERETO, AND EXPRESSLY DISCLAIMS ANY IMPLIED WARRANTIES,
     INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
     MERCHANTABILITY, NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR
     PURPOSE.  SOME STATES DO NOT ALLOW EXCLUSIONS OF AN IMPLIED
     WARRANTY, SO THIS DISCLAIMER MAY NOT APPLY TO LICENSEE.

 LIMITATION OF LIABILITY.
     NOTWITHSTANDING ANYTHING TO THE CONTRARY, EXCEPT FOR BODILY
     INJURY, EFINIX SHALL NOT BE LIABLE WITH RESPECT TO ANY SUBJECT
     MATTER OF THIS AGREEMENT UNDER TORT, CONTRACT, STRICT LIABILITY
     OR ANY OTHER LEGAL OR EQUITABLE THEORY (I) FOR ANY INDIRECT,
     SPECIAL, INCIDENTAL, EXEMPLARY OR CONSEQUENTIAL DAMAGES OF ANY
     CHARACTER INCLUDING, WITHOUT LIMITATION, DAMAGES FOR LOSS OF
     GOODWILL, DATA OR PROFIT, WORK STOPPAGE, OR COMPUTER FAILURE OR
     MALFUNCTION, OR IN ANY EVENT (II) FOR ANY AMOUNT IN EXCESS, IN
     THE AGGREGATE, OF THE FEE PAID BY LICENSEE TO EFINIX HEREUNDER
     (OR, IF THE FEE HAS BEEN WAIVED, $100), EVEN IF EFINIX SHALL HAVE
     BEEN INFORMED OF THE POSSIBILITY OF SUCH DAMAGES.  SOME STATES DO
     NOT ALLOW THE EXCLUSION OR LIMITATION OF INCIDENTAL OR
     CONSEQUENTIAL DAMAGES, SO THIS LIMITATION AND EXCLUSION MAY NOT
     APPLY TO LICENSEE.
'''
