from array import array
from typing import Optional, Union

from pyftdi.bits import BitSequence
from efx_pgm.efx_hw_server.server_pb2 import (BitSequenceMessage, JtagConfigureRequest, JtagConnectionRequest, JtagReadRequest,
                                              JtagChangeStateRequest, JtagWriteRequest, JtagSMPathRequest, JtagSMPath, JtagSMEventsRequest)
from efx_pgm.efx_hw_server.server_pb2 import Jtag2ApbWriteRequest, Jtag2ApbReadRequest, Jtag2ApbConfigureRequest, Jtag2ApbUnconfigureRequest
import grpc
from efx_pgm.efx_hw_client.grpc_client import EfxHwAPIClient
from efx_pgm.efx_hw_server.server_pb2_grpc import Jtag2ApbEngineHandlerStub
from contextlib import contextmanager

from tornado.escape import url_unescape, url_escape

def format_write_request(bseq: BitSequence, request: JtagWriteRequest = None) -> JtagWriteRequest:
    if request is None:
        request = JtagWriteRequest()
    request.data.data.extend(bseq.sequence())
    return request

def bs_to_message(bseq: BitSequence) -> BitSequenceMessage:
    result = BitSequenceMessage()
    result.data.extend(bseq.sequence())
    return result

def message_to_bs(message: BitSequenceMessage) -> BitSequence:
    result = BitSequence()
    for bit in message.data:
        result.append(bit)
    return result


class Jtag2ApbRemoteController:
    def __init__(self, **kwargs):
        self._api_client = None
        self._session = None
        self._params = kwargs
        self._conn_id = 0

    @contextmanager
    def get_handler_stub(self):
        with grpc.insecure_channel(self._api_client.get_base_url()) as channel:
            stub = Jtag2ApbEngineHandlerStub(channel)
            yield stub

    def configure(self, url, target_num, frequency, tap, user):
        tokens = url.split('/')
        base_url = tokens[2]
        self._api_client = EfxHwAPIClient(base_url=base_url)

        def get_param_from_url(u, param_name):
            return [i.split("=")[-1] for i in u.split("?", 1)[-1].split("&") if i.startswith(param_name + "=")][0]
        dev_url = get_param_from_url(url, 'dev_url')

        with self.get_handler_stub() as stub:
            request = Jtag2ApbConfigureRequest()
            request.header.connection_id = 0
            request.url = url
            request.target_num = target_num
            request.frequency = frequency
            request.tap = tap
            request.user = user
            request.trst = False
            response = stub.configure(request)
            resp = response.resp

            print('**********************************************')
            print(resp)
            print('**********************************************')
            return resp

    def unconfigure(self):
        with self.get_handler_stub() as stub:
            request = Jtag2ApbUnconfigureRequest()
            request.header.connection_id = 0
            response = stub.unconfigure(request)
            resp = response.resp
            print('**********************************************')
            print(resp)
            print('**********************************************')
            return resp

    #def close(self):
    #    print("JtagRemoteEngine close")
    #    with self.get_handler_stub() as stub:
    #        request = JtagConnectionRequest()
    #        request.header.connection_id = self._conn_id
    #        stub.close(request)

    #    if self._api_client:
    #        conn_client = self._api_client.get_connection_client()
    #        conn_client.delete_connection(self._conn_id)
    #        self._api_client = None
    #        self._conn_id = None

    #def purge(self):
    #    with self.get_handler_stub() as stub:
    #        request = JtagConnectionRequest()
    #        request.header.connection_id = self._conn_id
    #        stub.purge(request)

    #def reset(self):
    #    with self.get_handler_stub() as stub:
    #        request = JtagConnectionRequest()
    #        request.header.connection_id = self._conn_id
    #        stub.reset(request)

    def read(self, address: bytes) -> BitSequence:
        with self.get_handler_stub() as stub:
            request = Jtag2ApbReadRequest()
            request.header.connection_id = self._conn_id
            request.address = address
            response: BitSequenceMessage = stub.read(request)
            resp = message_to_bs(response)
            return resp

    def write(self, address: bytes, data: bytes, wrparity: BitSequence, use_last: bool = True):
        with self.get_handler_stub() as stub:
            request = Jtag2ApbWriteRequest()
            request.header.connection_id = self._conn_id
            request.address = address
            request.data = data
            request.wrparity.data.extend(wrparity.sequence())
            response: BitSequenceMessage = stub.write(request)
            resp = message_to_bs(response)
            return resp

    def burst_read(self, pi: bytes, address: bytes, in_value: bytes, wrparity: BitSequence,burst_size: bytes, use_last: bool = True):
        with self.get_handler_stub() as stub:
            data = BitSequence(0, length=32)
            if (in_value):
                data = data | in_value
            data = data | (BitSequence((min(pi, 0x7F)), length=32))
            data = data.tobytes()
            request = Jtag2ApbWriteRequest()
            request.header.connection_id = self._conn_id
            request.address = address
            request.data = bytes(data)
            response: BitSequenceMessage = stub.burst_read(request)
            resp = message_to_bs(response)
            return resp