import argparse
import os.path
import shutil
import fileinput
import re
import subprocess
import sys
import time
import os
import codecs
from pathlib import Path
from typing import Dict, List
from pyrsistent import pmap, freeze
from pyrsistent.typing import PMap, PVector
from typing import Mapping, Sequence
import xml.etree.ElementTree as ElementTree

def multiwordReplace(text, wordDic):
        """
        take a text and replace words that match a key in a dictionary with
        the associated value, return the changed text
        """
        rc = re.compile('|'.join(map(re.escape, wordDic)))
        def translate(match):
                return wordDic[match.group(0)]
        return rc.sub(translate, text)

def parse_var(s):
        """
        Parse a key, value pair, separated by '='
        That's the reverse of ShellArgs.

        On the command line (argparse) a declaration will typically look like:
                foo=hello
        or
                foo="hello world"
        """
        items = s.split('=')
        key = items[0].strip() # we remove blanks around keys, as is logical
        if len(items) > 1:
                        # rejoin the rest:
                        value = '='.join(items[1:])
        return (key, value)

def parse_parameters(params: List[str]) -> Dict[str, str]:
        """
        Parse the key-value pairs and return A dictionary
        """
        d = {}
        for item in params:
                        key, value = parse_var(item)
                        d[key] = value
        return d



def msb_len(n):
    bits = 0
    while n >> bits: bits += 1
    return bits

def intr_decode(n):
    intr_decode_dict = {
        0:"A",1:"B",2:"C",3:"D",4:"E",5:"F",6:"G",7:"H",
        8:"I",9:"J",10:"K",11:"L",12:"M",13:"N",14:"O",15:"P",
        16:"Q",17:"R",18:"S",19:"T",20:"U",21:"V",22:"W",23:"X"
    }
    return intr_decode_dict[n]

sl0=["0"]
sl1=["1"]
sl2=["2"]
sl3=["3"]
sl4=["4"]

top_spi="output\t\tsystem_spi_{}_io_sclk_write,\n\
output\t\tsystem_spi_{}_io_data_0_writeEnable,\n\
input\t\tsystem_spi_{}_io_data_0_read,\n\
output\t\tsystem_spi_{}_io_data_0_write,\n\
output\t\tsystem_spi_{}_io_data_1_writeEnable,\n\
input\t\tsystem_spi_{}_io_data_1_read,\n\
output\t\tsystem_spi_{}_io_data_1_write,\n\
output\t\tsystem_spi_{}_io_data_2_writeEnable,\n\
input\t\tsystem_spi_{}_io_data_2_read,\n\
output\t\tsystem_spi_{}_io_data_2_write,\n\
output\t\tsystem_spi_{}_io_data_3_writeEnable,\n\
input\t\tsystem_spi_{}_io_data_3_read,\n\
output\t\tsystem_spi_{}_io_data_3_write,\n\
output [3:0] system_spi_{}_io_ss,\n"
top_uart="output\t\tsystem_uart_{}_io_txd,\n\
input\t\tsystem_uart_{}_io_rxd,\n"
top_i2c="output\t\tsystem_i2c_{}_io_sda_writeEnable,\n\
output\t\tsystem_i2c_{}_io_sda_write,\n\
input\t\tsystem_i2c_{}_io_sda_read,\n\
output\t\tsystem_i2c_{}_io_scl_writeEnable,\n\
output\t\tsystem_i2c_{}_io_scl_write,\n\
input\t\tsystem_i2c_{}_io_scl_read,\n"
top_gpio="input [{}:0] system_gpio_{}_io_read,\n\
output [{}:0] system_gpio_{}_io_write,\n\
output [{}:0] system_gpio_{}_io_writeEnable,\n"
top_wdt="\
output\t\tsystem_watchdog_hardPanic_reset,\n"
top_hjtag="\
output\t\tjtagCtrl_tdi,\n\
input\t\tjtagCtrl_tdo,\n\
output\t\tjtagCtrl_enable,\n\
output\t\tjtagCtrl_capture,\n\
output\t\tjtagCtrl_shift,\n\
output\t\tjtagCtrl_update,\n\
output\t\tjtagCtrl_reset,\n\
input\t\tut_jtagCtrl_tdi,\n\
output\t\tut_jtagCtrl_tdo,\n\
input\t\tut_jtagCtrl_enable,\n\
input\t\tut_jtagCtrl_capture,\n\
input\t\tut_jtagCtrl_shift,\n\
input\t\tut_jtagCtrl_update,\n\
input\t\tut_jtagCtrl_reset,\n"
top_sjtag="\
input\t\tpin_io_jtag_tms,\n\
input\t\tpin_io_jtag_tdi,\n\
output\t\tpin_io_jtag_tdo,\n\
output\t\tio_jtag_tms,\n\
output\t\tio_jtag_tdi,\n\
input\t\tio_jtag_tdo,\n"
top_uintr="output\t\tuserInterrupt{},\n"
top_apb="\
output [{}:0] io_apbSlave_{}_PADDR,\n\
output\t\tio_apbSlave_{}_PSEL,\n\
output\t\tio_apbSlave_{}_PENABLE,\n\
input\t\tio_apbSlave_{}_PREADY,\n\
output\t\tio_apbSlave_{}_PWRITE,\n\
output [31:0] io_apbSlave_{}_PWDATA,\n\
input [31:0] io_apbSlave_{}_PRDATA,\n\
input\t\tio_apbSlave_{}_PSLVERROR,\n"
apb3_list0=["msb_len({PERI_APB_0_SIZE}-1)-1"]
apb3_list0.extend([x for x in sl0 for _ in range(8)])
apb3_list1=["msb_len({PERI_APB_1_SIZE}-1)-1"]
apb3_list1.extend([x for x in sl1 for _ in range(8)])
apb3_list2=["msb_len({PERI_APB_2_SIZE}-1)-1"]
apb3_list2.extend([x for x in sl2 for _ in range(8)])
apb3_list3=["msb_len({PERI_APB_3_SIZE}-1)-1"]
apb3_list3.extend([x for x in sl3 for _ in range(8)])
apb3_list4=["msb_len({PERI_APB_4_SIZE}-1)-1"]
apb3_list4.extend([x for x in sl4 for _ in range(8)])


wire_i2c_intr="wire\t\tsystem_i2c_{}_io_interrupt;\n"
wire_spi_intr="wire\t\tsystem_spi_{}_io_interrupt;\n"
wire_uart_intr="wire\t\tsystem_uart_{}_io_interrupt;\n"
wire_gpio_intr="\
wire\t\tsystem_gpio_{}_io_interrupt_0;\n\
wire\t\tsystem_gpio_{}_io_interrupt_1;\n"
wire_wdt_intr="wire\t\tsystem_watchdog_softPanic_interrupt;\n"
conn_i2c="\
assign system_i2c_{}_io_sda_writeEnable=!system_i2c_{}_io_sda_write;\n\
assign system_i2c_{}_io_scl_writeEnable=!system_i2c_{}_io_scl_write;\n"
conn_hjtag="\
assign jtagCtrl_capture = ut_jtagCtrl_capture;\n\
assign jtagCtrl_enable  = ut_jtagCtrl_enable;\n\
assign jtagCtrl_reset   = ut_jtagCtrl_reset;\n\
assign jtagCtrl_shift   = ut_jtagCtrl_shift;\n\
assign jtagCtrl_tdi     = ut_jtagCtrl_tdi;\n\
assign jtagCtrl_update  = ut_jtagCtrl_update;\n\
assign ut_jtagCtrl_tdo  = jtagCtrl_tdo;\n\n"
conn_sjtag="\
assign io_jtag_tdi = pin_io_jtag_tdi;\n\
assign io_jtag_tms = pin_io_jtag_tms;\n\
assign pin_io_jtag_tdo = io_jtag_tdo;\n\n"
conn_axis="\
/**/\n\
/*\tINFO: USER TO MODIFY CODES BELOW\t\t\t\t\t\t*/\n\
/*\tINFO: REFER EXAMPLE DESIGN FOR IMPLEMENTATION DETAILS\t*/\n\
/**/\n\
assign axiA_awready = 1'b1;\n\
assign axiA_wready = 1'b1;\n\
assign axiA_bresp = 2'b00;\n\
assign axiA_bvalid = 1'b0;\n\
assign axiA_arready = 1'b1;\n\
assign axiA_rdata = 32'd0;\n\
assign axiA_rresp = 2'b00;\n\
assign axiA_rlast = 1'b0;\n\
assign axiA_rvalid = 1'b0;\n\
//axiA_awaddr;\n\
//axiA_awlen;\n\
//axiA_awsize;\n\
//axiA_awburst;\n\
//axiA_awlock;\n\
//axiA_awcache;\n\
//axiA_awprot;\n\
//axiA_awqos;\n\
//axiA_awregion;\n\
//axiA_awvalid;\n\
//axiA_wdata;\n\
//axiA_wstrb;\n\
//axiA_wvalid;\n\
//axiA_wlast;\n\
//axiA_bready;\n\
//axiA_araddr;\n\
//axiA_arlen;\n\
//axiA_arsize;\n\
//axiA_arburst;\n\
//axiA_arlock;\n\
//axiA_arcache;\n\
//axiA_arprot;\n\
//axiA_arqos;\n\
//axiA_arregion;\n\
//axiA_arvalid;\n\
//axiA_rready;\n"

port_apb="\
.io_apbSlave_{}_PADDR(io_apbSlave_{}_PADDR),\n\
.io_apbSlave_{}_PSEL(io_apbSlave_{}_PSEL),\n\
.io_apbSlave_{}_PENABLE(io_apbSlave_{}_PENABLE),\n\
.io_apbSlave_{}_PREADY(io_apbSlave_{}_PREADY),\n\
.io_apbSlave_{}_PWRITE(io_apbSlave_{}_PWRITE),\n\
.io_apbSlave_{}_PWDATA(io_apbSlave_{}_PWDATA),\n\
.io_apbSlave_{}_PRDATA(io_apbSlave_{}_PRDATA),\n\
.io_apbSlave_{}_PSLVERROR(io_apbSlave_{}_PSLVERROR),\n"
port_gpio="\
.system_gpio_{}_io_interrupts_0(system_gpio_{}_io_interrupt_0),\n\
.system_gpio_{}_io_interrupts_1(system_gpio_{}_io_interrupt_1),\n\
.system_gpio_{}_io_read(system_gpio_{}_io_read),\n\
.system_gpio_{}_io_write(system_gpio_{}_io_write),\n\
.system_gpio_{}_io_writeEnable(system_gpio_{}_io_writeEnable),\n"
port_i2c="\
.system_i2c_{}_io_interrupt(system_i2c_{}_io_interrupt),\n\
.system_i2c_{}_io_sda_write(system_i2c_{}_io_sda_write),\n\
.system_i2c_{}_io_sda_read(system_i2c_{}_io_sda_read),\n\
.system_i2c_{}_io_scl_write(system_i2c_{}_io_scl_write),\n\
.system_i2c_{}_io_scl_read(system_i2c_{}_io_scl_read),\n"
port_uart="\
.system_uart_{}_io_interrupt(system_uart_{}_io_interrupt),\n\
.system_uart_{}_io_txd(system_uart_{}_io_txd),\n\
.system_uart_{}_io_rxd(system_uart_{}_io_rxd),\n"
port_spi="\
.system_spi_{}_io_interrupt(system_spi_{}_io_interrupt),\n\
.system_spi_{}_io_sclk_write(system_spi_{}_io_sclk_write),\n\
.system_spi_{}_io_data_0_writeEnable(system_spi_{}_io_data_0_writeEnable),\n\
.system_spi_{}_io_data_0_read(system_spi_{}_io_data_0_read),\n\
.system_spi_{}_io_data_0_write(system_spi_{}_io_data_0_write),\n\
.system_spi_{}_io_data_1_writeEnable(system_spi_{}_io_data_1_writeEnable),\n\
.system_spi_{}_io_data_1_read(system_spi_{}_io_data_1_read),\n\
.system_spi_{}_io_data_1_write(system_spi_{}_io_data_1_write),\n\
.system_spi_{}_io_data_2_writeEnable(system_spi_{}_io_data_2_writeEnable),\n\
.system_spi_{}_io_data_2_read(system_spi_{}_io_data_2_read),\n\
.system_spi_{}_io_data_2_write(system_spi_{}_io_data_2_write),\n\
.system_spi_{}_io_data_3_writeEnable(system_spi_{}_io_data_3_writeEnable),\n\
.system_spi_{}_io_data_3_read(system_spi_{}_io_data_3_read),\n\
.system_spi_{}_io_data_3_write(system_spi_{}_io_data_3_write),\n\
.system_spi_{}_io_ss(system_spi_{}_io_ss),\n"
port_wdt="\
.system_watchdog_hardPanic_reset(system_watchdog_hardPanic_reset),\n\
.system_watchdog_logic_panics_0(system_watchdog_softPanic_interrupt),\n"

tp_dict = pmap({
        "TP_spi_0"      : ["{PERI_SPI_0}==1 and {PERI_GEN}==1", top_spi,[x for x in sl0 for _ in range(14)]],
        "TP_spi_1"      : ["{PERI_SPI_1}==1 and {PERI_GEN}==1", top_spi,[x for x in sl1 for _ in range(14)]],
        "TP_spi_2"      : ["{PERI_SPI_2}==1 and {PERI_GEN}==1", top_spi,[x for x in sl2 for _ in range(14)]],
        "TP_uart_0"     : ["{PERI_UART_0}==1 and {PERI_GEN}==1",top_uart,[x for x in sl0 for _ in range(2)]],
        "TP_uart_1"     : ["{PERI_UART_1}==1 and {PERI_GEN}==1",top_uart,[x for x in sl1 for _ in range(2)]],
        "TP_uart_2"     : ["{PERI_UART_2}==1 and {PERI_GEN}==1",top_uart,[x for x in sl2 for _ in range(2)]],
        "TP_i2c_0"      : ["{PERI_I2C_0}==1 and {PERI_GEN}==1",top_i2c,[x for x in sl0 for _ in range(6)]],
        "TP_i2c_1"      : ["{PERI_I2C_1}==1 and {PERI_GEN}==1",top_i2c,[x for x in sl1 for _ in range(6)]],
        "TP_i2c_2"      : ["{PERI_I2C_2}==1 and {PERI_GEN}==1",top_i2c,[x for x in sl2 for _ in range(6)]],
        "TP_gpio_0"     : ["{PERI_GPIO_0}==1 and {PERI_GEN}==1",top_gpio,["{PERI_GPIO_0_WIDTH}-1","0","{PERI_GPIO_0_WIDTH}-1","0","{PERI_GPIO_0_WIDTH}-1","0"]],
        "TP_gpio_1"     : ["{PERI_GPIO_1}==1 and {PERI_GEN}==1",top_gpio,["{PERI_GPIO_1_WIDTH}-1","1","{PERI_GPIO_1_WIDTH}-1","1","{PERI_GPIO_1_WIDTH}-1","1"]],
        "TP_wdt_0"      : ["{PERI_WDT_0}==1 and {PERI_GEN}==1",top_wdt,[]],
        "TP_softap"     : ["{INTF_JTAG_TYPE}==1",top_sjtag,[]],
        "TP_hardtap"    : ["{INTF_JTAG_TYPE}==0",top_hjtag,[]],
        "TP_uintr_a"    : ["{PERI_COUNT}>=1",top_uintr,["A"]],
        "TP_uintr_b"    : ["{PERI_COUNT}>=2",top_uintr,["B"]],
        "TP_uintr_c"    : ["{PERI_COUNT}>=3",top_uintr,["C"]],
        "TP_uintr_d"    : ["{PERI_COUNT}>=4",top_uintr,["D"]],
        "TP_uintr_e"    : ["{PERI_COUNT}>=5",top_uintr,["E"]],
        "TP_uintr_f"    : ["{PERI_COUNT}>=6",top_uintr,["F"]],
        "TP_uintr_g"    : ["{PERI_COUNT}>=7",top_uintr,["G"]],
        "TP_uintr_h"    : ["{PERI_COUNT}>=8",top_uintr,["H"]],
        "TP_uintr_i"    : ["{PERI_COUNT}>=9",top_uintr,["I"]],
        "TP_uintr_j"    : ["{PERI_COUNT}>=10",top_uintr,["J"]],
        "TP_uintr_k"    : ["{PERI_COUNT}>=11",top_uintr,["K"]],
        "TP_uintr_l"    : ["{PERI_COUNT}>=12",top_uintr,["L"]],
        "TP_uintr_m"    : ["{PERI_COUNT}>=13",top_uintr,["M"]],
        "TP_uintr_n"    : ["{PERI_COUNT}>=14",top_uintr,["N"]],
        "TP_uintr_o"    : ["{PERI_COUNT}>=15",top_uintr,["O"]],
        "TP_uintr_p"    : ["{PERI_COUNT}>=16",top_uintr,["P"]],
        "TP_uintr_q"    : ["{PERI_COUNT}>=17",top_uintr,["Q"]],
        "TP_uintr_r"    : ["{PERI_COUNT}>=18",top_uintr,["R"]],
        "TP_uintr_s"    : ["{PERI_COUNT}>=19",top_uintr,["S"]],
        "TP_uintr_t"    : ["{PERI_COUNT}>=20",top_uintr,["T"]],
        "TP_uintr_u"    : ["{PERI_COUNT}>=21",top_uintr,["U"]],
        "TP_uintr_v"    : ["{PERI_COUNT}>=22",top_uintr,["V"]],
        "TP_uintr_w"    : ["{PERI_COUNT}>=23",top_uintr,["W"]],
        "TP_uintr_x"    : ["{PERI_COUNT}>=24",top_uintr,["X"]],
        "TP_apb3_0"     : ["{PERI_APB_0}==1 and {PERI_GEN}==1",top_apb,apb3_list0],
        "TP_apb3_1"     : ["{PERI_APB_1}==1 and {PERI_GEN}==1",top_apb,apb3_list1],
        "TP_apb3_2"     : ["{PERI_APB_2}==1 and {PERI_GEN}==1",top_apb,apb3_list2],
        "TP_apb3_3"     : ["{PERI_APB_3}==1 and {PERI_GEN}==1",top_apb,apb3_list3],
        "TP_apb3_4"     : ["{PERI_APB_4}==1 and {PERI_GEN}==1",top_apb,apb3_list4]
})

wire_dict = pmap({
    "WR_intr_spi_0"     : ["{PERI_SPI_0}==1 and {PERI_GEN}==1",wire_spi_intr,["0"]],
    "WR_intr_spi_1"     : ["{PERI_SPI_1}==1 and {PERI_GEN}==1",wire_spi_intr,["1"]],
    "WR_intr_spi_2"     : ["{PERI_SPI_2}==1 and {PERI_GEN}==1",wire_spi_intr,["2"]],
    "WR_intr_uart_0"    : ["{PERI_UART_0}==1 and {PERI_GEN}==1",wire_uart_intr,["0"]],
    "WR_intr_uart_1"    : ["{PERI_UART_1}==1 and {PERI_GEN}==1",wire_uart_intr,["1"]],
    "WR_intr_uart_2"    : ["{PERI_UART_2}==1 and {PERI_GEN}==1",wire_uart_intr,["2"]],
    "WR_intr_i2c_0"     : ["{PERI_I2C_0}==1 and {PERI_GEN}==1",wire_i2c_intr,["0"]],
    "WR_intr_i2c_1"     : ["{PERI_I2C_1}==1 and {PERI_GEN}==1",wire_i2c_intr,["1"]],
    "WR_intr_i2c_2"     : ["{PERI_I2C_2}==1 and {PERI_GEN}==1",wire_i2c_intr,["2"]],
    "WR_intr_gpio_0"    : ["{PERI_GPIO_0}==1 and {PERI_GEN}==1",wire_gpio_intr,["0","0"]],
    "WR_intr_gpio_1"    : ["{PERI_GPIO_1}==1 and {PERI_GEN}==1",wire_gpio_intr,["1","0"]],
    "WR_intr_wdt_0"     : ["{PERI_WDT_0}==1 and {PERI_GEN}==1",wire_wdt_intr,[]]

}) 

conn_jtag_dict = pmap({
        "CN_softap"     : ["{INTF_JTAG_TYPE}==1",conn_sjtag,[]],
        "CN_hardtap"    : ["{INTF_JTAG_TYPE}==0",conn_hjtag,[]]
})
conn_dict = pmap({
        "CN_axis"       : ["{PERI_GEN}==0 or {PERI_TCOUNT}==0" ,conn_axis,[]],
        "CN_i2c_0"      : ["{PERI_GEN}==1 and {PERI_I2C_0}==1",conn_i2c,[x for x in sl0 for _ in range(4)]],
        "CN_i2c_1"      : ["{PERI_GEN}==1 and {PERI_I2C_1}==1",conn_i2c,[x for x in sl1 for _ in range(4)]],
        "CN_i2c_2"      : ["{PERI_GEN}==1 and {PERI_I2C_2}==1",conn_i2c,[x for x in sl2 for _ in range(4)]]
})

port_dict = pmap({
        "PRT_spi_0"     : ["{PERI_SPI_0}==1 and {PERI_GEN}==1",port_spi,[x for x in sl0 for _ in range(30)]],
        "PRT_spi_1"     : ["{PERI_SPI_1}==1 and {PERI_GEN}==1",port_spi,[x for x in sl1 for _ in range(30)]],
        "PRT_spi_2"     : ["{PERI_SPI_2}==1 and {PERI_GEN}==1",port_spi,[x for x in sl2 for _ in range(30)]],
        "PRT_uart_0"    : ["{PERI_UART_0}==1 and {PERI_GEN}==1",port_uart,[x for x in sl0 for _ in range(6)]],
        "PRT_uart_1"    : ["{PERI_UART_1}==1 and {PERI_GEN}==1",port_uart,[x for x in sl1 for _ in range(6)]],
        "PRT_uart_2"    : ["{PERI_UART_2}==1 and {PERI_GEN}==1",port_uart,[x for x in sl2 for _ in range(6)]],
        "PRT_i2c_0"     : ["{PERI_I2C_0}==1 and {PERI_GEN}==1",port_i2c,[x for x in sl0 for _ in range(10)]],
        "PRT_i2c_1"     : ["{PERI_I2C_1}==1 and {PERI_GEN}==1",port_i2c,[x for x in sl1 for _ in range(10)]],
        "PRT_i2c_2"     : ["{PERI_I2C_2}==1 and {PERI_GEN}==1",port_i2c,[x for x in sl2 for _ in range(10)]],
        "PRT_gpio_0"    : ["{PERI_GPIO_0}==1 and {PERI_GEN}==1",port_gpio,[x for x in sl0 for _ in range(10)]],
        "PRT_gpio_1"    : ["{PERI_GPIO_1}==1 and {PERI_GEN}==1",port_gpio,[x for x in sl1 for _ in range(10)]],
        "PRT_apb_0"     : ["{PERI_APB_0}==1 and {PERI_GEN}==1",port_apb,[x for x in sl0 for _ in range(16)]],
        "PRT_apb_1"     : ["{PERI_APB_1}==1 and {PERI_GEN}==1",port_apb,[x for x in sl1 for _ in range(16)]],
        "PRT_apb_2"     : ["{PERI_APB_2}==1 and {PERI_GEN}==1",port_apb,[x for x in sl2 for _ in range(16)]],
        "PRT_apb_3"     : ["{PERI_APB_3}==1 and {PERI_GEN}==1",port_apb,[x for x in sl3 for _ in range(16)]],
        "PRT_apb_4"     : ["{PERI_APB_4}==1 and {PERI_GEN}==1",port_apb,[x for x in sl4 for _ in range(16)]],
        "PRT_wdt_0"     : ["{PERI_WDT_0}==1 and {PERI_GEN}==1",port_wdt,[]]

})

def _format_token(config: [], dic: PMap) -> Sequence:
        enable_condition = config[0]
        result = config[1]
        params = config[2]
        replaced_params = list(params)

        for field, value in dic.items():
                key = "{" + field + "}"

                for i in range(0, len(params)):
                        param = params[i]
                        if key in param:
                                temp = param.replace(key, str(value).strip('"'))
                                try:
                                        temp = eval(temp)
                                        replaced_params[i] = temp
                                except ValueError as e:
                                        replaced_params[i] = temp


                if key in enable_condition:
                        enable_condition = enable_condition.replace(key, str(value).strip('"'))

        if(eval(enable_condition)):
                return result.format(*replaced_params)
        else:
                return None

def main():
        parser = argparse.ArgumentParser()
        parser.add_argument('--out_path', help='Output Folder Path', required=True)
        parser.add_argument('--gen_name', help='Generated name of the IP', required=True)
        parser.add_argument('--parameters',
                                metavar="KEY=VALUE",
                                nargs="+",
                                help='Key-Value Pair for resolved parameters from the IPXACT'
                                "(do not put spaces before or after the = sign). "
                                "If a value contains spaces, you should define "
                                "it with double quotes: "
                                'foo="this is a sentence". Note that '
                                "values are always treated as strings."
                                , required=True)
        args = parser.parse_args()
        params = parse_parameters(args.parameters)

        #project ip location
        loc= Path(args.out_path, args.gen_name)
        #ipm python script location
        loc_script = Path(__file__).parent.absolute()

        """ 
        Step 0: Collecting params info
        """
        GenPeripherals=str(params['PERI_GEN']).strip('"')
        GenSpi0=str(params['PERI_SPI_0']).strip('"')
        GenSpi1=str(params['PERI_SPI_1']).strip('"')
        GenSpi2=str(params['PERI_SPI_2']).strip('"')
        GenI2c0=str(params['PERI_I2C_0']).strip('"')
        GenI2c1=str(params['PERI_I2C_1']).strip('"')
        GenI2c2=str(params['PERI_I2C_2']).strip('"')
        GenGpio0=str(params['PERI_GPIO_0']).strip('"')
        GenGpio1=str(params['PERI_GPIO_1']).strip('"')
        GenUart0=str(params['PERI_UART_0']).strip('"')
        GenUart1=str(params['PERI_UART_1']).strip('"')
        GenUart2=str(params['PERI_UART_2']).strip('"')
        GenWdt0=str(params['PERI_WDT_0']).strip('"')
        countIntr=int(str(params['INTF_UINTR']).strip('"')) 

        top_ports=''
        port=''
        wire=''
        intr=[]
        conn=''
        conn_intr=''
        conn_jtag=''

        for k, v in tp_dict.items():
                c = _format_token(v, params)
                if(c != None):
                        top_ports+=c

        for k, v in wire_dict.items():
                c = _format_token(v, params)
                if(c != None):
                        wire+=c

        for k, v in conn_dict.items():
                c = _format_token(v, params)
                if(c != None):
                        conn+=c

        for k, v in conn_jtag_dict.items():
                c = _format_token(v, params)
                if(c != None):
                        conn_jtag+=c

        for k, v in port_dict.items():
                c = _format_token(v, params)
                if(c != None):
                        port+=c

        if (GenPeripherals=='1'):
            if (GenUart0 == '1'):
                intr.append('system_uart_0_io_interrupt;\n')
            if (GenUart1 == '1'):
                intr.append('system_uart_1_io_interrupt;\n')
            if (GenUart2 == '1'):
                intr.append('system_uart_2_io_interrupt;\n')
            if (GenSpi0 == '1'):
                intr.append('system_spi_0_io_interrupt;\n')
            if (GenSpi1 == '1'):
                intr.append('system_spi_1_io_interrupt;\n')
            if (GenSpi2 == '1'):
                intr.append('system_spi_2_io_interrupt;\n')
            if (GenI2c0 == '1'):
                intr.append('system_i2c_0_io_interrupt;\n')
            if (GenI2c1 == '1'):
                intr.append('system_i2c_1_io_interrupt;\n')
            if (GenI2c2 == '1'):
                intr.append('system_i2c_2_io_interrupt;\n')
            if (GenGpio0 == '1'):
                intr.append('system_gpio_0_io_interrupt_0;\n')
                intr.append('system_gpio_0_io_interrupt_1;\n')
            if (GenGpio1 == '1'):
                intr.append('system_gpio_1_io_interrupt_0;\n')
                intr.append('system_gpio_1_io_interrupt_1;\n')
            if (GenWdt0 == '1'):
                intr.append('system_watchdog_softPanic_interrupt;\n')


            for p in range (len(intr)):
                c = 'assign userInterrupt'+intr_decode(p)+' = '+intr[p]
                conn_intr+=c
            
        """
        Step 1: open generated file and modify
        """
        tfname=args.gen_name+'.v'

        #read from target file
        tf=open(Path(loc, tfname),'r',encoding="utf-8")
        tfl=tf.read()
        tf.close

        #string slicing last character to remove newline char
        top_ports=top_ports[:-1]
        port=port[:-1]
        #replace the string
        rpDic={
        "//RPLPERI_TP@":top_ports,
        "//RPLPERI_WIRE@":wire,
        "//RPLPERI_INTRCN@":conn_intr,
        "//RPLPERI_JTCN@":conn_jtag,
        "//RPLPERI_CN@":conn,
        "//RPLPERI_PRT@":port,
        }
        rtfl=multiwordReplace(tfl, rpDic)

        #os.remove(Path(loc,tfname)
        #write to target file
        dtf=open(Path(loc,tfname),'w',encoding="utf-8")
        dtf.write(rtfl)
        dtf.close()


if __name__ == '__main__':
        main()

