import argparse
import os.path
import shutil
import fileinput
import re
import subprocess
import sys
import time
import os
import codecs
from pathlib import Path
from typing import Dict, List
from pyrsistent import pmap, freeze
from pyrsistent.typing import PMap, PVector
from typing import Mapping, Sequence
import platform

def multiwordReplace(text, wordDic):
        """
        take a text and replace words that match a key in a dictionary with
        the associated value, return the changed text
        """
        rc = re.compile('|'.join(map(re.escape, wordDic)))
        def translate(match):
                return wordDic[match.group(0)]
        return rc.sub(translate, text)

def parse_var(s):
        """
        Parse a key, value pair, separated by '='
        That's the reverse of ShellArgs.

        On the command line (argparse) a declaration will typically look like:
                foo=hello
        or
                foo="hello world"
        """
        items = s.split('=')
        key = items[0].strip() # we remove blanks around keys, as is logical
        if len(items) > 1:
                        # rejoin the rest:
                        value = '='.join(items[1:])
        return (key, value)

def parse_parameters(params: List[str]) -> Dict[str, str]:
        """
        Parse the key-value pairs and return A dictionary
        """
        d = {}
        for item in params:
                        key, value = parse_var(item)
                        d[key] = value
        return d


config_dict = pmap({
        "frequency_0": [
                "{PERI_GEN}==1",
                "--peripheralFrequency {}",
                ["{PERI_FREQ}*1000000"]
        ],
        "frequency_1": [
                "{PERI_GEN}==1",
                "--PeripheralClock true",
                [""]
        ],
        "apbSlave_0": [
                "{PERI_APB_0}==1 and {PERI_GEN}==1",
                "--apbSlave name=io_apbSlave_0,address=0x100000,size={}",
                ["{PERI_APB_0_SIZE}"]
        ],
        "apbSlave_1": [
                "{PERI_APB_1}==1 and {PERI_GEN}==1",
                "--apbSlave name=io_apbSlave_1,address=0x200000,size={}",
                ["{PERI_APB_1_SIZE}"]
        ],
        "apbSlave_2": [
                "{PERI_APB_2}==1 and {PERI_GEN}==1",
                "--apbSlave name=io_apbSlave_2,address=0x300000,size={}",
                ["{PERI_APB_2_SIZE}"]
        ],
        "apbSlave_3": [
                "{PERI_APB_3}==1 and {PERI_GEN}==1",
                "--apbSlave name=io_apbSlave_3,address=0x400000,size={}",
                ["{PERI_APB_3_SIZE}"]
        ],
        "apbSlave_4": [
                "{PERI_APB_4}==1 and {PERI_GEN}==1",
                "--apbSlave name=io_apbSlave_4,address=0x500000,size={}",
                ["{PERI_APB_4_SIZE}"]
        ],
        "gpio_0": [
                "{PERI_GPIO_0}==1 and {PERI_GEN}==1",
                "--gpio name=system_gpio_0_io,address=0x40000,width={},interrupts=\"0->9;1->10\"",
                ["{PERI_GPIO_0_WIDTH}"],
        ],
        "gpio_1": [
                "{PERI_GPIO_1}==1 and {PERI_GEN}==1",
                "--gpio name=system_gpio_1_io,address=0x41000,width={},interrupts=\"0->11;1->12\"",
                ["{PERI_GPIO_1_WIDTH}"]
        ],
        "uart_0": [
                "{PERI_UART_0}==1 and {PERI_GEN}==1",
                "--uart name=system_uart_0_io,address=0x10000,interruptId=0",
                [""]
        ],
        "uart_1": [
                "{PERI_UART_1}==1 and {PERI_GEN}==1",
                "--uart name=system_uart_1_io,address=0x11000,interruptId=1",
                [""]
        ],
        "uart_2": [
                "{PERI_UART_2}==1 and {PERI_GEN}==1",
                "--uart name=system_uart_2_io,address=0x12000,interruptId=2",
                [""]
        ],
        "spi_0": [
                "{PERI_SPI_0}==1 and {PERI_GEN}==1",
                "--spi name=system_spi_0_io,address=0x30000,interruptId=3,width=4,ssCount=4",
                [""]
        ],
        "spi_1": [
                "{PERI_SPI_1}==1 and {PERI_GEN}==1",
                "--spi name=system_spi_1_io,address=0x31000,interruptId=4,width=4,ssCount=4",
                [""]
        ],
        "spi_2": [
                "{PERI_SPI_2}==1 and {PERI_GEN}==1",
                "--spi name=system_spi_2_io,address=0x32000,interruptId=5,width=4,ssCount=4",
                [""]
        ],
        "i2c_0": [
                "{PERI_I2C_0}==1 and {PERI_GEN}==1",
                "--i2c name=system_i2c_0_io,address=0x20000,interruptId=6",
                [""]
        ],
        "i2c_1": [
                "{PERI_I2C_1}==1 and {PERI_GEN}==1",
                "--i2c name=system_i2c_1_io,address=0x21000,interruptId=7",
                [""]
        ],
        "i2c_2": [
                "{PERI_I2C_2}==1 and {PERI_GEN}==1",
                "--i2c name=system_i2c_2_io,address=0x22000,interruptId=8",
                [""]
        ],
        "wdt_0": [
                "{PERI_WDT_0}==1 and {PERI_GEN}==1",
                "--watchdog name=system_watchdog,address=0x50000,interruptId=13,prescalerWidth=24,counters=2,countersWidth=16",
                [""]
        ]

})

def _format_token(config: [], dic: PMap) -> Sequence:
        enable_condition = config[0]
        result = config[1]
        params = config[2]
        replaced_params = list(params)

        for field, value in dic.items():
                key = "{" + field + "}"

                for i in range(0, len(params)):
                        param = params[i]
                        if key in param:
                                temp = param.replace(key, str(value).strip('"'))
                                try:
                                        temp = eval(temp)
                                        replaced_params[i] = temp
                                except ValueError as e:
                                        replaced_params[i] = temp


                if key in enable_condition:
                        enable_condition = enable_condition.replace(key, str(value).strip('"'))

        if(eval(enable_condition)):
                return result.format(*replaced_params)
        else:
                return None

def generation_err():
    msg_printbar='*'*94+'\n'
    msg_err=msg_printbar+'[Error] Please contact Efinix team for further support on error messages printed below.\n\n'+msg_printbar
    print(msg_err)

def main():
        parser = argparse.ArgumentParser()
        parser.add_argument('--out_path', help='Output Folder Path', required=True)
        parser.add_argument('--user_project_path', help='User Project Path', required=True)
        parser.add_argument('--gen_name', help='Generated name of the IP', required=True)
        parser.add_argument('--ip_path', help='Location of IP libraries', required=True)
        parser.add_argument('--parameters',
                                metavar="KEY=VALUE",
                                nargs="+",
                                help='Key-Value Pair for resolved parameters from the IPXACT'
                                "(do not put spaces before or after the = sign). "
                                "If a value contains spaces, you should define "
                                "it with double quotes: "
                                'foo="this is a sentence". Note that '
                                "values are always treated as strings."
                                , required=True)
        args = parser.parse_args()
        params = parse_parameters(args.parameters)

        #project ip location
        loc = Path(args.out_path, args.gen_name)
        #project location
        loc_user = Path(args.user_project_path, args.gen_name)
        #ipm python script location
        loc_script = Path(__file__).parent.absolute()
        loc_ip=Path(args.ip_path, 'efx_soc', 'efx_soc', 'generator')

        project_path = loc_user.parents[0]

        #Standardize location script folder for server style use
        loc_outflow_tmp = Path(args.out_path, f'{args.gen_name}_outflow') # Use gen_name to make outflow path unique
        if loc_outflow_tmp.exists():
            shutil.rmtree(loc_outflow_tmp)

        if not loc_outflow_tmp.exists():
            loc_outflow_tmp.mkdir(parents=True, exist_ok=True)

        #make destination folder
        loc_source=Path(loc, 'source')
        if os.path.exists(Path(loc_source)):
            shutil.rmtree(Path(loc_source))

        if not os.path.exists(loc_source):
            os.makedirs(loc_source)

        targetConfig = Path(loc_script, 'peri_config')
        targetCopyConfig = Path(loc_source, 'peri_config')

        """
        Step 0: Collecting params info
        """
        GenPeripherals=str(params['PERI_GEN']).strip('"')
        TotalPeripherals=str(params['PERI_TCOUNT']).strip('"')

        if(GenPeripherals=='0' or TotalPeripherals=='0'):
            return

        """
        Step 1: Generate configuration file for peripheral devices
        """
        config = []
        for k, v in config_dict.items():
            c = _format_token(v, params)
            if(c != None):
                config.append(c)

        with open(targetCopyConfig, 'w', encoding='utf8') as f:
            for c in config:
                f.write(c + '\n')
        """
        Step 2: Parse configuration file to jar exe and move to destination
        """
        #generate peripheral
        currentPath = loc_outflow_tmp
        targetGen = Path(loc_ip, 'EfxSapphireSoc.jar')
        f=open(targetCopyConfig,"r")
        config=f.read()
        cmd_parse="".join(config.splitlines())
        cmd=cmd_parse.replace("--"," --")
        cmd_os = 'java -cp '+ str(targetGen) +' saxon.board.efinix.Axi4PeripheralEmulationGen '+ cmd

        '''
        NOTE: Example command:
        java -cp Axi4Peripheral.jar saxon.board.efinix.Axi4PeripheralEmulationGen
        --peripheralFrequency 50000000
        --gpio name=gpioA,address=0x000000,width=16,interrupts=0->0/1->0/2->0/3->0
        --uart name=uartA,address=0x10000,interruptId=0
        --spi name=spiA,address=0x14000,interruptId=0
        --apbSlave name=apbA,address=0x100000,size=0x100000
        '''
        try:
            subprocess.run(cmd_os, check = True ,shell=True, cwd=currentPath.absolute())
        except subprocess.CalledProcessError:
            return generation_err()

        #move file to destination
        shutil.move(Path(currentPath,'hardware','netlist','Axi4PeripheralTop.v'), Path(loc_source, 'Axi4PeripheralTop.v'))

        # Remove created outflow_tmp folder
        shutil.rmtree(loc_outflow_tmp)

if __name__ == '__main__':
        main()



