######################################################################
# Copyright (C) 2017 - 2020 Efinix Inc. All rights reserved.
#
# No portion of this code may be reused, modified or
# distributed in any way without the expressed written
# consent of Efinix Inc.
#
# pll_manual_clock.py : Demonstrates how to do manual clock configuration
#
# Revision : 1.0 - Initial release
######################################################################

# Get access to useful python package
import os
import sys
import argparse
import pprint
import configparser

from pathlib import Path
from typing import List, Dict, Union, Tuple

# Tell python where to reference IPM Home
ipm_home = os.environ['EFXIPM_HOME']
sys.path.append(ipm_home + "/bin")

# Tell python where to get Interface Designer's API package
pt_home = os.environ['EFXPT_HOME']
sys.path.append(pt_home + "/bin")

from api_service.design import DesignAPI  # Get access to design database API
from api_service.device import DeviceAPI  # Get access to device database API
import api_service.excp.design_excp as APIExcp  # Get access to API exception
# from pt.bin.api_service.design import DesignAPI  # Get access to design database API
# from pt.bin.api_service.device import DeviceAPI  # Get access to device database API
# import pt.bin.api_service.excp.design_excp as APIExcp  # Get access to API exception

from common.ipm_pt_map import IpmPTMap, IpmPTBlock, GPIOType

#cslau: Generate pinlist according to device
def gen_pinlist(package_pdic, device_name: str, interface: str):
    # Define a list of package to match
    package = ["676", "529", "484", "900" ,"1156"]

    # Check if any package is in the device_name and find a matching device in the dictionary
    matched_device = next((key for key in package_pdic if any(kw in key for kw in package if kw in device_name)), None)
    
    if matched_device:
        device_config = package_pdic[matched_device]
    else:
        print(f"No matching device found for package in '{device_name}'. Using default configuration.")
        device_config = package_pdic.get('Default')
    
    # Get the interface configuration if available
    interface_config = device_config.get(interface) if device_config else None

    return interface_config

def parse_hard_ip_ini_file(file_path: Path) -> Tuple[Dict[str, str], Dict[str, str]]:
    """
    Parse the key-value pairs and return A dictionary
    """
    param_dict = {}
    port_dict = {}
    hard_ip_ini_file = configparser.ConfigParser()
    hard_ip_ini_file.read(file_path)

    if 'parameters' in hard_ip_ini_file:
        all_params = hard_ip_ini_file['parameters']
        for param_name in all_params:
            key = str(param_name).upper()
            value = str(all_params[param_name])
            param_dict[key] = value
    else:
        raise Exception("No Parameters found in config ini file.")

    if 'ports' in hard_ip_ini_file:
        all_ports = hard_ip_ini_file['ports']
        for port_name in all_ports:
            key = str(port_name).upper()
            value = str(all_ports[port_name])
            port_dict[key] = value
    else:
        raise Exception("No Ports found in config ini file.")

    return param_dict, port_dict

def is_gpio_exists(design_api, inst_name: str) -> bool:
    object_id = design_api.get_gpio(inst_name)
    return object_id is not None

def is_block_exists(design_api, inst_name: str, block_type: str) -> bool:
    object_id = design_api.get_block(inst_name, block_type)
    return object_id is not None

def get_property_by_name_and_type(design_api, inst_name: str, block_type: str) -> Union[Dict[str, str], None]:
    object_id = design_api.get_block(inst_name, block_type)
    if object_id is not None:
        property: Dict[str, str] = design_api.get_all_property(block_type, object_id)
    else:
        property = None # write_to_console on error or something
    return property

def intr_decode(n):
    intr_decode_dict = {
        0:"A",1:"B",2:"C",3:"D",4:"E",5:"F",6:"G",7:"H",
        8:"I",9:"J",10:"K",11:"L",12:"M",13:"N",14:"O",15:"P",
        16:"Q",17:"R",18:"S",19:"T",20:"U",21:"V",22:"W",23:"X"
    }
    return intr_decode_dict[n]

def handle_default_pll(design_api, pll_name: str, pll_resource: str, pll_locked_sig: str,
                       pll_ref_freq: str, pll_ext_src: str, clkout_dict: Dict, export_list: List[str],
                       pll_ipm_param_name: str, ipm_pt_map: IpmPTMap):

    # Set property, non-defaults
    target_freq = {}
    clock_prop = ["M", "N", "O", "VCO_FREQ", "PLL_FREQ"]

    if is_block_exists(design_api, pll_name, "PLL"):
        clksrc_info = design_api.trace_ref_clock(pll_name, block_type="PLL")
        design_api.delete_gpio(clksrc_info[0]["NAME"])
        design_api.delete_block(pll_name,"PLL")
    design_api.create_block(pll_name,"PLL")
    design_api.gen_pll_ref_clock(pll_name, pll_res=pll_resource, refclk_src="EXTERNAL", refclk_name=f'{pll_name}_ext_clk', ext_refclk_no=pll_ext_src)
    design_api.set_property(pll_name,"LOCKED_PIN", pll_locked_sig, "PLL")
    design_api.set_property(pll_name,"REFCLK_FREQ", pll_ref_freq, "PLL")
    export_list.append(pll_name)
    export_list.append(f'{pll_name}_ext_clk')
    ipm_pt_map.set_block_item(IpmPTBlock.PLL, pll_ipm_param_name, pll_ipm_param_name, pll_name)
    ipm_pt_map.set_gpio_block_item(GPIOType.GPIO, f'{pll_name}_ext_clk')

    #kmlau:remove readback the current setting
    #else:
    #all_current_clkout = [
    #    "CLKOUT0_FREQ",
    #    "CLKOUT0_PHASE",
    #    "CLKOUT1_FREQ",
    #    "CLKOUT1_PHASE",
    #    "CLKOUT2_PHASE",
    #    "CLKOUT2_FREQ",
    #    "CLKOUT3_PHASE",
    #    "CLKOUT3_FREQ",
    #    "CLKOUT4_PHASE",
    #    "CLKOUT4_FREQ",
    #]
    #current_clkout_map = design_api.get_property(pll_name, all_current_clkout, block_type="PLL")
    #for key in current_clkout_map:
    #    clkout_val = current_clkout_map[key]
    #    if clkout_val is not None:
    #        target_freq[key] = clkout_val

    for clkout_int in clkout_dict:
        inner_clkout_dict = clkout_dict[clkout_int]

        current_clkout = f"CLKOUT{clkout_int}" # CLKOUT0
        clkout_name = f"{current_clkout}_EN" # CLKOUT0_EN
        pll_clkout_name = inner_clkout_dict["CLKOUT_NAME"] # pll_name_CLKOUT0
        pll_clkout_pin = f'{current_clkout}_PIN' # CLKOUT0_PIN
        clkout_freq_key = f'{current_clkout}_FREQ' # CLKOUT0_FREQ
        clkout_phase_key = f'{current_clkout}_PHASE' # CLKOUT0_PHASE

        design_api.set_property(pll_name,clkout_name,"1","PLL")

        design_api.set_property(pll_name,pll_clkout_pin, pll_clkout_name, "PLL")

        target_freq[clkout_freq_key] = inner_clkout_dict['CLKOUT_FREQ']
        target_freq[clkout_phase_key] = inner_clkout_dict['CLKOUT_PHASE']

    design_api.set_property(pll_name,"FEEDBACK_CLK", "CLK0", "PLL") # Explicitly set feedback clk to zeroeth clk

    # Auto calculate counters and output dividers with this desired frequencies
    list_of_property = design_api.auto_calc_pll_clock(pll_name, target_freq=target_freq)

    # Trace all auto-generated clock source
    # This will return a GPIO
    clksrc_info = design_api.trace_ref_clock(pll_name, block_type="PLL")
    pprint.pprint(clksrc_info)

    prop_map = design_api.get_property(pll_name, clock_prop, block_type="PLL")
    pprint.pprint(prop_map)

    # Inspect clock source-related properties
    clock_source_prop = ["REFCLK_SOURCE", "EXT_CLK"]

    prop_map = design_api.get_property(pll_name, clock_source_prop, block_type="PLL")
    pprint.pprint(prop_map)

def create_def_uart_gpio_block(design_api, inst_name, uart_pinlist, pin, export_list, ipm_pt_map):
    input_gpio = f"{inst_name}_io_rxd"
    output_gpio = f"{inst_name}_io_txd"

    if is_gpio_exists(design_api, input_gpio):
        design_api.delete_gpio(input_gpio)
    design_api.create_input_gpio(input_gpio)
    export_list.append(input_gpio)
    ipm_pt_map.set_gpio_block_item(GPIOType.UART, input_gpio)
    if(pin=='1'):
        design_api.assign_pkg_pin(input_gpio,uart_pinlist[0])

    if is_gpio_exists(design_api, output_gpio):
        design_api.delete_gpio(output_gpio)
    design_api.create_output_gpio(output_gpio)
    export_list.append(output_gpio)
    ipm_pt_map.set_gpio_block_item(GPIOType.UART, output_gpio)
    if(pin=='1'):
        design_api.assign_pkg_pin(output_gpio,uart_pinlist[1])

def delete_uart_gpio_block(design_api, inst_name):
    input_gpio = f"{inst_name}_io_rxd"
    output_gpio = f"{inst_name}_io_txd"

    if is_gpio_exists(design_api, input_gpio):
        design_api.delete_gpio(input_gpio)

    if is_gpio_exists(design_api, output_gpio):
        design_api.delete_gpio(output_gpio)

def create_def_spi_gpio_block(design_api, inst_name, spi_pinlist, pin, pll_assign, export_list, ipm_pt_map):
    inout_io_data_0 = f"{inst_name}_io_data_0"
    inout_io_data_1 = f"{inst_name}_io_data_1"
    output_gpio_sclk_write = f"{inst_name}_io_sclk_write"
    output_gpio_ss = f"{inst_name}_io_ss"

    if is_gpio_exists(design_api, inout_io_data_0):
        design_api.delete_gpio(inout_io_data_0)
    design_api.create_inout_gpio(inout_io_data_0)
    export_list.append(inout_io_data_0)
    ipm_pt_map.set_gpio_block_item(GPIOType.SPI, inout_io_data_0)

    if(pll_assign == '1'):
        design_api.set_property(inout_io_data_0, "IN_PIN", inout_io_data_0+"_read")
        design_api.set_property(inout_io_data_0, "IN_REG", "REG")
        design_api.set_property(inout_io_data_0, "IN_CLK_PIN", "io_peripheralClk")
        design_api.set_property(inout_io_data_0, "OUT_PIN", inout_io_data_0+"_write")
        design_api.set_property(inout_io_data_0, "OUT_REG", "REG")
        design_api.set_property(inout_io_data_0, "OUT_CLK_PIN", "io_peripheralClk")
        design_api.set_property(inout_io_data_0, "OE_PIN", inout_io_data_0+"_writeEnable")
        design_api.set_property(inout_io_data_0, "OE_REG", "REG")
        design_api.set_property(inout_io_data_0, "OE_CLK_PIN", "io_peripheralClk")
    else:
        design_api.set_property(inout_io_data_0, "IN_PIN", inout_io_data_0+"_read")
        design_api.set_property(inout_io_data_0, "OUT_PIN", inout_io_data_0+"_write")
        design_api.set_property(inout_io_data_0, "OE_PIN", inout_io_data_0+"_writeEnable")
    if(pin=='1'):
        design_api.assign_pkg_pin(inout_io_data_0,spi_pinlist[0])

    if is_gpio_exists(design_api, inout_io_data_1):
        design_api.delete_gpio(inout_io_data_1)
    design_api.create_inout_gpio(inout_io_data_1)
    export_list.append(inout_io_data_1)
    ipm_pt_map.set_gpio_block_item(GPIOType.SPI, inout_io_data_1)

    if(pll_assign == '1'):
        design_api.set_property(inout_io_data_1, "IN_PIN", inout_io_data_1+"_read")
        design_api.set_property(inout_io_data_1, "IN_REG", "REG")
        design_api.set_property(inout_io_data_1, "IN_CLK_PIN", "io_peripheralClk")
        design_api.set_property(inout_io_data_1, "OUT_PIN", inout_io_data_1+"_write")
        design_api.set_property(inout_io_data_1, "OUT_REG", "REG")
        design_api.set_property(inout_io_data_1, "OUT_CLK_PIN", "io_peripheralClk")
        design_api.set_property(inout_io_data_1, "OE_PIN", inout_io_data_1+"_writeEnable")
        design_api.set_property(inout_io_data_1, "OE_REG", "REG")
        design_api.set_property(inout_io_data_1, "OE_CLK_PIN", "io_peripheralClk")
    else:
        design_api.set_property(inout_io_data_1, "IN_PIN", inout_io_data_1+"_read")
        design_api.set_property(inout_io_data_1, "OUT_PIN", inout_io_data_1+"_write")
        design_api.set_property(inout_io_data_1, "OE_PIN", inout_io_data_1+"_writeEnable")
    if(pin=='1'):
        design_api.assign_pkg_pin(inout_io_data_1,spi_pinlist[1])

    if is_gpio_exists(design_api, output_gpio_sclk_write):
        design_api.delete_gpio(output_gpio_sclk_write)
    design_api.create_output_gpio(output_gpio_sclk_write)
    export_list.append(output_gpio_sclk_write)
    ipm_pt_map.set_gpio_block_item(GPIOType.SPI, output_gpio_sclk_write)

    if(pll_assign == '1'):
        design_api.set_property(output_gpio_sclk_write, "OUT_PIN", output_gpio_sclk_write)
        design_api.set_property(output_gpio_sclk_write, "OUT_REG", "REG")
        design_api.set_property(output_gpio_sclk_write, "OUT_CLK_PIN", "io_peripheralClk")
    else:
        design_api.set_property(output_gpio_sclk_write, "OUT_PIN", output_gpio_sclk_write)
    if(pin=='1'):
        design_api.assign_pkg_pin(output_gpio_sclk_write,spi_pinlist[2])

    if is_gpio_exists(design_api, output_gpio_ss):
        design_api.delete_gpio(output_gpio_ss)
    design_api.create_output_gpio(output_gpio_ss)
    export_list.append(output_gpio_ss)
    ipm_pt_map.set_gpio_block_item(GPIOType.SPI, output_gpio_ss)

    if(pll_assign == '1'):
        design_api.set_property(output_gpio_ss, "OUT_PIN", f"{output_gpio_ss}[0]")
        design_api.set_property(output_gpio_ss, "OUT_REG", "REG")
        design_api.set_property(output_gpio_ss, "OUT_CLK_PIN", "io_peripheralClk")
    else:
        design_api.set_property(output_gpio_ss, "OUT_PIN", f"{output_gpio_ss}[0]")
    if(pin=='1'):
        design_api.assign_pkg_pin(output_gpio_ss,spi_pinlist[3])

def delete_spi_gpio_block(design_api, inst_name):
    inout_io_data_0 = f"{inst_name}_io_data_0"
    inout_io_data_1 = f"{inst_name}_io_data_1"
    output_gpio_sclk_write = f"{inst_name}_io_sclk_write"
    output_gpio_ss = f"{inst_name}_io_ss"

    if is_gpio_exists(design_api, inout_io_data_0):
        design_api.delete_gpio(inout_io_data_0)

    if is_gpio_exists(design_api, inout_io_data_1):
        design_api.delete_gpio(inout_io_data_1)

    if is_gpio_exists(design_api, output_gpio_sclk_write):
        design_api.delete_gpio(output_gpio_sclk_write)

    if is_gpio_exists(design_api, output_gpio_ss):
        design_api.delete_gpio(output_gpio_ss)

def create_def_i2c_gpio_block(design_api, inst_name, i2c_pinlist, pin, export_list, ipm_pt_map):
    inout_scl = f"{inst_name}_io_scl"
    inout_sda = f"{inst_name}_io_sda"
    if is_gpio_exists(design_api, inout_scl):
        design_api.delete_gpio(inout_scl)
    design_api.create_inout_gpio(inout_scl)
    export_list.append(inout_scl)
    ipm_pt_map.set_gpio_block_item(GPIOType.I2C, inout_scl)

    design_api.set_property(inout_scl, "IN_PIN", inout_scl+"_read")
    design_api.set_property(inout_scl, "OUT_PIN", inout_scl+"_write")
    design_api.set_property(inout_scl, "OE_PIN", inout_scl+"_writeEnable")
    if(pin=='1'):
        design_api.assign_pkg_pin(inout_scl,i2c_pinlist[0])

    if is_gpio_exists(design_api, inout_sda):
        design_api.delete_gpio(inout_sda)
    design_api.create_inout_gpio(inout_sda)
    export_list.append(inout_sda)
    ipm_pt_map.set_gpio_block_item(GPIOType.I2C, inout_sda)

    design_api.set_property(inout_sda, "IN_PIN", inout_sda+"_read")
    design_api.set_property(inout_sda, "OUT_PIN", inout_sda+"_write")
    design_api.set_property(inout_sda, "OE_PIN", inout_sda+"_writeEnable")
    if(pin=='1'):
        design_api.assign_pkg_pin(inout_sda,i2c_pinlist[1])

def delete_i2c_gpio_block(design_api, inst_name):
    inout_scl = f"{inst_name}_io_scl"
    inout_sda = f"{inst_name}_io_sda"

    if is_gpio_exists(design_api, inout_scl):
        design_api.delete_gpio(inout_scl)

    if is_gpio_exists(design_api, inout_sda):
        design_api.delete_gpio(inout_sda)

def create_def_gpio_bus(design_api, inst_name, msb, lsb, gpio_pinlist, pin, export_list, ipm_pt_map):
    if not design_api.get_bus(inst_name) is None:
       design_api.delete_gpio(inst_name)
    design_api.create_inout_gpio(inst_name, msb, lsb)
    export_list.append(inst_name)
    ipm_pt_map.set_gpio_block_item(GPIOType.GPIO, inst_name)

    design_api.set_property(inst_name ,"IN_PIN" , f"{inst_name}_read")
    design_api.set_property(inst_name ,"OUT_PIN", f"{inst_name}_write")
    design_api.set_property(inst_name ,"OE_PIN" , f"{inst_name}_writeEnable")

    lower_bound = lsb
    if lsb == 0:
        lower_bound += 1

    upper_bound = msb
    upper_bound += 1

    # Start at least inst 1.
    for i in range(lower_bound, upper_bound):
        design_api.set_property(f'{inst_name}[{i}]',"DLY_ENA_PIN",f"{inst_name}_DLY_ENA[{i}]")
        design_api.set_property(f'{inst_name}[{i}]',"DLY_RST_PIN",f"{inst_name}_DLY_RST[{i}]")
        design_api.set_property(f'{inst_name}[{i}]',"DLY_INC_PIN",f"{inst_name}_DLY_CTRL[{i}]")
        design_api.set_property(f'{inst_name}[{i}]',"PULL_UP_ENA_PIN",f"{inst_name}_PULL_UP_ENA[{i}]")
    if(pin=='1'):
        for i in range(lsb,upper_bound):
            design_api.assign_pkg_pin(f'{inst_name}[{i}]', gpio_pinlist[i])

def delete_gpio_bus(design_api, inst_name):
    if not design_api.get_bus(inst_name) is None:
       design_api.delete_gpio(inst_name)

def create_def_soft_jtag_gpio_block(design_api, inst_name, soc_pin_name, pinlist, pin, export_list, ipm_pt_map):

    input_jtag_tms  = "pin_io_jtag_tms"
    input_jtag_tdi  = "pin_io_jtag_tdi"
    output_jtag_tdo = "pin_io_jtag_tdo"
    input_jtag_tck  = "io_jtag_tck"

    # Extract the device_specific config
    jtag_pinlist = pinlist["jtag_pinlist"]
    jtag_io_bank = pinlist["jtag_io_bank"]
    jtag_io_standard = pinlist["jtag_io_standard"]
    jtag_io_bank_voltage = pinlist["jtag_io_bank_voltage"]

    if is_gpio_exists(design_api, input_jtag_tms):
        design_api.delete_gpio(input_jtag_tms)
    design_api.create_input_gpio(input_jtag_tms)
    export_list.append(input_jtag_tms)
    ipm_pt_map.set_gpio_block_item(GPIOType.JTAG, input_jtag_tms)
    design_api.set_property(input_jtag_tms, "IO_STANDARD", jtag_io_standard)

    if is_gpio_exists(design_api, input_jtag_tdi):
        design_api.delete_gpio(input_jtag_tdi)
    design_api.create_input_gpio(input_jtag_tdi)
    export_list.append(input_jtag_tdi)
    ipm_pt_map.set_gpio_block_item(GPIOType.JTAG, input_jtag_tdi)
    design_api.set_property(input_jtag_tdi, "IO_STANDARD", jtag_io_standard)

    if is_gpio_exists(design_api, output_jtag_tdo):
        design_api.delete_gpio(output_jtag_tdo)
    design_api.create_output_gpio(output_jtag_tdo)
    export_list.append(output_jtag_tdo)
    ipm_pt_map.set_gpio_block_item(GPIOType.JTAG, output_jtag_tdo)
    design_api.set_property(output_jtag_tdo, "IO_STANDARD", jtag_io_standard)

    if is_gpio_exists(design_api, input_jtag_tck):
        design_api.delete_gpio(input_jtag_tck)
    design_api.create_input_gpio(input_jtag_tck)
    export_list.append(input_jtag_tck)
    ipm_pt_map.set_gpio_block_item(GPIOType.JTAG, input_jtag_tck)
    design_api.set_property(input_jtag_tck, "IO_STANDARD", jtag_io_standard)

    if(pin=='1'):
        design_api.assign_pkg_pin(input_jtag_tms, jtag_pinlist[0])
        design_api.assign_pkg_pin(input_jtag_tdi, jtag_pinlist[1])
        design_api.assign_pkg_pin(output_jtag_tdo, jtag_pinlist[2])
        design_api.assign_pkg_pin(input_jtag_tck, jtag_pinlist[3])

    #make sure the IO_BANK for the Soft JTAG is set to 3.3V
    if (design_api.get_iobank_voltage(jtag_io_bank) != jtag_io_bank_voltage):
        design_api.set_iobank_voltage(jtag_io_bank, jtag_io_bank_voltage)

def delete_soft_jtag_gpio_block(design_api, inst_name, soc_pin_name):
    input_jtag_tms  = "pin_io_jtag_tms"
    input_jtag_tdi  = "pin_io_jtag_tdi"
    output_jtag_tdo = "pin_io_jtag_tdo"
    input_jtag_tck  = "io_jtag_tck"

    if is_gpio_exists(design_api, input_jtag_tms):
        design_api.delete_gpio(input_jtag_tms)

    if is_gpio_exists(design_api, input_jtag_tdi):
        design_api.delete_gpio(input_jtag_tdi)

    if is_gpio_exists(design_api, output_jtag_tdo):
        design_api.delete_gpio(output_jtag_tdo)

    if is_gpio_exists(design_api, input_jtag_tck):
        design_api.delete_gpio(input_jtag_tck)

    design_api.set_property(soc_pin_name,"IO_JTAG_TCK_PIN","","SOC")
    design_api.set_property(soc_pin_name,"IO_JTAG_TDI_PIN","","SOC")
    design_api.set_property(soc_pin_name,"IO_JTAG_TDO_PIN","","SOC")
    design_api.set_property(soc_pin_name,"IO_JTAG_TMS_PIN","","SOC")

def create_def_hard_jtag(design_api, inst_name, resource_to_assign, export_list, ipm_pt_map):
    resource_to_jtag_map = {
        '8': 'JTAG_USER1',
        '9': 'JTAG_USER2',
        '10': 'JTAG_USER3',
        '11': 'JTAG_USER4',
    }
    if is_block_exists(design_api, inst_name, "JTAG"):
        design_api.delete_block(inst_name,"JTAG")
    design_api.create_block(inst_name,"JTAG")
    export_list.append(inst_name)
    ipm_pt_map.set_block_item(IpmPTBlock.JTAG, "HARD_JTAG_INST_NAME", "HARD_JTAG_INST_NAME", inst_name)

    val = resource_to_jtag_map[resource_to_assign]
    design_api.set_property(inst_name,"RESOURCE", val,"JTAG")
    design_api.assign_resource(inst_name, val,"JTAG")
    design_api.set_property(inst_name,"CAPTURE","ut_jtagCtrl_capture","JTAG")
    design_api.set_property(inst_name,"RESET","ut_jtagCtrl_reset","JTAG")
    design_api.set_property(inst_name,"SEL","ut_jtagCtrl_enable","JTAG")
    design_api.set_property(inst_name,"SHIFT","ut_jtagCtrl_shift","JTAG")
    design_api.set_property(inst_name,"TCK","jtagCtrl_tck","JTAG")
    design_api.set_property(inst_name,"TDI","ut_jtagCtrl_tdi","JTAG")
    design_api.set_property(inst_name,"UPDATE","ut_jtagCtrl_update","JTAG")
    design_api.set_property(inst_name,"TDO","ut_jtagCtrl_tdo","JTAG")
    design_api.set_property(inst_name,"TMS","","JTAG")
    design_api.set_property(inst_name,"DRCK","","JTAG")
    design_api.set_property(inst_name,"RUNTEST","","JTAG")

def main(project_name: str, device_name: str, output_dirname: str, params: Dict[str, str], ports: Dict[str, str], gen_name: str, user_project_path: str = '', peri_xml_file_path: str = ''):
    export_list = []
    is_verbose = True  # Set to True to see detail messages from API engine
    design = DesignAPI(is_verbose)
    device = DeviceAPI(is_verbose)
    is_soc_exists_flag = '0'

    peri_xml_default_path = Path(output_dirname) / f"{project_name}.peri.xml"
    project_peri_file_path = Path(peri_xml_file_path) if peri_xml_file_path else peri_xml_default_path
    is_load_existing_design = project_peri_file_path.exists()

    if is_load_existing_design:
        design.load(str(project_peri_file_path))
    else:
        # Create empty design
        design.create(project_name, device_name, output_dirname)

    ipm_pt_map = IpmPTMap()

    # SYSTEM CLOCK ----------------------------------------
    pll_soc_sys_clk_name = params["PLL_SOC_SYS_CLK_NAME"]
    pll_soc_sys_clk_resource = params["PLL_SOC_SYS_RESOURCE"]
    pll_soc_sys_clk_ext_src = params["PLL_SOC_SYS_EXT_CLK_SRC"]
    pll_soc_sys_clk_ref_freq = params["PLL_SOC_SYS_CLK_REF_FREQ"]
    pll_soc_sys_clkout0_freq = params["PLL_SOC_SYS_CLKOUT0_FREQ"]
    pll_soc_sys_clkout0_phase = params["PLL_SOC_SYS_CLKOUT0_PHASE"]
    if(pll_soc_sys_clk_resource == "PLL_BL1"):
        pll_soc_sys_clkout1_freq = params["MEM_FREQ"]
        pll_soc_sys_clkout1_phase = params["PLL_SOC_SYS_CLKOUT1_PHASE"]
        pll_soc_sys_clkout1_name = "io_memoryClk"
        pll_soc_sys_clkout2_freq = params["SYS_FREQ"]
        pll_soc_sys_clkout2_phase = params["PLL_SOC_SYS_CLKOUT2_PHASE"]
        pll_soc_sys_clkout2_name = "io_systemClk"
        pll_soc_sys_clkout3_freq = params["DDR_FREQ"]
        pll_soc_sys_clkout3_phase = params["PLL_SOC_SYS_CLKOUT3_PHASE"]
    else:
        pll_soc_sys_clkout1_freq = params["SYS_FREQ"]
        pll_soc_sys_clkout1_phase = params["PLL_SOC_SYS_CLKOUT1_PHASE"]
        pll_soc_sys_clkout1_name = "io_systemClk"
        pll_soc_sys_clkout2_freq = params["MEM_FREQ"]
        pll_soc_sys_clkout2_phase = params["PLL_SOC_SYS_CLKOUT2_PHASE"]
        pll_soc_sys_clkout2_name = "io_memoryClk"
        pll_soc_sys_clkout3_freq = params["DDR_FREQ"]
        pll_soc_sys_clkout3_phase = params["PLL_SOC_SYS_CLKOUT3_PHASE"]
    pll_soc_sys_locked="pll_system_locked"
    pll_res_assign=params["PLL_RES_ASSIGN"]

    pll_soc_sys_clkout_dict = {
        '0': {
            "CLKOUT_FREQ": pll_soc_sys_clkout0_freq,    # Output Clock 0 frequency in MHz
            "CLKOUT_PHASE": pll_soc_sys_clkout0_phase,  # Output Clock 0 phase in degree
            "CLKOUT_NAME" : "io_sysFbClk",              # Output Clock 0 name
        },
        '1': {
            "CLKOUT_FREQ": pll_soc_sys_clkout1_freq,    # Output Clock 1 frequency in MHz
            "CLKOUT_PHASE": pll_soc_sys_clkout1_phase,  # Output Clock 1 phase in degree
            "CLKOUT_NAME" : pll_soc_sys_clkout1_name,   # Output Clock 1 name
        },
        '2': {
            "CLKOUT_FREQ": pll_soc_sys_clkout2_freq,    # Output Clock 2 frequency in MHz
            "CLKOUT_PHASE": pll_soc_sys_clkout2_phase,  # Output Clock 2 phase in degree
            "CLKOUT_NAME" : pll_soc_sys_clkout2_name,   # Output Clock 2 name
        },
        '3': {
            "CLKOUT_FREQ": pll_soc_sys_clkout3_freq,    # Output Clock 3 frequency in MHz
            "CLKOUT_PHASE": pll_soc_sys_clkout3_phase,  # Output Clock 3 phase in degree
            "CLKOUT_NAME" : "ddrClk",                   # Output Clock 3 name
        }
    }
    if(pll_res_assign == '1'):
        handle_default_pll(design, pll_soc_sys_clk_name, pll_soc_sys_clk_resource, pll_soc_sys_locked, pll_soc_sys_clk_ref_freq, pll_soc_sys_clk_ext_src, pll_soc_sys_clkout_dict, export_list, "PLL_SOC_SYS_CLK_NAME", ipm_pt_map)

    # MEMORY CLOCK ----------------------------------------
    pll_soc_mem_clk_name = params["PLL_SOC_MEM_CLK_NAME"]
    pll_soc_mem_clk_resource = params["PLL_SOC_MEM_RESOURCE"]
    pll_soc_mem_clk_ext_src = params["PLL_SOC_MEM_EXT_CLK_SRC"]
    pll_soc_mem_clk_ref_freq = params["PLL_SOC_MEM_CLK_REF_FREQ"]
    pll_soc_mem_clkout0_freq = params["PLL_SOC_MEM_CLKOUT0_FREQ"]
    pll_soc_mem_clkout0_phase = params["PLL_SOC_MEM_CLKOUT0_PHASE"]
    pll_soc_mem_clkout1_freq = params["PERI_FREQ"]
    pll_soc_mem_clkout1_phase = params["PLL_SOC_MEM_CLKOUT1_PHASE"]
    pll_soc_mem_clkout2_freq = params["AXIM_FREQ"]
    pll_soc_mem_clkout2_phase = params["PLL_SOC_MEM_CLKOUT2_PHASE"]
    pll_soc_mem_clkout3_freq = params["CFU_FREQ"]
    pll_soc_mem_clkout3_phase = params["PLL_SOC_MEM_CLKOUT3_PHASE"]

    pll_soc_mem_locked="pll_peripheral_locked"
    pll_res_assign=params["PLL_RES_ASSIGN_2"]
    axi_master_en = params["INTF_AXIM"]
    axi_slave_en  = params["INTF_AXIS"]
    cust_inst0_en = params["INTF_CI_0"]
    cust_inst1_en = params["INTF_CI_1"]
    cust_inst2_en = params["INTF_CI_2"]
    cust_inst3_en = params["INTF_CI_3"]

    pll_soc_mem_clkout_dict = {
        '0': {
            "CLKOUT_FREQ": pll_soc_mem_clkout0_freq,        # Output Clock 0 frequency in MHz
            "CLKOUT_PHASE": pll_soc_mem_clkout0_phase,      # Output Clock 0 phase in degree
            "CLKOUT_NAME" : "io_memFbClk",                  # Output Clock 0 name
        },
        '1': {
            "CLKOUT_FREQ": str(pll_soc_mem_clkout1_freq),   # Output Clock 1 frequency in MHz
            "CLKOUT_PHASE": pll_soc_mem_clkout1_phase,      # Output Clock 1 phase in degree
            "CLKOUT_NAME" : "io_peripheralClk",             # Output Clock 1 name
        },
        '2': {
            "CLKOUT_FREQ": str(pll_soc_mem_clkout2_freq),   # Output Clock 2 frequency in MHz
            "CLKOUT_PHASE": pll_soc_mem_clkout2_phase,      # Output Clock 2 phase in degree
            "CLKOUT_NAME" : "io_ddrMasters_0_clk",          # Output Clock 2 name
        },
        '3': {
            "CLKOUT_FREQ": str(pll_soc_mem_clkout3_freq),   # Output Clock 3 frequency in MHz
            "CLKOUT_PHASE": pll_soc_mem_clkout3_phase,      # Output Clock 3 phase in degree
            "CLKOUT_NAME" : "io_cfuClk",                    # Output Clock 3 name
        }
    }
    if not(axi_slave_en == '1'):
        pll_soc_mem_clkout_dict.pop('1')
    if not(axi_master_en == '1'):
        pll_soc_mem_clkout_dict.pop('2')
    if (cust_inst0_en == '0' and cust_inst1_en == '0' and cust_inst2_en == '0' and cust_inst3_en == '0'):
        pll_soc_mem_clkout_dict.pop('3')

    if(pll_res_assign == '1'):
        handle_default_pll(design, pll_soc_mem_clk_name, pll_soc_mem_clk_resource, pll_soc_mem_locked, pll_soc_mem_clk_ref_freq, pll_soc_mem_clk_ext_src, pll_soc_mem_clkout_dict, export_list, "PLL_SOC_MEM_CLK_NAME", ipm_pt_map)

    # Auto assign pin?
    res_assign=params['PERI_RES_ASSIGN']
    pin_assign=params['PERI_PIN_ASSIGN']

    #cslau: Pin Assignment List for each supported package --------------------------
    # Default pin list
    # jtag_pinlist : ["io_jtag_tms","io_jtag_tdi","io_jtag_tdo","io_jtag_tck"]
    # spi_pdict    : ["dat0","dat1","sclk","ss"]
    # uart_pdict   : ["rxd", "txd"]
    # i2c_pdict    : ["scl","sda"]
    # gpio_pdict   : [0-3]

    # Current Supported Hardened Soc Package: ["C529","N484","N676","N900","N1156"]
    package_pdic = { 
        "C529" : {
            "jtag_pdict" : {
                "jtag_pinlist": ["P18","V19","T18","V20"], #GPIOL_80,GPIOL_46,GPIOL_83,GPIOL_45
                "jtag_io_bank": "TL1_TL5",
                "jtag_io_standard":"3.3_V_LVCMOS",
                "jtag_io_bank_voltage": "3.3"
            },

            "spi_pdict" : {
                "0":["H4","H5","G6","A7"],#GPIOB_N_40,GPIOB_P_40,GPIOB_P_39,GPIOB_N_38
                "1":["G15","G16","F16","F17"],#GPIOR_88,GPIOR_89,GPIOR_90,GPIOR_91
                "2":["B12","C14","C13","C12"]#GPIOR_96,GPIOR_97,GPIOR_98,GPIOR_99
            },

            "uart_pdict" : {
                "0":["E10","E9"],#GPIOR_145,GPIOR_165
                "1":["G17","A11"],#GPIOR_92,GPIOR_93
                "2":["D12","F12"]#GPIOR_100,GPIOR_101
            },

            "i2c_pdict":{
                "0":["R17","P17"],#GPIOL_85,GPIOL_84
                "1":["A13","A12"],#GPIOR_94,GPIOR_95
                "2":["F11","E13"]#GPIOR_141,GPIOR_103
            },  
            "gpio_pdict" : {
                "0":["R19","V4","U6","L18"],#GPIOT_P_19,GPIOL_23,GPIOL24,GPIOT_N_50
                "1":["E14","E16","F13","E15"]#GPIOR_104,GPIOR_105,GPIOR_106,GPIOR_107
            },

            "io_gpio_sw_n" : "F14"#GPIOR_108
        },
        "N484" : {
            "jtag_pdict" : {
                "jtag_pinlist": ["M8","K7","K6","K8"], #GPIOR_142,GPIOR_143,GPIOR_145,GPIOR_141
                "jtag_io_bank": "BR0",
                "jtag_io_standard":"3.3_V_LVCMOS",
                "jtag_io_bank_voltage": "3.3"
            },

            "spi_pdict" : {
                "0":["J4","J5","G4","C1"],#GPIOB_N_40,GPIOB_P_40,GPIOB_P_39,GPIOB_N_38
                "1":["H6","G6","H7","H8"],#GPIOR_166,GPIOR_167,GPIOR_168,GPIOR_169
                "2":["F5","F4","N6","N5"]#GPIOR_170,GPIOR_171,GPIOB_P_09,GPIOB_N_09
            },

            "uart_pdict":{
                "0":["N1","P1"],#GPIOB_P_04,GPIOB_N_04
                "1":["P3","T6"],#GPIOB_N_05,GPIOB_N_06
                "2":["M2","R2"]#GPIOB_N_07,GPIOB_N_03
            },

            "i2c_pdict":{
                "0":["F1","T16"],#GPIOB_N_30,GPIOT_N_14
                "1":["L21","L22"],#GPIOR_98,GPIOR_99
                "2":["N21","N22"]#GPIOR_95,GPIOR_96
            },

            "gpio_pdict" : {
                "0":["N7","P6","N3","N2"],#GPIOB_P_08,GPIOB_N_08,GPIOB_P_12,GPIOB_N_12
                "1":["K1","L1","P5","R4"]#GPIOB_P_13,GPIOB_N_13,GPIOB_P_14,GPIOB_N_14
            },

            "io_gpio_sw_n" : "K3"#GPIOB_N_24
        },

        "N1156" : {
            "jtag_pdict" : {
                "jtag_pinlist": ["AH27","AH28","AF26","AF27"], #GPIOL_45,GPIOL_46,GPIOL_86,GPIOL_84
                "jtag_io_bank": "TL1_TL5",
                "jtag_io_standard":"3.3_V_LVCMOS",
                "jtag_io_bank_voltage": "3.3"
            },

            "spi_pdict" : {
                "0":["T1","R1","N5","R3"],#GPIOB_N_40,GPIOB_P_40,GPIOB_P_39,GPIOB_N_38
                "1":["AE11","AD10","AC9","Y7"],#GPIOL_20,GPIOL_21,GPIOL_24,GPIOB_P_18
                "2":["H1","H2","J2","J3"]#GPIOR_170,GPIOR_171,GPIOB_P_09,GPIOB_N_09
            },

            "uart_pdict":{
                "0":["M12","N10"],#GPIOR_144,GPIOR_141
                "1":["M10","P12"],#GPIOR_140,GPIOR_143
                "2":["M2","R2"]#GPIOB_N_07,GPIOB_N_03
            },

            "i2c_pdict":{
                "0":["L3","K3"],#GPIOB_N_30,GPIOT_N_14
                "1":["U4","U3"],#GPIOB_P_47,GPIOB_N_47
                "2":["R5","T5"]#GPIOB_P_48,GPIOB_N_48
            },

            "gpio_pdict" : {
                "0":["T3","T4","N6","M6"],#GPIOB_N_33,GPIOB_P_33,GPIOB_N_41,GPIOB_P_41
                "1":["K1","L1","L4","L5"]#GPIOB_N_45,GPIOB_P_45,GPIOB_P_46,GPIOB_N_46
            },

            "io_gpio_sw_n" : "U1"#GPIOB_N_35
        },

        "N676" : {
            "jtag_pdict" : {
                "jtag_pinlist": ["G7","H5","G5","H7"], #GPIOR_129,GPIOR_127,GPIOR_128,GPIOR_126
                "jtag_io_bank": "BR0",
                "jtag_io_standard":"3.3_V_LVCMOS",
                "jtag_io_bank_voltage": "3.3"
            },

            "spi_pdict" : {
                "0":["A3","A4","B3","D3"],#GPIOB_N_27,GPIOB_P_27,GPIOB_P_26,GPIOB_N_25
                "1":["K22","J23","G21","G22"],#GPIOR_94,GPIOR_95,GPIOR_96,GPIOR_97
                "2":["H22","G23","H23","H24"]#GPIOR_98,GPIOR_99,GPIOR_100,GPIOR_101
            },

            "uart_pdict":{
                "0":["F19","F20"],#GPIOR_110,GPIOR_111
                "1":["F21","F23"],#GPIOR_113,GPIOR_114
                "2":["E22","D22"]#GPIOR_116,GPIOR_117
            },

            "i2c_pdict":{
                "0":["P25","N25"],#GPIOT_P_21,GPIOT_N_21
                "1":["P24","N24"],#GPIOT_P_24,GPIOT_N_24
                "2":["N26","M26"]#GPIOT_P_25,GPIOT_N_25
            },

            "gpio_pdict" : {
                "0":["P17","N17","R26","R25"],#GPIOT_P_17,GPIOT_N_17,GPIOT_P_18,GPIOT_N_18
                "1":["R22","R21","R23","P23"]#GPIOT_P_22,GPIOT_N_22,GPIOT_P_23,GPIOT_N_23
            },

            "io_gpio_sw_n" : "F25"#GPIOR_119
        },
        "N900" : {
            "jtag_pdict" : {
                "jtag_pinlist": ["U20","U21","W21","W22"], 
                "jtag_io_bank": "TL1_TL5",
                "jtag_io_standard":"3.3_V_LVCMOS",
                "jtag_io_bank_voltage": "3.3"
            },

            "spi_pdict" : {
                "0":["P2","R1","N5","P3"],
                "1":["P4","T2","AC9","Y7"],
                "2":["H1","J1","J2","J3"]
            },

            "uart_pdict":{
                "0":["P8","P9"],
                "1":["M10","N9"],
                "2":["M2","R2"]
            },

            "i2c_pdict":{
                "0":["L3","M7"],
                "1":["M8","U3"],
                "2":["R5","T5"]
            },

            "gpio_pdict" : {
                "0":["T3","T4","N6","M6"],
                "1":["K1","L2","L4","L5"]
            },

            "io_gpio_sw_n" : "U1" 
        },
        "Default" : {
            "jtag_pdict" : {
                "jtag_pinlist": [], 
                "jtag_io_bank": "",
                "jtag_io_standard":"3.3_V_LVCMOS",
                "jtag_io_bank_voltage": "3.3"
            },

            "spi_pdict" : {
                "0":["","","",""],
                "1":["","","",""],
                "2":["","","",""]
            },

            "uart_pdict":{
                "0":["",""],
                "1":["",""],
                "2":["",""]
            },

            "i2c_pdict":{
                "0":["",""],
                "1":["",""],
                "2":["",""]
            },

            "gpio_pdict" : {
                "0":["","","",""],
                "1":["","","",""]
            },

            "io_gpio_sw_n" : ""
        }
    }

    # UART -----------------------------------------------------------
    #default pinlist
    #rxd,txd
    #uart_pdict={
    #0:["E10","E9"],#GPIOR_145,GPIOR_165
    #1:["G17","A11"],#GPIOR_92,GPIOR_93
    #2:["D12","F12"]#GPIOR_100,GPIOR_101
    #}
    uart_pinlist = gen_pinlist(package_pdic,device_name,interface="uart_pdict")
    uart0_enabled = (params['PERI_UART_0']=='1' and params['PERI_GEN']=='1')
    uart1_enabled = (params['PERI_UART_1']=='1' and params['PERI_GEN']=='1')
    uart2_enabled = (params['PERI_UART_2']=='1' and params['PERI_GEN']=='1')

    uart0_inst_name = "system_uart_0"
    uart1_inst_name = "system_uart_1"
    uart2_inst_name = "system_uart_2"

    if uart0_enabled == True:
        if(res_assign == '1'):
            create_def_uart_gpio_block(design, uart0_inst_name, uart_pinlist["0"], pin_assign, export_list, ipm_pt_map)
    else:
        delete_uart_gpio_block(design, uart0_inst_name)

    if uart1_enabled == True:
        if(res_assign == '1'):
            create_def_uart_gpio_block(design, uart1_inst_name, uart_pinlist["1"], pin_assign, export_list, ipm_pt_map)
    else:
        delete_uart_gpio_block(design, uart1_inst_name)

    if uart2_enabled == True:
        if(res_assign == '1'):
            create_def_uart_gpio_block(design, uart2_inst_name, uart_pinlist["2"], pin_assign, export_list, ipm_pt_map)
    else:
        delete_uart_gpio_block(design, uart2_inst_name)

    # SPI -----------------------------------------------------------
    #default pinlist
    #dat0,dat1,sclk,ss
    #spi_pdict={
    #0:["H4","H5","G6","A7"],#GPIOB_N_40,GPIOB_P_40,GPIOB_P_39,GPIOB_N_38
    #1:["G15","G16","F16","F17"],#GPIOR_88,GPIOR_89,GPIOR_90,GPIOR_91
    #2:["B12","C14","C13","C12"]#GPIOR_96,GPIOR_97,GPIOR_98,GPIOR_99
    #}
    spi_pinlist = gen_pinlist(package_pdic,device_name,interface="spi_pdict")
    spi0_enabled = (params['PERI_SPI_0']=='1' and params['PERI_GEN']=='1')
    spi1_enabled = (params['PERI_SPI_1']=='1' and params['PERI_GEN']=='1')
    spi2_enabled = (params['PERI_SPI_2']=='1' and params['PERI_GEN']=='1')

    spi0_inst_name = "system_spi_0"
    spi1_inst_name = "system_spi_1"
    spi2_inst_name = "system_spi_2"

    if spi0_enabled == True:
        if(res_assign == '1'):
            create_def_spi_gpio_block(design, spi0_inst_name, spi_pinlist["0"], pin_assign,pll_res_assign, export_list, ipm_pt_map)
    else:
        delete_spi_gpio_block(design, spi0_inst_name)

    if spi1_enabled == True:
        if(res_assign == '1'):
            create_def_spi_gpio_block(design, spi1_inst_name, spi_pinlist["1"], pin_assign,pll_res_assign, export_list, ipm_pt_map)
    else:
        delete_spi_gpio_block(design, spi1_inst_name)

    if spi2_enabled == True:
        if(res_assign == '1'):
            create_def_spi_gpio_block(design, spi2_inst_name, spi_pinlist["2"], pin_assign,pll_res_assign, export_list, ipm_pt_map)
    else:
        delete_spi_gpio_block(design, spi2_inst_name)


    # I2C -----------------------------------------------------------
    #default pinlist
    #scl,sda
    #i2c_pdict={
    #0:["T19","R17"],#GPIOL_85,GPIOL_84
    #1:["A13","A12"],#GPIOR_94,GPIOR_95
    #2:["D13","E13"]#GPIOR_102,GPIOR_103
    #}
    i2c0_enabled = (params['PERI_I2C_0']=='1' and params['PERI_GEN']=='1')
    i2c1_enabled = (params['PERI_I2C_1']=='1' and params['PERI_GEN']=='1')
    i2c2_enabled = (params['PERI_I2C_2']=='1' and params['PERI_GEN']=='1')
    i2c_pinlist = gen_pinlist(package_pdic,device_name,interface="i2c_pdict")
    i2c0_inst_name = "system_i2c_0"
    i2c1_inst_name = "system_i2c_1"
    i2c2_inst_name = "system_i2c_2"

    if i2c0_enabled == True:
        if(res_assign == '1'):
            create_def_i2c_gpio_block(design, i2c0_inst_name, i2c_pinlist["0"], pin_assign, export_list, ipm_pt_map)
    else:
        delete_i2c_gpio_block(design, i2c0_inst_name)

    if i2c1_enabled == True:
        if(res_assign == '1'):
            create_def_i2c_gpio_block(design, i2c1_inst_name, i2c_pinlist["1"], pin_assign, export_list, ipm_pt_map)
    else:
        delete_i2c_gpio_block(design, i2c1_inst_name)

    if i2c2_enabled == True:
        if(res_assign == '1'):
            create_def_i2c_gpio_block(design, i2c2_inst_name, i2c_pinlist["2"], pin_assign, export_list, ipm_pt_map)
    else:
        delete_i2c_gpio_block(design, i2c2_inst_name)

    # GPIO -----------------------------------------------------------
    #default pinlist
    #gpio0-3
    #gpio_pdict={
    #0:["R19","V4","U6","L18"],#GPIOT_P_19,GPIOL_23,GPIOL24,GPIOT_N_50
    #1:["E14","E16","F13","E15"]#GPIOR_104,GPIOR_105,GPIOR_106,GPIOR_107
    #}
    gpio0_enabled = (params['PERI_GPIO_0']=='1' and params['PERI_GEN']=='1')
    gpio1_enabled = (params['PERI_GPIO_1']=='1' and params['PERI_GEN']=='1')

    gpio0_inst_name = "system_gpio_0_io"
    gpio1_inst_name = "system_gpio_1_io"

    gpio_pinlist = gen_pinlist(package_pdic,device_name,interface="gpio_pdict")

    if gpio0_enabled == True:
        gpio_width = params['PERI_GPIO_0_WIDTH']
        int_gpio_width = int(gpio_width)
        if int_gpio_width >= 4:
            int_gpio_width = 4
        int_gpio_width -= 1

        if(res_assign == '1'):
            create_def_gpio_bus(design, gpio0_inst_name, msb=int_gpio_width, lsb=0, gpio_pinlist=gpio_pinlist["0"],
                                pin=pin_assign, export_list=export_list, ipm_pt_map=ipm_pt_map)
    else:
        delete_gpio_bus(design, gpio0_inst_name)

    if gpio1_enabled == True:
        gpio_width = params['PERI_GPIO_1_WIDTH']
        int_gpio_width = int(gpio_width)
        if int_gpio_width >= 4:
            int_gpio_width = 4
        int_gpio_width -= 1

        if(res_assign == '1'):
            create_def_gpio_bus(design, gpio1_inst_name, msb=int_gpio_width, lsb=0, gpio_pinlist=gpio_pinlist["1"],
                                pin=pin_assign, export_list=export_list, ipm_pt_map=ipm_pt_map)
    else:
        delete_gpio_bus(design, gpio1_inst_name)

    if(res_assign == '1'):
        if is_gpio_exists(design, "io_gpio_sw_n"):
            design.delete_gpio("io_gpio_sw_n")
        design.create_input_gpio("io_gpio_sw_n")
        if(pin_assign == '1'):
            #cslau: Insert pin assignment based on the device selected. 
            gpio_sw_n_list = gen_pinlist(package_pdic,device_name, interface ="io_gpio_sw_n")
            design.assign_pkg_pin("io_gpio_sw_n",gpio_sw_n_list)
        export_list.append("io_gpio_sw_n")
        ipm_pt_map.set_gpio_block_item(GPIOType.GPIO, "io_gpio_sw_n")


    # SHOWCASE ----------------------------------------
    # kmlau: remove showcase statemets

    # DDR ---------------------------------------------------------
    res_assign=params['DDR_RES_ASSIGN']
    ddr_pin_name = params["DDR_PIN_NAME"]
    ddr_data_width = params["DDR_DATA_WIDTH"]
    ddr_memory_density = params["DDR_MEMORY_DENSITY"]
    ddr_memory_type = params["DDR_MEMORY_TYPE"]
    ddr_physical_rank = params["DDR_PHYSICAL_RANK"]

    if (res_assign == '1'):
        if(pll_soc_sys_clk_resource == "PLL_BL0"):
            ddr_clk_src="CLKIN 0"
        elif(pll_soc_sys_clk_resource == "PLL_BL1"):
            ddr_clk_src="CLKIN 1"
        else:
            ddr_clk_src="CLKIN 2"
        if is_block_exists(design, ddr_pin_name, "DDR"):
            design.delete_block(ddr_pin_name,"DDR")
        design.create_block(ddr_pin_name,"DDR")
        export_list.append(ddr_pin_name)
        ipm_pt_map.set_block_item(IpmPTBlock.DDR, "DDR_PIN_NAME", "DDR_PIN_NAME", ddr_pin_name)

        design.assign_resource(ddr_pin_name,"DDR_0", "DDR")
        design.set_property(ddr_pin_name,"CLKIN_SEL",ddr_clk_src,"DDR")
        design.set_property(ddr_pin_name,"MEMORY_TYPE",ddr_memory_type,"DDR")
        design.set_property(ddr_pin_name,"PHYSICAL_RANK",ddr_physical_rank,"DDR")
        design.set_property(ddr_pin_name,"DQ_WIDTH",ddr_data_width,"DDR")
        design.set_property(ddr_pin_name,"MEMORY_DENSITY",ddr_memory_density,"DDR")
        design.set_property(ddr_pin_name,"CTRL_CLK_PIN",pll_soc_sys_clkout_dict["0"]["CLKOUT_NAME"],"DDR")
        #kmlau: set AXI_1 for user disable, this interface shouldn't appear in GUI as the route has been occupied by SOC.
        design.set_property(ddr_pin_name,"TARGET1_EN","0","DDR")
        design.set_property(ddr_pin_name,"CFG_DONE_PIN","cfg_done","DDR")
        design.set_property(ddr_pin_name,"CFG_RESET_PIN","cfg_reset","DDR")
        design.set_property(ddr_pin_name,"CFG_SEL_PIN","cfg_sel","DDR")
        design.set_property(ddr_pin_name,"CFG_START_PIN","cfg_start","DDR")
        #TODO pin swizzling & CA BUS


    # SOC -----------------------------------
    pll_res_assign=params["PLL_RES_ASSIGN"]
    if(pll_soc_sys_clk_resource == "PLL_BL0"):
        soc_0_clk_src="Clock 0"
    elif(pll_soc_sys_clk_resource == "PLL_BL1"):
        soc_0_clk_src="Clock 1"
    else:
        soc_0_clk_src="Clock 2"

    soc_pin_name = params["SOC_PIN_NAME"]
    if is_block_exists(design, soc_pin_name, "SOC"):
        is_soc_exists_flag = '1'
        sys_clk_src = design.get_property(soc_pin_name, "SYS_CLK_SOURCE", block_type="SOC")
        mem_clk_src = design.get_property(soc_pin_name, "MEM_CLK_SOURCE", block_type="SOC")
        design.delete_block(soc_pin_name, "SOC")
    design.create_block(soc_pin_name,"SOC")
    export_list.append(soc_pin_name)
    ipm_pt_map.set_block_item(IpmPTBlock.SOC, "SOC_PIN_NAME", "SOC_PIN_NAME", soc_pin_name)

    design.assign_resource(soc_pin_name,"SOC_0", "SOC")
    if(pll_res_assign == '1'):
        design.set_property(soc_pin_name,"SYS_CLK_SOURCE",soc_0_clk_src,"SOC")
        design.set_property(soc_pin_name,"MEM_CLK_SOURCE",soc_0_clk_src,"SOC")
    else:
        if(is_soc_exists_flag == '1'):
            design.set_property(soc_pin_name,"SYS_CLK_SOURCE",sys_clk_src["SYS_CLK_SOURCE"],"SOC")
            design.set_property(soc_pin_name,"MEM_CLK_SOURCE",mem_clk_src["MEM_CLK_SOURCE"],"SOC")
        else:
            design.set_property(soc_pin_name,"SYS_CLK_SOURCE",soc_0_clk_src,"SOC")
            design.set_property(soc_pin_name,"MEM_CLK_SOURCE",soc_0_clk_src,"SOC")

    is_soc_exists_flag == '0'

    #l_prop=get_property_by_name_and_type(design,soc_pin_name,"SOC")
    #print(l_prop)

    # app path
    is_hexfile_enabled = params["APP_OVERWRITE"]
    ocr_file_path = params["APP_OVERWRITE_PATH"]
    ocr_file_path = ocr_file_path if is_hexfile_enabled == "1" else ''
    design.set_property(soc_pin_name,"OCR_FILE_PATH",ocr_file_path,"SOC")

    #misc pins
    design.set_property(soc_pin_name,"IO_ASYNCRESET_PIN","io_asyncReset","SOC")
    design.set_property(soc_pin_name,"IO_SYSTEMRESET_PIN","io_systemReset","SOC")

    design.set_property(soc_pin_name,"AXI_MASTER_EN",axi_master_en,"SOC")

    if axi_master_en == '1':
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_CLK_PIN","io_ddrMasters_0_clk","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_ADDR_PIN","io_ddrMasters_0_ar_payload_addr","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_BURST_PIN","io_ddrMasters_0_ar_payload_burst","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_CACHE_PIN","io_ddrMasters_0_ar_payload_cache","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_ID_PIN","io_ddrMasters_0_ar_payload_id","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_LEN_PIN","io_ddrMasters_0_ar_payload_len","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_LOCK_PIN","io_ddrMasters_0_ar_payload_lock","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_PROT_PIN","io_ddrMasters_0_ar_payload_prot","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_QOS_PIN","io_ddrMasters_0_ar_payload_qos","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_REGION_PIN","io_ddrMasters_0_ar_payload_region","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_PAYLOAD_SIZE_PIN","io_ddrMasters_0_ar_payload_size","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_READY_PIN","io_ddrMasters_0_ar_ready","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AR_VALID_PIN","io_ddrMasters_0_ar_valid","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_ADDR_PIN","io_ddrMasters_0_aw_payload_addr","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_ALLSTRB_PIN","io_ddrMasters_0_aw_payload_allStrb","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_BURST_PIN","io_ddrMasters_0_aw_payload_burst","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_CACHE_PIN","io_ddrMasters_0_aw_payload_cache","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_ID_PIN","io_ddrMasters_0_aw_payload_id","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_LEN_PIN","io_ddrMasters_0_aw_payload_len","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_LOCK_PIN","io_ddrMasters_0_aw_payload_lock","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_PROT_PIN","io_ddrMasters_0_aw_payload_prot","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_QOS_PIN","io_ddrMasters_0_aw_payload_qos","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_REGION_PIN","io_ddrMasters_0_aw_payload_region","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_PAYLOAD_SIZE_PIN","io_ddrMasters_0_aw_payload_size","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_READY_PIN","io_ddrMasters_0_aw_ready","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_AW_VALID_PIN","io_ddrMasters_0_aw_valid","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_B_PAYLOAD_ID_PIN","io_ddrMasters_0_b_payload_id","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_B_PAYLOAD_RESP_PIN","io_ddrMasters_0_b_payload_resp","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_B_READY_PIN","io_ddrMasters_0_b_ready","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_B_VALID_PIN","io_ddrMasters_0_b_valid","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_CLK_PIN","io_ddrMasters_0_clk","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_RESET_PIN","io_ddrMasters_0_reset","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_R_PAYLOAD_DATA_PIN","io_ddrMasters_0_r_payload_data","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_R_PAYLOAD_ID_PIN","io_ddrMasters_0_r_payload_id","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_R_PAYLOAD_LAST_PIN","io_ddrMasters_0_r_payload_last","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_R_PAYLOAD_RESP_PIN","io_ddrMasters_0_r_payload_resp","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_R_READY_PIN","io_ddrMasters_0_r_ready","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_R_VALID_PIN","io_ddrMasters_0_r_valid","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_W_PAYLOAD_DATA_PIN","io_ddrMasters_0_w_payload_data","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_W_PAYLOAD_LAST_PIN","io_ddrMasters_0_w_payload_last","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_W_PAYLOAD_STRB_PIN","io_ddrMasters_0_w_payload_strb","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_W_READY_PIN","io_ddrMasters_0_w_ready","SOC")
        design.set_property(soc_pin_name,"IO_DDRMASTERS_0_W_VALID_PIN","io_ddrMasters_0_w_valid","SOC")

    design.set_property(soc_pin_name,"AXI_SLAVE_EN",axi_slave_en,"SOC")
    if axi_slave_en == '1':
        design.set_property(soc_pin_name,"IO_PERIPHERALCLK_PIN","io_peripheralClk","SOC")
        design.set_property(soc_pin_name,"IO_PERIPHERALRESET_PIN","io_peripheralReset","SOC")
        design.set_property(soc_pin_name,"AXIAINTERRUPT_PIN","axiAInterrupt","SOC")
        design.set_property(soc_pin_name,"AXIA_ARADDR_PIN","axiA_araddr","SOC")
        design.set_property(soc_pin_name,"AXIA_ARBURST_PIN","axiA_arburst","SOC")
        design.set_property(soc_pin_name,"AXIA_ARCACHE_PIN","axiA_arcache","SOC")
        design.set_property(soc_pin_name,"AXIA_ARLEN_PIN","axiA_arlen","SOC")
        design.set_property(soc_pin_name,"AXIA_ARLOCK_PIN","axiA_arlock","SOC")
        design.set_property(soc_pin_name,"AXIA_ARPROT_PIN","axiA_arprot","SOC")
        design.set_property(soc_pin_name,"AXIA_ARQOS_PIN","axiA_arqos","SOC")
        design.set_property(soc_pin_name,"AXIA_ARREADY_PIN","axiA_arready","SOC")
        design.set_property(soc_pin_name,"AXIA_ARREGION_PIN","axiA_arregion","SOC")
        design.set_property(soc_pin_name,"AXIA_ARSIZE_PIN","axiA_arsize","SOC")
        design.set_property(soc_pin_name,"AXIA_ARVALID_PIN","axiA_arvalid","SOC")
        design.set_property(soc_pin_name,"AXIA_AWADDR_PIN","axiA_awaddr","SOC")
        design.set_property(soc_pin_name,"AXIA_AWBURST_PIN","axiA_awburst","SOC")
        design.set_property(soc_pin_name,"AXIA_AWCACHE_PIN","axiA_awcache","SOC")
        design.set_property(soc_pin_name,"AXIA_AWLEN_PIN","axiA_awlen","SOC")
        design.set_property(soc_pin_name,"AXIA_AWLOCK_PIN","axiA_awlock","SOC")
        design.set_property(soc_pin_name,"AXIA_AWPROT_PIN","axiA_awprot","SOC")
        design.set_property(soc_pin_name,"AXIA_AWQOS_PIN","axiA_awqos","SOC")
        design.set_property(soc_pin_name,"AXIA_AWREADY_PIN","axiA_awready","SOC")
        design.set_property(soc_pin_name,"AXIA_AWREGION_PIN","axiA_awregion","SOC")
        design.set_property(soc_pin_name,"AXIA_AWSIZE_PIN","axiA_awsize","SOC")
        design.set_property(soc_pin_name,"AXIA_AWVALID_PIN","axiA_awvalid","SOC")
        design.set_property(soc_pin_name,"AXIA_BREADY_PIN","axiA_bready","SOC")
        design.set_property(soc_pin_name,"AXIA_BRESP_PIN","axiA_bresp","SOC")
        design.set_property(soc_pin_name,"AXIA_BVALID_PIN","axiA_bvalid","SOC")
        design.set_property(soc_pin_name,"AXIA_RDATA_PIN","axiA_rdata","SOC")
        design.set_property(soc_pin_name,"AXIA_RLAST_PIN","axiA_rlast","SOC")
        design.set_property(soc_pin_name,"AXIA_RREADY_PIN","axiA_rready","SOC")
        design.set_property(soc_pin_name,"AXIA_RRESP_PIN","axiA_rresp","SOC")
        design.set_property(soc_pin_name,"AXIA_RVALID_PIN","axiA_rvalid","SOC")
        design.set_property(soc_pin_name,"AXIA_WDATA_PIN","axiA_wdata","SOC")
        design.set_property(soc_pin_name,"AXIA_WLAST_PIN","axiA_wlast","SOC")
        design.set_property(soc_pin_name,"AXIA_WREADY_PIN","axiA_wready","SOC")
        design.set_property(soc_pin_name,"AXIA_WSTRB_PIN","axiA_wstrb","SOC")
        design.set_property(soc_pin_name,"AXIA_WVALID_PIN","axiA_wvalid","SOC")

    # Customer instruction is either enabled for all or disabled for all
    design.set_property(soc_pin_name,"CUSTOM_INSTRUCTION_0_EN",cust_inst0_en,"SOC")
    design.set_property(soc_pin_name,"CUSTOM_INSTRUCTION_1_EN",cust_inst1_en,"SOC")
    design.set_property(soc_pin_name,"CUSTOM_INSTRUCTION_2_EN",cust_inst2_en,"SOC")
    design.set_property(soc_pin_name,"CUSTOM_INSTRUCTION_3_EN",cust_inst3_en,"SOC")

    if (cust_inst0_en == '1' or cust_inst1_en == '1' or cust_inst2_en == '1' or cust_inst3_en == '1'):
        design.set_property(soc_pin_name,"IO_CFUCLK_PIN","io_cfuClk","SOC")
        design.set_property(soc_pin_name,"IO_CFURESET_PIN","io_cfuReset","SOC")
    if (cust_inst0_en == '1'):
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_CMD_READY_PIN","cpu0_customInstruction_cmd_ready","SOC")
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_CMD_VALID_PIN","cpu0_customInstruction_cmd_valid","SOC")
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_FUNCTION_ID_PIN","cpu0_customInstruction_function_id","SOC")
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_INPUTS_0_PIN","cpu0_customInstruction_inputs_0","SOC")
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_INPUTS_1_PIN","cpu0_customInstruction_inputs_1","SOC")
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_OUTPUTS_0_PIN","cpu0_customInstruction_outputs_0","SOC")
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_RSP_READY_PIN","cpu0_customInstruction_rsp_ready","SOC")
        design.set_property(soc_pin_name,"CPU0_CUSTOMINSTRUCTION_RSP_VALID_PIN","cpu0_customInstruction_rsp_valid","SOC")
    if (cust_inst1_en == '1'):
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_CMD_READY_PIN","cpu1_customInstruction_cmd_ready","SOC")
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_CMD_VALID_PIN","cpu1_customInstruction_cmd_valid","SOC")
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_FUNCTION_ID_PIN","cpu1_customInstruction_function_id","SOC")
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_INPUTS_0_PIN","cpu1_customInstruction_inputs_0","SOC")
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_INPUTS_1_PIN","cpu1_customInstruction_inputs_1","SOC")
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_OUTPUTS_0_PIN","cpu1_customInstruction_outputs_0","SOC")
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_RSP_READY_PIN","cpu1_customInstruction_rsp_ready","SOC")
        design.set_property(soc_pin_name,"CPU1_CUSTOMINSTRUCTION_RSP_VALID_PIN","cpu1_customInstruction_rsp_valid","SOC")
    if (cust_inst2_en == '1'):
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_CMD_READY_PIN","cpu2_customInstruction_cmd_ready","SOC")
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_CMD_VALID_PIN","cpu2_customInstruction_cmd_valid","SOC")
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_FUNCTION_ID_PIN","cpu2_customInstruction_function_id","SOC")
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_INPUTS_0_PIN","cpu2_customInstruction_inputs_0","SOC")
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_INPUTS_1_PIN","cpu2_customInstruction_inputs_1","SOC")
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_OUTPUTS_0_PIN","cpu2_customInstruction_outputs_0","SOC")
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_RSP_READY_PIN","cpu2_customInstruction_rsp_ready","SOC")
        design.set_property(soc_pin_name,"CPU2_CUSTOMINSTRUCTION_RSP_VALID_PIN","cpu2_customInstruction_rsp_valid","SOC")
    if (cust_inst3_en == '1'):
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_CMD_READY_PIN","cpu3_customInstruction_cmd_ready","SOC")
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_CMD_VALID_PIN","cpu3_customInstruction_cmd_valid","SOC")
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_FUNCTION_ID_PIN","cpu3_customInstruction_function_id","SOC")
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_INPUTS_0_PIN","cpu3_customInstruction_inputs_0","SOC")
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_INPUTS_1_PIN","cpu3_customInstruction_inputs_1","SOC")
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_OUTPUTS_0_PIN","cpu3_customInstruction_outputs_0","SOC")
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_RSP_READY_PIN","cpu3_customInstruction_rsp_ready","SOC")
        design.set_property(soc_pin_name,"CPU3_CUSTOMINSTRUCTION_RSP_VALID_PIN","cpu3_customInstruction_rsp_valid","SOC")

    num_intr=int(params["INTF_UINTR"])
    for p in range(num_intr):
        design.set_property(soc_pin_name,f"USERINTERRUPT{intr_decode(p)}_PIN",f"userInterrupt{intr_decode(p)}","SOC")

    if not (num_intr == 24):
        for p in range(num_intr, 24):
            design.set_property(soc_pin_name,f"USERINTERRUPT{intr_decode(p)}_PIN","","SOC")


    # JTAG -----------------------------------------------------------
    # Default
    # jtag_ports = [
    #     "io_jtag_tms",
    #     "io_jtag_tdi",
    #     "io_jtag_tdo",
    #     "io_jtag_tck"
    # ]
    
    is_soft_jtag_enabled  = (params['INTF_JTAG_TYPE'] == '1')
    is_res_assign_enabled = (params['PERI_RES_ASSIGN'] == '1')
    soft_jtag_inst_name   = params["JTAG_GPIO_INST_NAME"]
    hard_jtag_inst_name   = params["HARD_JTAG_INST_NAME"]

    if is_soft_jtag_enabled == True:
        # Need to set this property even PERI_RES_ASSIGN is false.
        design.set_property(soc_pin_name,"JTAG_TYPE","CPU","SOC")
        design.set_property(soc_pin_name,"IO_JTAG_TCK_PIN","io_jtag_tck","SOC")
        design.set_property(soc_pin_name,"IO_JTAG_TDI_PIN","io_jtag_tdi","SOC")
        design.set_property(soc_pin_name,"IO_JTAG_TDO_PIN","io_jtag_tdo","SOC")
        design.set_property(soc_pin_name,"IO_JTAG_TMS_PIN","io_jtag_tms","SOC")
        
        if is_block_exists(design, hard_jtag_inst_name, "JTAG"):
            design.delete_block(hard_jtag_inst_name, "JTAG")
        if is_res_assign_enabled == True:
        # Will only enter this function if PERI_RES_ASSIGN is true. 
            jtag_pinlist = gen_pinlist(package_pdic,device_name,interface="jtag_pdict")
            create_def_soft_jtag_gpio_block(design, soft_jtag_inst_name, soc_pin_name, jtag_pinlist, pin_assign, export_list, ipm_pt_map)

    else:
        resource_to_assign = params['INTF_JTAG_TAP_SEL']
        design.set_property(soc_pin_name,"JTAG_TYPE","FPGA","SOC")
        delete_soft_jtag_gpio_block(design, soft_jtag_inst_name, soc_pin_name)
        create_def_hard_jtag(design, hard_jtag_inst_name, resource_to_assign, export_list, ipm_pt_map)

        design.set_property(soc_pin_name,"JTAGCTRL_TCK_PIN","jtagCtrl_tck","SOC")
        design.set_property(soc_pin_name,"JTAGCTRL_CAPTURE_PIN","jtagCtrl_capture","SOC")
        design.set_property(soc_pin_name,"JTAGCTRL_ENABLE_PIN","jtagCtrl_enable","SOC")
        design.set_property(soc_pin_name,"JTAGCTRL_RESET_PIN","jtagCtrl_reset","SOC")
        design.set_property(soc_pin_name,"JTAGCTRL_SHIFT_PIN","jtagCtrl_shift","SOC")
        design.set_property(soc_pin_name,"JTAGCTRL_TDI_PIN","jtagCtrl_tdi","SOC")
        design.set_property(soc_pin_name,"JTAGCTRL_TDO_PIN","jtagCtrl_tdo","SOC")
        design.set_property(soc_pin_name,"JTAGCTRL_UPDATE_PIN","jtagCtrl_update","SOC")

    # Validate --------------------------------------
    is_pass = design.check_design() # Since we didn't assign resource, it should always fail for now.
    error_list = design.get_design_check_issue()
    list_of_filtered_out_errors = [
        "Rule:pll_rule_resource,Msg:Resource name is empty",
        "Rule:ddr_rule_resource,Msg:Resource name is empty",
        "Rule:gpio_rule_resource,Msg:Resource name is empty",
        "Rule:jtag_rule_resource,Msg:Resource name is empty",
        "Rule:ddr_rule_axi0_empty_pins,Msg:Empty pin names found: ACLK_0",
        "Rule:ddr_rule_axi1_empty_pins,Msg:Empty pin names found: ACLK_1",
        "Rule:qcrv32_rule_resource,Msg:Resource is not a valid QCRV32 device instance.",
        "Rule:qcrv32_rule_mem_clk_resource",
        "Rule:qcrv32_rule_sys_clk_resource",
        "Rule:ddr_rule_phy_clock",
        "Rule:gpio_rule_io_standard_bank,Msg:Mismatch voltage",
        "SeverityType.warning",
        "SeverityType.info"
    ]

    has_error = False
    for error in error_list:
        is_filter_out_error = any(filtered_out_error for filtered_out_error in list_of_filtered_out_errors if filtered_out_error in error)
        if not is_filter_out_error:
            has_error = True
            print(error, file=sys.stderr)

    design_issue_path = Path(output_dirname) / "outflow" / "design_issue.xml"
    if design_issue_path.exists():
        try:
            design_issue_path.unlink()
            print('Successfully deleted design_issue.xml')
        except PermissionError:
            # This is not an error. I don't want to make it a showstopper.
            print(f"Lack permission to delete '{design_issue_path}'.")

    # Save the configured periphery design
    if not has_error:
        user_project_default_peri_path = f"{user_project_path}/{project_name}.peri.xml"
        user_project_peri_file_path = peri_xml_file_path if peri_xml_file_path else user_project_default_peri_path
        user_project_peri_file_isf = f"{user_project_path}/{project_name}.isf"
        user_project_ipm_pt_map_path = Path(f"{user_project_path}/ip/{gen_name}")

        # if is_load_existing_design:
        #     design.save_as(str(user_project_peri_file_path), overwrite=True)
        # else:
        #     design.save()
        design.save_as(user_project_peri_file_path, overwrite=True)
        design.export_design(block_inst=export_list, isf_file=user_project_peri_file_isf)
        # NOTE: Cannot generate unless we assign resource.
        # design.generate(enable_bitstream=False)
        if not user_project_ipm_pt_map_path.exists():
            user_project_ipm_pt_map_path.mkdir(parents=True, exist_ok=True)

        ipm_pt_map.save(str(user_project_ipm_pt_map_path))

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument('--device_name', help='Device currently set by the project', required=True)
    parser.add_argument('--project_name', help='Name of the project', required=True)
    parser.add_argument('--out_path', help='Output Folder Path. This is a tmp folder during upgrade.', required=True)
    parser.add_argument('--gen_name', help='Generated name of the IP.', required=True)
    parser.add_argument('--param_ini_path', help='Param Ini Path', required=True)
    parser.add_argument('--user_project_path', help='User Project Path. Same as out_path except during upgrade, it holds the actual user project path', required=True)
    parser.add_argument('--peri_xml_file_path', help='Peri Xml File Path in project', required=True)
    # parser.add_argument('--parameters',
    #         metavar="KEY=VALUE",
    #         nargs="+",
    #         help='Key-Value Pair for resolved parameters from the IPXACT'
    #         "(do not put spaces before or after the = sign). "
    #         "If a value contains spaces, you should define "
    #         "it with double quotes: "
    #         'foo="this is a sentence". Note that '
    #         "values are always treated as strings."
    #         , required=True)
    args = parser.parse_args()
    params, ports = parse_hard_ip_ini_file(args.param_ini_path)

    '''
    Parameters to consider passing in from efx_ipmgr:
        is_verbose
    '''

    gen_name = args.gen_name

    # NOTE: Just do some hack first for ip_project_path. We want to get velocity up:
    ip_project_path = Path(args.out_path)
    ip_project_path.mkdir(parents=True, exist_ok=True)

    user_project_path = args.user_project_path

    project_path = ip_project_path.parent # Get project path without /ip folder

    device_name = args.device_name  # Matches Device name from Efinity's Project Editor
    project_name = args.project_name
    output_dirname = str(project_path) # New periphery design will be generated in this directory
    peri_xml_file_path = args.peri_xml_file_path

    main(project_name, device_name, output_dirname, params, ports, gen_name, user_project_path, peri_xml_file_path)
