"""

Copyright (C) 2020-2023 Efinix Inc. All rights reserved.

No portion of this code may be reused, modified or
distributed in any way without the expressed written
consent of Efinix Inc.

Created on August 8, 2020

@author: jhcheah
"""

import copy
import functools
import json
import os
import shutil
import sys
import textwrap
import traceback
from pathlib import Path
from typing import Any, Callable, Dict, Iterable, List, Tuple, Type, Union

from networkx import DiGraph
from PyQt5 import QtCore, QtGui, QtWidgets, uic
from PyQt5.QtCore import Qt

import gui.api_wrapper as wrapper
from blkviewer.widgets import BlockDiagramViewer
from efx_ipmgr.api_v2 import ValidatedParamResult
from efx_ipmgr.production_api.type.component_type import Component
from efx_ipmgr.production_api.type.template_type import (FilesetTemplate,
                                                         ParameterTemplate)
from efx_ipmgr.production_api.type.vlnv_type import VLNV
from efx_ipmgr.production_api.utility.utility import \
    create_new_parameter_template_by_value
from gui.enums import Action
from gui.json_generator import (BaseGenerator, BoolValues, JsonKey,
                                NoteGenerator, QBitStringLineEdit,
                                QBitStringLineEditGenerator,
                                QCheckBoxGenerator, QComboBoxGenerator,
                                QFileDialogGenerator, QLineEditGenerator,
                                QSpinBoxGenerator,
                                QTableCellConnectivityWidgetGenerator,
                                QTableCellWidget, QTableCellWidgetGenerator,
                                ResolveType, Sizes, WidgetType)
from gui.peri_handler import PeriHandler
from gui.Ui_ip_config import Ui_ip_config_window

sys.path.append(os.path.join(os.path.dirname(__file__), '..'))

import efinity_service_pb2
import icons_rc

from common.console import BufferedConsole, MsgTarget
from common.logger import LogLevel
from common.web_util import WebLauncher
from efx_ipmgr.production_api.type_class.to_object import toObject


# Refer: https://www.learnpyqt.com/tutorials/multithreading-pyqt-applications-qthreadpool/
class GenerateIpSignals(QtCore.QObject):
    """
    Signals for generate ip

    Supported signals are:

    finished
        No data

    error
        string of traceback.format_exc()

    result
        Accepts a ResultCode, which is returned after processing.

    """
    finished = QtCore.pyqtSignal()
    error = QtCore.pyqtSignal(str)
    result = QtCore.pyqtSignal(wrapper.ResultCode)

class GenerateIpRunnable(QtCore.QRunnable):

    def __init__(
            self,
            vlnv: VLNV,
            params: List[ParameterTemplate],
            fileset_list: List[FilesetTemplate],
            output_path: str,
            module_name: str,
            device_name: str,
            family_name: str,
            settings_json_path: str,
            project_name: str,
            peri_xml_file_path: str,
            is_del_settings_json: bool = False,
            *args,
            **kwargs,
            ):
        """
        Convert wrapper.generate_ip to a QRunnable.
        All params are the same type as wrapper.generate_ip
        """
        super(GenerateIpRunnable, self).__init__()

        self.vlnv = vlnv
        self.params = params
        self.fileset_list = fileset_list
        self.output_path = output_path
        self.module_name = module_name
        self.device_name = device_name
        self.family_name = family_name
        self.settings_json_path = settings_json_path
        self.is_del_settings_json = is_del_settings_json
        self.project_name = project_name
        self.peri_xml_file_path = peri_xml_file_path
        # Store constructor arguments (re-used for processing)
        self.args = args
        self.kwargs = kwargs
        self.signals = GenerateIpSignals()

        self.setAutoDelete(True)

    @QtCore.pyqtSlot()
    def run(self):
        try:
            gen_result = wrapper.generate_ip(
                vlnv=self.vlnv,
                params=self.params,
                filesets=self.fileset_list,
                out_path=self.output_path,
                gen_name=self.module_name,
                device_name=self.device_name,
                family_name=self.family_name,
                setting_path=self.settings_json_path,
                is_del_settings_json=self.is_del_settings_json,
                skip_put_params=True,
                project_name=self.project_name,
                peri_xml_file_path=self.peri_xml_file_path
            )
        except Exception:
            traceback.print_exc()
            self.signals.error.emit(traceback.format_exc())
        else:
            self.signals.result.emit(gen_result)  # Return the result of the processing
        finally:
            self.signals.finished.emit()  # Done


class IpConfigWindow(QtWidgets.QMainWindow, Ui_ip_config_window):

    def __init__(
            self,
            parent = None,
            vlnv: VLNV = None,
            project_path: str = None,
            project_name: str = None,
            settings_json_path: str = None,
            action: str = "",
            selected_device: str = '',
            selected_device_family: str = '',
            console: Type[BufferedConsole] = None,
            grpc_stub = None,
            peri_xml_file_path: str = ""
        ):
        self._grpc_stub = grpc_stub
        self.console = console
        wrapper.init_console(console)
        self.write_to_console("[IP-CONFIG-START-BEGIN] Initializing IP Config Window...")

        QtWidgets.QMainWindow.__init__(self, parent)

        self.project_path = project_path
        self.project_name = project_name
        self.peri_xml_file_path = peri_xml_file_path
        self.settings_json_path = settings_json_path
        self.action = action
        self.json_ui_template = None

        if project_path is None:
            proj_path = Path(f"{project_path}").absolute()
            if proj_path.exists() == False:
                proj_path.mkdir(parents=True, exist_ok=True)
                Path(f"{proj_path}/ip").mkdir(parents=True, exist_ok=True)
            self.project_path = f"{proj_path}"

        self.selected_device = selected_device
        self.selected_device_family = selected_device_family

        self.ui_dialog = Ui_ip_config_window()
        self.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui_dialog.setupUi(self)
        self.err_icon = QtGui.QIcon()
        self.err_icon.addFile(":/icons/error.svg")

        self.blk_viewer = None
        self.is_blk_viewer_loaded = False
        self.is_window_loaded = False
        self.is_ignore_mainwindow_close = False
        self.is_closing = False

        self.focused_widget_name = None
        self.focused_widget_model = None
        self.cursor_pos = 0
        self.tab_scroll_values = {}

        self.module_name = ""
        self.vlnv = vlnv
        self.arguments = None
        self.threadPool = QtCore.QThreadPool()

        self.ipmsg_dict = {
            "efx_jade_soc" : ('Note: The Sapphire SoC is available in the Efinity software 2021.1 or higher. '
                                'Efinix® will be obsoleting the Jade SoC and replacing it with Sapphire SoC in '
                                'an upcoming Efinity software release. You cannot migrate automatically from the '
                                'Jade SoC to the Sapphire SoC. Therefore, Efinix® recommends that you use the '
                                'Sapphire SoC for all new designs. You can continue to use the Jade SoC with the '
                                'Efinity software 2021.2 or lower. However, the Jade SoC will not be supported in '
                                'future Efinity releases.'),
            "efx_opal_soc" : ('Note: The Sapphire SoC is available in the Efinity software 2021.1 or higher. '
                                'Efinix® will be obsoleting the Opal SoC and replacing it with Sapphire SoC in '
                                'an upcoming Efinity software release. You cannot migrate automatically from the '
                                'Opal SoC to the Sapphire SoC. Therefore, Efinix® recommends that you use the '
                                'Sapphire SoC for all new designs. You can continue to use the Opal SoC with the '
                                'Efinity software 2021.2 or lower. However, the Opal SoC will not be supported in '
                                'future Efinity releases.'),
            "efx_ruby_soc" : ('Note: The Sapphire SoC is available in the Efinity software 2021.1 or higher. '
                                'Efinix® will be obsoleting the Ruby SoC and replacing it with Sapphire SoC in '
                                'an upcoming Efinity software release. You cannot migrate automatically from the '
                                'Ruby SoC to the Sapphire SoC. Therefore, Efinix® recommends that you use the '
                                'Sapphire SoC for all new designs. You can continue to use the Ruby SoC with the '
                                'Efinity software 2021.2 or lower. However, the Ruby SoC will not be supported in '
                                'future Efinity releases.'),
        }

        # TODO: Add testing where if action == New OR Browse, settings_json_path is empty.
        # Else if action ==  RECONFIGURE or IMPORT, settings_json mustn't be empty so we raise error it is.
        self.project_ipm_path = ''
        self.ip_generation_setting = None
        if self.settings_json_path != "" and self.settings_json_path is not None:
            self.ip_generation_setting = wrapper.get_parsed_settings_json(settings_path=self.settings_json_path)
            if self.ip_generation_setting is not None:
                if (self.action == Action.RECONFIGURE.value or self.action == Action.IMPORT.value):
                    self.module_name = self.ip_generation_setting.gen_name

                if vlnv is None:
                    self.vlnv: VLNV = self.ip_generation_setting.vlnv

                self.project_ipm_path = Path(self.project_path, 'ip', self.module_name, 'ipm')

        if self.vlnv is None:
            raise Exception("VLNV is not found because VLNV is None")

        self.load_pickle_success = False
        if self.ip_generation_setting and self.action == Action.IMPORT.value:
            # NOTE: If user is trying to import IP through action = Import, the settings.json file must be of a valid version.
            is_migrated_process_complete = self.migrate_imported_ip(settings_json_path)
            if not is_migrated_process_complete:
                self.close()
                return

        else:
            # When ip_generation_setting exists, project_ipm_path = Path, else project_ipm_path = ''
            self.load_pickle_success = wrapper.refresh_ip_sources(
                opt=None, vlnv=self.vlnv, project_ipm_path=self.project_ipm_path,device=self.selected_device, family=self.selected_device_family)

        IpConfigWindow.vlnv = self.vlnv
        loaded_component, loaded_graph = wrapper.get_model(self.vlnv)
        self.temp_component: Component = loaded_component
        self.temp_graph: DiGraph = loaded_graph

        if self.temp_component.isHardIP == '1\'b1':
            if self.vlnv.vendor == 'efinixinc.com' and self.vlnv.library == 'soc' and self.vlnv.name == 'efx_hard_soc':
                self.module_name = 'EfxSapphireHpSoc_slb'
                self.ui_dialog.le_inst.setText(self.module_name)
                self.ui_dialog.le_inst.setStyleSheet("color: #515151")
                self.ui_dialog.le_inst.setEnabled(False)

        self.abs_path_to_html_doc = wrapper.get_ip_documentation(vlnv=self.vlnv)
        self.web_launcher = WebLauncher(self.console)

        self.vlnv_dict = self.vlnv.to_dict()
        self.vlnv_title = f"{self.vlnv.name} {self.vlnv.version}"

        selected_file_suffix = None
        self.last_file_dialog_selection: Tuple[str, Union[str, None]] = (self.project_path, selected_file_suffix)
        self.params = None
        self.default_params = None

        self.setup_params(self.load_pickle_success, have_settings_json=bool(self.ip_generation_setting is not None))

        self.gui_fileset_deliverables = wrapper.get_ip_filesets(vlnv=self.vlnv,
                                                    temp_component_template=toObject(self.temp_component))
        self.fileset_to_isGenerate_dict = {deliverable.display_name:True for deliverable in self.gui_fileset_deliverables}
        if self.ip_generation_setting:
            '''
            Example using Ti180:
            Expected behavior for reconfigure ip:
            Ti180 = True
            |-----------------------------------------------------
            |settings.json         |          deliverable page   |
            |----------------------------------------------------|
            |no Ti180              |           show as unticked  |
            |got Ti180             |           show as ticked    |
            ------------------------------------------------------

            Ti180 = False
            |-----------------------------------------------------
            |settings.json         |          deliverable page   |
            |----------------------------------------------------|
            |no Ti180              |           Not show          |
            |got Ti180             |           Not show          |
            ------------------------------------------------------
            '''
            self.fileset_to_isGenerate_dict = {}
            for deliverable in self.gui_fileset_deliverables:
                if deliverable.fileset_template.get_ip_fileset_name() in self.ip_generation_setting.output_filesets:
                    self.fileset_to_isGenerate_dict[deliverable.display_name]=True
                else:
                    self.fileset_to_isGenerate_dict[deliverable.display_name]=False

        self.tab_widget = None
        self.setup_window()

        if self.action == Action.RECONFIGURE.value and self.module_name != "":
            self.ui_dialog.le_inst.setText(self.module_name)
            self.ui_dialog.le_inst.setStyleSheet("color: #515151")
            self.ui_dialog.le_inst.setEnabled(False)
        elif self.action == Action.IMPORT.value and self.module_name !="":
            self.ui_dialog.le_inst.setText(self.module_name)

        if self.action == Action.BROWSE.value:
            self.ui_dialog.pb_generate.setEnabled(False)

        # Make all connections
        self.module_name_timer = QtCore.QTimer(self)
        self.module_name_timer.timeout.connect(self.update_block_diagram)
        self.ui_dialog.le_inst.textChanged.connect(self.set_module_name)
        self.ui_dialog.pb_help.clicked.connect(self.open_html_doc)
        self.ui_dialog.pb_generate.clicked.connect(self.show_confirmation_box)
        self.ui_dialog.pb_close.clicked.connect(self.close)
        self.installEventFilter(self)


        self.write_to_console("[IP-CONFIG-START-DONE] IP Config Window initialized")

    def write_to_console(self, msg, log_level = LogLevel.INFO, dest = MsgTarget.CONSOLE_AND_LOG):
        extra = self.console.generate_extra_dict(__name__)
        self.console.write(msg=msg, log_level=log_level, dest=dest, extra=extra)

    def eventFilter(self, obj, event):
        if (event.type() == QtCore.QEvent.KeyPress):
            if event.key() == QtCore.Qt.Key_Return:
                if isinstance(obj, QtWidgets.QMainWindow):
                    self.show_confirmation_box()
                elif isinstance(obj, QTableCellWidget):
                    QTableCellWidgetGenerator.handle_key_return(obj, event)
                elif isinstance(obj, QBitStringLineEdit):
                    QBitStringLineEditGenerator.handle_key_return(obj, event)
                elif isinstance(obj, QtWidgets.QLineEdit):
                    QLineEditGenerator.handle_key_return(obj, event)
                elif isinstance(obj, QtWidgets.QComboBox):
                    QComboBoxGenerator.handle_key_return(obj, event)
                elif isinstance(obj, QtWidgets.QSpinBox):
                    QSpinBoxGenerator.handle_key_return(obj, event)
                elif isinstance(obj, QtWidgets.QCheckBox):
                    QCheckBoxGenerator.handle_key_return(obj, event)
                elif isinstance(obj, QtWidgets.QFileDialog):
                    QFileDialogGenerator.handle_key_return(obj, event)
                return True

        if event.type() == QtCore.QEvent.Wheel:
            # Disable mouse wheel operation on these widgets. It affects the whole editor scrolling
            # since the widgets scrolling takes precedent.
            if isinstance(obj, QtWidgets.QComboBox) or isinstance(obj, QtWidgets.QSpinBox) or isinstance(obj, QtWidgets.QDoubleSpinBox):
                event.ignore()
                return True

        return super(IpConfigWindow, self).eventFilter(obj, event)

    def closeEvent(self, event):
        """Reimplement closeEvent"""
        if self.is_ignore_mainwindow_close == True or self.is_closing == True:
            event.ignore()
        else:
            # Close signal is sometimes fired twice
            self.is_closing = True

            # Clean up console
            self.write_to_console("[IP-CONFIG-STOP] IP Config Window has closed")
            self.console.stop()

            # Stop webview
            if self.blk_viewer is not None:
                self.blk_viewer.close()

    def migrate_imported_ip(self, settings_json_path: Path) -> bool:
        is_migrate_process_completed = True

        list_of_ips_matching_vlnv = wrapper.get_list_of_ips(
            vendor=self.vlnv.vendor, library=self.vlnv.library, name=self.vlnv.name,
            device_name=self.selected_device, family_name=self.selected_device_family
        )
        latest_vlnv: VLNV = list_of_ips_matching_vlnv[0] # Only one IP should ever match

        settings_json_vlnv_version = VLNV.get_int_version_from_str(self.vlnv.version)
        latest_vlnv_version = VLNV.get_int_version_from_str(latest_vlnv.version)

        self.load_pickle_success = wrapper.refresh_ip_sources(
                opt=None, vlnv=latest_vlnv, project_ipm_path=self.project_ipm_path,
                device=self.selected_device, family=self.selected_device_family)

        if settings_json_vlnv_version >= latest_vlnv_version:
            # Only handle migrate ip if settings_json_vlnv_version <= latest_vlnv_version
            return is_migrate_process_completed

        # Do migration step

        # TODO: Consider using progress dialog?
        # progress_dialog = QtWidgets.QProgressDialog(self)
        # progress_dialog.setWindowTitle("Warning")
        # progress_dialog.setLabelText("The imported IP is outdated. Attempting IP migration...")
        # progress_dialog.setMinimum(0)
        # progress_dialog.setMaximum(0)
        # progress_dialog.setCancelButton(None)
        # progress_dialog.setModal(True)
        # progress_dialog.setWindowModality(Qt.WindowModal)
        # progress_dialog.setWindowFlags(Qt.Window | Qt.WindowTitleHint | Qt.CustomizeWindowHint | Qt.WindowStaysOnTopHint)
        # progress_dialog.setAutoClose(True)

        button_clicked = QtWidgets.QMessageBox.warning(None, "Warning", "The imported IP is outdated. Attempting IP migration...", QtWidgets.QMessageBox.Ok | QtWidgets.QMessageBox.Cancel)
        if button_clicked != QtWidgets.QMessageBox.Ok:
            return (not is_migrate_process_completed)

        QtWidgets.QApplication.setOverrideCursor(Qt.WaitCursor)
        result = wrapper.migrate_ip(
            from_vlnv=self.vlnv, to_vlnv=latest_vlnv, setting_path=settings_json_path)

        if isinstance(result, str):
            QtWidgets.QApplication.restoreOverrideCursor()
            QtWidgets.QMessageBox.critical(self, "Error", f"IP migration failed: {result}")
            return (not is_migrate_process_completed)

        migrated_params, migrated_filesets, migrated_comp, migrated_graph = result
        self.vlnv = latest_vlnv
        wrapper.update_model(self.vlnv, migrated_comp, migrated_graph)

        self.ip_generation_setting.set_params(migrated_params)
        self.ip_generation_setting.set_output_info(migrated_filesets)

        QtWidgets.QApplication.restoreOverrideCursor()
        return is_migrate_process_completed

    def setup_params(self, load_pickle_success = False, have_settings_json = False):
        self.write_to_console("[IP-CONFIG-SETUP-PARAMS-BEGIN] Setting up params...")
        self.validated_param_map: Dict[str, ValidatedParamResult] = {}

        # initialize param_map
        self.params = wrapper.get_ip_params(vlnv=self.vlnv, device_name=self.selected_device, family_name=self.selected_device_family)
        self.default_params = copy.deepcopy(self.params)
        for p_temp in self.params:
            if( p_temp.name not in self.validated_param_map ):
                self.validated_param_map[p_temp.name] = ValidatedParamResult(result=True, error_tuple_msg=({}, {}, {}), param_template=p_temp)

        if load_pickle_success == False:
            self.default_params = copy.deepcopy(self.params)
            self.default_validated_param_map = copy.deepcopy(self.validated_param_map)
            self.default_component = copy.deepcopy(self.temp_component)
            self.default_graph = copy.deepcopy(self.temp_graph)

        if load_pickle_success == False and have_settings_json:
            self.init_from_settings()

        if self.temp_component.isHardIP == '1\'b1':
            # NOTE: We do not read values from peri.xml, for now.
            # self.setup_hard_ip_params()
            pass

        self.write_to_console("[IP-CONFIG-SETUP-PARAMS-DONE] Param setup done.")

    def init_from_settings(self):
        # Initalize from param
        if self.ip_generation_setting is not None:
            self.params: List[ParameterTemplate] = []
            for key in self.ip_generation_setting.params:
                old_validated_param_result = self.validated_param_map[key]
                user_value = self.ip_generation_setting.params[key]

                new_param_template = create_new_parameter_template_by_value(old_validated_param_result.param_template, user_value)
                old_validated_param_result.param_template = new_param_template
                self.params.append(new_param_template)

            validated_params_result, parameter_temp_list, component, graph = wrapper.validate_params(vlnv=self.vlnv, params_to_chk=self.params)
            self.temp_component = component
            self.temp_graph = graph
            self.params = parameter_temp_list
            if not validated_params_result.result:
                exception_dict, error_dict, warning_dict = validated_params_result.error_tuple_msg
                for parameter_template in parameter_temp_list:
                    param_name = parameter_template.name

                    is_no_error_or_exception = True
                    new_exception_dict = {}
                    new_error_dict = {}
                    new_warning_dict = {}

                    if param_name in exception_dict:
                        is_no_error_or_exception = False
                        new_exception_dict = {param_name: exception_dict[param_name]}

                    if param_name in error_dict:
                        is_no_error_or_exception = False
                        new_error_dict = {param_name: error_dict[param_name]}

                    if param_name in warning_dict:
                        new_warning_dict = {param_name: warning_dict[param_name]}

                    self.validated_param_map[param_name] = ValidatedParamResult(
                        result=is_no_error_or_exception,
                        error_tuple_msg=(new_exception_dict, new_error_dict, new_warning_dict),
                        param_template=parameter_template
                    )

            else:
                for parameter_template in parameter_temp_list:
                    validated_params_result = self.validated_param_map[parameter_template.name]
                    validated_params_result.param_template = parameter_template

            #print(json.dumps(list(p.to_dict() for p in self.params), indent=4, sort_keys=True))

    def get_hard_ip_property_map(self):
        """
        NOTE: Use IP Packager to generate the PT-IPM property map functionality. Hard code for now.
        This PT-IPM Property map only contains PLL for now.
        """
        hard_ip_property_map = wrapper.get_ipm_pt_map_json(vlnv=self.vlnv)
        return hard_ip_property_map

    def setup_hard_ip_params(self):
        '''
        In this setup_hard_ip_params, I only attempt to load PLL from peri.xml, as they contain the only values that I should
        get from PT. The rest of the parameters from IP Config Window.
        '''
        self.hard_ip_map = {}

        if self.temp_component.isHardIP == '1\'b1':
            self.hard_ip_map = self.get_hard_ip_property_map()
            handle_hard_ip = PeriHandler(self.project_name, self.hard_ip_map, self.console)
            pll_dict = handle_hard_ip.get_properties_from_peri_file('PLL')
            # print(json.dumps(pll_dict, indent=4))

            # CASE 1: Peri.xml is empty or doesn't exist yet.
            # CASE 2: Peri.xml already exists but it only contains user-defined PLL and no block created from hard IP
            if len(pll_dict) == 0:
                return

            # CASE 3: When user opens IP Manager, and peri.xml already exists, and it contains PT Params created from hard IP.
            for pll_name in pll_dict:
                pll_prop_dict = pll_dict[pll_name]

                for pll_prop_name in pll_prop_dict:
                    pll_val = pll_prop_dict[pll_prop_name]
                    if pll_val is None:
                        continue # Use default value

                    curr_validated_param_result = self.validated_param_map[pll_prop_name]
                    template = curr_validated_param_result.param_template
                    if "PHASE" in pll_prop_name:
                        pll_val_float = float(pll_val)
                        pll_val_int = int(pll_val_float)
                        pll_val = f"{pll_val_int}"

                    if template.datatype == 'string':
                        pll_val = f"\"{pll_val}\""

                    new_param_template = create_new_parameter_template_by_value(curr_validated_param_result.param_template, pll_val)
                    curr_validated_param_result.param_template = new_param_template

            self.params = [validated_param_result.param_template for validated_param_result in self.validated_param_map.values()]

            validated_params_result, parameter_temp_list, component, graph = wrapper.validate_params(vlnv=self.vlnv, params_to_chk=self.params)
            self.temp_component = component
            self.temp_graph = graph
            self.params = parameter_temp_list
            if not validated_params_result.result:
                exception_dict, error_dict, warning_dict = validated_params_result.error_tuple_msg
                for parameter_template in parameter_temp_list:
                    param_name = parameter_template.name

                    is_no_error_or_exception = True
                    new_exception_dict = {}
                    new_error_dict = {}
                    new_warning_dict = {}

                    if param_name in exception_dict:
                        is_no_error_or_exception = False
                        new_exception_dict = {param_name: exception_dict[param_name]}

                    if param_name in error_dict:
                        is_no_error_or_exception = False
                        new_error_dict = {param_name: error_dict[param_name]}

                    if param_name in warning_dict:
                        new_warning_dict = {param_name: warning_dict[param_name]}

                    self.validated_param_map[param_name] = ValidatedParamResult(
                        result=is_no_error_or_exception,
                        error_tuple_msg=(new_exception_dict, new_error_dict, new_warning_dict),
                        param_template=parameter_template
                    )

            else:
                for parameter_template in parameter_temp_list:
                    validated_params_result = self.validated_param_map[parameter_template.name]
                    validated_params_result.param_template = parameter_template

    def setup_window(self, is_reset_template = False):
        if (self.json_ui_template is None) or (is_reset_template):
            self.json_ui_template = wrapper.get_ip_template(vlnv=self.vlnv)

        if self.blk_viewer is None:
            self.write_to_console("[IP-CONFIG-SETUP-BLOCK-DIAGRAM-BEGIN] Setting up block diagram viewer...")
            self.blk_viewer = BlockDiagramViewer(Sizes.blk_width, Sizes.blk_height, self, self.vlnv_title, context_menu=False)
            self.blk_viewer.setFrameShape(QtWidgets.QFrame.StyledPanel)
            self.blk_viewer.setAutoFillBackground(True)
            self.blk_viewer.setBackgroundRole(QtGui.QPalette.Light)
            self.blk_viewer.onInitFinished.connect(lambda: self.generate_diagram_on_load(True, False))
            self.ui_dialog.blk_vlayout.addWidget(self.blk_viewer)
            self.write_to_console("[IP-CONFIG-SETUP-BLOCK-DIAGRAM-DONE] Block diagram viewer set up.")

        # Initialize tab widget
        self.tab_widget = QtWidgets.QTabWidget()
        self.tab_widget.setMinimumWidth(Sizes.min_tab_widget_width)
        self.tab_widget.setMinimumHeight(Sizes.tab_widget_height)
        self.tab_widget.resize(Sizes.tab_widget_width, Sizes.tab_widget_height)
        self.tab_widget.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Expanding)
        self.ui_dialog.tab_vlayout.addWidget(self.tab_widget)
        self.generate_tabs_from_json()
        self.generate_diagram_on_load(False, True)

        # Setting up additional message for IP. This message will appear under the "module" name.
        if(self.vlnv.name not in self.ipmsg_dict):
            self.ui_dialog.lbl_ipmsg.setVisible(False)
            self.ui_dialog.ip_msg_hlayout.setContentsMargins(0, 0, 0, 0)
        else:
            self.ui_dialog.lbl_ipmsg.setText(self.ipmsg_dict[self.vlnv.name].strip())


    def reset_window(self, is_reset_template = False, is_reset_default = False):
        self.get_latest_slider_position()
        curr_index = self.tab_widget.currentIndex()
        self.ui_dialog.tab_vlayout.removeWidget(self.tab_widget)
        self.tab_widget.deleteLater()
        self.tab_widget = None

        if is_reset_default:
            QtWidgets.QApplication.setOverrideCursor(Qt.WaitCursor)
            if not self.load_pickle_success:
                self.params = copy.deepcopy(self.default_params)
                self.validated_param_map = copy.deepcopy(self.default_validated_param_map)
                self.temp_component = copy.deepcopy(self.default_component)
                self.temp_graph = copy.deepcopy(self.default_graph)
                wrapper.update_model(self.vlnv, self.temp_component, self.temp_graph)

            else:
                wrapper.refresh_ip_sources(opt=None, vlnv=self.vlnv, project_ipm_path=self.project_ipm_path,
                                           device=self.selected_device, family=self.selected_device_family)

                loaded_component, loaded_graph = wrapper.get_model(self.vlnv)
                self.temp_component: Component = loaded_component
                self.temp_graph: DiGraph = loaded_graph

                self.default_component = copy.deepcopy(self.temp_component)
                self.default_graph = copy.deepcopy(self.temp_graph)

                self.validated_param_map: Dict[str, ValidatedParamResult] = {}
                self.params = wrapper.get_ip_params(vlnv=self.vlnv,
                                                    device_name=self.selected_device,
                                                    family_name=self.selected_device_family)
                self.default_params = copy.deepcopy(self.params)
                for p_temp in self.params:
                    if( p_temp.name not in self.validated_param_map ):
                        self.validated_param_map[p_temp.name] = ValidatedParamResult(result=True,
                                                                                     error_tuple_msg=({}, {}, {}),
                                                                                     param_template=p_temp)
                self.default_validated_param_map = copy.deepcopy(self.validated_param_map)

                # Only run once to get default value
                self.load_pickle_success = False

            QtWidgets.QApplication.restoreOverrideCursor()

        # YQ: update deliverable with expression
        self.gui_fileset_deliverables = wrapper.get_ip_filesets(vlnv=self.vlnv,
                                                    temp_component_template=toObject(self.temp_component))

        old_fileset_to_isGenerate_dict = self.fileset_to_isGenerate_dict

        self.fileset_to_isGenerate_dict = {}
        for deliverable in self.gui_fileset_deliverables:
            display_name = deliverable.display_name
            if deliverable.display_name in old_fileset_to_isGenerate_dict:
                self.fileset_to_isGenerate_dict[display_name] = old_fileset_to_isGenerate_dict[display_name]
            else:
                self.fileset_to_isGenerate_dict[display_name] = True

        self.setup_window(is_reset_template)
        self.tab_widget.setCurrentIndex(curr_index)


    def set_module_name(self, text):

        self.module_name = text
        self.module_name_timer.start(400)

    def update_block_diagram(self):
        self.generate_block_diagram()
        self.module_name_timer.stop()

    def open_html_doc(self):
        if self.abs_path_to_html_doc is not None:
            self.web_launcher.open_in_browser(self.abs_path_to_html_doc)

    def generate_ip(self):
        self.write_to_console("[IP-CONFIG-GENERATE-BEGIN] Setup IP generation")
        self.ui_dialog.pb_generate.setEnabled(False)
        self.ui_dialog.pb_close.setEnabled(False)

        self.write_to_console(f"[IP-CONFIG-GENERATE] Project_path: {self.project_path}/ip")
        dlvrb_names_msg = ""
        fileset_list: List[FilesetTemplate] = []
        for dlvrb in self.gui_fileset_deliverables:
            is_optional = dlvrb.is_optional
            is_include_in_list = self.fileset_to_isGenerate_dict[dlvrb.display_name]
            dlvrb_names_msg += f"{dlvrb.display_name}\n"
            if is_optional == False or is_include_in_list == True:
                fileset_list.append(dlvrb.fileset_template)
        self.write_to_console(f"[IP-CONFIG-GENERATE] Chosen deliverables: \n{dlvrb_names_msg}")

        QtWidgets.QApplication.setOverrideCursor(Qt.WaitCursor)

        # for p_obj in self.params:
        #     if( hasattr(p_obj, "data_type")
        #         and p_obj.data_type == DataType.STRING.value
        #         and p_obj.param_value.value != ""):
        #         # HACK: For IPM-206 and IPM-201. Hitting a ValueError issue that will require
        #         # a lot of changes to IP_Component.xml for a proper fix (if expecting int,
        #         # then we should not set type="string" in ip_component.xml)

        #         # Add if statement to prevent adding double quote in connectivity page
        #         if type(p_obj.param_value.value) != str:
        #             p_obj.param_value.value = f"\"{p_obj.param_value.value}\""
        #     elif( hasattr(p_obj, "data_type")
        #         and hasattr(p_obj, "row_label")
        #         and p_obj.data_type == DataType.HEXADECIMAL.value
        #         and p_obj.param_value.value != ""):
        #         hexadecimal_padding_size = 10
        #         hex_representation = QTableCellWidgetGenerator.convert_to_base(
        #             p_obj.data_type, p_obj.param_value.value, hexadecimal_padding_size)
        #         p_obj.param_value.value = f"\"{hex_representation}\""

        genIpRunnable = GenerateIpRunnable(
            self.vlnv,
            self.params,
            fileset_list,
            f"{self.project_path}",
            self.module_name,
            self.selected_device,
            self.selected_device_family,
            self.settings_json_path,
            project_name=self.project_name,
            peri_xml_file_path=self.peri_xml_file_path
        )

        # Disable to test api_wrapper and api. Cannot debug child process
        genIpRunnable.signals.result.connect(self.post_generate_ip)
        genIpRunnable.signals.error.connect(self.post_generate_error)
        genIpRunnable.signals.finished.connect(self.post_generate_finish)

        self.write_to_console("[IP-CONFIG-GENERATE] Have thread pool start GenerateIpRunnable...")
        self.threadPool.globalInstance().start(genIpRunnable)

        # Enable to test api wrapper and api.
        # gen_result = wrapper.generate_ip(
        #     vlnv=self.vlnv,
        #     params=self.params,
        #     filesets=fileset_list,
        #     out_path=f"{self.project_path}",
        #     gen_name=self.module_name,
        #     device_name=self.selected_device,
        #     family_name=self.selected_device_family,
        #     setting_path=self.settings_json_path,
        #     is_del_settings_json=False,
        #     skip_put_params=True,
        #     project_name=self.project_name,
        # )

        # self.post_generate_ip(gen_result)

    def post_generate_finish(self):
        self.write_to_console("[IP-CONFIG-GENERATE-STOP] IP Generation thread finished run", dest=MsgTarget.LOG_ONLY)

    def post_generate_error(self, formatted_traceback):
        # Written in case generate_ip fails before try and except section is even hit
        self.console.wait_until_all_msg_processed()
        traceback_arr = formatted_traceback.split('\n')
        self.write_to_console("[IP-CONFIG-GENERATE-ERROR] Exception caught from GenerateIpRunnable..")
        for line in traceback_arr:
            self.write_to_console(line, log_level=LogLevel.EXCEPTION)
        QtWidgets.QApplication.restoreOverrideCursor()

        QtWidgets.QMessageBox.critical(self, "Generation Failed", "ERROR: IP generation has failed.")

        self.update() # Trigger a gui update to have summary page be refreshed.

        self.ui_dialog.pb_generate.setEnabled(True)
        self.ui_dialog.pb_close.setEnabled(True)

        self.is_ignore_mainwindow_close = False

        self.write_to_console("[IP-CONFIG-GENERATE-DONE] IP generation done")

    def post_generate_ip(self, result: wrapper.ResultCode):
        self.console.wait_until_all_msg_processed()
        self.write_to_console("[IP-CONFIG-GENERATE-DONE] Result returned from GenerateIpRunnable..")
        QtWidgets.QApplication.restoreOverrideCursor()

        resultCode = wrapper.ResultCode(result)
        self.write_to_console(f"[IP-CONFIG-GENERATE-DONE] Generate status: {resultCode.msg()}")

        if resultCode.is_pass():
            QtWidgets.QMessageBox.information(None, "Generation Success", "The IP has been successfully generated.")
            if self._grpc_stub and (self.action == Action.NEW.value or self.action == Action.IMPORT.value):
                response = self._grpc_stub.AddIP(
                    efinity_service_pb2.AddIPRequest(
                    settings_json_path=f"{self.project_path}/ip/{self.module_name}/settings.json"))
        else:
            if self.action == Action.NEW.value or self.action == Action.IMPORT.value:
                created_folder_path = Path(self.project_path, 'ip', self.module_name)
                if created_folder_path.exists():
                    shutil.rmtree(Path(self.project_path, 'ip', self.module_name))

            QtWidgets.QMessageBox.critical(self, "Generation Failed", "ERROR: IP generation has failed.")
            self.update() # Trigger a gui update to have summary page be refreshed.

        self.ui_dialog.pb_generate.setEnabled(True)
        self.ui_dialog.pb_close.setEnabled(True)

        self.is_ignore_mainwindow_close = False


        if resultCode.is_pass():
            self.close()

    def check_errors_on_generate(self) -> Tuple[bool, str]:
        params_with_errors_list = []
        are_there_any_errors = False
        invalid_params_str = ""

        for param_name, validated_param_result in self.validated_param_map.items():
            if validated_param_result.result == False:
                params_with_errors_list.append(validated_param_result.param_template.displayName)
                are_there_any_errors = True

        if are_there_any_errors:
            invalid_params_str = "\n".join(params_with_errors_list)

        return (are_there_any_errors, invalid_params_str)

    def on_click_generate(self, dialog, ok_button, cancel_button):
        page = self.tab_widget.findChild(QtWidgets.QWidget, "Summary")
        self.tab_widget.setCurrentWidget(page)

        ok_button.setDisabled(True)
        cancel_button.setDisabled(True)
        dialog.setWindowFlags(dialog.windowFlags() | Qt.CustomizeWindowHint)
        dialog.setWindowFlags(dialog.windowFlags() & ~Qt.WindowCloseButtonHint)

        # Use dialog.exec to prevent side effects from main window event loop
        dialog.exec()

        self.generate_ip()

    def create_confirmation_box(self) -> QtWidgets.QDialog:
        dialog = QtWidgets.QDialog()
        dialog.setWindowTitle("Review generation configuration")
        dialog.resize(480, 360)
        generate_conf_vlayout = QtWidgets.QVBoxLayout()
        generate_conf_vlayout.setContentsMargins(9, 9, 9, 9)

        # Confirm Label -----------------------------------
        count = 0
        text = ''
        if self.temp_component.isHardIP == '1\'b1' and self.action == Action.RECONFIGURE.value:
            text += '<b><i>***WARNING:***</i>'
            text += '<br>  Reconfiguring this IP will reset all'
            text += '<br>  PLL/DDR/GPIO/JTAG/QCRV32 properties of'
            text += '<br>  IPM-Generated Interface Designer blocks. </b><br><br>'
        text += 'Selected deliverables:<br><br>'
        warning_targets = set()
        for display_name in self.fileset_to_isGenerate_dict:
            if self.fileset_to_isGenerate_dict[display_name] == True:
                count += 1
                text += f'{count}) {display_name}<br>'
                if 'example' in display_name.lower():
                    warning_targets.add('example design')
                if 'testbench' in display_name.lower():
                    warning_targets.add('testbench')

        # Remarks: Tell user that only example project / testbench are tested
        if self.params != self.default_params and len(warning_targets) != 0:
            target_str = '/'.join(warning_targets)
            text +=  '<br><b><i>***Important:***</i>'
            text += f'<br>  The {target_str} was only designed to'
            text +=  '<br>  work with default IP parameters.</b>'

        text += '<br><br>Generate IP with these deliverables?<br>'

        lbl_confirmation = QtWidgets.QLabel(text)

        generate_conf_vlayout.addWidget(lbl_confirmation)

        conf_spacer = QtWidgets.QSpacerItem(20, 20, QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Expanding)
        generate_conf_vlayout.addSpacerItem(conf_spacer)

        # Dialog button -----------------------------------
        dialog_h_layout = QtWidgets.QHBoxLayout()
        ok_button = QtWidgets.QPushButton("Generate")

        # Do not allow the ok button to close the dialog.
        ok_button.setDefault(False)

        def on_finish():
            self.is_ignore_mainwindow_close = False
        dialog.finished.connect(on_finish)

        cancel_button = QtWidgets.QPushButton("Cancel")
        cancel_button.clicked.connect(dialog.reject)

        ok_button.clicked.connect(
            lambda: self.on_click_generate(dialog, ok_button, cancel_button)
        )
        dialog_h_layout.addWidget(ok_button)
        dialog_h_layout.addWidget(cancel_button)

        generate_conf_vlayout.addLayout(dialog_h_layout)
        dialog.setLayout(generate_conf_vlayout)

        return dialog

    def show_confirmation_box(self):
        if self.action != Action.BROWSE.value:
            result = wrapper.check_module_name(self.module_name)
            is_module_name_exist = self.is_module_name_exists(self.module_name, self.project_path)
            keyword_result = wrapper.check_with_other_language_keyword(self.module_name)
            are_there_errors_in_params, invalid_params_str = self.check_errors_on_generate()
            is_confirmation_pop_up_enabled = self.ui_dialog.checkBox_cnfrm.isChecked()

            if result[0] == False:
                QtWidgets.QMessageBox.critical(self, "Error", result[1])

            elif is_module_name_exist:
                QtWidgets.QMessageBox.critical(self,
                    "Error",
                    f"The module name {self.module_name} already exists under the current \nproject path: {self.project_path}ip")

            elif are_there_errors_in_params:
                QtWidgets.QMessageBox.critical(self,
                    "Error: Invalid Parameters!",
                    f"The following parameters contain errors:\n\n{invalid_params_str}\n\nPlease fix them first.\n\nGeneration disallowed.")

            elif is_confirmation_pop_up_enabled:
                if keyword_result[0] == False:
                    QtWidgets.QMessageBox.warning(None, "Warning", keyword_result[1])
                self.is_ignore_mainwindow_close = True

                # Use dialog.exec to match the dialog.exec in on_click_generate
                dialog = self.create_confirmation_box()
                dialog.exec()

            else:
                # If checkBox_cnfrm is disabled. Then we skip confirmation box
                self.is_ignore_mainwindow_close = True
                page = self.tab_widget.findChild(QtWidgets.QWidget, "Summary")
                self.tab_widget.setCurrentWidget(page)

                self.generate_ip()

    def is_module_name_exists(self, module_name, project_path):
        if self.action != Action.RECONFIGURE.value and self.action != Action.BROWSE.value:
            project_ip_folder = Path(f"{project_path}/ip")
            ip_folder_list = [Path(f).name for f in os.scandir(project_ip_folder) if f.is_dir()]
            return module_name in ip_folder_list
        return False

    def store_focused_widget_details(self, generator_class):
        self.focused_widget_name = None
        self.cursor_pos = 0

        widget = generator_class.get_generated_display_widget()
        self.focused_widget_name = widget.objectName()
        if isinstance(generator_class, QSpinBoxGenerator):
            self.cursor_pos = widget.lineEdit().cursorPosition()
        elif isinstance(generator_class, QLineEditGenerator):
            self.cursor_pos = widget.cursorPosition()
        else:
            # NOTE: This function will create infinite loop for QTableCellWidget and QBitStringLineEdit
            # due to logic on focus out. Disable this functionality when there is no cursor position
            self.focused_widget_name = None
            self.focused_widget_model = None
            self.cursor_pos = 0

    def update_focused_widget(self, widget_model):
        widget = widget_model.get_generated_display_widget()
        if self.focused_widget_name is not None and self.focused_widget_name == widget.objectName():
            self.focused_widget_model = widget_model

    def refocus_widget(self):
        if self.focused_widget_model is None:
            return

        widget = self.focused_widget_model.get_generated_display_widget()
        widget.setFocus()

        if isinstance(self.focused_widget_model, QLineEditGenerator):
            widget.setCursorPosition(self.cursor_pos)
        elif isinstance(self.focused_widget_model, QSpinBoxGenerator):
            widget.lineEdit().setCursorPosition(self.cursor_pos)

    def send_to_server(self,
                       generator_class: Type[BaseGenerator],
                       param_template: ParameterTemplate,
                       col: str,
                       changed_value: Any
                       ):

        new_value = generator_class.on_change(param_template, changed_value, col)

        self.store_focused_widget_details(generator_class)

        validated_param_result, new_component, new_graph = wrapper.validate_param(
            vlnv=self.vlnv,
            component=self.temp_component,
            graph=self.temp_graph,
            param_name=param_template.name,
            updated_param_value=new_value
            )

        new_component_template = toObject(new_component)
        # update all the other param to get updated result.
        for key in new_component.parameters:
            self.validated_param_map[key] = ValidatedParamResult(result=True, error_tuple_msg=({}, {}, {}), param_template=new_component_template.parameters[key])
            # self.validated_param_map[key] = new_component_template.parameters[key]

        if validated_param_result.param_template is not None:
            self.validated_param_map[param_template.name] = validated_param_result
            self.params = [validated_param_result.param_template for validated_param_result in self.validated_param_map.values()]
        else:
            current_result = self.validated_param_map[param_template.name]
            current_result.result = validated_param_result.result
            current_result.param_template = create_new_parameter_template_by_value(param_template, changed_value)
            current_result.error_tuple_msg = validated_param_result.error_tuple_msg
            current_result.side_effect_list = validated_param_result.side_effect_list

        # Map error message to param
        exception_dict, error_dict, _ = validated_param_result.error_tuple_msg

        # Handle error_dict ---------------------------------------
        temp_dict = {key:{} for key in self.validated_param_map.keys()}

        for err_msg in error_dict:
            for param_name in self.validated_param_map.keys():
                if ("param " + param_name + " ") in err_msg:
                    temp_dict[param_name][err_msg] = error_dict[err_msg]

        # Replace error_dict content for each param
        for key in self.validated_param_map.keys():
            current_validated_param_result = self.validated_param_map[key]
            curr_exception_dict, _, some_dict = current_validated_param_result.error_tuple_msg
            current_validated_param_result.error_tuple_msg = (curr_exception_dict, temp_dict[key], some_dict)
            if len(temp_dict[key]):
                current_validated_param_result.result = False
            else:
                current_validated_param_result.result = True
            self.validated_param_map[key] = current_validated_param_result

         # Handle exception_dict -----------------------------------
        if len(exception_dict) > 0:
            current_validated_param_result = self.validated_param_map[param_template.name]
            current_validated_param_result.result = False
            self.validated_param_map[param_template.name] = current_validated_param_result

        # NOTE: Validate_params will return the same component and graph passed in if error is found. No chance of NONE.
        self.temp_component = new_component
        self.temp_graph = new_graph

        self.reset_window()

    def pretty_print_json(self, json_doc):
        print(json.dumps(json_doc, indent=4, sort_keys=True))

    def get_latest_slider_position(self):
        for key in self.tab_scroll_values.keys():
            child_dict = self.tab_scroll_values[key]
            vert_scrollBar = child_dict["scroll_bar"]
            child_dict["value"] = vert_scrollBar.value()

    def set_latest_slider_values(self, tab_name, minimum, maximum):
        for key in self.tab_scroll_values.keys():
            child_dict = self.tab_scroll_values[key]
            vert_scrollBar = child_dict["scroll_bar"]
            vert_scrollBar.setRange(minimum, maximum)
            vert_scrollBar.setValue(child_dict["value"])

    def on_click_reset_params(self):
        self.reset_window(is_reset_default=True)
        self.generate_block_diagram()

    def generate_reset_default_option_bar_widget(self):
        option_bar_spacer = QtWidgets.QSpacerItem(20, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)
        option_bar_push_button = QtWidgets.QPushButton("Reset default settings")
        option_bar_push_button.clicked.connect(self.on_click_reset_params)
        option_bar_push_button.setMinimumWidth(150)
        option_bar_push_button.setMaximumWidth(150)

        option_bar = QtWidgets.QWidget()
        option_bar.setObjectName("OptionBar")
        option_bar.setSizePolicy(QtWidgets.QSizePolicy.Preferred, QtWidgets.QSizePolicy.Preferred)
        option_bar.setMaximumHeight(49)
        option_bar.setAutoFillBackground(True)
        option_bar.setStyleSheet("QWidget#OptionBar { border-top: 1px solid #ABABAB; }")

        option_bar_h_layout = QtWidgets.QHBoxLayout(option_bar)
        option_bar_h_layout.setContentsMargins(9, 10, 14, 9)
        option_bar_h_layout.addSpacerItem(option_bar_spacer)
        option_bar_h_layout.addWidget(option_bar_push_button)

        return option_bar

    def generate_tabs_from_json(self):
        all_tabs = self.json_ui_template["tabs"]

        # Loop through all array_of_tabs data
        tab_idx = 0
        for tab in all_tabs:
            tab_name: str = tab[JsonKey.NAME.value]
            widgets = tab[JsonKey.WIDGETS.value]
            tables = None

            if JsonKey.TABLE.value in tab:
                tables = tab[JsonKey.TABLE.value]

            # Declare the widgets
            new_tab_widget = QtWidgets.QWidget()
            new_tab_widget.setBackgroundRole(QtGui.QPalette.Light)
            new_tab_widget.setAutoFillBackground(True)

            scroll_area_root_layout = QtWidgets.QVBoxLayout(new_tab_widget)
            scroll_area_root_layout.setContentsMargins(0, 0, 0, 0)
            scroll_area_root_layout.setSpacing(0)

            scroll_area = QtWidgets.QScrollArea()
            scroll_area.setAutoFillBackground(True)
            scroll_area.setFrameShape(QtWidgets.QFrame.NoFrame)
            scroll_area.setWidgetResizable(True)
            scroll_area.setVerticalScrollBarPolicy(QtCore.Qt.ScrollBarAlwaysOn)

            scroll_area_widget = QtWidgets.QWidget()

            # Add the child widgets
            scroll_area_root_layout.addWidget(scroll_area)
            scroll_area.setWidget(scroll_area_widget)

            v_layout = QtWidgets.QVBoxLayout(scroll_area_widget)
            v_layout.setContentsMargins(0, 0, 0, 0)
            v_layout.setSizeConstraint(QtWidgets.QVBoxLayout.SetMinimumSize)

            self.generate_widgets(v_layout, widgets)
            if tables is not None:
                # if tab_name == "Connectivity":
                #     self.generate_connectivity_tables(v_layout, tables)
                # else:
                self.generate_tables(v_layout, tables)

            v_layout_spacer = QtWidgets.QSpacerItem(20, 20, QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Expanding)
            v_layout.addSpacerItem(v_layout_spacer)

            if tab_name == "General":
                option_bar_widget = self.generate_reset_default_option_bar_widget()
                scroll_area_root_layout.addWidget(option_bar_widget)

            self.tab_widget.addTab(new_tab_widget, tab_name)


            with_err = False
            for json_widget_dict in widgets:
                param_name = json_widget_dict[JsonKey.PARAM_NAME.value]
                if param_name in self.validated_param_map:
                    validated_param_result = self.validated_param_map[param_name]
                    if validated_param_result.result == False:
                        with_err = True

            if with_err:
                self.tab_widget.setTabIcon(tab_idx, self.err_icon)

            # Solve IPM 103: Take care of scroll to top on reset_window bug.
            v_scroll_bar = scroll_area.verticalScrollBar()
            v_scroll_bar.setTracking(False)

            if tab_name in self.tab_scroll_values.keys():
                self.tab_scroll_values[tab_name]["scroll_bar"] = v_scroll_bar
            else:
                self.tab_scroll_values[tab_name] = {
                    "scroll_bar": v_scroll_bar,
                    "value": 0,
                }

            # TODO: Find out why range change is automatically emitted by v_scroll_bar
            v_scroll_bar.rangeChanged.connect(
                functools.partial(self.set_latest_slider_values, tab_name)
            )

            tab_idx += 1

            self.tab_widget.currentChanged.connect(self.on_tab_changed)


        self.refocus_widget()
        self.add_deliverables_tab()
        self.add_summary_tab(all_tabs)

    def generate_widgets(self, scroll_area_v_layout: QtWidgets.QVBoxLayout, widgets: List[Dict]):
        for json_widget_dict in widgets:
            widget_type = json_widget_dict[JsonKey.TYPE.value]

            if widget_type == WidgetType.NOTE.value:
                note_generator = NoteGenerator(json_widget_dict)
                widget_container = note_generator.generate_widget()
                scroll_area_v_layout.addWidget(widget_container)
                continue

            param_name = json_widget_dict[JsonKey.PARAM_NAME.value]
            validated_param_result = self.validated_param_map[param_name]
            param_template = validated_param_result.param_template

            if widget_type == WidgetType.QLINEEDIT.value:
                if json_widget_dict[JsonKey.WIDGET_DATATYPE.value] == "HEXADECIMAL":
                    generator_class = QBitStringLineEditGenerator(json_widget_dict, validated_param_result)
                else:
                    generator_class = QLineEditGenerator(json_widget_dict, validated_param_result)
            elif widget_type == WidgetType.QCOMBOBOX.value:
                generator_class = QComboBoxGenerator(json_widget_dict, validated_param_result)
            elif widget_type == WidgetType.QSPINBOX.value:
                generator_class = QSpinBoxGenerator(json_widget_dict, validated_param_result)
            elif widget_type == WidgetType.QCHECKBOX.value:
                generator_class = QCheckBoxGenerator(json_widget_dict, validated_param_result)
            elif widget_type == WidgetType.QFILEDIALOG.value:
                generator_class = QFileDialogGenerator(json_widget_dict, validated_param_result, self.last_file_dialog_selection)
            else:
                continue

            # if it is not present, don't show.
            if param_template.isPresentGUI == BoolValues.FALSE.value:
                continue

            widget_container = generator_class.generate_widget()
            scroll_area_v_layout.addWidget(widget_container)

            gen_widget = generator_class.get_generated_display_widget()
            if param_template.isEditableGUI == BoolValues.FALSE.value or param_template.resolve == ResolveType.IMMEDIATE.value:
                gen_widget.setEnabled(False)
                continue

            gen_widget.installEventFilter(self)
            self.update_focused_widget(generator_class)

            connection_func = generator_class.get_connection_func()
            connection_func(
                functools.partial(self.send_to_server, generator_class, param_template, ""))

            if widget_type == WidgetType.QSPINBOX.value:
                generator_class.reset_special_text_value()

    '''
    NOTE: Table Example:
    "tables": [
        {
            "id": "0",
            "elements": {
                "AXI_S0": [
                    {
                        "param_name": "TABLE0_AXI_S0__MIN",
                        "type": "LineEdit",
                        "data_type": "HEXADECIMAL",
                        "row_label": "AXI_S0",
                        "column_label": "MIN"
                    },
                    {
                        "param_name": "TABLE0_AXI_S0__MAX",
                        "type": "LineEdit",
                        "data_type": "HEXADECIMAL",
                        "row_label": "AXI_S0",
                        "column_label": "MAX"
                    }
                ],
                "AXI_S1": [
                    {
                        "param_name": "TABLE0_AXI_S1__MIN",
                        "type": "LineEdit",
                        "data_type": "HEXADECIMAL",
                        "row_label": "AXI_S1",
                        "column_label": "MIN"
                    },
                    {
                        "param_name": "TABLE0_AXI_S1__MAX",
                        "type": "LineEdit",
                        "data_type": "HEXADECIMAL",
                        "row_label": "AXI_S1",
                        "column_label": "MAX"
                    }
                ]
            }
        }
    ]
    '''

    #IPM-780: Fix scrolling issues in IPM tabs
    def on_tab_changed(self, index):
        current_tab_widget = self.tab_widget.widget(index)
        scroll_area = current_tab_widget.findChild(QtWidgets.QScrollArea)
        if scroll_area:
            scroll_area_widget = scroll_area.widget()
            scroll_area_widget.updateGeometry()
            scroll_area_widget.adjustSize()

    def generate_tables(self, scroll_area_v_layout: QtWidgets.QVBoxLayout, tables: List[Dict]):
        for table in tables:
            id = table[JsonKey.TABLE_ID.value]
            row_lbl_to_elem_list_dict: Dict[str, List] = table[JsonKey.TABLE_ELEMENTS.value]

            table_widget = QtWidgets.QTableWidget()
            row_lbl_list: List[str] = list(row_lbl_to_elem_list_dict.keys())
            col_lbl_list: List[str] = []

            row_count = len(row_lbl_list)
            if row_count > 0:
                scroll_area_v_layout.addWidget(table_widget)

                warning_text = f"NOTE: New flow for tables, evaluation for each cell is only done after editing is finished."
                warning_text += f"\nPress enter on edit finish to trigger validation or click outside of the table cell"
                warning_lbl = QtWidgets.QLabel(warning_text)
                scroll_area_v_layout.addWidget(warning_lbl)

                first_row_lbl = row_lbl_list[0]
                all_columns_of_first_row = row_lbl_to_elem_list_dict[first_row_lbl]
                column_count = len(all_columns_of_first_row)

                for tbl_element in all_columns_of_first_row:
                    col_lbl = tbl_element[JsonKey.TABLE_COLUMN_LABEL.value]
                    col_lbl_list.append(col_lbl)

                table_widget.setRowCount(row_count)
                table_widget.setColumnCount(column_count)

            for row_idx, row_lbl in enumerate(row_lbl_to_elem_list_dict):
                all_columns_of_row = row_lbl_to_elem_list_dict[row_lbl]

                for col_idx, tbl_element in enumerate(all_columns_of_row):
                    param_name = tbl_element[JsonKey.PARAM_NAME.value]
                    validated_param_result = self.validated_param_map[param_name]
                    param_template = validated_param_result.param_template

                    # We must set row count first before we're able to set item and show value
                    # because on change row count, any inserted text gets reset
                    # pass param value to json_generator
                    generator_class = QTableCellWidgetGenerator(tbl_element, validated_param_result)

                    widget_container = generator_class.generate_widget()
                    table_widget.setCellWidget(row_idx, col_idx, widget_container)

                    # if it is not present, don't show.
                    if param_template.isPresentGUI == BoolValues.FALSE.value:
                        table_widget.hideRow(row_idx)
                        continue

                    gen_widget = generator_class.get_generated_display_widget()
                    if param_template.isEditableGUI == BoolValues.FALSE.value or param_template.resolve == ResolveType.IMMEDIATE.value:
                        gen_widget.setEnabled(False)
                        continue

                    generator_class.generate_error_message(scroll_area_v_layout)

                    gen_widget.installEventFilter(self)
                    # self.update_focused_widget(generator_class)

                    generator_class.set_connection_func(self.send_to_server)
                    # connection_func = generator_class.get_connection_func()
                    # connection_func(
                    #     functools.partial(self.send_to_server, generator_class, param_template, "")
                    # )

            if len(col_lbl_list) > 0:
                table_widget.setHorizontalHeaderLabels(col_lbl_list)
                table_widget.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.Stretch)
                table_widget.horizontalHeader().setDefaultAlignment(QtCore.Qt.AlignCenter)

            if len(row_lbl_list) > 0:
                table_widget.setVerticalHeaderLabels(row_lbl_list)
                table_widget.verticalHeader().setMinimumWidth(80)
                table_widget.verticalHeader().setDefaultAlignment(QtCore.Qt.AlignCenter)

            if len(col_lbl_list) > 0 and len(row_lbl_list) > 0:
                table_widget.setStyleSheet('''
                QTableView QTableCornerButton::section {
                    border-left: 0px outset #CACACA;
                    border-top: 0px outset #CACACA;
                    border-right: 1px outset #CACACA;
                    border-bottom: 1px outset #CACACA;
                    background-color: #F6F6F6;
                }
                ''')

    def generate_connectivity_tables(self, scroll_area_v_layout: QtWidgets.QVBoxLayout, tables: List[Dict]):
        # TODO: This logic is incomplete but it has low priority. Disabling for now.
        for table in tables:
            id = table[JsonKey.TABLE_ID.value]
            row_lbl_to_elem_list_dict: Dict[str, List] = table[JsonKey.TABLE_ELEMENTS.value]

            table_widget = QtWidgets.QTableWidget()

            row_lbl_list: List[str] = list(row_lbl_to_elem_list_dict.keys())
            col_lbl_list: List[str] = []

            row_count = len(row_lbl_list)
            if row_count > 0:
                scroll_area_v_layout.addWidget(table_widget)

                s_port_validated_param_result = self.validated_param_map["S_PORTS"]
                s_port_value = s_port_validated_param_result.param_template.value

                column_count = int(s_port_value)

                table_widget.setRowCount(row_count)
                table_widget.setColumnCount(column_count)

            # for row in range(m_port_value)
            for row_idx, row_lbl in enumerate(row_lbl_to_elem_list_dict):
                all_columns_of_row = row_lbl_to_elem_list_dict[row_lbl]

                for col in range(column_count):
                    for tbl_element in all_columns_of_row:
                        param_name = tbl_element[JsonKey.PARAM_NAME.value]
                        validated_param_result = self.validated_param_map[param_name]
                        param_template = validated_param_result.param_template

                        # only run when len(param_value) >= s_port_value
                        # To handle when len(param_value) < s_port_value, index_out_range
                        if len(param_template.value) >= column_count:
                            param_value = param_template.value[col]

                            # We must set row count first before we're able to set item and show value
                            # because on change row count, any inserted text gets reset
                            # pass param_obj.param_value.value[col] to json_generator
                            generator_class = QTableCellConnectivityWidgetGenerator(tbl_element, param_template, param_value)

                            widget_container = generator_class.generate_widget()
                            table_widget.setCellWidget(row_idx, col, widget_container)

                            # if it is not present, don't show.
                            if param_template.isPresentGUI == BoolValues.FALSE.value:
                                table_widget.hideRow(row_idx)
                                continue

                            gen_widget = generator_class.get_generated_display_widget()
                            if param_template.isEditableGUI == BoolValues.FALSE.value or param_template.resolve == ResolveType.IMMEDIATE.value:
                                gen_widget.setEnabled(False)
                                continue

                            generator_class.generate_error_message(scroll_area_v_layout)

                            gen_widget.installEventFilter(self)
                            self.update_focused_widget(generator_class)

                            connection_func = generator_class.get_connection_func()
                            connection_func(
                                functools.partial(self.send_to_server, generator_class, param_template, col)
                            )

                            col_name = "S"+str(col)
                            col_lbl_list.append(col_name)
                # add "" to param_value so 00000000 won't become 0
                param_template = create_new_parameter_template_by_value("\""+param_template.value+"\"", "")
                # print(row_idx,": ",param_obj.param_value.value)

            if len(col_lbl_list) > 0:
                table_widget.setHorizontalHeaderLabels(col_lbl_list)
                table_widget.horizontalHeader().setSectionResizeMode(QtWidgets.QHeaderView.ResizeToContents)
                table_widget.horizontalHeader().setDefaultAlignment(QtCore.Qt.AlignCenter)

            if len(row_lbl_list) > 0:
                table_widget.setVerticalHeaderLabels(row_lbl_list)
                table_widget.verticalHeader().setMinimumWidth(80)
                table_widget.verticalHeader().setDefaultAlignment(QtCore.Qt.AlignCenter)

            if len(col_lbl_list) > 0 and len(row_lbl_list) > 0:
                table_widget.setStyleSheet('''
                QTableView QTableCornerButton::section {
                    border-left: 0px outset #CACACA;
                    border-top: 0px outset #CACACA;
                    border-right: 1px outset #CACACA;
                    border-bottom: 1px outset #CACACA;
                    background-color: #F6F6F6;
                }

                QTableView{
                selection-background-color: white;
                }
                ''')

    def generate_diagram_on_load(self, is_blk_viewer_loaded, is_window_loaded):
        if self.is_blk_viewer_loaded and self.is_window_loaded:
            self.generate_block_diagram()
        else:
            # A short delay is needed to load it the first time. Otherwise, no diagram shows
            if is_blk_viewer_loaded:
                self.is_blk_viewer_loaded = is_blk_viewer_loaded

            if is_window_loaded:
                self.is_window_loaded = is_window_loaded

            if self.is_blk_viewer_loaded and self.is_window_loaded:
                QtCore.QTimer.singleShot(200, self.generate_block_diagram)

    def generate_block_diagram(self):
        self.blk_viewer.begin_load_block()
        component_obj = toObject(self.temp_component)
        list_of_ports = component_obj.ports
        port_template_dict_payload = []
        for port in list_of_ports:
            port_template = list_of_ports[port]
            if port_template.isPresent == "True":
                port_dict = port_template.to_dict()
                left_right_val = port_template.get_left_right_value()
                if left_right_val is not None:
                    msb_val = left_right_val[0]
                    lsb_val = left_right_val[1]

                    port_dict['gui_left_right_val'] = [msb_val, lsb_val]
                port_template_dict_payload.append(port_dict)

        block_name = self.ui_dialog.le_inst.displayText()
        # Wrap the block name if it exceeds a certain length
        max_length = 10
        wrapped_block_name = '\n'.join(textwrap.wrap(block_name, width=max_length))

        data = {
            'block_name': wrapped_block_name,
            'payload': port_template_dict_payload
        }

        # print(data)
        self.blk_viewer.load_block(data)
        self.blk_viewer.end_load_block()

    def add_deliverables_tab(self):
        # Add deliverables -------------------------------------
        # Declare the widget
        deliverables_tab = QtWidgets.QWidget()

        scroll_area_root_layout = QtWidgets.QVBoxLayout(deliverables_tab)
        scroll_area_root_layout.setContentsMargins(0, 0, 0, 0)

        scroll_area = QtWidgets.QScrollArea()
        scroll_area.setBackgroundRole(QtGui.QPalette.Light)
        scroll_area.setFrameShape(QtWidgets.QFrame.NoFrame)

        scroll_area_widget = QtWidgets.QWidget()
        scroll_area.setWidgetResizable(True)

        # Add the child widgets
        scroll_area_root_layout.addWidget(scroll_area)
        scroll_area.setWidget(scroll_area_widget)

        vlayout_dlvrb = QtWidgets.QVBoxLayout(scroll_area_widget)
        vlayout_dlvrb.setContentsMargins(9, 9, 9, 9)

        # Deliverables --------------------------------------
        lbl_dlvrb_header = QtWidgets.QLabel("Generate these deliverables:")
        lbl_dlvrb_header.setContentsMargins(0, 0, 0, 5)
        vlayout_dlvrb.addWidget(lbl_dlvrb_header)

        for dlvrb in self.gui_fileset_deliverables:
            fileset_display_name = dlvrb.display_name
            is_optional = dlvrb.is_optional

            checkBox_dlvrb = QtWidgets.QCheckBox(f"{fileset_display_name}")
            checkBox_dlvrb.setChecked(self.fileset_to_isGenerate_dict[fileset_display_name])
            checkBox_dlvrb.setContentsMargins(0, 5, 0, 5)
            checkBox_dlvrb.stateChanged.connect(
                functools.partial(self.update_deliverables, fileset_display_name)
            )

            if (is_optional == False):
                checkBox_dlvrb.setEnabled(False)

            vlayout_dlvrb.addWidget(checkBox_dlvrb)

        vlayout_dlvrb_spacer = QtWidgets.QSpacerItem(20, 20, QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Expanding)
        vlayout_dlvrb.addSpacerItem(vlayout_dlvrb_spacer)

        self.tab_widget.addTab(deliverables_tab, "Deliverables")

    def update_deliverables(self, fileset_display_name: int, state: int):
        if state == Qt.Unchecked:
            self.fileset_to_isGenerate_dict[fileset_display_name] = False
        elif state == Qt.Checked:
            self.fileset_to_isGenerate_dict[fileset_display_name] = True

        self.reset_window()


    def add_summary_tab(self, all_tabs):
        # Add summary -------------------------------------
        # Declare the widget
        summary_tab = QtWidgets.QWidget()
        summary_tab.setObjectName('Summary')
        summary_tab.setBackgroundRole(QtGui.QPalette.Light)
        summary_tab.setAutoFillBackground(True)

        root_layout = QtWidgets.QVBoxLayout(summary_tab)
        root_layout.setContentsMargins(0, 0, 0, 0)
        root_layout.setSpacing(0)

        v_splitter = QtWidgets.QSplitter(Qt.Vertical)
        v_splitter.setContentsMargins(0, 0, 0, 0)
        root_layout.addWidget(v_splitter)

        scroll_area = QtWidgets.QScrollArea()
        scroll_area.setBackgroundRole(QtGui.QPalette.Light)
        scroll_area.setFrameShape(QtWidgets.QFrame.NoFrame)
        scroll_area.setStyleSheet("QScrollArea { border: 0px; border-bottom: 1px solid #ABABAB; }")

        scroll_area_widget = QtWidgets.QWidget()
        scroll_area.setWidgetResizable(True)

        # Add the child widgets
        v_splitter.addWidget(scroll_area)
        scroll_area.setWidget(scroll_area_widget)

        vlayout_summary = QtWidgets.QVBoxLayout(scroll_area_widget)
        vlayout_summary.setContentsMargins(9, 9, 9, 9)

        # Details section --------------------------------------
        def create_lbl_details(key: str) -> QtWidgets.QLabel:
            return QtWidgets.QLabel(f"{key.capitalize()}: {self.vlnv_dict[key]}")

        self.create_summary_section(
            header_lbl_text = "<h2>Details</h2>",
            vlayout_summary =  vlayout_summary,
            iterable_list = self.vlnv_dict,
            create_lbl_func = create_lbl_details,
            spacer_items = [20, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed]
        )

        # Deliverables section ----------------------------------
        def create_lbl_dlvrb(fileset: str) -> QtWidgets.QLabel:
            return QtWidgets.QLabel(f"{fileset}: {self.fileset_to_isGenerate_dict[fileset]}")

        self.create_summary_section(
            header_lbl_text = "<h2>Deliverables</h2>",
            vlayout_summary =  vlayout_summary,
            iterable_list = self.fileset_to_isGenerate_dict,
            create_lbl_func = create_lbl_dlvrb,
            spacer_items = [20, 20, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed]
        )

        # Param section ---------------------------------------
        lbl_param_header = QtWidgets.QLabel("<h2>Configurations</h2>")
        lbl_param_header.setContentsMargins(0, 0, 0, 5)
        vlayout_summary.addWidget(lbl_param_header)

        def create_lbl_param(tab_widget: Any) -> Union[QtWidgets.QLabel, None]:
            widget_type = tab_widget[JsonKey.TYPE.value]
            if widget_type ==  WidgetType.NOTE.value:
                return None

            param_name = tab_widget[JsonKey.PARAM_NAME.value]
            validated_param_result = self.validated_param_map[param_name]
            param_temp = validated_param_result.param_template
            if param_temp.displayName.strip() == "":
                param_temp = param_temp._replace(displayName = param_temp.name)

            return QtWidgets.QLabel(f"{param_temp.displayName}: {param_temp.value}")

        def create_tbl_lbl_param(tbl_element: Any) -> Union[QtWidgets.QLabel, None]:
            param_name = tbl_element[JsonKey.PARAM_NAME.value]
            validated_param_result = self.validated_param_map[param_name]
            param_temp = validated_param_result.param_template
            if param_temp.isPresentGUI == BoolValues.FALSE.value:
                return None

            param_obj_row_label = tbl_element[JsonKey.TABLE_ROW_LABEL.value]
            param_obj_column_label = tbl_element[JsonKey.TABLE_COLUMN_LABEL.value]
            param_disp_name = param_obj_row_label + "_" + param_obj_column_label

            param_value = param_temp.value

            return QtWidgets.QLabel(f"{param_disp_name}: {param_value}")

        for tab in all_tabs:
            tab_name = tab[JsonKey.NAME.value]
            widgets = tab[JsonKey.WIDGETS.value]

            if len(widgets) > 0:
                self.create_summary_section(
                    header_lbl_text = f"<h3><u>{tab_name}</u></h3>",
                    vlayout_summary =  vlayout_summary,
                    iterable_list = widgets,
                    create_lbl_func = create_lbl_param,
                    spacer_items = [5, 5, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed]
                )

            if JsonKey.TABLE.value in tab:
                tables = tab[JsonKey.TABLE.value]

                if len(tables) > 0:
                    self.create_summary_section(
                        header_lbl_text = f"<h3><u>{tab_name}</u></h3>",
                        vlayout_summary =  vlayout_summary,
                        iterable_list = tables,
                        create_lbl_func = create_tbl_lbl_param,
                        spacer_items = [5, 5, QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed],
                        is_table = True
                    )

        vlayout_summary_spacer = QtWidgets.QSpacerItem(20, 20, QtWidgets.QSizePolicy.Fixed, QtWidgets.QSizePolicy.Expanding)
        vlayout_summary.addSpacerItem(vlayout_summary_spacer)

        console_widget = self.console.get_console_widget()
        if console_widget is not None:
            console_parent = QtWidgets.QWidget()

            console_layout = QtWidgets.QVBoxLayout(console_parent)
            console_layout.setContentsMargins(0, 0, 0, 0)
            console_layout.setSpacing(0)

            console_header = QtWidgets.QLabel("<h2>Console</h2>")
            console_header.setContentsMargins(9, 9, 9, 9)
            console_layout.addWidget(console_header)
            console_layout.addWidget(console_widget)

            v_splitter.addWidget(console_parent)
            v_splitter.setSizes([2000, 1000])


        self.tab_widget.addTab(summary_tab, "Summary")

    def create_summary_section(
        self,
        header_lbl_text: str,
        vlayout_summary: QtWidgets.QVBoxLayout,
        iterable_list: Iterable,
        create_lbl_func: Callable[[Any], QtWidgets.QLabel],
        spacer_items: List,
        is_table: bool = False
    ):
        lbl_header = QtWidgets.QLabel(header_lbl_text)
        vlayout_summary.addWidget(lbl_header)

        lbl_width = self.tab_widget.width()
        labels_per_line = 2
        list_length = len(iterable_list)
        if is_table:
            # TODO: Cleanup summary section
            first_table_entry = iterable_list[0]
            first_elements = first_table_entry[JsonKey.TABLE_ELEMENTS.value]
            row_length = len(first_elements)
            list_length = list_length * row_length

        def style_lbl_summary(
            lbl_summary: QtWidgets.QLabel,
            hlayout_line: QtWidgets.QHBoxLayout,
            label_count: int,
            list_length: int
            ) -> QtWidgets.QHBoxLayout:

            lbl_summary.setContentsMargins(1, 5, 0, 5)
            lbl_summary.setWordWrap(True)
            lbl_summary.setMinimumWidth(int(lbl_width * 0.45))
            size_policy = lbl_summary.sizePolicy()
            size_policy.setHorizontalStretch(1) # 50% of hLayout
            lbl_summary.setSizePolicy(size_policy)

            hlayout_line.addWidget(lbl_summary)
            if label_count % labels_per_line == 0 or label_count == list_length:
                vlayout_summary.addLayout(hlayout_line)
                hlayout_line = QtWidgets.QHBoxLayout()
            # handle problem when number of master is odd, it doesn't show on summary page
            elif label_count % labels_per_line == 1:
                vlayout_summary.addLayout(hlayout_line)

            return hlayout_line

        hlayout_line = QtWidgets.QHBoxLayout()
        label_count = 0
        for key in iterable_list:
            if is_table:
                table_entry = key
                row_lbl_to_elem_list_dict = table_entry[JsonKey.TABLE_ELEMENTS.value]
                for row_lbl in row_lbl_to_elem_list_dict:
                    all_columns_of_row = row_lbl_to_elem_list_dict[row_lbl]

                    for tbl_element in all_columns_of_row:
                        lbl_summary = create_lbl_func(tbl_element)
                        if lbl_summary != None:
                            label_count += 1
                            hlayout_line = style_lbl_summary(lbl_summary, hlayout_line, label_count, list_length)
            else:
                lbl_summary = create_lbl_func(key)
                if lbl_summary != None:
                    label_count += 1
                    hlayout_line = style_lbl_summary(lbl_summary, hlayout_line, label_count, list_length)

        # pass list items as arguments to QSpacerItem
        spacer = QtWidgets.QSpacerItem(*spacer_items)
        vlayout_summary.addSpacerItem(spacer)


if __name__ == '__main__':
    pass
