////////////////////////////////////////////////////////////////////////////
//           _____
//          / _______    Copyright (C) 2013-2024 Efinix Inc. All rights reserved.
//         / /       \
//        / /  ..    /   jtagReqResInterface.v
//       / / .'     /
//    __/ /.'      /     Description: Jtag to user interface

//   __   \       /
//  /_/ /\ \_____/ /
// ____/  \_______/
//
// *******************************
// Revisions:
// 0.1 Add Burst Operation
// 0.0 Initial rev
//
// *******************************
// `timescale 1 ns / 1 ps

`include "version.vh"

module jtag2ReqResInterface #
(
	parameter integer ADDR_WIDTH = 16,
	parameter integer DATA_WIDTH = 16,
	parameter integer DATA_PARITY_WIDTH = 4,
    parameter integer DATA_BURST_PTTRN = 2,
    parameter integer DATA_BURST_SIZE = 4
)
(
	input wire CAPTURE,
	input wire DRCK,
	input wire RESET,
	input wire RUNTEST,
	input wire SEL,
	input wire SHIFT,
	input wire UPDATE,
	input wire TCK,
	input wire TDI,
	input wire TMS,
	output wire TDO,

	input wire clk,
	input wire rst,

    output wire request,
    output wire rnw,
    output wire [ADDR_WIDTH-1:0] address,
    output wire [DATA_WIDTH-1:0] wrData,
    output wire [DATA_PARITY_WIDTH-1:0] wrData_parity,

    input done,
    input [1:0] resp,
    input [DATA_WIDTH-1:0] rdData
);

    reg  init_ff;
    reg  init_ff2;
    reg rnw_register;
    reg rnw_burst_register;
    reg  [DATA_PARITY_WIDTH-1:0] wrData_parity_register;
    reg  [DATA_PARITY_WIDTH-1:0] wrData_burst_parity_register;

    reg [ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH+DATA_BURST_PTTRN+DATA_BURST_SIZE:0] r_shift_regs;
    reg [DATA_WIDTH*2+1:0] r_send ='d0;
    reg [ADDR_WIDTH-1:0] address_base = 'd0;
    reg [DATA_WIDTH-1:0] wrData_register ='d0;
    
    
    reg [DATA_BURST_PTTRN-1:0] burst_pttrn     = 'd0;

    reg [DATA_BURST_SIZE-1:0] burst_size = 'd0;
    reg [2*DATA_WIDTH-1:0] burst_read_data = 'd0;
    reg [DATA_WIDTH-1:0] wrData_burst_register;
    reg [DATA_BURST_SIZE:0] offset = 'd0;

    reg burst_request    = 1'b0;
    reg request_register = 1'b0;

    reg [4:0] burst_state = STATE_idle;
    reg [4:0] prev_burst_state = STATE_idle;

    localparam STATE_idle           = 5'h0;
    localparam STATE_Rburst         = 5'h1; //Set Read First address
    localparam STATE_WrPi           = 5'h2; //Set Pi Value
    localparam STATE_WrFin          = 5'h3; //Set Clear Eye_Surf Value
    localparam STATE_WeyesurfStat   = 5'h4; //Set Read first address
    localparam STATE_Wburst         = 5'h5; //Set Enable Eye_surf
    localparam STATE_Finish         = 5'h6; //Wait clear Eye_surf
    localparam STATE_Wait           = 5'h7;

    ////////////////////////////////////////////////////////////
    //Generate a pulse to initiate transaction.
    //latch rising edge
    
    assign request =  (request_register) || burst_request;
    assign address = (burst_state == STATE_idle )  ? address_base : address_base + offset;
    assign rnw = (burst_state == STATE_idle ) ? rnw_register : rnw_burst_register;
    assign wrData_parity = (burst_state == STATE_idle ) ? wrData_parity_register : wrData_burst_parity_register;
    assign wrData = (burst_state == STATE_idle) ? wrData_register : wrData_burst_register;
    
    always @(posedge clk)
        if ((!init_ff2) && init_ff && address_base > 0 && burst_state == STATE_idle) begin 
            request_register <= 1'b1;
        end else 
            request_register <= 1'b0;

    always @(posedge clk)
    begin
        if (rst )
            begin
                init_ff <= 1'b0;
                init_ff2 <= 1'b0;
            end
        else
            begin
                init_ff <= i2_update;
                init_ff2 <= init_ff;
            end
    end

    reg [3:0] i_update;
    always @( posedge clk )
    begin
    	if ( rst )
    		begin
    			i_update <= 4'h0;
    		end
    	else
    		begin
    			i_update <= {i_update[2:0],UPDATE};
    		end
    end

    assign i2_update = (!i_update[1] & i_update[3]);

    ////////////////////////////////////////////////////////////

    always @( posedge TCK )
    begin
    	if ( rst )
    		r_shift_regs <= 0;
    	else
    		if ( SHIFT && SEL )
                // shift in MSB to LSB while LSB get shift out to TDO
    			r_shift_regs <= { TDI, r_shift_regs[ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH+DATA_BURST_PTTRN+DATA_BURST_SIZE:1] };
    		else if ( CAPTURE && SEL )
                r_shift_regs <= r_send;
            // clear buffer to ensure clean write/read next cycle
    		else if ( RESET && SEL )
    		    r_shift_regs <= 0;
    end

    wire update;
    assign update = UPDATE && SEL;
    always @( posedge (update) ) 
		begin
    	if ( rst ) 
			begin
    			rnw_register	<= 1'h0;
                wrData_parity_register <= {(4){1'h0}};
                burst_pttrn     <= {(4){1'h0}};
    			address_base	<= {(ADDR_WIDTH){1'h0}};
    			wrData_register <= {(DATA_WIDTH){1'h0}};
        	end 
		else 
			begin 
    	    //if ( UPDATE && SEL )
    	    //	begin
    	    		rnw_register	        <= r_shift_regs[ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH+DATA_BURST_SIZE+DATA_BURST_PTTRN];
                    burst_pttrn             <= r_shift_regs[ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH+DATA_BURST_SIZE+DATA_BURST_PTTRN-1 : ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH+DATA_BURST_SIZE];
                    wrData_parity_register  <= r_shift_regs[ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH-1:ADDR_WIDTH+DATA_WIDTH];
    	    		address_base            <= r_shift_regs[ADDR_WIDTH+DATA_WIDTH-1:DATA_WIDTH];
    	    		wrData_register	        <= r_shift_regs[DATA_WIDTH-1:0];
    	    //	end
        	end
    end
    
    assign TDO = r_shift_regs[0];
      
    always @( posedge clk ) begin
        case (burst_state)
            STATE_idle: begin            
                if (done && i2_update && address_base > 0) begin
                    if(burst_pttrn == 2'b01) begin
                        //Debugger Request Burst operation
                        prev_burst_state            <= burst_state;
                        burst_state                     <= STATE_Wait;
                        burst_size                      <= 1'b1;//r_shift_regs[ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH+DATA_BURST_SIZE-1:ADDR_WIDTH+DATA_WIDTH+DATA_PARITY_WIDTH];
                        burst_request                   <= 1'b1;
                        wrData_burst_parity_register    <= wrData_parity_register;
                        wrData_burst_register           <= wrData_register;
                        rnw_burst_register              <= 1'b1;
                        burst_read_data                 <= 'd0;
                    end else begin 
                        //Debugger Request normal Read/write operation
                        burst_size                      <= 1'b0;
                    end  
                    
                    offset                  <= 'd0;
                end else begin
                    burst_request           <= 1'b0;
                    wrData_burst_parity_register <= 'd0;
                    rnw_burst_register      <= 'd0;
                end

                if ( rst )
                    r_send <= 0;
                // special case, if burst_pattern is 2'b11, we just read out
                // the version number
                else if (done && burst_pttrn == 2'b11)
                        r_send <= {00, `SERDES_DEBUGGER_VERSION};
                else if (done && rnw  && burst_pttrn == 2'b00 )
                        r_send <= resp;
                else if (done && !rnw && burst_pttrn == 2'b00)
                        r_send <= {resp, rdData};
                else if (done && burst_pttrn != 2'b00)
                        r_send <= {resp, burst_read_data[2*DATA_WIDTH-1:0]};
                else 
                    r_send <= 'd0;
            end
            
            STATE_WrPi: begin 
                prev_burst_state            <= burst_state;
                if (done) begin
                    burst_state             <= STATE_Wait;
                    burst_request           <= 1'b1;
                    wrData_burst_register   <= wrData_burst_register | (1'b1 << 10);
                end
            end
        
            STATE_Wburst: begin 
                if (done) begin 
                    burst_state                     <= STATE_Wait;
                    burst_request                   <= 1'b1;
                    rnw_burst_register              <= 1'b0;
                    wrData_burst_parity_register    <= 4'b0000;
                end 
                prev_burst_state                    <= burst_state;
            end

            STATE_WeyesurfStat: begin 
                if (done) begin
                    if (rdData[8]) begin  
                        offset                  <= 4'd2;
                        burst_state             <= STATE_Wait;
                        burst_request           <= 1'b1;
                        rnw_burst_register      <= 1'b0;
                        burst_size              <= burst_size - 1'b1;
                    end else begin 
                        burst_state             <= STATE_Wburst;
                    end
                end 
                prev_burst_state            <= burst_state;
            end
            
            STATE_Rburst: begin 
                if (done) begin 
                    offset                  <= offset + 1'b1;
                    burst_read_data[2*DATA_WIDTH-1:DATA_WIDTH]         <= rdData;
                    burst_state             <= STATE_Wait;
                    burst_request           <= 1'b1;
                end 
                prev_burst_state            <= burst_state;
            end

            STATE_Finish: begin 
                if (done) begin 
                    burst_read_data[DATA_WIDTH-1:0]             <= rdData;
                    burst_state                     <= STATE_Wait;
                    burst_request                   <= 1'b1;
                    wrData_burst_parity_register    <= wrData_parity_register;
                    wrData_burst_register           <= wrData_register;
                    rnw_burst_register              <= 1'b1;
                end    
                prev_burst_state                    <= burst_state;
            end

            STATE_WrFin: begin 
                if (done) begin 
                    burst_state     <= STATE_idle;
                end 
            end
            
            STATE_Wait: begin
                burst_request <= 1'b0;
                if (!done) begin 
                    case (prev_burst_state)
                        STATE_idle :  burst_state       <= STATE_WrPi;
                        STATE_WrPi : burst_state        <= STATE_Wburst;
                        STATE_Wburst : burst_state      <= STATE_WeyesurfStat;
                        STATE_WeyesurfStat : burst_state  <= STATE_Rburst;
                        STATE_Rburst : burst_state      <= STATE_Finish;
                        STATE_Finish : burst_state      <= STATE_WrFin;
                        default: burst_state            <= STATE_idle;
                    endcase
                end 
            end
            
            default: burst_state            <= STATE_idle;

        endcase
    end
endmodule
