import sys
import os
import glob
from pathlib import Path
from time import sleep
from contextlib import contextmanager
from typing import Mapping, Sequence
import argparse
import subprocess
from array import array as Array
from time import time as now
from hashlib import sha1
#import pyreversebytes
#import pytobytes
import cProfile

sys.path.append(str(Path(os.environ['EFINITY_HOME'], 'debugger', 'bin')))
sys.path.append(str(Path(os.environ['EFINITY_HOME'], 'pgm', 'bin')))

from pyftdi.bits import BitSequence
from efx_dbg.jtag import JtagManager, JtagSession
from efx_pgm.usb_resolver import UsbResolver
from efx_pgm.efx_hw_client.remote.jtag2apb import Jtag2ApbRemoteController

class Options:
    class FtdiProgramArgs:
         def __init__(self):
            self.mode = 'jtag'
            self.output_file = ''
            self.urls = ['', '']
            self.xml = ''
            self.num = 1
            self.url = ''

    def __init__(self):
        self.ftdi_options = Options.FtdiProgramArgs()

def reverseBytes(_bytes: bytes) -> bytes:
    def _reverseByte(_byte) -> int:
        return int('{:08b}'.format(_byte)[::-1], 2)
    return bytes(tuple(map(lambda x: _reverseByte(x), _bytes)))

class Jtag2Apb:
    def __init__(self, jtag_session):
        self.jtag_session = jtag_session

    def write(self, address: bytes, wrdata: bytes, wrdata_parity: str='1111'):
        bseq_write_byte_data = bytes(tuple(reversed(wrdata))) + bytes(tuple(reversed(address)))
        bseq_write_bit_data = BitSequence(wrdata_parity + "0000" + "00" + "1", length=11) #parity + burst_size + burst_pttrn + read/write (1=write)

        with self.jtag_session.get_engine() as jtag_engine:
            jtag_engine.write_dr_bytes2(bseq_write_bit_data, bseq_write_byte_data)
            resp = jtag_engine.read_dr(2)
            jtag_engine.go_idle()
            return resp

    def read(self, address: bytes):
        bseq_read_byte_data = bytes((0,0)) + bytes(tuple(reversed(address)))
        bseq_read_bit_data = BitSequence("0000"+ "0000"+"00"+ "0", length=11)#parity + burst_size + burst_pttrn + read/write (0=read)

        with self.jtag_session.get_engine() as jtag_engine:
            jtag_engine.write_dr_bytes2(bseq_read_bit_data, bseq_read_byte_data)
            sleep(0.001)
            resp = jtag_engine.read_dr(34)
            jtag_engine.go_idle()
            return resp

    # read version number by using burst_pttrn '11'
    def read_version(self):
        # need to have non-zero address to enter proper state 
        bseq_read_byte_data = bytes((0,0,0,0,0,0,2,0))
        bseq_read_bit_data = BitSequence("0000"+ "0000"+"11"+ "0", length=11)#parity + burst_pttrn + burst_size + read/write

        with self.jtag_session.get_engine() as jtag_engine:
            jtag_engine.write_dr_bytes2(bseq_read_bit_data, bseq_read_byte_data)
            sleep(0.001)
            resp = jtag_engine.read_dr(34)
            jtag_engine.go_idle()
            return resp
        
    def burst_read(self,pi, address: bytes, in_value: BitSequence, burst_size: int = 0):
        data = BitSequence(0, length=32)
        if(in_value):
            data = data | in_value
        data = data | (BitSequence((min(pi, 0x7F)), length=32))
        bseq_read_byte_data = bytes(tuple(reversed(bytes(data.tobytes())))) + bytes(tuple(reversed(address)))
        bseq_read_bit_data = BitSequence("1111"+"1000"+"10"+ "0", length=11)#parity + burst_size + burst_pttrn + read/write

        with self.jtag_session.get_engine() as jtag_engine:
            jtag_engine.write_dr_bytes2(bseq_read_bit_data, bseq_read_byte_data)
            sleep(0.01)
            #Read two data payload 
            resp = jtag_engine.read_dr(66)
            jtag_engine.go_idle()
            return resp

class Jtag2ApbRemote:
    def __init__(self, url, chip_num, tap, user):
        self.client = Jtag2ApbRemoteController()
        configure_resp = self.client.configure(url, target_num=chip_num, frequency=6E6, tap=tap, user=user)

    def unconfigure(self):
        unconfig_resp = self.client.unconfigure()
        return unconfig_resp

    def write(self, address: bytes, wrdata: bytes, wrdata_parity: BitSequence=BitSequence('1111')):
        write_resp = self.client.write(address=address, data=wrdata, wrparity=wrdata_parity)
        return write_resp

    def read(self, address: bytes):
        read_resp = self.client.read(address=address)
        return read_resp
    
    def read_version(self):
        read_resp = self.client.read_version()
        return read_resp
    
    def burst_read(self, pi: int, address: bytes, in_value: bytes, wrdata_parity: BitSequence=BitSequence('1111'), burst_size: int = 0):
        read_resp = self.client.burst_read(pi = pi, address=address, in_value=in_value, wrparity=wrdata_parity, burst_size = burst_size)
        return read_resp
