# Copyright (C) 2019 Efinix Inc. All rights reserved.
#
# No portion of this code may be reused, modified or
# distributed in any way without the expressed written
# consent of Efinix Inc.
#
# Created on Jul 29, 2019
#
# @author: samh
"""Main GUI module for Efinix Debuggers.

This module is for the actual feature implementations of the debugger GUI.
UI elements are drawn in Qt Designer, saved as ``*.ui`` under ``resource`` folder,
and compiled into ``Ui_*.py`` using ``pyuic``.

All modules in efx_dbg are formatted with ``yapf --style=google``.

Attributes:
    LOGGER (StyleAdapter): A StyleAdapter that inherits LoggerAdapter such that
        logging messages can use comma based string formatting for variables.

    RADIX_TO_BASE (dict): Shorthand for translating radix strings `Bin`, `Dec`,
        or `Hex`, to int bases 2, 10, or 16.

    EFXPGM_HOME (str): A copy of the environment variable if it exists.

    EFXDBG_HOME (str): A copy of the environment variable if it exists.

.. _Google Python Style Guide:
       http://google.github.io/styleguide/pyguide.html
"""

import pprint
from abc import abstractmethod
import argparse
from enum import Enum
import glob
import json
import logging
import math
import os
import re
from shutil import which
import subprocess
import sys
import traceback
from typing import Callable, List, Optional, Tuple, Union, Dict, Any # pylint: disable=E0611
import copy
from time import sleep
from logging.handlers import RotatingFileHandler
from pathlib import Path
from functools import partial
from configparser import ParsingError, ConfigParser
import shutil
import tempfile
import random

import grpc
import efx_dbg.efinity_service_pb2
import efx_dbg.efinity_service_pb2_grpc

import usb.core
from pyftdi.bits import BitSequence
from pyftdi.ftdi import FtdiError
from pyftdi.jtag import JtagError
from pyftdi.usbtools import UsbToolsError
from PyQt5.QtGui import (QPainter, QPixmap, QIcon, QRegExpValidator, QKeyEvent,
                         QPalette, QFont, QCloseEvent, QFontDatabase,
                         QBrush, QColor)
from PyQt5.QtCore import (QSize, Qt, QEvent, QAbstractItemModel, QModelIndex,
                          QRegExp, QTimer, QObject, pyqtSignal, pyqtSlot,
                          QSettings, QVariant, QFileSystemWatcher)
from PyQt5.QtWidgets import (QDockWidget, QFileDialog, QHeaderView, QLabel, QTableWidget,
                             QFileIconProvider, QMessageBox, QWidget,
                             QTableWidgetItem, QDialog, QFrame,
                             QStyledItemDelegate, QStyleOptionViewItem,
                             QComboBox, QApplication, QAbstractItemView,
                             QGroupBox, QListWidgetItem, QDialogButtonBox,
                             QMainWindow, QActionGroup, QLineEdit,
                             QMenu, QToolButton)

# sys.path.append(os.path.join(os.environ["EFXPGM_HOME"], 'bin'))
import efx_pgm.util.bitstream_util as bitstream_util
import efx_pgm.util.app_logger
from efx_pgm.detect_jtag import EmptyDeviceText, NoUrlError
from efx_pgm.util.config_util import HwToolsConfigUtil
from _version import __version__

EFINITY_HOME = os.environ['EFINITY_HOME']
sys.path.append(os.path.join(EFINITY_HOME, 'scripts'))
from _app_info import _AppInfo

from efx_dbg.efx_dbg_gtkw import EfxGTKSave
from efx_dbg.jcf import JcfParser
from efx_dbg.jtag import (DebugSession, JtagManager, JtagSession)
from efx_dbg.exception import (ConfigExistingUrlError,
                               DeviceNotFoundError, EfxDbgCorruptedDataError, InvalidJsonError,
                               InvalidProfileError, UnknownUrlError,
                               UnknownUsbSernumError, InvalidDeviceIdError,
                               InvalidInterfaceError)
from efx_dbg.programmer import DbgPgmEngine, DbgPgmWin
from efx_dbg.rtl_gen import (write_debug_template_verilog, LogicNRtlDesignLimitError,
                             write_debug_template_vhdl, write_debug_top_verilog)
from efx_dbg.define import (CoreSpec, DebugProfile,
                            DebugProfileEncoder, DebugUsbTarget, LogLevel, LogicNSpec, StyleAdapter, VioSpec,
                            TCoreSpec, LogicNCaptureMode, LogicNTriggerMode,
                            DebugProfileBuilder)
from efx_dbg.util import change_int_base, log_trace, show_help_manual, gen_next_writable_file_path, HasOPXArch
from efx_dbg.widgets import HexLineEdit, VioActiveHighButton, VioActiveLowButton, VioToggleButton
from efx_dbg.adapter import QVioController, QLogicNController
from efx_dbg.exception import EfxDbgMisMatchCoreUUIDError

from efx_dbg.Ui_LaGen import Ui_LaGen
from efx_dbg.Ui_LaDebug import Ui_LaDebug
from efx_dbg.Ui_LaAddProbes import Ui_LaAddProbes
from efx_dbg.Ui_VioGen import Ui_VioGen
from efx_dbg.Ui_VioDebug import Ui_VioDebug
from efx_dbg.Ui_DebuggerAbout import Ui_DebuggerAbout
DOCKED_GUI = True if not 'DOCKED_GUI' in os.environ else (
    os.environ['DOCKED_GUI'].lower() in ['1', 'true', ''])
if DOCKED_GUI:
    from efx_dbg.Ui_DockedMainWindow import Ui_DockedMainWindow
else:
    from efx_dbg.Ui_DebuggerMainWindow import Ui_DebuggerMainWindow


LOGGER = logging.getLogger(__name__)

RADIX_TO_BASE = {'Dec': 10, 'Hex': 16, 'Bin': 2}

#QAPP = [None]

if 'EFXPGM_HOME' in os.environ:
    EFXPGM_HOME = os.environ['EFXPGM_HOME']

if 'EFXDBG_HOME' in os.environ:
    EFXDBG_HOME = os.environ['EFXDBG_HOME']
#else:
#    raise EnvironmentError('EFXDBG_HOME not in os.environ')

CLEAR_PROFILE_ON_IMPORT = True

global_grpc_stub = None
global_grpc_port = None
global_gui = None

def table_widget_item(text: Optional[str] = None,
                      value: Optional[QVariant] = None,
                      user_value: Optional[QVariant] = None,
                      no_drop: bool = False,
                      no_edit: bool = False,
                      gray: bool = False):
    if text:
        item = QTableWidgetItem(text)
    else:
        item = QTableWidgetItem()
    if value is not None:
        item.setData(Qt.EditRole, value)
    if user_value is not None:
        item.setData(Qt.UserRole, user_value)
    if no_drop:
        item.setFlags(item.flags() & ~Qt.ItemIsDropEnabled)
    if no_edit:
        item.setFlags(item.flags() & ~Qt.ItemIsEditable)
    if gray:
        item.setBackground(QBrush(Qt.gray))
    return item


def table_remove_selected_toprow(table: QTableWidget) -> None:
    selected = table.selectedRanges()
    row = selected[0].topRow() if selected else -1
    LOGGER.debug('selected={}, row={}'.format(selected, row))
    if row >= 0:
        table.removeRow(row)

def widgets_setEnabled(widgets: List[QWidget],
                            enable: bool) -> None:
    for widget in widgets:
        widget.setEnabled(enable)

QObjectType = type(QObject)

class DbgAboutDialog(QDialog):

    def __init__(self, license_file: str='', parent=None):
        super().__init__(parent)
        self.ui = Ui_DebuggerAbout()
        self.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui.setupUi(self)
        self.ui.label_appversion.setText("{} {} Debugger {}".format(
            _AppInfo.GUI_COMPANY_NAME, _AppInfo.APP_NAME, __version__))
        self.ui.textBrowser_about_info.setFontPointSize(10)
        self.ui.textBrowser_about_info.setText(" ")
        _uiFont = QFont("Ubuntu", 11)
        self.ui.textBrowser_about_info.setAlignment(Qt.AlignHCenter)
        self.ui.textBrowser_about_info.setCurrentFont(_uiFont)
        self.ui.textBrowser_about_info.insertPlainText("\n{}".format(_AppInfo.COPYRIGHT_TEXT))

class AbstractGenWidget(QFrame):

    error_received = pyqtSignal(str, str)

    def __init__(self, parent=None, name: str = ''):
        super().__init__(parent=parent)
        self._name = name

    @abstractmethod
    def title(self) -> str:
        pass

    @abstractmethod
    def get_spec(self, spec: CoreSpec):
        pass

    @abstractmethod
    def load_spec(self, spec: CoreSpec):
        pass


class VioDebugDelegate(QStyledItemDelegate):
    """Delegate for specifiying editors on the VIO setup table"""
    commit_index = pyqtSignal(QModelIndex)
    control_type_updated = pyqtSignal(QModelIndex)

    def __init__(self,
                 table: QTableWidget,
                 parent=None,
                 style_edited: bool = False):
        super().__init__(parent)
        self._style_edited = style_edited
        self._table = table

    def eventFilter(self, editor: QObject, event: QEvent) -> bool:  # pylint: disable=C0103
        """For avoiding focusOut when returnPressed"""
        if (event.type() == QEvent.KeyPress and
            (event.key() == Qt.Key_Enter or event.key() == Qt.Key_Return) and
                isinstance(editor, HexLineEdit) and self._style_edited):
            LOGGER.info('Emiting commitData')
            self.commitData.emit(editor)
            return False
        #LOGGER.debug(
        #    'VioGenDelegate returning super().eventFilter(editor={}, event={})',
        #    editor, event.type())
        return super().eventFilter(editor, event)

    def createEditor(  # pylint: disable=C0103
            self, parent: QWidget,
            option: QStyleOptionViewItem, index: QModelIndex):
        if index.column() == 2:
            editor = super().createEditor(parent, option, index)
            editor.setMinimum(1)
            editor.setMaximum(256)
            # Didn't commitData here bcoz editingFinished and valueChanged
            # behaves differently
            return editor
        if index.column() == 3:
            width = index.sibling(index.row(), 2).data(Qt.EditRole)
            editor = QComboBox(parent)
            if width == 1:
                editor.addItems(["Bin"])
            else:
                editor.addItems(['Bin', 'Hex', 'Dec'])
            # Helps commitData when item in comboBox is already selected but not
            # focusOut yet; Qt's commitData only happens on focusOut
            editor.activated.connect(self.emit_commitData)
            return editor
        if index.column() == 4:
            width = index.sibling(index.row(), 2).data(Qt.EditRole)
            radix = index.sibling(index.row(), 3).data(Qt.EditRole)
            value = index.sibling(index.row(), 4).data(Qt.EditRole)
            control_type = index.sibling(index.row(), 5).data(Qt.EditRole)
            if control_type == 'Active-High Button':
                editor = VioActiveHighButton(parent)
                editor.button_activate.connect(self.emit_commitData)
                editor.button_restore.connect(self.emit_commitData)
            elif control_type == 'Active-Low Button':
                editor = VioActiveLowButton(parent)
                editor.button_activate.connect(self.emit_commitData)
                editor.button_restore.connect(self.emit_commitData)
            elif control_type == 'Toggle':
                editor = VioToggleButton(value, parent)
                editor.button_activate.connect(self.emit_commitData)
            else:
                editor = HexLineEdit(parent, style_edited=self._style_edited)
                editor.set_bitwidth(width)
                editor.setDisplayIntegerBase(RADIX_TO_BASE[radix])
            LOGGER.debug('at {}, 4; width={}, radix={}'.format(index.row(), width,
                         radix))
            return editor

        if index.column() == 5:
            width = index.sibling(index.row(), 2).data(Qt.EditRole)
            editor = QComboBox(parent)
            if width == 1:
                editor.addItems(["Hex", "Active-High Button", "Active-Low Button", "Toggle"])
            else:
                editor.addItems(["Hex"])
            control_type = index.data(Qt.EditRole)
            cb_index = editor.findText(control_type)
            if cb_index != -1:
                editor.setCurrentIndex(cb_index)
            editor.currentIndexChanged.connect(self.on_control_type_updated)
            return editor

        if index.column() == 6:
            width = index.sibling(index.row(), 2).data(Qt.EditRole)
            radix = index.sibling(index.row(), 3).data(Qt.EditRole)
            editor = HexLineEdit(parent, style_edited=self._style_edited, empty = True)
            editor.set_bitwidth(width)
            editor.setDisplayIntegerBase(RADIX_TO_BASE[radix])
            return editor
        return super().createEditor(parent, option, index)

    def setEditorData(self, editor: QObject, index: QModelIndex):  # pylint: disable=C0103
        if index.column() == 2:
            value = index.data(Qt.EditRole)
            editor.setValue(value)
            LOGGER.debug('at {}, 2; value={}'.format(index.row(), value))
        elif index.column() == 3:
            current_text = index.data(Qt.EditRole)
            if isinstance(editor, QComboBox):
                cb_index = editor.findText(current_text)
                if cb_index >= 0:
                    editor.setCurrentIndex(cb_index)
                LOGGER.debug('at {}, 3; value={}'.format(index.row(), cb_index))
        elif index.column() == 4:
            text = index.data(Qt.EditRole)
            radix = index.sibling(index.row(), 3).data(Qt.EditRole)
            if isinstance(editor, (HexLineEdit, VioToggleButton)):
                editor.setText(text)
                editor.setDisplayIntegerBase(RADIX_TO_BASE[radix])
            LOGGER.debug('at {}, 4; value={}, radix={}'.format(index.row(), text,
                         radix))
        elif index.column() == 6:
            text = index.data(Qt.EditRole)
            radix = index.sibling(index.row(), 3).data(Qt.EditRole)
            editor.setText(text)
            editor.setDisplayIntegerBase(RADIX_TO_BASE[radix])
            LOGGER.debug('at {}, 6; value={}, radix={}'.format(index.row(), text,
                         radix))
        else:
            super().setEditorData(editor, index)

    def setModelData(  # pylint: disable=C0103
            self, editor: QObject, model: QAbstractItemModel,
            index: QModelIndex):
        if index.column() == 2:
            #editor.interpretText()
            value = editor.value()
            model.setData(index, value, Qt.EditRole)
            LOGGER.debug('at {}, 2; value={}'.format(index.row(), value))
            # Trigger column 4 validator fixup immediately
            self.update_value(index)
        elif index.column() == 3:
            new_radix = editor.currentText()
            old_radix = index.data(Qt.EditRole)
            model.setData(index, new_radix, Qt.EditRole)
            LOGGER.debug('at {}, 3; new_radix={}'.format(index.row(), new_radix))
            self.update_value(index)
            bitwidth = index.sibling(index.row(), 2).data(Qt.EditRole)
            old_text = index.sibling(index.row(), 4).data(Qt.EditRole)
            new_text = change_int_base(old_text, RADIX_TO_BASE[old_radix],
                                       RADIX_TO_BASE[new_radix], bitwidth)
            LOGGER.debug('Pre setData {}, 4;'.format(index.row()))
            model.setData(index.sibling(index.row(), 4), new_text, Qt.EditRole)
            LOGGER.debug('Post setData {}, 4; new_text={}'.format(index.row(),
                         new_text))
            old_text = index.sibling(index.row(), 6).data(Qt.EditRole)
            if old_text:
                new_text = change_int_base(old_text, RADIX_TO_BASE[old_radix],
                                        RADIX_TO_BASE[new_radix], bitwidth)
                LOGGER.debug('Pre setData {}, 6;'.format(index.row()))
                model.setData(index.sibling(index.row(), 6), new_text, Qt.EditRole)
                LOGGER.debug('Post setData {}, 6; new_text={}'.format(index.row(),
                        new_text))
        elif index.column() == 4:
            text = editor.text()
            # Fixes bug when somehow empty string '' passed thru validator and came here
            if text:
                model.setData(index, text, Qt.EditRole)
                LOGGER.info('at {}, 4; value={}'.format(index.row(), text))
                self.commit_index.emit(index)

        elif index.column() == 5:
            new_contro_type = editor.currentText()
            model.setData(index, new_contro_type, Qt.DisplayRole)
            model.setData(index, new_contro_type, Qt.EditRole)
            # Update the value to match with the button inital's
            value_cell_index = index.sibling(index.row(), 4)
            if new_contro_type == "Active-High Button":
                model.setData(value_cell_index, "0", Qt.EditRole)
                self.commit_index.emit(value_cell_index)
            elif new_contro_type == "Active-Low Button":
                model.setData(value_cell_index, "1", Qt.EditRole)
                self.commit_index.emit(value_cell_index)
            self.control_type_updated.emit(index)

        elif index.column() == 6:
            text = editor.text()
            text = text.lower()
            model.setData(index, text, Qt.EditRole)
            self.commit_index.emit(index)

        else:
            super().setModelData(editor, model, index)

    def updateEditorGeometry(  # pylint: disable=C0103
            self, editor: QObject,
            option: QStyleOptionViewItem, index: QModelIndex):
        if index.column() >= 2:
            editor.setGeometry(option.rect)
        else:
            super().updateEditorGeometry(editor, option, index)

    def update_value(self, index: QModelIndex):
        """Start edit and then key_return to trigger validator fixup for column 4"""
        index_col1 = index.sibling(index.row(), 1)
        if index.column() == 2 and index_col1.data(Qt.EditRole) == 'OUT':
            item = self._table.item(index.row(), 4)
            # Start editing
            self._table.editItem(item)
            widget = self._table.indexWidget(index.sibling(index.row(), 4))
            key_return = QKeyEvent(QEvent.KeyPress,
                                   Qt.Key_Return,
                                   Qt.NoModifier)
            # Finish editing
            QApplication.sendEvent(widget, key_return)

    def emit_commitData(self):
        self.commitData.emit(self.sender())

    def on_control_type_updated(self, new_idx):
        self.commitData.emit(self.sender())
        self.closeEditor.emit(self.sender())

class VioGenDelegate(QStyledItemDelegate):
    """Delegate for specifiying editors on the VIO setup table"""
    commit_index = pyqtSignal(QModelIndex)

    def __init__(self,
                 table: QTableWidget,
                 parent=None,
                 style_edited: bool = False):
        super().__init__(parent)
        self._style_edited = style_edited
        self._table = table

    def eventFilter(self, editor: QObject, event: QEvent) -> bool:  # pylint: disable=C0103
        """For avoiding focusOut when returnPressed"""
        if (event.type() == QEvent.KeyPress and
            (event.key() == Qt.Key_Enter or event.key() == Qt.Key_Return) and
                isinstance(editor, HexLineEdit) and self._style_edited):
            LOGGER.info('Emiting commitData')
            self.commitData.emit(editor)
            return False
        #LOGGER.debug(
        #    'VioGenDelegate returning super().eventFilter(editor={}, event={})',
        #    editor, event.type())
        return super().eventFilter(editor, event)

    def createEditor(  # pylint: disable=C0103
            self, parent: QWidget,
            option: QStyleOptionViewItem, index: QModelIndex):
        if index.column() == 2:
            editor = super().createEditor(parent, option, index)
            editor.setMinimum(1)
            editor.setMaximum(256)
            # Didn't commitData here bcoz editingFinished and valueChanged
            # behaves differently
            return editor
        if index.column() == 3:
            width = index.sibling(index.row(), 2).data(Qt.EditRole)
            editor = QComboBox(parent)
            editor.addItems(['Bin', 'Hex', 'Dec'])
            # Helps commitData when item in comboBox is already selected but not
            # focusOut yet; Qt's commitData only happens on focusOut
            editor.activated.connect(self.emit_commitData)
            return editor
        if index.column() == 4:
            editor = HexLineEdit(parent, style_edited=self._style_edited)
            width = index.sibling(index.row(), 2).data(Qt.EditRole)
            radix = index.sibling(index.row(), 3).data(Qt.EditRole)
            editor.set_bitwidth(width)
            editor.setDisplayIntegerBase(RADIX_TO_BASE[radix])
            LOGGER.debug('at {}, 4; width={}, radix={}'.format(index.row(), width,
                         radix))
            return editor
        return super().createEditor(parent, option, index)

    def setEditorData(self, editor: QObject, index: QModelIndex):  # pylint: disable=C0103
        if index.column() == 2:
            value = index.data(Qt.EditRole)
            editor.setValue(value)
            LOGGER.debug('at {}, 2; value={}'.format(index.row(), value))
        elif index.column() == 3:
            current_text = index.data(Qt.EditRole)
            cb_index = editor.findText(current_text)
            if cb_index >= 0:
                editor.setCurrentIndex(cb_index)
            LOGGER.debug('at {}, 3; value={}'.format(index.row(), cb_index))
        elif index.column() == 4:
            text = index.data(Qt.EditRole)
            radix = index.sibling(index.row(), 3).data(Qt.EditRole)
            editor.setText(text)
            editor.setDisplayIntegerBase(RADIX_TO_BASE[radix])
            LOGGER.debug('at {}, 4; value={}, radix={}'.format(index.row(), text,
                         radix))
        else:
            super().setEditorData(editor, index)

    def setModelData(  # pylint: disable=C0103
            self, editor: QObject, model: QAbstractItemModel,
            index: QModelIndex):
        if index.column() == 2:
            #editor.interpretText()
            value = editor.value()
            model.setData(index, value, Qt.EditRole)
            LOGGER.debug('at {}, 2; value={}'.format(index.row(), value))
            # Trigger column 4 validator fixup immediately
            self.update_value(index)
        elif index.column() == 3:
            new_radix = editor.currentText()
            old_radix = index.data(Qt.EditRole)
            model.setData(index, new_radix, Qt.EditRole)
            LOGGER.debug('at {}, 3; new_radix={}'.format(index.row(), new_radix))
            self.update_value(index)
            bitwidth = index.sibling(index.row(), 2).data(Qt.EditRole)
            old_text = index.sibling(index.row(), 4).data(Qt.EditRole)
            new_text = change_int_base(old_text, RADIX_TO_BASE[old_radix],
                                       RADIX_TO_BASE[new_radix], bitwidth)
            LOGGER.debug('Pre setData {}, 4;'.format(index.row()))
            model.setData(index.sibling(index.row(), 4), new_text, Qt.EditRole)
            LOGGER.debug('Post setData {}, 4; new_text={}'.format(index.row(),
                         new_text))
        elif index.column() == 4:
            text = editor.text()
            # Fixes bug when somehow empty string '' passed thru validator and came here
            if text:
                model.setData(index, text, Qt.EditRole)
                LOGGER.debug('at {}, 4; value={}'.format(index.row(), text))
                self.commit_index.emit(index)
        else:
            super().setModelData(editor, model, index)

    def updateEditorGeometry(  # pylint: disable=C0103
            self, editor: QObject,
            option: QStyleOptionViewItem, index: QModelIndex):
        if index.column() >= 2:
            editor.setGeometry(option.rect)
        else:
            super().updateEditorGeometry(editor, option, index)

    def update_value(self, index: QModelIndex):
        """Start edit and then key_return to trigger validator fixup for column 4"""
        index_col1 = index.sibling(index.row(), 1)
        if index.column() == 2 and index_col1.data(Qt.EditRole) == 'OUT':
            item = self._table.item(index.row(), 4)
            # Start editing
            self._table.editItem(item)
            widget = self._table.indexWidget(index.sibling(index.row(), 4))
            key_return = QKeyEvent(QEvent.KeyPress,
                                   Qt.Key_Return,
                                   Qt.NoModifier)
            # Finish editing
            QApplication.sendEvent(widget, key_return)

    def emit_commitData(self):
        self.commitData.emit(self.sender())


class VioGenWidget(AbstractGenWidget):

    """Widget added to the setup-tab of Debugger GUI for each VIO instance"""

    def __init__(self, parent=None, name: str = 'VIO_n'):

        super().__init__(parent=parent, name=name)

        self.ui = Ui_VioGen()  # pylint: disable=C0103
        self.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui.setupUi(self)
        self.ui.le_name.setText(self._name)

        self.delegate = VioGenDelegate(table=self.ui.tableWidget)
        self.ui.tableWidget.setEditTriggers(
            QAbstractItemView.AllEditTriggers)
        self.ui.tableWidget.setItemDelegate(self.delegate)
        self.ui.tableWidget.verticalHeader().setSectionsMovable(True)
        self.ui.tableWidget.setSortingEnabled(True)

        self.ui.pb_add_probe_in.clicked.connect(lambda: self.on_add_probe(True))
        self.ui.pb_add_probe_out.clicked.connect(lambda: self.on_add_probe(False))
        self.ui.pb_remove_probe.clicked.connect(self.on_remove_probe)

        self.ui.pb_help.clicked.connect(lambda: show_help_manual('VIO_GEN'))

    def title(self) -> str:
        return self.ui.le_name.text().strip()

    def on_remove_probe(self):
        table_remove_selected_toprow(self.ui.tableWidget)

    def _gen_probe_name(self, is_input: bool) -> str:
        rows = self.ui.tableWidget.rowCount()
        names = [
            self.ui.tableWidget.item(r, 0).data(Qt.EditRole)
            for r in range(rows)
            if self.ui.tableWidget.item(r, 0)
        ]
        prefix = 'probe' if is_input else 'source'
        idx = 0
        while prefix + str(idx) in names:
            idx += 1
        return prefix + str(idx)

    def on_add_probe(self, is_input: bool):
        self.ui.tableWidget.setSortingEnabled(False)
        row = self.ui.tableWidget.rowCount()
        self.ui.tableWidget.insertRow(row)
        #(':')(u'↕')(u'⇕')(u'⇵')(u'⇳')(u'⬍')
        item = QTableWidgetItem(u'↕')
        self.ui.tableWidget.setVerticalHeaderItem(row, item)

        probe_str = self._gen_probe_name(is_input)
        item = table_widget_item(text=probe_str, no_drop=True)
        self.ui.tableWidget.setItem(row, 0, item)

        dir_str = 'IN' if is_input else 'OUT'
        type_str = 'Probe' if is_input else 'Source'
        #item = table_widget_item(text=dir_str, no_drop=True, no_edit=True)
        item = table_widget_item(text=type_str,
                                 user_value=dir_str,
                                 no_drop=True,
                                 no_edit=True)
        self.ui.tableWidget.setItem(row, 1, item)

        item = table_widget_item(value=1, no_drop=True)
        LOGGER.debug('Calling setItem at {}, 2'.format(row))
        self.ui.tableWidget.setItem(row, 2, item)
        self.ui.tableWidget.openPersistentEditor(item)

        kwargs_off = {'no_drop': True, 'no_edit': True, 'gray': True}
        kwargs_on = {'value': 'Hex', 'no_drop': True}
        kwargs = kwargs_off if is_input else kwargs_on
        item = table_widget_item(**kwargs)
        LOGGER.debug('Calling setItem at {}, 3'.format(row))
        self.ui.tableWidget.setItem(row, 3, item)
        if not is_input:
            self.ui.tableWidget.openPersistentEditor(item)

        kwargs_on = {'value': '0', 'no_drop': True}
        kwargs = kwargs_off if is_input else kwargs_on
        item = table_widget_item(**kwargs)
        LOGGER.debug('tableWidget.setItem({}, 4)'.format(row))
        self.ui.tableWidget.setItem(row, 4, item)

        self.ui.tableWidget.setSortingEnabled(True)

    def get_spec(self, spec: VioSpec):
        table = self.ui.tableWidget
        for vr in range(table.rowCount()):  # pylint: disable=C0103
            row = table.verticalHeader().logicalIndex(vr)
            probe_name = table.item(row, 0).data(Qt.EditRole)
            #probe_dir = table.item(row, 1).data(Qt.EditRole)
            probe_dir = table.item(row, 1).data(Qt.UserRole)
            probe_width = table.item(row, 2).data(Qt.EditRole)
            if probe_dir == 'OUT':
                radix = table.item(row, 3).data(Qt.EditRole)
                text = table.item(row, 4).data(Qt.EditRole)
                probe_init = hex(int(text, RADIX_TO_BASE[radix]))
                spec.add_probe(probe_name, probe_width, probe_init)
            else:
                spec.add_probe(probe_name, probe_width)

    def load_spec(self, spec: VioSpec):
        table = self.ui.tableWidget
        LOGGER.debug('VioGenWidget table.item.setData foreach spec.get_probes')
        for row, (name, width, val) in enumerate(spec.get_probes()):  # pylint: disable=C0103
            self.on_add_probe(is_input=(val is None))
            table.item(row, 0).setData(Qt.EditRole, name)
            LOGGER.debug('table.item({}, 2).setData()'.format(row))
            table.item(row, 2).setData(Qt.EditRole, width)
            if val is not None:
                LOGGER.debug('table.item({}, 4).setData()'.format(row))
                table.item(row, 4).setData(Qt.EditRole, val[2:])


class LogicNGenDelegate(QStyledItemDelegate):
    """"""
    ptype_id = LogicNSpec.LogicNProbe.TID

    def createEditor(  # pylint: disable=C0103
            self, parent: QWidget,
            option: QStyleOptionViewItem, index: QModelIndex):
        if index.column() == 1:
            editor = super().createEditor(parent, option, index)
            editor.setMinimum(1)
            editor.setMaximum(1024)
            return editor
        if index.column() == 2:
            editor = QComboBox(parent)
            editor.addItems(list(self.ptype_id.keys()))
            editor.activated.connect(self.emit_commitData)
            return editor
        return super().createEditor(parent, option, index)

    def setEditorData(self, editor: QObject, index: QModelIndex):  # pylint: disable=C0103
        if index.column() == 2:
            current_text = index.data(Qt.EditRole)
            cb_index = editor.findText(current_text)
            if cb_index >= 0:
                editor.setCurrentIndex(cb_index)
            LOGGER.debug('at {}, 3; current_text={}, cb_index={}'.format(index.row(),
                         current_text, cb_index))
        else:
            super().setEditorData(editor, index)

    def setModelData(  # pylint: disable=C0103
            self, editor: QObject, model: QAbstractItemModel,
            index: QModelIndex):
        if index.column() == 2:
            text = editor.currentText()
            model.setData(index, text, Qt.EditRole)
        else:
            super().setModelData(editor, model, index)

    def updateEditorGeometry(  # pylint: disable=C0103
            self, editor: QObject,
            option: QStyleOptionViewItem, index: QModelIndex):
        if index.column() == 2:
            editor.setGeometry(option.rect)
        else:
            super().updateEditorGeometry(editor, option, index)

    def emit_commitData(self):
        self.commitData.emit(self.sender())


# Not in use
class IconProvider(QFileIconProvider):

    def icon(self, fileInfo):
        if fileInfo.isDir():
            return QIcon("../resource/debugger.svg")
        return QFileIconProvider.icon(self, fileInfo)


class LogicNGenWidget(AbstractGenWidget):
    """"""
    ptype_id = LogicNSpec.LogicNProbe.TID
    ptype_str = LogicNSpec.LogicNProbe.TSTR

    def __init__(self, parent=None, name: str = 'LogicN_n'):

        super().__init__(parent=parent, name=name)

        self.ui = Ui_LaGen()  # pylint: disable=C0103
        self.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui.setupUi(self)
        self.ui.le_name.setText(self._name)

        for i in range(3, 18):
            twos_pwr = 2**i
            self.ui.cb_data_depth.addItem(str(twos_pwr), twos_pwr)

        # Support input pipeline stage from 0 to 6
        for i in range(7):
            self.ui.cb_input_pipeline.addItem(str(i), i)

        # Default input pipeline stage to 1
        idx_input_pipeline = self.ui.cb_input_pipeline.findText(str(1))
        if idx_input_pipeline != -1:
            self.ui.cb_input_pipeline.setCurrentIndex(idx_input_pipeline)

        # Default data width 1024
        idx = self.ui.cb_data_depth.findText(str(1024))
        if idx != -1:
            self.ui.cb_data_depth.setCurrentIndex(idx)

        self.delegate = LogicNGenDelegate()
        self.ui.tableWidget.setItemDelegate(self.delegate)
        self.ui.tableWidget.verticalHeader().setSectionsMovable(True)
        self.ui.tableWidget.setSortingEnabled(True)

        self.ui.pb_add_probe_in.clicked.connect(self.on_add_probe)
        self.ui.pb_remove_probe.clicked.connect(self.on_remove_probe)

        self.ui.pb_help.clicked.connect(lambda: show_help_manual('LA_GEN'))

    def title(self) -> str:
        return self.ui.le_name.text().strip()

    def _gen_probe_name(self) -> str:
        rows = self.ui.tableWidget.rowCount()
        names = [
            self.ui.tableWidget.item(r, 0).data(Qt.EditRole)
            for r in range(rows)
            if self.ui.tableWidget.item(r, 0)
        ]
        prefix = 'probe'
        idx = 0
        while prefix + str(idx) in names:
            idx += 1
        return prefix + str(idx)

    def on_add_probe(self):
        row = self.ui.tableWidget.rowCount()
        if (row + 1) > LogicNSpec.MAX_NUM_OF_PROBES:
            self.error_received.emit("Add Probe", "Number of probes exceed the supported limit.")
            return
        self.ui.tableWidget.setSortingEnabled(False)
        self.ui.tableWidget.insertRow(row)
        item = QTableWidgetItem(u'↕')
        self.ui.tableWidget.setVerticalHeaderItem(row, item)

        probe_str = self._gen_probe_name()
        item = table_widget_item(text=probe_str, no_drop=True)
        self.ui.tableWidget.setItem(row, 0, item)

        item = table_widget_item(value=1, no_drop=True)
        LOGGER.debug('Calling setItem at {}, 1'.format(row))
        self.ui.tableWidget.setItem(row, 1, item)
        self.ui.tableWidget.openPersistentEditor(item)

        item = table_widget_item(value='DATA AND TRIGGER', no_drop=True)
        LOGGER.debug('Calling setItem at {}, 2'.format(row))
        self.ui.tableWidget.setItem(row, 2, item)
        self.ui.tableWidget.openPersistentEditor(item)
        self.ui.tableWidget.setSortingEnabled(True)

    def on_remove_probe(self):
        table_remove_selected_toprow(self.ui.tableWidget)

    def get_spec(self, spec: LogicNSpec):
        spec.set_attr('data_depth', self.ui.cb_data_depth.currentData())
        spec.set_attr('trigin_en', self.ui.chk_trigin.isChecked())
        spec.set_attr('trigout_en', self.ui.chk_trigout.isChecked())
        spec.set_attr('capture_control', self.ui.chk_cap_control.isChecked())
        spec.set_attr('input_pipeline', self.ui.cb_input_pipeline.currentData())

        table = self.ui.tableWidget
        for i in range(table.rowCount()):
            row = table.verticalHeader().logicalIndex(i)
            name = table.item(row, 0).data(Qt.EditRole)
            width = table.item(row, 1).data(Qt.EditRole)
            ptype_str = table.item(row, 2).data(Qt.EditRole)
            spec.add_probe(name, width, self.ptype_id[ptype_str])

    def load_spec(self, spec: LogicNSpec):
        idx = self.ui.cb_data_depth.findText(str(spec.get_depth()))
        self.ui.cb_data_depth.setCurrentIndex(idx)
        self.ui.chk_trigin.setChecked(spec.get_attr('trigin_en'))
        self.ui.chk_trigout.setChecked(spec.get_attr('trigout_en'))
        self.ui.chk_cap_control.setChecked(
            spec.get_attr('capture_control', default=False))


        table = self.ui.tableWidget
        for row, (name, width, ptype, idx) in enumerate(spec.get_probes()):
            self.on_add_probe()
            table.item(row, 0).setData(Qt.EditRole, name)
            table.item(row, 1).setData(Qt.EditRole, width)
            table.item(row, 2).setData(Qt.EditRole, self.ptype_str[ptype])

        if spec.get_attr('auto_inserted'):
            children = self.findChildren(QWidget)
            for child in children:
                LOGGER.debug('type(child) = {}'.format(type(child)))
                child.setEnabled(False)


class AbstractDebugWidget(QGroupBox):
    """"""

    def __init__(self, spec: TCoreSpec, parent=None):
        super().__init__(parent)
        self._spec = spec
        self._name = spec.get_name()

    @abstractmethod
    def load_profile(self, profile: DebugProfile) -> None:
        """
        Load the data from debug profile to GUI
        """
        raise NotImplementedError

    # @abstractmethod
    # def on_timer(self, jtag_url: Optional[str]) -> None:
    #     pass

    def is_setting_changed(self) -> bool:
        """
        Return True if the  debugger widget session is changed
        """
        raise NotImplementedError

    @pyqtSlot()
    def on_debug_core_pre_connect(self) -> None:
        """
        Slot to update the GUI when start connecting with the debug core
        """
        raise NotImplementedError

    @pyqtSlot()
    def on_debug_core_post_connect(self, dbg_sesh: DebugSession) -> None:
        """
        Slot to update the GUI when connection with debug core is establish successfully
        """
        raise NotImplementedError

    @pyqtSlot()
    def on_debug_core_error(self, error: Exception) -> None:
        """
        Slot to update the GUI when there is error between the connection with debug core
        """
        raise NotImplementedError

    @pyqtSlot()
    def on_debug_core_pre_disconnect(self) -> None:
        """
        Slot to update the GUI when start disconnecting the debug core
        """
        raise NotImplementedError

    @pyqtSlot()
    def on_debug_core_post_disconnect(self) -> None:
        """
        Slot to update the GUI when connection with debug core is closed successfully
        """
        raise NotImplementedError

    @pyqtSlot()
    def on_close(self) -> None:
        """
        Slot to do some cleanup when the debugger application is closing
        """
        pass


class VioGUIContext(QObject):

    state_changed = pyqtSignal()
    setting_changed = pyqtSignal()

    def __init__(self, initial_state: dict, initial_setting: dict):
        super(VioGUIContext, self).__init__()
        self._state = copy.deepcopy(initial_state)
        self._setting = copy.deepcopy(initial_setting)
        self._default_setting = copy.deepcopy(initial_setting)

    @property
    def state(self):
        return copy.deepcopy(self._state)

    @property
    def setting(self):
        return copy.deepcopy(self._setting)

    def is_setting_changed(self):
        return self._default_setting != self._setting

    @pyqtSlot(object)
    def dispatch_state_change(self, new_state):
        """
        Update the internal state
        """
        self._state = {**self._state, **new_state}
        self.state_changed.emit()

    @pyqtSlot(object)
    def dispatch_setting_overwrite(self, new_setting):
        """
        Update the internal setting by overwriting the whole dict,
        Also update the default setting
        """
        self._default_setting = copy.deepcopy(new_setting)
        self._setting = copy.deepcopy(new_setting)
        self.setting_changed.emit()

    @pyqtSlot(str, object)
    def dispatch_setting_change(self, setting_type, setting_value):
        """
        Update the internal setting
        """
        self._setting[setting_type] = setting_value
        LOGGER.debug(f'new_setting = {pprint.pformat(self._setting)}')
        self.setting_changed.emit()


class NewVioDebugWidget(AbstractDebugWidget):
    """Widget added to the Run-tab of GUI for each VIO instance"""
    commit_probe = pyqtSignal(str, list)
    error_received = pyqtSignal(Exception)

    TABLE_COLUMN_NAME = 0
    TABLE_COLUMN_TYPE = 1
    TABLE_COLUMN_WIDTH = 2
    TABLE_COLUMN_RADIX = 3
    TABLE_COLUMN_VALUE = 4
    TABLE_COLUMN_CONTROL = 5

    TABLE_RADIX_VALUE_HEX = 'Hex'
    TABLE_RADIX_VALUE_DEC = 'Dec'
    TABLE_RADIX_VALUE_BIN = 'Bin'

    TABLE_TYPE_VALUE_IN = 'Probe'
    TABLE_TYPE_VALUE_OUT = 'Source'

    def __init__(self, spec: VioSpec, init_setting=None, parent=None, ui=None):
        super().__init__(spec, parent)


        # Default setting
        probes_in = []
        for name, width in self._spec.get_probe_ins():
            probes_in.append({
                'name': name,
                'radix': 'Hex'
            })

        probes_out = []
        for name, width, init_value in self._spec.get_probe_outs():
            probes_out.append({
                'name': name,
                'radix': 'Hex'
            })
        self._default_setting = {
            'probes_in': probes_in,
            'probes_out': probes_out
        }

        if not init_setting:
            init_setting = self._default_setting

        # Initial state
        self._context = VioGUIContext({
            'connected': False
        }, init_setting)

        self._cached_setting = init_setting
        self._controller = None

        self._probe_outs = []
        self.ui = Ui_VioDebug()
        self.ui.setupUi(self)
        self.setTitle(self._name)
        if DOCKED_GUI:
            self.setStyleSheet(
                "QGroupBox { color: rgb(0, 0, 0, 0); border: 0px;}")

        self.delegate = VioDebugDelegate(table=self.ui.tableWidget,
                                       style_edited=True)
        self.delegate.commit_index.connect(self.on_commit_index)
        self.ui.tableWidget.setEnabled(False)

        self.ui.tableWidget.setEditTriggers(
            QAbstractItemView.AllEditTriggers)
        self.ui.tableWidget.setItemDelegate(self.delegate)
        self.ui.tableWidget.verticalHeader().setSectionsMovable(True)
        self.ui.tableWidget.horizontalHeader().setSectionResizeMode(4, QHeaderView.ResizeMode.Stretch)
        #self.ui.tableWidget.horizontalHeader().setSectionResizeMode(5, QHeaderView.ResizeMode.ResizeToContents)
        self.ui.tableWidget.horizontalHeader().setStretchLastSection(1)
        self.delegate.control_type_updated.connect(self.on_update_control)

        self._context.state_changed.connect(self.on_debug_core_state_changed)
        self._context.setting_changed.connect(self.on_debug_core_setting_changed)
        self._connect_signal_slot()

    def _connect_signal_slot(self):
        """
        Connect signal
        """
        self._show_help_clicked = lambda: show_help_manual('VIO_RUN')
        self._probe_table_changed = lambda item: self.on_table_item_changed(self.ui.tableWidget, item)

        self.delegate.commit_index.connect(self.on_commit_index)
        self.ui.pb_help.clicked.connect(self._show_help_clicked)
        self.ui.tableWidget.itemChanged.connect(self._probe_table_changed)

    def _disconnect_signal_slot(self):
        """
        Disconnect signal
        """
        self.delegate.commit_index.disconnect(self.on_commit_index)
        self.ui.pb_help.clicked.disconnect(self._show_help_clicked)
        self.ui.tableWidget.itemChanged.disconnect(self._probe_table_changed)

    def on_debug_core_state_changed(self):
        """
        Update GUI according to VIO state
        """
        new_state = self._context.state
        self.ui.tableWidget.setEnabled(new_state['connected'])

    def on_debug_core_setting_changed(self):
        """
        Update GUI according to VIO setting
        """
        new_setting = self._context.setting

        fields_changed = []
        for new, old in zip(new_setting['probes_in'], self._cached_setting['probes_in']):
            if new != old:
                fields_changed.append(new)
        for new, old in zip(new_setting['probes_out'], self._cached_setting['probes_out']):
            if new != old:
                fields_changed.append(new)

        self._disconnect_signal_slot()
        for item in fields_changed:
            target_item = None
            for row in range(self.ui.tableWidget.rowCount()):
                target_item = self.ui.tableWidget.item(row, self.TABLE_COLUMN_NAME)
                if target_item is None:
                    continue
                name = target_item.data(Qt.EditRole)
                if name != item['name']:
                    continue
                target_item = self.ui.tableWidget.item(row, self.TABLE_COLUMN_RADIX)
                if target_item is None:
                    continue
                target_item.setData(Qt.EditRole, item['radix'])
        self._connect_signal_slot()

        self._cached_setting = new_setting

    def on_table_item_changed(self, table: QTableWidget, item: QTableWidgetItem):
        row = item.row()
        col = item.column()

        # We only interest in radix change
        if col != self.TABLE_COLUMN_RADIX:
            return

        radix = item.data(Qt.EditRole)
        target_item = table.item(row, self.TABLE_COLUMN_NAME)
        if target_item is None:
            return
        name = target_item.data(Qt.EditRole)
        target_item = table.item(row, self.TABLE_COLUMN_TYPE)
        if target_item is None:
            return
        probe_type = target_item.data(Qt.EditRole)
        LOGGER.debug('item = {}, row = {}, radix = {}, name = {}, probe_type = {}'.format(
            item, row, radix, name, probe_type))
        # Update setting and dispatch change in context
        key = None
        if probe_type == self.TABLE_TYPE_VALUE_IN:
            key = 'probes_in'
        elif probe_type == self.TABLE_TYPE_VALUE_OUT:
            key = 'probes_out'
        else:
            return

        new_setting = self._context.setting
        for p_setting in new_setting[key]:
            if p_setting['name'] == name:
                p_setting['radix'] = radix
        self._context.dispatch_setting_change(key, new_setting[key])

    def on_debug_core_pre_connect(self):
        if self._context.state['connected']:
            return

    def on_debug_core_post_connect(self, dbg_sesh: DebugSession):
        if self._context.state['connected']:
            return

        self._controller = QVioController(dbg_sesh, self._spec.get_name())
        self._controller.data_changed.connect(self.on_data_changed)
        self._controller.probe()
        self._controller.error_received.connect(self.on_debug_core_error)

        new_state = dict(self._context.state)
        new_state['connected'] = True
        self._context.dispatch_state_change(new_state)

    def on_debug_core_pre_disconnect(self):
        LOGGER.debug("IN")
        if not self._context.state['connected']:
            return

        if self._controller:
            self._controller.data_changed.disconnect(self.on_data_changed)
            self._controller.error_received.disconnect(self.on_debug_core_error)
            self._controller = None
        LOGGER.debug("OUT")

    def on_debug_core_post_disconnect(self):
        """
        Restore default UI state
        """
        if not self._context.state['connected']:
            return

        new_state = dict(self._context.state)
        new_state['connected'] = False
        self._context.dispatch_state_change(new_state)

    def on_debug_core_error(self, error: Exception):
        self.error_received.emit(error)

    def on_commit_index(self, index: QModelIndex):
        LOGGER.debug('index.row, col={}, {}'.format(index.row(), index.column()))
        if self._probe_outs and index.column() == 4:
            name_list = [n for n, _, _ in self._spec.get_probe_outs()]
            name = index.sibling(index.row(), 0).data(Qt.EditRole)
            width = index.sibling(index.row(), 2).data(Qt.EditRole)
            radix = index.sibling(index.row(), 3).data(Qt.EditRole)
            val = index.sibling(index.row(), 4).data(Qt.EditRole)
            outid = name_list.index(name)
            new_text = change_int_base(val, RADIX_TO_BASE[radix], 2, width)
            self._probe_outs[outid] = BitSequence(new_text, msb=True)

            if not DOCKED_GUI:
                self.commit_probe.emit(self._spec.get_name(), self._probe_outs)
            else:
                self.update_hw(self._probe_outs)

    def on_update_control(self, index: QModelIndex):
        if not index.isValid():
            return
        # Trigger the delegate to re-create the editor based on new type
        item = self.ui.tableWidget.item(index.row(), 4)
        if item:
            self.ui.tableWidget.closePersistentEditor(item)
            self.ui.tableWidget.openPersistentEditor(item)

    def _insert_row(self, name: str, width: int, is_input: bool = True):
        row = self.ui.tableWidget.rowCount()
        self.ui.tableWidget.insertRow(row)
        #(':')(u'↕')(u'⇕')(u'⇵')(u'⇳')(u'⬍')
        item = QTableWidgetItem(u'↕')
        self.ui.tableWidget.setVerticalHeaderItem(row, item)

        item = table_widget_item(text=name, no_drop=True, no_edit=True)
        self.ui.tableWidget.setItem(row, 0, item)
        dir_str = 'IN' if is_input else 'OUT'
        type_str = 'Probe' if is_input else 'Source'
        #item = table_widget_item(text=dir_str, no_drop=True, no_edit=True)
        item = table_widget_item(text=type_str,
                                 user_value=dir_str,
                                 no_drop=True,
                                 no_edit=True)
        self.ui.tableWidget.setItem(row, 1, item)
        item = table_widget_item(value=width, no_drop=True, no_edit=True)
        self.ui.tableWidget.setItem(row, 2, item)
        if width == 1:
            item = table_widget_item(value='Bin', no_drop=True, no_edit=True)
        else:
            item = table_widget_item(value='Hex', no_drop=True)
        self.ui.tableWidget.setItem(row, 3, item)
        self.ui.tableWidget.openPersistentEditor(item)
        item = table_widget_item(value='0', no_drop=True, no_edit=is_input)
        self.ui.tableWidget.setItem(row, 4, item)
        # if not is_input and width == 1:
            # self.ui.tableWidget.openPersistentEditor(item)

        if not is_input:
            item = table_widget_item(value='Hex', no_drop=True, no_edit=False)
        else:
            item = table_widget_item(no_drop=True, no_edit=True, gray=True)
        self.ui.tableWidget.setItem(row, 5, item)

        if not is_input:
            self.ui.tableWidget.openPersistentEditor(item)

        if not is_input:
            item = table_widget_item(no_drop=True, no_edit=True, gray=True)
        else:
            item = table_widget_item(no_drop=False, no_edit=False)
        self.ui.tableWidget.setItem(row, 6, item)

    def load_profile(self, profile):
        LOGGER.info('VioDebugWidget _insert_row() foreach spec.get_probes')
        for n, w, v in self._spec.get_probes():  # pylint: disable=C0103
            self._insert_row(n, w, is_input=(v is None))
        self.ui.tableWidget.clearFocus()
        session = profile.get_core_session(self._name)
        if session:
            self._context.dispatch_setting_overwrite(session['setting'])

    def save_profile(self, profile):
        LOGGER.debug("Called")
        profile.set_core_session(name=self._name,
                                 session=dict(self._context.setting))
        LOGGER.debug('Updated = {}'.format(profile))

    def _update_row(self, row: int, bseq: BitSequence):
        table = self.ui.tableWidget
        bitw = table.item(row, 2).data(Qt.EditRole)
        radix = table.item(row, 3).data(Qt.EditRole)
        base = RADIX_TO_BASE[radix]
        new_text = change_int_base(repr(bseq), 2, base, bitw)
        table.item(row, 4).setData(Qt.EditRole, new_text)
        compared_value = table.item(row, 6).data(Qt.EditRole)
        compared_type = table.item(row, 1).data(Qt.EditRole)
        if compared_type == 'Probe' and not table.item(row,6).isSelected():
            if compared_value:
                if compared_value == new_text:
                    table.item(row, 6).setBackground(QColor(130,232,133))
                else:
                    table.item(row, 6).setBackground(QColor(253,132,132))
            else:
                table.item(row, 6).setBackground(QBrush(Qt.transparent))

    def update_gui(self, probe_ins: List[BitSequence],
                   probe_outs: List[BitSequence]):
        self._probe_outs = probe_outs
        idx_i = idx_o = 0
        for row, (_, _, init) in enumerate(self._spec.get_probes()):
            if init is None:
                bseq = probe_ins[idx_i]
                idx_i += 1
            else:
                bseq = probe_outs[idx_o]
                idx_o += 1
            self._update_row(row, bseq)

    def update_hw(self, probes_out):
        if self._context.state['connected']:
            self._controller.commit(probes_out)

    def on_data_changed(self, probes_in):
        """
        Update the GUI according to probes_in value
        """
        probes_out = self._controller.get_property('probe_out')
        self.update_gui(probes_in, probes_out)
        if self._context.state['connected']:
            self._controller.probe()

    def is_setting_changed(self):
        return self._context.is_setting_changed()


class LogicNAddProbesDialog(QDialog):
    """"""

    def __init__(self,
                 spec: LogicNSpec,
                 data_probe: bool,
                 trig_probe: bool,
                 parent=None):
        super().__init__(parent)
        self._spec = spec

        self.ui = Ui_LaAddProbes()
        self.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui.setupUi(self)
        self.setModal(True)
        for (name, width, _, pid) in self._spec.get_probes(data_en=data_probe,
                                                           trig_en=trig_probe):
            text = (name + '[{}:0]'.format(width - 1)) if width > 1 else name
            item = QListWidgetItem()
            item.setData(Qt.UserRole, (text, width, pid))
            item.setData(Qt.DisplayRole, text)
            self.ui.listWidget.insertItem(self.ui.listWidget.count(), item)
        self.ui.buttonBox.button(
            QDialogButtonBox.Ok).setEnabled(False)

        self.ui.listWidget.itemSelectionChanged.connect(
            self.on_selection_changed)

    def on_selection_changed(self):
        if len(self.ui.listWidget.selectedItems()) == 0:
            self.ui.buttonBox.button(
                QDialogButtonBox.Ok).setEnabled(False)
        else:
            self.ui.buttonBox.button(
                QDialogButtonBox.Ok).setEnabled(True)

    def on_probe_removed(self, pid):
        # Find the probe and set the flag
        list_widget = self.ui.listWidget
        for i in range(list_widget.count()):
            item = list_widget.item(i)
            _, _, probe_id = item.data(Qt.UserRole)
            if probe_id == pid:
                item.setFlags(item.flags() | Qt.ItemIsSelectable |
                               Qt.ItemIsEnabled)


class LogicNTriggerDelegate(QStyledItemDelegate):
    """"""

    def __init__(self, parent=None, is_vec: bool = False):
        super().__init__(parent)
        self._is_vec = is_vec

    @staticmethod
    def _validator_pattern(width: int, radix_base: int) -> str:
        LOGGER.debug('width = {}, radix_base = {}\n'.format(width, radix_base))
        pattern = ''
        if radix_base == 10:
            max_str = str(2**width - 1)
            pattern = ''.join(['[0-' + c + ']' for c in max_str])
            pattern += '$'
            nchar = len(max_str)
            if nchar > 1:
                pattern += '|[0-{}]?\\d{{1,{}}}$'.format(
                    int(max_str[0]) - 1, nchar - 1)
        elif radix_base == 2:
            pattern = '[0-1Xx]{{{0},{0}}}$'.format(width)
        elif radix_base == 16:
            remainder = width % 4
            pattern = {
                0: '',
                1: '[0-1Xx]',
                2: '[0-3Xx]',
                3: '[0-7Xx]',
            }.get(remainder)
            if width >= 4:
                if remainder != 0:
                    pattern += '?'
                pattern += '[0-9A-Fa-fXx]{{1,{}}}'.format(width // 4)
            pattern += '$'

        LOGGER.debug('pattern = {}\n'.format(pattern))
        return pattern

    def createEditor(self, parent: QWidget,
                     option: QStyleOptionViewItem,
                     index: QModelIndex):
        if index.column() == 1:
            editor = QComboBox(parent)
            if not self._is_vec:
                editor.addItem('==', 1)
                #editor.addItem('!= (not equal)', 2)
            else:
                # these uses same value as in compare unit rtl
                editor.addItem('==', 1)
                editor.addItem('!=', 2)
                editor.addItem('<', 3)
                editor.addItem('<=', 4)
                editor.addItem('>', 5)
                editor.addItem('>=', 6)
            #editor.activated.connect(self.emit_commitData)
            return editor
        if index.column() == 2:
            editor = QComboBox(parent)
            editor.addItem('Bin', 2)
            editor.addItem('Hex', 16)
            editor.addItem('Dec', 10)
            #editor.activated.connect(self.emit_commitData)
            return editor
        if index.column() == 3:
            if not self._is_vec:
                editor = QComboBox(parent)
                # these uses same value as in compare unit rtl
                editor.addItem('0 (logical zero)', 1)
                editor.addItem('1 (logical one)', 2)
                editor.addItem("X (don't care)", 3)
                editor.addItem('R (0-to-1 transition)', 4)
                editor.addItem('F (1-to-0 transition)', 5)
                editor.addItem('B (both transitions)', 6)
                editor.addItem('N (no transitions)', 7)
                #editor.activated.connect(self.emit_commitData)
            else:
                editor = super().createEditor(parent, option, index)
                width = index.data(Qt.UserRole)
                radix_base = index.sibling(index.row(), 2).data(Qt.UserRole)
                pattern = self._validator_pattern(width, radix_base)
                regex = QRegExp(pattern)
                validator = QRegExpValidator(regex, self)
                editor.setValidator(validator)
                editor.textEdited.connect(lambda txt: editor.setText(txt.upper()))
            return editor
        return super().createEditor(parent, option, index)

    def _use_combobox(self, index: QModelIndex):
        return (index.column() >= 1 and
                index.column() <= 2) or (index.column() == 3 and
                                         not self._is_vec)

    def setEditorData(self, editor: QObject, index: QModelIndex):
        if self._use_combobox(index):
            pattern = index.data(Qt.UserRole)
            cb_index = editor.findData(pattern)
            if cb_index >= 0:
                editor.setCurrentIndex(cb_index)
            LOGGER.debug('at {}, {}; UserRole data={}, cb_index={}'.format(
                         index.row(), index.column(), pattern, cb_index))
        else:
            super().setEditorData(editor, index)

    @staticmethod
    def _change_base(org_val: str, org_base: int, new_base: int,
                     width: int) -> str:
        LOGGER.debug('org_val={}, org_base={}, new_base={}, width={}'.format(org_val,
                     org_base, new_base, width))
        if org_base == new_base:
            return org_val

        if org_base == 16 and new_base == 2:
            hex2bin = {
                'X': 'XXXX',
                '0': '0000',
                '1': '0001',
                '2': '0010',
                '3': '0011',
                '4': '0100',
                '5': '0101',
                '6': '0110',
                '7': '0111',
                '8': '1000',
                '9': '1001',
                'A': '1010',
                'B': '1011',
                'C': '1100',
                'D': '1101',
                'E': '1110',
                'F': '1111'
            }
            new_val = ''.join([hex2bin[c] for c in org_val])
            return new_val[-width:]

        if org_base == 2 and new_base == 16:
            bin2hex = {
                'XXXX': 'X',
                '0000': '0',
                '0001': '1',
                '0010': '2',
                '0011': '3',
                '0100': '4',
                '0101': '5',
                '0110': '6',
                '0111': '7',
                '1000': '8',
                '1001': '9',
                '1010': 'A',
                '1011': 'B',
                '1100': 'C',
                '1101': 'D',
                '1110': 'E',
                '1111': 'F'
            }
            remainder = len(org_val) % 4
            nchar = ('0' if org_val[0] != 'X' else 'X') * (4 - remainder)
            in_val = nchar + org_val if remainder != 0 else org_val
            new_val = ''
            for i in range(0, len(in_val), 4):
                hex_ = bin2hex.get(in_val[i:i + 4])
                if not hex_:
                    break
                new_val += hex_
            print(
                'remainder={}, nchar={}, in_val={}, new_val={}, hex_={}'.format(
                    remainder, nchar, in_val, new_val, hex_))
            if not hex_:
                return '0' * (len(in_val) // 4)
            return new_val

        if new_base == 10:
            if 'X' in org_val:
                return '0'
            return str(int(org_val, org_base))

        if org_base == 10:
            if new_base == 2:
                return '{:0{w}b}'.format(int(org_val), w=width)
            if new_base == 16:
                return '{:0{w}X}'.format(int(org_val), w=(width - 1) // 4 + 1)

        raise ValueError

    def setModelData(self, editor: QObject,
                     model: QAbstractItemModel,
                     index: QModelIndex):
        if self._use_combobox(index):
            data = editor.currentData()
            text = editor.currentText()

            index_col3 = index.sibling(index.row(), 3)
            data_col3 = index_col3.data(Qt.EditRole)
            if (index.column() == 1) and data_col3 and (
                    'X' in data_col3) and data > 2:
                # Block <,<=,>,>= if there is X in value
                # TODO prompt message
                return

            # When change radix(col2), change value(col3)
            if index.column() == 2 and self._is_vec:
                org_base = index.data(Qt.UserRole)
                new_base = data
                #index_col3 = index.sibling(index.row(), 3)
                #org_val = index_col3.data(Qt.EditRole)
                org_val = data_col3
                width = index_col3.data(Qt.UserRole)
                new_val = self._change_base(org_val, org_base, new_base, width)
                LOGGER.debug('_change_base new_val={}'.format(new_val))
                model.setData(index_col3, new_val, Qt.EditRole)

            model.setData(index, data, Qt.UserRole)
            model.setData(index, text, Qt.EditRole)
            LOGGER.debug('at {}, {}; UserRole data={}'.format(index.row(),
                         index.column(), data))
            LOGGER.debug('at {}, {}; EditRole data={}'.format(index.row(),
                         index.column(), text))

        elif index.column() == 3:
            text = editor.text()
            index_col1 = index.sibling(index.row(), 1)
            data_col1 = index_col1.data(Qt.UserRole)
            if 'X' in text and data_col1 > 2:
                # Block <,<=,>,>= if there is X in value
                # TODO prompt message
                return
            super().setModelData(editor, model, index)
        else:
            super().setModelData(editor, model, index)

    def updateEditorGeometry(self, editor: QObject,
                             option: QStyleOptionViewItem,
                             index: QModelIndex):
        if self._use_combobox(index):
            editor.setGeometry(option.rect)
        else:
            super().updateEditorGeometry(editor, option, index)

    def emit_commitData(self):
        self.commitData.emit(self.sender())


def open_wave_viewer(gtkw_path: str, err_call: Callable[[Exception], None], repscript_path: str='', is_always_launch_new_viewer: bool=False):
    hwtc_util = HwToolsConfigUtil()
    repperiod = hwtc_util.read_setting(sect='debugger', opt='gtkwave_repperiod')

    try:
        env = os.environ.copy()
        env.pop('LD_LIBRARY_PATH', None)
        if not is_always_launch_new_viewer:
            return subprocess.Popen(['gtkwave', gtkw_path, f'--repscript={repscript_path}', f'--repperiod={repperiod}'], env=env)
        else:
            return subprocess.Popen(['gtkwave', gtkw_path], env=env)
        #['gtkwave', gtkw_path, '--rcvar', "'do_initial_zoom_fit'", 'yes'])
    except (OSError, ValueError) as exc:
        err_call(exc)

def gtkw_reload(err_call: Callable[[Exception], None]):
    try:
        return subprocess.run([
            'gconftool-2', '--type', 'string', '--set',
            '/com.geda.gtkwave/0/reload', '0'
        ])
    except (OSError, ValueError) as exc:
        err_call(exc)


class LogicNGUIContext(QObject):

    state_changed = pyqtSignal()
    setting_changed = pyqtSignal()

    def __init__(self, initial_state, initial_setting):
        super(LogicNGUIContext, self).__init__()
        self._logicN_state = copy.deepcopy(initial_state)

        # Inital settings
        self._default_setting = copy.deepcopy(initial_setting)
        self._logicN_setting = copy.deepcopy(initial_setting)
        """
        self._logicN_setting = {
            'probes' : [
                {
                    'trigger_setting': {
                        'trig_pttrn': None,
                        'trig_cmpr': None,
                        'trig_mask': None
                    },
                    'capture_setting': {
                        'cap_pttn': None,
                        'cap_cmpr': None,
                        'cap_mask': None
                    }
                }
            ] * len(self._spec.get_probes()),
            'num_window': 1,
            'window_depth': self._spec.get_depth(),
            'tu_pttrn': None,
            'cu_pttrn': None
        }
        """

    @property
    def state(self):
        return copy.deepcopy(self._logicN_state)

    @property
    def setting(self):
        return copy.deepcopy(self._logicN_setting)

    def is_setting_changed(self):
        print("default = {}".format(self._default_setting))
        print("current = {}".format(self._logicN_setting))
        return self._default_setting != self._logicN_setting

    @pyqtSlot(str, object)
    def dispatch_setting_change(self, setting_type, setting_value):
        """
        Update the internal setting
        """
        self._logicN_setting[setting_type] = setting_value
        self.setting_changed.emit()

    @pyqtSlot(object)
    def dispatch_setting_overwrite(self, new_setting):
        """
        Update the internal setting by overwriting the whole dict,
        Also update the default setting
        """
        self._default_setting = copy.deepcopy(new_setting)
        self._logicN_setting = copy.deepcopy(new_setting)
        self.setting_changed.emit()

    @pyqtSlot(object)
    def dispatch_state_change(self, new_state):
        """
        Update the internal state
        """
        self._logicN_state = {**self._logicN_state, **new_state}
        self.state_changed.emit()


class NewLogicNDebugWidget(AbstractDebugWidget):
    gtk_trig_pos = None
    is_continous_trig = False
    onstop = False
    error_received = pyqtSignal(object)
    parent = None

    TABLE_COLUMN_NAME = 0
    TABLE_COLUMN_OP = 1
    TABLE_COLUMN_RADIX = 2
    TABLE_COLUMN_VALUE = 3

    def __init__(self, spec: LogicNSpec , init_setting=None, parent=None):
        super(NewLogicNDebugWidget, self).__init__(spec, parent)
        self.parent = parent

        # Inital state
        self._logicN_state = {
            'state': 'IDLE',
            'sample_cnt': 0,
            'running': False,
            'connected': False
        }

        # Gernerate initial settings
        probe_settings = []
        for pname, pwidth, ptype, pid in self._spec.get_probes():
            probe_settings.append(
                {
                    'pid': pid
                }
            )

        # Inital settings
        self._default_setting = {
            'probes' : probe_settings,
            'num_window': 1,
            'window_depth': self._spec.get_depth(),
            'tu_pttrn': 'AND',
            'cu_pttrn': 'AND',
            'trig_pos': self._spec.get_depth() // 2,
            'capture_mode': LogicNCaptureMode.ALWAYS,
            'trigger_mode': LogicNTriggerMode.BASIC,
            'skip_count': 0
        }
        self._last_directory = os.getcwd()
        self._ctrl = None
        self._context = LogicNGUIContext(self._logicN_state, self._default_setting)
        self._cached_setting = self._default_setting

        self._wave_viewer = None
        self._standalone_wave_viewers = []
        self._is_always_launch_new_viewer = False
        self._addprobes_dialog = LogicNAddProbesDialog(spec=self._spec,
                                                       data_probe=False,
                                                       trig_probe=True,
                                                       parent=self)
        self._bit_trig_delegate = LogicNTriggerDelegate(parent=self,
                                                        is_vec=False)
        self._vec_trig_delegate = LogicNTriggerDelegate(parent=self,
                                                        is_vec=True)

        self._addcapture_dialog = LogicNAddProbesDialog(spec=self._spec,
                                                        data_probe=True,
                                                        trig_probe=False,
                                                        parent=self)
        self._bit_capt_delegate = LogicNTriggerDelegate(parent=self,
                                                        is_vec=False)
        self._vec_capt_delegate = LogicNTriggerDelegate(parent=self,
                                                        is_vec=True)
        self.ui = Ui_LaDebug()
        self.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui.setupUi(self)
        self.setTitle(self._name)
        self._state_le = [
            self.ui.le_state_idle, self.ui.le_state_wait, self.ui.le_state_post,
            self.ui.le_state_full
        ]
        self._widgets_to_freeze = [
            self.ui.le_filepath, self.ui.pb_filepath_select,
            self.ui.chk_overwrite_wave, self.ui.pb_add_trigger,
            self.ui.pb_rm_trigger, self.ui.cb_trigger_cndt,
            self.ui.tableWidget_trigger, self.ui.sb_trig_pos,
            self.ui.sb_num_window, self.ui.cb_data_depth,
            self.ui.cb_n_trig
        ]

        self._default_wave_path = os.path.join(
            os.getcwd(),
            self.title() + '_waveform' + '.vcd')

        for display_name, pattern in self.available_cmp_pattern:
            self.ui.cb_trigger_cndt.addItem(display_name, pattern)
            self.ui.cb_capture_cndt.addItem(display_name, pattern)
        self.tmp_continous_gtk_dir = tempfile.mkdtemp()
        self.tmp_continous_vcd_path = os.path.join(self.tmp_continous_gtk_dir,self.title() + 'tmp_continous.vcd')
        self.tmp_repscript_path = os.path.join(self.tmp_continous_gtk_dir,self.title() + 'tmp_repscript.tcl')
        self.tmp_mem_path = os.path.join(self.tmp_continous_gtk_dir,self.title() + 'tmp.mem')

        f = open(self.tmp_continous_vcd_path, 'w')
        f.close()
        f = open(self.tmp_mem_path, 'w')
        f.close()

        if os.name == 'nt':
            self.tmp_mem_path = re.sub(r'\\', '/', rf'{self.tmp_mem_path}')
        else:
            self.tmp_mem_path = self.tmp_mem_path

        script = f'set fp [open "{str(self.tmp_mem_path)}" r]\nset status [gets $fp]\nset trigpos [gets $fp]\nclose $fp\nif {{ $status == "1" }} {{\nset fp [open "{str(self.tmp_mem_path)}" w]\nif {{ $trigpos != "" }} {{\ngtkwave::setNamedMarker A $trigpos;\n}}\ngtkwave::reLoadFile;\nclose $fp\n}}\n'

        repscriptfile = open(str(self.tmp_repscript_path), 'w')
        repscriptfile.writelines(script)
        repscriptfile.close()

        # Setup GUI according to initial state
        self.ui.le_filepath.setText(self._default_wave_path)
        clr_but = self.ui.le_filepath.findChild(QToolButton)
        clr_but.clicked.connect(self.on_clearbutton_press)

        self.ui.pb_run.setEnabled(False)
        self.ui.pb_run_imdt.setEnabled(False)
        self.ui.pb_run_cont.setEnabled(False)
        self.ui.pb_stop.setEnabled(False)
        self.ui.widget_capture_sel.setEnabled(False)

        # Smallest window depth is 8 due to design limitation
        self.ui.cb_data_depth.clear()
        for i in range(int(math.log(self._spec.get_depth(), 2)), 3 - 1, -1):
            twos_pwr = 2**i
            self.ui.cb_data_depth.addItem(str(twos_pwr), twos_pwr)

        idx = self.ui.cb_data_depth.findText(str(self._spec.get_depth()))
        if idx != -1:
            self.ui.cb_data_depth.setCurrentIndex(idx)

        self.ui.sb_num_window.setMinimum(1)
        self.ui.sb_num_window.setMaximum(self._spec.get_depth() // 8)

        self.ui.pgb_seg_sample.setMaximum(self._spec.get_depth())
        self.ui.pgb_seg_count.setMaximum(1)
        self.ui.pgb_total_sample.setMaximum(self._spec.get_depth())

        self.on_logicN_state_changed()
        self._setup_signal_and_slot()

    @property
    def available_cmp_pattern(self):
        """
        Available global compare pattern
        """
        return [
            ("Global 'AND'", 'AND'),
            ("Global 'OR'", 'OR'),
            ("Global 'NAND'", 'NAND'),
            ("Global 'NOR'", 'NOR')
        ]

    @property
    def current_capture_mode(self):
        return self._context.setting['capture_mode']

    def _setup_signal_and_slot(self):
        self._connect_signal_slot()
        # Connect context with the controller
        self._context.state_changed.connect(self.on_logicN_state_changed)
        self._context.setting_changed.connect(self.on_logicN_setting_changed)

    def _connect_signal_slot(self):
        """
        Connect UI signals to corresponding handler
        Notice that we need to store the lambda function otherwise we failed to disconnect them
        """
        self._on_run = lambda: self.on_run(False)
        self._on_run_imdt = lambda: self.on_run(True)
        self._on_run_cont = lambda: self.on_run(False, True)
        self._show_help = lambda: show_help_manual('LA_RUN')
        self._tu_pttrn_changed = lambda text: self._context.dispatch_setting_change('tu_pttrn', [t[1] for t in self.available_cmp_pattern if t[0] == text][0])
        self._cu_pttrn_changed = lambda text: self._context.dispatch_setting_change('cu_pttrn', [t[1] for t in self.available_cmp_pattern if t[0] == text][0])
        self._n_changed = lambda value: self._context.dispatch_setting_change('skip_count', value)
        self._trig_pos_changed = lambda value: self._context.dispatch_setting_change('trig_pos', value)
        self._trigger_table_changed = lambda item: self.on_table_item_changed('trigger_setting', self.ui.tableWidget_trigger, item)
        self._cappture_table_changed = lambda item: self.on_table_item_changed('capture_setting', self.ui.tableWidget_capture, item)

        self.ui.pb_run.clicked.connect(self._on_run)
        self.ui.pb_run_imdt.clicked.connect(self._on_run_imdt)
        self.ui.pb_run_cont.clicked.connect(self._on_run_cont)
        self.ui.pb_stop.clicked.connect(self.on_stop)

        self.ui.le_filepath.textChanged.connect(self.on_filepath_changed)
        self.ui.le_filepath.editingFinished.connect(self.on_filepath_edited)
        self.ui.pb_filepath_select.clicked.connect(self.on_waveform_filepath)

        self.ui.pb_add_trigger.clicked.connect(self._addprobes_dialog.open)
        self._addprobes_dialog.finished.connect(self.on_add_trigger)
        self.ui.pb_rm_trigger.clicked.connect(self.on_remove_trigger)

        self.ui.pb_add_capture.clicked.connect(self._addcapture_dialog.open)
        self._addcapture_dialog.finished.connect(self.on_add_capture_cond)
        self.ui.pb_rm_capture.clicked.connect(self.on_remove_capture_cond)

        self.ui.pb_help.clicked.connect(self._show_help)
        self.ui.cb_data_depth.currentIndexChanged[str].connect(self.on_window_depth_changed)
        self.ui.sb_num_window.valueChanged[int].connect(self.on_number_of_window_changed)
        self.ui.cb_capture_mode.currentIndexChanged[str].connect(self.on_capture_mode_changed)

        self.ui.cb_trigger_cndt.currentIndexChanged[str].connect(self._tu_pttrn_changed)
        self.ui.cb_capture_cndt.currentIndexChanged[str].connect(self._cu_pttrn_changed)
        self.ui.sb_trig_pos.valueChanged.connect(self._trig_pos_changed)
        self.ui.tableWidget_trigger.itemChanged.connect(self._trigger_table_changed)
        self.ui.tableWidget_capture.itemChanged.connect(self._cappture_table_changed)
        self.ui.cb_n_trig.valueChanged.connect(self._n_changed)


    def _disconnect_signal_slot(self):
        """
        Disconnect handler from UI signals to prevent recurrsive trigger while we updating the UI by code
        """
        self.ui.pb_run.clicked.disconnect(self._on_run)
        self.ui.pb_run_imdt.clicked.disconnect(self._on_run_imdt)
        self.ui.pb_run_cont.clicked.disconnect(self._on_run_cont)
        self.ui.pb_stop.clicked.disconnect(self.on_stop)

        self.ui.le_filepath.textChanged.disconnect(self.on_filepath_changed)
        self.ui.le_filepath.editingFinished.disconnect(self.on_filepath_edited)
        self.ui.pb_filepath_select.clicked.disconnect(self.on_waveform_filepath)

        self.ui.pb_add_trigger.clicked.disconnect(self._addprobes_dialog.open)
        self._addprobes_dialog.finished.disconnect(self.on_add_trigger)
        self.ui.pb_rm_trigger.clicked.disconnect(self.on_remove_trigger)

        self.ui.pb_add_capture.clicked.disconnect(self._addcapture_dialog.open)
        self._addcapture_dialog.finished.disconnect(self.on_add_capture_cond)
        self.ui.pb_rm_capture.clicked.disconnect(self.on_remove_capture_cond)

        self.ui.pb_help.clicked.disconnect(self._show_help)
        self.ui.cb_data_depth.currentIndexChanged[str].disconnect(self.on_window_depth_changed)
        self.ui.sb_num_window.valueChanged[int].disconnect(self.on_number_of_window_changed)
        self.ui.cb_capture_mode.currentIndexChanged[str].disconnect(self.on_capture_mode_changed)

        self.ui.cb_trigger_cndt.currentIndexChanged[str].disconnect(self._tu_pttrn_changed)
        self.ui.cb_capture_cndt.currentIndexChanged[str].disconnect(self._cu_pttrn_changed)
        self.ui.sb_trig_pos.valueChanged.disconnect(self._trig_pos_changed)
        self.ui.tableWidget_trigger.itemChanged.disconnect(self._trigger_table_changed)
        self.ui.tableWidget_capture.itemChanged.disconnect(self._cappture_table_changed)
        self.ui.cb_n_trig.valueChanged.disconnect(self._n_changed)

    def _display_progress(self, in_prog: int, num_window: int, window_depth: int):
        window_used = in_prog // window_depth
        self.ui.pgb_seg_count.setValue(window_used)
        self.ui.label_seg_count.setText('Segment {} of {}'.format(
            window_used, num_window))

        segment_sample = in_prog % window_depth
        self.ui.pgb_seg_sample.setValue(segment_sample)
        self.ui.label_seg_sample.setText('Segment sample {} of {}'.format(
            segment_sample, window_depth))

        self.ui.pgb_total_sample.setValue(in_prog)
        self.ui.label_total_sample.setText('Total sample {} of {}'.format(
            in_prog, num_window * window_depth))

    def _get_selected_probe(self, dialog: LogicNAddProbesDialog):
        """
        Return selected probe as tuple (name, width, pid)
        """
        selected_list = dialog.ui.listWidget.selectedItems()
        if not selected_list:
            return None

        list_item = selected_list[0]
        # list_item.setFlags(list_item.flags() & ~Qt.ItemIsSelectable &
        #                     ~Qt.ItemIsEnabled)
        pname, pwidth, pid = list_item.data(Qt.UserRole)
        return pname, pwidth, pid

    def _add_probe_to_table(self, probe: Tuple[str, int, int],
                            table: QTableWidget,
                            bit_delegate: LogicNTriggerDelegate,
                            vect_delegate: LogicNTriggerDelegate,
                            init_value: Tuple[int, int, str]=None ) -> None:
        """
        Add probe to able

        Arugment:
            probe: Tuple(name, width, pid),
            init_value: Tuple(pattern, radix, compare_value)
        Special Requirement:
            probe name should be xxx[3:0] for bus
        """
        pname, pwidth, pid = probe
        if pwidth < 1:
            raise ValueError('Probe width should be positive integer')

        if pwidth > 1:
            if not pname.endswith('[{}:0]'.format(pwidth-1)):
                raise ValueError("Probe name should be in formation of name[width-1:0]")

        delegate = bit_delegate if pwidth == 1 else vect_delegate
        row = table.rowCount()
        table.insertRow(row)
        table.setItemDelegateForRow(row, delegate)
        LOGGER.debug("Add, pid = {}, pwidth = {}, pname = {}".format(pid, pwidth, pname))

        if init_value is None:
            init_value = (1, 2, '0' * pwidth)
        LOGGER.debug("Add, init_value = {}".format(init_value))
        item0 = table_widget_item(text=pname,
                                user_value=(pid, pwidth),
                                no_edit=True)
        item1 = table_widget_item(user_value=init_value[0])
        item2 = table_widget_item(user_value=init_value[1])
        arg3 = {
            'user_value': init_value[0]
        } if pwidth == 1 else {
            'text': init_value[2],
            'user_value': pwidth
        }
        item3 = table_widget_item(**arg3)
        suffix = '[{}:0]'.format(pwidth - 1)
        name = 'probe' + str(pid)
        text = name + suffix if pwidth > 1 else name
        item4 = table_widget_item(text=text, no_edit=True)
        table.setItem(row, 0, item0)
        table.setItem(row, 1, item1)
        table.setItem(row, 2, item2)
        table.setItem(row, 3, item3)
        table.setItem(row, 4, item4)

        # Caution! setCurrentItem needed for currentChanged to closeEditor
        key_return = QKeyEvent(QEvent.KeyPress, Qt.Key_Return,
                                     Qt.NoModifier)
        table.setCurrentItem(item1)
        table.editItem(item1)
        editor = table.indexWidget(table.indexFromItem(item1))
        QApplication.sendEvent(editor, key_return)
        table.setCurrentItem(item2)
        table.editItem(item2)
        editor = table.indexWidget(table.indexFromItem(item2))
        QApplication.sendEvent(editor, key_return)
        table.setCurrentItem(item3)
        table.editItem(item3)
        editor = table.indexWidget(table.indexFromItem(item3))
        QApplication.sendEvent(editor, key_return)

        if pwidth == 1:
            item2.setFlags(item2.flags() & ~Qt.ItemIsEditable)
        for r in range(table.rowCount()):
            for c in range(table.columnCount()):
                target_item = table.item(r, c)
                if target_item:
                    LOGGER.debug("row = {}, col = {}, user_data = {}, edit_data = {}".format(r, c,
                        target_item.data(Qt.UserRole), target_item.data(Qt.EditRole)))
        LOGGER.debug("Add, OUT")

    def _remove_probe_from_table(self, table):
        for i in range(table.rowCount() - 1, -1, -1):
            table.removeRow(i)

        table_remove_selected_toprow(table)

    def _update_control_widgets(self, is_running: bool):
        """
        Update the enable status for control widgets according to the state (is_running or not)
        """
        if self.is_continous_trig:
            self.ui.pb_run.setEnabled(False)
            self.ui.pb_run_imdt.setEnabled(False)
            self.ui.pb_run_cont.setEnabled(False)
            self.ui.pb_stop.setEnabled(True)
            self.ui.le_filepath.setEnabled(False)
            return

        if not is_running:
            self.ui.pb_run.setEnabled(True)
            self.ui.pb_run_imdt.setEnabled(True)
            self.ui.pb_run_cont.setEnabled(True)
            self.ui.pb_stop.setEnabled(False)
            self.ui.le_filepath.setEnabled(True)
            widgets_setEnabled(self._widgets_to_freeze, True)
            self.ui.cb_capture_mode.setEnabled(
                self._spec.get_attr('capture_control', default=False))
            self.ui.widget_capture_sel.setEnabled(self._context.setting['capture_mode'] == LogicNCaptureMode.BASIC)
        else:
            self.ui.pb_run.setEnabled(False)
            self.ui.pb_run_imdt.setEnabled(False)
            self.ui.pb_run_cont.setEnabled(False)
            self.ui.pb_stop.setEnabled(True)
            self.ui.le_filepath.setEnabled(True)
            widgets_setEnabled(self._widgets_to_freeze, False)
            self.ui.cb_capture_mode.setEnabled(False)
            self.ui.widget_capture_sel.setEnabled(False)

    def _display_state(self, state: str):
        color = lambda s: 'white' if state != s else 'rgb(0, 255, 0)'
        # TODO PRE_TRIG
        state_str = ['IDLE', 'WAIT_TRIG', 'POST_TRIG', 'FULL']
        for le, s_str in zip(self._state_le, state_str):
            le.setStyleSheet("background-color: {};".format(color(s_str)))

    def on_close(self):
        try:
            for wave_viewer in self._standalone_wave_viewers:
                if wave_viewer is not None:
                    wave_viewer.terminate()

            self._standalone_wave_viewers.clear()
            if self._wave_viewer and self._wave_viewer.poll() is None:
                self._wave_viewer.terminate()
            self._wave_viewer = None

            if os.path.isdir(self.tmp_continous_gtk_dir):
                shutil.rmtree(self.tmp_continous_gtk_dir)

        except Exception as exc:
            # The application is closing, catch everything when close
            pass

    def on_table_item_changed(self, setting_key: str, table: QTableWidget, item: QTableWidgetItem) -> None:
        """
        Update the context according to the table item,
        Pair with _add_probe_to_table, as data stored in the table as user data
        """
        new_setting = self._context.setting
        LOGGER.debug('table = {}, item = {}, row = {}, col = {}, user_data = {}, edit_data = {}'.format(
                    table, item, item.row(), item.column(),
                    item.data(Qt.UserRole), item.data(Qt.EditRole)))

        row = item.row()
        # Test the row is inserted and initialized
        for c in range(table.columnCount()):
            target_item = table.item(row, c)
            if not target_item:
                LOGGER.debug("OUT")
                return


        for r in range(table.rowCount()):
            for c in range(table.columnCount()):
                target_item = table.item(r, c)
                if target_item:
                    LOGGER.debug("row = {}, col = {}, user_data = {}, edit_data = {}".format(r, c,
                        target_item.data(Qt.UserRole), target_item.data(Qt.EditRole)))

        # Get pid, pwidth from the 0 th col
        col = 0
        target_item = table.item(row, col)
        pid, pwidth = target_item.data(Qt.UserRole)
        LOGGER.debug("pid = {}, pwidth = {}".format(pid, pwidth))

        # Calculate the pttrn, cmpr, mask
        cmpr, mask = 0, (1 << pwidth) - 1
        raw_cmpr = 0
        radix_base = 2
        if pwidth == 1:
            pttrn_col = 3
        else:
            pttrn_col = 1
            raw_cmpr = table.item(row, 3).data(Qt.EditRole)
            radix_base = table.item(row, 2).data(Qt.UserRole)
        pttrn = table.item(row, pttrn_col).data(Qt.UserRole)
        LOGGER.debug('pid = {}, pwidth = {}, pttrn = {}, cmpr = {}, radix = {}'.format(pid, pwidth, pttrn, raw_cmpr, radix_base))

        # Finally, dispatch change in context
        updated_probes = list(self._context.setting['probes'])
        for p in updated_probes:
            if p.get('pid', -1) == pid:
                p[setting_key] = {
                    'pttrn': pttrn,
                    'value': raw_cmpr,
                    'radix': radix_base,
                }

        self._context.dispatch_setting_change('probes', updated_probes)

    def on_number_of_window_changed(self, num_win: int):
        new_window_depth = self._context.setting['window_depth']
        total_sample = num_win * self._context.setting['window_depth']
        while total_sample > self._spec.get_depth():
            new_window_depth = new_window_depth // 2
            total_sample = num_win * new_window_depth

        new_trig_pos = self._context.setting['trig_pos']
        if new_trig_pos > new_window_depth:
            new_trig_pos = new_window_depth - 1

        self._context.dispatch_setting_change('window_depth', new_window_depth)
        self._context.dispatch_setting_change('num_window', num_win)
        self._context.dispatch_setting_change('trig_pos', new_trig_pos)


    def on_window_depth_changed(self, depth: int):
        new_depth = int(depth)
        new_num_window = self._context.setting['num_window']
        new_trig_pos = self._context.setting['trig_pos']
        max_num_window = self._spec.get_depth() // new_depth
        if new_num_window > max_num_window:
            new_num_window = max_num_window

        if new_trig_pos > new_depth:
            new_trig_pos = new_depth - 1

        self._context.dispatch_setting_change('window_depth', new_depth)
        self._context.dispatch_setting_change('num_window', new_num_window)
        self._context.dispatch_setting_change('trig_pos', new_trig_pos)

    def on_logicN_setting_changed(self):
        new_setting = self._context.setting
        LOGGER.debug('new_setting = {}'.format(pprint.pformat(new_setting)))

        # Check the setting is valid
        if not all(key in ['probes', 'num_window', 'window_depth', 'tu_pttrn',
                   'cu_pttrn', 'trig_pos', 'capture_mode', 'trigger_mode', 'skip_count']
                   for key in list(new_setting.keys())):
            LOGGER.warning("Invalid setting, extra key found")
            return

        # Detect setting change, TODO: Better way?
        fields_changed = []
        for key in new_setting.keys():
            if self._cached_setting == None:
                fields_changed.append(key)
            elif new_setting[key] != self._cached_setting[key]:
                fields_changed.append(key)

        self._disconnect_signal_slot()
        LOGGER.debug('fields = {}'.format(fields_changed))
        LOGGER.debug('DEBUG, new_setting = {}, cached_setting = {}'.format(new_setting['probes'], self._cached_setting['probes']))
        # Update accordingly
        for field in fields_changed:
            if field == 'capture_mode':
                idx = self.ui.cb_capture_mode.findText(new_setting['capture_mode'])
                if idx != -1:
                    self.ui.cb_capture_mode.setCurrentIndex(idx)
                self.ui.widget_capture_sel.setEnabled(new_setting['capture_mode'] == LogicNCaptureMode.BASIC)
            elif field == 'trigger_mode':
                pass
            elif field == 'tu_pttrn':
                idx = self.ui.cb_trigger_cndt.findData(new_setting['tu_pttrn'], Qt.UserRole)
                if idx != -1:
                    self.ui.cb_trigger_cndt.setCurrentIndex(idx)
            elif field == 'cu_pttrn':
                idx = self.ui.cb_capture_cndt.findData(new_setting['cu_pttrn'], Qt.UserRole)
                if idx != -1:
                    self.ui.cb_capture_cndt.setCurrentIndex(idx)
            elif field == 'trig_pos':
                self.gtk_trig_pos = self.ui.sb_trig_pos.value()
                self.ui.sb_trig_pos.setValue(new_setting['trig_pos'])
            elif field == 'num_window' or field == 'window_depth':
                self.ui.sb_num_window.setValue(new_setting['num_window'])
                idx = self.ui.cb_data_depth.findText(str(new_setting['window_depth']))
                if idx != -1:
                    self.ui.cb_data_depth.setCurrentIndex(idx)
                self.ui.pgb_seg_sample.setMaximum(new_setting['window_depth'])
                self.ui.pgb_seg_count.setMaximum(new_setting['num_window'])
                self.ui.pgb_total_sample.setMaximum(new_setting['num_window'] * new_setting['window_depth'])

                self.ui.label_trig_pos_range.setText('[0 - {}]'.format(new_setting['window_depth'] - 1))
                self.ui.sb_trig_pos.setMinimum(0)
                self.ui.sb_trig_pos.setMaximum(new_setting['window_depth'] - 1)
            elif field == 'skip_count':
                idx = self.ui.cb_n_trig.value()
                self.ui.cb_n_trig.setValue(idx)

            elif field == 'probes':

                def apply_probe_setting_to_table(probe_dialog: LogicNAddProbesDialog, table: QTableWidget, setting: Dict[str, Any]):
                    if setting is None:
                        raise ValueError('Invalid setting, expecting dictionary')
                    if not all(key in ['pttrn', 'value', 'radix'] for key in setting.keys()):
                        raise ValueError("Invalid setting dictionary, missing required keys")

                    # Check added to table or not
                    in_table = False
                    for row in range(table.rowCount()):
                        target_item = table.item(row, self.TABLE_COLUMN_NAME)
                        if target_item is None:
                            continue
                        (pid, _) = target_item.data(Qt.UserRole)
                        if pid != p['pid']:
                            continue
                        in_table = True
                        # Update the row data
                        for key, column in [('pttrn', self.TABLE_COLUMN_OP),
                                            ('value', self.TABLE_COLUMN_VALUE),
                                            ('radix', self.TABLE_COLUMN_RADIX)]:
                            target_item = table.item(row, column)
                            if target_item is None:
                                break
                            # TODO: This part only for API mode to set the data in model, skipped normal GUI mode
                            # LOGGER.debug("setData, target_item = {}, row = {}, column = {}, key = {}".format(target_item, row, column, key))
                            # target_item.setData(Qt.EditRole, p.get(key))

                    LOGGER.debug("in_table = {}".format(in_table))
                    if not in_table:
                        p_obj = self._spec.get_probe(p['pid'])
                        pwidth = p_obj.width
                        pname = '{}[{}:0]'.format(p_obj.name, pwidth - 1) if pwidth > 1 else p_obj.name
                        # Mark the probe as selected such that user cannot choose in dialog
                        probe_list_widget = probe_dialog.listWidget
                        for row in range(probe_list_widget.count()):
                            target_item = probe_list_widget.item(row)
                            if target_item is None:
                                continue
                            user_data = target_item.data(Qt.UserRole)
                            if user_data == (pname, pwidth, p['pid']):
                                target_item.setFlags(target_item.flags() & ~Qt.ItemIsSelectable &
                                                    ~Qt.ItemIsEnabled)
                        LOGGER.debug('apply_probe_setting_to_table')
                        self._add_probe_to_table(probe=(pname, pwidth, p['pid']),
                                                    table=table,
                                                    bit_delegate=self._bit_trig_delegate,
                                                    vect_delegate=self._vec_trig_delegate,
                                                    init_value=(setting['pttrn'],
                                                                setting['radix'],
                                                                setting['value']))
                    # We don't handle the case where probes are deleted from the table, TODO


                for p in new_setting['probes']:
                    trigger_setting = p.get('trigger_setting')
                    capture_setting = p.get('capture_setting')
                    if trigger_setting is not None:
                        apply_probe_setting_to_table(probe_dialog=self._addprobes_dialog.ui,
                                                     table=self.ui.tableWidget_trigger,
                                                     setting=trigger_setting)

                    if capture_setting is not None:
                        apply_probe_setting_to_table(probe_dialog=self._addcapture_dialog.ui,
                                                     table=self.ui.tableWidget_capture,
                                                     setting=capture_setting)

            # Other field dont need to update
        self._connect_signal_slot()

        # Apply data back to gui (disconnect those slot first, prevent recursive trigger)?
        self._cached_setting = copy.deepcopy(new_setting)

    def on_clearbutton_press(self):
        self.ui.le_filepath.setText('')
        self.ui.le_filepath.setFocus()

    def on_logicN_state_changed(self):
        """
        Update the GUI when LogicN state changed.
        Dataflow:
            HW -> Context -> GUI
        """

        new_state = self._context.state
        LOGGER.debug('new_state = {}'.format(new_state))

        try:
            self.on_debug_core_state_updated({
                'status': new_state['state'],
                'sample_cnt': new_state['sample_cnt'],
                'running': new_state['running'],
                'connected': new_state['connected']
            })

            if not self.onstop:
                if new_state['state'] == 'FULL' and new_state['running']:
                    self.load_result()
                    return

                if self.is_continous_trig:
                    if new_state['state'] == 'IDLE' and not new_state['running']:
                        self.on_run(cont=True)
                    return

            if self.onstop:
                if new_state['running']:
                    self.go_idle()

                if self.is_continous_trig:
                    if new_state['state'] == 'IDLE' and not new_state['running']:
                        self.parent.ui.actionAlwaysLaunchNewWaveWindow.setEnabled(True)
                        self.is_continous_trig = False
                        self._update_control_widgets(False)


        except (TypeError, RuntimeError) as ex:
            # Reset the GUI state on error
            self.on_debug_core_state_updated({
                'status': 'IDLE',
                'sample_cnt': 0,
                'running': False,
                'connected': False
            })
            # Trigger Main Application for disconnect
            self.error_received.emit(str(ex))

        # Cached the state
        self._logicN_state = new_state

    def on_run(self, imdt: bool = False, cont: bool = False) -> None:
        self.onstop = False
        if not self.ui.le_filepath.text() or os.path.isdir(self.ui.le_filepath.text()):
            self.ui.le_filepath.setFocus()
            QMessageBox.warning(self, "Efinity Debugger",
                        "Invalid Waveform Path. Debugger would not run")
            return

        if cont:
            self.is_continous_trig = True
            self._is_continous_trig = True
            self.parent.ui.actionAlwaysLaunchNewWaveWindow.setEnabled(False)
        else:
            self.is_continous_trig = False
            self._is_continous_trig = False
            #self.parent.ui.actionAlwaysLaunchNewWaveWindow.setEnabled(True)


        table = self.ui.tableWidget_trigger
        if table.rowCount() <=  0 and not imdt:
            QMessageBox.warning(self, "Run Trigger",
                "No trigger condition is set. Set some trigger condition or the logic analyser will never trigger.")
            return

        new_state = dict(self._context.state)
        new_state['running'] = True
        self._context.dispatch_state_change(new_state)

        setting = self._context.setting
        self._ctrl.run(setting, imdt)


    def on_stop(self):
        self.onstop = True
        self._ctrl.stop()
        #self._update_control_widgets(False)
        #self.is_continous_trig = False


    def on_add_trigger(self, result: int):
        if result == QDialog.Accepted:
            probe = self._get_selected_probe(self._addprobes_dialog)
            if probe:
                name, width, pid = probe
                new_setting = copy.deepcopy(self._context.setting)
                LOGGER.debug('Before dispatch, old = {}'.format(self._context.setting))
                updated_probes = list(new_setting['probes'])
                for p in updated_probes:
                    if p.get('pid', -1) == pid:
                        p['trigger_setting'] = {
                            'pttrn': 1,
                            'value': '0' * width,
                            'radix': 2,
                        }
                LOGGER.debug('After dispatch, old = {}'.format(self._context.setting))
                self._context.dispatch_setting_change('probes', updated_probes)
        self._addprobes_dialog.ui.listWidget.clearSelection()

    def on_remove_trigger(self):
        table = self.ui.tableWidget_trigger
        selected = table.selectedRanges()
        if not selected:
            return
        row = selected[0].topRow()
        col = selected[0].leftColumn()

        (pid, pwidth) = table.item(row, col).data(Qt.UserRole)
        self._addprobes_dialog.on_probe_removed(pid)
        self._remove_probe_from_table(table)

        updated_probes = list(self._context.setting['probes'])
        for p in updated_probes:
            if p.get('pid', -1) == pid:
                probe_setting = p.get('trigger_setting', None)
                if probe_setting is not None:
                    del p['trigger_setting']
        self._context.dispatch_setting_change('probes', updated_probes)

    def on_add_capture_cond(self, result: int):
        if result == QDialog.Accepted:
            probe = self._get_selected_probe(self._addcapture_dialog)
            if probe:
                self._add_probe_to_table(probe=probe, table=self.ui.tableWidget_capture,
                                         bit_delegate=self._bit_capt_delegate,
                                         vect_delegate=self._vec_capt_delegate)
        self._addcapture_dialog.ui.listWidget.clearSelection()

    def on_remove_capture_cond(self):
        table = self.ui.tableWidget_capture
        selected = table.selectedRanges()
        if not selected:
            return
        row = selected[0].topRow()
        col = selected[0].leftColumn()

        (pid, pwidth) = table.item(row, col).data(Qt.UserRole)
        self._addcapture_dialog.on_probe_removed(pid)
        self._remove_probe_from_table(table)

        updated_probes = list(self._context.setting['probes'])
        for p in updated_probes:
            if p.get('pid', -1) == pid:
                probe_setting = p.get('capture_setting', None)
                if probe_setting is not None:
                    del p['capture_setting']
        self._context.dispatch_setting_change('probes', updated_probes)

    def on_capture_mode_changed(self, mode: str):
        new_mode = mode
        if new_mode != LogicNCaptureMode.ALWAYS and new_mode != LogicNCaptureMode.BASIC:
            self.error_received.emit('Invalid Capture Mode')
            return
        self._context.dispatch_setting_change('capture_mode', new_mode)

    def on_waveform_received(self, wave_path: str, prev_wave_path: str, fix_wave_path: str) -> None:
        """
        Display waveform
        """
        #checked_exe = ['gtkwave', 'shmidcat', 'tail', 'gconftool-2']
        checked_exe = ['gtkwave']
        exe_paths = [which(exe) for exe in checked_exe]
        if None not in exe_paths:
            root, ext = os.path.splitext(wave_path)
            gtkw_path = root + '.gtkw'
            trig_pos = self.ui.sb_trig_pos.value()
            marker = trig_pos
            markers = {}
            window_depth = self._context.setting['window_depth']

            if os.name == 'nt':
                filter_proc = os.path.join(EFXDBG_HOME, 'bin', 'efx_dbg',
                                           'gtkw_filter', 'gtkw_filter.exe')
            else:
                filter_proc = os.path.join(EFXDBG_HOME, 'bin', 'efx_dbg',
                                           'gtkw_filter', 'gtkw_filter')
            filter_argv = 'win_depth={}'.format(window_depth)

            # GTKWave Limitation, it only support 26 markers
            marker_list = [
                'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm',
                'n', 'o', 'p'
                'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z'
            ]
            for idx, marker in enumerate(marker_list):
                if idx > self.ui.sb_num_window.value():
                    break
                markers[marker] = (idx * window_depth + trig_pos)
            print(markers)

            prev_gtkw_path = ''
            overwrite = self.ui.chk_overwrite_wave.isChecked()
            if not overwrite and os.path.isfile(gtkw_path):
                gtkw_path, _ = gen_next_writable_file_path(gtkw_path)

            prev_wave_path_prefix, _ = os.path.splitext(prev_wave_path)
            prev_gtkw_path = prev_wave_path_prefix + '.gtkw'

            if( os.path.exists(prev_gtkw_path) ):
                shutil.copyfile(prev_gtkw_path, gtkw_path)

                # Replace the existing wave_path with new path
                # Read in the file
                with open(gtkw_path, 'r') as file:
                    filedata = file.read()

                # Replace the target string
                filedata = filedata.replace(prev_wave_path, str(fix_wave_path))
                filedata = filedata.replace(prev_gtkw_path, str(gtkw_path))

                # Write the file out again
                with open(gtkw_path, 'w') as file:
                    file.write(filedata)
            else:
                with open(gtkw_path, 'w') as f:
                    gtkw = EfxGTKSave(f)
                    gtkw.dumpfile(fix_wave_path)
                    gtkw.dumpfile_mtime(dump_path=fix_wave_path)
                    gtkw.dumpfile_size(dump_path=fix_wave_path)
                    gtkw.savefile(save_path=gtkw_path)
                    gtkw.timestart()
                    gtkw.size(1000, 600)
                    gtkw.pos()
                    # gtkw.zoom_markers(zoom=0.0, marker=marker, a=marker, b=new_marker)
                    gtkw.zoom_markers(zoom=0.0, marker=marker, **markers)
                    gtkw.signals_width(126)
                    gtkw.sst_expanded(True)
                    for name, width, _, _ in self._spec.get_probes(trig_en=False):
                        suffix = '[{}:0]'.format(width - 1) if width > 1 else ''
                        # if '/' in name:
                              # Remarks: Need escape \ if the name contains '/'
                        #     name = rf"\{name}"
                        #gtkw.trace('top.' + name + suffix, transaction_filter_proc=filter_proc,
                        #           transaction_filter_argv=filter_argv)
                        gtkw.trace('top.' + name + suffix)

            if os.path.isfile(gtkw_path):
                if self._is_continous_trig or not self._is_always_launch_new_viewer:
                    #shutil.copyfile(gtkw_path, self.tmp_continous_gtk_path)
                    wave_viewer = open_wave_viewer(gtkw_path, lambda e: self.error(str(e)), self.tmp_repscript_path)

                    self._wave_viewer = wave_viewer
                else:
                    wave_viewer = open_wave_viewer(gtkw_path, lambda e: self.error(str(e)), is_always_launch_new_viewer=True)
                    self._standalone_wave_viewers.append(wave_viewer)


    def on_result_received(self, capture_buff):
        """
        Save to waveform file
        """
        def save_waveform(_wave_path: str, overwrite: bool, capture_buf, window_depth):
            gtkstatus = '-1'
            with open(_wave_path, 'w') as f:
                self._ctrl.export_vcd(f, capture_buff, window_depth)

            if (self.is_continous_trig or not self._is_always_launch_new_viewer):
                with open(self.tmp_mem_path, 'r') as f:
                    gtkstatus = f.read()

                if gtkstatus == '':
                    if self.gtk_trig_pos == None:
                        with open(self.tmp_mem_path, 'w') as f:
                            f.write("1")
                    else:
                        with open(self.tmp_mem_path, 'w') as f:
                            f.write(f"1\n{str(self.gtk_trig_pos)}")
                    self.gtk_trig_pos = None

        wave_path = self.ui.le_filepath.text()
        overwrite = self.ui.chk_overwrite_wave.isChecked()
        window_depth = self._context.setting['window_depth']

        prev_wave_path = ''
        if not overwrite and os.path.isfile(wave_path):
            new_wave_path, prev_wave_path = gen_next_writable_file_path(wave_path)
            #shutil.copyfile(prev_wave_path, new_wave_path)
            wave_path = new_wave_path

        if (self._is_continous_trig or not self._is_always_launch_new_viewer):
            if self._wave_viewer:
                if self._wave_viewer.poll() is None:
                    save_waveform(self.tmp_continous_vcd_path, overwrite, capture_buff, window_depth)
                    shutil.copyfile(self.tmp_continous_vcd_path, wave_path)
                    # sometime we encountered late is_cont_trig if  FULL and IDLE happenned too close ...
                    self._is_continous_trig = self.is_continous_trig
                    return

            LOGGER.debug('wave_path = {}, overwrite = {}'.format(self.tmp_continous_vcd_path, overwrite))
            save_waveform(self.tmp_continous_vcd_path, overwrite, capture_buff, window_depth)
            shutil.copyfile(self.tmp_continous_vcd_path, wave_path)
            self.on_waveform_received(str(wave_path), str(prev_wave_path), str(self.tmp_continous_vcd_path))
            self._is_continous_trig = self.is_continous_trig
        else:
            LOGGER.debug('wave_path = {}, overwrite = {}'.format(wave_path, overwrite))
            save_waveform(wave_path, overwrite, capture_buff, window_depth)
            self.on_waveform_received(str(wave_path), str(prev_wave_path), str(wave_path))
            self._is_continous_trig = self.is_continous_trig


    def on_filepath_edited(self) -> None:
        """
        Set default path if lineedit is clear
        """
        le_filepath = self.ui.le_filepath
        if not le_filepath.text():
            le_filepath.setText(self._default_wave_path)
        if (not le_filepath.isModified()):
            return
        self.ui.le_filepath.setModified(False)
        if not le_filepath.text() or os.path.isdir(le_filepath.text()):
            le_filepath.setFocus()
            QMessageBox.warning(self, "Efinity Debugger",
                        "Invalid Waveform Path. Debugger would not run")

    def on_filepath_changed(self, s):
        le_filepath = self.ui.le_filepath
        if not s or os.path.isdir(self.ui.le_filepath.text()):
            le_filepath.setStyleSheet("border: 2px solid red;")
        else :
            le_filepath.setStyleSheet("border: 1px solid silver;")

    def on_waveform_filepath(self):
        """Opens up file select dialog on open button clicked"""
        title = 'Select path for waveform file'
        dialog = QFileDialog(self, title, self._last_directory)
        dialog.setNameFilter("Value Change Dump Waveform (*.vcd)")
        dialog.setFileMode(QFileDialog.AnyFile)
        dialog.setViewMode(QFileDialog.Detail)
        dialog.selectFile('')
        filename_list = []
        if dialog.exec():
            filename_list = dialog.selectedFiles()
            if filename_list:
                self._last_directory = dialog.directory().path()
                wave_filename = filename_list[0]
                self.ui.le_filepath.setText(wave_filename)

    def on_debug_core_error(self, error: Exception) -> None:
        self.error_received.emit(error)

    def on_debug_core_pre_connect(self):
        if self._context.state['connected']:
            return

    def on_debug_core_post_connect(self, dbg_sesh: DebugSession):
        if self._context.state['connected']:
            return

        self._ctrl = QLogicNController(dbg_sesh, self._spec.get_name())
        self._ctrl.state_changed.connect(self._context.dispatch_state_change)
        self._ctrl.data_received.connect(self.on_result_received)
        self._ctrl.error_received.connect(self.on_debug_core_error)

        new_state = dict(self._context.state)
        new_state['connected'] = True
        self._context.dispatch_state_change(new_state)
        self._ctrl.start_monitor()

    def on_debug_core_pre_disconnect(self):
        if not self._context.state['connected']:
            return

        if self._ctrl:
            self._ctrl.stop_monitor()
            self._ctrl.state_changed.disconnect(self._context.dispatch_state_change)
            self._ctrl.data_received.disconnect(self.on_result_received)
            self._ctrl.error_received.disconnect(self.on_debug_core_error)
            self._ctrl = None

    def on_debug_core_post_disconnect(self):
        if not self._context.state['connected']:
            return

        # # Reset the GUI to default state
        # current_setting = self._context.setting

        # self._display_state('UNKNOWN')
        # self._display_progress(0, current_setting['num_window'],
        #                        current_setting['window_depth'])
        # self.ui.pb_run.setEnabled(False)
        # self.ui.pb_run_imdt.setEnabled(False)
        # self.ui.pb_stop.setEnabled(False)
        # self.ui.widget_capture_sel.setEnabled(False)
        # self.ui.cb_capture_mode.setEnabled(False)
        # widgets_setEnabled(self._widgets_to_freeze, False)

        self.parent.ui.actionAlwaysLaunchNewWaveWindow.setEnabled(True)

        new_state = dict(self._context.state)
        new_state['connected'] = False
        new_state['running'] = False
        self._context.dispatch_state_change(new_state)

    def on_debug_core_state_updated(self, debug_core_state: dict):
        """
        Triggered after the new state is queried from the Logic N Debugger Core

        Expecting debug_core_state
        debug_core_state = {
            'status': Any['IDLE', 'PRE_TRIG', 'WAIT_TRIG', 'POST_TRIG', 'FULL'],
            'sample_cnt' : Number,
            'running': Bool,
            'connected': Bool
        }
        """
        current_setting = self._context.setting

        # Check the state obj is invalid
        if not all(key in ['status', 'sample_cnt', 'running', 'connected'] for key in list(debug_core_state.keys())):
            raise TypeError('Status Object should only contains "status" / "sample_cnt"/ "running"/"connected" keys.')

        if debug_core_state['status'] not in ['IDLE', 'PRE_TRIG', 'WAIT_TRIG', 'POST_TRIG', 'FULL']:
            raise TypeError('Invalid state, should be any of ["IDLE", "PRE_TRIG", "WAIT_TRIG", "POST_TRIG", "FULL"]')

        if not isinstance(debug_core_state['sample_cnt'], int):
            raise TypeError('debug_core_state["sample_cnt"] must be int, not {}'.format(type(debug_core_state['sample_cnt'])))

        if not isinstance(debug_core_state['running'], bool):
            raise TypeError('debug_core_state["running"] must be bool, not {}'.format(type(debug_core_state['running'])))

        if not isinstance(debug_core_state['connected'], bool):
            raise TypeError('debug_core_state["connected"] must be bool, not {}'.format(type(debug_core_state['connected'])))

        # Check the correctness of the new state by comparing with the old one
        still_running = self._logicN_state['running'] and debug_core_state['running']
        if still_running:
            # sample_cnt should be increasing or no change
            if debug_core_state['sample_cnt'] < self._logicN_state['sample_cnt'] and not (debug_core_state['sample_cnt'] == 0 and self.ui.cb_data_depth.currentData() == 131072):
                raise RuntimeError(f"Invalid sample_cnt => {debug_core_state['sample_cnt']} => {self._logicN_state['sample_cnt']}, probably error between communication with debug core.")

            # sample_cnt should not exceed the limit set when kick-starting
            if debug_core_state['sample_cnt'] > self._spec.get_depth():
                raise RuntimeError(f"Invalid sample_cnt => {debug_core_state['sample_cnt']} => {self._spec.get_depth()}, probably error between communication with debug core.")

        if debug_core_state['connected']:
            pass
        else:
            self._display_state('UNKNOWN')
            self._display_progress(0,
                                   current_setting['num_window'],
                                   current_setting['window_depth'])
            self.ui.pb_run.setEnabled(False)
            self.ui.pb_run_imdt.setEnabled(False)
            self.ui.pb_run_cont.setEnabled(False)
            self.ui.pb_stop.setEnabled(False)
            self.ui.widget_capture_sel.setEnabled(False)
            self.ui.cb_capture_mode.setEnabled(False)
            widgets_setEnabled(self._widgets_to_freeze, False)
            return

        # Update the GUI
        self._display_state(debug_core_state['status'])
        self._display_progress(debug_core_state['sample_cnt'],
                               current_setting['num_window'],
                               current_setting['window_depth'])
        self._update_control_widgets(debug_core_state['running'])

    def is_setting_changed(self):
        return self._context.is_setting_changed()

    """
    Slot
    """
    def load_result(self):
        """
        """
        state = self._context.state
        if state['state'] != 'FULL':
            # TODO: Error
            return
        self._ctrl.read()
        new_state = dict(self._context.state)
        new_state['running'] = False
        self._context.dispatch_state_change(new_state)

    def go_idle(self):
        """
        """
        state = self._context.state
        if state['state'] != 'FULL':
            # TODO: Error
            return
        self._ctrl.go_idle()
        new_state = dict(self._context.state)
        new_state['running'] = False
        self._context.dispatch_state_change(new_state)

    def load_profile(self, profile: DebugProfile):
        """
        Load the settings from the LogicNSpec / inital settings
        """
        LOGGER.debug("Called")
        self.ui.label_num_window_range.setText('[1 - {}]'.format(
            self._spec.get_depth() // 8))
        self.ui.label_data_depth_range.setText('[1 - {}]'.format(
            self._spec.get_depth()))
        self.ui.label_trig_pos_range.setText(
            '[0 - {}]'.format(self._spec.get_depth() - 1))
        self.ui.sb_trig_pos.setMaximum(self._spec.get_depth() - 1)
        self.ui.sb_trig_pos.setValue(self._spec.get_depth() // 2)

        self.ui.cb_n_trig.setMaximum(1000)
        self.ui.cb_n_trig.setMinimum(1)
        self.ui.cb_n_trig.setValue(1)

        session = profile.get_core_session(self._name)
        if session:
            LOGGER.debug('ILA, session = {}'.format(session))
            self._context.dispatch_setting_overwrite(session['setting'])

    def save_profile(self, profile: DebugProfile):
        """
        Save the setting to profile
        """
        LOGGER.debug("Called")
        profile.set_core_session(name=self._name,
                                 session=dict(self._context.setting))
        LOGGER.debug('Updated = {}'.format(profile))

    def update_wave_viewer_config(self, always_launch_new_win: bool):
        self._is_always_launch_new_viewer = always_launch_new_win

class GuiPerspective(Enum):
    PROFILE = 1
    DEBUG = 2


class GenWindow(QMainWindow):

    def __init__(self, parent: QWidget = None):
        super(GenWindow, self).__init__(parent)
        self._jtag_docks = []

    def get_widgets(self) -> List[QWidget]:
        return [dock.widget() for dock in self._jtag_docks]

    # used by test_gui.py
    def get_widget_count(self) -> int:
        return len(self._jtag_docks)

    # used by test_gui.py
    def get_widget_by_index(self, index: int) -> QWidget:
        return self._jtag_docks[index].widget()

    def remove_gen_widgets(self):
        """Remove all debug core gen widgets"""
        for dock in self._jtag_docks:
            dock.widget().deleteLater()
            dock.deleteLater()
        self._jtag_docks.clear()

    def on_core_name_update(self, widget_id: int):
        for dock in self._jtag_docks:
            if id(dock.widget()) == widget_id:
                names = [dock.widget()._name for dock in self._jtag_docks]
                if dock.widget().title() in names:
                    dock.widget().ui.le_name.setText(dock.widget()._name)
                else:
                    dock.widget()._name = dock.widget().title()
                    dock.widget().ui.le_name.setText(dock.widget().title())
                    dock.setWindowTitle(dock.widget().title())
                    dock.setObjectName(dock.widget().title())

    def on_core_add(self, name: str = "", ctype: str = "") -> QWidget:
        if not name:
            names = [dock.widget()._name for dock in self._jtag_docks]
            idx = 0
            while ctype.lower() + str(idx) in names:
                idx += 1
            name = ctype.lower() + str(idx)
        cls = {"vio": VioGenWidget, "la": LogicNGenWidget}.get(ctype)
        if not cls:
            raise ValueError('Unknown debugger type "{}".'.format(ctype))

        dockWidget = QDockWidget(self)
        widget = cls(parent=dockWidget, name=name)
        widget.error_received.connect(self.on_core_gen_error)
        widget.ui.pb_remove_core.clicked.connect(lambda: self.on_core_remove(id(widget)))
        widget.ui.le_name.editingFinished.connect(lambda: self.on_core_name_update(id(widget)))
        dockWidget.setFeatures(QDockWidget.DockWidgetFloatable | QDockWidget.DockWidgetMovable)
        dockWidget.setWidget(widget)
        dockWidget.setWindowTitle(widget.title())
        dockWidget.setObjectName(widget.title())

        self.addDockWidget(Qt.TopDockWidgetArea, dockWidget)
        if len(self._jtag_docks) >= 1:
            self.tabifyDockWidget(self._jtag_docks[-1], dockWidget)
        dockWidget.show()
        dockWidget.raise_()
        self._jtag_docks.append(dockWidget)

        return widget

    def on_core_gen_error(self, msg_title: str, msg_body: str):
        QMessageBox.warning(self, msg_title, msg_body)

    def on_core_remove(self, widget_id: int):
        for dock in self._jtag_docks:
            if id(dock.widget()) == widget_id:
                self._jtag_docks.remove(dock)
                dock.widget().deleteLater()
                dock.deleteLater()

    def show(self):
        super(GenWindow, self).show()
        for dockwidget in self._jtag_docks:
            dockwidget.show()

    def hide(self):
        super(GenWindow, self).hide()
        for dockwidget in self._jtag_docks:
            dockwidget.hide()

class DbgGui(QMainWindow):  # pylint: disable=R0902, R0904
    """Debugger GUI main window"""
    MSG_I001 = 'Ready'
    MSG_I002 = ('Select the correct USB JTAG device and click '
                'Connect Debugger to start debugging.')
    MSG_W003 = ('EFXDBG_HOME not in os.environ. Debug core '
                'generation disabled.')
    MSG_W004 = 'InvalidJsonError: '
    MSG_W005 = 'No target device available for JTAG configuration.'
    MSG_W006 = ('Debug profile not available. '
                'Please go to Perspectives -> Profile Editor, '
                'do ADD/IMPORT and then GENERATE before debugging.')
    MSG_W007 = ('Debug Core UUID mismatch with Debug Profile, Please re-generated the Core.')
    MSG_W008 = ('Debug Profile has been updated. Would you want to refresh it?')

    HW_TOOL_CONFIG_SECTION = 'debugger'
    HW_TOOL_CONFIG_KEY_ALWAYS_LAUNCH_NEW_WAVE_WIN = 'always_launch_new_wave_window'

    pre_jtag_config = pyqtSignal()
    post_jtag_config = pyqtSignal(object)
    pre_jtag_unconfig = pyqtSignal()
    post_jtag_unconfig = pyqtSignal()

    def __init__(self, args):
        super().__init__()
        self._args = args
        self._jtag_sesh: Optional[JtagSession] = None
        self._dbg_sesh: Optional[DebugSession] = None
        self._jtag_url = None
        self._prev_tab = -1
        self._profile = None
        self._timer_interval = 40
        self.titanium = None
        # Last directory in the open image dialog
        self._last_directory = os.getcwd()
        self._dbg_docks = []  # type: List[QDockWidget]

        if DOCKED_GUI:
            self.ui = Ui_DockedMainWindow()
        else:
            self.ui = Ui_DebuggerMainWindow()  # pylint: disable=C0103
        self.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        self.ui.setupUi(self)

        self.genwindow = GenWindow(self)
        self.ui.verticalLayout_setup.addWidget(self.genwindow)

        self.update_window_file_path()

        if DOCKED_GUI:
            centralwidget = self.takeCentralWidget()
            #centralwidget.hide()
            #centralwidget.deleteLater()

            # Qt.DockWidgetArea(1), 0x1 left, 0x2 right, 0x4 top, 0x8 bottom
            #self.setTabPosition(Qt.LeftDockWidgetArea, QTabWidget.North)
            #self.setTabPosition(Qt.RightDockWidgetArea, QTabWidget.North)
            #self.setTabPosition(Qt.BottomDockWidgetArea, QTabWidget.North)

            self.resize(1400, 900)

            self.ui.actionViewDebug.setChecked(True)
            self.perspectives = QActionGroup(self)
            self.perspectives.addAction(self.ui.actionViewProfile)
            self.perspectives.addAction(self.ui.actionViewDebug)
            self.perspectives.setExclusive(True)

            self.reset_dock_layout(init=True)

            self._dbg_docks.append(self.ui.dockWidget_debug1)

            self.ui.le_status = QLineEdit(centralwidget)
            self.ui.le_status.setReadOnly(True)
            self.ui.le_status.setObjectName("le_status")
            self.ui.le_status.setText("Ready")

            self.ui.toolBar_status.addWidget(self.ui.le_status)

            btn_reload_profile = QToolButton(self)
            btn_reload_profile.setToolButtonStyle(Qt.ToolButtonTextBesideIcon)
            btn_reload_profile.setIcon(QIcon(':/resource/dbg-import-profile.svg'))
            btn_reload_profile.setText("Reload Profile")
            btn_reload_profile.clicked.connect(self.reload_profile)
            self.action_reload_profile = self.ui.toolBar_status.addWidget(btn_reload_profile)
            self.action_reload_profile.setVisible(False)

            self.ui.toolBar_status.addSeparator()
            self.ui.toolBar_status.addAction(self.ui.actionHelp)

            self.ui.toolBar_main.toggleViewAction().setEnabled(False)
            self.ui.toolBar_status.toggleViewAction().setEnabled(False)
            #self.ui.toolBar_help.toggleViewAction().setEnabled(False)
            self.ui.pb_config_jtag.setEnabled(False)
            #self.ui.cb_jchain_target.setEnabled(False)

        #self.ui.label_jtag_info.setText('')

        palette = self.ui.pte_user_guide.palette()
        background = palette.window().color()
        palette.setColor(QPalette.Active, QPalette.Base, background)
        self.ui.pte_user_guide.setPalette(palette)

        # self.ui.pb_unconfig_jtag.setEnabled(False)

        self.pgm_engine = DbgPgmEngine()
        def programmer_error(exception):
            self.error(str(exception))
            self.on_unconfig(force_quit=True)

        self.pgm_engine.error_caught.connect(lambda str: programmer_error(str))
        self.pgm_win = DbgPgmWin(self.ui.pte_console, engine=self.pgm_engine)

        self.pgm_win.error_caught.connect(
            lambda exception: programmer_error(exception))
        self.console = self.pgm_win.console

        # Set the font for console
        EFINITY_HOME = os.environ['EFINITY_HOME']
        APP_INI = os.path.join(EFINITY_HOME, "bin", "style.ini")
        settings = QSettings(APP_INI, QSettings.IniFormat)
        font_name = settings.value("debugger/console_font_name", "Source Code Pro")
        font_size = settings.value("debugger/console_font_size", "11")
        font = QFont(font_name, int(font_size))
        console_stylesheet = 'QWidget {{ font-family: {}; font-size: {}pt; }}'.format(font_name, font_size)
        self.ui.pte_console.setStyleSheet(console_stylesheet)

        self.device_db = bitstream_util.get_device_db()
        # self.detect_jtag = DetectJTAGDbg(self.console, self)
        self.detect_jtag = self.pgm_win.detect_jtag

        self.status_colors = {'RED':QPixmap(os.path.join(":/icons/dot-red.svg")),
                 'YELLOW':QPixmap(os.path.join(":/icons/dot-yellow.svg")),
                 'GREEN':QPixmap(os.path.join(":/icons/dot-green.svg")),
                 'GRAY':QPixmap(os.path.join(":/icons/dot-gray.svg"))}

        # This is how we replace with our gui
        kwargs = {
            'cb_usb_target': self.ui.cb_usb_target,
            #'pb_refresh_usb_target': self.ui.pb_refresh_usb_target,
            'lbl_usb_id': self.ui.label_usb_id,
            'cb_prog_image': self.ui.cb_prog_image,
            'le_detected_device': self.ui.le_detected_device,
            'pb_select_image': self.ui.pb_select_image,
            #'cb_prog_mode': mock.MagicMock(), #self.ui.cb_prog_mode,
            #'pb_adv_image_gen': mock.MagicMock(), #self.ui.pb_adv_image_gen,
            'pb_program': self.ui.pb_program,
            'pb_stop_program': self.ui.pb_stop_prog,
            'cb_chip_num': self.ui.cb_chip_num,
            # 'actionImport_JCF': self.ui.actionImport_JCF,
            'lbl_prog_status': self.ui.lbl_prog_status,
            'lbl_last_updated': self.ui.lbl_last_updated,
            'pb_refresh_status': self.ui.pb_refresh_status,
            'pb_ti_jtag_status': self.ui.pb_ti_jtag_status,
            'pb_edit_remote_host': self.ui.pb_edit_remote_host,
            'actionEnable_Legacy_Mode': self.ui.actionEnable_Legacy_Mode

        }

        self._programmer_widgets_to_freeze = [
            self.ui.cb_usb_target,
            self.ui.cb_user_instr,
            self.ui.le_detected_device,
            self.ui.cb_prog_image,
            self.ui.pb_program,
            self.ui.pb_stop_prog,
            self.ui.pb_select_image,
            self.ui.pb_import_jcf,
            self.ui.pb_refresh_usb_target,
        ]

        self.ui.pb_program.clicked.connect(self.programmer_watcher_start)
        self.ui.pb_import_jcf.clicked.connect(self.on_dbg_import_jcf)

        self.pgm_win.override_ui(**kwargs)
        self.pgm_engine.run_nogui(self.pgm_win)
        # self.pgm_win.on_prog_mode_changed('JTAG')
        self.pgm_win.cb_prog_mode.setCurrentText('JTAG')

        # Set the programming mode
        # self.pgm_win.on_prog_mode_changed('JTAG')

        #self.ui.cb_debug_target.currentIndexChanged[int].connect(self.on_debug_target_changed)

        self.menu_add_core = QMenu()
        action_add_vio = self.menu_add_core.addAction('Virtual IO')
        action_add_la = self.menu_add_core.addAction('Logic Analyzer')
        action_add_vio.triggered.connect(lambda: self.genwindow.on_core_add(ctype="vio"))
        action_add_la.triggered.connect(lambda: self.genwindow.on_core_add(ctype="la"))
        self.ui.pb_add_core.setMenu(self.menu_add_core)

        self.ui.pb_import.clicked.connect(self.on_import)
        self.ui.pb_generate.clicked.connect(self.on_generate)

        #self.ui.cb_usb_target.currentIndexChanged.connect(self.on_debug_refresh)
        #self.ui.pb_refresh_debug_target.clicked.connect(self.on_debug_refresh)
        self.ui.pb_refresh_usb_target.clicked.connect(self.on_debug_refresh)
        self.ui.pb_config_jtag.pressed.connect(self.on_configure)
        self.ui.pb_unconfig_jtag.clicked.connect(self.on_unconfig)
        self.ui.cb_usb_target.currentIndexChanged.connect(self.on_usb_target_changed)

        self.ui.pb_next.clicked.connect(self.on_next)
        self.ui.actionHelp.triggered.connect(self.on_show_help)
        self.ui.actionAbout.triggered.connect(self.on_show_about)
        self.ui.actionEnable_Legacy_Mode.toggled.connect(self.on_toggle_legacy_mode)
        self.ui.actionImport_JCF.triggered.connect(self.on_dbg_import_jcf)

        self.ui.actionNewProfile.triggered.connect(self.on_new_profile)
        self.ui.actionOpenProfile.triggered.connect(self.on_import)
        self.ui.actionSaveAs.triggered.connect(self.on_save_as)

        self.ui.pb_ti_jtag_status.clicked.connect(self.on_show_ti_jtag_status)
        self.ui.actionAlwaysLaunchNewWaveWindow.toggled.connect(self.on_wave_viewer_config_update)

        if not DOCKED_GUI:
            self.ui.tabWidget.currentChanged.connect(self.on_change_tab)
        else:
            self.ui.cb_chip_num.activated.connect(
                self.on_jchain_target_changed)
            self.ui.actionViewProfile.triggered.connect(
                lambda: self.reset_dock_layout(view=GuiPerspective.PROFILE))
            self.ui.actionViewDebug.triggered.connect(
                lambda: self.reset_dock_layout(view=GuiPerspective.DEBUG))

        self.PROFILE_FILE = os.path.join(self._args.project_home,
                                         'debug_profile.json')
        if self._args.profile_file:
            if os.path.isabs(self._args.profile_file):
                self.PROFILE_FILE = self._args.profile_file
            else:
                self.PROFILE_FILE = os.path.join(self._args.project_home,
                                                 self._args.profile_file)

        self.PROFILE_FILE = os.path.abspath(self.PROFILE_FILE)

        self.programmer_timer = QTimer()
        self.programmer_timer.setInterval(1000)
        self.programmer_timer.timeout.connect(self.programmer_watcher_stop)
        self.programmer_timer.start()

        self.profile_watcher = QFileSystemWatcher(self)
        self.profile_watcher.fileChanged.connect(self.on_profile_file_modified)
        self.on_startup(self.PROFILE_FILE)

        self.jtag_available = None
        self.apply_hwtools_config()

    def apply_hwtools_config(self):
        """
        Apply settings defined in hw_tools.ini
        """
        self.hwtconfig_util = HwToolsConfigUtil()
        if self.hwtconfig_util.config_file == self.hwtconfig_util.backup_file:
            self.console.write_warning_msg("WARNING: Unable to decode hw_tools.json, using default settings")
            self.ui.actionAlwaysLaunchNewWaveWindow.setChecked(False)
            return

        val = self.hwtconfig_util.read_setting(sect='debugger', opt='always_launch_new_wave_window')
        self.ui.actionAlwaysLaunchNewWaveWindow.setChecked(val)

    def on_profile_file_modified(self, file_path):
        # Check whether the debug cores has been updated or not
        # We only take care about those debug cores
        def is_dbg_cores_updated(file_path):
            try:
                new_profile = DebugProfileBuilder.from_file(file_path)
                if new_profile.is_valid() and self._profile is not None:
                    # Check if the uuid is updated
                    if len(new_profile.core_list) != len(self._profile.core_list):
                        LOGGER.debug(f'Number of core different: {len(new_profile.core_list)} vs {len(self._profile.core_list)}')
                        return True

                    for old_spec in self._profile.core_list:
                        new_spec = new_profile.get_core_spec(old_spec.get_name())
                        if new_spec == None:
                            # Old Core removed
                            LOGGER.debug(f'Old core removed: {old_spec.get_name()}')
                            return True

                        if new_spec.get_type() != old_spec.get_type():
                            LOGGER.debug(f'Spec type different: New {new_spec.get_type()} vs {old_spec.get_type()}')
                            return True

                        if new_spec.get_uuid() != old_spec.get_uuid():
                            LOGGER.debug(f'UUID different: New {new_spec.get_uuid()} vs {old_spec.get_uuid()}')
                            return True
                return False
            except Exception as exc:
                import traceback
                LOGGER.debug(traceback.format_exc())
                return False

        changed = is_dbg_cores_updated(file_path)
        if changed:
            if self.action_reload_profile.isVisible() == False:
                # Only show it when user haven't reload the profile
                self.console.write(f"Debug Profile: {file_path} is modifed.", colour_code="#ff6700")
                self.console.write(f"Please reload it.", colour_code="#ff6700")
            self.warning(DbgGui.MSG_W008)
            self.action_reload_profile.setVisible(True)

    def reload_profile(self):
        self.clear_message()
        if self._jtag_url:
            self.on_unconfig()
            self.console.write(f'Disconnected active debugger connections.')

        self.import_profile_from_file(self.PROFILE_FILE)
        self.action_reload_profile.setVisible(False)
        self.console.write(f'Loaded Debug Profile from file: {self.PROFILE_FILE}', is_timestamp=True)

    def update_window_file_path(self, filepath: str = "unsaved profile"):
        self.setWindowTitle("{} Debugger - {}".format(_AppInfo.APP_NAME, filepath))
        self.ui.dockWidget_setup.setWindowTitle("Profile - " + filepath)

    def on_save_as(self) -> bool:
        filename = QFileDialog.getSaveFileName(self, "Save as", os.getcwd(), "Debug Profile File (*.json)")[0]

        if not filename:
            return False

        if not filename.endswith(".json"):
            filename += ".json"
            self.PROFILE_FILE = os.path.relpath(filename)
        msg = ("The debug profile will be saved to the following destinations:\n*file will be replaced if it already exists\n\n"
                + filename + "\n"
                + os.path.dirname(filename) + "/debug_top.v\n"
                + os.path.dirname(filename) + "/debug_TEMPLATE.v\n"
                + os.path.dirname(filename) + "/debug_TEMPLATE.vhd")
        msg_box = QMessageBox(self)
        msg_box.setIcon(QMessageBox.Question)
        msg_box.setWindowTitle("Confirmation")
        msg_box.setText(msg)
        msg_box.setStandardButtons(QMessageBox.Ok | QMessageBox.Cancel)
        ok_btn = msg_box.button(QMessageBox.Ok)
        ok_btn.setText("Proceed")
        result = msg_box.exec_()

        if result != QMessageBox.Ok:
            self.console.write("Save action was cancelled")
            return False

        self.on_generate(filename)
        return True

    def on_new_profile(self):
        msg_box = QMessageBox()
        msg_box.setWindowTitle("Confirmation")
        msg_box.setText("Do you want to save the changes you made (if any) before resetting to a new debug profile?\n\n"
                        "Your changes will be lost if you don't save them.\n")
        msg_box.setStandardButtons(QMessageBox.Discard | QMessageBox.Cancel | QMessageBox.Save)
        button_discard = msg_box.button(QMessageBox.Discard)
        button_discard.setText("Don't Save")
        button_save = msg_box.button(QMessageBox.Save)
        button_save.setText("Save...")
        msg_box.setDefaultButton(QMessageBox.Save)
        result = msg_box.exec_()

        if result == QMessageBox.Discard:
            self.genwindow.remove_gen_widgets()
            self.remove_debug_widgets()
            self.update_window_file_path()
        elif result == QMessageBox.Save:
            if self.on_save_as():
                self.genwindow.remove_gen_widgets()
                self.remove_debug_widgets()
                self.update_window_file_path()

        msg_box.close()

    def programmer_watcher_start(self):
        # Start the timer
        LOGGER.info("Programmer started, start watcher")
        self.programmer_timer.start()
        self.ui.pb_config_jtag.setEnabled(False)
        self.ui.pb_refresh_usb_target.setEnabled(False)
        self.ui.pb_unconfig_jtag.setEnabled(False)

    def programmer_watcher_stop(self):
        programmer_running = self.pgm_win.programmer.is_running()
        if not programmer_running:
            LOGGER.info("Programmer stopped, stop watcher")
            self.programmer_timer.stop()
            self.ui.pb_config_jtag.setEnabled(True)
            self.ui.pb_refresh_usb_target.setEnabled(True)
            self.ui.pb_unconfig_jtag.setEnabled(True)

    def on_jchain_target_changed(self, index):
        # self.remove_gen_widgets()
        # self.remove_debug_widgets()
        # TODO: call change_target of JtagChainEngine
        #target_num = self.ui.cb_chip_num.currentData()
        #LOGGER.debug('cb_jchain_target.currentData()={}', target_num)
        #usb_target = self.ui.cb_usb_target.currentIndex()
        #prog_usb_target = self.pgm_win.programmer.resolved_usb[usb_target]
        #(_, _, desc, (v, p), urls) = prog_usb_target
        #url = urls[-1]
        #engine, _, _ = JtagManager().get_configured_jtag(url)
        #engine.update_target(target_num)
        #JtagManager().clear_core_history(url)
        pass

    @pyqtSlot(int)
    def on_usb_target_changed(self, index):
        self.pgm_win.on_usb_target_changed(index)
        self.pgm_win.cb_prog_mode.setCurrentText('JTAG')
        self.titanium = self.ui.pb_ti_jtag_status.isEnabled()

    def on_refresh_status(self):
        '''
        Refresh user mode detection
        '''
        self.pgm_win.on_refresh_status()

    def on_show_ti_jtag_status(self):
        '''
        Show Advanced Device Status GUI
        '''
        if self._jtag_url:
            self.console.write_error_msg("Debugger must be disconnected to use Advanced Device Status widget!")
            return
        self.pgm_win.on_show_ti_jtag_status()

    def on_toggle_legacy_mode(self):
        self.pgm_win.on_toggle_legacy_mode()

    def reset_dock_layout(self, view=GuiPerspective.DEBUG, init=False):
        if DOCKED_GUI:
            self.removeDockWidget(self.ui.dockWidget_setup)
            self.removeDockWidget(self.ui.dockWidget_user_guide)
            self.removeDockWidget(self.ui.dockWidget_program)
            self.removeDockWidget(self.ui.dockWidget_run)
            for dock in self._dbg_docks:
                self.removeDockWidget(dock)

            if not init:
                # Flush pervious message
                self.info(self.MSG_I001)

            if view == GuiPerspective.DEBUG:
                self.genwindow.hide()
                self.addDockWidget(Qt.LeftDockWidgetArea,
                                   self.ui.dockWidget_debug1)
                self.addDockWidget(Qt.RightDockWidgetArea,
                                   self.ui.dockWidget_program)
                self.addDockWidget(Qt.RightDockWidgetArea,
                                   self.ui.dockWidget_run)
                self.ui.dockWidget_debug1.show()
                self.ui.dockWidget_program.show()
                self.ui.dockWidget_run.show()
                self.resizeDocks(
                    [self.ui.dockWidget_debug1, self.ui.dockWidget_program],
                    [900, 700], Qt.Horizontal
                )
                self.resizeDocks(
                    [self.ui.dockWidget_program, self.ui.dockWidget_run],
                    [300, 600], Qt.Vertical)
                if len(self._dbg_docks) > 1:
                    for dock in self._dbg_docks[1:]:
                        self.tabifyDockWidget(self.ui.dockWidget_debug1, dock)
                        dock.show()
                if self._dbg_sesh is None:
                    self.ui.pb_unconfig_jtag.setEnabled(False)

            elif view == GuiPerspective.PROFILE:
                self.addDockWidget(Qt.TopDockWidgetArea,
                                   self.ui.dockWidget_setup)
                self.addDockWidget(Qt.BottomDockWidgetArea,
                                   self.ui.dockWidget_run)
                self.ui.dockWidget_setup.show()
                self.ui.dockWidget_run.show()
                self.resizeDocks(
                    [self.ui.dockWidget_setup, self.ui.dockWidget_run],
                    [600, 300], Qt.Vertical)
                self.genwindow.show()
                self.assert_generate()

    @staticmethod
    def on_show_help():
        """Launch html help in browser"""
        import platform
        import webbrowser
        control = None
        if platform.system() == 'Linux':
            control = webbrowser.get('firefox')
            if control is None:
                control = webbrowser.get('chrome')
        else:
            control = webbrowser.get('windows-default')
        if control is not None:
            if 'EFXDBG_HOME' in os.environ:
                help_file = os.path.join(EFXDBG_HOME, 'doc', 'topics',
                                         'hlp-dbg-intro.html')
                control.open_new_tab('file://{}'.format(help_file))

    @staticmethod
    def on_show_about():
        """
        Show the about dialog
        """
        about = DbgAboutDialog()
        about.exec()

    def on_startup(self, profile_file: str):
        """Import profile at the end of constructor"""
        if os.path.exists(profile_file) and os.path.isfile(profile_file):
            self.import_profile_from_file(profile_file)
            # By default opens in Debug perspective
            # Refresh once on gui start
            # self.on_debug_refresh()
            # self.pgm_win.on_usb_target_refreshed()
        else:
            # No need to refresh if open in Editor perspective
            self.ui.actionViewProfile.trigger()
            # self.on_usb_target_changed()
        self.titanium = self.ui.pb_ti_jtag_status.isEnabled()

    def closeEvent(self, event: QCloseEvent):  # pylint: disable=C0103, W0613
        """Handles stopping things when the GUI quits"""
        LOGGER.debug("closeEvent")
        self.on_unconfig()

        # Check if each debugger widgets has something to save in debug profile, ask the User
        setting_changed = False
        for item in self._dbg_docks:
            widget = item.widget()
            if isinstance(widget, AbstractDebugWidget):
                setting_changed = widget.is_setting_changed()
                if setting_changed:
                    break

        if setting_changed and not self._args.no_save:
            buttonReply = QMessageBox.question(self, 'Debugger GUI', "Would you like to save your debugger setting?", QMessageBox.Yes | QMessageBox.No, QMessageBox.No)
            if buttonReply == QMessageBox.Yes:
                for item in self._dbg_docks:
                    widget = item.widget()
                    if isinstance(widget, AbstractDebugWidget):
                        widget.save_profile(self._profile)
                if self._profile:
                    with open(self.PROFILE_FILE, 'w') as f:
                        json.dump(self._profile, f, indent=4, cls=DebugProfileEncoder)
            else:
                print('No clicked.')

        for item in self._dbg_docks:
            widget = item.widget()
            if isinstance(widget, AbstractDebugWidget):
                widget.on_close()

        self.pgm_win.on_exit()

    def assert_generate(self):
        """Show warnning message if EFXDBG_HOME is not in ENV"""
        if 'EFXDBG_HOME' not in os.environ:
            self.warning(self.MSG_W003)
            self.ui.pb_generate.setEnabled(False)

    def show_message(self, msg: str):
        """Common call for showing msg in GUI top status line"""
        self.ui.le_status.setText(msg)

    def clear_message(self):
        """Clear message in status line"""
        self.ui.le_status.setStyleSheet('')
        self.ui.le_status.clear()

    def debug(self, msg: str):
        """Shows debug msg in GUI top status line"""
        self.ui.le_status.setStyleSheet('background-color: white;')
        self.show_message(msg)

    def error(self, msg: str):
        """Shows error in GUI top status line"""
        self.ui.le_status.setStyleSheet('background-color: red;')
        self.show_message(msg)

    def info(self, msg: str, color: bool = False):
        """Shows info in GUI top status line"""
        if color:
            style = 'background-color: green; color: white'
            self.ui.le_status.setStyleSheet(style)
        else:
            self.ui.le_status.setStyleSheet('background-color: white;')
        self.show_message(msg)

    def warning(self, msg: str):
        """Shows warning in GUI top status line"""
        self.ui.le_status.setStyleSheet('background-color: yellow;')
        self.show_message(msg)

    def import_profile_from_file(self, file_path: str):
        """
        Load the debug profile json
        """
        LOGGER.debug("IN")
        if CLEAR_PROFILE_ON_IMPORT:
            self.genwindow.remove_gen_widgets()

        for path in self.profile_watcher.files():
            self.profile_watcher.removePath(path)

        try:
            debug_profile = DebugProfileBuilder.from_file(file_path)
            self._profile = debug_profile
            self.load_gen_widgets_from_profile()
            self.stop_jtag_and_load_debug_table()
            self.profile_watcher.addPath(file_path)
        except InvalidJsonError as exc:  # pylint: disable=C0103
            self.warning(self.MSG_W004 + str(exc))
            return
        except InvalidProfileError as exc:
            self.warning(str(exc))
            return

        LOGGER.debug("OUT")

    def on_import(self):
        """Opens up file select dialog on import button clicked"""
        title = 'Import Debug Profile'
        dialog = QFileDialog(self, title, self._last_directory)
        dialog.setWindowFlag(Qt.WindowContextHelpButtonHint, False)
        dialog.setNameFilter("Debug Profile File (*.json)")
        dialog.setFileMode(QFileDialog.ExistingFile)
        dialog.setViewMode(QFileDialog.Detail)
        dialog.selectFile('')
        filename_list = []
        if dialog.exec():
            filename_list = dialog.selectedFiles()
            if filename_list:
                self._last_directory = dialog.directory().path()
                profile_filename = filename_list[0]
                self.import_profile_from_file(profile_filename)
                self.update_window_file_path(profile_filename)

    def load_gen_widgets_from_profile(self):
        """Insert data from spec obj into dock widget"""
        for spec in self._profile.list_cores():
            widget = self.genwindow.on_core_add(spec.get_name(), ctype=spec.get_type())
            widget.load_spec(spec)

    def get_profile_from_widgets(self) -> DebugProfile:  # pylint: disable=R0914
        """Read widgets in scrollArea to generate DebugProfile object"""
        profile = DebugProfile()
        for widget in self.genwindow.get_widgets():
            widget_type = type(widget).__name__  # eg. VioGenWidget
            LOGGER.debug('widget_type={}'.format(widget_type))
            match = re.match(r'(\w+)GenWidget', widget_type)
            if not match:
                continue
            core_type = {'Vio': 'vio', 'LogicN': 'la'}.get(match.group(1))
            core_name = widget.title()
            core_spec = profile.add_core(core_name, core_type, True)
            widget.get_spec(core_spec)
        return profile

    def stop_jtag_and_load_debug_table(self):
        """Handles re-generate or import"""
        if not DOCKED_GUI:
            self.on_unconfig()  # dont stop, for jtag chain
        # Load profile into RUN tab
        self.load_debug_widgets_from_profile()

    def on_generate(self, profile_filepath: str = None) -> None:
        if not profile_filepath:
            profile_filepath = os.path.abspath(self.PROFILE_FILE)
        self.update_window_file_path(profile_filepath)

        """Get profile from setup tab and gen RTL on button clicked"""
        self._profile = self.get_profile_from_widgets()
        #print('profile:\n' + json.dumps(self._profile, indent=4))
        with open(profile_filepath, 'w') as f:  # pylint: disable=C0103
            json.dump(self._profile, f, indent=4, cls=DebugProfileEncoder)
        msg = 'Generated profile file = {}'.format(profile_filepath)
        LOGGER.debug(msg)
        self.console.write(msg)

        self.stop_jtag_and_load_debug_table()

        # HD-187 Require GBUF for JTAG TCK in Titanium
        call = [ partial(write_debug_top_verilog, insert_gbuf=HasOPXArch(self._args.family)), write_debug_template_verilog, write_debug_template_vhdl]
        fname = ['debug_top.v', 'debug_TEMPLATE.v', 'debug_TEMPLATE.vhd']
        fpath = [os.path.join(os.path.dirname(profile_filepath), f) for f in fname]
        for fn, p in zip(call, fpath):
            try:
                fn(p, self._profile)
            except LogicNRtlDesignLimitError as e:
                QMessageBox.warning(self, "Warning Debugger Probe", str(e))
        msg = 'Generated RTL and instantiation template files: {}'.format(fpath)
        LOGGER.debug(msg)
        self.console.write(msg)

        self.info(self.MSG_I001)
        if not DOCKED_GUI:
            self.ui.tabWidget.setCurrentIndex(1)
        else:
            #self.ui.dockWidget_user_guide.raise_()
            pass

    def on_next(self):
        """Change tab focus on button clicked"""
        if not DOCKED_GUI:
            self.ui.tabWidget.setCurrentIndex(2)
        else:
            self.ui.dockWidget_program.raise_()

    def on_change_tab(self, index: int):
        """Pause timer when tab focus is changed manually"""
        if index == 3:
            urls = JtagManager().view_configured_jtag()
            if not urls:
                # refresh USB target when change into debug tab
                # self.on_debug_refresh()
                pass
            else:
                pass
        elif self._prev_tab == 3:
            pass
        self._prev_tab = index

    @staticmethod
    def _guess_port_tap(vid: int, pid: int, desc: str) -> Tuple[int, str]:  # pylint: disable=W0613
        if (pid == 0x6010) or (pid == 0x6011):
            default_tup = (2, 'efx')
            swap_var = os.environ.get("SWAP_FTDI")
            if swap_var is not None and swap_var == "1":
                default_tup = (1, 'efx')
        else:
            default_tup = (1, 'efx')
        port_tap = {
            'Trion T20 Development Board': (2, 'efx'),
            'Digilent USB Device': (1, 'xlnx'),
            'C232HM-DDHSL-0': (1, 'efx'),
            'Dual RS232-HS': (2, 'efx')
        }.get(desc, default_tup)
        return port_tap

    #def on_debug_target_changed(self, index: int):
    #    if index >= 0:
    #        (bus, addr, v, p, url, tap) = self.ui.cb_debug_target.currentData()
    #        text = 'Bus %03d Device %03d: ID %04X:%04X' % (bus, addr, v, p)
    #        self.console.write('URL to be used for pyftdi: {}'.format(url))
    #        self.ui.label_usb_id.setText(text)
    #    else:
    #        self.ui.label_usb_id.setText('')

    def on_debug_refresh(self):
        try:
            # HD-330: need to disconnect signal to prevent triggering of different event
            self.ui.cb_usb_target.currentIndexChanged.disconnect(self.on_usb_target_changed)
            self.pgm_win.on_usb_target_refreshed()
            self.ui.cb_usb_target.currentIndexChanged.connect(self.on_usb_target_changed)
            self.titanium = self.ui.pb_ti_jtag_status.isEnabled()
            dev_list = self.pgm_win.programmer.resolved_usb
            if not dev_list:
                self.ui.pb_config_jtag.setEnabled(False)
                self.warning(self.MSG_W005)
            else:
                if self._profile is None:
                    self.ui.pb_config_jtag.setEnabled(False)
                    self.warning(self.MSG_W006)
                else:
                    self.ui.pb_config_jtag.setEnabled(True)
                    self.info(self.MSG_I002, color=True)
        except Exception as exc:
            self.error("ERROR: {}".format(str(exc)))

    def remove_debug_widgets(self):
        if not DOCKED_GUI:
            LOGGER.debug('Pre removeWidget(), Layout_debug.count() = {}'.format(
                         self.ui.verticalLayout_debug.count()))
            # QLayout itemAt() index changes when item is deleted,
            # use count down for removeWidget and delete
            for i in range(self.ui.verticalLayout_debug.count() - 1, -1, -1):
                widget = self.ui.verticalLayout_debug.itemAt(i).widget()
                LOGGER.debug('type(widget)={}'.format(type(widget).__name__))
                self.ui.verticalLayout_debug.removeWidget(widget)
                widget.deleteLater()
            LOGGER.debug('Post removeWidget(), Layout_debug.count() = {}'.format(
                         self.ui.verticalLayout_debug.count()))
        else:
            for i in range(len(self._dbg_docks) - 1, -1, -1):
                if i == 0:
                    dock = self._dbg_docks[0]
                    widget = dock.widget()
                    LOGGER.debug('i=0, \ndock: {}\nwidget: {}'.format(dock, widget))
                    if widget:
                        widget.setParent(None)
                        LOGGER.debug('widget.deleteLater: {} '.format(widget))
                        if widget == self.ui.dockWidget_debug1:
                            pass
                        elif isinstance(widget, AbstractDebugWidget):
                            self.pre_jtag_config.disconnect(widget.on_debug_core_pre_connect)
                            self.post_jtag_config.disconnect(widget.on_debug_core_post_connect)
                            self.pre_jtag_unconfig.disconnect(widget.on_debug_core_pre_disconnect)
                            self.post_jtag_unconfig.disconnect(widget.on_debug_core_post_disconnect)
                            widget.error_received.disconnect(self.on_error)
                        widget.deleteLater()
                        dock.setWindowTitle('Debug')
                else:
                    dock = self._dbg_docks.pop(i)
                    widget = dock.widget()
                    if widget == self.ui.dockWidget_debug1:
                        pass
                    elif isinstance(widget, AbstractDebugWidget):
                        self.pre_jtag_config.disconnect(widget.on_debug_core_pre_connect)
                        self.post_jtag_config.disconnect(widget.on_debug_core_post_connect)
                        self.pre_jtag_unconfig.disconnect(widget.on_debug_core_pre_disconnect)
                        self.post_jtag_unconfig.disconnect(widget.on_debug_core_post_disconnect)
                        widget.error_received.disconnect(self.on_error)
                    LOGGER.debug('dock.deleteLater: {}'.format(dock))
                    dock.deleteLater()

    def add_widget_to_dock(self, widget: QWidget):
        if len(self._dbg_docks) == 1 and not self._dbg_docks[0].widget():
            dock = self._dbg_docks[0]
            dock.setWindowTitle(widget.title())
            LOGGER.debug(
                'dock.setWidget, dock: {}\nwidget: {}, widget title={}'.format(dock,
                widget, widget.title()))
            dock.setWidget(widget)
        else:
            dock = QDockWidget(self)
            dock.setFeatures(QDockWidget.DockWidgetFloatable |
                             QDockWidget.DockWidgetClosable |
                             QDockWidget.DockWidgetMovable)
            dock.setWindowTitle(widget.title())
            dock.setObjectName(widget.title())
            #self.setStyleSheet(self.styleSheet() +
            #                   ("\nQDockWidget#" + str(dock.objectName()) +
            #                    "::title{background-color: rgb(50, 147, 208)}\n"
            #                    "QDockWidget#" + str(dock.objectName()) +
            #                    "{color: white}"))
            LOGGER.debug('dock.setWidget, dock: {}\nwidget: {}'.format(dock, widget))
            dock.setWidget(widget)
            if self.ui.actionViewDebug.isChecked():
                self.tabifyDockWidget(self.ui.dockWidget_debug1, dock)
            self._dbg_docks.append(dock)

    def load_debug_widgets_from_profile(self):
        """
        Create and configure Debug Widget according to the profile
        """
        self.remove_debug_widgets()
        for spec in self._profile.list_cores():
            ctype = spec.get_type()
            cls = {'vio': NewVioDebugWidget, 'la': NewLogicNDebugWidget}.get(ctype)
            if not cls:
                raise ValueError('Unknown debugger type "{}".'.format(ctype))

            widget = cls(spec, parent=self)
            self.add_widget_to_dock(widget)
            self.pre_jtag_config.connect(widget.on_debug_core_pre_connect)
            self.post_jtag_config.connect(widget.on_debug_core_post_connect)
            self.pre_jtag_unconfig.connect(widget.on_debug_core_pre_disconnect)
            self.post_jtag_unconfig.connect(widget.on_debug_core_post_disconnect)
            widget.error_received.connect(self.on_error)
            widget.load_profile(self._profile)

        if not DOCKED_GUI:
            LOGGER.debug('Post addWidget(), verticalLayout_debug.count()=%d',
                         self.ui.verticalLayout_debug.count())

    def open_profile_file_and_bind_jtag(self, file_path: str):
        try:
            with open(file_path, 'r') as fp:  # pylint: disable=C0103
                try:
                    JtagManager().add_debug_profile(fp, self._jtag_session)
                except (  # pylint: disable=C0103
                        InvalidJsonError, InvalidProfileError,
                        UnknownUrlError) as exc:
                    self.error(str(exc))
                    return
        except OSError as exc:
            self.warning(str(exc))
            return

    def get_current_debug_target(self) -> DebugUsbTarget:
        """
        Get the current target selected to debug

        Side effect:
        Will touch the connection to query some information

        Raise:
            AssertionError
        """
        usb_target = self.ui.cb_usb_target.currentIndex()
        resolved_usb = self.pgm_win.programmer.resolved_usb
        assert usb_target >= 0 and usb_target < len(resolved_usb), 'No programming cable detected'

        prog_usb_target = resolved_usb[usb_target]

        desc = prog_usb_target.description
        v = prog_usb_target.vendor_id
        p = prog_usb_target.product_id
        url_list = prog_usb_target.URLS

        urls = [None, None, None, None] # Hard-code it to be 4 that support FT4232
        if len(url_list) == 1:
            urls[0] = url_list[0]
            urls[1] = url_list[0]
        elif len(url_list) >= 2:
            urls[0] = url_list[0]
            urls[1] = url_list[1]

        def get_url_from_bus(bus: str) -> Union[str, None]:
            if bus == "A" and len(urls) > 0:
                return urls[0]
            elif bus == "B" and len(urls) > 1:
                return urls[1]
            elif bus == "C" and len(urls) > 2:
                return urls[2]
            elif bus == "D" and len(urls) > 3:
                return urls[3]
            return None

        def parse_pinout_to_bus_and_idx(
            val: str) -> Tuple[Union[str, None], Union[int, None]]:
            """
            Return:
            bus: str in ["A", "B", "C", "D"] or None
            idx: int or None
            """
            pattern_str = r"FTDI_([ABC])DBUS(\d)"
            p = re.compile(pattern_str)
            match = re.search(p, val)

            bus, idx = None, None
            if match:
                bus = match.group(1)
                idx = int(match.group(2))
            return bus, idx
        # url = urls[-1]
        best_profile = self.pgm_win.programmer.get_board_profile(prog_usb_target)
        bus, _ = parse_pinout_to_bus_and_idx(best_profile.pinout.TCK)
        if bus is None:
            bus = 'B'

        url = get_url_from_bus(bus)

        (_, tap) = self._guess_port_tap(v, p, desc)

        if self.pgm_win.xml:
            chain_target_idx = int(self.ui.cb_chip_num.currentText()[0])
            chain_devices = JcfParser(self.pgm_win.xml).all_devices()
        else:
            chain_target_idx = 1
            chain_devices = None

        user = self.ui.cb_user_instr.currentText()
        tap = JtagManager().detect_jtag_tap(url=url, jcfs=chain_devices, chip_num=chain_target_idx)
        result = DebugUsbTarget(url=url,
                                jtag_user=user,
                                chain_devices=chain_devices,
                                chain_target_idx=chain_target_idx,
                                tap=tap)
        LOGGER.info(f'result = {result}')
        return result

    def get_debug_profile(self) -> DebugProfile:
        # return self.get_profile_from_widgets()
        return self._profile

    @log_trace
    def on_configure(self):
        """
        Raise:
            Exception
        """
        if not self._profile:
            self.console.write_error_msg('Debug profile must be imported before connecting debugger!')
            return

        self.pre_jtag_config.emit()
        self.clear_message()

        debug_usb_target = self.get_current_debug_target()
        self.console.write(f'Connecting to URL: {debug_usb_target.url}')
        self.console.write(f'Connecting to JTAG_TAP: {debug_usb_target.tap}')

        try:
            first_chip = self.ui.cb_chip_num.currentIndex()

            self._jtag_sesh = JtagManager().config_jtag(url=debug_usb_target.url,
                                                           tap=debug_usb_target.tap,
                                                           user=debug_usb_target.jtag_user,
                                                           jcfs=debug_usb_target.chain_devices,
                                                           chip_num=debug_usb_target.chain_target_idx)

            self._profile = self.get_debug_profile()
            self._dbg_sesh = JtagManager().config_debug(self._jtag_sesh, self._profile)

            JtagManager().validate_debug_cores(self._dbg_sesh)

        except EfxDbgMisMatchCoreUUIDError as exc:
            self.on_unconfig(force_quit=True)
            self.error(str(exc.msg))
            if exc.core_name is not None:
                if exc.expected == CoreSpec.UNINITIALIZED_UUID:
                    msg = 'Debug Core UUID is missing from profile for {}!'.format(exc.core_name)
                elif exc.received == CoreSpec.UNINITIALIZED_UUID:
                    msg = 'No Debug Core UUID detected on {}! \
                            Check that appropriate the JTAG USER is selected.'.format(exc.core_name)
                else:
                    msg = 'Debug Core UUID mismatch detected on {}! \
                            \n Received UUID: {}\n Expected UUID: {}'\
                            .format(exc.core_name, exc.received, exc.expected)
                self.console.write_error_msg(msg)
                LOGGER.info(msg)
            return
        except (  # pylint: disable=C0103
                ConfigExistingUrlError, DeviceNotFoundError,
                UnknownUsbSernumError) as exc:
            self.error(str(exc))
            return
        except (FtdiError, UsbToolsError, usb.core.USBError) as exc:  # pylint: disable=C0103
            self.error(str(exc) + '. Check the usb driver installation for Windows.')
            return

        self.ui.pb_config_jtag.setEnabled(False)
        self.ui.pb_unconfig_jtag.setEnabled(True)
        self.ui.pb_refresh_status.setEnabled(False)
        self.ui.pb_ti_jtag_status.setEnabled(False)
        self.ui.pb_edit_remote_host.setEnabled(False)

        for widget in self._programmer_widgets_to_freeze:
            if not isinstance(widget, QWidget):
                raise TypeError("Unknown type for programmer widget: {}".format(widget))
            widget.setEnabled(False)

        if self.pgm_win.xml:
            self.ui.cb_chip_num.setCurrentIndex(first_chip)

        self.post_jtag_config.emit(self._dbg_sesh)

    @log_trace
    def on_unconfig(self, force_quit=False):
        self.pre_jtag_unconfig.emit()

        try:
            if self._dbg_sesh:
                JtagManager().unconfig_debug(self._dbg_sesh)
                self._dbg_sesh = None

            if self._jtag_sesh:
                JtagManager().unconfig_jtag(self._jtag_sesh)
                self.console.write(f'Disconnected from URL: {self._jtag_sesh.url}')
                self._jtag_sesh = None

            self.info(self.MSG_I001)
        except Exception as exc:
            self.error(str(exc))
            self.console.write_error_msg(traceback.format_exc())
        finally:
            self._dbg_sesh = None
            self._jtag_sesh = None

        self.ui.pb_config_jtag.setEnabled(True)
        self.ui.pb_unconfig_jtag.setEnabled(False)
        self.ui.pb_refresh_status.setEnabled(True)
        if self.titanium:
            self.ui.pb_ti_jtag_status.setEnabled(True)
        self.ui.pb_edit_remote_host.setEnabled(True)

        for widget in self._programmer_widgets_to_freeze:
            if not isinstance(widget, QWidget):
                raise TypeError("Unknown type for programmer widget: {}".format(widget))
            widget.setEnabled(True)

        self.post_jtag_unconfig.emit()

    def on_error(self, error: Exception):
        LOGGER.error('error = {}'.format(str(error)))
        self.on_unconfig(force_quit=True)
        self.on_debug_refresh()
        if isinstance(error, EfxDbgCorruptedDataError):
            self.error('ERROR: Data corruption due to USB connection issue.')
        else:
            self.error('ERROR: {}'.format(str(error)))
        self.console.write_error_msg('ERROR: {}'.format(str(error)))

    def is_connected(self) -> bool:
        return self._jtag_sesh is not None and self._dbg_sesh is not None

    def on_wave_viewer_config_update(self):
        always_launch_new_win = self.ui.actionAlwaysLaunchNewWaveWindow.isChecked()
        if self.hwtconfig_util:
            self.hwtconfig_util.write_setting(sect='debugger', opt='always_launch_new_wave_window', val=always_launch_new_win)
        for item in self._dbg_docks:
            widget = item.widget()
            if isinstance(widget, NewLogicNDebugWidget):
                widget.update_wave_viewer_config(always_launch_new_win=always_launch_new_win)

    def on_dbg_import_jcf(self):
        self.pgm_win.cb_prog_mode.setCurrentText('JTAG')
        self.pgm_win.on_import_jcf()

def parse_cmdline():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        '--project_home',
        type=str,
        default='.',
        help=
        ('Path of project directory when starting Debugger GUI from Efinity.'
         'In stand-alone mode, debug profile JSON with be saved in current dir.'
        ))
    parser.add_argument('--debug',
                        action='store_true',
                        help='Enable debug messages on stdout.')
    parser.add_argument('--cls',
                        choices=['DbgGui'],
                        default='DbgGui',
                        help='Class for main GUI to launch. Default: DbgGui')
    parser.add_argument('--netdb',
                        default='default.netdb.json',
                        help='Input file for debug signals selection.')
    parser.add_argument('--profile_file',
                        default='',
                        type=str,
                        help='Debug Profile json file.')

    parser.add_argument('--no_save',
                        action='store_false',
                        help='Dont save the debug profile when quit')

    parser.add_argument("--device", help="Device name as per Efinity project")
    parser.add_argument("--family", help="Device family")

    parser.add_argument('--log_path', default=None, help=argparse.SUPPRESS)
    parser.add_argument('--parent_port', default='0', help=argparse.SUPPRESS)

    args = parser.parse_args()
    return args


def gui_factory(args: argparse.Namespace):
    classname = args.cls
    try:
        cls = globals()[classname]
    except KeyError as exc:
        raise Exception('GUI class {} is not yet implemented.'.format(
            str(exc))) from exc
    return cls(args)

def set_stay_on_top(enable=False):
    global global_gui
    if not global_gui:
        return

    if enable:
        global_gui.setWindowFlag(Qt.WindowStaysOnTopHint, True)
        # W/A to ensure window bring to top on Windows, as Windows precent focus stealing
        global_gui.setAttribute(Qt.WA_ShowWithoutActivating)
        # Bring back minimize Window
        global_gui.showNormal()
    else:
        global_gui.setWindowFlag(Qt.WindowStaysOnTopHint, False)

    global_gui.show()


def check_parent_aliveness():
    """
    Ping server to check if parent process is still running
    """

    message = "Hello from Debugger GUI"
    try:
        response = global_grpc_stub.Ping(efx_dbg.efinity_service_pb2.PingData(data = message))
        if response.data == "Raise":
            set_stay_on_top(True)
            QTimer.singleShot(10, set_stay_on_top)
    except grpc.RpcError as rpc_error:
        global global_gui, global_grpc_port
        global_gui.info('Efinity not reachable at port {}. Exiting...'.format(global_grpc_port))
        global_gui.closeEvent()
        sys.exit(0)


def main():
    args = parse_cmdline()
    # Support overwriting pgm log path
    efx_pgm.util.app_logger.LoggerManager.set_log_path(args.log_path)

    logging.basicConfig(
        format=
        '[%(threadName)12s] %(levelname)6s: %(name)16s: %(funcName)32s(): %(message)s'
    )
    logging.addLevelName(LogLevel.LOG_LEVEL_JTAG, 'jtag')
    logging.addLevelName(LogLevel.LOG_LEVEL_TRACE, 'trace')
    # if args.debug:
    #     logging.getLogger().setLevel(logging.DEBUG)
    # logging.getLogger('efx_dbg.engine').setLevel(logging.DEBUG)
    jtag_log_path = 'efinity_dbg.jtag.log'
    if args.project_home:
        jtag_log_path = Path(args.project_home) / 'efinity_dbg.jtag.log'

    jtag_formatter = logging.Formatter(
            '%(asctime)s %(levelname)s: %(name)s (%(lineno)d): %(funcName)s(): %(message)s'
    )
    jtag_log_handler = RotatingFileHandler(jtag_log_path, 'w', maxBytes=5*1024*1024, backupCount=1)
    jtag_log_handler.setFormatter(jtag_formatter)
    logging.getLogger('efx_dbg.jtag').propagate = False
    logging.getLogger('efx_dbg.jtag').setLevel(logging.DEBUG)
    logging.getLogger('efx_dbg.jtag').addHandler(jtag_log_handler)
    logging.getLogger('asyncio').setLevel(logging.CRITICAL)

    # Enable High DPI support
    QApplication.setAttribute(Qt.AA_EnableHighDpiScaling, True)
    QApplication.setAttribute(Qt.AA_UseHighDpiPixmaps, True)
    QApplication.setHighDpiScaleFactorRoundingPolicy(Qt.HighDpiScaleFactorRoundingPolicy.PassThrough)
    qapp = QApplication(sys.argv)
    qapp.setStyle("fusion")
    qapp.setStyleSheet(
        "QToolTip { background-color: #f0f8ff; color: #007fbe; border: none; }")

    # Load the fonts
    QFontDatabase().addApplicationFont(":/fonts/SourceCodePro-Bold.ttf")
    QFontDatabase().addApplicationFont(":/fonts/SourceCodePro-BoldItalic.ttf")
    QFontDatabase().addApplicationFont(":/fonts/SourceCodePro-Italic.ttf")
    QFontDatabase().addApplicationFont(":/fonts/SourceCodePro-Regular.ttf")
    QFontDatabase().addApplicationFont(":/fonts/SourceSansPro-Bold.ttf")
    QFontDatabase().addApplicationFont(":/fonts/SourceSansPro-BoldItalic.ttf")
    QFontDatabase().addApplicationFont(":/fonts/SourceSansPro-Italic.ttf")
    QFontDatabase().addApplicationFont(":/fonts/SourceSansPro-Regular.ttf")

    # Load the setting files
    EFINITY_HOME = os.environ['EFINITY_HOME']
    APP_INI = os.path.join(EFINITY_HOME, "bin", "style.ini")
    settings = QSettings(APP_INI, QSettings.IniFormat)

    # Set the fonts global wise
    font_name = settings.value("debugger/global_font_name", "Source Sans Pro")
    font_size = settings.value("debugger/global_font_size", "11")
    font = QFont(font_name, int(font_size))
    qapp.setFont(font)

    global global_gui
    global_gui = gui_factory(args)
    global_gui.show()

    if int(getattr(args, "parent_port", 0)):
        global global_grpc_stub, global_grpc_port
        global_grpc_port = int(args.parent_port)
        timer = QTimer()
        try:
            channel = grpc.insecure_channel(f'127.0.0.1:{global_grpc_port}')
            global_grpc_stub = efx_dbg.efinity_service_pb2_grpc.MiscServiceStub(channel)
            timer.timeout.connect(check_parent_aliveness)
            timer.start(1000) # 1s TTL
        except grpc.RpcError as rpc_error:
            pass

    sys.exit(qapp.exec_())


if __name__ == '__main__':
    main()
